# tracks.rb

class StarHistory

    include Tioga
    include Math
    include FigureConstants
    
    
    def mark_point(x, y, dx, dy, n)
        mark_spot(x, y)
        t.show_label('text' => sprintf('%i', n), 'x' => x+dx, 'y' => y+dy,
            'color' => Crimson,
            'scale' => 0.7, 'justification' => CENTERED, 'alignment' => ALIGNED_AT_MIDHEIGHT)
    end
    
    
    def add_info_line(xs, ys)
        t.line_type = Line_Type_Dash
        t.line_color = SlateGray
        t.line_width = 1
        t.append_points_to_path(xs, ys)
        t.stroke
    end
    
    
    def add_ZAMS(xs, ys)
        add_info_line(xs, ys)
    end
    
    
    def add_eta(ary, eta)
        xs = ary[0]; ys = ary[1]
        add_info_line(xs, ys)
        j = ys.where_closest(t.bounds_ymin + 0.08 * t.bounds_height)
        t.show_label('text' => sprintf('$\eta$=%i', eta), 'x' => xs[j], 'y' => ys[j]-0.05,
            'color' => Crimson,
            'scale' => 0.6, 'justification' => CENTERED, 'alignment' => ALIGNED_AT_MIDHEIGHT)
    end
    
    
    def plot_lgg_lgTeff(right_margin=0, left_margin=0)
        setup_data
        t.landscape unless t.in_subplot
        t.title_shift += 0.5
        margin = 0.1
        t.subplot('right_margin' => right_margin) do
            background
            xs = d.log_Teff
            tmax = xs.max; tmin = xs.min; xmargin = margin*(tmax-tmin)
            left = tmax + xmargin; right = tmin - xmargin 
            ys = d.log_g
            lmax = ys.max; lmin = ys.min; ymargin = margin*(lmax-lmin)
            bottom = lmax + ymargin; top = lmin - ymargin
            t.show_plot('left_boundary' => left, 'right_boundary' => right,
                'top_boundary' => top, 'bottom_boundary' => bottom) do
                t.show_xlabel('log $T_{eff}$')
                t.show_ylabel('log g')
                #add_ZAMS(@zams_log_Surface_Temp, @zams_log_Luminosity)
                stroke_track(xs, ys, Blue, nil, Line_Type_Solid, 0)
            end
        end
    end
    
    
    def plot_lgg_Teff(right_margin=0, left_margin=0)
        setup_data
        t.landscape unless t.in_subplot
        t.title_shift += 0.5
        margin = 0.1
        t.subplot('right_margin' => right_margin) do
            background
            xs = d.log_Teff.exp10
            tmax = xs.max; tmin = xs.min; xmargin = margin*(tmax-tmin)
            left = tmax + xmargin; right = tmin - xmargin 
            ys = d.log_g
            lmax = ys.max; lmin = ys.min; ymargin = margin*(lmax-lmin)
            bottom = lmax + ymargin; top = lmin - ymargin
            t.show_plot('left_boundary' => left, 'right_boundary' => right,
                'top_boundary' => top, 'bottom_boundary' => bottom) do
                t.show_xlabel('$T_{eff}$')
                t.show_ylabel('log g')
                #add_ZAMS(@zams_log_Surface_Temp, @zams_log_Luminosity)
                stroke_track(xs, ys, Blue, nil, Line_Type_Solid, 0)
            end
            f = File.open('logg_Teff.data', 'w')
            xs.length.times do |i|
                f.printf("%20g %20g %20g %20g\n", xs[i], ys[i], d.star_age[i], d.star_mass[i])
            end
            f.close
        end
    end
    
    
    def plot_A_vs_B(
            xs, ys, xlabel, ylabel, title, 
            right_margin, left_margin, reverse_xaxis,
            ymin = nil, ymax = nil, xmin = nil, xmax = nil, decorate_line = nil)
        setup_data
        t.landscape unless t.in_subplot
        t.set_aspect_ratio(0.85)
        t.title_shift += 0.5
        margin = 0.1
        t.line_width = 0.8
        t.subplot('right_margin' => right_margin) do
            background
            xmax = xs.max if xmax == nil
            xmin = xs.min if xmin == nil
            xmargin = margin*(xmax-xmin)
            if reverse_xaxis
                left = xmax + xmargin; right = xmin - xmargin
            else
                right = xmax + xmargin; left = xmin - xmargin 
            end
            ymax = ys.max if ymax == nil
            ymin = ys.min if ymin == nil
            ymargin = margin*(ymax-ymin)
            top = ymax + ymargin; bottom = ymin - ymargin
            t.show_plot('left_boundary' => left, 'right_boundary' => right,
                'top_boundary' => top, 'bottom_boundary' => bottom) do
                t.show_xlabel(xlabel)
                t.show_ylabel(ylabel)
                t.show_title(title) unless title == nil
                stroke_track(xs, ys, Black, nil, Line_Type_Solid, 0)
                unless decorate_line == nil
                    str = sprintf("%s(%g, %g, %g, %g)", decorate_line, ymin, ymax, xmin, xmax)
                    puts "eval #{str}"
                    eval(str) 
                end
            end
        end
    end
    
    
    def plot_H_R(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.log_Teff, 
            d.log_L, 
            'log $\mathrm{T_{eff}}$ (K)',  'log L ($\mathrm{L_\odot}$)', 
            nil, right_margin, left_margin, true,
            nil, nil, nil, nil, 'decorate_H_R')
    end
    
    
    def plot_H_R_2(right_margin=0, left_margin=0)
        xs = d.log_Teff.exp10
        ys = d.log_L.exp10
        plot_A_vs_B(
            xs, ys, 
            '$\mathrm{T_{eff}}$ (K)', 
            'L/$\mathrm{L_\odot}$', 
            nil, right_margin, left_margin, true,
            nil, nil, nil, nil, 'decorate_H_R')
    end
    
    
    def plot_H_R_3(right_margin=0, left_margin=0)
        xs = d.log_Teff.exp10
        ys = d.log_L
        plot_A_vs_B(
            xs, ys, 
            'Teff (K)', 
            'log (L/$\mathrm{L_\odot}$)', 
            nil, right_margin, left_margin, true,
            nil, nil, nil, nil, 'decorate_H_R')
    end
    
    
    
    def plot_H_R_RGB(right_margin=0, left_margin=0)
        xs = d.log_Teff.exp10
        ys = d.log_L.exp10
        plot_A_vs_B(
            xs, ys, 
            'Teff (K)', 
            'Luminosity ($\mathrm{L_\odot}$)', 
            nil, right_margin, left_margin, true,
            nil, nil, nil, nil, 'decorate_H_R_RGB')
    end
    
    
    def decorate_H_R_RGB(ymin, ymax, xmin, xmax)
        dx0 = 0
        dx0 = @labels_HR_dx unless @labels_HR_dx == nil
        dy0 = 0
        dy0 = @labels_HR_dy unless @labels_HR_dy == nil
        dx = 0; dy = 0
        
        k = d.log_LHe.where_max
        he_flash = d.star_age[k]
        k = d.log_L.where_max
        peak = d.star_age[k] # Gyr
        puts "peak age #{d.star_age[k]}"
        puts "last age #{d.star_age[-1]}"
        puts "last - peak #{d.star_age[-1] - d.star_age[k]}"

        ages = [ -7000, -1600, -800, -400, -200, -100, -50, -25, 
                -12, -6, -3, -1, 0, 0.01, 0.054, 0.82, 70, 100, 114 ] # in Myr
        xs = d.log_Teff
        ys = d.log_L
        justification = LEFT_JUSTIFIED
        dx = -0.004
        dy = -0.3
        ages.each do |age|
            x = Dvector.linear_interpolate(peak + age*1e-3, d.star_age, xs)
            y = Dvector.linear_interpolate(peak + age*1e-3, d.star_age, ys)

            if age < 0
                txt = sprintf(" %3.0f ", age) # age in Myr
            elsif age == 0
                txt = ' 0'
            elsif age > 1
                txt = sprintf(" %3.0f ", age) # age in Myr
            else
                txt = sprintf(" %3.3f ", age) # age in Myr
            end
            dx = 0.004 if age > 0
            dy = 0.3 if age > 0
            justification = RIGHT_JUSTIFIED if age > 0
            t.line_width = 0.5
            t.stroke_line(x, y, x+dx+dx0, y+dy+dy0)
            t.show_marker('x' => x+dx+dx0, 'y' => y+dy+dy0, 
                'font' => Helvetica, 'string' => txt, 'justification' => justification,
                'scale' => 0.5, 'color' => Black)
            
        end

        x0 = 3.54
        y = 0.6
        dy = 0.2
        scale = 0.6
        t.show_label('text' => 'M_0 = 1.00 $\mathrm{M_\odot}$', 'x' => x0, 'y' => y,
            'color' => Black, 
            'scale' => scale, 'justification' => LEFT_JUSTIFIED, 'alignment' => ALIGNED_AT_MIDHEIGHT)
        y = y - dy
        t.show_label('text' => 'Y_0 = 0.28', 'x' => x0, 'y' => y,
            'color' => Black, 
            'scale' => scale, 'justification' => LEFT_JUSTIFIED, 'alignment' => ALIGNED_AT_MIDHEIGHT)
        y = y - dy
        t.show_label('text' => 'Z_0 = 0.02', 'x' => x0, 'y' => y,
            'color' => Black, 
            'scale' => scale, 'justification' => LEFT_JUSTIFIED, 'alignment' => ALIGNED_AT_MIDHEIGHT)
        
        y = 3.35
        t.show_label('text' => 'times (Myrs) relative to peak RGB luminosity', 'x' => 3.76, 'y' => y,
            'color' => Black, 
            'scale' => 0.5, 'justification' => LEFT_JUSTIFIED, 'alignment' => ALIGNED_AT_MIDHEIGHT)
        
        return
    end
    
    
    def decorate_H_R(ymin, ymax, xmin, xmax)
        return if @labels_HR == nil
        @labels_HR.each do |pair|
            k = d.model_number.where_closest(pair[0])
            txt = pair[1]
            puts "HR label at #{k} #{txt}"
            if k != nil && k < xs.length
                if pair.length == 4
                    dx = pair[2]; dy = pair[3]
                else
                    dx = 0; dy = 0
                end
                t.show_marker('x' => xs[k], 'y' => ys[k], 
                    'marker' => Circle, 'justification' => CENTERED,
                    'scale' => 0.4, 'color' => Black)
                t.show_marker('x' => xs[k]+dx+dx0, 'y' => ys[k]+dy+dy0, 
                    'font' => Helvetica, 'string' => txt, 'justification' => LEFT_JUSTIFIED,
                    'scale' => 0.5, 'color' => Black)
            end
        end
    end
    
    
    def plot_T_RHO(right_margin=0, left_margin=0)
        #t.set_aspect_ratio(0.85)
        plot_A_vs_B(
            d.log_center_Rho, 
            d.log_center_T, 
            'log $\rho_{\rm c}$ (g cm^{-3})', 
            'log T_c \rm (K)', 
            nil, right_margin, left_margin, false,
            nil, nil, nil, nil, 'decorate_T_RHO')
    end
    
    
    def decorate_T_RHO(ymin, ymax, xmin, xmax)
        show_gamma_4_thirds
        return if @labels_T_RHO == nil
        xs = d.log_center_Rho
        ys = d.log_center_T
        dx0 = 0
        dx0 = @labels_T_RHO_dx unless @labels_T_RHO_dx == nil
        dy0 = 0
        dy0 = @labels_T_RHO_dy unless @labels_T_RHO_dy == nil
        @labels_T_RHO.each do |pair|
            k = d.model_number.where_closest(pair[0])
            txt = pair[1]
            puts "T Rho label at #{k} #{txt}"
            if k != nil && k < xs.length
                if pair.length == 4
                    dx = pair[2]; dy = pair[3]
                else
                    dx = 0; dy = 0
                end
                t.show_marker('x' => xs[k], 'y' => ys[k], 
                    'marker' => Circle, 'justification' => CENTERED,
                    'scale' => 0.4, 'color' => Black)
                t.show_marker('x' => xs[k]+dx+dx0, 'y' => ys[k]+dy+dy0, 
                    'font' => Helvetica, 'string' => txt, 'justification' => LEFT_JUSTIFIED,
                    'scale' => 0.5, 'color' => Black)
            end
        end
    end
    
    
    def show_gamma_4_thirds
        # show where electron to baryon ratio is twice that expected
        t.line_width = 0.8
        t.line_type = Line_Type_Dash
        t.append_points_to_path(@gamma_4_thirds[0],@gamma_4_thirds[1])
        t.stroke_color = Crimson
        t.stroke
        t.show_label('text' => '$\Gamma_{1} < 4/3$', 'x' => 2.8, 'y' => 9.1,
                    'color' => Crimson, 
                    'scale' => 0.9, 'justification' => CENTERED, 
                    'alignment' => ALIGNED_AT_BASELINE)
    end
    
    
    def plot_core_mass_luminosity(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.h1_boundary_mass, 
            d.log_L, 
            'Core mass', 
            'log Luminosity $\mathrm{L_\odot}$', 
            nil, right_margin, left_margin, false)
    end
    
    
    def plot_surf_He4_logL(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.log_L, 
            d.surface_he4, 
            'log Luminosity $\mathrm{L_\odot}$', 
            'surface He4', 
            nil, right_margin, left_margin, false)
    end
    
    
    def log_inertia_ratio_delta_nu(right_margin=0, left_margin=0)
        name = 'int_k_r_dr_nu_max_Sl1'
        col_num = d.column_names.index(name)
        intertia_ratio = d.columns[col_num]
        plot_A_vs_B(
            d.delta_nu, 
            intertia_ratio, 
            '$\Delta \nu \ \mathrm{(\mu Hz)}$', 
            'log intertia ratio', 
            nil, right_margin, left_margin, false,
            ymin = nil, ymax = 2, xmin = 2, xmax = 10, decorate_line = nil)
    end
    
    
    def plot_luminosity_kr_integral(right_margin=0, left_margin=0)
        name = 'int_k_r_dr_nu_max_Sl1'
        col_num = d.column_names.index(name)
        intertia_ratio = d.columns[col_num]
        plot_A_vs_B(
            d.log_L, 
            intertia_ratio, 
            'log Luminosity $\mathrm{L_\odot}$', 
            'log exp(2I)', 
            nil, right_margin, left_margin, false,
            ymin = 0, ymax = 2.3, xmin = 1.45, xmax = 2.35, decorate_line = nil)
    end
    
    
    def plot_delta_wkb_delta_nu(right_margin=0, left_margin=0)
        name = 'int_k_r_dr_nu_max_Sl1'
        col_num = d.column_names.index(name)
        intertia_ratio = d.columns[col_num]
        plot_A_vs_B(
            d.delta_nu, 
            intertia_ratio, 
            '$\Delta \nu \ \mathrm{(\mu Hz)}$', 
            'log exp(2I)', 
            nil, right_margin, left_margin, false,
            ymin = 0, ymax = 3, xmin = 1, xmax = 5, decorate_line = nil)
    end
    
    
    def plot_delta_Pg_delta_nu(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.delta_nu, 
            d.delta_Pg, 
            '$\Delta \nu \ \mathrm{(\mu Hz)}$', 
            '$\Delta \mathrm{P_{\!g} \ (s)}$', 
            nil, right_margin, left_margin, false,
            ymin = nil, ymax = nil, xmin = nil, xmax = nil, decorate_line = nil)
    end
    
    
    def plot_nu_max_Teff(right_margin=0, left_margin=0)
        xs = d.log_Teff.exp10
        ys = d.nu_max
        plot_A_vs_B(
            xs, ys, 
            '$\mathrm{T_{eff}}$ (K)', 
            '$\nu_\mathrm{max} \ \mathrm{(\mu Hz)}$', 
            nil, right_margin, left_margin, true,
            nil, nil, nil, nil, 'decorate_H_R')
    end
    
    
    def plot_nu_max_delta_nu(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.delta_nu, 
            d.nu_max, 
            '$\Delta \nu \ \mathrm{(\mu Hz)}$', 
            '$\nu_\mathrm{max} \ \mathrm{(\mu Hz)}$', 
            nil, right_margin, left_margin, false,
            ymin = nil, ymax = nil, xmin = nil, xmax = nil, decorate_line = nil)
    end
    
    
    def plot_log_delta_Pg_delta_nu(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.delta_nu, 
            d.log_delta_Pg, 
            '$\Delta \nu \ \mathrm{(\mu Hz)}$', 
            'log $\Delta \mathrm{P_{\!g} \ (s)}$', 
            nil, right_margin, left_margin, false,
            ymin = nil, ymax = nil, xmin = nil, xmax = nil, decorate_line = nil)
    end
    
    
    def plot_nu_max_34_div_delta_nu_vs_delta_nu(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.delta_nu, 
            d.nu_max_3_4th_div_delta_nu, 
            '$\Delta \nu \ \mathrm{(\mu Hz)}$', 
            '$\nu_{max}^{0.75} /  \Delta \nu$', 
            nil, right_margin, left_margin, false,
            ymin = nil, ymax = nil, xmin = nil, xmax = nil, decorate_line = nil)
    end
    
    
    def plot_luminosity_Tcentral(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.log_center_T, 
            d.log_L, 
            'log Center T (K)', 
            'log Luminosity $\mathrm{L_\odot}$', 
            nil, right_margin, left_margin, false)
    end
    
    
    def plot_lgTeff_central_He(right_margin=0, left_margin=0)
        plot_A_vs_B(
            d.center_he4, 
            d.log_Teff, 
            'central helium mass fraction', 
            'log Teff', 
            nil, right_margin, left_margin, true)
    end
    

    def plot_H_R_T_RHO
        setup_data
        t.set_portrait
        t.rescale(0.55)
        right_margin = 0.2
        left_margin = 0.85
        t.subplot('bottom_margin' => 0.57) do
            plot_H_R(right_margin, left_margin)
        end
        t.subplot('top_margin' => 0.57) do
            plot_T_RHO(right_margin, left_margin)
        end
    end
    

    def plot_H_R_2_T_RHO
        setup_data
        t.set_portrait
        t.rescale(0.55)
        right_margin = 0.2
        left_margin = 0.85
        t.subplot('bottom_margin' => 0.57) do
            plot_H_R_2(right_margin, left_margin)
        end
        t.subplot('top_margin' => 0.57) do
            plot_T_RHO(right_margin, left_margin)
        end
    end
    

    def plot_H_R_3_T_RHO
        setup_data
        t.set_portrait
        t.rescale(0.55)
        right_margin = 0.2
        left_margin = 0.85
        t.subplot('bottom_margin' => 0.76) do
            plot_H_R_2(right_margin, left_margin)
        end
        t.subplot('top_margin' => 0.38, 'bottom_margin' => 0.38) do
            plot_T_RHO(right_margin, left_margin)
        end
        t.subplot('top_margin' => 0.76) do
           ys = d.log_L.exp10
           plot_A_vs_B(
               d.log_center_T, 
               ys, 
               'log Central T (K)', 
               'Luminosity ($\mathrm{L_\odot}$)', 
               nil, right_margin, left_margin, false)
        end
    end
    

    def init_for_plot_many
        @labels = [ 'first', 'second' ]
        @show_flags = [ true, true ]        
        @colors = [ Blue, Green ]
        @line_types = [ Line_Type_Solid, Line_Type_Dash ]
    end 

    
    def min_of_many(ary, flags, limit = nil)
        return nil if ary == nil || ary.size == 0
        have_min = false
        ymin = nil
        if flags.length != ary.length
            puts "must have same length array for flags and data in min_of_many"
            return nil
        end
        ary.length.times { |i|
            if flags[i]
                if have_min
                    amin = ary[i].min
                    ymin = amin if amin < ymin
                else
                    ymin = ary[i].min
                    have_min = true
                end
            end
        }
        ymin = limit if limit != nil && ymin < limit
        return ymin
    end
    
    
    def max_of_many(ary, flags, limit = nil)
        return nil if ary == nil || ary.size == 0
        have_max = false
        ymax = nil
        if flags.length != ary.length
            puts "must have same length array for flags and data in max_of_many"
            return nil
        end
        ary.length.times { |i|
            if flags[i]
                if have_max
                    amax = ary[i].max
                    ymax = amax if amax > ymax
                else
                    ymax = ary[i].max
                    have_max = true
                end
            end
        }
        ymax = limit if limit != nil && ymax > limit
        return ymax
    end

    
    
    def plot_many_A_vs_B(dict)
        yary = dict['yary']
        xary = dict['xary']
        ylabel = dict['ylabel']
        title = dict['title']
        title = ylabel if title == nil

        xmin = dict['xmin']
        xmax = dict['xmax']

        ymin = dict['ymin']
        ymax = dict['ymax']
        ymargin = dict['ymargin']
        
        xlabel = dict['xlabel']
        reverse_xaxis = dict['reverse_xaxis']

        decorate = dict['decorate']

        #t.landscape unless t.in_subplot
        t.title_shift += 0.5
        margin = 0.1

        legend_left_margin = 0.66
        legend_top_margin = 0.06

        t.legend_text_dy = 1.2
        t.legend_line_x1 = 3
        t.legend_text_xstart = 4
        t.ylabel_shift = 1.2
        
        last_p = xary.length-1

        t.show_plot_with_legend('legend_scale' => 1) do 

            background

            t.show_title(title) unless title == nil
            t.show_xlabel(xlabel); t.no_xlabel
            t.show_ylabel(ylabel); t.no_ylabel
            
            have_xmax = xmax != nil
            have_xmin = xmin != nil
            xmax = Dvector.max_of_many(xary) unless have_xmax
            xmin = Dvector.min_of_many(xary) unless have_xmin
            xmargin = margin*(xmax-xmin)
            
            if have_xmax
                right = xmax
            else
                right = xmax + xmargin 
            end
            
            if have_xmin
                left = xmin
            else
                left = xmin - xmargin 
            end
            
            if reverse_xaxis
               temp = left; left = right; right = temp
            end
            
            have_ymax = ymax != nil
            have_ymin = ymin != nil
            ymax = Dvector.max_of_many(yary) unless have_ymax
            ymin = Dvector.min_of_many(yary) unless have_ymin
            ymargin = margin*(ymax-ymin)
                   
            if have_ymax
                top = ymax
            else
                top = ymax + ymargin 
            end
            
            if have_ymin
                bottom = ymin
            else
                bottom = ymin - ymargin 
            end

            t.show_plot('left_boundary' => left, 'right_boundary' => right,
                'top_boundary' => top, 'bottom_boundary' => bottom) do
                @colors.length.times do |i|
                    if @show_flags[i] && xary[i] != nil && yary[i] != nil && xary[i].length == yary[i].length
                        stroke_track(xary[i], yary[i], @colors[i], @labels[i], @line_types[i], 0)
                    end
                end 
                unless decorate == nil
                    str = sprintf("%s(%g, %g, %g, %g)", decorate, ymin, ymax, xmin, xmax)
                    puts "eval #{str}"
                    eval(str) 
                end
            end

        end
    end

    
    
    def rlo_axis_style
        t.xaxis_major_tick_length = 0.4
        t.yaxis_major_tick_length = 0.4
        t.xaxis_minor_tick_length = 0.15
        t.yaxis_minor_tick_length = 0.15
    end
    
    
    def plot_rlo1
        t.set_aspect_ratio(0.5)
        rlo_axis_style
        num_plots = 2
        row_margin = 0.01
        d = @logs[0]
        names = d.column_names
        xmin = 8.1 #7.5  #2.3
        xmax = 9.8
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('star_mass'), 'mass', false, 
                nil, nil, xmin, xmax, 0.018, 0.4)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            #plot_column(names.index('period_hr'), 'orbital period (hr)', false, 
            #    nil, nil, xmin, xmax, 0.9, 12)
            plot_column(names.index('period_minutes'), 'orbital period (minutes)', false, 
                nil, nil, xmin, xmax, 50, 230)
        end
    end
    
    
    def plot_rlo2
        t.set_aspect_ratio(0.5)
        rlo_axis_style
        num_plots = 3
        row_margin = 0.01
        d = @logs[0]
        names = d.column_names
        xmin = 8.1 #7.5  #2.3
        xmax = 9.8
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 1, 'row_margin' => row_margin)) do
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            plot_column(names.index('star_mass'), 'mass ($\mathrm{M_\odot}$)', false, 
                nil, nil, xmin, xmax, 0.018, 0.4)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 2, 'row_margin' => row_margin)) do 
            t.title_visible = false
            t.xlabel_visible = false
            t.xaxis_type = AXIS_WITH_TICKS_ONLY
            #plot_column(names.index('period_hr'), 'orbital period (hr)', false, 
            #    nil, nil, xmin, xmax, 0.9, 12)
            plot_column(names.index('lg_mdot'), 'log mdot ($\mathrm{M_\odot}$ yr^{-1})', false, 
                nil, nil, xmin, xmax, -17, -8)
        end
        t.subplot(t.row_margins('num_rows' => num_plots, 'row' => 3, 'row_margin' => row_margin)) do 
            t.title_visible = false
            #plot_column(names.index('period_hr'), 'orbital period (hr)', false, 
            #    nil, nil, xmin, xmax, 0.9, 12)
            plot_column(names.index('period_minutes'), 'period (min)', false, 
                nil, nil, xmin, xmax, 50, 230)
        end
    end
    
    
    def plot_rlo_R_vs_M(right_margin=0, left_margin=0)
        ymin = nil
        ymax = nil
        xmin = nil
        xmax = 0.21
        radius = d.log_R.exp10
        plot_A_vs_B(
            d.star_mass, 
            radius, 
            'M_2\ ($\mathrm{M_\odot}$)',  'R_2\ ($\mathrm{R_\odot}$)', 
            nil, 0, 0, false,
            ymin, ymax, xmin, xmax, decorate_rlo_R_vs_M)
        #write_rlo_table
    end
    
    
    def decorate_rlo_R_vs_M
        return
        xs = [ 0.07, 0.09 ]
        ys = [ ]
        t.show_marker('xs' => xs, 'ys' => ys, 
            'marker' => Plus, 'justification' => CENTERED,
            'scale' => 0.4, 'color' => Black)
    end 
    
    
    def plot_rlo_Porb_mdot(right_margin=0, left_margin=0)
        ymin = -12
        ymax = nil
        xmin = nil
        xmax = nil
        plot_A_vs_B(
            d.period_hr.safe_log10, 
            d.log_abs_mdot, 
            'log P_{\rm orb} \rm (hr)',  'log \dot M\ ($\mathrm{M_\odot}$ yr^{-1})', 
            nil, right_margin, left_margin, false,
            ymin, ymax, xmin, xmax)
        #write_rlo_table
    end
    
    
    def plot_rlo_Porb_mass(right_margin=0, left_margin=0)
        ymin = nil
        ymax = nil
        xmin = nil
        xmax = nil
        plot_A_vs_B(
            d.star_mass.safe_log10, 
            d.period_hr.safe_log10, 
            'log M\ ($\mathrm{M_\odot}$)', 'log P_{\rm orb} \rm (hr)',  
            nil, right_margin, left_margin, false,
            ymin, ymax, xmin, xmax)
    end
    
    def write_rlo_table
        puts "write_rlo_table"
        k = d.star_mass.where_first_lt(0.2)
        offset = d.star_age[k]
        star_ages = d.star_age.sub(offset)
        last_age = star_ages[-1]
        dlogR = d.log_R.dup
        dlogM = d.star_mass.log10
        (dlogR.size-1).times do |i|
            dlogR[i] = dlogR[i] - dlogR[i+1]
            dlogM[i] = dlogM[i] - dlogM[i+1]
        end
        dlogR[-1] = dlogR[-2]
        dlogM[-1] = dlogM[-2]
        25.times { dlogR = dlogR.convolve([1,2,1],1); dlogM = dlogM.convolve([1,2,1],1) }
        chi = dlogR.div(dlogM)
        filename = 'rlo_hist.data'
        f = File.open(filename,"w")
        f.puts "time(yrs)      Mdonor(Msun)   Porb(minutes)       Porb(hr)  Mdot(msun/yr) rldonor(rsun)  " +
             "rlaccre(rsun)   R              dlnR_dlnM     Teff            log_Tc          log_L           " +
             "lg_LWD          e_thermal      t_mdot_yrs     t_KH_yrs        B             atm_col_den   jdot_div_jdotgr"
        age_prev = nil
        if true # by age
            interval = 0.001   # use 0.1 for the jdot = 3 cases
            n = (last_age/interval).to_i 
            n.times { |i| age_prev = write_rlo_line(f,i*interval,d,star_ages,chi,age_prev) }
        else # by model
            dn = 100
            n = d.model_number.length/dn 
            n.times { |i| age_prev = write_rlo_line(f,star_ages[i*dn],d,star_ages,chi,age_prev) }
        end
        write_rlo_line(f,last_age,d,star_ages,chi,age_prev)
        f.close
        puts "done write_rlo_table to " + filename
    end
    
    def write_rlo_line(f,age,d,star_ages,chi,age_prev)
        k = star_ages.where_closest(age)
        
        return age_prev if k == nil
        return age_prev if age_prev != nil && (star_ages[k] - age_prev).abs < 0.001
        
        secyer = 3.1558149984e7
        lsun = 3.8418e33
        if d.e_thermal == nil
            e_thermal = 0
            b = 0
            t_mdot = 0
            t_KH = 0
        else
            e_thermal = d.e_thermal[k]
            t_mdot = (d.star_mass[k])/(10.0**d.log_abs_mdot[k]/secyer + 1e-99) # s
            t_mdot = 1e99 if t_mdot > 1e99
            t_KH = e_thermal/(10**d.log_L[k]*lsun) # s
            b = t_KH/t_mdot
        end
        lg_LWD = log10(6e-3*(10.0**d.log_abs_mdot[k]/1e-10)*(d.star_mass[k]/0.9)**0.4 + 1e-99)
        f.puts sprintf(
            "%12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e   " +
            "%12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e   %12.6e",
            star_ages[k]*@age_Scale,
            d.star_mass[k], 
            d.period_hr[k]*60.0, 
            d.period_hr[k],
            10.0**d.log_abs_mdot[k], 
            d.rl_donor[k], 
            d.rl_accretor[k],
            10.0**d.log_R[k], 
            chi[k], 
            10.0**d.log_Teff[k], 
            d.log_center_T[k], 
            d.log_L[k], 
            lg_LWD, 
            e_thermal, 
            t_mdot/secyer, 
            t_KH/secyer, 
            b,
            d.atmosphere_column_density[k], 
            d.jdot_div_jdotgr[k])
        return star_ages[k]
    end
    


    
    
    def plot_As_vs_Bs(
            xs_ary, ys_ary, xlabel, ylabel, title, 
            right_margin, left_margin, reverse_xaxis, decorate_line = nil, legend_proc = nil,
            ymin = nil, ymax = nil, xmin = nil, xmax = nil, margin = nil, 
            reverse_yaxis = false, aspect_ratio = nil, 
            line_types = nil, line_widths = nil, line_colors = nil)
        setup_data
        #t.landscape unless t.in_subplot
        if aspect_ratio == nil
            t.portrait unless t.in_subplot
        else
            t.set_aspect_ratio(aspect_ratio)
        end
        t.title_shift += 0.5
        margin = 0.1 if margin == nil
        t.subplot('right_margin' => right_margin) do
            background
            xmax = max_of_many(xs_ary) if xmax == nil
            xmin = min_of_many(xs_ary) if xmin == nil
            xmargin = margin*(xmax-xmin)
            if reverse_xaxis
                left = xmax + xmargin; right = xmin - xmargin
            else
                right = xmax + xmargin; left = xmin - xmargin 
            end
            ymax = max_of_many(ys_ary) if ymax == nil
            ymin = min_of_many(ys_ary) if ymin == nil
            ymargin = margin*(ymax-ymin)
            if reverse_yaxis
                bottom = ymax + ymargin; top = ymin - ymargin
            else
                top = ymax + ymargin; bottom = ymin - ymargin
            end
            t.show_plot('left_boundary' => left, 'right_boundary' => right,
                'top_boundary' => top, 'bottom_boundary' => bottom) do
                eval(legend_proc) unless legend_proc == nil
                t.show_xlabel(xlabel)
                t.show_ylabel(ylabel)
                t.show_title(title) unless title == nil
                n = xs_ary.length
                n.times { |i|
                    if line_widths == nil
                        line_width = 0.7
                    else
                        line_width = line_widths[i]
                    end
                    t.line_width = line_width
                    if line_types == nil
                        line_type = Line_Type_Solid
                    else
                        line_type = line_types[i]
                    end
                    if line_colors == nil
                        color = Black
                    else
                        color = line_colors[i]
                    end
                    stroke_track(xs_ary[i], ys_ary[i], color, nil, line_type, 0) 
                    eval(sprintf("%s(%i, %i)", decorate_line, i, n)) unless decorate_line == nil
                }
            end
        end
    end




end
