! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
 
      module mod_other_mixing
      
      ! you can add your own mixing routine for use in addition to the standard ones.
      ! don't edit this file -- instead copy null_other_mixing from below and edit it.
      ! it goes in your src/run_star_extras file.
      ! edit the extras_controls routine to set s% other_mixing
      ! e.g.,
         !    s% other_mixing => my_mixing_routine

         
         
      ! NOTE: if you'd like to have some inlist controls for your routine,
      ! you can use the x_ctrl array of real(dp) variables that is in &controls
      ! e.g., in the &controls inlist, you can set
      !     x_ctrl(1) = my_special_param
      ! then in your routine, you can access that by
      !     s% x_ctrl(1)
      ! of course before you can use s, you need to get it using the id argument.
      ! here's an example of how to do that -- add these lines at the start of your routine:
      !         use star_lib, only: star_ptr
      !         type (star_info), pointer :: s
      !         call star_ptr(id, s, ierr)
      !         if (ierr /= 0) then ! OOPS
      !            return
      !         end if
      ! 
      ! for integer control values, you can use x_integer_ctrl
      ! for logical control values, you can use x_logical_ctrl




      use star_def
      use alert_lib
      use const_def
      use mlt_def

      implicit none
      
      
      real(dp), parameter :: pavel_delta_lgT = 0.15d0, pavel_f_mix = 0.01  

      
      
      contains
      
      
      subroutine null_other_mixing(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine null_other_mixing
      
      
      ! examples
                  
      
      subroutine pavel_mixing2(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr         
         integer :: k
         type (star_info), pointer :: s
         real(dp) :: a, gradmu_alt, gradr, grada, &
            dq00, dqm1, dqsum, T, kap, rho, cp, thermal_diffusivity    
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         a = s% x_ctrl(1)            
         do k = 2, s% nz ! start at 2 since no mixing across surface
            gradmu_alt = s% gradmu_alt(k) ! abar/(1 + zbar) -- assumes complete ionization
            if (gradmu_alt >= 0) cycle
            gradr = s% gradr(k)
            grada = s% grada_at_face(k)
            if (gradr >= grada) cycle
            ! interpolate values at face
            dq00 = s% dq(k)
            dqm1 = s% dq(k-1)
            dqsum = dq00 + dqm1
            kap = (dqm1*s% opacity(k) + dq00*s% opacity(k-1))/dqsum
            rho = (dqm1*s% rho(k) + dq00*s% rho(k-1))/dqsum
            cp = (dqm1*s% cp(k) + dq00*s% cp(k-1))/dqsum
            T = (dqm1*s% T(k) + dq00*s% T(k-1))/dqsum
            thermal_diffusivity = 4*crad*clight*T**3/(3*kap*rho**2*cp)
            ! gradmu_alt < 0 and gradr < grada
            s% D_mix(k) = 2*pi**2*(gradmu_alt/(gradr - grada))*thermal_diffusivity*a**2
            s% mixing_type(k) = thermo_haline_mixing
         end do         
      end subroutine pavel_mixing2
                  
      
      subroutine pavel_mixing1(id, ierr)
         ! this implements one of the RGB extra-mixing schemes used by Pavel Denissenkov

         use chem_def, only: i3alf, ih1, ihe4, ic12
         use mlt_def, only: convective_mixing, overshoot_mixing
         use rates_def, only: i_rate
         
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         
         integer :: k_bce, k_b, k_mix, k, i, nz, h1, he4
         real(dp) :: lnT_mix, max_c12, L_He
         real(dp), parameter :: &
            he_min_qlimit = 0.2d0, max_c12_limit = 0.02d0, max_L_He = 0.1d0
         type (star_info), pointer :: s
            
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return

         nz = s% nz
         h1 = s% net_iso(ih1)
         he4 = s% net_iso(ihe4)
         
         ! have we gone beyond the RGB?
         max_c12 = maxval(s% xa(s% net_iso(ic12),:))
         if (max_c12 > max_c12_limit) then
            if (dbg) write(*,1) 'pavel_mixing1: max_c12 > max_c12_limit', max_c12, max_c12_limit
            return
         end if
         
         ! is the He flash starting?
         L_He = (s% mstar/Lsun)*dot_product(s% dq(:), s% eps_nuc_categories(i_rate, i3alf, :))
         if (L_He > max_L_He) then
            if (dbg) write(*,1) 'pavel_mixing1: L_He > max_L_He', L_He, max_L_He
            return
         end if
         
         ! find the bottom of the convective envelope (k_bce)
         k_bce = 0
         do i = 1, s% num_conv_boundaries
            if (s% top_conv_bdy(i)) cycle
            k = s% conv_bdy_loc(i)
            if (k_bce == 0) then
               k_bce = k
            else if (s% q(k) < s% q(k_bce)) then
               k_bce = k
            end if
         end do
         if (k_bce == 0) then
            if (dbg) write(*,1) 'pavel_mixing1: no convective zone'
            return
         end if

         if (s% mixing_type(k_bce) /= convective_mixing) then
            k_bce = k_bce-1
            if (s% mixing_type(k_bce) /= convective_mixing) then
               write(*,*) 'confusion in search for H burn convection zone'
               stop 1
            end if
         end if
         if (dbg) write(*,2) 'pavel_mixing1: k_bce', k_bce, s% r(k_bce)/Rsun
         
         ! find bottom of H burn shell (k_b)
         k_b = 0
         do k = k_bce, nz
            if (s% xa(h1,k) < 1d-4) then
               k_b = k; exit
            end if
         end do
         if (k_b == 0) then
            if (dbg) write(*,1) 'pavel_mixing1: no H burn shell'
            return
         end if
         if (dbg) write(*,2) 'pavel_mixing1: k_b', k_b, s% r(k_b)/Rsun
         
         lnT_mix = s% lnT(k_b) - pavel_delta_lgT*ln10
         ! find the mixing boundary (k_mix)
         k_mix = 0
         do k = k_b-1, k_bce, -1
            if (s% lnT(k) < lnT_mix) then
               k_mix = k; exit
            end if
         end do
         if (k_mix == 0) then
            if (dbg) write(*,1) 'pavel_mixing1: no mixing boundary'
            return
         end if
         if (dbg) write(*,2) 'pavel_mixing1: k_mix', k_mix, s% r(k_mix)/Rsun
         
         ! set mixing
!$OMP PARALLEL DO PRIVATE(k)
         do k = k_bce+1, nz
            call do1(k)
         end do
!$OMP END PARALLEL DO

         ! NOTE: need to fix this to update mixing_type_change_dq

         if (dbg) write(*,1) 'pavel_mixing1: finished'
         
         
         contains
         
         
         subroutine do1(k)
            integer, intent(in) :: k
            
            real(dp) :: X, T, r, rho, kappa, Cp, &
               nu_rad, nu_mol, nu, thermal_diffusivity, D
            
            X = s% xa(h1,k)
            T = s% T(k)
            r = s% r(k)
            rho = s% rho(k)
            kappa = s% opacity(k)
            Cp = s% Cp(k)
            
            nu_rad = 4*crad*T**4/(15*clight*kappa*rho**2) ! radiative viscosity
            nu_mol = 1.84d-17*(1+7*X)*T**2.5/rho ! molecular viscosity
            nu = nu_rad + nu_mol ! total viscosity
            
            if (k > k_mix) then
               D = nu
            else
               thermal_diffusivity = 4*crad*clight*T**3/(3*kappa*rho**2*Cp)
               D = nu + pavel_f_mix*thermal_diffusivity ! diffusion coefficient
            end if
            
            s% D_mix(k) = D ! cm^2/sec
            s% mixing_type(k) = overshoot_mixing
         
         end subroutine do1
         

      end subroutine pavel_mixing1
      

      end module mod_other_mixing
      
      
      
      
