! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module write_model

      use star_private_def
      use const_def
      use alert_lib
      use read_model

      implicit none
      

      contains
      
      
     
      subroutine do_write_model(id, filename, prev_flag_in, ierr)
         use utils_lib  
         use chem_def
         use read_model
         integer, intent(in) :: id
         character (len=*), intent(in) :: filename
         logical, intent(in) :: prev_flag_in
         integer, intent(out) :: ierr 
             
         integer :: iounit, i, nvar_hydro, nz, species, file_type
         integer, pointer :: chem_id(:)
         type (star_info), pointer :: s
         logical :: lnPgas_flag, lnE_flag, lnTdot_flag, lnddot_flag, v_flag, prev_flag, rotation_flag
         integer :: time_vals(8)
         character (len=10) :: date_str, time_str, zone_str
         
         1 format(a32, 2x, 1pe26.16)
         11 format(a32, 2x, 1pe26.16, 2x, a, 2x, 1pe26.16)
         2 format(a32, 2x, i30)
         3 format(a32, 3x, a8)
         4 format(a32, 3x, a)

         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            call alert(ierr, 'bad star id for write_model')
            return
         end if
         
         chem_id => s% chem_id
         nvar_hydro = s% nvar_hydro
         nz = s% nz
         lnPgas_flag = s% lnPgas_flag
         lnE_flag = s% lnE_flag
         lnTdot_flag = s% lnTdot_flag
         lnddot_flag = s% lnddot_flag
         v_flag = s% v_flag
         rotation_flag = s% rotation_flag
         species = s% species
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         open(unit=iounit, file=trim(filename), action='write', status='replace')
         write(iounit,'(a)') '! note: initial lines of file can contain comments'
         write(iounit,'(a)') '!'
         prev_flag = (prev_flag_in .and. s% nz_old == s% nz)
         file_type = 0
         if (lnPgas_flag) file_type = file_type + 2**bit_for_lnPgas
         if (lnE_flag) file_type = file_type + 2**bit_for_lnE
         if (lnTdot_flag) file_type = file_type + 2**bit_for_lnTdot
         if (lnddot_flag) file_type = file_type + 2**bit_for_lnddot
         if (prev_flag) file_type = file_type + 2**bit_for_2models
         if (v_flag) file_type = file_type + 2**bit_for_velocity
         if (rotation_flag) file_type = file_type + 2**bit_for_rotation
         write(iounit, '(i14)', advance='no') file_type
         if (BTEST(file_type, bit_for_lnPgas)) &
            write(iounit,'(a)',advance='no') ', with lnPgas variables'
         if (BTEST(file_type, bit_for_lnE)) &
            write(iounit,'(a)',advance='no') ', with lnE variables'
         if (BTEST(file_type, bit_for_lnTdot)) &
            write(iounit,'(a)',advance='no') ', with lnTdot variables'
         if (BTEST(file_type, bit_for_lnddot)) &
            write(iounit,'(a)',advance='no') ', with lnddot variables'
         if (BTEST(file_type, bit_for_2models)) then
            write(iounit,'(a)',advance='no') ' -- two consecutive models for mesa/star'
         else
            write(iounit,'(a)',advance='no') ' -- single model for mesa/star'
         end if
         if (BTEST(file_type, bit_for_velocity)) &
            write(iounit,'(a)',advance='no') ', with velocity variables (v)'
         if (BTEST(file_type, bit_for_rotation)) &
            write(iounit,'(a)',advance='no') ', with angular velocity variables (omega)'
         write(iounit,'(a)',advance='no') &
            '. cgs units. lnd=ln(density), lnT=ln(temperature), lnR=ln(radius), L=luminosity'
         write(iounit,'(a)',advance='no') &
            ', dq=fraction of total mstar in cell; remaining cols are mass fractions.'
         write(iounit,*)
         call date_and_time(date_str, time_str, zone_str, time_vals)
         ! write property list
         write(iounit, *) ! blank line before start of property list
         write(iounit, 3) 'year_month_day_when_created', date_str(1:8)
         write(iounit, 1) 'M/Msun', s% star_mass
         write(iounit, 2) 'model_number', s% model_number
         write(iounit, 1) 'star_age', s% star_age
         write(iounit, 1) 'initial_z', s% initial_z
         write(iounit, 2) 'n_shells', nz
         write(iounit, 4) 'net_name', "'" // trim(s% net_name) // "'"
         write(iounit, 2) 'species', species
         if (s% M_center /= 0) write(iounit, 11) 'xmstar', s% xmstar, &
            '! mass above core (g).  mass of core:', s% M_center
         if (s% R_center /= 0) write(iounit, 11) 'R_center', s% R_center, &
            '! radius of core (cm).  avg core density (g/cm^3):', &
            s% M_center/(4*pi/3*s% R_center**3)
         if (s% L_center /= 0) write(iounit, 11) 'L_center', s% L_center, &
            '! luminosity of core (erg/s).  avg core eps (erg/g/s):', &
            s% L_center/max(1d0,s% M_center)
         if (s% tau_factor /= 1) write(iounit, 1) 'tau_factor', s% tau_factor
         write(iounit, *) ! blank line for end of property list

         call header
         call write_info(s% nz, s% xh, s% xa, s% dq)
         if (prev_flag) then         
            write(iounit, *)
            write(iounit, '(a)') '        previous model'
            ! write prev properties
            write(iounit, *)
            write(iounit, 2) 'previous n_shells', s% nz_old
            write(iounit, 1) 'previous mass (grams)', s% mstar_old
            write(iounit, 1) 'timestep (seconds)', s% dt
            write(iounit, 1) 'dt_next (seconds)', s% dt_next
            write(iounit, *)
            call header
            call write_info(s% nz_old, s% xh_old, s% xa_old, s% dq_old)
         end if
         close(iounit)
         call free_iounit(iounit)   
         
         contains
         
         subroutine header
            write(iounit, fmt='(10x, a9, 1x, 99(a26, 1x))', advance='no') &
                  'lnd', 'lnT', 'lnR', 'L', 'dq'
            if (lnTdot_flag) write(iounit, fmt='(a26, 1x)', advance='no') 'lnTdot'
            if (lnddot_flag) write(iounit, fmt='(a26, 1x)', advance='no') 'lnddot'
            if (lnPgas_flag) write(iounit, fmt='(a26, 1x)', advance='no') 'lnPgas'
            if (lnE_flag) write(iounit, fmt='(a26, 1x)', advance='no') 'lnE'
            if (v_flag) write(iounit, fmt='(a26, 1x)', advance='no') 'v'
            if (rotation_flag) write(iounit, fmt='(a26, 1x)', advance='no') 'omega'
            do i=1, species
               write(iounit, fmt='(a26, 1x)', advance='no') chem_isos% name(chem_id(i))
            end do
            write(iounit, *)
         end subroutine header
         
         subroutine write_info(n, xh, xa, dq)
            integer, intent(in) :: n
            real(dp), intent(in) :: xh(nvar_hydro, n), xa(species, n), dq(n)
            integer :: k
            do k=1, n
               write(iounit, fmt='(i5, 1x, 99(1pe26.16, 1x))', advance='no') k,  &
                  s% lnd(k), s% lnT(k), s% lnR(k), s% L(k), dq(k)
               if (lnTdot_flag) &
                  write(iounit, fmt='(1pe26.16, 1x)', advance='no') s% lnTdot(k)
               if (lnddot_flag) &
                  write(iounit, fmt='(1pe26.16, 1x)', advance='no') s% lnddot(k)
               if (lnPgas_flag) &
                  write(iounit, fmt='(1pe26.16, 1x)', advance='no') s% lnPgas(k)
               if (lnE_flag) &
                  write(iounit, fmt='(1pe26.16, 1x)', advance='no') s% lnE(k)
               if (v_flag) &
                  write(iounit, fmt='(1pe26.16, 1x)', advance='no') s% velocity(k)
               if (rotation_flag) &
                  write(iounit, fmt='(1pe26.16, 1x)', advance='no') s% omega(k)
               do i=1, species
                  write(iounit, fmt='(1pe26.16, 1x)', advance='no') xa(i, k)
               end do
               write(iounit, *)
            end do
         end subroutine write_info
         
      end subroutine do_write_model
      
      
                     

      end module write_model
