! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module winds

      use star_private_def
      use alert_lib
      use const_def
      use chem_def, only: ih1, ihe4, ic12, ic13, in14, io16

      implicit none


      contains
      
      
      
      ! NOTE: don't assume that vars are all set when doing winds.
      ! use values from s% xh(:,:) and s% xa(:,:) only.
      ! e.g., must use s% xh(s% i_lnT,1) instead of s% lnT(1)
      
      
      subroutine eval_super_eddington_wind(s, L, M, R, ierr)
         ! determine mdot for super eddington wind
         use eos_def, only: i_lnfree_e
         type (star_info), pointer :: s
         real(dp), intent(in) :: L, M, R
         integer, intent(out) :: ierr

         real(dp) :: Ledd, Leff, vesc2, wind_vel, kap
            
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0
         s% super_eddington_wind_mdot = 0 
         
         if (s% super_eddington_wind_eta <= 0) return 

         kap = s% surf_opacity
         Ledd = s% prev_Ledd ! NOTE: Ledd for this is averaged over mass to tau = 100
         Leff = L/s% super_eddington_wind_Ledd_factor
         if (Leff <= Ledd) then
            if (.false. .or. Leff > 0.95d0*Ledd) then
               write(*,'(a40,99f20.5)') 'L/Ledd', L/Ledd, L/Lsun, Ledd/Lsun
            end if
            return
         end if
         vesc2 = s% cgrav(1)*M/R
         s% super_eddington_wind_mdot = s% super_eddington_wind_eta*(Leff - Ledd)/vesc2
         !wind_vel = 1d-5*sqrt(2*vesc2)  !1d-5*sqrt(2*(Leff-Ledd)/s% super_eddington_wind_mdot)
         !if (dbg) write(*,1) 'wind Leff/Ledd, lg mdot, v(km/sec)', Leff/Ledd, &
         !   log10(s% super_eddington_wind_mdot/(Msun/secyer)), wind_vel
         
         write(*,'(a60,i12,992f12.5)') 'super eddington wind: model number, L/Ledd, lg_Mdot, M', &
            s% model_number, L/Ledd, log10(s% super_eddington_wind_mdot/(Msun/secyer)), M/Msun
         
      end subroutine eval_super_eddington_wind
      
      
      ! NOTE: don't assume that vars are all set at this point.
      ! use values from s% xh(:,:) and s% xa(:,:) only.
      real(dp) function eval_nova_wind(s, L_surf, R, Teff, ierr) ! value in Msun/year
         type (star_info), pointer :: s
         real(dp), intent(in) :: L_surf, R, Teff ! Lsun, Rsun, K
         integer, intent(out) :: ierr   
         real(dp) :: roche_lobe_radius ! Rsun    
         real(dp) :: mdot
         include 'formats.dek'
         ierr = 0
         eval_nova_wind = 0
         if (s% nova_wind_eta <= 0) return
         if (L_surf < s% nova_wind_min_L) return
         if (Teff > s% nova_wind_max_Teff) return
         
         roche_lobe_radius = s% nova_roche_lobe_radius
         if (R >= roche_lobe_radius) then
            mdot = s% nova_RLO_mdot*Msun/secyer ! in gm per second
            write(*,1) 'nova RLO log mdot', log10(s% nova_wind_eta*mdot/(Msun/secyer))
         else ! eqn 23 of Kato and Hachisu, ApJ 437:802-826, 1994
            mdot = 10d0**(1.49d0*log10(Teff/1d5) + s% nova_wind_b) ! in gm per second
            write(*,1) 'nova wind log mdot', log10(s% nova_wind_eta*mdot/(Msun/secyer))
         end if
                
         eval_nova_wind = s% nova_wind_eta*mdot/(Msun/secyer)
         
      end function eval_nova_wind
      
      
      ! NOTE: don't assume that vars are all set at this point.
      ! use values from s% xh(:,:) and s% xa(:,:) only.
      real(dp) function eval_rlo_wind(s, L_surf, R, Teff, ierr) ! value in Msun/year
         type (star_info), pointer :: s
         real(dp), intent(in) :: L_surf, R, Teff ! Lsun, Rsun, K
         integer, intent(out) :: ierr   
         real(dp) :: roche_lobe_radius ! Rsun    
         real(dp) :: rho, p, grav, hp, v_th, h, rho_exponent, rho_rl, rho_rl0, mdot
         include 'formats.dek'
         ierr = 0
         eval_rlo_wind = 0
         if (s% rlo_wind_eta <= 0) return
         if (L_surf < s% rlo_wind_min_L) return
         if (Teff > s% rlo_wind_max_Teff) return
         roche_lobe_radius = s% rlo_wind_roche_lobe_radius
         if (R < roche_lobe_radius .or. s% rlo_wind_scale_height <= 0) then
            return
         end if
         mdot = s% rlo_wind_base_mdot*exp((R - roche_lobe_radius)/s% rlo_wind_scale_height)
         eval_rlo_wind = s% rlo_wind_eta*mdot ! Msun/year
         write(*,1) 'log rlo wind, dR/H', log10(eval_rlo_wind), &
            (R - roche_lobe_radius)/s% rlo_wind_scale_height
      end function eval_rlo_wind
      
      
      ! NOTE: don't assume that vars are all set at this point.
      ! use values from s% xh(:,:) and s% xa(:,:) only.
      subroutine eval_supersonic_wind(s, is_infalling, ierr)
         ! determine mdot for supersonic wind
         use utils_lib
         use micro, only: eval_csound_and_rho
         use num_lib
         type (star_info), pointer :: s
         logical, intent(out) :: is_infalling
         integer, intent(out) :: ierr

         integer :: k, base, max_base
         real(dp) :: v, csound, mdot, max_eject_rate, max_mdot, &
            v_div_cs, v_div_cs_min, r, rho
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0
         is_infalling = .false.
         s% supersonic_wind_ejection_mass = 0 
         s% supersonic_wind_mdot = 0 
         s% supersonic_wind_base_cell = 0
         if (s% supersonic_wind_eta == 0 .or. .not. s% v_flag) then
            !if (dbg) write(*,*) '(s% supersonic_wind_eta == 0 .or. .not. s% v_flag)'
            return
         end if
         
         max_base = s% nz / 5
         base = 0
         v_div_cs_min = 0
         do k = 1, max_base
            v = s% xh(s% i_vel, k)
            if (s% eject_supersonic_infall_flag .and. v < 0) then
               is_infalling = .true.; v = -v
            end if
            if (k == 1) then
               csound = s% surf_csound
               rho = s% surf_rho
            else
               csound = eval_csound_and_rho(s, k, rho, ierr)
               if (ierr /= 0) then
                  if (dbg) write(*,2) 'eval_csound eval_csound_and_rho', k
                  return
               end if
            end if
            v_div_cs = v/csound
            if (dbg .and. (k > 1 .or. v_div_cs > 1d-1)) write(*,2) 'v_div_cs', k, v_div_cs
            if (v_div_cs < 1) exit
            if (k == 1) then
               v_div_cs_min = v_div_cs
               if (dbg) write(*,2) 'v_div_cs/v_div_cs_min', k, v_div_cs/v_div_cs_min, &
                  v_div_cs, v_div_cs_min
            else if (0.90d0*v_div_cs > v_div_cs_min) then
               if (dbg) write(*,2) 'v_div_cs/v_div_cs_min', k, v_div_cs/v_div_cs_min, &
                  v_div_cs, v_div_cs_min
               exit
            end if
            base = k
         end do
         if (base == 0) return
         if (base == 1) then
            s% supersonic_wind_ejection_mass = s% dq(1)*s% xmstar
            r = exp(s% xh(s% i_lnR,1))
            v = s% xh(s% i_vel,1)
            k = 1
         else
            s% supersonic_wind_ejection_mass = sum(s% dq(1:base-1))*s% xmstar
            r = exp(s% xh(s% i_lnR,base))
            v = s% xh(s% i_vel,base)
            k = base-1
         end if
         if (s% lnPgas_flag) then ! get rho for k
            csound = eval_csound_and_rho(s, k, rho, ierr)
            if (ierr /= 0) then
               if (dbg) write(*,2) 'eval_csound eval_csound_and_rho', k
               return
            end if
         else
            rho = exp(s% xh(s% i_xlnd,k) - lnd_offset)
         end if
         mdot = 4*pi*r**2*rho*abs(v)

         mdot = mdot*s% supersonic_wind_eta
         if (dbg) write(*,1) 'ss mdot', mdot
         max_eject_rate = s% max_supersonic_wind_mdot
         max_mdot = max_eject_rate*Msun/secyer
         if (dbg) write(*,1) 'ss max_mdot', max_mdot, max_eject_rate
         s% supersonic_wind_mdot = min(mdot, max_mdot)
         if (dbg) write(*,1) 'supersonic_wind_mdot', s% supersonic_wind_mdot
         s% supersonic_wind_base_cell = base
         
         if (dbg) write(*,3) 'ss_wind model_number, base', s% model_number, base
         if (dbg) write(*,1) 'ss wind ejection mass/Msun', s% supersonic_wind_ejection_mass/Msun
         if (dbg) write(*,1) 'supersonic_wind_eta', s% supersonic_wind_eta
         write(*,*)
      
      end subroutine eval_supersonic_wind
      
      
      ! NOTE: don't assume that vars are all set at this point.
      ! use values from s% xh(:,:) only.
      ! do_winds can reduce s% dt if necessary to limit mass loss per step
      subroutine do_winds(s, L_phot, M_phot, T_phot, ierr)
         use utils_lib, only: is_bad_num
         use chem_def
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         real(dp), intent(in) :: L_phot, M_phot, T_phot ! photosphere values (cgs)
         integer, intent(out) :: ierr
         
         integer :: k, j, h1, he4, nz, base
         real(dp) :: max_ejection_mass, wind_mdot, env_mass, wind, alfa, total_H, &
            X, Y, Z, w1, w2, T_high, T_low, L1, M1, R1, T1, &
            center_h1, center_he4, surface_h1, surface_he4, mdot
         character (len=256) :: message
         character (len=32) :: scheme
         logical :: is_infalling, using_wind_scheme_mdot
         real(dp), parameter :: Zsolar = 0.019d0 ! for Vink et al formula

         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         if (dbg) write(*,1) 'enter do_winds mass_change', s% mass_change
         
         ierr = 0
         L1 = L_phot
         M1 = M_phot
         T1 = T_phot
         R1 = sqrt(L1/(pi*crad*clight*T1**4))
         h1 = s% net_iso(ih1)
         he4 = s% net_iso(ihe4)
         nz = s% nz
         wind_mdot = 0
         using_wind_scheme_mdot = .false.
         
         call eval_supersonic_wind(s, is_infalling, ierr)
         if (ierr /= 0) then
            if (dbg .or. s% report_ierr) write(*, *) 'do_winds: eval_supersonic_wind ierr'
            return
         end if
         
         if (s% supersonic_wind_ejection_mass /= 0) then
            ! limit mstar_dot according to the current timestep
            ! so that don't remove more than desired.
            wind_mdot = s% supersonic_wind_mdot
            max_ejection_mass = s% supersonic_wind_ejection_mass
            if (wind_mdot*s% dt > max_ejection_mass) wind_mdot = max_ejection_mass/s% dt
            if (is_infalling) then
               write(*,'(a)') 'artificial wind to remove supersonically infalling envelope'
            else
               write(*,'(a)') 'supersonic wind'
            end if
         end if
         
         call eval_super_eddington_wind(s, L1, M1, R1, ierr)
         if (ierr /= 0) then
            if (dbg .or. s% report_ierr) write(*, *) 'do_winds: eval_super_eddington_wind ierr'
            return
         end if
         
         if (s% super_eddington_wind_mdot > wind_mdot) then
            wind_mdot = s% super_eddington_wind_mdot
            if (dbg) write(*,1) 'super eddington wind mdot', &
               log10(s% super_eddington_wind_mdot/(Msun/secyer))


            if (.false.) then
               s% mstar_dot = -s% super_eddington_wind_mdot
               if (dbg) write(*,1) 'return super eddington wind mdot', &
                  log10(s% super_eddington_wind_mdot/(Msun/secyer))
               if (dbg) write(*,*)
               return
            end if


         end if
         
         if (dbg) write(*,1) 's% remove_H_wind_mdot', s% remove_H_wind_mdot
         if (s% remove_H_wind_mdot > 0 .and. h1 > 0) then
            total_H = dot_product(s% xa(h1,1:nz), s% dm(1:nz))/Msun
            write(*,1) 'total_H', total_H
            if (dbg) write(*,1) 's% remove_H_wind_H_mass_limit', s% remove_H_wind_H_mass_limit
            if (total_H > s% remove_H_wind_H_mass_limit) then
               if (s% remove_H_wind_mdot*Msun/secyer > wind_mdot) then
                  wind_mdot = s% remove_H_wind_mdot*Msun/secyer
                  if (dbg) write(*,1) 'use remove H wind mdot', &
                     log10(s% remove_H_wind_mdot/(Msun/secyer))
               end if
            end if
         end if
            
         if (s% flash_wind_mdot > 0) then
            if (s% doing_flash_wind) then
               if (R1/Rsun <= s% flash_wind_full_off) then
                  s% doing_flash_wind = .false.
                  write(*,*) 's% doing_flash_wind = .false.'
               end if
            else if (R1/Rsun >= s% flash_wind_starts) then
               s% doing_flash_wind = .true.
               write(*,*) 's% doing_flash_wind = .true.'
            end if
            if (s% doing_flash_wind .and. s% flash_wind_mdot*Msun/secyer > wind_mdot) then
               if (R1/Rsun < s% flash_wind_declines) then
                  s% flash_wind_mdot = s% flash_wind_mdot * &
                     (s% flash_wind_declines - R1/Rsun) / &
                        (s% flash_wind_declines - s% flash_wind_full_off)
               else if (R1/Rsun > s% flash_wind_starts) then
                  s% flash_wind_mdot = s% flash_wind_mdot*(2d0**((R1/Rsun)/s% flash_wind_starts - 1d0))
               end if
               if (s% flash_wind_mdot*Msun/secyer > wind_mdot) then
                  wind_mdot = s% flash_wind_mdot*Msun/secyer
                  write(*,1) 'flash wind r', R1/Rsun
               end if
            end if
         end if
         
         mdot = eval_rlo_wind(s, L1/Lsun, R1/Rsun, T1, ierr)
         if (ierr /= 0) then
            if (dbg .or. s% report_ierr) write(*, *) 'do_winds: eval_rlo_wind ierr'
            return
         end if
         s% doing_rlo_wind = (mdot /= 0)
         if (s% doing_rlo_wind .and. mdot > wind_mdot) then
            if (dbg) write(*,1) 's% doing_rlo_wind mdot', mdot
            wind_mdot = mdot*Msun/secyer
         end if
         
         if (s% nova_wind_eta /= 0 .and. L1/Lsun > s% nova_wind_min_L .and. &
               T1 < s% nova_min_Teff_for_accretion .and. s% mass_change > 0) then
            mdot = 0
         else
            mdot = eval_nova_wind(s, L1/Lsun, R1/Rsun, T1, ierr)
            if (ierr /= 0) then
               if (dbg .or. s% report_ierr) write(*, *) 'do_winds: eval_nova_wind ierr'
               return
            end if
         end if
         s% doing_nova_wind = (mdot /= 0)
         if (s% doing_nova_wind .and. mdot > wind_mdot) then
            if (dbg) write(*,1) 's% doing_nova_wind mdot', mdot
            wind_mdot = mdot*Msun/secyer
         end if
         
         if (h1 > 0) then
            center_h1 = s% xa(h1,nz)
            surface_h1 = s% xa(h1,1)
         else
            center_h1 = 0
            surface_h1 = 0
         end if
         if (he4 > 0) then
            center_he4 = s% xa(he4,nz)
            surface_he4 = s% xa(he4,1)
         else
            center_he4 = 0
            surface_he4 = 0
         end if
         
         if (center_h1 < 0.01d0 .and. center_he4 < s% RGB_to_AGB_wind_switch) then
            scheme = s% AGB_wind_scheme
            if (dbg) write(*,1) 'using AGB_wind_scheme: "' // trim(scheme) // '"', &
                        center_h1, center_he4, s% RGB_to_AGB_wind_switch
         else
            scheme = s% RGB_wind_scheme
            if (dbg) write(*,*) 'using RGB_wind_scheme: "' // trim(scheme) // '"'
         end if
                     
         if (len_trim(scheme) == 0) then
            if ((s% mass_change > 0 .and. wind_mdot == 0) .or. &
                (s% mass_change < 0 .and. -s% mass_change*Msun/secyer > wind_mdot)) then
               if (dbg) write(*,*) 'mass_change mdot', s% mass_change
               wind_mdot = -s% mass_change*Msun/secyer
            end if
         else
            env_mass = s% xmstar
            do k=1,nz
               if (s% xa(h1,k) <= s% h1_boundary_limit) then
                  env_mass = s% xmstar*(1 - s% q(k))
                  exit
               end if
            end do
            if (dbg) then
               write(*,1) 'env_mass', env_mass/Msun
               write(*,1) 's% wind_envelope_limit', s% wind_envelope_limit
            end if
            if (s% wind_envelope_limit > 0 .and. env_mass < s% wind_envelope_limit*Msun) then
               wind = 0
            else
               wind = 4d-13*(L1*R1/M1)/(Lsun*Rsun/Msun) ! in Msun/year
               if (dbg) write(*,1) 'wind', wind
               if (wind <= 0 .or. is_bad_num(wind)) then
                  ierr = -1
                  write(message,*) 'bad value for wind :', wind,L1,R1,M1
                  call alert(ierr, message)
                  if (dbg) stop 'debug: bad value for wind'
                  return
               end if
               X = surface_h1
               Y = surface_he4
               Z = 1 - (X + Y)
               if (scheme == 'Dutch') then
                  T_high = 11000
                  T_low = 10000
                  if (s% Dutch_wind_eta == 0 .and. (.not. dbg)) then
                     wind = 0
                  else if (T1 <= T_low) then
                     call eval_de_Jager_wind(wind)
                     if (dbg) write(*,1) 'Dutch_wind = de Jager', log10(wind), T1, T_low, T_high
                  else if (surface_h1 < 0.4d0) then ! helium rich Wolf-Rayet star: Nugis & Lamers
                     wind = 1d-11 * (L1/Lsun)**1.29d0 * Y**1.7d0 * Z**0.5d0
                     if (dbg) write(*,1) 'Dutch_wind = Nugis & Lamers', log10(wind)
                  else
                     if (T1 >= T_high) then
                        call eval_vink_wind(wind)
                     else ! transition
                        call eval_de_Jager_wind(w1)
                        call eval_vink_wind(w2)
                        alfa = (T1 - T_low)/(T_high - T_low)
                        wind = (1-alfa)*w1 + alfa*w2
                     end if
                  end if
                  wind = s% Dutch_wind_eta * wind
               else if (scheme == 'Reimers') then
                  wind = wind * s% Reimers_wind_eta
                  if (dbg) then
                     write(*,1) 's% Reimers_wind_eta', s% Reimers_wind_eta
                     write(*,1) 'Reimers_wind', wind
                     write(*,1) 'L1/Lsun', L1/Lsun
                     write(*,1) 'R1/Rsun', R1/Rsun
                     write(*,1) 'M1/Msun', M1/Msun
                     write(*,1) 'Reimers_wind_etaReimers_wind_eta', s% Reimers_wind_eta
                     write(*,1) 'wind', wind
                     write(*,1) 'log10 wind', log10(wind)
                     write(*,*)
                     stop 'debug: winds'
                  end if
               else if (scheme == 'Kudritzki') then
                  call eval_Kudritzki_wind(wind)
                  if (dbg) write(*,1) 'Kudritzki_wind', wind
               else if (scheme == 'Blocker') then
                  call eval_blocker_wind(wind)
                  if (dbg) write(*,1) 'Blocker_wind', wind
               else if (scheme == 'de Jager') then
                  call eval_de_Jager_wind(wind)
                  wind = s% de_Jager_wind_eta * wind
                  if (dbg) write(*,1) 'de_Jager_wind', wind
               else if (scheme == 'Nieuwenhuijzen') then
                  call eval_Nieuwenhuijzen_wind(wind)
                  wind = s% Nieuwenhuijzen_wind_eta * wind
                  if (dbg) write(*,1) 'Nieuwenhuijzen_wind', wind
               else if (scheme == 'other') then
                  if (dbg) write(*,*) 'call other_wind'
                  call s% other_wind(s% id, L1, M1, R1, T1, wind, ierr)
                  if (ierr /= 0) return
               else 
                  ierr = -1
                  write(message,*) 'unknown name for wind scheme : ' // trim(scheme)
                  call alert(ierr, message)
                  if (dbg) stop 'debug: bad value for wind scheme'
                  return
               end if
               
               if (wind*Msun/secyer > wind_mdot) then
                  using_wind_scheme_mdot = .true.
                  wind_mdot = wind*Msun/secyer
                  if (dbg) write(*,1) 'use wind scheme mdot', wind_mdot
               end if
               
            end if
         end if

         if (wind_mdot >= 0 .and. s% min_wind > 0 .and. &
               wind_mdot < s% min_wind*Msun/secyer) then
            if (dbg) write(*,1) 'use s% min_wind', s% min_wind
            wind_mdot = s% min_wind*Msun/secyer
         end if
         
         s% mstar_dot = -wind_mdot
         if (s% mstar_dot < 0 .and. using_wind_scheme_mdot) then
            call rotation_enhancement(ierr)
            if (ierr /= 0) then
               if (dbg .or. s% report_ierr) write(*, *) 'do_winds: rotation_enhancement ierr'
               return
            end if
            call limit_change_in_mdot
         end if         
         
         if (dbg) then
            write(*,1) 'final lg s% mstar_dot/(Msun/secyer)', safe_log10(abs(s% mstar_dot/(Msun/secyer)))
            write(*,*)
         end if
         


         contains
         
         
         subroutine limit_change_in_mdot
            real(dp) :: abs_mdot_prev, abs_mdot
            abs_mdot_prev = abs(s% mstar_dot_old)
            abs_mdot = abs(s% mstar_dot)
            if (abs_mdot <= s% min_abs_mdot_for_change_limits) return
            if (abs_mdot*abs_mdot_prev <= 0) return ! no limit when change sign
            abs_mdot = min(abs_mdot, abs_mdot_prev*s% max_abs_mdot_factor)
            abs_mdot = max(abs_mdot, abs_mdot_prev*s% min_abs_mdot_factor)
            s% mstar_dot = sign(abs_mdot, s% mstar_dot)
         end subroutine limit_change_in_mdot
         
         
         subroutine rotation_enhancement(ierr)
            use eos_def, only: i_lnfree_e
            
            integer, intent(out) :: ierr
            ! as in Heger, Langer, and Woosley, 2000, ApJ, 528:368-396.  section 2.6
            ! Mdot = Mdot_no_rotation/(1 - Osurf/Osurf_crit)^mdot_omega_power
            ! where Osurf = angular velocity at surface
            !       Osurf_crit^2 = (1 - Gamma_edd)*G*M/R_equitorial^3
            !       Gamma_edd = kappa*L/(4 pi c G M), Eddington factor
            real(dp) :: Ledd, v_div_v_crit, Gamma_edd, enhancement, wind_mdot, &
               kh_timescale, mdot_lim, mdot_prev, prev_rotational_mdot_boost, &
               dmsfac, dmskhf, wind_mdot_lim

            include 'formats.dek'
            
            ierr = 0
            if (.not. s% rotation_flag) return
            if (s% mdot_omega_power <= 0) return
            if (s% mstar_dot >= 0) return

            wind_mdot = -s% mstar_dot
            
            kh_timescale = s% cgrav(1)*M1**2/(2*R1*L1)
            dmskhf = s% rotational_mdot_kh_fac
            dmsfac = s% rotational_mdot_boost_fac
            wind_mdot_lim = min(dmskhf*M1/kh_timescale, wind_mdot*dmsfac)

            Ledd = s% prev_Ledd
            Gamma_edd = L1/Ledd
            v_div_v_crit = s% v_div_v_crit_surf
            
            enhancement = max(1d-22, 1d0 - v_div_v_crit)**(-s% mdot_omega_power)
            if (enhancement > s% lim_trace_rotational_mdot_boost) then
               write(*,1) 'mdot rotation_enhancement factor', &
                  enhancement, Gamma_edd, v_div_v_crit
            end if
            if (s% max_rotational_mdot_boost > 0 .and. &
                  enhancement > s% max_rotational_mdot_boost) then
               enhancement = s% max_rotational_mdot_boost
               !write(*,1) 'reduce to max_rotational_mdot_boost', &
               !      s% max_rotational_mdot_boost
            end if
            
            wind_mdot = min(wind_mdot_lim, wind_mdot*enhancement)
            s% mstar_dot = -wind_mdot
            s% rotational_mdot_boost = enhancement

         end subroutine rotation_enhancement
         
         
         subroutine eval_vink_wind(w)
            real(dp), intent(inout) :: w
            real(dp) :: alfa, w1, w2, Teff_jump, logMdot, dT, vinf_div_vesc

            ! alfa = 1 for hot side, = 0 for cool side
            if (T1 > 27500d0) then
               alfa = 1
            else if (T1 < 22500d0) then
               alfa = 0
            else ! use Vink et al 2001, eqns 14 and 15 to set "jump" temperature
               Teff_jump = 1d3*(61.2d0 + 2.59d0*(-13.636d0 + 0.889d0*log10(Z/Zsolar)))
               dT = 100d0
               if (T1 > Teff_jump + dT) then
                  alfa = 1
               else if (T1 < Teff_jump - dT) then
                  alfa = 0
               else
                  alfa = (T1 - (Teff_jump - dT)) / (2*dT)
               end if
            end if
            
            if (alfa > 0) then ! eval hot side wind (eqn 24)
               vinf_div_vesc = 2.6d0 ! this is the hot side galactic value
               vinf_div_vesc = vinf_div_vesc*(Z/Zsolar)**0.13d0 ! corrected for Z
               logMdot = &
                  - 6.697d0 &
                  + 2.194d0*log10(L1/Lsun/1d5) &
                  - 1.313d0*log10(M1/Msun/30) &
                  - 1.226d0*log10(vinf_div_vesc/2d0) &
                  + 0.933d0*log10(T1/4d4) &
                  - 10.92d0*log10(T1/4d4)**2 &
                  + 0.85d0*log10(Z/Zsolar)
               w1 = 10**logMdot
            else
               w1 = 0
            end if
            
            if (alfa < 1) then ! eval cool side wind (eqn 25)
               vinf_div_vesc = 1.3d0 ! this is the cool side galactic value
               vinf_div_vesc = vinf_div_vesc*(Z/Zsolar)**0.13d0 ! corrected for Z
               logMdot = &
                  - 6.688d0 &
                  + 2.210d0*log10(L1/Lsun/1d5) &
                  - 1.339d0*log10(M1/Msun/30) &
                  - 1.601d0*log10(vinf_div_vesc/2d0) &
                  + 1.07d0*log10(T1/2d4) &
                  + 0.85d0*log10(Z/Zsolar)
               w2 = 10**logMdot
            else
               w2 = 0
            end if
            
            w = alfa*w1 + (1 - alfa)*w2
            
            if (dbg) write(*,*) 'vink wind', w
            
         end subroutine eval_vink_wind
         
         
         subroutine eval_Kudritzki_wind(w)
            real(dp), intent(inout) :: w
            real(dp) :: xlogl, xlteff, stmass, xsurf, xmdfic, vinfkm, zstar
            xlogl = log10(L1/Lsun)
            xlteff = log10(T1)
            stmass = M1/Msun
            xsurf = surface_h1
            zstar = -1d99 ! not used
            ! output xmdfic = mass loss rate [Msun/yr]
            ! output vinfkm = v_infinity
            call kuma(xlogl,xlteff,stmass,xsurf,xmdfic,vinfkm,zstar)
            w = xmdfic*s% Kudritzki_wind_eta
            if (dbg) write(*,*) 'lg eval_Kudritzki_wind', log10(w)
         end subroutine eval_Kudritzki_wind

         
         subroutine eval_blocker_wind(w)
            real(dp), intent(inout) :: w
            w = w * s% Blocker_wind_eta * &
               4.83d-9 * ((M1/Msun)**(-2.1d0)) * ((L1/Lsun)**2.7d0)
            if (dbg) write(*,*) 'blocker wind', w
         end subroutine eval_blocker_wind
         
         
         subroutine eval_de_Jager_wind(w)
            ! de Jager, C., Nieuwenhuijzen, H., & van der Hucht, K. A. 1988, A&AS, 72, 259.
            real(dp), intent(out) :: w
            real(dp) :: log10w
            include 'formats.dek'
            log10w = 1.769d0*log10(L1/Lsun) - 1.676d0*log10(T1) - 8.158d0
            w = 10d0**log10w
            if (dbg) then
               write(*,1) 'de_Jager log10(wind)', log10w
            end if
         end subroutine eval_de_Jager_wind
         
         
         subroutine eval_Nieuwenhuijzen_wind(w)
            ! Nieuwenhuijzen, H.; de Jager, C. 1990, A&A, 231, 134 (eqn 2)
            real(dp), intent(out) :: w
            real(dp) :: log10w
            include 'formats.dek'
            log10w = -14.02d0 + &
                     1.24d0*log10(L1/Lsun) + &
                     0.16d0*log10(M1/Msun) + &
                     0.81d0*log10(R1/Rsun)
            w = 10d0**log10w
            if (dbg) then
               write(*,1) 'Nieuwenhuijzen log10(wind)', log10w
            end if
         end subroutine eval_Nieuwenhuijzen_wind
         
         
         
         
         
      end subroutine do_winds




      end module winds
