! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module struct_burn_mix

      use star_private_def
      use alert_lib
      use const_def

      implicit none

      contains


      integer function do_struct_burn_mix(s, skip_global_corr_coeff_limit) 
         use solve_hydro, only: do_hydro_converge
         use utils_lib, only:is_bad_num
         use hydro_vars, only: check_rs, set_vars
         use rates_def, only: num_rvs
         use chem_def, only: num_categories, ih1
         use report, only: sum_Ebinding
         use solve_burn, only: do_solve_burn
         
         type (star_info), pointer :: s
         logical, intent(in) :: skip_global_corr_coeff_limit
         
         integer :: ierr, pass1, pass2, pass3, pass4
         logical :: trace
         real(dp) :: dt
         integer, parameter :: struct_hydro = 2**struct_hydro_bit_pos
         integer, parameter :: struct_thermo = 2**struct_thermo_bit_pos
         integer, parameter :: burn = 2**burn_bit_pos
         integer, parameter :: mix = 2**mix_bit_pos
         logical, parameter :: check = .false. 
         logical, parameter :: dbg = .false. 
         include 'formats.dek'
         
         ierr = 0
         
         do_struct_burn_mix = keep_going
         trace = s% trace_evolve
         s% op_split_diff = 0
         dt = s% dt
         
         if (check) then
            call check_rs(s, ierr)
            if (ierr /= 0) then
               do_struct_burn_mix = retry
               return
            end if
         end if
         
         pass2 = 0; pass3 = 0; pass4 = 0
         select case(s% operator_coupling_choice)

            case (-7)
               if (trace) write(*,*) 'doing operator_coupling_choice = -7. (mb)_s'
               call do_op_split(s, skip_global_corr_coeff_limit, dt, 0d0, .false., ierr)
               if (ierr /= 0) do_struct_burn_mix = retry
               return
            case (-6)
               if (trace) write(*,*) 'doing operator_coupling_choice = -6. s_(mb)'
               call do_op_split(s, skip_global_corr_coeff_limit, 0d0, dt, .false., ierr)
               if (ierr /= 0) do_struct_burn_mix = retry
               return
            case (-5)
               if (trace) write(*,*) 'doing operator_coupling_choice = -5. (mb)_s_(mb)'
               call do_op_split(s, skip_global_corr_coeff_limit, dt/2, dt/2, .false., ierr)
               if (ierr /= 0) do_struct_burn_mix = retry
               return

            case (-4)
               if (trace) write(*,*) 'doing operator_coupling_choice = -4. mbm_s'
               call do_op_split(s, skip_global_corr_coeff_limit, dt, 0d0, .true., ierr)
               if (ierr /= 0) do_struct_burn_mix = retry
               return
            case (-3)
               if (trace) write(*,*) 'doing operator_coupling_choice = -3. s_mbm'
               call do_op_split(s, skip_global_corr_coeff_limit, 0d0, dt, .true., ierr)
               if (ierr /= 0) do_struct_burn_mix = retry
               return
            case (-2)
               if (trace) write(*,*) 'doing operator_coupling_choice = -2. mbm_s_mbm'
               call do_op_split(s, skip_global_corr_coeff_limit, dt/2, dt/2, .true., ierr)
               if (ierr /= 0) do_struct_burn_mix = retry
               return
               
            case (-1) ! -1 -- experimental
               stop 'no -1 option'
               if (trace) write(*,*) 'doing operator_coupling_choice = -1'
               pass1 = struct_hydro + struct_thermo
               
            case (0) ! 0 -- fully coupled
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = struct_hydro + struct_thermo + burn + mix
               else
                  pass1 = struct_hydro
                  pass2 = struct_thermo + burn + mix
               end if
            case (1) ! 1 -- burn, then struct + mix
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = burn
                  pass2 = struct_hydro + struct_thermo + mix
               else
                  pass1 = burn
                  pass2 = struct_hydro
                  pass3 = struct_thermo + mix
               end if
            case (2) ! 2 -- mix, then struct + burn
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = mix
                  pass2 = struct_hydro + struct_thermo + burn
               else
                  pass1 = mix
                  pass2 = struct_hydro
                  pass3 = struct_thermo + burn
               end if
            case (3) ! 3 -- burn + mix, then struct
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = burn + mix
                  pass2 = struct_hydro + struct_thermo
               else
                  pass1 = burn + mix
                  pass2 = struct_hydro
                  pass3 = struct_thermo
               end if
            case (4) ! 4 -- struct + mix, then burn
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = struct_hydro + struct_thermo + mix
                  pass2 = burn
               else
                  pass1 = struct_hydro
                  pass2 = struct_thermo + mix
                  pass3 = burn
               end if
            case (5) ! 5 -- struct + burn, then mix
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = struct_hydro + struct_thermo + burn
                  pass2 = mix
               else
                  pass1 = struct_hydro
                  pass2 = struct_thermo + burn
                  pass3 = mix
               end if
            case (6) ! 6 -- struct, then burn + mix
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = struct_hydro + struct_thermo
                  pass2 = burn + mix
               else
                  pass1 = struct_hydro
                  pass2 = struct_thermo
                  pass3 = burn + mix
               end if
            case (7) !  7 -- struct, then burn, then mix
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = struct_hydro + struct_thermo
                  pass2 = burn
                  pass3 = mix
               else
                  pass1 = struct_hydro
                  pass2 = struct_thermo
                  pass3 = burn
                  pass4 = mix
               end if
            case (8) !  8 -- struct, then mix, then burn
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = struct_hydro + struct_thermo
                  pass2 = mix
                  pass3 = burn
               else
                  pass1 = struct_hydro
                  pass2 = struct_thermo
                  pass3 = mix
                  pass4 = burn
               end if
            case (9) !  9 -- burn, then struct, then mix
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = burn
                  pass2 = struct_hydro + struct_thermo
                  pass3 = mix
               else
                  pass1 = burn
                  pass2 = struct_hydro
                  pass3 = struct_thermo
                  pass4 = mix
               end if
            case (10) ! 10 -- burn, then mix, then struct
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = burn
                  pass2 = mix
                  pass3 = struct_hydro + struct_thermo
               else
                  pass1 = burn
                  pass2 = mix
                  pass3 = struct_hydro
                  pass4 = struct_thermo
               end if
            case (11) ! 11 -- mix, then struct, then burn
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = mix
                  pass2 = struct_hydro + struct_thermo
                  pass3 = burn
               else
                  pass1 = mix
                  pass2 = struct_hydro
                  pass3 = struct_thermo
                  pass4 = burn
               end if
            case (12) ! 12 -- mix, then burn, then struct
               if (s% dynamics_coupling_choice == 0) then
                  pass1 = mix
                  pass2 = burn
                  pass3 = struct_hydro + struct_thermo
               else
                  pass1 = mix
                  pass2 = burn
                  pass3 = struct_hydro
                  pass4 = struct_thermo
               end if
            case default
               write(*,*) 'bad value for operator_coupling_choice', s% operator_coupling_choice
               ierr = -1
               return
         end select         
         
         call do_pass(pass1, 1, s% dt, ierr)
         if (ierr /= 0) return
         
         if (pass2 == 0) return
         
         call do_pass(pass2, 2, s% dt, ierr)
         if (ierr /= 0) return
         
         if (pass3 == 0) return
         
         call do_pass(pass3, 3, s% dt, ierr)
         if (ierr /= 0) return
         
         if (pass4 == 0) return
         
         call do_pass(pass4, 4, s% dt, ierr)
         if (ierr /= 0) return
         
         
         contains


         subroutine do_pass(pass_solve, pass, dt, ierr)
            use chem_def, only: ih1
            use solve_mix, only: do_solve_mix
            integer, intent(in) :: pass_solve, pass
            real(dp), intent(in) :: dt 
            integer, intent(out) :: ierr
            
            integer :: k, h1, species, nz
            
            include 'formats.dek'
            
            if (pass_solve == 0) return
            
            species = s% species
            nz = s% nz
            
            s% do_struct_hydro = BTEST(pass_solve, struct_hydro_bit_pos)
            s% do_struct_thermo = BTEST(pass_solve, struct_thermo_bit_pos)
            s% do_burn = BTEST(pass_solve, burn_bit_pos)
            s% do_mix = BTEST(pass_solve, mix_bit_pos)
            
            if (trace) then
               write(*,*)
               write(*,*) 'hydro,thermo,burn,mix', &
                  s% do_struct_hydro, s% do_struct_thermo, s% do_burn, s% do_mix
            end if
            
            if (pass == 1) then
               !s% xs_pre_pass => s% xs_old
               !s% xa_pre_pass => s% xa_pre_hydro
            else
               !allocate(s% xs_pre_pass(s% nvar, nz),stat=ierr)
               !if (ierr /= 0) return
               !s% xs_pre_pass(1:s% nvar_hydro,1:nz) = s% xs(1:s% nvar_hydro,1:nz)
               !allocate(s% xa_pre_pass(species, nz),stat=ierr)
               !if (ierr /= 0) return
               !s% xa_pre_pass(1:species,1:nz) = s% xa(1:species,1:nz)
            end if
            
            do_struct_burn_mix = &
               do_hydro_converge(s,s% newton_itermin,skip_global_corr_coeff_limit,dt)
            
            if (pass /= 1) then
               !deallocate(s% xs_pre_pass)
               !deallocate(s% xa_pre_pass)
            end if
            
            if (do_struct_burn_mix /= keep_going) ierr = -1
            if (failed('do_hydro_converge')) then
               if (dbg) write(*,*) 's% model_number', s% model_number
               if (dbg) stop 'debug: do_struct_burn_mix failed in do_hydro_converge'
               if (dbg) write(*,*) 'do_struct_burn_mix failed in do_hydro_converge', &
                  s% model_number
               return
            end if
            if (dbg) write(*,'(/,a)') 'done do_hydro_converge'

            if (check) then
               call check_rs(s, ierr)
               if (ierr /= 0) then
                  do_struct_burn_mix = retry
                  return
               end if
            end if
            
         end subroutine do_pass
         
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            if (do_struct_burn_mix /= keep_going) then
               failed = .true.
               if (dbg) write(*,*) 'failed ' // trim(str)
               return
            end if
            failed = .false.
         end function failed
         

      end function do_struct_burn_mix


      subroutine do_op_split( &
            s, skip_global_corr_coeff_limit, mbmdt1, mbmdt2, split_burn_mix, ierr)
         use star_utils, only: update_time, total_times
         use solve_hydro, only: do_hydro_converge
         use mix_info, only: get_convection_sigmas
         use num_lib, only: safe_log10
         use chem_def, only: ih1
         use solve_split_burn_mix, only: do_solve_split_burn_mix
         use solve_coupled_burn_mix, only: do_isolve_burn_mix

         type (star_info), pointer :: s
         logical, intent(in) :: skip_global_corr_coeff_limit
         real(dp), intent(in) :: mbmdt1, mbmdt2
         logical, intent(in) :: split_burn_mix
         integer, intent(out) :: ierr
         
         integer :: i, j, k
         integer :: time0, clock_rate, hydro_result
         real(dp) :: dt, total_all_before
         
         logical :: trace
         
         include 'formats.dek'
         
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
                  
         ierr = 0
         trace = s% op_split_trace     
         dt = mbmdt1 + mbmdt2  

         call get_convection_sigmas(s, dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'get_convection_sigmas failed'
            return
         end if
         
         if (mbmdt1 > 0) then
            if (trace) write(*,2) 'call burn_mix', s% model_number
            if (split_burn_mix) then
               call do_solve_split_burn_mix(s, mbmdt1, ierr) 
            else
               call do_isolve_burn_mix(s, mbmdt1, ierr)
            end if
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'burn_mix failed'
               return
            end if
            if (trace) write(*,2) 'done burn_mix', s% model_number, mbmdt1/secyer, &
               s% xa(s% net_iso(ih1),s% nz), s% lnd(s% nz)/ln10, s% lnT(s% nz)/ln10
         end if

         s% do_struct_hydro = .true.
         s% do_struct_thermo = .true.
         s% do_burn = .false.
         s% do_mix = .false.

         if (trace) write(*,2) 'call do_hydro_converge', s% model_number
         hydro_result = &
            do_hydro_converge(s, s% newton_itermin, skip_global_corr_coeff_limit, dt)
         if (hydro_result /= keep_going) then
            if (trace) write(*,*) 'struct failed to converge'
            if (s% report_ierr) write(*,*) 'struct failed in do_op_split'
            ierr = -1
            return
         end if
      
         if (trace) write(*,2) 'done do_hydro_converge', s% model_number, dt/secyer, &
            s% xa(s% net_iso(ih1),s% nz), s% lnd(s% nz)/ln10, s% lnT(s% nz)/ln10
         
         if (mbmdt2 > 0) then
            if (trace) write(*,2) 'call burn_mix', s% model_number
            if (split_burn_mix) then
               call do_solve_split_burn_mix(s, mbmdt2, ierr) 
            else
               call do_isolve_burn_mix(s, mbmdt1, ierr)
            end if
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'burn_mix failed'
               return
            end if
            if (trace) write(*,2) 'done burn_mix', s% model_number, mbmdt2/secyer, &
               s% xa(s% net_iso(ih1),s% nz), s% lnd(s% nz)/ln10, s% lnT(s% nz)/ln10
         end if
                     
         if (s% doing_timing) &
            call update_time(s, time0, total_all_before, s% time_op_split_control)
            
         if (trace) write(*,*)

      end subroutine do_op_split


      end module struct_burn_mix


