! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module star_utils

      use star_private_def
      use alert_lib
      use const_def
      use num_lib

      implicit none
      
      
      real(dp), parameter :: multiplier = 1d0 
         ! can make non-zero for debugging of partials
      logical, parameter :: try_arcsinh = .false.  ! convergence problems around 0
      real(dp), parameter :: arcsinh_mult = 1d0

      
      contains
      
      
      real(dp) function L_to_FL(s,L) result(FL)
         type (star_info), pointer :: s         
         real(dp), intent(in) :: L ! erg/sec
         if (.not. s% use_transformed_L) then
            FL = L/multiplier
         else if (try_arcsinh) then         
            FL = arcsinh(arcsinh_mult*L)            
         else if (L + s% FL_offset >= 1) then
            FL = log(L + s% FL_offset)
         else if (-L + s% FL_offset >= 1) then
            FL = -log(-L + s% FL_offset)
         else
            FL = 0d0
         end if
      end function L_to_FL
      
      
      real(dp) function FL_to_L(s,FL) result(L)
         type (star_info), pointer :: s         
         real(dp), intent(in) :: FL
         if (.not. s% use_transformed_L) then
            L = FL*multiplier
         else if (try_arcsinh) then         
            L = sinh(FL)/arcsinh_mult         
         else if (FL > 0) then
            L = exp(FL) - s% FL_offset
         else if (FL < 0) then
            L = -(exp(-FL) - s% FL_offset)
         else
            L = 0d0
         end if
      end function FL_to_L
      
      
      real(dp) function L_to_dFL_dL(s,L) result(dFL_dL)
         type (star_info), pointer :: s         
         real(dp), intent(in) :: L ! erg/sec
         if (.not. s% use_transformed_L) then
            dFL_dL = 1d0/multiplier
         else if (try_arcsinh) then         
            dFL_dL = arcsinh_mult*d_arcsinh_dx(arcsinh_mult*L)            
         else if ((L + s% FL_offset) >= 1) then ! FL = log((L + s% FL_offset))
            dFL_dL = 1d0/(L + s% FL_offset)
         else if ((-L + s% FL_offset) >= 1) then ! FL = -log((-L + s% FL_offset))
            dFL_dL = 1d0/(-L + s% FL_offset)
         else ! FL = 0
            dFL_dL = 1d0
         end if
      end function L_to_dFL_dL
      
      
      real(dp) function L_to_dL_dFL(s,L) result(dL_dFL)
         type (star_info), pointer :: s         
         real(dp), intent(in) :: L ! erg/sec
         dL_dFL = 1d0/L_to_dFL_dL(s,L)
      end function L_to_dL_dFL
      
      
      subroutine foreach_cell(s,nzlo,nzhi,use_omp,do1,ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: use_omp
         interface
            subroutine do1(s,k,ierr)
               use star_private_def
               type (star_info), pointer :: s         
               integer, intent(in) :: k
               integer, intent(out) :: ierr
            end subroutine do1
         end interface
         integer, intent(out) :: ierr
         
         integer :: k, op_err
         logical :: okay
         ierr = 0
         
         if (nzlo == nzhi) then
            call do1(s,nzlo,ierr)
            return
         end if
         
         if (use_omp) then
            okay = .true.
!$OMP PARALLEL DO PRIVATE(k,op_err) SCHEDULE(STATIC,10)
            do k = nzlo, nzhi
               if (.not. okay) cycle
               op_err = 0
               call do1(s,k,op_err)
               if (op_err /= 0) okay = .false. ! cannot just exit from a parallel loop
            end do
!$OMP END PARALLEL DO
            if (.not. okay) ierr = -1
         else
            do k = nzlo, nzhi
               call do1(s,k,ierr)
               if (ierr /= 0) exit
            end do
         end if
      
      end subroutine foreach_cell
      
      
      subroutine foreach_cell_dynamic(s,nzlo,nzhi,use_omp,do1,ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: use_omp
         interface
            subroutine do1(s,k,ierr)
               use star_private_def
               type (star_info), pointer :: s         
               integer, intent(in) :: k
               integer, intent(out) :: ierr
            end subroutine do1
         end interface
         integer, intent(out) :: ierr
         
         integer :: k, op_err
         ierr = 0
         
         if (nzlo == nzhi) then
            call do1(s,nzlo,ierr)
            return
         end if
         
         if (use_omp) then
!$OMP PARALLEL DO PRIVATE(k,op_err) SCHEDULE(DYNAMIC,10)
            do k = nzlo, nzhi
               if (ierr /= 0) cycle
               op_err = 0
               call do1(s,k,op_err)
               if (op_err /= 0) ierr = op_err
            end do
!$OMP END PARALLEL DO
         else
            do k = nzlo, nzhi
               call do1(s,k,ierr)
               if (ierr /= 0) exit
            end do
         end if
      
      end subroutine foreach_cell_dynamic


      real(dp) function sum_Egrav(s)
         type (star_info), pointer :: s
         integer :: k
         real(dp) :: dq_prev, dq_cur
         dq_cur = 0
         sum_Egrav = 0
         do k = 1, s% nz-1
            dq_prev = dq_cur
            dq_cur = s% dq(k)
            sum_Egrav = sum_Egrav + 0.5d0*(dq_prev + dq_cur)*s% q(k)/s% r(k)
         end do
         k = s% nz
         sum_Egrav = sum_Egrav + (0.5d0*s% dq(k-1) + s% dq(k))*s% q(k)/s% r(k)
         sum_Egrav = -sum_Egrav*s% cgrav(k)*s% mstar**2
      end function sum_Egrav


      real(dp) function sum_Etherm(s)
         type (star_info), pointer :: s
         integer :: nz
         nz = s% nz
         sum_Etherm = dot_product(s% dm(1:nz), exp(s% lnE(1:nz)))
      end function sum_Etherm


      real(dp) function sum_Ebinding(s)
         use chem_def
         type (star_info), pointer :: s
         integer :: cid, j
         real(dp) :: m1, n1, E1, Etotal
         integer :: nz
         nz = s% nz
         Etotal = 0
         do j=1, s% species
            cid = s% chem_id(j)
            m1 = dot_product(s% xa(j,1:nz),s% dm(1:nz)) ! grams of species j
            n1 = m1*avo/chem_isos% W (cid) ! number of species j nuclei
            E1 = ev2erg*1d6*chem_isos% binding_energy(cid) ! ergs binding energy per nuclei of species j
            Etotal = Etotal + E1*n1
         end do
         sum_Ebinding = Etotal
      end function sum_Ebinding
      
      
      real(dp) function sum_L_nuc(s)
         type (star_info), pointer :: s
         integer :: nz
         nz = s% nz
         sum_L_nuc = dot_product(s% dm(1:nz), s% eps_nuc(1:nz))
      end function sum_L_nuc
      
      
      real(dp) function sum_L_grav(s)
         type (star_info), pointer :: s
         integer :: nz
         nz = s% nz
         sum_L_grav = s% eps_grav_factor*dot_product(s% dm(1:nz), s% eps_grav(1:nz))
      end function sum_L_grav
      
      
      subroutine get_average_Y_and_Z(s, nzlo, nzhi, y_avg, z_avg, ierr)
         use chem_def
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         real(dp), intent(out) :: y_avg, z_avg
         integer, intent(out) :: ierr
         
         integer :: k, nz,  h1, h2, he3, he4
         real(dp) :: total_mass_h, total_mass_he, total_mass_z, &
            cell_mass, total_mass
         
         ierr = 0
         
         nz = s% nz
         h1 = s% net_iso(ih1)
         h2 = s% net_iso(ih2)
         he3 = s% net_iso(ihe3)
         he4 = s% net_iso(ihe4)
         total_mass=0; total_mass_h=0; total_mass_he=0; total_mass_z=0
         do k=nzlo, nzhi
            cell_mass = s% dm(k)
            total_mass = total_mass + cell_mass
            total_mass_h = total_mass_h + cell_mass*s% xa(h1, k)
            if (h2 /= 0) total_mass_h = total_mass_h + cell_mass*s% xa(h2, k)
            total_mass_he = total_mass_he + cell_mass*s% xa(he4, k)
            if (he3 /= 0) total_mass_he = total_mass_he + cell_mass*s% xa(he3, k)
         end do
         total_mass_z = total_mass - (total_mass_h + total_mass_he)
         z_avg = total_mass_z / total_mass
         y_avg = total_mass_he / total_mass
         
      end subroutine get_average_Y_and_Z


      real(dp) function eval_current_y(s, nzlo, nzhi, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         real(dp) :: y_avg, z_avg
         call get_average_Y_and_Z(s, nzlo, nzhi, y_avg, z_avg, ierr)
         eval_current_y = y_avg
      end function eval_current_y


      real(dp) function eval_current_z(s, nzlo, nzhi, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         real(dp) :: y_avg, z_avg
         call get_average_Y_and_Z(s, nzlo, nzhi, y_avg, z_avg, ierr)
         eval_current_z = z_avg
      end function eval_current_z


      real(dp) function eval_current_abundance(s, j, nzlo, nzhi, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: j, nzlo, nzhi
         integer, intent(out) :: ierr
         integer :: k, nz
         real(dp) :: cell_mass, jmass, total_mass
         
         ierr = 0
         
         if (j == 0) then
            eval_current_abundance = 0
            return
         end if
         
         nz = s% nz
         total_mass=0; jmass=0
         do k=nzlo, nzhi
            cell_mass = s% dm(k)
            total_mass = total_mass + cell_mass
            jmass = jmass + cell_mass*s% xa(j, k)
         end do
         eval_current_abundance = jmass / total_mass
         
      end function eval_current_abundance

      
      subroutine smooth_abundances(s, cnt, nzlo, nzhi, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: cnt ! make this many passes
         integer, intent(in) :: nzlo, nzhi ! only smooth zones nzlo to nzhi inclusive
         integer, intent(out) :: ierr
         integer :: k, j, nz
         ierr = 0
         nz = s% nz
         do j = 1, cnt
            do k = max(nzlo,2), min(nzhi, nz)
               s% xa(:,k) = (s% xa(:,k-1) + s% xa(:,k) + s% xa(:,k+1))/3
            end do
            if (nzhi == nz) s% xa(:,nz) = (s% xa(:,nz-1) + s% xa(:,nz) + s% xa(:,nz))/3
            if (nzlo == 1) s% xa(:,1) = (s% xa(:,2) + s% xa(:,1) + s% xa(:,1))/3
         end do
      end subroutine smooth_abundances
      
      
      integer function k_for_q(s, q)
         ! return k s.t. q(k) >= q > q(k)-dq(k)
         type (star_info), pointer :: s         
         real(dp), intent(in) :: q
         integer :: k, nz
         nz = s% nz
         if (q >= 1) then
            k_for_q = 1; return
         else if (q <= s% q(nz)) then
            k_for_q = nz; return
         end if
         do k = 1, nz-1
            if (q > s% q(k+1)) then
               k_for_q = k; return
            end if
         end do
         k_for_q = nz
      end function k_for_q
  
  
      subroutine get_string_for_model_number(num, n, photo_digits)
         character (len=256), intent(out) :: num
         integer, intent(in) :: n, photo_digits
         integer :: val
         include 'formats.dek'
         !write(*,2) 'n', n
         !write(*,2) 'photo_digits', photo_digits
         val = mod(n, 10**photo_digits) ! wrap around
         !write(*,2) '10**photo_digits', 10**photo_digits
         !write(*,2) 'val', val
         if (val == 0) then
            write(num,*) n
            num = adjustl(num)
            return
         end if
         write(num,*) val
         num = adjustl(num)
         do while (len_trim(num) < photo_digits)
            !write(*,2) 'num = ' // trim(num), len_trim(num)
            num = '0' // trim(num)
         end do
         num = 'x' // trim(num)
         !write(*,*) 'num = ' // trim(num)
      end subroutine get_string_for_model_number
         
         
      subroutine report_xa_bad_nums(s,ierr)         
         use utils_lib, only: is_bad_num
         type (star_info), pointer :: s
         integer, intent(out) :: ierr         
         integer :: k, j
         ierr = 0
         do k=1,s% nz
            do j=1,s% species
               if (is_bad_num(s% xa(j,k))) then
                  ierr = -1
                  write(*,*) j, k, s% xa(j,k)
               end if
            end do
         end do
      end subroutine report_xa_bad_nums
      
      
      subroutine normalize_dqs(nz, dq, ierr) 
         ! rescale dq's so that add to 1.000
         ! work in from boundaries to meet at largest dq
         integer, intent(in) :: nz
         real(dp), intent(inout) :: dq(nz)
         integer, intent(out) :: ierr
         integer :: k, midq
         real(dp) :: dqsum1, dqsum2
         include 'formats.dek'
         midq = maxloc(dq(1:nz),dim=1)
         ! surface inward
         dqsum1 = 0
         do k=1, midq
            dqsum1 = dqsum1 + dq(k)
            if (dq(k) <= 0) then
               ierr = -1
               !write(*,2) 'normalize_dqs: bad dq(k)', k, dq(k)
               return
            end if
         end do
         ! center outward
         dqsum2 = 0
         do k=nz, midq+1, -1
            dqsum2 = dqsum2 + dq(k)
            if (dq(k) <= 0) then
               ierr = -1
               !write(*,2) 'normalize_dqs: bad dq(k)', k, dq(k)
               return
            end if
         end do
         !write(*,1) 'normalize_dqs: dqsum1+dqsum2', dqsum1+dqsum2
         dq(1:nz) = dq(1:nz)/(dqsum1 + dqsum2)
      end subroutine normalize_dqs
      
      
      subroutine set_qs(nz, q, dq, ierr) ! set q's using dq's
         integer, intent(in) :: nz
         real(dp), intent(inout) :: dq(nz)
         real(dp), intent(out) :: q(nz)
         integer, intent(out) :: ierr
         integer :: k   
         include 'formats.dek'      
         ierr = 0
         q(1) = 1
         do k=2,nz-1
            q(k) = q(k-1) - dq(k-1)
         end do
         q(nz) = dq(nz)
         if (q(nz) >= q(nz-1)) then
            q(nz) = q(nz-1) - dq(nz-1)
            dq(nz) = q(nz)
            if (dq(nz) <= 0) then
               ierr = -1
               return
            end if
         end if
      end subroutine set_qs
      
      
      subroutine set_xqs(nz, xq, dq, ierr) ! set xq's using dq's
         integer, intent(in) :: nz
         real(dp), intent(inout) :: dq(nz)
         real(dp), intent(out) :: xq(nz)
         integer, intent(out) :: ierr
         integer :: k
         include 'formats.dek'      
         ierr = 0
         xq(1) = 0
         do k=2,nz-1
            xq(k) = xq(k-1) + dq(k-1)
         end do
         xq(nz) = 1 - dq(nz)
         if (xq(nz) < xq(nz-1)) then
            xq(nz) = xq(nz-1) + dq(nz-1)
            dq(nz) = 1 - xq(nz)
            if (dq(nz) <= 0) then
               ierr = -1
               return
            end if
         end if
      end subroutine set_xqs

         
      real(dp) function interp_val_to_pt(v,k,sz,dq)
         use interp_1d_lib, only: interp_4_to_1
         integer, intent(in) :: k, sz
         real(dp), pointer :: v(:), dq(:)
         integer :: ierr
         include 'formats.dek'
         if (k == 1) then
            interp_val_to_pt = v(k)
            return
         end if
         if (k > 2 .and. k < sz) then
            ierr = 0
            call interp_4_to_1( &
               0.5d0*(dq(k-2)+dq(k-1)), &
               0.5d0*(dq(k-1)+dq(k)), &
               0.5d0*(dq(k)+dq(k+1)), &
               0.5d0*dq(k-2)+dq(k-1), &
               v(k-2), v(k-1), v(k), v(k+1), &
               interp_val_to_pt, ierr)
            if (ierr == 0) return
         endif
         interp_val_to_pt = (v(k)*dq(k-1) + v(k-1)*dq(k))/(dq(k-1) + dq(k))
      end function interp_val_to_pt
      
      
      real(dp) function interp_xa_to_pt(xa,j,k,sz,dq)
         use interp_1d_lib, only: interp_4_to_1
         real(dp), pointer :: xa(:,:), dq(:)
         integer, intent(in) :: j, k, sz
         integer :: ierr
         include 'formats.dek'
         if (j == 0) then
            interp_xa_to_pt = 0
            return
         end if
         if (k == 1) then
            interp_xa_to_pt = xa(j,k)
            return
         end if
         if (k > 2 .and. k < sz) then
            ierr = 0
            call interp_4_to_1( &
               0.5d0*(dq(k-2)+dq(k-1)), &
               0.5d0*(dq(k-1)+dq(k)), &
               0.5d0*(dq(k)+dq(k+1)), &
               0.5d0*dq(k-2)+dq(k-1), &
               xa(j,k-2), xa(j,k-1), xa(j,k), xa(j,k+1), &
               interp_xa_to_pt, ierr)
            if (ierr == 0) return
         endif
         interp_xa_to_pt = (xa(j,k)*dq(k-1) + xa(j,k-1)*dq(k))/(dq(k-1) + dq(k))
      end function interp_xa_to_pt

      
      real(dp) function get_dtau1(s)
         type (star_info), pointer :: s  
         get_dtau1 = s% dm(1)*s% opacity(1)/(4*pi*s% rmid(1)**2)
      end function get_dtau1
      
      
      subroutine get_tau(s, tau)
         type (star_info), pointer :: s  
         real(dp), pointer :: tau(:) 
         ! tau(k) is optical depth at outer boundary of cell k
         real(dp) :: dtau, dr
         integer :: k
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         dtau = get_dtau1(s)
         tau(1) = s% tau_factor*s% tau_base 
         do k = 2, s% nz
            tau(k) = tau(k-1) + dtau
            dtau = s% dm(k)*s% opacity(k)/(4*pi*s% rmid(k)**2)
         end do
      end subroutine get_tau
         
         
      integer function find_cell_for_mass(s, m)
         type (star_info), pointer :: s
         real(dp), intent(in) :: m
         integer :: k
         find_cell_for_mass = s% nz
         do k = 1, s% nz-1
            if (s% m(k) >= m .and. m > s% m(k+1)) then
               find_cell_for_mass = k
               return
            end if
         end do
      end function find_cell_for_mass
      
      
      subroutine get_delta_Pg(s, nu_max, delta_Pg)
         use utils_lib, only: is_bad_num
         type (star_info), pointer :: s  
         real(dp), intent(in) :: nu_max ! microHz
         real(dp), intent(out) :: delta_Pg ! seconds
         ! g-mode period spacing for l=1
         real(dp) :: integral, N2, omega2, kr2, L2, el, &
            dr, r, r2, cs2, sl2, I_integral, I_integral_limit
         integer :: k
         include 'formats.dek'
         delta_Pg = 0
         integral = 0
         I_integral = 0
         I_integral_limit = 0.5
         omega2 = (2*pi*nu_max/1d6)**2
         el = 1
         L2 = el*(el+1)
         do k = 2, s% nz
            N2 = s% brunt_N2(k)
            r = s% r(k)
            r2 = r*r
            cs2 = s% csound_at_face(k)**2
            sl2 = L2*cs2/r2
            dr = s% rmid(k-1) - s% rmid(k)
            !if (N2 > 0) then
            !   write(*,*) 'omega2 >= sl2 omega2 < N2', omega2 >= sl2, omega2 < N2
            !   write(*,2) '      omega2, sl2, N2', k, omega2, sl2, N2
            !end if
            if (omega2 >= sl2) cycle
            if (omega2 < N2) then ! in g-cavity
               integral = integral + sqrt(N2)*dr/r
            else ! in decay region
               if (integral == 0) cycle ! ! haven't been in g-cavity yet
               !if (I_integral == 0) write(*,2) 'enter decay', k
               ! in decay region below g-cavity; I_integral estimates decay
               kr2 = (1 - n2/omega2)*(1 - Sl2/omega2)*omega2/cs2
               I_integral = I_integral + sqrt(-kr2)*dr
               if (I_integral > I_integral_limit) exit
            end if
         end do
         
         !write(*,2) 'omega2 nu_max integral I_integral', &
         !   s% model_number, omega2, nu_max, integral, I_integral

         if (integral == 0) return
         delta_Pg = sqrt(2d0)*pi**2/integral
         if (is_bad_num(delta_Pg)) delta_Pg = 0
         
      end subroutine get_delta_Pg
      
      
      real(dp) function get_tau_at_r(s, r)
         type (star_info), pointer :: s  
         real(dp), intent(in) :: r
         real(dp) :: dtau, dr, tau_m1, tau_00
         integer :: k
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         dtau = get_dtau1(s)
         tau_00 = s% tau_factor*s% tau_base 
         get_tau_at_r = tau_00
         if (r >= s% r(1)) return
         do k = 2, s% nz
            tau_m1 = tau_00
            tau_00 = tau_m1 + dtau
            if (r < s% r(k-1) .and. r >= s% r(k)) then
               get_tau_at_r = &
                  (tau_00*(s% r(k-1)-r) + tau_m1*(r-s% r(k)))/(s% r(k-1)-s% r(k))
               return
            end if
            dtau = s% dm(k)*s% opacity(k)/(4*pi*s% rmid(k)**2)
         end do
      end function get_tau_at_r
      
      
      integer function find_tau_phot(s, tau00, taup1, ierr) 
         ! return k for the cell containing optical depth = 2/3
         type (star_info), pointer :: s  
         real(dp), intent(out) :: tau00, taup1
         integer, intent(out) :: ierr
         integer :: k
         real(dp) :: dtau
         
         real(dp), parameter :: tau_phot = 2d0/3d0
         
         include 'formats.dek'
         ierr = 0
         tau00 = 0
         taup1 = 0
         find_tau_phot = 1
         if (s% tau_factor >= 1) return
         tau00 = s% tau_factor*s% tau_base
         do k = 1, s% nz
            dtau = s% dm(k)*s% opacity(k)/(4*pi*s% rmid(k)**2)
            taup1 = tau00 + dtau
            if (taup1 >= tau_phot) then
               find_tau_phot = k
               return
            end if
            tau00 = taup1
         end do
         ierr = -1
      end function find_tau_phot

            
      subroutine interp_q( &
            nz2, nvar_hydro, species, qval, xh, xa, q, dq, struct, comp, ierr)
         use num_lib, only: binary_search
         integer, intent(in) :: nz2, nvar_hydro, species
         real(dp), intent(in) :: qval, xh(nvar_hydro,nz2), xa(species,nz2), q(nz2), dq(nz2)
         real(dp), intent(out) :: struct(nvar_hydro), comp(species)
         integer, intent(out) :: ierr         
         integer :: k
         real(dp) :: alfa         
         ierr = 0         
         if (qval <= q(nz2)) then
            if (nvar_hydro > 0) &
               struct(1:nvar_hydro) = xh(1:nvar_hydro,nz2)
            if (species > 0) &
               comp(1:species) = xa(1:species,nz2)
            return
         end if
         k = binary_search(nz2, 0, q, qval)
         if (qval <= q(k) .and. qval > q(k+1)) then
            alfa = (qval - q(k+1)) / dq(k)
            if (nvar_hydro > 0) &
               struct(1:nvar_hydro) = &
                  alfa*xh(1:nvar_hydro,k) + (1-alfa)*xh(1:nvar_hydro,k+1)
            if (species > 0) &
               comp(1:species) = alfa*xa(1:species,k) + (1-alfa)*xa(1:species,k+1)
            return
         end if
         ierr = -1         
      end subroutine interp_q

      
      subroutine std_write_internals_to_file(id, num)
         integer, intent(in) :: num, id
         character (len=256) :: fname
         integer :: ierr
         ierr = 0
         write(fname, '(a, i1, a)') 'plot_data/internals', mod(abs(num), 10), '.data'
         write(*,*) 'call write_internals_to_file ' // trim(fname)
         call write_internals_to_file(id, fname, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in write_internals_to_file ' // trim(fname)
         end if
      end subroutine std_write_internals_to_file
      
      
      subroutine write_internals_to_file(id, filename, ierr)
         use utils_lib
         character (len=*), intent(in) :: filename
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         integer :: iounit
         ierr = 0
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         open(iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
         if (ierr == 0) then
            call write_internals(id, iounit, ierr)
            close(iounit)
         else
            write(*, *) 'failed to open internals file ' // trim(filename)
         end if
         call free_iounit(iounit)
      end subroutine write_internals_to_file
      
      
      subroutine write_internals(id, iounit, ierr)
         use chem_def
         integer, intent(in) :: iounit, id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s 
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            write(*,*) 'write_internals: get_star_ptr ierr', ierr
            return
         end if
         call write_model_info(s, iounit, ierr)
      end subroutine write_internals
      
      
      subroutine write_model_info(s, iounit, ierr)
         use chem_def
         use num_lib, only: safe_log10
         type (star_info), pointer :: s 
         integer, intent(in) :: iounit
         integer, intent(out) :: ierr
         integer, pointer :: chem_id(:)
         integer :: k, i, nz, species
         
         integer :: he4

         ierr = 0
         nz = s% nz
         species = s% species
         chem_id => s% chem_id
         he4 = s% net_iso(ihe4)
         
         write(iounit,'(a)') '            mass         initial_z       n_shells'
         write(iounit,'(2x,2f15.4,i15)') s% star_mass, s% initial_z, nz
         write(iounit,fmt='(i5)',advance='no') 1
         do i=2,88
            write(iounit,fmt='(i12,15x)',advance='no') i
         end do
         write(iounit,*)
         write(iounit,fmt='(a5,1x,99(a26,1x))',advance='no') &
               'grid', 'r', 'm', 'log_dq', &
               'log10d', 'log10T', 'log10m', 'log10r', 'L', 'r_div_rstar', &
               'log10P', 'log10Pgas', 'chiT', 'chiRho', &
               'dlnRho_dlnPgas_const_T', 'dlnRho_dlnT_const_Pgas', &
               'd_dlnd_const_lnT', 'd_dlnT_const_lnd', 'xscale_lnd', 'xscale_lnT', &
               'dequP_dlnPgas', 'term2', 'term1', 'Prad_div_P'
         ! composition info
         do i=1, species
            write(iounit, fmt='(a26, 1x)', advance='no') trim(chem_isos% name(chem_id(i)))
         end do
         do i=1, species
            write(iounit, fmt='(a26, 1x)', advance='no') 'lg_' // trim(chem_isos% name(chem_id(i)))
         end do
         write(iounit,fmt=*)
               
         do k=1, nz
            write(iounit,'(i5,1x,99(1pe26.16,1x))',advance='no') k,  &
               s% r(k)/Rsun, s% m(k)/Msun, safe_log10(s% dq(k)), &
               s% lnd(k)/ln10, s% lnT(k)/ln10, log10(s% m(k)),  &
               s% lnR(k)/ln10, s% L(k)/Lsun, s% r(k)/s% r(1), &
               s% lnP(k)/ln10, s% lnPgas(k)/ln10, s% chiT(k), s% chiRho(k), &
               s% dlnRho_dlnPgas_const_T(k), s% dlnRho_dlnT_const_Pgas(k), &
               s% profile_extra(k,1), s% profile_extra(k,2), &
               s% profile_extra(k,3), s% profile_extra(k,4), &
               s% profile_extra(k,5), s% profile_extra(k,6), &
               s% profile_extra(k,7), s% profile_extra(k,8)
            do i=1, species
               write(iounit, fmt='(1pe26.16, 1x)', advance='no') s% xa(i, k)
            end do               
            do i=1, species
               write(iounit, fmt='(1pe26.16, 1x)', advance='no') safe_log10(s% xa(i, k))
            end do               
            write(iounit,*)
         end do
      
      end subroutine write_model_info

      
      subroutine std_dump_model_info_for_ndiff(s, num, ierr)
         use utils_lib
         type (star_info), pointer :: s 
         integer, intent(in) :: num
         integer, intent(out) :: ierr
         character (len=256) :: fname
         integer :: iounit
         ierr = 0
         write(fname, '(a, i1)') 'n', mod(abs(num), 10)
         write(*,*) 'dump_model_info_for_ndiff ' // trim(fname)
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         open(iounit, file=trim(fname), action='write', status='replace', iostat=ierr)
         if (ierr /= 0) then
            call free_iounit(iounit)
            return
         end if
         call dump_model_info_for_ndiff(s, iounit, ierr)
         call free_iounit(iounit)
      end subroutine std_dump_model_info_for_ndiff

      
      subroutine dump_model_info_for_ndiff(s, iounit, ierr)
         use chem_def
         use num_lib, only: safe_log10
         type (star_info), pointer :: s 
         integer, intent(in) :: iounit
         integer, intent(out) :: ierr
         integer, pointer :: chem_id(:)
         integer :: k, j, nz, species
         include 'formats.dek'
         ierr = 0
         nz = s% nz
         species = s% species
         chem_id => s% chem_id
         write(iounit,*) 'nz', nz
         write(iounit,1) 'star_mass', s% star_mass
         write(iounit,1) 'initial_z', s% initial_z
         do k=1, nz
            do j=1, s% nvar_hydro
               write(iounit,2) trim(s% nameofvar(j)), k, s% xh(j,k)
            end do
            do j=1,species
               write(iounit,2) trim(chem_isos% name(chem_id(j))), k, clip(s% xa(j,k))
            end do
         end do
         
         contains
         
         real(dp) function clip(x)
            real(dp), intent(in) :: x
            if (.true. .or. x > 1d-30) then
               clip = x
            else
               clip = 0d0
            end if
         end function clip
         
      end subroutine dump_model_info_for_ndiff


      subroutine set_tau_base(id, tau_base, ierr)
         integer, intent(in) :: id
         real(dp), intent(in) :: tau_base
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         s% tau_base = tau_base
      end subroutine set_tau_base
      

      real(dp) function dt_Courant(s)
         type (star_info), pointer :: s 
         integer :: k
         real(dp) :: dt
         dt_Courant = s% r(s% nz)/s% csound(s% nz)
         do k=1, s% nz-1
            dt = (s% r(k) - s% r(k+1))/s% csound(k)
            if (dt < dt_Courant) dt_Courant = dt
         end do
      end function dt_Courant


      real(dp) function xscale_height(s, k, dbg)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         logical, intent(in) :: dbg
         real(dp) :: Hp, alt_Hp, alfa, beta, rho_face, P_face
         include 'formats.dek'
         
         ! obsolete -- just use s% scale_height(k) instead
         stop 'star_utils: xscale_height'
         
         if (k == 1) then
            alfa = 1
         else
            alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
         end if
         beta = 1 - alfa
         if (alfa == 1) then
            rho_face = s% rho(k)
            P_face = s% P(k)
         else
            rho_face = alfa*s% rho(k) + beta*s% rho(k-1)
            P_face = alfa*s% P(k) + beta*s% P(k-1)
         end if
         Hp = P_face/(rho_face*s% grav(k))
         if (s% alt_scale_height_flag) then
            alt_Hp = sqrt(P_face / s% cgrav(k)) / rho_face
            xscale_height = min(Hp, alt_Hp)
         else
            alt_Hp = -1 ! for dbg
            xscale_height = Hp
         end if
         if (.not. dbg) return
         write(*,1) 'xscale_height', xscale_height
         write(*,1) 'Hp', Hp
         write(*,1) 'alt_Hp', alt_Hp
         write(*,1) 'P_face', P_face
         write(*,1) 'rho_face', rho_face
         write(*,1) 's% cgrav(k)', s% cgrav(k)
         write(*,1) 's% grav(k)', s% grav(k)
         write(*,1) 'alfa', alfa
         write(*,*)
      end function xscale_height


      real(dp) function eval_Ledd(s)
         use mlt_def, only: no_mixing
         type (star_info), pointer :: s
         real(dp) :: dtau, dr, tau, dqsum, kap_sum, kap
         integer :: k
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         dtau = get_dtau1(s)
         tau = s% tau_factor*s% tau_base
         dqsum = s% dq(1)
         kap_sum = s% dq(1)*s% opacity(1)
         do k = 2, s% nz
            tau = tau + dtau
            if (tau > s% surf_avg_tau) exit
            dtau = s% dm(k)*s% opacity(k)/(4*pi*s% rmid(k)**2)
            dqsum = dqsum + s% dq(k)
            kap_sum = kap_sum + s% dq(k)*s% opacity(k)
         end do
         if (dqsum > 0) then
            kap = kap_sum/dqsum
         else
            kap = s% opacity(1)
         end if
         eval_Ledd = 4*pi*clight*s% cgrav(1)*s% mstar/kap
      end function eval_Ledd
         
         
      real(dp) function total_angular_momentum(s) result(J)
         type (star_info), pointer :: s 
         integer :: k
         real(dp) :: dmm1, dm00, dm
         include 'formats.dek'
         J = 0
         if (.not. s% rotation_flag) return
         dm00 = 0
         do k = 1, s% nz
            dmm1 = dm00
            dm00 = s% dm(k)
            if (k == s% nz) then
               dm = 0.5d0*dmm1+dm00
            else
               dm = 0.5d0*(dmm1+dm00)
            end if
            J = J + dm*s% j_rot(k)
         end do
      end function total_angular_momentum
         
         
      real(dp) function total_extra_angular_momentum(s,dt) result(J)
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         integer :: k
         real(dp) :: dmm1, dm00, dm
         include 'formats.dek'
         J = 0
         if (.not. s% rotation_flag) return
         dm00 = 0
         do k = 1, s% nz
            dmm1 = dm00
            dm00 = s% dm(k)
            if (k == s% nz) then
               dm = 0.5d0*dmm1+dm00
            else
               dm = 0.5d0*(dmm1+dm00)
            end if
            J = J + dm*dt* &
               (s% extra_omegadot(k)*s% i_rot(k) + s% extra_jdot(k))
         end do
      end function total_extra_angular_momentum


      real(dp) function surf_omega(s)
         type (star_info), pointer :: s
         real(dp) :: dtau, dr, tau, dqsum, omega_sum
         integer :: k
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         surf_omega = 0
         if (.not. s% rotation_flag) return
         dtau = get_dtau1(s)
         tau = s% tau_factor*s% tau_base
         dqsum = s% dq(1)
         omega_sum = s% dq(1)*s% omega(1)
         do k = 2, s% nz
            tau = tau + dtau
            if (tau > s% surf_avg_tau) exit
            dtau = s% dm(k)*s% opacity(k)/(4*pi*s% rmid(k)**2)
            dqsum = dqsum + s% dq(k)
            omega_sum = omega_sum + s% dq(k)*s% omega(k)
         end do
         if (dqsum > 0) then
            surf_omega = omega_sum/dqsum
         else
            surf_omega = s% omega(1)
         end if
      end function surf_omega
      
      
      real(dp) function surf_omega_crit(s)
         type (star_info), pointer :: s 
         real(dp) :: dtau, dr, tau, dqsum, omega_crit_sum
         integer :: k
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         surf_omega_crit = 0
         if (.not. s% rotation_flag) return
         dtau = get_dtau1(s)
         tau = s% tau_factor*s% tau_base
         dqsum = s% dq(1)
         omega_crit_sum = s% dq(1)*omega_crit(s,1)
         do k = 2, s% nz
            tau = tau + dtau
            if (tau > s% surf_avg_tau) exit
            dtau = s% dm(k)*s% opacity(k)/(4*pi*s% rmid(k)**2)
            dqsum = dqsum + s% dq(k)
            omega_crit_sum = omega_crit_sum + s% dq(k)*omega_crit(s,k)
         end do
         if (dqsum > 0) then
            surf_omega_crit = omega_crit_sum/dqsum
         else
            surf_omega_crit = 0
         end if
         surf_omega_crit = max(surf_omega_crit, 1d-99) ! so can safely divide by it
      end function surf_omega_crit
      
      
      real(dp) function surf_v_div_v_crit(s)
         type (star_info), pointer :: s 
         real(dp) :: dtau, dr, tau, dqsum, v_div_v_crit, v_div_v_crit_sum
         integer :: k
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         surf_v_div_v_crit = 0
         if (.not. s% rotation_flag) return
         dtau = get_dtau1(s)
         tau = s% tau_factor*s% tau_base
         dqsum = s% dq(1)
         v_div_v_crit = min(1d0, s% omega(1)/max(1d-99,omega_crit(s,1)))
         v_div_v_crit_sum = s% dq(1)*v_div_v_crit
         do k = 2, s% nz
            tau = tau + dtau
            if (tau > s% surf_avg_tau) exit
            dtau = s% dm(k)*s% opacity(k)/(4*pi*s% rmid(k)**2)
            dqsum = dqsum + s% dq(k)
            v_div_v_crit = min(1d0, s% omega(k)/max(1d-99,omega_crit(s,k)))
            v_div_v_crit_sum = v_div_v_crit_sum + s% dq(k)*v_div_v_crit
         end do
         if (dqsum > 0) then
            surf_v_div_v_crit = v_div_v_crit_sum/dqsum
         else
            surf_v_div_v_crit = 0
         end if
      end function surf_v_div_v_crit


      real(dp) function omega_crit(s, k) 
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp) :: gamma_Edd, Ledd
         include 'formats.dek'
         if (.not. s% rotation_flag) then
            omega_crit = 0
            return
         end if
         Ledd = 4*pi*clight*s% cgrav(k)*s% m(k)/s% opacity(k)
         gamma_Edd = 1d0 - min(s% L(k)/Ledd, 0.9999d0)
         omega_crit = sqrt(gamma_Edd*s% cgrav(k)*s% m(k)/s% r(k)**3)
      end function omega_crit

         
      real(dp) function eval_irradiation_heat(s,k)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp) :: irradiation_dq, xq, eps
         eval_irradiation_heat = 0
         if (s% irradiation_flux /= 0) then
            irradiation_dq = s% area(1)*s% column_depth_for_irradiation/s% xmstar
            xq = 1 - s% q(k)
            if (irradiation_dq > xq) then ! add irradiation heat for cell k
               eps = 0.25d0 * s% irradiation_flux / s% column_depth_for_irradiation
               if (irradiation_dq < xq + s% dq(k)) then ! only part of cell gets heated
                  eval_irradiation_heat = eps*(irradiation_dq - xq)/s% dq(k)
               else ! all of cell gets heated
                  eval_irradiation_heat = eps
               end if
            end if
         end if
      end function eval_irradiation_heat

      
      subroutine update_time(s, time0, total_all_before, total)
         type (star_info), pointer :: s 
         integer, intent(in) :: time0
         real(dp), intent(in) :: total_all_before
         real(dp), intent(inout) :: total
         real(dp) :: total_all_after, other_stuff
         integer :: time1, clock_rate
         call system_clock(time1,clock_rate)
         total_all_after = total_times(s)
         other_stuff = total_all_after - total_all_before 
            ! don't double count any other stuff
         total = total + (dble(time1-time0)/clock_rate - other_stuff)
      end subroutine update_time
      
      
      real(dp) function total_times(s)
         type (star_info), pointer :: s 
         total_times = &
            s% time_eos + &
            s% time_kap + &
            s% time_net + &
            s% time_neu + &
            s% time_do_mesh + &
            s% time_do_mesh_plan + &
            s% time_do_mesh_adjust + &
            s% time_do_adjust_mass + &
            s% time_do_report + &
            s% time_next_timestep + &
            s% time_write_profile + &
            s% time_write_log + &
            s% time_write_photo + &
            s% time_pgstar + &
            s% time_set_basic_vars + &
            s% time_set_rotation_vars + &
            s% time_set_mlt_vars + &
            s% time_set_eqn_vars + &
            s% time_eval_eqns + &
            s% time_set_newton_vars + &
            s% time_newton_mtx + &
            s% time_newton_xscale + &
            s% time_newton_eval_eqn + &
            s% time_newton_size_equ + &
            s% time_newton_size_B + &
            s% time_newton_enter_setmatrix + &
            s% time_newton_self + &
            !s% time_newton_test + &     <<<<<< this isn't a separate item, so don't include in total
            s% time_solve_burn_in_net + &
            s% time_solve_burn_non_net + &
            s% time_solve_mix + &
            s% time_op_split_control + &
            s% time_check_model + &
            s% time_prep_new_step + &
            s% time_prep_new_try + &
            s% time_prep_for_retry + &
            s% time_do_winds + &
            s% time_save_for_d_dt + &
            s% time_diffusion + &
            s% time_evolve_set_vars + &
            s% time_save_pre_hydro + &
            s% time_do_brunt + &
            s% time_struct_burn_mix + &
            s% time_check_newly_non_conv
      end function total_times
      
      
      subroutine dump_model_for_diff(s,io)
         use chem_def, only: chem_isos
         type (star_info), pointer :: s
         integer, intent(in) :: io         
         integer :: k, j, i
         include 'formats.dek'      
         write(io,1) 'dump_model_for_diff'   
         do k = 1, s% nz
            do j = 1, s% nvar_hydro
               write(io,3) s% nameofvar(j), k, j, s% xh(j,k)
            end do
            do j = s% nvar_hydro+1, s% nvar
               i = j-s% nvar_hydro
               write(io,3) 'var_' // trim(chem_isos% name(s% chem_id(i))), k, i, s% xa(i,k)
            end do
         end do      
      end subroutine dump_model_for_diff
      
      
      subroutine smooth(dc, sz)
         real(dp), dimension(:), pointer :: dc
         integer, intent(in) :: sz
         integer :: k
         k = 1
         dc(k) = (3*dc(k) + dc(k+1))/4
         do k=2,sz-1
            dc(k) = (dc(k-1) + 2*dc(k) + dc(k+1))/4
         end do
         k = sz
         dc(k) = (dc(k-1) + 3*dc(k))/4
      end subroutine smooth

      
      subroutine dq_smooth_nonconv(s, dqsm, dc, work)
         use mlt_def, only: convective_mixing
         type (star_info), pointer :: s
         real(dp), intent(in) :: dqsm
         real(dp), dimension(:) :: dc, work
      
         real(dp) :: q0, qcntr, qsurf, dq, dqsum
         integer :: nz, k, kcntr, ksurf
         
         if (dqsm <= 0) return
         nz = s% nz
         work(1:nz) = dc(1:nz)
         dq = dqsm
         do k = 1, nz
            if (s% mixing_type(k) == convective_mixing) cycle
            q0 = s% q(k)
            qcntr = q0 - dq*0.5
            qsurf = q0 + dq*0.5
            kcntr = k
            do while (kcntr < nz .and. s% q(kcntr) > qcntr)
               if (s% mixing_type(kcntr+1) == convective_mixing) exit
               kcntr = kcntr+1
            end do
            ksurf = k
            do while (ksurf > 1 .and. s% q(ksurf) < qsurf)
               if (s% mixing_type(ksurf-1) == convective_mixing) exit
               ksurf = ksurf-1
            end do
            dqsum = sum(s% dq(ksurf:kcntr))
            dc(k) = dot_product(s% dq(ksurf:kcntr),work(ksurf:kcntr))/dqsum
         end do
         
      end subroutine dq_smooth_nonconv

      
      subroutine dr_div_R_smooth_nonconv(s, dr_div_R_width, cell_dr, v, work)
         use mlt_def, only: convective_mixing
         use utils_lib, only: is_bad_num
         
         type (star_info), pointer :: s
         real(dp), intent(in) :: dr_div_R_width
         real(dp), dimension(:) :: cell_dr, v, work
      
         real(dp) :: r0, rcntr, rsurf, dr, drsum, rstar
         integer :: nz, k, j, kcntr, ksurf
         
         include 'formats.dek'
         
         if (dr_div_R_width <= 0) return
         nz = s% nz
         rstar = s% r(1)
         work(1:nz) = v(1:nz)
         dr = dr_div_R_width*rstar
         do k = 1, nz
            if (s% mixing_type(k) == convective_mixing) cycle
            r0 = s% rmid(k)
            rcntr = r0 - dr*0.5
            rsurf = r0 + dr*0.5
            kcntr = k
            do while (kcntr < nz .and. s% rmid(kcntr) > rcntr)
               if (s% mixing_type(kcntr+1) == convective_mixing) exit
               kcntr = kcntr+1
            end do
            ksurf = k
            do while (ksurf > 1 .and. s% rmid(ksurf) < rsurf)
               if (s% mixing_type(ksurf-1) == convective_mixing) exit
               ksurf = ksurf-1
            end do
            drsum = sum(cell_dr(ksurf:kcntr))
            v(k) = dot_product(cell_dr(ksurf:kcntr),work(ksurf:kcntr))/drsum
            if (is_bad_num(v(k))) then
               write(*,2) 'v(k)', k, v(k)
               write(*,2) 'drsum', k, drsum
               do j=ksurf,kcntr
                  write(*,2) 'work(j)', j, work(j)
                  write(*,2) 'cell_dr(j)', j, cell_dr(j)
               end do
               stop 'debug dr_div_R_smooth_nonconv'
            end if
         end do
         
      end subroutine dr_div_R_smooth_nonconv
      
      
      subroutine get_XYZ(s, xa, X, Y, Z)
         use chem_def, only: ih1, ih2, ihe3, ihe4
         type (star_info), pointer :: s
         real(dp), intent(in) :: xa(:)
         real(dp), intent(out) :: X, Y, Z
         X = 0
         if (s% net_iso(ih1) /= 0) X = X + xa(s% net_iso(ih1))
         if (s% net_iso(ih2) /= 0) X = X + xa(s% net_iso(ih2))
         Y = 0
         if (s% net_iso(ihe3) /= 0) Y = Y + xa(s% net_iso(ihe3))
         if (s% net_iso(ihe4) /= 0) Y = Y + xa(s% net_iso(ihe4))
         Z = min(1d0, max(0d0, 1d0 - (X + Y)))
      end subroutine get_XYZ
      
      
      end module star_utils
