! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module solve_split_burn_mix

      use star_private_def

      implicit none

      
      contains
      
      
      ! evolve abundances using current structure
      ! this also sets eps_nuc for use in luminosity equation
      subroutine do_solve_split_burn_mix(s, dt, ierr)
         use solve_burn, only: setup_interpolants_for_burn
         use num_lib, only: safe_log10
         use const_def
         use rates_def
         use chem_def
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr

         real(dp), pointer, dimension(:) :: times
         real(dp), pointer, dimension(:,:,:) :: log10Ts_f, log10Rhos_f, etas_f
         integer :: max_num_times, num_times, result, nz
         real(dp) :: power_total
         logical :: trace, need_final_eps_nuc
         integer :: j
         
         include 'formats.dek'
         
         ierr = 0
         trace = s% op_split_burn_mix_trace
         nz = s% nz
         
         if (trace) write(*,1) 'start do_solve_split_burn_mix', dt/secyer, &
            s% xa(s% net_iso(ih1),s% nz), s% lnd(s% nz)/ln10, s% lnT(s% nz)/ln10
         
         need_final_eps_nuc = .true.
         
         max_num_times = 1
         call setup_interpolants_for_burn( &
            s, dt, max_num_times, &
            num_times, times, log10Ts_f, log10Rhos_f, etas_f, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'setup_interpolants_for_burn failed'
            call dealloc
            return
         end if
         
         result = do_split_mix_burn(s, 0d0, dt, need_final_eps_nuc, .true., &
            num_times, times, log10Ts_f, log10Rhos_f, etas_f)
         if (result /= keep_going) then
            if (trace) write(*,*) 'burn+mix failed'
            if (s% report_ierr) write(*,*) 'burn+mix failed in do_op_split'
            ierr = -1
            call dealloc
            return
         end if
         
         call dealloc
         
         s% Lnuc_1 = dot_product(s% dm(1:nz), s% eps_nuc(1:nz))/Lsun
         s% Lphoto_1 = &
            dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
         s% Lnucneu_1 = dot_product(s% dm(1:nz), s% eps_nuc_neu_total(1:nz))/Lsun
         power_total = 0
         do j=1,num_categories
            if (j == iphoto) cycle
            power_total = power_total + &
               dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,j,1:nz))/Lsun
         end do
         s% Ltot_1 = power_total

         if (trace) write(*,1) 'end do_solve_split_burn_mix', dt/secyer, &
            s% xa(s% net_iso(ih1),s% nz), s% lnd(s% nz)/ln10, s% lnT(s% nz)/ln10
         
         contains
         
         
         subroutine dealloc
            deallocate(times, log10Ts_f, log10Rhos_f, etas_f)
         end subroutine dealloc
         
      
      end subroutine do_solve_split_burn_mix


      integer function do_split_mix_burn( &
            s, t_start, t_end, need_final_eps_nuc, strang_style, &
            num_times, times, log10Ts_f, log10Rhos_f, etas_f)
         use chem_def, only: chem_isos
         use utils_lib, only: is_bad_num
         
         type (star_info), pointer :: s
         real(dp), intent(in) :: t_start, t_end
         logical, intent(in) :: need_final_eps_nuc, strang_style
         integer, intent(in) :: num_times ! for interpolation of lnT and lnRho
         real(dp), pointer, dimension(:) :: times
         real(dp), pointer, dimension(:,:,:) :: log10Ts_f, log10Rhos_f, etas_f

         logical, parameter :: use_prev_burn = .false.
         real(dp) :: dt, t
         integer :: species, nz, ierr
         logical :: trace
         real(dp), dimension (:,:), pointer :: ptr
         real(dp), pointer :: dxdt_source_term(:,:) 
            ! (species,nz)  or null if no source term.

         include 'formats.dek'
         
         ierr = 0
         nz = s% nz
         species = s% species
         dt = t_end - t_start
         t = 0
         dxdt_source_term => null()
      
         trace = s% op_split_burn_mix_trace

         if (strang_style) then
            if (.not. mix_1st_half()) return
            if (.not. burn_full()) return
            if (.not. mix_2nd_half()) return
         else
            if (.not. burn_full()) return
            if (.not. mix_full()) return
         end if
         
         
         contains
         
         
         logical function mix_full()
            use solve_mix, only: do_solve_mix
            mix_full = .true.
            do_split_mix_burn = do_solve_mix(s, t, t + dt, dxdt_source_term)
            if (do_split_mix_burn /= keep_going) then
               if (s% report_ierr .or. trace) &
                  write(*,*) 'do_split_mix_burn: mix_full failed'
               mix_full = .false.
            end if
         end function mix_full
         
         
         logical function mix_1st_half()
            use solve_mix, only: do_solve_mix
            mix_1st_half = .true.
            do_split_mix_burn = do_solve_mix(s, t, t + dt/2, dxdt_source_term)
            if (do_split_mix_burn /= keep_going) then
               if (s% report_ierr .or. trace) &
                  write(*,*) 'do_split_mix_burn: mix_1st_half failed'
               mix_1st_half = .false.
            end if
         end function mix_1st_half
         
         
         logical function mix_2nd_half()
            use solve_mix, only: do_solve_mix
            mix_2nd_half = .true.
            do_split_mix_burn = do_solve_mix(s, t + dt/2, t + dt, dxdt_source_term)
            if (do_split_mix_burn /= keep_going) then
               if (s% report_ierr .or. trace) &
                  write(*,*) 'do_split_mix_burn: mix_1st_half failed'
               mix_2nd_half = .false.
            end if
         end function mix_2nd_half
         
         
         logical function burn_full()
            use solve_burn, only: do_solve_burn
            burn_full = .true.
            do_split_mix_burn = do_solve_burn( &
               s, t, t + dt, dt, dxdt_source_term, need_final_eps_nuc, &
               num_times, times, log10Ts_f, log10Rhos_f, etas_f)
            if (do_split_mix_burn /= keep_going) then
               if (s% report_ierr .or. trace) &
                  write(*,*) 'do_split_mix_burn: burn_full failed'
               burn_full = .false.
            end if
         end function burn_full
         
         
         logical function burn_1st_half()
            use solve_burn, only: do_solve_burn
            burn_1st_half = .true.
            do_split_mix_burn = do_solve_burn( &
               s, t, t + dt/2, dt, dxdt_source_term, .false., &
               num_times, times, log10Ts_f, log10Rhos_f, etas_f)
            if (do_split_mix_burn /= keep_going) then
               if (s% report_ierr .or. trace) &
                  write(*,*) 'do_split_mix_burn: burn_1st_half failed'
               burn_1st_half = .false.
            end if
         end function burn_1st_half
         
         
         logical function burn_2nd_half()
            use solve_burn, only: do_solve_burn
            burn_2nd_half = .true.
            do_split_mix_burn = do_solve_burn( &
               s, t + dt/2, t + dt, dt, dxdt_source_term, need_final_eps_nuc, &
               num_times, times, log10Ts_f, log10Rhos_f, etas_f)
            if (do_split_mix_burn /= keep_going) then
               if (s% report_ierr .or. trace) &
                  write(*,*) 'do_split_mix_burn: burn_2nd_half failed'
               burn_2nd_half = .false.
            end if
         end function burn_2nd_half
         
         
      end function do_split_mix_burn


      end module solve_split_burn_mix


