! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module solve_hydro

      use star_private_def
      use alert_lib
      use const_def

      implicit none

      integer, parameter :: stencil_neighbors = 1 
            ! number of neighbors on each side (e.g., =1 for 3 point stencil)


      integer, parameter :: nstep_write_solve = 1 ! make this a control


      contains
      

      integer function do_hydro_converge(s, itermin, skip_global_corr_coeff_limit, dt)
         ! return keep_going, retry, backup, or terminate
         
         ! do not require that functions have been evaluated for starting configuration.
         ! when finish, will have functions evaluated for the final set of primary variables.

         use mtx_lib
         use mtx_def
         use num_def
         use hydro_vars, only: set_all_vars_except_mixing_info
         !use hydro_seulex, only: do_hydro_seulex
         use hydro_bimd, only: do_hydro_bimd
         use hydro_rotation, only: get_rotation_sigmas
         use mix_info, only: get_convection_sigmas
         use star_utils, only: update_time, total_times, FL_to_L

         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin
         logical, intent(in) :: skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt 
         character (len=64) :: hydro_decsol
         integer :: lid, lrd, ierr, nvar, nz, k, mljac, mujac, n, nzmax
         integer :: time0, clock_rate, hydro_lwork, hydro_liwork, hydro_lqwork
         real(dp) :: total_all_before
         real(dp) :: sparse_nzmax_factor
         logical :: report, dumping, numerical_jacobian
         
         include 'formats.dek'
         
         do_hydro_converge = terminate
      
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if

         ierr = 0
         call set_thomas_blck_sprs_clip_lim(s% thomas_blck_sprs_clip_lim)
         
         if (s% do_burn .or. s% do_mix) then
            nvar = s% nvar
         else
            nvar = s% nvar_hydro
         end if
         
         nz = s% nz
         n = nz*nvar
         mljac = (stencil_neighbors+1)*nvar-1
         mujac = mljac
         nzmax = 0
         
         if (nvar <= s% hydro_decsol_switch) then
            hydro_decsol = s% small_mtx_decsol
         else
            hydro_decsol = s% large_mtx_decsol
         end if
         
         s% hydro_decsol_option = decsol_option(hydro_decsol, ierr)         
         if (ierr /= 0) then
            write(*, *) 'bad value for hydro_decsol ' // trim(hydro_decsol)
            do_hydro_converge = terminate
            return
         end if
         
         s% hydro_matrix_type = banded_matrix_type
         select case(s% hydro_decsol_option)
            case (lapack)
               call lapack_work_sizes(n, lrd, lid)
            case (block_thomas_dble)
               call block_thomas_dble_work_sizes(nvar, nz, lrd, lid)
               s% hydro_matrix_type = block_tridiag_dble_matrix_type
            case (block_thomas_quad)
               call block_thomas_quad_work_sizes(nvar, nz, lrd, lid)
               s% hydro_matrix_type = block_tridiag_quad_matrix_type
            case (block_thomas_klu)
               call block_thomas_klu_work_sizes(nvar, nz, lrd, lid)
               s% hydro_matrix_type = block_tridiag_dble_matrix_type
            case (block_dc_mt_dble)
               call block_dc_mt_dble_work_sizes(nvar, nz, lrd, lid)
               s% hydro_matrix_type = block_tridiag_dble_matrix_type
            case (block_dc_mt_quad)
               call block_dc_mt_quad_work_sizes(nvar, nz, lrd, lid)
               s% hydro_matrix_type = block_tridiag_quad_matrix_type
            case (block_dc_mt_klu)
               call block_dc_mt_klu_work_sizes(nvar, nz, lrd, lid)
               s% hydro_matrix_type = block_tridiag_dble_matrix_type
            case (mkl_pardiso)
               sparse_nzmax_factor = max(1d-10, min(1d0, s% sparse_non_zero_max_factor))
               nzmax = n*(mljac+mujac+1)*sparse_nzmax_factor
               call mkl_pardiso_work_sizes(n, nzmax, lrd, lid)
            case (klu)
               sparse_nzmax_factor = max(1d-10, min(1d0, s% sparse_non_zero_max_factor))
               nzmax = n*(mljac+mujac+1)*sparse_nzmax_factor
               call klu_work_sizes(n, nzmax, lrd, lid)
            case default
               write(*,*) 'do_hydro_converge: invalid setting for hydro_decsol_option', s% hydro_decsol_option
               do_hydro_converge = terminate
               s% result_reason = nonzero_ierr 
               return
         end select
         
         report = (s% report_hydro_solver_progress .or. s% report_ierr)
         s% hydro_call_number = s% hydro_call_number + 1
         dumping = (s% hydro_call_number == s% hydro_dump_call_number)         
         if (s% report_hydro_solver_progress) then
            write(*,*)
            write(*,2) 'hydro_call_number, dt, dt/secyer, log dt/yr', &
               s% hydro_call_number, dt, dt/secyer, log10(dt/secyer)
         end if
         
         numerical_jacobian = s% hydro_numerical_jacobian .and. &
            ((s% hydro_call_number == s% hydro_dump_call_number) &
            .or. (s% hydro_dump_call_number < 0))
      
         if (s% doing_first_model_of_run) &
            s% L_phot_old = FL_to_L(s,s% xh(s% i_FL,1))/Lsun
         
         if (s% do_mix .or. s% operator_coupling_choice == -1) then
            call get_convection_sigmas(s, dt, ierr)
            if (ierr /= 0) return
         end if
         
         if (s% rotation_flag) then
            call get_rotation_sigmas(s, 1, s% nz, dt, ierr)
            if (ierr /= 0) return
         end if
         
         if (s% v_flag) &
            s% csound_init(1:nz) = sqrt(s% gamma1(1:nz)*s% P(1:nz)/s% rho(1:nz))
         
         call alloc_for_decsol(ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'do_hydro_converge: alloc_for_decsol failed'
            do_hydro_converge = retry
            s% result_reason = nonzero_ierr 
            return
         end if
                  
         if (.not. s% doing_hydro_newton) then
            
            !do_hydro_converge = do_hydro_seulex( &
            !   s, dt, report, block_dc_mt_dble_decsolblk, &
            !   lrd, s% rpar_decsol, lid, s% ipar_decsol)
         
         
            write(*,*) 'do_hydro_converge call BiMDd'  
            call BiMDd
            stop 'do_hydro_converge'


            !stop '.not. s% doing_hydro_newton'
            do_hydro_converge = do_hydro_bimd( &
               s, dt, report, block_dc_mt_dble_decsolblk, lrd, lid)
               
            
         else
            
            call work_sizes_for_newton(ierr)
            if (ierr /= 0) then
               if (s% report_ierr) write(*, *) 'do_hydro_converge: work_sizes_for_newton failed'
               do_hydro_converge = retry
               s% result_reason = nonzero_ierr 
               return
            end if
         
            call alloc_for_newton(ierr)
            if (ierr /= 0) return
      
            do_hydro_converge = do_hydro_newton( &
               s, itermin, skip_global_corr_coeff_limit, dt, &
               report, dumping, numerical_jacobian, &
               nz, nvar, lrd, s% rpar_decsol, lid, s% ipar_decsol, &
               s% hydro_work, s% hydro_work_quad, hydro_lwork, &
               s% hydro_iwork, hydro_liwork, &
               s% hydro_qwork, hydro_lqwork)
         
         end if
         
         if (do_hydro_converge == keep_going .and. s% lnE_flag) then ! reset lnE vars
            do k=1,nz
               s% xh(s% i_lnE,k) = s% lnE(k)
            end do
         end if
         
         if (s% doing_timing) &
            call update_time(s, time0, total_all_before, s% time_struct_burn_mix)
         
         
         contains
         
         
         subroutine alloc_for_decsol(ierr)
            integer, intent(out) :: ierr
            include 'formats.dek'
            ierr = 0
            if (.not. associated(s% ipar_decsol)) then
               allocate(s% ipar_decsol(lid))
            else if (size(s% ipar_decsol, dim=1) < lid) then
               deallocate(s% ipar_decsol)
               allocate(s% ipar_decsol(int(1.3*lid)+100))
            end if
            
            if (.not. associated(s% rpar_decsol)) then
               allocate(s% rpar_decsol(lrd))
            else if (size(s% rpar_decsol, dim=1) < lrd) then
               deallocate(s% rpar_decsol)
               allocate(s% rpar_decsol(int(1.3*lrd)+100))
            end if            
         end subroutine alloc_for_decsol
         
         
         subroutine alloc_for_newton(ierr)
            integer, intent(out) :: ierr
            include 'formats.dek'
            ierr = 0
            
            if (.not. associated(s% hydro_iwork)) then
               allocate(s% hydro_iwork(hydro_liwork))
            else if (size(s% hydro_iwork, dim=1) < hydro_liwork) then
               deallocate(s% hydro_iwork)
               allocate(s% hydro_iwork(int(1.3*hydro_liwork)+100))
            end if
            
               if (.not. associated(s% hydro_work)) then
                  allocate(s% hydro_work(hydro_lwork))
               else if (size(s% hydro_work, dim=1) < hydro_lwork) then
                  deallocate(s% hydro_work)
                  allocate(s% hydro_work(int(1.3*hydro_lwork)+100))
               end if
            
            if (s% use_quad_newton) then
               if (.not. associated(s% hydro_work_quad)) then
                  allocate(s% hydro_work_quad(hydro_lwork))
               else if (size(s% hydro_work_quad, dim=1) < hydro_lwork) then
                  deallocate(s% hydro_work_quad)
                  allocate(s% hydro_work_quad(int(1.3*hydro_lwork)+100))
               end if
            end if

            if (.not. associated(s% hydro_qwork)) then
               allocate(s% hydro_qwork(hydro_lqwork))
            else if (size(s% hydro_qwork, dim=1) < hydro_lqwork) then
               deallocate(s% hydro_qwork)
               allocate(s% hydro_qwork(int(1.3*hydro_lqwork)+100))
            end if
            
         end subroutine alloc_for_newton
      
   
         subroutine work_sizes_for_newton(ierr)
            use num_lib, only: newton_quad_work_sizes, newton_dble_work_sizes
            integer, intent(out) :: ierr
            include 'formats.dek'
            ierr = 0
            if (s% use_quad_newton) then
               call newton_quad_work_sizes(mljac, mujac, nvar, nz, 0, &
                  s% hydro_matrix_type, hydro_lwork, hydro_liwork, hydro_lqwork, ierr)
            else
               call newton_dble_work_sizes(mljac, mujac, nvar, nz, 0, &
                  s% hydro_matrix_type, hydro_lwork, hydro_liwork, hydro_lqwork, ierr)
            end if
         end subroutine work_sizes_for_newton


      end function do_hydro_converge
      

      integer function do_hydro_newton( &
            s, itermin, skip_global_corr_coeff_limit, dt, &
            report, dumping, numerical_jacobian, &
            nz, nvar, lrd, rpar_decsol, lid, ipar_decsol, &
            newton_work, newton_work_quad, newton_lwork, &
            newton_iwork, newton_liwork, &
            newton_qwork, newton_lqwork)
         ! return keep_going, retry, backup, or terminate
         
         ! when using newton for hydro step, 
         ! do not require that functions have been evaluated for starting configuration.
         ! when finish, will have functions evaluated for the final set of primary variables.
         ! for example, the reaction rates will have been computed, so they can be used
         ! as initial values in the following burn and mix.
         
         use num_def
         use num_lib
         use utils_lib, only: is_bad_num, has_bad_num
         use alloc

         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nz, nvar
         logical, intent(in) :: skip_global_corr_coeff_limit, &
            report, dumping, numerical_jacobian
         real(dp), intent(in) :: dt
         integer, intent(in) :: lrd, lid
         
         real(dp), pointer :: dx_dble(:,:)
         real(qp), pointer :: dx_quad(:,:)
         integer, pointer :: ipar_decsol(:) ! (lid)
         real(dp), pointer :: rpar_decsol(:) ! (lrd)
         integer, intent(in) :: newton_lwork, newton_liwork, newton_lqwork
         real(dp), pointer :: newton_work(:) ! (newton_lwork)
         real(qp), pointer :: newton_work_quad(:) ! (newton_lwork)
         integer, pointer :: newton_iwork(:) ! (newton_liwork)         
         real(qp), pointer :: newton_qwork(:) ! (newton_lqwork)
         logical :: converged
         integer :: i, k, species, ierr, alph, j1, j2
         real(dp) :: tol_correction_norm, tol_max_correction, varscale
         logical, parameter :: check_for_bad_nums = .false.

         real(dp), parameter :: xscale_min = 1d-3

         include 'formats.dek'

         species = s% species
         do_hydro_newton = keep_going
         
         if (s% number_of_backups_in_a_row < 3) then
            tol_correction_norm = s% tol_correction_norm
            tol_max_correction = s% tol_max_correction
         else
            tol_correction_norm = s% tol_correction_norm_alt
            tol_max_correction = s% tol_max_correction_alt
         end if
         
         ! parameters for newton
         newton_iwork(1:num_iwork_params) = 0
         newton_work(1:num_work_params) = 0
         
         if ((s% doing_first_model_of_run) .or. s% model_number <= s% last_backup) &
            newton_iwork(i_try_really_hard) = 1 ! try_really_hard for 1st model or after a backup
         newton_iwork(i_itermin) = itermin
         
         newton_iwork(i_max_iter_for_enforce_resid_tol) = s% max_iter_for_resid_tol1
         newton_iwork(i_max_iter_for_resid_tol2) = s% max_iter_for_resid_tol2
         newton_iwork(i_max_iter_for_resid_tol3) = s% max_iter_for_resid_tol3
         
         if (s% refine_solution) then
            newton_iwork(i_refine_solution) = 1
         else
            newton_iwork(i_refine_solution) = 0
         end if
         
         if (s% refine_mtx_solution) then
            newton_iwork(i_refine_mtx_solution) = 1
         else
            newton_iwork(i_refine_mtx_solution) = 0
         end if
         
         newton_iwork(i_max_iterations_for_jacobian) = s% max_iterations_for_jacobian
         if (s% model_number < s% last_backup-1) then
            newton_iwork(i_max_iterations_for_jacobian) = 1
            newton_iwork(i_max_tries) = s% max_tries_after_backup2
            !write(*,*) 'use max_tries_after_backup2', s% max_tries_after_backup2
         else if (s% model_number < s% last_backup) then
            newton_iwork(i_max_iterations_for_jacobian) = 1
            newton_iwork(i_max_tries) = s% max_tries_after_backup
            !write(*,*) 'use max_tries_after_backup', s% max_tries_after_backup
         else if (s% retry_cnt > 0) then
            newton_iwork(i_max_tries) = s% max_tries_for_retry
            !write(*,*) 'use max_tries_for_retry', s% max_tries_for_retry
         else if (s% doing_first_model_of_run) then
            newton_iwork(i_max_tries) = s% max_tries1
         else
            newton_iwork(i_max_tries) = s% max_tries
         end if
         
         newton_iwork(i_tiny_corr_coeff_limit) = s% tiny_corr_coeff_limit
         if (dumping .or. s% report_hydro_solver_progress) then
            newton_iwork(i_debug) = 1
         else
            newton_iwork(i_debug) = 0
         end if
         newton_iwork(i_model_number) = s% model_number
         if (s% model_number > s% model_number_for_last_jacobian) then
            newton_iwork(i_num_solves) = 0
            newton_iwork(i_num_jacobians) = 0
         else
            newton_iwork(i_num_solves) = s% num_solves
            newton_iwork(i_num_jacobians) = s% num_jacobians         
         end if

         newton_iwork(i_min_for_check_D_norm_converging) = s% min_for_check_D_norm_converging

         newton_work(r_tol_residual_norm) = s% tol_residual_norm1
         newton_work(r_tol_max_residual) = s% tol_max_residual1
         newton_work(r_tol_residual_norm2) = s% tol_residual_norm2
         newton_work(r_tol_max_residual2) = s% tol_max_residual2
         newton_work(r_tol_residual_norm3) = s% tol_residual_norm3
         newton_work(r_tol_max_residual3) = s% tol_max_residual3

         newton_work(r_tol_max_correction) = tol_max_correction
         
         newton_work(r_target_corr_factor) = s% target_corr_factor
         newton_work(r_tol_abs_slope_min) = -1 ! unused
         newton_work(r_tol_corr_resid_product) = -1 ! unused
         
         newton_work(r_scale_correction_norm) = s% scale_correction_norm
         newton_work(r_corr_param_factor) = s% corr_param_factor
         newton_work(r_scale_max_correction) = s% scale_max_correction
         newton_work(r_corr_norm_jump_limit) = s% corr_norm_jump_limit
         newton_work(r_max_corr_jump_limit) = s% max_corr_jump_limit
         newton_work(r_resid_norm_jump_limit) = s% resid_norm_jump_limit
         newton_work(r_max_resid_jump_limit) = s% max_resid_jump_limit
         newton_work(r_slope_alert_level) = s% slope_alert_level
         newton_work(r_slope_crisis_level) = s% slope_crisis_level
         newton_work(r_tiny_corr_factor) = s% tiny_corr_factor
         newton_work(r_dt) = dt
         newton_work(r_D_norm_kappa) = s% D_norm_kappa

         if (skip_global_corr_coeff_limit) then
            newton_work(r_corr_coeff_limit) = 1       
         else
            newton_work(r_corr_coeff_limit) = s% corr_coeff_limit         
         end if

         newton_work(r_sparse_non_zero_max_factor) = s% sparse_non_zero_max_factor
         
         if (check_for_bad_nums) then
            if (has_bad_num(species*nz, s% xa)) then
               write(*, *) &
                  'bad num in xa before calling hydro_newton_step: model_number', s% model_number
               do_hydro_newton = terminate
               return
            end if
         end if
            
         if (s% use_quad_newton) then        
            dx_dble => null()         
            call get_2d_quad_array(s, dx_quad, nvar, nz, nz_alloc_extra, 'newton', ierr)
            if (ierr /= 0) return         
            dx_quad(1:s% nvar_hydro,1:nz) = 0 ! set this if want to make initial guess for hydro vars
            if (nvar >= s% i_chem1) then
               do k = 1, nz
                  j2 = 1
                  do j1 = s% i_chem1, nvar
                     dx_quad(j1,k) = s% xa(j2,k) - s% xa_pre_hydro(j2,k)
                     j2 = j2+1
                  end do
               end do
            end if
         else         
            dx_quad => null()         
            call get_2d_work_array(s, dx_dble, nvar, nz, nz_alloc_extra, 'newton', ierr)
            if (ierr /= 0) return         
            dx_dble(1:s% nvar_hydro,1:nz) = 0 ! set this if want to make initial guess for hydro vars
            if (nvar >= s% i_chem1) then
               do k = 1, nz
                  j2 = 1
                  do j1 = s% i_chem1, nvar
                     dx_dble(j1,k) = s% xa(j2,k) - s% xa_pre_hydro(j2,k)
                     j2 = j2+1
                  end do
               end do
            end if            
         end if        

         call hydro_newton_step( &
            s, nz, s% nvar_hydro, nvar, dx_dble, dx_quad, dt, &
            tol_correction_norm, numerical_jacobian, &
            lrd, rpar_decsol, lid, ipar_decsol, &
            newton_work, newton_work_quad, newton_lwork, &
            newton_iwork, newton_liwork, &
            newton_qwork, newton_lqwork, &
            converged, ierr)
         
         if (s% use_quad_newton) then
            call return_2d_quad_array(s, dx_quad, 'newton')
         else
            call return_2d_work_array(s, dx_dble, 'newton')
         end if        

         if (dumping) stop 'debug: dumping hydro_newton' 
         
         if (check_for_bad_nums) then
            if (has_bad_num(species*nz, s% xa)) then
               write(*, *) 'bad num in xa after calling hydro_newton_step: model_number', s% model_number
               do_hydro_newton = terminate
               return
            else
               write(*, *) &
                  'no bad nums in xa after calling hydro_newton_step: model_number', s% model_number
            end if
         end if
         
         if (ierr /= 0) then
            if (report) then
               write(*, *) 'hydro_newton_step returned ierr', ierr
               write(*, *) 's% model_number', s% model_number
               write(*, *) 'nz', nz
               write(*, *) 's% num_retries', s% num_retries
               write(*, *) 's% num_backups', s% num_backups
               write(*, *) 
            end if
            do_hydro_newton = retry
            s% result_reason = nonzero_ierr 
            return
         end if         
            
         s% num_solves = newton_iwork(i_num_solves)
         s% num_jacobians = newton_iwork(i_num_jacobians)
         s% total_num_jacobians = s% total_num_jacobians + s% num_jacobians
         
         if (converged) then ! sanity checks before accept it
            converged = check_after_converge(s, report, ierr)
         end if

         if (.not. converged) then
            do_hydro_newton = retry
            s% result_reason = hydro_failed_to_converge
            if (report) then
               write(*, *) 'hydro_newton_step failed to converge'
               write(*,2) 's% model_number', s% model_number
               write(*,2) 's% hydro_call_number', s% hydro_call_number
               write(*,2) 'nz', nz
               write(*,2) 's% num_retries', s% num_retries
               write(*,2) 's% num_backups', s% num_backups
               write(*,2) 's% number_of_backups_in_a_row', s% number_of_backups_in_a_row
               write(*,1) 'log dt/secyer', log10(dt/secyer)
               write(*, *) 
            end if
            return
         end if

      end function do_hydro_newton
      
      
      logical function check_after_converge(s, report, ierr) result(converged)
         type (star_info), pointer :: s
         logical, intent(in) :: report
         integer, intent(out) :: ierr
         integer :: k, nz
         include 'formats.dek'
         ierr = 0
         nz = s% nz
         converged = .true.
         if (s% L(1) <= 0) then
            if (report) write(*,*) 'after hydro, negative surface luminosity'
            converged = .false.
            return
         end if
         if (s% R_center > 0) then
            if (s% R_center > exp(s% lnR(nz))) then
               if (report) &
                  write(*,2) 'volume < 0 in cell nz', nz, &
                     s% R_center - exp(s% lnR(nz)), s% R_center, exp(s% lnR(nz)), &
                     s% dm(nz), s% rho(nz), s% dq(nz)
               converged = .false.
               return
            end if
         end if
         do k=1,nz-1
            if (s% lnR(k) <= s% lnR(k+1)) then
               if (report) write(*,2) 'after hydro, negative cell volume in cell k', &
                     k, s% lnR(k) - s% lnR(k+1), s% lnR(k), s% lnR(k+1), & 
                     s% lnR_pre_hydro(k) - s% lnR_pre_hydro(k+1), s% lnR_pre_hydro(k), s% lnR_pre_hydro(k+1)
               converged = .false.; exit
               stop 'check_after_converge'
            else if (s% lnT(k) > ln10*12) then
               if (report) write(*,2) 'after hydro, logT > 12 in cell k', k, s% lnT(k)
               converged = .false.!; exit
            else if (s% lnT(k) < ln10) then
               if (report) write(*,*) 'after hydro, logT < 1 in cell k', k
               converged = .false.!; exit
            else if (s% lnd(k) > ln10*12) then
               if (report) write(*,*) 'after hydro, logRho > 12 in cell k', k
               converged = .false.!; exit
            else if (s% lnd(k) < -ln10*20) then
               if (report) write(*,*) 'after hydro, logRho < -20 in cell k', k
               converged = .false.!; exit
            end if
         end do
      end function check_after_converge
      
      
      subroutine hydro_newton_step( &
            s, nz, nvar_hydro, nvar, dx_dble, dx_quad, dt, &
            tol_correction_norm, numerical_jacobian, &
            lrd, rpar_decsol, lid, ipar_decsol, &
            newton_work, newton_work_quad, newton_lwork, &
            newton_iwork, newton_liwork, &
            newton_qwork, newton_lqwork, &
            converged, ierr)
	      use num_def
	      use num_lib
	      use chem_def
         use mtx_lib
         use mtx_def
         use alloc
         use hydro_mtx_dble, only: ipar_id, ipar_first_call, hydro_lipar, &
            rpar_dt, hydro_lrpar

         type (star_info), pointer :: s         
         integer, intent(in) :: nz, nvar_hydro, nvar
         real(dp), pointer :: dx_dble(:,:)
         real(qp), pointer :: dx_quad(:,:)
         real(dp), intent(in) :: dt
         real(dp), intent(in) :: tol_correction_norm
         logical, intent(in) :: numerical_jacobian
         integer, intent(in) :: lrd, lid
         integer, intent(inout), pointer :: ipar_decsol(:) ! (lid)
         real(dp), intent(inout), pointer :: rpar_decsol(:) ! (lrd)
         integer, intent(in) :: newton_lwork, newton_liwork, newton_lqwork
         real(dp), intent(inout), pointer :: newton_work(:) ! (newton_lwork)
         real(qp), intent(inout), pointer :: newton_work_quad(:) ! (newton_lwork)
         integer, intent(inout), pointer :: newton_iwork(:) ! (newton_liwork)
         real(qp), intent(inout), pointer :: newton_qwork(:) ! (newton_lqwork)
         logical, intent(out) :: converged
         integer, intent(out) :: ierr

         integer, parameter :: lipar=hydro_lipar, lrpar=hydro_lrpar
         integer, target :: ipar_target(lipar)
         real(dp), target :: rpar_target(lrpar)
         integer, pointer :: ipar(:)
         real(dp), pointer :: rpar(:)

         integer :: mljac, mujac, ldysec, i, k, j, matrix_type
         logical :: failure
         real(dp) :: varscale
         real(dp), parameter :: xscale_min = 1
         integer, parameter :: nsec = 0
         real(dp), target :: ysec_dble_arry(nz, nsec)
         real(qp), target :: ysec_quad_arry(nz, nsec)
         real(dp), pointer :: ysec_dble(:,:), dx_init_dble(:,:), x_scale_dble(:,:)
         real(qp), pointer :: ysec_quad(:,:), dx_init_quad(:,:), x_scale_quad(:,:)
         
         logical, parameter :: dbg = .false.

         include 'formats.dek'
         
         ldysec = nz

         ierr = 0
         
         ysec_dble => ysec_dble_arry
         ysec_quad => ysec_quad_arry
         
         if (dbg) write(*, *) 'enter hydro_newton_step'
         
         s% numerical_jacobian = numerical_jacobian
                  
         mljac = 2*nvar-1
         mujac = mljac
         
         ipar => ipar_target
         ipar(ipar_id) = s% id
         ipar(ipar_first_call) = 1

         rpar => rpar_target         
         rpar(rpar_dt) = dt
         
         call check_sizes(s, ierr)
         if (ierr /= 0) then
            write(*,*) 'check_sizes failed'
            return
         end if

         if (s% use_quad_newton) then
            call get_2d_quad_array(s, x_scale_quad, nvar, nz, nz_alloc_extra, 'hydro_newton_step', ierr)
            if (ierr /= 0) return
            call get_2d_quad_array(s, dx_init_quad, nvar, nz, nz_alloc_extra, 'hydro_newton_step', ierr)
            if (ierr /= 0) return
         else
            call get_2d_work_array(s, x_scale_dble, nvar, nz, nz_alloc_extra, 'hydro_newton_step', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, dx_init_dble, nvar, nz, nz_alloc_extra, 'hydro_newton_step', ierr)
            if (ierr /= 0) return
         end if

         do i = 1, nvar
            if (i <= s% nvar_hydro) then
               varscale = maxval(abs(s% xh(i,1:nz)))
               varscale = max(xscale_min, varscale)
            else
               varscale = 1
            end if
            if (s% use_quad_newton) then
               x_scale_quad(i, 1:nz) = varscale
            else
               x_scale_dble(i, 1:nz) = varscale
            end if
         end do
         
         if (s% use_quad_newton) then
            do k = 1, nz
               do j = 1, nvar
                  dx_init_quad(j,k) = dx_quad(j,k)
               end do
            end do
         else
            do k = 1, nz
               do j = 1, nvar
                  dx_init_dble(j,k) = dx_dble(j,k)
               end do
            end do
         end if
         
         if (s% matrix_type /= 0) then ! s% matrix_type is a control parameter
            matrix_type = s% matrix_type
         else ! s% hydro_matrix_type is the matrix_type currently in use.
            if (s% hydro_matrix_type <= 0) then
               if (s% hydro_decsol_option == block_dc_mt_dble .or. &
                   s% hydro_decsol_option == block_thomas_dble) then
                  s% hydro_matrix_type = block_tridiag_dble_matrix_type
               else if (s% hydro_decsol_option == block_dc_mt_quad .or. &
                   s% hydro_decsol_option == block_thomas_quad) then
                  s% hydro_matrix_type = block_tridiag_quad_matrix_type
               else
                  s% hydro_matrix_type = banded_matrix_type
               end if
            end if
            if (s% use_quad_newton) s% hydro_matrix_type = block_tridiag_quad_matrix_type
            matrix_type = s% hydro_matrix_type
         end if
            			
         if (dbg) write(*, *) 'call newton'
         select case(s% hydro_decsol_option)
         
            case (lapack)
               call newt(lapack_decsol, null_decsolblk, null_decsols, null_decsolblk_quad, ierr)
               
            case (block_thomas_dble)
               if (matrix_type /= block_tridiag_dble_matrix_type) then
                  write(*,'(a)') 'matrix_type must be block_tridiag_dble_matrix_type for block_thomas_dble'
                  ierr = -1
                  return
               end if
               call newt(null_decsol, block_thomas_dble_decsolblk, null_decsols, null_decsolblk_quad, ierr)
               
            case (block_thomas_quad)
               if (matrix_type /= block_tridiag_quad_matrix_type) then
                  write(*,'(a)') 'matrix_type must be block_tridiag_quad_matrix_type for block_thomas_quad'
                  ierr = -1
                  return
               end if
               call newt(null_decsol, null_decsolblk, null_decsols, block_thomas_quad_decsolblk, ierr)
               
            case (block_thomas_klu)
               if (matrix_type /= block_tridiag_dble_matrix_type) then
                  write(*,'(a)') 'matrix_type must be block_tridiag_dble_matrix_type for block_thomas_klu'
                  ierr = -1
                  return
               end if
               call newt(null_decsol, block_thomas_klu_decsolblk, null_decsols, null_decsolblk_quad, ierr)
               
            case (block_dc_mt_dble)
               if (matrix_type /= block_tridiag_dble_matrix_type) then
                  write(*,'(a)') 'matrix_type must be block_tridiag_dble_matrix_type for block_dc_mt_dble'
                  ierr = -1
                  return
               end if
               call newt(null_decsol, block_dc_mt_dble_decsolblk, null_decsols, null_decsolblk_quad, ierr)
               
            case (block_dc_mt_quad)
               if (matrix_type /= block_tridiag_quad_matrix_type) then
                  write(*,'(a)') 'matrix_type must be block_tridiag_quad_matrix_type for block_dc_mt_quad'
                  ierr = -1
                  return
               end if
               call newt(null_decsol, null_decsolblk, null_decsols, block_dc_mt_quad_decsolblk, ierr)
               
            case (block_dc_mt_klu)
               if (matrix_type /= block_tridiag_dble_matrix_type) then
                  write(*,'(a)') 'matrix_type must be block_tridiag_dble_matrix_type for block_dc_mt_klu'
                  ierr = -1
                  return
               end if
               call newt(null_decsol, block_dc_mt_klu_decsolblk, null_decsols, null_decsolblk_quad, ierr)

            case (mkl_pardiso)
            
               call newt(null_decsol, null_decsolblk, mkl_pardiso_decsols, null_decsolblk_quad, ierr)
               
            !case (klu)
               
               !call newt(null_decsol, null_decsolblk, klu_decsols, null_decsolblk_quad, ierr)
               
            case default
            
               write(*,*) 'invalid hydro_decsol_option', s% hydro_decsol_option
               ierr = -1
               
         end select

         if (ierr /= 0 .and. s% report_ierr) then
            write(*,*) 'newton failed for hydro'
         end if
         
         converged = (ierr == 0) .and. (.not. failure)            
         if (converged) then         
            if (s% use_quad_newton) then
               do k=1,nz
                  do j=1,nvar_hydro
                     s% xh(j,k) = s% xh_pre_hydro(j,k) + dx_quad(j,k)
                  end do
               end do
            else
               do k=1,nz
                  do j=1,nvar_hydro
                     s% xh(j,k) = s% xh_pre_hydro(j,k) + dx_dble(j,k)
                  end do
               end do
            end if
            ! s% xa has already been updated by final call to set_newton_vars from newton solver
         end if
                  
         if (s% use_quad_newton) then
            call return_2d_quad_array(s, x_scale_quad, 'hydro_newton_step')            
            call return_2d_quad_array(s, dx_init_quad, 'hydro_newton_step')            
         else
            call return_2d_work_array(s, x_scale_dble, 'hydro_newton_step')            
            call return_2d_work_array(s, dx_init_dble, 'hydro_newton_step')            
         end if
         
         
         contains
         
         
         subroutine newt(decsol, decsolblk, decsols, decsolblk_quad, ierr)
            use chem_def
            use hydro_newton_procs_dble
            use star_utils, only: total_times
            interface
               include "mtx_decsol.dek"
               include "mtx_decsolblk_dble.dek"
               include "mtx_decsols.dek"
               include "mtx_decsolblk_quad.dek"
            end interface
            integer, intent(out) :: ierr
            integer :: time0, time1, clock_rate
            integer :: k, j
            real(dp) :: total_other_time, total_mtx_time, total_all_before, &
               total_all_after, time_callbacks, elapsed_time, time_self
            include 'formats.dek'
            
            if (s% doing_timing) then
               call system_clock(time0,clock_rate)
               newton_work(r_mtx_time) = 1
               newton_work(r_test_time) = 1
               total_all_before = total_times(s)
            else
               newton_work(r_mtx_time) = 0
               newton_work(r_test_time) = 0
               total_all_before = 0
            end if
            
            newton_iwork(i_caller_id) = s% id
            
            if (s% use_quad_newton) then
               call newt_quad(decsol, decsolblk, decsols, decsolblk_quad, ierr)
            else
               call newton_dble( &
                  nz, nvar, dx_dble, dx_init_dble, &
                  matrix_type, mljac, mujac, &
                  decsol, decsolblk, decsols, decsolblk_quad, &
                  lrd, rpar_decsol, lid, ipar_decsol, &
                  s% hydro_decsol_option, tol_correction_norm, &
                  default_set_primaries, default_set_secondaries, &
                  set_xscale, default_Bdomain, default_xdomain, eval_equations, sizequ, &
                  sizeB, inspectB, enter_setmatrix, exit_setmatrix, &
                  default_failed_in_setmatrix, force_another_iteration, &
                  x_scale_dble, s% equ_dble, ldysec, nsec, ysec_dble, &
                  newton_work, newton_lwork, &
                  newton_iwork, newton_liwork, &
                  newton_qwork, newton_lqwork, &
                  s% AF_dble, lrpar, rpar, lipar, ipar, failure, ierr)
            end if
               
            s% D_norm_err_est = newton_work(r_D_norm_err_est)
            
            if (s% doing_timing) then ! subtract time_newton_mtx
               call system_clock(time1,clock_rate)
               total_all_after = total_times(s)
               time_callbacks = total_all_after - total_all_before
               elapsed_time = dble(time1-time0)/clock_rate
               total_other_time = elapsed_time - time_callbacks
               total_mtx_time = newton_work(r_mtx_time)
               time_self = total_other_time - total_mtx_time
               s% time_newton_self = s% time_newton_self + time_self
               s% time_newton_mtx = s% time_newton_mtx + total_mtx_time
               s% time_newton_test = s% time_newton_test + &
                  newton_work(r_test_time) - (total_mtx_time + time_callbacks)
            end if

         end subroutine newt
         
         
         subroutine newt_quad(decsol, decsolblk, decsols, decsolblk_quad, ierr)
            use chem_def
            use hydro_newton_procs_quad
            use star_utils, only: total_times
            interface
               include "mtx_decsol.dek"
               include "mtx_decsolblk_dble.dek"
               include "mtx_decsols.dek"
               include "mtx_decsolblk_quad.dek"
            end interface
            integer, intent(out) :: ierr
            real(fltp), parameter :: one = 1
            include 'formats.dek'
            
            newton_work_quad(1:num_work_params) = newton_work(1:num_work_params)
            call newton_quad( &
               nz, nvar, dx_quad, dx_init_quad, &
               matrix_type, mljac, mujac, &
               decsolblk_quad, &
               lrd, rpar_decsol, lid, ipar_decsol, &
               s% hydro_decsol_option, one*tol_correction_norm, &
               default_set_primaries_quad, default_set_secondaries_quad, &
               set_xscale, default_Bdomain_quad, default_xdomain_quad, &
               eval_equations, sizequ, sizeB, inspectB, enter_setmatrix, exit_setmatrix, &
               default_failed_in_setmatrix_quad, force_another_iteration, &
               x_scale_quad, s% equ_quad, ldysec, nsec, ysec_quad, &
               newton_work_quad, newton_lwork, &
               newton_iwork, newton_liwork, &
               newton_qwork, newton_lqwork, &
               s% AF_quad, lrpar, rpar, lipar, ipar, failure, ierr)
            newton_work(1:num_work_params) = newton_work_quad(1:num_work_params)

         end subroutine newt_quad
         
      
      end subroutine hydro_newton_step
      
      
      ! about to declare victory... but may want to do another iteration
      ! 1 means force another iteration
      ! 0 means don't need to force another
      ! -1 means failure. newton returns with non-convergence.
      integer function force_another_iteration(iter, itermin, lrpar, rpar, lipar, ipar)
         use hydro_mtx_dble, only: ipar_id
         integer, intent(in) :: iter ! have finished this many iterations and have converged
         integer, intent(in) :: itermin ! this is the requested minimum.  iter may be < itermin.
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)

         type (star_info), pointer :: s   
         integer :: id, ierr, k
         
         include 'formats.dek'
         
         if (iter < itermin) then
            force_another_iteration = 1
         else
            force_another_iteration = 0
         end if

         id = ipar(ipar_id)
         ierr = 0

         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            force_another_iteration = -1
            return
         end if
         
         if (s% L(1) <= 0) then
            !write(*,1) 'Lsurf/Lsun', s% L(1)/Lsun
            force_another_iteration = 1
            return
         end if
         
      end function force_another_iteration

      

      end module solve_hydro


