! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module rotation_mix_info

      use const_def
      use num_lib
      use utils_lib
      use star_private_def
      use mlt_def
      use star_utils, only: dq_smooth_nonconv, smooth

      implicit none
      
      
      
      real(dp), parameter :: Ri_crit = 0.25d0 ! critical Richardson number
      real(dp), parameter :: R_crit = 2500d0 ! critical Reynolds number
      
      integer, parameter :: i_DSI = 1
      integer, parameter :: i_SH = i_DSI + 1
      integer, parameter :: i_SSI = i_SH + 1
      integer, parameter :: i_ES = i_SSI + 1
      integer, parameter :: i_GSF = i_ES + 1
      integer, parameter :: i_ST = i_GSF + 1
      integer, parameter :: num_instabilities = i_ST
      
      
      
      contains     


      subroutine set_rotation_mixing_info(s, ierr)
         
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         real(dp) :: f_mu, q
         real(dp), pointer :: smooth_work(:,:)
         ! the following are all defined at cell boundaries
         real(dp), dimension(:), pointer :: & ! just copies of pointers
            r, m, L, j_rot, gradT, grada, grav
         real(dp), dimension(:), pointer :: & ! allocated temporary storage
            csound, rho, T, P, cp, chiRho, abar, zbar, gradT_sub_grada, &
            opacity, gamma1, mu_alt, eps_tot, delta, &
            scale_height, &
            omega, &
            dRho, dr, dPressure, domega, d_mu, d_j_rot, &
            dRho_dr, dRho_dr_ad, dr2omega, H_T, &
            domega_dlnR, Hj, dlnR_domega, &
            t_dyn, t_kh, &
            Ri_mu, Ri_T, Ri, &
            ve0, ve_mu, &
            v_ssi, h_ssi, visc, Ris_1, Ris_2, &
            v_es, H_es, &
            v_gsf, H_gsf, &
            N2, N2_mu, &
            work, work2


            
         logical, pointer :: unstable(:,:) ! (num_instabilities, nz)
         
         integer :: nz, i, j, k, isoftmu, which
         logical :: okay
         
         ierr = 0
         nz = s% nz
         
         call setup(ierr)
         if (failed('setup for set_rotation_mixing_info')) return

         s% D_DSI(1:nz) = 0
         s% D_SH(1:nz) = 0
         s% D_SSI(1:nz) = 0
         s% D_ES(1:nz) = 0
         s% D_GSF(1:nz) = 0
         s% D_ST(1:nz) = 0
         s% nu_ST(1:nz) = 0
         s% dynamo_B_r(1:nz) = 0
         s% dynamo_B_phi(1:nz) = 0

         unstable(:,1:nz) = .false.
         okay = .true.
         
!$OMP PARALLEL DO PRIVATE(which, k, q)
         do which = 1, num_instabilities
         
            if (.not. okay) cycle
            
            select case (which)
            
               case (i_DSI)
               
                  if (s% D_DSI_factor > 0) then
                     call set_D_DSI(ierr)
                     if (failed('set_D_DSI')) okay = .false.
                     call dq_smooth_nonconv(s, s% dq_smooth_DSI, s% D_DSI, smooth_work(:,which))
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% D_DSI)
                     call zero_if_tiny(s,s% D_DSI)
                  end if

               case (i_SH)
      
                  if (s% D_SH_factor > 0) then
                     call set_D_SH(ierr)
                     if (failed('set_D_SH')) okay = .false.
                     call dq_smooth_nonconv(s, s% dq_smooth_SH, s% D_SH, smooth_work(:,which))
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% D_SH)
                     call zero_if_tiny(s,s% D_SH)
                  end if

               case (i_SSI)
      
                  if (s% D_SSI_factor > 0) then
                     call set_D_SSI(ierr)
                     if (failed('set_D_SSI')) okay = .false.
                     call dq_smooth_nonconv(s, s% dq_smooth_SSI, s% D_SSI, smooth_work(:,which))
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% D_SSI)
                     call zero_if_tiny(s,s% D_SSI)
                  end if
      
               case (i_ES)
               
                  if (s% D_ES_factor > 0) then
                     call set_D_ES(ierr)
                     if (failed('set_D_ES')) okay = .false.
                     call dq_smooth_nonconv(s, s% dq_smooth_ES, s% D_ES, smooth_work(:,which))
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% D_ES)
                     call zero_if_tiny(s,s% D_ES)
                  end if
      
               case (i_GSF)
      
                  if (s% D_GSF_factor > 0) then
                     call set_D_GSF(ierr)
                     if (failed('set_D_GSF')) okay = .false.
                     call dq_smooth_nonconv(s, s% dq_smooth_GSF, s% D_GSF, smooth_work(:,which))
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% D_GSF)
                     call zero_if_tiny(s,s% D_GSF)
                  end if

               case (i_ST)
               
                  if (s% D_ST_factor > 0 .or. s% nu_ST_factor > 0) then
                     call set_ST(ierr)
                     if (failed('set_ST')) okay = .false.         
                     call dq_smooth_nonconv(s, s% dq_smooth_ST, s% D_ST, smooth_work(:,which))
                     call dq_smooth_nonconv(s, s% dq_smooth_ST, s% nu_ST, smooth_work(:,which))
                     ! calculate B_r and B_phi
                     do k = 1, nz
                        q = max(1d-30,min(1d30,abs(domega_dlnr(k)/omega(k)))) ! shear            
                        s% dynamo_B_r(k) = & ! eqn 11, H05
                           ((4.D0*pi*rho(k)*s% nu_ST(k)*q/r(k))**2*omega(k)*s% nu_ST(k))**0.25D0
                        s% dynamo_B_phi(k) = & ! eqn 12, H05
                           ((4.D0*pi*rho(k)*omega(k)*q*r(k))**2*omega(k)*s% nu_ST(k))**0.25d0
                     end do
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% D_ST)
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% nu_ST)
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% dynamo_B_r)
                     call zero_if_convective(nz, s% mixing_type, s% D_mix, s% dynamo_B_phi)
                     call zero_if_tiny(s,s% D_ST)
                  end if
                  
               case default
                  stop 'bad case for rotation_mix_info'
                  
            end select
         
         end do
!$OMP END PARALLEL DO
                        
         call dealloc

                  
         contains         
         

         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            logical, pointer :: l(:)
            call get_work_array(s, csound, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, rho, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, T, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, P, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, cp, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, chiRho, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, abar, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, zbar, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, gradT_sub_grada, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, opacity, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, gamma1, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, mu_alt, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, omega, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, eps_tot, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dRho, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dr, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dPressure, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, domega, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, d_j_rot, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, d_mu, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dRho_dr, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dRho_dr_ad, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dr2omega, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, domega_dlnR, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnR_domega, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, delta, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, Ri, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, Ri_mu, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, Ri_T, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, t_dyn, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, t_kh, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, ve0, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, ve_mu, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, scale_height, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, H_T, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, v_ssi, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, h_ssi, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, visc, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, Hj, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, Ris_1, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, Ris_2, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, v_es, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, H_es, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, v_gsf, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, H_gsf, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, N2, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, N2_mu, nz, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return            
            call get_2d_work_array(s, smooth_work, nz, num_instabilities, nz_alloc_extra, 'rotation_mix_info', ierr)
            if (ierr /= 0) return
            call get_logical_work_array(s, l, nz*num_instabilities, nz_alloc_extra, ierr)
            if (ierr /= 0) return
            call set_logical_pointer_2(unstable, l, num_instabilities, nz)
         end subroutine do_alloc
         
         
         subroutine dealloc
            use alloc
            use utils_lib
            logical, pointer :: l(:)
            call return_work_array(s, csound, 'rotation_mix_info')            
            call return_work_array(s, rho, 'rotation_mix_info')            
            call return_work_array(s, T, 'rotation_mix_info')            
            call return_work_array(s, P, 'rotation_mix_info')            
            call return_work_array(s, cp, 'rotation_mix_info')            
            call return_work_array(s, chiRho, 'rotation_mix_info')            
            call return_work_array(s, abar, 'rotation_mix_info')            
            call return_work_array(s, zbar, 'rotation_mix_info')            
            call return_work_array(s, gradT_sub_grada, 'rotation_mix_info')            
            call return_work_array(s, opacity, 'rotation_mix_info')            
            call return_work_array(s, gamma1, 'rotation_mix_info')            
            call return_work_array(s, mu_alt, 'rotation_mix_info')            
            call return_work_array(s, omega, 'rotation_mix_info')            
            call return_work_array(s, eps_tot, 'rotation_mix_info')            
            call return_work_array(s, dRho, 'rotation_mix_info')            
            call return_work_array(s, dr, 'rotation_mix_info')            
            call return_work_array(s, dPressure, 'rotation_mix_info')            
            call return_work_array(s, domega, 'rotation_mix_info')            
            call return_work_array(s, d_j_rot, 'rotation_mix_info')            
            call return_work_array(s, d_mu, 'rotation_mix_info')            
            call return_work_array(s, dRho_dr, 'rotation_mix_info')            
            call return_work_array(s, dRho_dr_ad, 'rotation_mix_info')            
            call return_work_array(s, dr2omega, 'rotation_mix_info')            
            call return_work_array(s, domega_dlnR, 'rotation_mix_info')            
            call return_work_array(s, dlnR_domega, 'rotation_mix_info')            
            call return_work_array(s, delta, 'rotation_mix_info')            
            call return_work_array(s, Ri, 'rotation_mix_info')            
            call return_work_array(s, Ri_mu, 'rotation_mix_info')            
            call return_work_array(s, Ri_T, 'rotation_mix_info')            
            call return_work_array(s, t_dyn, 'rotation_mix_info')            
            call return_work_array(s, t_kh, 'rotation_mix_info')            
            call return_work_array(s, ve0, 'rotation_mix_info')            
            call return_work_array(s, ve_mu, 'rotation_mix_info')            
            call return_work_array(s, scale_height, 'rotation_mix_info')            
            call return_work_array(s, H_T, 'rotation_mix_info')            
            call return_work_array(s, v_ssi, 'rotation_mix_info')            
            call return_work_array(s, h_ssi, 'rotation_mix_info')            
            call return_work_array(s, visc, 'rotation_mix_info')            
            call return_work_array(s, Hj, 'rotation_mix_info')            
            call return_work_array(s, Ris_1, 'rotation_mix_info')            
            call return_work_array(s, Ris_2, 'rotation_mix_info')            
            call return_work_array(s, v_es, 'rotation_mix_info')            
            call return_work_array(s, H_es, 'rotation_mix_info')            
            call return_work_array(s, v_gsf, 'rotation_mix_info')            
            call return_work_array(s, H_gsf, 'rotation_mix_info')            
            call return_work_array(s, N2, 'rotation_mix_info')            
            call return_work_array(s, N2_mu, 'rotation_mix_info')            
            call return_2d_work_array(s, smooth_work, 'rotation_mix_info')
            call set_logical_pointer_1(l, unstable, nz*num_instabilities)
            call return_logical_work_array(s, l)
         end subroutine dealloc

         
         subroutine setup(ierr)
            integer, intent(out) :: ierr
            integer :: i, k, j
            real(dp) :: &
               bracket_term, ri0, alfa, beta, &
               eps_totm1, eps_tot00, scale_height2, dlnRho_dlnP, dlnT_dlnP
               
            include 'formats.dek'
            
            ierr = 0
         
            f_mu = s% am_gradmu_factor
            if (f_mu < 0) then
               isoftmu = 1
               f_mu = -f_mu
            else
               isoftmu = 0
            end if
            
            ! copy some pointers
            r => s% r
            m => s% m
            L => s% L
            j_rot => s% j_rot
            gradT => s% gradT
            grada => s% grada_at_face
            grav => s% grav
            
            call do_alloc(ierr)
            if (ierr /= 0) return
            
            ! interpolate by mass to get values at cell boundaries
            eps_tot00 = s% eps_nuc(1) - s% non_nuc_neu(1)
               ! skip eps_grav, extra_heat, and irradiation_heat
               ! because they haven't been evaluated yet when this is called.
            eps_tot(1) = eps_tot00
            csound(1) = s% csound(1)
            rho(1) = s% rho(1)
            T(1) = s% T(1)
            P(1) = s% P(1)
            cp(1) = s% cp(1)
            chiRho(1) = s% chiRho(1)
            abar(1) = s% abar(1)
            zbar(1) = s% zbar(1)
            opacity(1) = s% opacity(1)
            gamma1(1) = s% gamma1(1)
            mu_alt(1) = s% mu_alt(1)
            delta(1) = s% chiT(1)/s% chiRho(1)
            do k = 2, nz
               alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
               beta = 1 - alfa
               eps_totm1 = eps_tot00
               eps_tot00 = s% eps_nuc(k) - s% non_nuc_neu(k)
               eps_tot(k) = alfa*eps_tot00 + beta*eps_totm1
               csound(k) = alfa*s% csound(k) + beta*s% csound(k-1)
               rho(k) = alfa*s% rho(k) + beta*s% rho(k-1)
               T(k) = alfa*s% T(k) + beta*s% T(k-1)
               P(k) = alfa*s% P(k) + beta*s% P(k-1)
               cp(k) = alfa*s% cp(k) + beta*s% cp(k-1)
               chiRho(k) = alfa*s% chiRho(k) + beta*s% chiRho(k-1)
               abar(k) = alfa*s% abar(k) + beta*s% abar(k-1)
               zbar(k) = alfa*s% zbar(k) + beta*s% zbar(k-1)
               opacity(k) = alfa*s% opacity(k) + beta*s% opacity(k-1)
               gamma1(k) = alfa*s% gamma1(k) + beta*s% gamma1(k-1)
               mu_alt(k) = alfa*s% mu_alt(k) + beta*s% mu_alt(k-1)
               delta(k) = alfa*s% chiT(k)/s% chiRho(k) + beta*s% chiT(k-1)/s% chiRho(k-1)
            end do
            
            do i = 1, nz
               gradT_sub_grada(i) = s% gradT(i) - s% grada_at_face(i)
               gradT_sub_grada(i) = & ! make sure it isn't too close to 0
                  sign(max(abs(gradT_sub_grada(i)),1d-99),gradT_sub_grada(i))
               scale_height(i) = P(i)*r(i)**2/(s% cgrav(i)*m(i)*rho(i))
               scale_height2 = sqrt(P(i)/s% cgrav(i))/rho(i)
               if (scale_height2 < scale_height(i)) scale_height(i) = scale_height2
            end do

            omega(1:nz) = s% omega(1:nz)
            call smooth(omega,nz)
            
            ! differences (at cell boundaries)
            do i = 2, nz-1
               dRho(i) = rho(i-1) - rho(i)
               dr(i) = max(1d0, 0.5D0*(r(i-1) - r(i+1)))
               dPressure(i) = min(-1d-10, P(i-1) - P(i))
               d_mu(i) = mu_alt(i-1) - mu_alt(i)
               d_j_rot(i) = max(1d-10, j_rot(i-1) - j_rot(i))
               domega(i) = 0.5D0*(omega(i-1) - omega(i+1))
            end do
            dRho(1) = 0; dRho(nz) = 0
            dr(1) = max(1d0,r(1)-r(2)); dr(nz) = r(nz)-s% R_center
            dPressure(1) = 0; dPressure(nz) = 0
            d_mu(1) = 0; d_mu(nz) = 0
            d_j_rot(1) = 0; d_j_rot(nz) = 0
            domega(1) = 0; domega(nz) = 0
            
            call smooth(dRho,nz)
            call smooth(dr,nz)
            call smooth(dPressure,nz)
            call smooth(d_mu,nz)
            call smooth(d_j_rot,nz)
            call smooth(domega,nz)
         
            do i = 2, nz-1            
               dRho_dr(i) = dRho(i)/dr(i)
               dRho_dr_ad(i) = rho(i)*dPressure(i)/(P(i)*gamma1(i)*dr(i))
               dr2omega(i) = 4.5d0*j_rot(i)*d_j_rot(i)/dr(i) ! d(r^2 omega)^2/dr using i = (2/3)*r^2
               domega_dlnR(i) = domega(i)*r(i)/dr(i)
               if (gradT(i) > 1d-20) then
                  H_T(i) = scale_height(i)/gradT(i) ! -dr/dlnT, scale height of temperature
               else
                  H_T(i) = scale_height(i)
               endif
               Hj(i) = max(1d-10, j_rot(i)*dr(i)/d_j_rot(i))
                  ! dr/dlnj, scale height of angular momentum              
            end do
            dRho_dr(1) = 0; dRho_dr(nz) = 0
            dRho_dr_ad(1) = 0; dRho_dr_ad(nz) = 0
            dr2omega(1) = 0; dr2omega(nz) = 0
            domega_dlnR(1) = 0; domega_dlnR(nz) = 0
            H_T(1) = H_T(2); H_T(nz) = H_T(nz-1)
            Hj(1) = Hj(2); Hj(nz) = Hj(nz-1)
         
            do i = 1, nz
               dlnRho_dlnP = s% grad_density(i)
               dlnT_dlnP = s% grad_temperature(i)
               N2(i) = -grav(i)/scale_height(i)*(1/gamma1(i) - dlnRho_dlnP)
               N2_mu(i) = -grav(i)/scale_height(i)*(1/chiRho(i) - delta(i)*dlnT_dlnP - dlnRho_dlnP)
            end do
         
            call smooth(domega_dlnR,nz)
            call smooth(Hj,nz)
            call smooth(N2,nz)
            call smooth(N2_mu,nz)
            
            s% domega_dlnR(1:nz) = domega_dlnR(1:nz)
            
            do k=1,nz
               omega(k) = max(omega(k),1d-99)
            end do
            
            ! safe inverse of domega/dlnR
            do i = 2, nz-1
               dlnR_domega(i) = sign(1d0/max(abs(domega_dlnR(i)),1d-30),domega_dlnR(i))
            end do
            dlnR_domega(1) = 0; dlnR_domega(nz) = 0
            
            ! timescales
            do i = 1, nz
               t_dyn(i) = sqrt(r(i)**3/(s% cgrav(i)*m(i)))
               t_kh(i) = s% cgrav(i)*m(i)**2/(r(i)*max(1d0,L(i)-m(i)*eps_tot(i))) ! Heger 2000, eqn 37
            end do
         
            ! Richardson numbers (Heger 2000, eqn 20)
            do i = 2, nz-1
               ri0 = (rho(i)*delta(i)/P(i))*(dlnR_domega(i)*grav(i))**2
               Ri_T(i) = ri0*(-gradT_sub_grada(i))
               Ri_mu(i) = ri0*f_mu*s% gradL_composition_term(i)
            end do
            Ri_T(1) = 0; Ri_T(nz) = 0
            Ri_mu(1) = 0; Ri_mu(nz) = 0
            Ri(1:nz) = Ri_T(1:nz) + Ri_mu(1:nz)
         
            ! velocities for ES and GSF
            if (s% D_ES_factor > 0 .or. s% D_GSF_factor > 0) then
               do i = 1, nz ! Heger 2000, eqns 35 and 36
                  ! the bracket_term blows up at center since r^2/L and r^2/m can both -> Inf
                  ! so bullet proof by including lower bounds
                  bracket_term = &
                     2*r(i)**2*(eps_tot(i)/max(1d-3*Lsun,abs(L(i))) - 1/max(1d-3*Msun,m(i))) - &
                     3/(4*pi*rho(i)*max(1d-3*Rsun,r(i)))        
                  ve0(i) = grada(i)*omega(i)**2*r(i)**3*L(i)*bracket_term/ &
                           (gradT_sub_grada(i)*delta(i)*(s% cgrav(i)*m(i))**2)                  
                  ve_mu(i) = (scale_height(i)/t_kh(i))* &
                           (f_mu*s% gradL_composition_term(i))/(gradT_sub_grada(i))


                  !if (.false. .and. s% m(i) < 0.1*Msun .and. s% D_ES(i) > 1d-20) then
                  if (.false. .and. s% model_number == 17149 .and. i == 492) then
                     write(*,2) 've0(i)', i, ve0(i)
                     write(*,2) 'grada(i)', i, grada(i)
                     write(*,2) 'gradT(i)', i, gradT(i)
                     write(*,2) 'gradT_sub_grada(i)', i, gradT_sub_grada(i)
                     write(*,2) 's% gradL_composition_term(i)', i, s% gradL_composition_term(i)
                     write(*,2) 'omega(i)', i, omega(i)
                     write(*,2) 'eps_tot(i)', i, eps_tot(i)
                     write(*,2) 'r(i)/Rsun', i, r(i)/Rsun
                     write(*,2) 'L(i)/Lsun', i, L(i)/Lsun
                     write(*,2) '2*r**2*eps_tot/L', i, 2*r(i)**2*eps_tot(i)/max(1d0,L(i))
                     write(*,2) '2*r**2/m', i, 2*r(i)**2/m(i)
                     write(*,2) '3/(4*pi*rho*r)', i, 3/(4*pi*rho(i)*r(i))
                     write(*,2) 'bracket_term', i, bracket_term
                     write(*,2) 'delta(i)', i, delta(i)
                     write(*,2) 'm(i)/msun', i, m(i)/msun
                     write(*,2) 've_mu(i)', i, ve_mu(i)
                     write(*,2)
                     !stop 'setup for calculation of rotation diffusion coefficients'
                  end if


                  if (is_bad_num(ve0(i))) then
                     write(*,2) 've0(i)', i, ve0(i)
                     write(*,2) 'grada(i)', i, grada(i)
                     write(*,2) 'gradT(i)', i, gradT(i)
                     write(*,2) 'gradT_sub_grada(i)', i, gradT_sub_grada(i)
                     write(*,2) 's% gradL_composition_term(i)', i, s% gradL_composition_term(i)
                     write(*,2) 'omega(i)', i, omega(i)
                     write(*,2) '2*r**2*eps_tot/L', i, 2*r(i)**2*eps_tot(i)/max(1d0,L(i))
                     write(*,2) '2*r**2/m', i, 2*r(i)**2/m(i)
                     write(*,2) '3/(4*pi*rho*r)', i, 3/(4*pi*rho(i)*r(i))
                     write(*,2)
                     
                     write(*,2) 'bracket_term', i, bracket_term
                     write(*,2) 'r(i)/Rsun', i, r(i)/Rsun
                     write(*,2) 'eps_tot(i)', i, eps_tot(i)
                     write(*,2) 's% eps_nuc(i)', i, s% eps_nuc(i)
                     write(*,2) 's% non_nuc_neu(i)', i, s% non_nuc_neu(i)
                     write(*,2) 's% eps_nuc(i-1)', i-1, s% eps_nuc(i-1)
                     write(*,2) 's% non_nuc_neu(i-1)', i-1, s% non_nuc_neu(i-1)
                     write(*,2) 'L(i)/Lsun', i, L(i)/Lsun
                     write(*,2) 'm(i)/msun', i, m(i)/msun
                     write(*,2) 'rho(i)', i, rho(i)
                     write(*,2)
                     
                     write(*,2) 'delta(i)', i, delta(i)
                     write(*,2) 've_mu(i)', i, ve_mu(i)
                     write(*,2)
                     stop 'setup for calculation of rotation diffusion coefficients'
                  end if

               end do
            end if
            
         end subroutine setup
         
         
         subroutine set_D_DSI(ierr)
            integer, intent(out) :: ierr
            integer :: i, k, kbot, ktop
            real(dp) :: instability_height, height, D
            ierr = 0
            kbot = nz
            do i = nz-1, 1, -1
               if (Ri(i) < Ri_crit .and. s% mixing_type(i) /= convective_mixing) then
                  unstable(i_DSI,i) = .true.
                  if (.not. unstable(i_DSI,i+1)) kbot = i
               end if
               if (unstable(i_DSI,i+1) .and. &
                     (i == 1 .or. .not. unstable(i_DSI,i)) .and. kbot > 1) then
                  if (unstable(i_DSI,i)) then
                     ktop = i
                  else
                     ktop = i+1
                  end if
                  instability_height = r(ktop) - r(kbot)
                  do k = ktop, kbot
                     height = min(instability_height, scale_height(k))
                     D = s% D_DSI_factor*height**2/t_dyn(k)
                     s% D_DSI(k) = min(D, scale_height(k)*csound(k))
                  end do
               end if
            end do
         end subroutine set_D_DSI
         
         
         subroutine set_D_SH(ierr) ! comment in Langer code says "DO NOT USE"
            integer, intent(out) :: ierr
            integer :: i, k, kbot, ktop
            real(dp) :: instability_height, height, D
            ierr = 0
            kbot = nz
            do i = nz-1, 1, -1
               D = grav(i)/rho(i)*(dRho_dr_ad(i)-dRho_dr(i))+dr2omega(i)/(r(i)**3)
               if (D < 0 .and. s% mixing_type(i) /= convective_mixing) then
                  unstable(i_SH,i) = .true.
                  if (.not. unstable(i_SH,i+1)) kbot = i
                  s% D_SH(i) = D ! save for later
               end if
               if (unstable(i_SH,i+1) .and. &
                     (i == 1 .or. .not. unstable(i_SH,i)) .and. kbot > 1) then
                  if (unstable(i_SH,i)) then
                     ktop = i
                  else
                     ktop = i+1
                  end if
                  instability_height = r(ktop) - r(kbot)
                  do k = ktop, kbot
                     height = min(instability_height, scale_height(k))
                     ! use the previously calculated value saved in D_SH
                     D = s% D_SH_factor*(height*s% D_SH(k)*r(k)/grav(k))**2/t_dyn(k)
                     s% D_SH(k) = min(D, scale_height(k)*csound(k))
                  end do
               end if
            end do
         end subroutine set_D_SH
         
         
         subroutine set_D_SSI(ierr)
            use chem_def
            integer, intent(out) :: ierr
            integer :: i, k, kbot, ktop
            real(dp) :: &
               D, rho6, mu_e, gamma, qe3, qe4, rm23, ctmp, xi2, &
               lambda, dynvisc, T_F, Prandtl
            integer :: ivisc
            ierr = 0
            
            ! switch if T_center < T_F
            T_F = 5.930D9*( &
               sqrt(1.D0+1.018D0*(zbar(1)/abar(1)*1.D-6*rho(1))**(2.D0/3.D0))-1d0)
            if (T(1) < T_F) then
               ivisc = 2
            else
               ivisc = 1
            end if

            qe3 = qe*qe*qe
            qe4 = qe3*qe
            
            kbot = nz
            do i = nz-1, 1, -1
               rho6 = rho(i)*1d-6
               gamma = 0.2275d0*zbar(i)**2*(rho6/abar(i))**(1d0/3d0)*1.d8/T(i)   
                  ! gamma => eq (5) of Itoh et al (1987)
                  
               ! electron viscosity according to Nandkumar & Pethick 1984 MNRAS
               mu_e = abar(i)/zbar(i)
               rm23 = (rho6/mu_e)**(2./3.)
               ctmp = 1 + 1.018*rm23
               xi2 = sqrt(pi/3.)*log(zbar(i))/3. + 2.*log(1.32+2.33/sqrt(gamma)) - &
                     0.475*(1.+2.036*rm23)/ctmp + 0.276*rm23/ctmp
               if (ivisc .eq. 1) then
                  ! Coulomb logarithm accoring to Spitzer (1962)
                  lambda = 1.50*T(i)*kerg/(qe3*zbar(i)**2)* &
                           sqrt(mp*abar(i)*kerg*T(i)/(pi*rho(i)*zbar(i)))
                  lambda = max(lambda,1.00001)
                  ! dynamic (shear) viscosity according to Spitzer (1962)
                  dynvisc = 0.406D0*sqrt(mp*abar(i)*(kerg*T(i))**5)/ &
                              (zbar(i)**4*qe4*log(lambda))
               else ! if (ivisc .eq. 2) then
                  ! dynamic shear viscosity according to Wallenborn and Bauss (1978)
                  ! See also Itoh et al 1987 ApJ 317,733
                  ! fitting formula for eta* in Eq (12) of Itoh et al. 1987
                  ctmp = -0.016321227+1.0198850*gamma**(-1.9217970) + &
                          0.024113535*gamma**(0.49999098)
                  ! dynamic shear viscosity
                  dynvisc = 5.53d3*zbar(i)*rho6**(5./6.)*ctmp/abar(i)**(1d0/3d0)
               endif

               ! add contibution of radiation
               dynvisc = dynvisc + 4.D0*crad*T(i)**4/(15.D0*clight*opacity(i)*rho(i))
               ! add contibution of electrons
               dynvisc = dynvisc + 1.893d6*rm23**(2.5)/(zbar(i)*ctmp*xi2)
               ! kinematic shear viscosity
               visc(i) = dynvisc/rho(i)
               Prandtl = dynvisc ! following Langer, we drop the rest of this!
               Ris_1(i) = 0.125D0*R_crit*Prandtl*Ri_T(i)
               if (Ris_1(i) <= Ri_crit .and. s% mixing_type(i) /= convective_mixing) then
                  Ris_2(i) = Ri_mu(i)
                  if (Ris_2(i) <= Ri_crit) then
                     unstable(i_SSI,i) = .true.
                     if (.not. unstable(i_SSI,i+1)) kbot = i
                  end if
               else
                  Ris_2(i) = 0
               end if               
               
               if (unstable(i_SSI,i+1) .and. &
                     (i == 1 .or. .not. unstable(i_SSI,i)) .and. kbot > 1) then
                  if (unstable(i_SSI,i)) then
                     ktop = i
                  else
                     ktop = i+1
                  end if
                  
                  do k = ktop, kbot ! Heger 2000, eqn 31
                     v_ssi(k)=sqrt(visc(k)/R_crit*abs(domega_dlnR(k)))
                  end do
                  
                  H_ssi(kbot) = (v_ssi(kbot-1) + v_ssi(kbot))*(r(kbot-1) - r(kbot))/ &
                                 max(1d-99,abs(v_ssi(kbot-1) - v_ssi(kbot)))
                  do k = kbot-1, ktop+1, -1
                     H_ssi(k) = (v_ssi(k+1) + v_ssi(k-1))*dr(k)/ &
                                 max(1d-99,abs(v_ssi(k+1) - v_ssi(k-1)))
                  end do
                  H_ssi(ktop) = (v_ssi(ktop) + v_ssi(ktop+1))*(r(ktop) - r(ktop+1))/ &
                                 max(1d-99,abs(v_ssi(ktop) - v_ssi(ktop+1)))

                  do k = ktop, kbot ! Heger 2000, eqn 34
                     H_ssi(k) = min(H_ssi(k),scale_height(k))
                     v_ssi(k) = min(v_ssi(k),csound(k))
                     D = s% D_SSI_factor*H_ssi(k)*v_ssi(k)* &
                           (1d0-max(0d0,max(Ris_1(k),Ris_2(k))/Ri_crit))**2
                     s% D_SSI(k) = min(D, scale_height(k)*csound(k))
                  end do
                  
               end if
               
            end do
            
         end subroutine set_D_SSI
         
         
         subroutine set_D_ES(ierr)
            integer, intent(out) :: ierr
            integer :: i, k, kbot, ktop
            real(dp) :: instability_height, D, v
            include 'formats.dek'
            ierr = 0
            kbot = nz
            do i = nz-1, 1, -1
            
               if (isoftmu == 0) then
                  v = abs(ve0(i)) - abs(ve_mu(i)) ! heger 2000, eqn 38
                  if (v > 0) then
                     unstable(i_ES,i) = .true.
                     if (.not. unstable(i_ES,i+1)) kbot = i
                     v_es(i) = v
                  else
                     v_es(i) = 0
                  end if
               else if (ve0(i) > 0) then
                  unstable(i_ES,i) = .true.
                  if (.not. unstable(i_ES,i+1)) kbot = i
                  v_es(i) = ve0(i)*exp(-abs(ve_mu(i)/ve0(i)))
               end if
               
               if (.false.) then
                  write(*,2) 've0(i)', i, ve0(i)
                  write(*,2) 've_mu(i)', i, ve_mu(i)
                  write(*,2) 'v_es(i)', i, v_es(i)
                  stop 'ES'
               end if
                              
               if (unstable(i_ES,i+1) .and. &
                     (i == 1 .or. .not. unstable(i_ES,i)) .and. kbot > 1) then
                  if (unstable(i_ES,i)) then
                     ktop = i
                  else
                     ktop = i+1
                  end if
                  instability_height = r(ktop) - r(kbot)
                  
                  ! heger 2000, eqn 39
                  H_es(kbot) = (v_es(kbot-1) + v_es(kbot))*(r(kbot-1) - r(kbot))/ &
                              max(1d-99,abs(v_es(kbot-1) - v_es(kbot)))
                  do k = kbot-1, ktop+1, -1
                     H_es(k) = (v_es(k+1) + v_es(k-1))*dr(k)/ &
                              max(1d-99,abs(v_es(k+1) - v_es(k-1)))
                  end do
                  H_es(ktop) = (v_es(ktop) + v_es(ktop+1))*(r(ktop) - r(ktop+1))/ &
                              max(1d-99,abs(v_es(ktop) - v_es(ktop+1)))

                  do k = ktop, kbot
                     if (s% mixing_type(k) == convective_mixing) cycle
                     H_es(k) = min(instability_height, H_es(k), scale_height(k))
                     v_es(k) = min(v_es(k), csound(k))
                     D = s% D_ES_factor*H_es(k)*v_es(k)
                     s% D_ES(k) = min(D, scale_height(k)*csound(k))
                     
                     if (.false. .and. s% m(k) < 0.1*Msun .and. s% D_ES(k) > 1d10) then
                        write(*,2) 's% D_ES(k)', k, s% D_ES(k)
                        write(*,2) 'log s% D_ES(k)', k, log10(s% D_ES(k))
                        write(*,2) 'H_es(k)', k, H_es(k)
                        write(*,2) 'scale_height(k)', k, scale_height(k)
                        write(*,2) 'instability_height', k, instability_height
                        write(*,2) 'v_es(k)', k, v_es(k)
                        write(*,2) 'csound(k)', k, csound(k)
                        write(*,2) 've0(k)', k, ve0(k)
                        write(*,2) 've_mu(k)', k, ve_mu(k)
                        write(*,2) 'nz', nz
                        write(*,*)
                        
                        !stop 'set_D_ES'
                     end if
                     
                  end do
                  
               end if
               
            end do
            
         end subroutine set_D_ES
         
         
         subroutine set_D_GSF(ierr)
            integer, intent(out) :: ierr
            integer :: i, k, kbot, ktop
            real(dp) :: instability_height, D, v, v_diff
            include 'formats.dek'
            ierr = 0
            kbot = nz
            do i = nz-1, 1, -1
               
               ! heger 2000, eqn 42
               v = ve0(i)*2*H_T(i)*r(i)/Hj(i)**2/(1 + 2*omega(i)*dlnR_domega(i))
               if (is_bad_num(v)) then
                  write(*,2) 'bad v for GSF', i, v
                  write(*,2) 've0(i)', i, ve0(i)
                  write(*,2) 'H_T(i)', i, H_T(i)
                  write(*,2) 'r(i)', i, r(i)
                  write(*,2) 'Hj(i)', i, Hj(i)
                  write(*,2) 'omega(i)', i, omega(i)
                  write(*,2) 'dlnR_domega(i)', i, dlnR_domega(i)
                  stop 'set_D_GSF'
                  v = 0
               end if
               if (isoftmu == 0) then
                  v_diff = abs(v) - abs(ve_mu(i)) ! heger 2000, eqn 43
                  if (v_diff > 0 .and. s% mixing_type(i) /= convective_mixing) then
                     unstable(i_GSF,i) = .true.
                     if (.not. unstable(i_GSF,i+1)) kbot = i
                     v_gsf(i) = v_diff
                  else
                     v_gsf(i) = 0
                  end if
               else 
                  if (v /= 0 .and. s% mixing_type(i) /= convective_mixing) then
                     unstable(i_GSF,i) = .true.
                     if (.not. unstable(i_GSF,i+1)) kbot = i
                     v_gsf(i) = v*exp(-abs(ve_mu(i)/v))
                  else 
                     v_gsf(i) = 0
                  end if 
               end if              
                              
               if (unstable(i_GSF,i+1) .and. &
                     (i == 1 .or. .not. unstable(i_GSF,i)) .and. kbot > 1) then
                  if (unstable(i_GSF,i)) then
                     ktop = i
                  else
                     ktop = i+1
                  end if
                  instability_height = r(ktop) - r(kbot)
                  
                  ! heger 2000, eqn 45
                  H_gsf(kbot) = (v_gsf(kbot-1) + v_gsf(kbot))*(r(kbot-1) - r(kbot))/ &
                              max(1d-99,abs(v_gsf(kbot-1) - v_gsf(kbot)))
                  do k = kbot-1, ktop+1, -1
                     H_gsf(k) = (v_gsf(k+1) + v_gsf(k-1))*dr(k)/ &
                              max(1d-99,abs(v_gsf(k+1) - v_gsf(k-1)))
                  end do
                  H_gsf(ktop) = (v_gsf(ktop) + v_gsf(ktop+1))*(r(ktop) - r(ktop+1))/ &
                              max(1d-99,abs(v_gsf(ktop) - v_gsf(ktop+1)))

                  do k = ktop, kbot
                     H_gsf(k) = min(instability_height, H_gsf(k), scale_height(k))
                     v_gsf(k) = min(v_gsf(k), csound(k))
                     D = s% D_GSF_factor*H_gsf(k)*v_gsf(k)
                     s% D_GSF(k) = min(D, scale_height(k)*csound(k))
                  end do
                  
               end if
               
            end do

         end subroutine set_D_GSF
         
         
         subroutine set_ST(ierr)
            integer, intent(out) :: ierr
            
            integer :: nz, k, j
            real(dp) :: xmagfmu, xmagft, xmagfdif, xmagfnu, &
               N2_T, Z_i, m_i, Hp, lnLambda, eta, kap, q, eta_div_r2_omega, &
               sqrt_4_pi_rho, S_0, S_1, eta_e, eta_1, eta_e1a, eta_e1b, &
               omega_div_N, q0, N_T, kap_div_r2_N, q1, S_1a, S_1b, S_m, q_m, &
               eta_m, nu_e, fq, v_conv, nu_sc

            include 'formats.dek'
         
            ierr = 0
            nz = s% nz
         
            s% D_ST(1:nz) = 0
            s% nu_ST(1:nz) = 0
            s% dynamo_B_r(1:nz) = 0
            s% dynamo_B_phi(1:nz) = 0
         
            xmagfmu = 1
            xmagft = 1
            xmagfdif = 1
            xmagfnu = 1
         
            do k = 2, nz-1
               if (s% mixing_type(k) == convective_mixing) cycle ! skip it
                  
               N2_T = N2(k) - N2_mu(k)
               Z_i = zbar(k)
               m_i = mp*abar(k)
               Hp = scale_height(k)
            
               lnLambda = log(2/(3*qe**3)*sqrt(m_i*(kerg*T(k))**3/(pi*rho(k)*Z_i**5)))
               ! until a more general description is found, (lower) limit it to 1 [AH]
               if (lnLambda < 1) lnLambda = 1
            
               kap = 16*boltz_sigma*T(k)**3/(3*opacity(k)*rho(k)**2*Cp(k)) ! thermal diffusivity
               q = max(1d-30,min(1d30,abs(domega_dlnr(k)/omega(k)))) ! shear            
               eta = 7d11*lnLambda/T(k)**1.5d0 ! magnetic diffusivity -- eqn 5
               eta_div_r2_omega = eta/(r(k)**2*omega(k))
            
               sqrt_4_pi_rho = sqrt(4*pi*rho(k))
               
               S_0 = 0; S_1 = 0; q1 = 0; q0 = 0
               eta_e = 0; eta_1 = 0; eta_e1a = 0; eta_e1b = 0
               if (N2_mu(k) > 0) then
                  omega_div_N = omega(k)/sqrt(N2_mu(k)*xmagfmu)
                  q0 = omega_div_N**(-1.5D0)*eta_div_r2_omega**0.25D0 ! eqn 39
                  if (is_bad_num(q0)) then
                     write(*,2) 'q0', k, q0
                     write(*,2) 'omega_div_N', k, omega_div_N
                     write(*,2) 'eta_div_r2_omega', k, eta_div_r2_omega
                     write(*,2) 'N2_mu(k)', k, N2_mu(k)
                     write(*,2) 'omega', k, omega
                     stop 'set_ST'
                  end if
                  S_0 = rho(k)*omega(k)**2*r(k)**2*q**3*omega_div_N**4
                  eta_e = q**4*omega_div_N**6*r(k)**2*omega(k)
               end if
            
               if (N2_T > 0) then
                  N_T = sqrt(N2_T*xmagft)
                  omega_div_N = omega(k)/N_T
                  kap_div_r2_N = kap/(r(k)**2*N_T)
                  q1 = (omega_div_N**(-7)*eta_div_r2_omega*omega_div_N*(eta/kap)**3)**0.25D0 ! eqn 40
                  S_1a = rho(k)*(omega(k)*r(k))**2*q*sqrt(omega_div_N*kap_div_r2_N)
                  S_1b = rho(k)*omega(k)**2*r(k)**2*q**3*omega_div_N**4
                  S_1 = max(S_1a,S_1b)
                  eta_e1a = r(k)**2*omega(k)*q*(omega_div_N*kap_div_r2_N)**0.75D0
                  eta_e1b = q**4*omega_div_N**6*r(k)**2*omega(k)
                  eta_1 = max(eta_e1a,eta_e1b)
               end if

               if (N2(k) < 0.0D0) then !.... convective
                  S_m = 0
                  q_m = 0
                  eta_m = 0
               else if ((N2_T > 0.D0) .and. (N2_mu(k) > 0.D0)) then !.... radiative region
                  S_m = S_0*S_1/(S_0+S_1) 
                  q_m = q0 + q1
                  eta_m = eta_e*eta_1/(eta_e+eta_1)
               else if ((N2_T <= 0.D0) .and. (N2_mu(k) > 0.D0)) then !.... semiconvection
                  S_m = S_0
                  q_m = q0
                  eta_m = eta_e
               else if ((N2_T > 0.D0) .and. (N2_mu(k) <= 0.D0)) then !.... thermohaline convection
                  S_m = S_1
                  q_m = q1
                  eta_m = eta_1
               else !.... convection
                  S_m = 0
                  q_m = 0
                  eta_m = 0
               end if

               nu_e = S_m/(omega(k)*q*rho(k)) ! eqn 2, H05
               if (is_bad_num(nu_e)) then
                  write(*,2) 'nu_e', k, nu_e
                  write(*,2) 'S_m', k, S_m
                  write(*,2) 'S_0', k, S_0
                  write(*,2) 'S_1', k, S_1
                  write(*,2) 'S_1a', k, S_1a
                  write(*,2) 'S_1b', k, S_1b
                  write(*,2) 'omega_div_N', k, omega_div_N
                  write(*,2) 'kap_div_r2_N', k, kap_div_r2_N
                  write(*,2) 'omega', k, omega(k)
                  write(*,2) 'domega_dlnr', k, domega_dlnr(k)
                  write(*,2) 'q', k, q
                  write(*,2) 'rho', k, rho(k)
                  stop 'set_ST'
               end if

               fq = max(0d0, 1d0 - min(1d0, q_m/q))
               eta_m = fq*eta_m
               nu_e = fq*nu_e

               if ((N2_T <= 0.D0) .and. (N2_mu(k) > 0.D0) .and. (N2(k) > 0.D0)) then ! semiconvective
                  v_conv = (grav(k)*delta(k)*Hp*max(0.D0,L(k))/ &
                           (64*pi*rho(k)*cp(k)*T(k)*r(k)**2))**(1d0/3d0) ! eqn 8, H05
                  nu_sc = (1d0/3d0)*Hp*v_conv ! eqn 7, H05
                  nu_e = sqrt(nu_e*nu_sc) ! eqn 9, H05
                  if (is_bad_num(nu_e)) then
                     write(*,2) 'nu_e', k, nu_e
                     write(*,2) 'nu_sc', k, nu_sc
                     stop 'set_ST'
                  end if
               end if
   
               s% D_ST(k) = min(eta_m,nu_e)  *xmagfdif
               s% nu_ST(k) = nu_e  *xmagfnu

               !if (s% D_ST(k) /= 0) write(*,2) 's% D_ST(k)', k, s% D_ST(k)

               if (is_bad_num(s% D_ST(k))) then
                  write(*,2) 's% D_ST(k)', k, s% D_ST(k)
                  write(*,2) 'fq', k, fq
                  write(*,2) 'eta_m', k, eta_m
                  write(*,2) 'nu_e', k, nu_e
                  stop 'set_ST'
               end if

                  
               !if (.false. .and. r(k) < 0.601*Rsun .and. r(k) > 0.599*Rsun .and. s% D_ST(k) > 1) then
               if (.false. .and. s% D_ST(k) > 1d60) then
                  write(*,2) 's% D_ST(k)', k, s% D_ST(k)
                  write(*,2) 'fq', k, fq
                  write(*,2) 'eta_m', k, eta_m
                  write(*,2) 'nu_e', k, nu_e
                  write(*,2) 'eta_e', k, eta_e
                  write(*,2) 'eta_1', k, eta_1
                  write(*,2) 'eta_e1a', k, eta_e1a
                  write(*,2) 'eta_e1b', k, eta_e1b
                  write(*,2) 'q', k, q
                  write(*,2) 'domega_dlnr(k)', k, domega_dlnr(k)
                  write(*,2) 'omega(k)', k, omega(k)
                  write(*,2) 'domega(k)', k, domega(k)
                  write(*,2) 'dr(k)', k, dr(k)
                  write(*,2) 'nz', nz
                  do j = max(1,k-5), min(nz,k+5)
                     write(*,2) 'omega domega domega_dlnr dr', j, omega(j), domega(j), domega_dlnr(j), dr(j)
                  end do
                  stop 'set_ST'
               end if
               
               s% D_ST(k) = s% D_ST_factor*s% D_ST(k)
               s% nu_ST(k) = s% nu_ST_factor*s% nu_ST(k)

            end do

            s% D_ST(1) = s% D_ST(2)
            s% D_ST(nz) = s% D_ST(nz-1)

            s% nu_ST(1) = s% nu_ST(2)
            s% nu_ST(nz) = s% nu_ST(nz-1)
         
         end subroutine set_ST
         
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            if (ierr == 0) then
               failed = .false.
               return
            end if
            if (s% report_ierr) &
               write(*,*) 'set_rotation_mixing_info failed in call to ' // trim(str)
            failed = .true.
         end function failed
         
         
      end subroutine set_rotation_mixing_info
      
      
      subroutine zero_if_convective(nz, mixing_type, D_mix, dc)
         integer, intent(in) :: nz
         integer, dimension(:), pointer :: mixing_type
         real(dp), dimension(:), pointer :: D_mix, dc
         integer :: k
         do k=1,nz
            if (mixing_type(k) == convective_mixing .and. dc(k) < D_mix(k)) dc(k) = 0
         end do
      end subroutine zero_if_convective
      
      
      subroutine zero_if_tiny(s, dc)
         type (star_info), pointer :: s
         real(dp), dimension(:), pointer :: dc
         integer :: k
         real(dp) :: tiny
         tiny = s% clip_D_limit
         do k=1,s% nz
            if (dc(k) < tiny) dc(k) = 0
         end do
      end subroutine zero_if_tiny



      end module rotation_mix_info
