! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module pulsation_info
      
      use star_private_def
      use const_def
      use num_lib, only: safe_log10

      implicit none
      
      
      integer, parameter :: FGONG_format = 1
      integer, parameter :: OSC_format = 2
      integer, parameter :: GYRE_format = 3
      integer, parameter :: Saio_format = 4


      contains
      
          
      subroutine save_pulsation_info( &
            id, add_atmosphere, pulsation_info_format, filename, ierr)
         use utils_lib, only: StrLowCase
         integer, intent(in) :: id
         logical, intent(in) :: add_atmosphere
         character (len=*), intent(in) :: pulsation_info_format, filename
         integer, intent(out) :: ierr
         integer :: info_format
         character (len=256) :: format_lowercase
         format_lowercase = StrLowCase(pulsation_info_format)
         if (format_lowercase == 'fgong') then
            info_format = FGONG_format
         else if (format_lowercase == 'osc') then
            info_format = OSC_format
         else if (format_lowercase == 'gyre') then
            info_format = GYRE_format
         else if (format_lowercase == 'saio') then
            info_format = Saio_format
         else
            write(*,*) 'unknown format for pulsation info ' // trim(pulsation_info_format)
            ierr = -1
            return
         end if
         call do_write_pulsation_info( &
            info_format, id, add_atmosphere, filename, ierr)
      end subroutine save_pulsation_info
            
          
      subroutine do_write_pulsation_info( &
            which_format, id, add_atmosphere, filename, ierr)
         use atm_def
         use utils_lib  
         use star_utils, only: set_xqs
         use create_atm, only: do_create_atm
         use alloc
         integer, intent(in) :: which_format, id
         logical, intent(in) :: add_atmosphere
         character (len=*), intent(in) :: filename
         integer, intent(out) :: ierr 
         
         type (star_info), pointer :: s
         integer :: NN, i, k, j, IVAR, nvars, iounit, io_txt, iconst, atm_pts
         integer :: h1, h2, he3, he4, li7, be7, be9, c12, c13, n14, n15, o16, o17, o18, ne20, si28
         integer, parameter :: IVERS = 300, IABUND = 14
         real(dp) :: alfa, beta, eps_nuc, d_eps_nuc_dlnT, d_eps_nuc_dlnd, &
            eps_grav, d_eps_grav_dlnTm1, d_eps_grav_dlnT00, d_eps_grav_dlnT, &
            d_eps_grav_dlndm1, d_eps_grav_dlnd00, d_eps_grav_dlnd, kap, Lrad, &
            r_outer, m_outer, P_prev
         real(dp), pointer :: VAR(:,:), GLOB(:)
         real(dp), pointer :: xq(:)
         logical :: fgong, osc
         
         include 'formats.dek'
         
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            write(*,*) 'bad star id for write_pulsation_info'
            return
         end if

         fgong = .false.
         osc = .false.
         if (which_format == FGONG_format) then
            fgong = .true.
         else if (which_format == OSC_format) then
            osc = .true.
         else if (which_format == GYRE_format) then
         else if (which_format == Saio_format) then
         else 
            ierr = -1
            write(*,*) 'bad value for format in do_write_pulsation_info'
            return
         end if

         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         
         io_txt = -1
         
         open(unit=iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open ' // trim(filename)
            call free_iounit(iounit)
            return
         end if
         
         if (which_format == GYRE_format) then
            nn = s% nz
            call get_work_array(s, xq, nn, 0, 'get_pulsation_info', ierr)
            if (ierr /= 0) return     
            call set_xqs(nn, xq, s% dq, ierr)       
            if (ierr /= 0) return     

            write(iounit,'(i6,99e20.12)') NN, s% m(1), s% r(1), s% L(1)
            do k=NN,1,-1
               kap = pt(s% opacity,k)
               write(iounit,'(i6,99e20.12)') NN-k+1, &
                  s% r(k), s% q(k)/max(1d-99,xq(k)), s% L(k), &
                  pt(s% P,k), pt(s% T,k), pt(s% rho,k), s% gradT(k), s% brunt_N2(k), &
                  pt(s% cv,k), pt(s% cp,k), pt(s% chiT,k), pt(s% chiRho,k), &
                  kap, pt(s% d_opacity_dlnT,k)/kap, pt(s% d_opacity_dlnd,k)/kap, &
                  pt(s% eps_nuc,k), pt(s% d_epsnuc_dlnT,k), pt(s% d_epsnuc_dlnd,k)
            end do
            
            close(iounit)
            call free_iounit(iounit)
            
            return
            
         end if
         
         if (which_format == Saio_format) then

            if (add_atmosphere) then
               call do_create_atm(s, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in do_create_atm'
                  return
               end if
               atm_pts = s% atm_structure_num_pts
               nn = s% nz + atm_pts - 2 ! skip 2 points at junction of interior and envelope
               r_outer = s% r(1) + s% atm_structure(atm_delta_r,atm_pts) 
               m_outer = s% mstar !+ s% atm_structure(atm_delta_m,atm_pts)
            else
               atm_pts = 0
               nn = s% nz
               r_outer = s% r(1)
               m_outer = s% mstar
            end if
            
            io_txt = alloc_iounit(ierr)
            if (ierr /= 0) return

            open(unit=io_txt, file = trim(filename) // '.txt', action='write', status='replace', iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed to open saio txt file'
               close(iounit)
               call free_iounit(iounit)
               return
            end if
            
            !write(*,2) 'atm_pts', atm_pts

            write(iounit,'(i6,99(1pe16.9))') &
               nn, m_outer, log10(s% L(1)/Lsun), log10(r_outer/Rsun), s% star_age
            if (.true.) write(io_txt,'(a6,99a20)') &
               'i', 'r', 'logtau', 'logT', 'logrho', &
               'logP', 'cv', 'chiRho', 'chiT', 'gradT', 'grada', &
               'kap', 'dlnkap_dlnd', 'dlnkap_dlnT', &
               'Lrad', 'X', 'Y', 'L', 'm', &
               'eps_nuc', 'deps_dlnd', 'deps_dlnT', 'logW'
            P_prev = 1d99
            do j=nn,1,-1
               call write1_saio_point(j)
            end do
            
            close(iounit)
            call free_iounit(iounit)
            
            close(io_txt)
            call free_iounit(io_txt)
            
            if (associated(s% atm_structure)) then
               deallocate(s% atm_structure)
               nullify(s% atm_structure)
            end if

            return
            
         end if
         
         call get_pulsation_info(which_format, id, add_atmosphere, nn, iconst, ivar, glob, var, ierr)
         if (ierr /= 0) return
         
         if (fgong) then
            nvars = IVAR
         else if (osc) then
            nvars = IVAR + IABUND
         else
            nvars = 0
         end if
                  
         ! header (4 lines)
         if (fgong) then
            write(iounit,*) 'FGONG file'
            write(iounit,*) 
            write(iounit,*) 
            write(iounit,*) 
         else if (osc) then
            write(iounit,*) 'OSC file'
            write(iounit,*) 
            write(iounit,*) 
            write(iounit,*) 
         end if
         
         if (osc) then ! abundance info
            write(iounit,'(a)') &
               '14 H1 H2 He3 He4 Li7 Be7 C12 C13 N14 N15 O16 O17 Be9 Si28'
            write(iounit,'(5I10)') NN, ICONST, IVAR, IABUND, IVERS
         else if (fgong) then
            write(iounit,'(4I10)') NN, ICONST, IVAR, IVERS
         end if
         
   99    format(1P5E16.9,x)
         
         if (fgong .or. osc) then
            write(iounit,99) (GLOB(i),i=1,ICONST)
            do k=1,NN
               write(iounit,99) (VAR(j,k),j=1,nvars)
            end do
         end if






         if (osc .and. s% write_pulsation_plot_data) then
            close(iounit)
            write(*,*) 'write osc_plot.data'
            write(*,2) 's% atm_structure_num_pts', s% atm_structure_num_pts
            open(unit=iounit, file = 'osc_plot.data', action='write', status='replace', iostat=ierr)
            write(iounit,'(99a24)') &
               'k', &
               'r_div_Rsun', &
               'logR', &
               'logT', &
               'logP', &
               'logRho', &
               'l_div_Lsun', &
               'brunt_N2', &
               'log_brunt_N2', &
               'r', &
               'lnq', &
               'temperature', &
               'pressure', &
               'density', &
               'gradT', &
               'luminosity', &
               'opacity', &
               'eps', &
               'gamma1', &
               'grada', &
               'chiT_div_chiRho', &
               'cp', &
               'free_e', &
               'brunt_A', &
               'omega', &
               'dlnkap_dlnT', &
               'dlnkap_dlnRho', &
               'depsnuc_dlnT', &
               'depsnuc_dlnRho', &
               'P_div_Pgas', &
               'gradr', &
               'h1', &
               'h2', &
               'he3', &
               'he4', &
               'li7', &
               'c12', &
               'c13', &
               'c14', &
               'n14', &
               'n15', &
               'o16', &
               'o17', &
               'be9', &
               'si28'
            do k=1,nn
               write(iounit,'(i24)',advance='no') k
               write(iounit,'(e24.12)',advance='no') var(1,k)/Rsun
               write(iounit,'(e24.12)',advance='no') safe_log10(var(1,k)/Rsun)
               write(iounit,'(e24.12)',advance='no') safe_log10(var(3,k))
               write(iounit,'(e24.12)',advance='no') safe_log10(var(4,k))
               write(iounit,'(e24.12)',advance='no') safe_log10(var(5,k))
               write(iounit,'(e24.12)',advance='no') var(7,k)/Lsun
               write(iounit,'(e24.12)',advance='no') var(15,k)*s% cgrav(1)*s% mstar/max(1d0,var(1,k)**3)
               write(iounit,'(e24.12)',advance='no') safe_log10(var(15,k)*s% cgrav(1)*s% mstar/max(1d0,var(1,k)**3))
               do j=1,nvars
                  write(iounit,'(e24.12)',advance='no') var(j,k)
               end do
               write(iounit,*)
            end do
         end if
                  



         if (fgong .and. s% write_pulsation_plot_data) then
            close(iounit)
            write(*,*) 'write fgong_plot.data'
            write(*,2) 's% atm_structure_num_pts', s% atm_structure_num_pts
            open(unit=iounit, file = 'fgong_plot.data', action='write', status='replace', iostat=ierr)
            write(iounit,'(99a24)') &
               'k', &
               'r_div_Rsun', &
               'logR', &
               'logT', &
               'logP', &
               'logRho', &
               'l_div_Lsun', &
               'brunt_N2', &
               'log_brunt_N2', &
               'r', &
               'lnq', &
               'temperature', &
               'pressure', &
               'density', &
               'h1', &
               'luminosity', &
               'opacity', &
               'eps', &
               'gamma1', &
               'grada', &
               'chiT_div_chiRho', &
               'cp', &
               'free_e', &
               'brunt_A', &
               'dxdt_nuc_h1', &
               'metallicity', &
               'dr', &
               'eps_grav', &
               'he3', &
               'c12', &
               'c13', &
               'n14', &
               'o16', &
               'h2', &
               'he4', &
               'li7', &
               'be7', &
               'n15', &
               'o17', &
               'o18', &
               'ne20'
            do k=1,nn
               write(iounit,'(i24)',advance='no') k
               write(iounit,'(e24.12)',advance='no') var(1,k)/Rsun
               write(iounit,'(e24.12)',advance='no') safe_log10(var(1,k)/Rsun)
               write(iounit,'(e24.12)',advance='no') safe_log10(var(3,k))
               write(iounit,'(e24.12)',advance='no') safe_log10(var(4,k))
               write(iounit,'(e24.12)',advance='no') safe_log10(var(5,k))
               write(iounit,'(e24.12)',advance='no') var(7,k)/Lsun
               write(iounit,'(e24.12)',advance='no') var(15,k)*s% cgrav(1)*s% mstar/max(1d0,var(1,k)**3)
               write(iounit,'(e24.12)',advance='no') safe_log10(var(15,k)*s% cgrav(1)*s% mstar/max(1d0,var(1,k)**3))
               do j=1,nvars
                  write(iounit,'(e24.12)',advance='no') var(j,k)
               end do
               write(iounit,*)
            end do
         end if
                  









         
         call dealloc
         
         close(iounit)
         call free_iounit(iounit)   
         
         if (fgong) then
            write(*, '(/,a, i7,/)') 'save FGONG ' // trim(filename), s% model_number
            if (s% extra_terminal_iounit > 0) &
               write(s% extra_terminal_iounit, '(/,a, i7,/)') &
                  'save FGONG' // trim(filename), s% model_number
         else if (osc) then
            write(*, '(/,a, i7,/)') 'save OSC ' // trim(filename), s% model_number
            if (s% extra_terminal_iounit > 0) &
               write(s% extra_terminal_iounit, '(/,a, i7,/)') &
                  'save OSC' // trim(filename), s% model_number
         end if
         
         
         contains
         
         
         subroutine dealloc
            deallocate(VAR,GLOB)
         end subroutine dealloc
         
         
         real(dp) function pt(v,k)
            use star_utils, only: interp_val_to_pt
            integer, intent(in) :: k
            real(dp), pointer :: v(:)
            pt = interp_val_to_pt(v,k,s% nz,s% dq)
         end function pt
         
         
         subroutine write1_saio_point(j)
            integer, intent(in) :: j
            integer :: k
            real(dp) :: &
               cgrav, r, m, Lrad, T, rho, &
               P, eps_nuc, kap, cv, chiRho, &
               chiT, d_epsnuc_dlnd, d_epsnuc_dlnT, &
               dlnKap_dlnd, dlnKap_dlnT, &
               gradT, grada_at_face, X, Y, L, tau, lnPgas
            include 'formats.dek'
            if (j < atm_pts) then ! in atmosphere
               k = atm_pts-j+1 ! k is index of point in atm_structure
                  ! we skip the point at the base of the atm to smooth the transition
               r = s% r(1) + s% atm_structure(atm_delta_r,k)
               m = s% mstar !+ s% atm_structure(atm_delta_m,k)
               T = exp(s% atm_structure(atm_lnT,k))
               rho = exp(s% atm_structure(atm_lnd,k))
               P = exp(s% atm_structure(atm_lnP,k))
               eps_nuc = 0
               kap = s% atm_structure(atm_kap,k)
               cv = s% atm_structure(atm_cv,k)
               chiRho = s% atm_structure(atm_chiRho,k)
               chiT = s% atm_structure(atm_chiT,k)
               d_epsnuc_dlnd = 0
               d_epsnuc_dlnT = 0
               dlnKap_dlnd = s% atm_structure(atm_dlnkap_dlnd,k)
               dlnKap_dlnT = s% atm_structure(atm_dlnkap_dlnT,k)
               gradT = s% atm_structure(atm_gradT,k)
               grada_at_face = s% atm_structure(atm_grada,k)
               X = pt(s% X,1)
               Y = pt(s% Y,1)
               L = s% L(1)
               cgrav = s% cgrav(1)
               tau = s% atm_structure(atm_tau,k)
               lnPgas = s% atm_structure(atm_lnPgas,k)
               !write(*,2) 'write1_saio_point: atm_tau', &
               !   k, s% atm_structure(atm_tau,k), log10(s% atm_structure(atm_tau,k))
               !if (k < 20) write(*,2) 'atmosphere P, tau', k, P, tau
               if (P >= P_prev) then
                  write(*,2) 'WARNING: atmosphere P >= P_prev', k, P, P_prev
                  write(*,'(a)') ' try decreasing create_atm_max_step_size to get more atm points'
                  !write(*,2) 's% P(1)', 1, s% P(1)
                  !stop
               end if
            else ! j > atm_pts
               if (atm_pts == 0) then
                  k = j
               else
                  k = j - atm_pts + 2 ! skip k==1
                  ! we skip the point at the top of the interior to smooth the transition
               end if
               kap = pt(s% opacity,k)
               r = s% r(k)
               m = s% m(k)
               T = pt(s% T,k)
               rho = pt(s% rho,k)
               P = pt(s% P,k)
               eps_nuc = pt(s% eps_nuc,k)
               cv = pt(s% cv,k)
               chiRho = pt(s% chiRho,k)
               chiT = pt(s% chiT,k)
               d_epsnuc_dlnd = pt(s% d_epsnuc_dlnd,k)
               d_epsnuc_dlnT = pt(s% d_epsnuc_dlnT,k)
               dlnKap_dlnd = pt(s% d_opacity_dlnd,k)/kap
               dlnKap_dlnT = pt(s% d_opacity_dlnT,k)/kap
               gradT = s% gradT(k)
               grada_at_face = s% grada_at_face(k)
               X = pt(s% X,k)
               Y = pt(s% Y,k)
               L = s% L(k)
               cgrav = s% cgrav(k)
               tau = s% tau(k)
               lnPgas = pt(s% lnPgas,k)
               !if (k < 20) write(*,2) 'interior P, tau', k, P, tau
               if (P >= P_prev) then
                  write(*,2) 'interior P >= P_prev', &
                     k, P - P_prev, P, P_prev, s% P(k-1), s% P(k), s% P(k+1)
                  !stop 'write1_saio_point'
               end if
            end if
            if (tau >= 2d0/3d0) then
               Lrad = (16*pi*clight*crad*cgrav*m*T**4*gradT) / (3*P*kap)
            else
               Lrad = L
            end if
            write(iounit,'(5(1pe16.9))') &
               r, m, Lrad, T, rho, &
               P, eps_nuc, kap, cv, chiRho, &
               chiT, d_epsnuc_dlnd, d_epsnuc_dlnT, &
               dlnKap_dlnd, dlnKap_dlnT, &
               gradT, grada_at_face, X, Y, L
            if (.true.) write(io_txt,'(i6,99e20.10)') &
               j, r/Rsun, log10(tau), log10(T), log10(rho), &
               log10(P), cv, chiRho, chiT, gradT, grada_at_face, &
               kap, dlnKap_dlnd, dlnKap_dlnT, &
               Lrad/Lsun, X, Y, L/Lsun, m/Msun, &
               eps_nuc, d_epsnuc_dlnd, d_epsnuc_dlnT, &
               lnPgas/ln10 - 4*log10(T)
            P_prev = P
         end subroutine write1_saio_point

      
      end subroutine do_write_pulsation_info
      
      
      subroutine get_fgong_or_osc_center_info(s, species, fgong, glob, var, ierr)
         use micro, only: eos_get
         use opacities, only: get1_kap
         use chem_lib, only: composition_info
         use chem_def
         use eos_def
         type (star_info), pointer :: s
         integer, intent(in) :: species
         logical, intent(in) :: fgong
         real(dp) :: glob(:) ! (iconst)  ! uses glob(2) = R
         real(dp) :: var(:) ! (ivar)
         integer, intent(out) :: ierr

         real(dp), dimension(num_eos_basic_results) :: &
            res, d_eos_dlnd, d_eos_dlnT
         integer :: nz, j
         integer :: h1, h2, he3, he4, li7, be7, be9, c12, c13, &
            n14, n15, o16, o17, o18, ne20, si28
         real(dp) :: &
            xa(species), X, Y, Z, X0, Y0, Z0, lnY, delta_lnY, delta_Y, &
            abar, zbar, approx_abar, approx_zbar, lnfree_e, &
            z2bar, ye, xsum, dabar_dx(species), dzbar_dx(species), &
            r12, r22, lnT0, d2lnT0, lnP0, d2lnP0, lnd0, d2lnd0, &
            T, rho, P, R, kap, dlnkap_dlnd, dlnkap_dlnT, d2
         
         logical, parameter :: dbg = .true.
         
         include 'formats.dek'
         
         nz = s% nz
         h1 = s% net_iso(ih1)
         h2 = s% net_iso(ih2)
         he3 = s% net_iso(ihe3)
         he4 = s% net_iso(ihe4)
         li7 = s% net_iso(ili7)
         be7 = s% net_iso(ibe7)
         be9 = s% net_iso(ibe9)
         c12 = s% net_iso(ic12)
         c13 = s% net_iso(ic13)
         n14 = s% net_iso(in14)
         n15 = s% net_iso(in15)
         o16 = s% net_iso(io16)
         o17 = s% net_iso(io17)
         o18 = s% net_iso(io18)
         ne20 = s% net_iso(ine20)
         si28 = s% net_iso(isi28)

         ! get center values by quadratic fit to rmid of nz and nz-1
         ! requiring d*/dr = 0 at r = 0.
         r12 = s% rmid(nz)**2
         r22 = s% rmid(nz-1)**2

         call get_center(s% lnT(nz), s% lnT(nz-1), lnT0, d2lnT0)
         call get_center(s% lnP(nz), s% lnP(nz-1), lnP0, d2lnP0)
         call get_center(s% lnd(nz), s% lnd(nz-1), lnd0, d2lnd0)
         T = exp(lnT0)
         rho = exp(lnd0)
         P = exp(lnP0)
         
         R = glob(2)
         glob(11) = R**2*d2lnP0
         glob(12) = R**2*d2lnd0
         
         !write(*,2) 'glob', 2, glob(2)
         !write(*,2) 'glob', 11, glob(11)
         !write(*,2) 'glob', 12, glob(12)
         
         do j=1,species
            call get_center(s% xa(j,nz), s% xa(j,nz-1), xa(j), d2)
         end do
         
         call composition_info(species, s% chem_id, xa, X, Y, &
            abar, zbar, z2bar, ye, approx_abar, approx_zbar, xsum, dabar_dx, dzbar_dx)  
         Z = 1 - (X+Y)

         call eos_get( &
               s, 0, Z, X, abar, zbar, approx_abar, approx_zbar, xa, &
               rho, lnd0/ln10, T, lnT0/ln10, &
               res, d_eos_dlnd, d_eos_dlnT, ierr)
         if (ierr /= 0) then
            if (dbg) then
               write(*,*) 'failed in eos_get'
               stop
            end if
            return
         end if
         
         lnfree_e = res(i_lnfree_e)
         kap = get1_kap( &
            s, 0, zbar, xa, 0d0, 1d0, lnd0/ln10, lnT0/ln10, &
            lnfree_e, d_eos_dlnd(i_lnfree_e), d_eos_dlnT(i_lnfree_e), &
            dlnkap_dlnd, dlnkap_dlnT, ierr)
         if (ierr /= 0) then
            if (dbg) then
               write(*,*) 'failed in get1_kap'
               stop
            end if
            return
         end if
         
         VAR(1) = 0d0 ! r
         VAR(2) = -99*ln10 !lnq
         VAR(3) = T
         VAR(4) = P
         VAR(5) = rho
         if (fgong) then
            VAR(6) = xpt(h1)
         else
            VAR(6) = s% gradT(nz)
         end if
         VAR(7) = 0d0 ! L
         VAR(8) = kap
         VAR(9) = s% eps_nuc(nz) + s% eps_grav(nz)
         VAR(10) = res(i_gamma1)
         VAR(11) = res(i_grad_ad)
         VAR(12) = res(i_chiT)/res(i_chiRho)
         VAR(13) = res(i_cp)
         VAR(14) = exp(lnfree_e)
         VAR(15) = 0d0 ! pt_A
                     
         if (fgong) then
            VAR(16) = s% dxdt_nuc(h1,nz)
            VAR(17) = 1d0 - (xpt(h1) + xpt(he3) + xpt(he4))
            VAR(18) = R
            VAR(19) = s% eps_grav(nz)
            VAR(21) = xpt(he3)
            VAR(22) = xpt(c12)
            VAR(23) = xpt(c13)
            VAR(24) = xpt(n14)
            VAR(25) = xpt(o16)
            VAR(29) = xpt(h2)
            VAR(30) = xpt(he4)
            VAR(31) = xpt(li7)
            VAR(32) = xpt(be7)
            VAR(33) = xpt(n15)
            VAR(34) = xpt(o17)
            VAR(35) = xpt(o18)
            VAR(36) = xpt(ne20)
         else
            if (s% rotation_flag) VAR(16) = s% omega(nz)
            VAR(17) = dlnkap_dlnT
            VAR(18) = dlnkap_dlnd
            VAR(19) = s% d_epsnuc_dlnT(nz)
            VAR(20) = s% d_epsnuc_dlnd(nz)
            VAR(21) = s% P(nz)/s% Pgas(nz)
            VAR(22) = s% gradr(nz)
            VAR(23) = xpt(h1)
            VAR(24) = xpt(h2)
            VAR(25) = xpt(he3)
            VAR(26) = xpt(he4)
            VAR(27) = xpt(li7)
            VAR(28) = xpt(be7)
            VAR(29) = xpt(c12)
            VAR(30) = xpt(c13)
            VAR(31) = xpt(n14)
            VAR(32) = xpt(n15)
            VAR(33) = xpt(o16)
            VAR(34) = xpt(o17)
            VAR(35) = xpt(be9)
            VAR(36) = xpt(si28)
         end if
         
         
         contains
         
         subroutine get_center(f1, f2, f0, c)
            real(dp), intent(in) :: f1, f2
            real(dp), intent(out) :: f0, c
            f0 = (f1*r22 - f2*r12)/(r22 - r12)
            c = 2*(f2 - f1)/(r22 - r12)
         end subroutine get_center
         
         real(dp) function xpt(j)
            integer, intent(in) :: j
            if (j == 0) then
               xpt = 0d0
            else
               xpt = xa(j)
            end if
         end function xpt

      
      end subroutine get_fgong_or_osc_center_info
      
      
      subroutine get_fgong_info(id, add_atmosphere, nn, iconst, ivar, glob, var, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: add_atmosphere
         integer, intent(out) :: nn, iconst, ivar
         real(dp), pointer :: glob(:) ! (iconst) -- will be allocated
         real(dp), pointer :: var(:,:) ! (ivar,nn) -- will be allocated
         integer, intent(out) :: ierr
         call get_pulsation_info( &
            FGONG_format, id, add_atmosphere, nn, iconst, ivar, glob, var, ierr)
      end subroutine get_fgong_info
      
      
      subroutine get_osc_info(id, add_atmosphere, nn, iconst, ivar, glob, var, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: add_atmosphere
         integer, intent(out) :: nn, iconst, ivar
         real(dp), pointer :: glob(:) ! (iconst) -- will be allocated
         real(dp), pointer :: var(:,:) ! (ivar,nn) -- will be allocated
         integer, intent(out) :: ierr
         call get_pulsation_info( &
            OSC_format, id, add_atmosphere, nn, iconst, ivar, glob, var, ierr)
      end subroutine get_osc_info
      
      
      subroutine get_pulsation_info( &
            which_format, id, add_atmosphere, nn, iconst, ivar, glob, var, ierr)
         use chem_def
         use atm_def
         use create_atm, only: do_create_atm
         integer, intent(in) :: which_format, id
         logical, intent(in) :: add_atmosphere
         integer, intent(out) :: nn, iconst, ivar
         real(dp), pointer :: glob(:) ! (iconst) -- will be allocated
         real(dp), pointer :: var(:,:) ! (ivar,nn) -- will be allocated
         integer, intent(out) :: ierr

         type (star_info), pointer :: s
         integer :: i, k, j, nvars, atm_pts, nz
         integer :: h1, h2, he3, he4, li7, be7, be9, c12, c13, &
            n14, n15, o16, o17, o18, ne20, si28
         integer, parameter :: IVERS = 300, IABUND = 14
         real(dp) :: kap, eps_nuc, r_outer, m_outer, lnq
         logical :: fgong, osc
         
         include 'formats.dek'
         
         ierr = 0
         ICONST = 15
         
         allocate(glob(iconst),stat=ierr)
         if (ierr /= 0) return
         
         fgong = .false.
         osc = .false.
         if (which_format == FGONG_format) then
            fgong = .true.
         else if (which_format == OSC_format) then
            osc = .true.
         else 
            ierr = -1
            write(*,*) 'bad value for format in do_write_pulsation_info'
            return
         end if
         
         if (fgong) then
            IVAR = 40
            nvars = IVAR
         else
            IVAR = 22
            nvars = IVAR + IABUND
         end if
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            write(*,*) 'bad star id for get_pulsation_info'
            return
         end if
         nz = s% nz
         
         if (add_atmosphere) then
            call do_create_atm(s, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in do_create_atm'
               return
            end if
            atm_pts = s% atm_structure_num_pts
            nn = nz + atm_pts - 2
         else
            atm_pts = 0
            nn = nz
         end if
         r_outer = Rsun*s% photosphere_r ! according to Andrea Miglio
         m_outer = s% mstar

         if (fgong .or. osc) nn = nn+1 ! add point at r=0
         
         h1 = s% net_iso(ih1)
         h2 = s% net_iso(ih2)
         he3 = s% net_iso(ihe3)
         he4 = s% net_iso(ihe4)
         li7 = s% net_iso(ili7)
         be7 = s% net_iso(ibe7)
         be9 = s% net_iso(ibe9)
         c12 = s% net_iso(ic12)
         c13 = s% net_iso(ic13)
         n14 = s% net_iso(in14)
         n15 = s% net_iso(in15)
         o16 = s% net_iso(io16)
         o17 = s% net_iso(io17)
         o18 = s% net_iso(io18)
         ne20 = s% net_iso(ine20)
         si28 = s% net_iso(isi28)
         
         ! globals
         GLOB = 0
         GLOB(1) = m_outer
         GLOB(2) = r_outer
         GLOB(3) = s% L(1)
         GLOB(4) = s% initial_z
         GLOB(6) = s% mixing_length_alpha
         if (fgong) then
            GLOB(13) = s% star_age
         else if (osc) then
            GLOB(5) = 1 - (s% initial_y + s% initial_z)
            if (s% largest_conv_mixing_region /= 0) then
               k = s% mixing_region_bottom(s% largest_conv_mixing_region)
               GLOB(7) = s% xa(h1,k)
               GLOB(8) = s% xa(he3,k) + s% xa(he4,k)
            end if
            GLOB(11) = s% star_age
            if (s% rotation_flag) then
               GLOB(12) = s% omega(1)
            else
               GLOB(12) = 0
            end if
         end if
         GLOB(14) = s% Teff
         
         ! point variables  <<<<   NOTE: convert cell averages to point values.
         allocate(VAR(nvars,NN+1),stat=ierr) ! pulse may need to increase NN so alloc NN+1
         if (ierr /= 0) return 
         VAR = 0
         do j = 1, NN ! set var(:,j)
            if (atm_pts == 0) then
               k = j
               if (k <= nz) then
                  lnq = log(s% q(k))
               else
                  lnq = -1d99
               end if
            else 
               if (j < atm_pts) then
                  k = atm_pts-j+1 ! k is index of point in atm_structure
                  ! we skip the point at the base of the atm to smooth the transition
                  call store_atm_info(j, k)
                  cycle
               end if
               k = j - atm_pts + 2 ! skip k==1
                  ! we skip the point at the top of the interior to smooth the transition
               lnq = log(s% m(k)/m_outer)
            end if
            if ((j == NN) .and. (fgong .or. osc)) then
               call get_fgong_or_osc_center_info( &
                     s, s% species, fgong, glob, var(:,nn), ierr)
               if (osc) then
                  GLOB(11) = s% star_age
                  if (s% rotation_flag) then
                     GLOB(12) = s% omega(1)
                  else
                     GLOB(12) = 0
                  end if
               end if
            else if (atm_pts == 0 .or. k > 1) then
               call store_point_info(j,k)
            end if
         end do

         if (associated(s% atm_structure)) then
            deallocate(s% atm_structure)
            nullify(s% atm_structure)
         end if
         
         
         contains
         
         
         subroutine store_point_info(j,k)
            integer, intent(in) :: j, k
            VAR(1,j) = s% r(k)
            VAR(2,j) = lnq
            VAR(3,j) = pt(s% T,k)
            VAR(4,j) = pt(s% P,k)
            VAR(5,j) = pt(s% rho,k)
            if (fgong) then
               VAR(6,j) = xpt(h1,k)
            else
               VAR(6,j) = s% gradT(k)
            end if
            VAR(7,j) = s% L(k)
            kap = pt(s% opacity,k)
            VAR(8,j) = kap
            eps_nuc = pt(s% eps_nuc,k)
            VAR(9,j) = eps_nuc + pt(s% eps_grav,k)
            VAR(10,j) = pt(s% gamma1,k)
            VAR(11,j) = pt(s% grada,k)
            VAR(12,j) = pt(s% chiT,k)/pt(s% chiRho,k)
            VAR(13,j) = pt(s% cp,k)
            VAR(14,j) = exp(pt(s% lnfree_e,k))
            VAR(15,j) = pt_A(k)
                        
            if (fgong) then
               VAR(16,j) = pt_dxdt_nuc_h1(k)
               VAR(17,j) = 1d0 - (xpt(h1,k) + xpt(he3,k) + xpt(he4,k))
               VAR(18,j) = r_outer - s% r(k)
               VAR(19,j) = pt(s% eps_grav,k)
               VAR(21,j) = xpt(he3,k)
               VAR(22,j) = xpt(c12,k)
               VAR(23,j) = xpt(c13,k)
               VAR(24,j) = xpt(n14,k)
               VAR(25,j) = xpt(o16,k)
               VAR(29,j) = xpt(h2,k)
               VAR(30,j) = xpt(he4,k)
               VAR(31,j) = xpt(li7,k)
               VAR(32,j) = xpt(be7,k)
               VAR(33,j) = xpt(n15,k)
               VAR(34,j) = xpt(o17,k)
               VAR(35,j) = xpt(o18,k)
               VAR(36,j) = xpt(ne20,k)
            else if (osc) then
               if (s% rotation_flag) VAR(16,j) = s% omega(k)
               VAR(17,j) = pt(s% d_opacity_dlnT,k)/kap
               VAR(18,j) = pt(s% d_opacity_dlnd,k)/kap
               VAR(19,j) = pt(s% d_epsnuc_dlnT,k)
               VAR(20,j) = pt(s% d_epsnuc_dlnd,k)
               VAR(21,j) = pt(s% P,k)/pt(s% Pgas,k)
               VAR(22,j) = s% gradr(k)
               VAR(23,j) = xpt(h1,k)
               VAR(24,j) = xpt(h2,k)
               VAR(25,j) = xpt(he3,k)
               VAR(26,j) = xpt(he4,k)
               VAR(27,j) = xpt(li7,k)
               VAR(28,j) = xpt(be7,k)
               VAR(29,j) = xpt(c12,k)
               VAR(30,j) = xpt(c13,k)
               VAR(31,j) = xpt(n14,k)
               VAR(32,j) = xpt(n15,k)
               VAR(33,j) = xpt(o16,k)
               VAR(34,j) = xpt(o17,k)
               VAR(35,j) = xpt(be9,k)
               VAR(36,j) = xpt(si28,k)
            else
               write(*,*) 'confusion about format in store_point_info'
               stop 1
            end if
         end subroutine store_point_info
         
         
         subroutine store_atm_info(j,k)
            integer, intent(in) :: j, k ! k is index of point in atm_structure
            real(dp) :: grav, N2, M, r, rho, P, Pgas, chiT, chiRho, grada, gradT
            r = s% r(1) + s% atm_structure(atm_delta_r,k)
            VAR(1,j) = r
            M = s% mstar !+ s% atm_structure(atm_delta_m,k)
            VAR(2,j) = log(M/m_outer)
            VAR(3,j) = exp(s% atm_structure(atm_lnT,k))
            P = exp(s% atm_structure(atm_lnP,k))
            VAR(4,j) = P
            rho = exp(s% atm_structure(atm_lnd,k))
            VAR(5,j) = rho
            gradT = s% atm_structure(atm_gradT,k)
            if (fgong) then
               VAR(6,j) = xpt(h1,1)
            else
               VAR(6,j) = gradT
            end if
            VAR(7,j) = s% L(1)
            kap = s% atm_structure(atm_kap,k)
            VAR(8,j) = kap
            eps_nuc = 0 ! eps_nuc
            VAR(9,j) = 0 ! eps_nuc + eps_grav
            VAR(10,j) = s% atm_structure(atm_gamma1,k)
            grada = s% atm_structure(atm_grada,k)
            VAR(11,j) = grada
            chiT = s% atm_structure(atm_chiT,k)
            chiRho = s% atm_structure(atm_chiRho,k)
            VAR(12,j) = chiT/chiRho
            VAR(13,j) = s% atm_structure(atm_cp,k)
            VAR(14,j) = exp(s% atm_structure(atm_lnfree_e,k))
            grav = s% cgrav(1)*M/(r**2)
            N2 = grav**2*(rho/P)*(chiT/chiRho)*(grada - gradT)
            VAR(15,j) = N2*r/grav ! A
                        
            if (fgong) then
               VAR(16,j) = 0 ! dxdt_nuc_h1
               VAR(17,j) = 1d0 - (xpt(h1,1) + xpt(he3,1) + xpt(he4,1))
               VAR(18,j) = r_outer - r
               VAR(19,j) = 0 ! eps_grav
               VAR(21,j) = xpt(he3,1)
               VAR(22,j) = xpt(c12,1)
               VAR(23,j) = xpt(c13,1)
               VAR(24,j) = xpt(n14,1)
               VAR(25,j) = xpt(o16,1)
               VAR(29,j) = xpt(h2,1)
               VAR(30,j) = xpt(he4,1)
               VAR(31,j) = xpt(li7,1)
               VAR(32,j) = xpt(be7,1)
               VAR(33,j) = xpt(n15,1)
               VAR(34,j) = xpt(o17,1)
               VAR(35,j) = xpt(o18,1)
               VAR(36,j) = xpt(ne20,1)
            else if (osc) then
               if (s% rotation_flag) VAR(16,j) = 0 ! omega
               VAR(17,j) = s% atm_structure(atm_dlnkap_dlnT,k)
               VAR(18,j) = s% atm_structure(atm_dlnkap_dlnd,k)
               VAR(19,j) = 0 ! d_epsnuc_dlnT
               VAR(20,j) = 0 ! d_epsnuc_dlnd
               Pgas = exp(s% atm_structure(atm_lnPgas,k))
               VAR(21,j) = P/Pgas
               VAR(22,j) = s% atm_structure(atm_gradr,k)
               VAR(23,j) = xpt(h1,1)
               VAR(24,j) = xpt(h2,1)
               VAR(25,j) = xpt(he3,1)
               VAR(26,j) = xpt(he4,1)
               VAR(27,j) = xpt(li7,1)
               VAR(28,j) = xpt(be7,1)
               VAR(29,j) = xpt(c12,1)
               VAR(30,j) = xpt(c13,1)
               VAR(31,j) = xpt(n14,1)
               VAR(32,j) = xpt(n15,1)
               VAR(33,j) = xpt(o16,1)
               VAR(34,j) = xpt(o17,1)
               VAR(35,j) = xpt(be9,1)
               VAR(36,j) = xpt(si28,1)
            else
               write(*,*) 'confusion about format in store_atm_info'
               stop 1
            end if
         end subroutine store_atm_info
         
         
         real(dp) function pt(v,k)
            use star_utils, only: interp_val_to_pt
            integer, intent(in) :: k
            real(dp), pointer :: v(:)
            pt = interp_val_to_pt(v,k,s% nz,s% dq)
         end function pt
         
         real(dp) function xpt(j,k)
            use star_utils, only: interp_xa_to_pt
            integer, intent(in) :: j,k
            xpt = interp_xa_to_pt(s% xa,j,k,nz,s% dq)
         end function xpt
         
         real(dp) function pt_dxdt_nuc_h1(k)
            integer, intent(in) :: k
            if (k == 1) then
               pt_dxdt_nuc_h1 = s% dxdt_nuc(h1,k)
            else
               pt_dxdt_nuc_h1 = &
                  (s% dxdt_nuc(h1,k)*s% dq(k-1) + s% dxdt_nuc(h1,k-1)*s% dq(k))/(s% dq(k-1) + s% dq(k))
            endif
         end function pt_dxdt_nuc_h1
         
         real(dp) function pt_A(k) ! N2*r/g at point k
            ! reminder: gradB(k) and gradT(k) are the values at face(k)
            integer, intent(in) :: k
            pt_A = s% brunt_N2(k)*s% r(k)/s% grav(k)
         end function pt_A


      end subroutine get_pulsation_info

                      
      subroutine do_write_short_format(id, filename, ierr)
         use utils_lib  
         integer, intent(in) :: id
         character (len=*), intent(in) :: filename
         integer, intent(out) :: ierr 
         
         type (star_info), pointer :: s
         integer :: iounit, k
         real(dp) :: m, r, T, rho, v, ye, omega
         
         include 'formats.dek'

         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         open(unit=iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open ' // trim(filename)
            call free_iounit(iounit)
            return
         end if

         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            write(*,*) 'bad star id for write_pulsation_info'
            call free_iounit(iounit)
            return
         end if                  
         
   99    format(i6,1p,99e20.10)
         write(iounit,99) s% nz
         ! write cell center values from star center to surface
         do k = s% nz, 1, -1
            m = s% m(k) - 0.5*s% dm(k)
            T = s% T(k)
            rho = s% rho(k)
            ye = s% ye(k)
            if (k == s% nz) then
               r = 0.5*s% r(k)
               v = 0.5*s% v(k)
            else
               r = 0.5*(s% r(k) + s% r(k+1))
               v = 0.5*(s% v(k) + s% v(k+1))
            end if
            if (.not. s% rotation_flag) then
               omega = 0
            else if (k == s% nz) then
               omega = s% omega(k)
            else
               omega = 0.5*(s% omega(k) + s% omega(k+1))
            end if
            write(iounit,99) 1 + s% nz - k, m, r, T, rho, v, ye, omega
         end do

         close(iounit)
         call free_iounit(iounit)   
         
         write(*, '(/,a, i7,/)') 'save short format ' // trim(filename), s% model_number
         if (s% extra_terminal_iounit > 0) &
            write(s% extra_terminal_iounit, '(/,a, i7,/)') &
               'save short format' // trim(filename), s% model_number
         
      
      end subroutine do_write_short_format
      


      end module pulsation_info
      
