! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module profile_getval
      
      use star_private_def
      use const_def
      use num_lib, only: safe_log10
      use utils_lib, only: is_bad_num
      use star_utils, only: interp_val_to_pt

      implicit none
      
      
      contains
      
      
      subroutine getval_for_profile(s, c, k, val, int_flag, int_val)
         use chem_def
         use rates_def
         use mlt_def
         use ionization_def
         use star_utils, only: interp_val_to_pt, omega_crit
         type (star_info), pointer :: s
         integer, intent(in) :: c, k
         real(dp), intent(out) :: val
         integer, intent(out) :: int_val
         logical, intent(inout) :: int_flag 
         real(dp) :: cno, z, frac, eps, eps_alt, L_rad, L_edd, P_face, rho_face
         integer :: j, nz, ionization_k, klo, khi
         double precision :: ionization_res(num_ion_vals)
         
         include 'formats.dek'
         
         val = 0; int_val = 0; int_flag = .false.
         nz = s% nz
         ionization_k = 0
         
         select case(c)
            case (p_zone)
               val = dble(k)
               int_val = k
               int_flag = .true.
            case (p_luminosity)
               val = s% L(k)/Lsun
            case (p_luminosity_rad)
               L_rad = get_L_rad(k)
               val = L_rad/Lsun
            case (p_luminosity_conv)
               L_rad = get_L_rad(k)
               val = (s% L(k) - L_rad)/Lsun
               
            case (p_lum_rad_div_lum_Edd_sub_fourPrad_div_PchiT)
               L_rad = get_L_rad(k)
               L_edd = get_Ledd(k)
               val = L_rad/L_edd - 4*s% Prad(k)/(s% P(k)*s% chiT(k))
               
            case (p_lum_rad_div_lum_Edd)
               L_rad = get_L_rad(k)
               L_edd = get_Ledd(k)
               val = L_rad/L_edd
               
            case (p_lum_conv_div_lum_Edd)
               L_rad = get_L_rad(k)
               L_edd = get_Ledd(k)
               val = (s% L(k) - L_rad)/L_edd
               
            case (p_lum_conv_div_lum_rad)
               L_rad = get_L_rad(k)
               val = (s% L(k) - L_rad)/L_rad

            case (p_lum_rad_div_L)
               L_rad = get_L_rad(k)
               val = L_rad/max(1d0,s% L(k))
            case (p_lum_conv_div_L)
               L_rad = get_L_rad(k)
               val = (s% L(k) - L_rad)/max(1d0,s% L(k))

            case (p_log_Lrad)
               L_rad = get_L_rad(k)
               val = safe_log10(L_rad/Lsun)
            case (p_log_Lrad_div_L)
               L_rad = get_L_rad(k)
               val = safe_log10(L_rad/s% L(k))
            case (p_log_Lrad_div_Ledd)
               L_rad = get_L_rad(k)
               L_edd = get_Ledd(k)
               val = safe_log10(L_rad/L_edd)
            case (p_grav)
               val = s% grav(k)
            case (p_g_div_r)
               val = s% grav(k)/s% r(k)
            case (p_r_div_g)
               val = s% r(k)/s% grav(k)
            case (p_signed_log_eps_grav)
               val = s% eps_grav(k)
               val = sign(1d0,val)*log10(max(1d0,abs(val)))
            case (p_net_nuclear_energy)
               val = s% eps_nuc(k) - s% eps_nuc_neu_total(k) - s% non_nuc_neu(k)
               val = sign(1d0,val)*log10(max(1d0,abs(val)))
            case (p_net_energy)
               val = s% eps_nuc(k) - s% eps_nuc_neu_total(k) - s% non_nuc_neu(k) + s% eps_grav(k)
               val = sign(1d0,val)*log10(max(1d0,abs(val)))
            case (p_signed_log_power)
               val = s% L(k)
               val = sign(1d0,val)*log10(max(1d0,abs(val)))
            case (p_logL)
               val = log10(max(1d-12,s% L(k)/Lsun))
            case (p_log_Ledd)
               val = log10(get_Ledd(k)/Lsun)
            case (p_log_L_div_Ledd)
               val = log10(max(1d-12,s% L(k)/get_Ledd(k)))
            case (p_velocity)
               val = s% velocity(k)
            case (p_v_div_r)
               val = s% velocity(k)/s% r(k)
            case (p_radius)
               val = s% r(k)/Rsun
            case (p_rmid)
               val = s% rmid(k)/Rsun
            case (p_logR)
               val = safe_log10(s% r(k)/Rsun)
               
            case (p_grav_gr_factor)
               val = 1d0/sqrt(1d0 - &
                        2*s% cgrav(k)*s% m(k)/(s% r(k)*clight**2))
            case (p_log_grav_gr_factor)
               val = safe_log10(1d0/sqrt(1d0 - &
                        2*s% cgrav(k)*s% m(k)/(s% r(k)*clight**2)))
                        
            case (p_gr_mass_excess)
               if (s% use_gr_factors) then
                  val = s% gr_m(k)/s% m(k) - 1d0
               else
                  val = 0
               end if
            case (p_gr_time_excess)
               if (s% use_gr_factors) then
                  val = s% gr_efi(k) - 1d0
               else
                  val = 0
               end if
            case (p_gr_space_excess)
               if (s% use_gr_factors) then
                  val = s% gr_gam(k) - 1d0
               else
                  val = 0
               end if
            case (p_gr_enthalpy_excess)
               if (s% use_gr_factors) then
                  val = s% gr_w(k) - 1d0
               else
                  val = 0
               end if

            case (p_avg_charge_H)
               val = get_ion_info(ion_iZ_H,k)
            case (p_avg_charge_He)
               val = get_ion_info(ion_iZ_He,k)
            case (p_avg_charge_C)
               val = get_ion_info(ion_iZ_C,k)
            case (p_avg_charge_N)
               val = get_ion_info(ion_iZ_N,k)
            case (p_avg_charge_O)
               val = get_ion_info(ion_iZ_O,k)
            case (p_avg_charge_Ne)
               val = get_ion_info(ion_iZ_Ne,k)
            case (p_avg_charge_Mg)
               val = get_ion_info(ion_iZ_Mg,k)
            case (p_avg_charge_Si)
               val = get_ion_info(ion_iZ_Si,k)
            case (p_avg_charge_Fe)
               val = get_ion_info(ion_iZ_Fe,k)
 
            case (p_neutral_fraction_H)
               val = get_ion_info(ion_ifneut_H,k)
            case (p_neutral_fraction_He)
               val = get_ion_info(ion_ifneut_He,k)
            case (p_neutral_fraction_C)
               val = get_ion_info(ion_ifneut_C,k)
            case (p_neutral_fraction_N)
               val = get_ion_info(ion_ifneut_N,k)
            case (p_neutral_fraction_O)
               val = get_ion_info(ion_ifneut_O,k)
            case (p_neutral_fraction_Ne)
               val = get_ion_info(ion_ifneut_Ne,k)
            case (p_neutral_fraction_Mg)
               val = get_ion_info(ion_ifneut_Mg,k)
            case (p_neutral_fraction_Si)
               val = get_ion_info(ion_ifneut_Si,k)
            case (p_neutral_fraction_Fe)
               val = get_ion_info(ion_ifneut_Fe,k)
 
            case (p_q)
               val = s% q(k)
            case (p_dq)
               val = s% dq(k)
            case (p_logtau)
               val = safe_log10(s% tau(k))
            case (p_mass)
               val = s% m(k)/Msun
            case (p_mmid)
               val = (s% M_center + s% xmstar*(s% q(k) - s% dq(k)/2))/Msun
            case (p_x)
               val = s% X(k)
            case (p_log_x)
               val = safe_log10(s% X(k))
            case (p_y)
               val = s% Y(k)
            case (p_log_y)
               val = safe_log10(s% Y(k))
            case (p_z)
               val = 1d0 - (s% X(k) + s% Y(k))
            case (p_log_z)
               val = safe_log10(1d0 - (s% X(k) + s% Y(k)))
            case (p_xm)
               val = s% xmstar*sum(s% dq(1:k-1))/Msun
            case (p_logxm)
               val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/Msun)
            case (p_logxq)
               val = safe_log10(sum(s% dq(1:k-1)))
            case (p_logdq)
               val = safe_log10(s% dq(k))
            case (p_log_column_depth)
               val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/(4*pi*s% r(k)**2))
            case (p_log_radial_depth)
               val = safe_log10(s% r(1) - s% r(k))
               
            case (p_r_div_R)
               val = s% r(k)/s% r(1)
            case (p_dr)
               if (k == s% nz) then
                  val = s% r(k) - s% R_center
               else
                  val = s% r(k) - s% r(k+1)
               end if
            case (p_log_dr)
               if (k == s% nz) then
                  val = s% r(k) - s% R_center
               else
                  val = s% r(k) - s% r(k+1)
               end if
               val = safe_log10(val)
            case (p_acoustic_radius)
               val = sum(s% dr_div_csound(k:nz))
            case (p_acoustic_r_div_R_phot)
               val = sum(s% dr_div_csound(k:nz))/s% photosphere_acoustic_r
            case (p_dr_div_cs)
               if (k == s% nz) then
                  val = s% r(k) - s% R_center
               else
                  val = s% r(k) - s% r(k+1)
               end if
               val = val/s% csound(k)
            case (p_log_dr_div_cs)
               if (k == s% nz) then
                  val = s% r(k) - s% R_center
               else
                  val = s% r(k) - s% r(k+1)
               end if
               val = safe_log10(val/s% csound(k))
            case (p_dr_div_cs_yr)
               if (k == s% nz) then
                  val = s% r(k) - s% R_center
               else
                  val = s% r(k) - s% r(k+1)
               end if
               val = val/s% csound(k)/secyer
            case (p_log_dr_div_cs_yr)
               if (k == s% nz) then
                  val = s% r(k) - s% R_center
               else
                  val = s% r(k) - s% r(k+1)
               end if
               val = safe_log10(val/s% csound(k)/secyer)
            case (p_pgas_div_ptotal)
               val = s% Pgas(k)/s% P(k)
               
            case (p_pturbulence_div_ptotal)
               val = s% gamma1(k)/3*(s% conv_vel(k)/s% csound(k))**2

            case (p_fourPrad_div_PchiT)
               val = 4*s% Prad(k)/(s% P(k)*s% chiT(k))

            case (p_dq_ratio)
               if (k == 1 .or. k == s% nz) then
                  val = 1
               else
                  val = s% dq(k-1)/s% dq(k)
               end if
            case (p_tau)
               val = s% tau(k)
            case (p_log_opacity)
               val = safe_log10(s% opacity(k))
            case (p_energy)
               val = exp(s% lnE(k))
            case (p_logM)
               val = safe_log10(s% m(k)/Msun)
            case (p_temperature)
               val = s% T(k)
            case (p_logT)
               val = s% lnT(k)/ln10
            case (p_rho)
               val = s% rho(k)
            case (p_logRho)
               val = s% lnd(k)/ln10
            case (p_pgas)
               val = s% Pgas(k)
            case (p_logPgas)
               val = s% lnPgas(k)/ln10
            case (p_prad)
               val = s% Prad(k)
            case (p_pressure)
               val = s% P(k)
            case (p_logP)
               val = s% lnP(k)/ln10
            case (p_logE)
               val = s% lnE(k)/ln10
            case (p_grada)
               val = s% grada(k)
            case (p_dE_dRho)
               val = s% dE_dRho(k)
            case (p_cv)
               val = s% cv(k)
            case (p_cp)
               val = s% cp(k)
            case (p_logS)
               val = s% lnS(k)/ln10
            case (p_logS_per_baryon)
               val = s% lnS(k)/ln10 + log10(amu)
            case (p_gamma1)
               val = s% gamma1(k)
            case (p_gamma3)
               val = s% gamma3(k)
            case (p_eta)
               val = s% eta(k)
            case (p_theta_e)
               val = s% theta_e(k)
            case (p_gam)
               val = s% gam(k)
            case (p_mu)
               val = s% mu(k)
            case (p_v_div_csound)
               val = s% v_div_csound(k)
            case (p_csound)
               val = s% csound(k)
            case (p_csound_at_face)
               val = s% csound_at_face(k)
            case (p_scale_height)
               val = s% scale_height(k)/Rsun
            case (p_entropy)
               val = s% entropy(k)
            case (p_free_e)
               val = exp(s% lnfree_e(k))
            case (p_logfree_e)
               val = s% lnfree_e(k)/ln10
            case (p_chiRho)
               val = s% chiRho(k)
            case (p_chiT)
               val = s% chiT(k)

            case (p_dlnRho_dlnT_const_Pgas)
               if (s% lnPgas_flag) then
                  val = s% dlnRho_dlnT_const_Pgas(k)
               else
                  val = 0
               end if
            case (p_dlnRho_dlnPgas_const_T)
               if (s% lnPgas_flag) then
                  val = s% dlnRho_dlnPgas_const_T(k)
               else
                  val = 0
               end if

            case (p_abar)
               val = s% abar(k)
            case (p_zbar)
               val = s% zbar(k)
            case (p_z2bar)
               val = s% z2bar(k)
            case (p_ye)
               val = s% ye(k)
            case (p_opacity)
               val = s% opacity(k)
            case (p_dkap_dlnrho_at_face)
               val = interp_val_to_pt(s% d_opacity_dlnd,k,nz,s% dq)
            case (p_dkap_dlnt_at_face)
               val = interp_val_to_pt(s% d_opacity_dlnT,k,nz,s% dq)
            case (p_eps_nuc)
               val = s% eps_nuc(k)
            case (p_d_epsnuc_dlnd)
               val = s% d_epsnuc_dlnd(k)
            case (p_d_lnepsnuc_dlnd)
               val = s% d_epsnuc_dlnd(k)/max(1d0,abs(s% eps_nuc(k)))
            case (p_d_epsnuc_dlnT)
               val = s% d_epsnuc_dlnT(k)
            case (p_d_lnepsnuc_dlnT)
               val = s% d_epsnuc_dlnT(k)/max(1d0,abs(s% eps_nuc(k)))
            case (p_deps_dlnd_at_face)
               val = interp_val_to_pt(s% d_epsnuc_dlnd,k,nz,s% dq)
            case (p_deps_dlnT_at_face)
               val = interp_val_to_pt(s% d_epsnuc_dlnT,k,nz,s% dq)
            case (p_eps_nuc_neu_total)
               val = s% eps_nuc_neu_total(k)
            case (p_non_nuc_neu)
               val = s% non_nuc_neu(k)
            case (p_nonnucneu_plas)
               val = s% nonnucneu_plas(k)
            case (p_nonnucneu_brem)
               val = s% nonnucneu_brem(k)
            case (p_nonnucneu_phot)
               val = s% nonnucneu_phot(k)
            case (p_nonnucneu_pair)
               val = s% nonnucneu_pair(k)
            case (p_nonnucneu_reco)
               val = s% nonnucneu_reco(k)
            case (p_log_irradiation_heat)
               val = safe_log10(s% irradiation_heat(k))
            case (p_cgrav_factor)
               val = s% cgrav(k)/standard_cgrav
               
            case (p_extra_heat)
               val = s% extra_heat(k)
            case (p_extra_L)
               val = dot_product(s% dm(k:s% nz),s% extra_heat(k:s% nz))/Lsun
            case (p_log_extra_L)
               val = safe_log10(dot_product(s% dm(k:s% nz),s% extra_heat(k:s% nz))/Lsun)
               
            case (p_logPvisc)
               if (s% v_flag .and. s% use_artificial_viscosity) then
                  val = safe_log10(s% Pvisc(k))
               else
                  val = 0
               end if
            case (p_eps_grav)
               val = s% eps_grav(k)
            case (p_env_eps_grav)
               val = -s% gradT_sub_grada(k)*s% grav(k)*s% mstar_dot*s% Cp(k)*s% T(k) / &
                        (4*pi*s% r(k)**2*s% P(k))
            case (p_mlt_mixing_type)
               val = dble(s% mlt_mixing_type(k))
               int_val = s% mlt_mixing_type(k)
               int_flag = .true.
            case (p_mlt_mixing_length)
               val = s% mlt_mixing_length(k)
            case (p_mlt_Gamma)
               val = s% mlt_Gamma(k)
            case (p_mlt_Zeta)
               if (abs(s% gradr(k) - s% grada_at_face(k)) > 1d-20) then
                  val = (s% gradr(k) - s% gradT(k))/(s% gradr(k) - s% grada_at_face(k))
               else
                  val = 0
               end if
            case (p_dlnmu_X)
               val = s% dlnmu_X(k)
            case (p_gradmu_X)
               val = s% gradmu_X(k)
               
            case (p_grad_density)
               val = s% grad_density(k)
            case (p_grad_temperature)
               val = s% grad_temperature(k)
            case (p_gradmu)
               val = s% gradmu(k)
            case (p_mu_alt)
               val = s% mu_alt(k)
            case (p_gradmu_alt)
               val = s% gradmu_alt(k)
            case (p_gradL_composition_term)
               val = s% gradL_composition_term(k)
            case (p_gradL)
               val = s% gradL(k)
            case (p_sch_stable)
               if (s% grada(k) > s% gradr(k)) then
                  val = 1
               else
                  val = 0
               end if
            case (p_ledoux_stable)
               if (s% gradL(k) > s% gradr(k)) then
                  val = 1
               else
                  val = 0
               end if
            case (p_stability_type)
               if (s% grada(k) > s% gradr(k)) then ! sch stable
                  if (s% gradL(k) > s% gradr(k)) then ! ledoux stable
                     val = no_mixing
                  else ! ledoux unstable
                     val = thermo_haline_mixing
                  end if
               else ! sch unstable
                  if (s% gradL(k) > s% gradr(k)) then ! ledoux stable
                     val = semiconvective_mixing
                  else ! ledoux unstable
                     val = convective_mixing
                  end if
               end if

            case (p_gradr_for_mixing)
               val = s% gradr_pre_hydro(k)
            case (p_gradL_for_mixing)
               val = s% gradL_pre_hydro(k)
            case (p_gradT_for_mixing)
               val = s% gradT_pre_hydro(k)
            case (p_grada_for_mixing)
               val = s% grada_at_face_pre_hydro(k)
 
             case (p_gradL_comp_for_mixing)
               val = s% gradL_comp_pre_hydro(k)
            case (p_gradmu_for_mixing)
               val = s% gradmu_pre_hydro(k)
            case (p_gradmu_alt_for_mixing)
               val = s% gradmu_alt_pre_hydro(k)
            case (p_gradmu_X_for_mixing)
               val = s% gradmu_X_pre_hydro(k)
              
            case (p_log_D_conv)
               if (s% mixing_type(k) == convective_mixing) then
                  val = safe_log10(s% D_mix_non_rotation(k))
               else
                  val = -99
               end if
            case (p_log_D_semi)
               if (s% mixing_type(k) == semiconvective_mixing) then
                  val = safe_log10(s% D_mix_non_rotation(k))
               else
                  val = -99
               end if
            case (p_log_D_ovr)
               if (s% mixing_type(k) == overshoot_mixing) then
                  val = safe_log10(s% D_mix_non_rotation(k))
               else
                  val = -99
               end if
            case (p_log_D_th)
               if (s% mixing_type(k) == thermo_haline_mixing) then
                  val = safe_log10(s% D_mix_non_rotation(k))
               else
                  val = -99
               end if
            case (p_log_D_mix_non_rotation)
               val = safe_log10(s% D_mix_non_rotation(k))
            case (p_log_D_mix)
               val = safe_log10(s% D_mix(k))
            case (p_log_sig_mix)
               val = safe_log10(s% sig(k))
            case (p_log_sig_div_siglim)
               val = safe_log10(s% sig_div_siglim(k))

            case (p_rp_div_GMrho_face)
               P_face = interp_val_to_pt(s% P,k,s% nz,s% dq)
               rho_face = interp_val_to_pt(s% rho,k,s% nz,s% dq)
               val = s% r(1)*P_face/(s% cgrav(k)*s% m(1)*rho_face)
               
            case (p_log_conv_vel_frac_current)
               val = safe_log10(s% conv_vel_frac_current(k))
            case (p_log_conv_vel)
               val = safe_log10(s% conv_vel(k))
            case (p_conv_vel_div_L_vel)
               val = s% conv_vel(k)/max(1d0,get_L_vel(k))
            case (p_conv_vel_div_csound)
               val = s% conv_vel(k)/s% csound(k)
            case (p_newly_nonconvective)
               if (s% newly_nonconvective(k)) then
                  int_val = 1; val = 1
               else
                  int_val = 0; val = 0
               end if
               int_flag = .true.
            case (p_mixing_type)
               val = dble(s% mixing_type(k))
               int_val = s% mixing_type(k)
               int_flag = .true.
            case (p_conv_mixing_type) ! OBSOLETE
               val = dble(s% mixing_type(k))
               int_val = s% mixing_type(k)
               int_flag = .true.
            case (p_conv_dP_term)
               val = s% conv_dP_term_factor*s% conv_dP_term(k)
            case (p_log_mlt_D_mix)
               val = safe_log10(s% mlt_D(k))
            case (p_pressure_scale_height)
               val = s% P(k)/(s% rho(k)*s% grav(k))/Rsun
            case (p_gradT)
               val = s% gradT(k)
            case (p_gradr)
               val = s% gradr(k)

            case (p_omega)
               val = if_rot(s% omega,k)

            case (p_log_omega)
               val = safe_log10(if_rot(s% omega,k))
            case (p_log_j_rot)
               val = safe_log10(if_rot(s% j_rot,k))
            case (p_log_J_inside)
               if (s% rotation_flag) then
                  val = safe_log10(dot_product(s% j_rot(k:s% nz), s% dm(k:s% nz)))
               else
                  val = -99
               end if
            case (p_log_J_div_M53)
               if (s% rotation_flag) then
                  val = safe_log10( &
                     dot_product(s% j_rot(k:s% nz), s% dm(k:s% nz))*1d-50/(s% m(k)/Msun)**(5d0/3d0))
               else
                  val = -99
               end if

            case (p_shear)
               if (s% rotation_flag .and. s% omega(k) > 0) then
                  val = -s% domega_dlnR(k)/s% omega(k)
               else
                  val = 0
               end if
            case (p_log_abs_shear)
               if (s% rotation_flag .and. s% omega(k) > 0) then
                  val = safe_log10(abs(s% domega_dlnR(k)/s% omega(k)))
               else
                  val = -99
               end if
            case (p_i_rot)
               val = if_rot(s% i_rot,k)
            case (p_j_rot)
               val = if_rot(s% j_rot,k)
            case (p_v_rot)
               val = if_rot(s% omega,k)*s% r(k)*1d-5 ! km/sec
            case (p_fp_rot)
               val = if_rot(s% fp_rot,k)
            case (p_ft_rot)
               val = if_rot(s% ft_rot,k)
            case (p_log_am_nu)
               val = safe_log10(if_rot(s% am_nu,k))

            case (p_r_polar)
               val = if_rot(s% r_polar,k)/Rsun
            case (p_log_r_polar)
               val = safe_log10(if_rot(s% r_polar,k)/Rsun)
            case (p_r_equitorial)
               val = if_rot(s% r_equitorial,k)/Rsun
            case (p_log_r_equitorial)
               val = safe_log10(if_rot(s% r_equitorial,k)/Rsun)
            case (p_r_e_div_r_p)
               if (s% rotation_flag) then
                  val = s% r_equitorial(k)/s% r_polar(k)
               else
                  val = 0
               end if
            case (p_omega_crit)
               if (s% rotation_flag) then
                  val = omega_crit(s,k)
               else
                  val = 0
               end if
            case (p_omega_div_omega_crit)
               if (s% rotation_flag) then
                  val = omega_crit(s,k)
                  if (val < 1d-50) then
                     val = 0
                  else
                     val = s% omega(k)/val
                  end if
               else
                  val = 0
               end if

            case (p_am_domega_dlnR)
               val = if_rot(s% domega_dlnR,k)
            case (p_am_log_sig)
               val = safe_log10(if_rot(s% am_sig,k))
               
            case (p_am_log_D_DSI)
               val = safe_log10(if_rot(s% D_DSI,k))
            case (p_am_log_D_SH)
               val = safe_log10(if_rot(s% D_SH,k))
            case (p_am_log_D_SSI)
               val = safe_log10(if_rot(s% D_SSI,k))
            case (p_am_log_D_ES)
               val = safe_log10(if_rot(s% D_ES,k))
            case (p_am_log_D_GSF)
               val = safe_log10(if_rot(s% D_GSF,k))
            case (p_am_log_D_ST)
               val = safe_log10(if_rot(s% D_ST,k))
            case (p_am_log_nu_ST)
               val = safe_log10(if_rot(s% nu_ST,k))

            case (p_dynamo_log_B_r)
               val = safe_log10(if_rot(s% dynamo_B_r,k))
            case (p_dynamo_log_B_phi)
               val = safe_log10(if_rot(s% dynamo_B_phi,k))

            case (p_gradr_sub_grada)
               val = s% gradr(k) - s% grada_at_face(k)
            case (p_gradT_sub_grada)
               val = s% gradT(k) - s% grada_at_face(k)
            case (p_super_ad)
               val = max(0d0, s% gradT(k) - s% grada_at_face(k))
            case (p_dlnP_dm)
               val = s% dlnP_dm(k)
            case (p_dlnT_dm)
               val = s% dlnT_dm(k)
            case (p_dL_dm)
               val = s% dL_dm(k)
            case (p_dlnP_dlnm)
               val = s% dlnP_dm(k)*s% m(k)
            case (p_dlnT_dlnm)
               val = s% dlnT_dm(k)*s% m(k)
            case (p_accel_div_grav)
               if (s% v_flag) then
                  val = s% dv_dt(k)/s% grav(k)
               else
                  val = 0
               end if
            case (p_dlnPgas_dt)
               val = s% dlnPgas_dt(k)
            case (p_dlnE_var_dt)
               val = s% dlnE_var_dt(k)
            case (p_dlnd_dt)
               val = s% dlnd_dt(k)
            case (p_lnddot)
               val = s% lnddot(k)
            case (p_dlnT_dt)
               val = s% dlnT_dt(k)
            case (p_lnTdot)
               val = s% lnTdot(k)
            case (p_dlnR_dt)
               val = s% dlnR_dt(k)
            case (p_signed_dlnd)
               val = 1d6*s% dlnd_dt(k)*s% dt
               val = sign(1d0,val)*log10(max(1d0,abs(val)))
            case (p_signed_dlnT)
               val = 1d6*s% dlnT_dt(k)*s% dt
               val = sign(1d0,val)*log10(max(1d0,abs(val)))
            case (p_dv_dt)
               val = s% dv_dt(k)
            case (p_cno_div_z)
               cno = s% xa(s% net_iso(ic12),k) + s% xa(s% net_iso(in14),k) + s% xa(s% net_iso(io16),k)
               z = 1 - (s% xa(s% net_iso(ih1),k) + s% xa(s% net_iso(ihe4),k))
               if (z > 1d-50) then
                  val = cno/z
               else
                  val = 0
               end if
            case (p_delta_r)
               if (k < s% nz) then
                  val = s% r(k) - s% r(k+1)
               else
                  val = s% r(k) - s% R_center
               end if
            case (p_delta_v)
               if (.not. s% v_flag) then
                  val = 0
               else if (k < s% nz) then
                  val = s% v(k+1) - s% v(k)
               else
                  val = -s% v(k)
               end if
            case (p_dt_dv_div_dr)
               if (.not. s% v_flag) then
                  val = 0
               else if (k < s% nz) then
                  val = s% dt*(s% v(k+1) - s% v(k))/(s% r(k) - s% r(k+1))
               else
                  val = -s% dt*s% v(k)/s% r(k)
               end if

            case (p_dlnH1_dlnP)
               val = get_dlogX_dlogP(ih1, k)
            case (p_dlnHe3_dlnP)
               val = get_dlogX_dlogP(ihe3, k)
            case (p_dlnHe4_dlnP)
               val = get_dlogX_dlogP(ihe4, k)
            case (p_dlnC12_dlnP)
               val = get_dlogX_dlogP(ic12, k)
            case (p_dlnC13_dlnP)
               val = get_dlogX_dlogP(ic13, k)
            case (p_dlnN14_dlnP)
               val = get_dlogX_dlogP(in14, k)
            case (p_dlnO16_dlnP)
               val = get_dlogX_dlogP(io16, k)
            case (p_dlnNe20_dlnP)
               val = get_dlogX_dlogP(ine20, k)
            case (p_dlnMg24_dlnP)
               val = get_dlogX_dlogP(img24, k)
            case (p_dlnSi28_dlnP)
               val = get_dlogX_dlogP(isi28, k)

            case (p_dlog_pp_dlogP)
               val = get_dlog_eps_dlogP(ipp, k)
            case (p_dlog_cno_dlogP)
               val = get_dlog_eps_dlogP(icno, k)
            case (p_dlog_3alf_dlogP)
               val = get_dlog_eps_dlogP(i3alf, k)

            case (p_dlog_burn_c_dlogP)
               val = get_dlog_eps_dlogP(i_burn_c, k)
            case (p_dlog_burn_n_dlogP)
               val = get_dlog_eps_dlogP(i_burn_n, k)
            case (p_dlog_burn_o_dlogP)
               val = get_dlog_eps_dlogP(i_burn_o, k)

            case (p_dlog_burn_ne_dlogP)
               val = get_dlog_eps_dlogP(i_burn_ne, k)
            case (p_dlog_burn_na_dlogP)
               val = get_dlog_eps_dlogP(i_burn_na, k)
            case (p_dlog_burn_mg_dlogP)
               val = get_dlog_eps_dlogP(i_burn_mg, k)

            case (p_dlog_cc_dlogP)
               val = get_dlog_eps_dlogP(icc, k)
            case (p_dlog_co_dlogP)
               val = get_dlog_eps_dlogP(ico, k)
            case (p_dlog_oo_dlogP)
               val = get_dlog_eps_dlogP(ioo, k)

            case (p_dlog_burn_si_dlogP)
               val = get_dlog_eps_dlogP(i_burn_si, k)
            case (p_dlog_burn_s_dlogP)
               val = get_dlog_eps_dlogP(i_burn_s, k)
            case (p_dlog_burn_ar_dlogP)
               val = get_dlog_eps_dlogP(i_burn_ar, k)
            case (p_dlog_burn_ca_dlogP)
               val = get_dlog_eps_dlogP(i_burn_ca, k)
            case (p_dlog_burn_ti_dlogP)
               val = get_dlog_eps_dlogP(i_burn_ti, k)
            case (p_dlog_burn_cr_dlogP)
               val = get_dlog_eps_dlogP(i_burn_cr, k)
            case (p_dlog_burn_fe_dlogP)
               val = get_dlog_eps_dlogP(i_burn_fe, k)
            case (p_dlog_pnhe4_dlogP)
               val = get_dlog_eps_dlogP(ipnhe4, k)
            case (p_dlog_photo_dlogP)
               val = get_dlog_eps_dlogP(iphoto, k)
            case (p_dlog_other_dlogP)
               val = get_dlog_eps_dlogP(iother, k)
            case (p_binding_energy)
               val = exp(s% lnE(k)) + &
                     s% P(k)/s% rho(k) - s% cgrav(k)*s% m(k)/s% r(k) + &
                     0.5d0*s% velocity(k)**2
            case (p_binding_energy_integral)
               val = dot_product(s% dm(1:k), exp(s% lnE(1:k)) + &
                     s% P(1:k)/s% rho(1:k) - s% cgrav(k)*s% m(1:k)/s% r(1:k) + &
                     0.5d0*s% velocity(1:k)**2)
                     
                     
            case (p_dlnX_dr)
               klo = max(1,k-1)
               khi = min(nz,k+1)
               val = log(max(1d-99,max(1d-99,s% X(klo))/max(1d-99,s% X(khi))))  &
                              /  (s% rmid(klo) - s% rmid(khi))
            case (p_dlnY_dr)
               klo = max(1,k-1)
               khi = min(nz,k+1)
               val = log(max(1d-99,max(1d-99,s% Y(klo))/max(1d-99,s% Y(khi))))  &
                              /  (s% rmid(klo) - s% rmid(khi))
            case (p_dlnRho_dr)
               klo = max(1,k-1)
               khi = min(nz,k+1)
               val = (s% lnd(klo) - s% lnd(khi))/(s% rmid(klo) - s% rmid(khi))

            case (p_brunt_g_r_div_cs2)
               val = s% grav(k)*s% r(k)/max(1d-20,s% csound_at_face(k))**2
            case (p_brunt_dlnRho_dlnR)
               val = s% brunt_dlnRho_dlnR(k)

            case (p_brunt_chiY)
               val = s% brunt_chiY(k)
            case (p_brunt_dlnY_dlnP)
               val = s% brunt_dlnY_dlnP(k)
            case (p_brunt_B)
               val = s% brunt_B(k)
            case (p_brunt_nonB)
               val = -s% gradT_sub_grada(k)
               
            case (p_brunt_N2)
               val = s% brunt_N2(k)
            case (p_brunt_N2_composition_term)
               val = s% brunt_B(k) - s% gradT_sub_grada(k)
               if (abs(val) > 1d-20) then
                  val = s% brunt_B(k)*s% brunt_N2(k)/val
               else
                  val = 0
               end if
            case (p_brunt_N2_structure_term)
               val = s% brunt_B(k) - s% gradT_sub_grada(k)
               if (abs(val) > 1d-20) then
                  val = -s% gradT_sub_grada(k)*s% brunt_N2(k)/val
               else
                  val = 0
               end if
               
            case (p_brunt_A)
               val = s% brunt_N2(k)*s% r(k)/s% grav(k)
            case (p_log_brunt_N2_dimensionless)
               val = safe_log10(s% brunt_N2(k)/(3*s% cgrav(1)*s% mstar/s% r(1)**3))
            case (p_brunt_N2_dimensionless)
               val = s% brunt_N2(k)/(3*s% cgrav(1)*s% mstar/s% r(1)**3)
            case (p_brunt_N_dimensionless)
               val = sqrt(max(0d0,s% brunt_N2(k))/(3*s% cgrav(1)*s% mstar/s% r(1)**3))
            case (p_brunt_N)
               val = sqrt(max(0d0,s% brunt_N2(k)))
            case (p_brunt_frequency) ! cycles per day
               val = (24d0*60d0*60d0/(2*pi))*sqrt(max(0d0,s% brunt_N2(k)))
            case (p_log_brunt_N)
               val = safe_log10(sqrt(max(0d0,s% brunt_N2(k))))
            case (p_log_brunt_N2)
               val = safe_log10(s% brunt_N2(k))
            case (p_lamb_S)
               val = sqrt(2d0)*s% csound_at_face(k)/s% r(k) ! for l=1
            case (p_lamb_S2)
               val = 2d0*(s% csound_at_face(k)/s% r(k))**2 ! for l=1               
               
            case (p_brunt_nu) ! micro Hz
               val = s% brunt_N2(k)
               val = (1d6/(2*pi))*sqrt(max(0d0,val))
            case (p_brunt_nu_composition_term)
               val = s% brunt_B(k) - s% gradT_sub_grada(k)
               if (abs(val) > 1d-20) then
                  val = s% brunt_B(k)*s% brunt_N2(k)/val
                  val = (1d6/(2*pi))*sqrt(max(0d0,val))
               else
                  val = 0
               end if
            case (p_brunt_nu_structure_term)
               val = s% brunt_B(k) - s% gradT_sub_grada(k)
               if (abs(val) > 1d-20) then
                  val = -s% gradT_sub_grada(k)*s% brunt_N2(k)/val
                  val = (1d6/(2*pi))*sqrt(max(0d0,val))
               else
                  val = 0
               end if
            case (p_lamb_Sl1)
               val = (1d6/(2*pi))*sqrt(2d0)*s% csound_at_face(k)/s% r(k) ! microHz
            case (p_lamb_Sl2)
               val = (1d6/(2*pi))*sqrt(6d0)*s% csound_at_face(k)/s% r(k) ! microHz
            case (p_lamb_Sl3)
               val = (1d6/(2*pi))*sqrt(12d0)*s% csound_at_face(k)/s% r(k) ! microHz
               
            case (p_log_brunt_nu) ! micro Hz
               val = safe_log10((1d6/(2*pi))*sqrt(max(0d0,s% brunt_N2(k))))
            case (p_log_lamb_Sl1)
               val = safe_log10((1d6/(2*pi))*sqrt(2d0)*s% csound_at_face(k)/s% r(k)) ! microHz
            case (p_log_lamb_Sl2)
               val = safe_log10((1d6/(2*pi))*sqrt(6d0)*s% csound_at_face(k)/s% r(k)) ! microHz
            case (p_log_lamb_Sl3)
               val = safe_log10((1d6/(2*pi))*sqrt(12d0)*s% csound_at_face(k)/s% r(k)) ! microHz
            case (p_log_lamb_Sl10)
               val = safe_log10((1d6/(2*pi))*sqrt(110d0)*s% csound_at_face(k)/s% r(k)) ! microHz
               
            case (p_brunt_N_div_r_integral)
               val = get_brunt_N_div_r_integral(k)
               
            case (p_k_r_integral)
               val = get_k_r_integral(k,1,1d0)
               
            case (p_log_err_ratio_max)
               val = 0
               if (associated(s% seulex_error_vectors)) then
                  if (size(s% seulex_error_vectors, dim=2) >= s% nz) then
                     val = safe_log10(maxval(abs(s% seulex_error_vectors(1:s% nvar,k,s% seulex_rows-1))))
                  end if
               end if
               
            case (p_sign_brunt_N2)
               val = sign(1d0,s% brunt_N2(k))
            case (p_cs_at_cell_bdy)
               val = s% csound_at_face(k)
            case (p_log_mdot_cs) ! log10(4 Pi r^2 csound rho / (Msun/year))
               val = safe_log10(4*pi*s% r(k)**2*s% csound(k)*s% rho(k)/(Msun/secyer))
            case (p_log_mdot_v) ! log10(4 Pi r^2 v rho / (Msun/year))
               val = safe_log10(4*pi*s% r(k)**2*s% velocity(k)*s% rho(k)/(Msun/secyer))
            case (p_log_L_div_CpTMdot)
               if (s% star_mdot == 0) then
                  val = 0
               else
                  val = safe_log10(s% L(k)/(s% cp(k)*s% T(k)*abs(s% star_mdot)*(Msun/secyer)))
               end if
            case (p_logQ)
               val = s% lnd(k)/ln10 - 2*s% lnT(k)/ln10 + 12
               
            case default
               write(*,*) 'FATAL ERROR in profile columns specifications', c, k
               write(*,*) 'between ' // trim(profile_column_name(c-1)) // ' and ' // &
                  trim(profile_column_name(c+1))
               val = 0
               stop 'profile_getval'
               
         end select
         
         
         contains
         
         real(dp) function get_Ledd(k) result(Ledd)
            integer, intent(in) :: k
            Ledd = pi4*clight*s% cgrav(k)*s% m(k)/s% opacity(k)
         end function get_Ledd
         
         
         real(dp) function get_L_vel(k) result(v) ! velocity if L carried by convection
            integer, intent(in) :: k
            real(dp) :: rho_face
            integer :: j
            if (k == 1) then
               j = 2
            else
               j = k
            end if
            rho_face = interp_val_to_pt(s% rho,j,nz,s% dq)
            v = (max(1d0,s% L(k))/(4*pi*s% r(k)**2*rho_face))**(1d0/3d0)
         end function get_L_vel
         

         real(dp) function get_k_r_integral(k_in, el, nu_factor)
            integer, intent(in) :: k_in
            integer, intent(in) :: el
            real(dp), intent(in) :: nu_factor
            real(dp) :: integral, integral_for_k, &
               cs2, r2, n2, sl2, omega2, L2, kr2, dr
            integer :: k, k1, k_inner, k_outer         
            include 'formats.dek'

            if (k_in == 1) then
               get_k_r_integral = 1
               return
            end if
            
            get_k_r_integral = 0
            L2 = el*(el+1)
            omega2 = (1d-6*2*pi*s% nu_max*nu_factor)**2
            
            ! k_inner and k_outer are bounds of evanescent region
            
            ! k_outer is outermost k where Sl2 <= omega2 at k-1 and Sl2 > omega2 at k
            ! 1st find outermost where Sl2 <= omega2
            k1 = 0
            do k = 2, s% nz
               r2 = s% r(k)**2
               cs2 = s% csound_at_face(k)**2
               sl2 = L2*cs2/r2
               if (sl2 <= omega2) then
                  k1 = k; exit
               end if
            end do
            if (k1 == 0) return
            ! then find next k where Sl2 >= omega2
            k_outer = 0
            do k = k1+1, s% nz
               r2 = s% r(k)**2
               cs2 = s% csound_at_face(k)**2
               sl2 = L2*cs2/r2
               if (sl2 > omega2) then
                  k_outer = k; exit
               end if
            end do
            if (k_outer == 0) return
            if (k_in <= k_outer) then
               get_k_r_integral = 1
               return
            end if
            
            ! k_inner is next k where N2 >= omega2 at k+1 and N2 < omega2 at k
            k_inner = 0
            do k = k_outer+1, s% nz
               if (s% brunt_N2(k) >= omega2) then
                  k_inner= k; exit
               end if
            end do
            if (k_inner == 0) return
            if (k_in > k_inner) then
               get_k_r_integral = 1
               return
            end if

            integral = 0; integral_for_k = 0
            get_k_r_integral = 0
            do k = k_inner, k_outer, -1
               r2 = s% r(k)**2
               cs2 = s% csound_at_face(k)**2
               n2 = s% brunt_N2(k)
               sl2 = L2*cs2/r2
               kr2 = (1 - n2/omega2)*(1 - Sl2/omega2)/cs2
               dr = s% rmid(k-1) - s% rmid(k)
               if (kr2 < 0 .and. omega2 < Sl2) integral = integral + sqrt(-kr2)*dr
               if (k == k_in) integral_for_k = integral
            end do
            if (integral < 1d-99) return
            get_k_r_integral = integral_for_k/integral
            
            if (is_bad_num(get_k_r_integral)) then
               write(*,2) 'get_k_r_integral', k_in, integral_for_k, integral
               stop 'get_k_r_integral'
            end if
            
         end function get_k_r_integral
         

         real(dp) function get_brunt_N_div_r_integral(k_in)
            integer, intent(in) :: k_in
            real(dp) :: integral, integral_for_k, dr
            integer :: k
            integral = 0
            get_brunt_N_div_r_integral = 1
            if (k_in == 1) return
            get_brunt_N_div_r_integral = 0
            do k = s% nz, 2, -1
               dr = s% rmid(k-1) - s% rmid(k)
               if (s% brunt_N2(k) > 0) &
                  integral = integral + sqrt(s% brunt_N2(k))*dr/s% r(k)
               if (k == k_in) integral_for_k = integral
            end do
            if (integral < 1d-99) return
            get_brunt_N_div_r_integral = integral_for_k/integral
         end function get_brunt_N_div_r_integral

         
         real(dp) function get_L_rad(k)
            integer, intent(in) :: k
            integer :: j
            real(dp) :: kap_face, del_m, del_T4
            if (k == 1) then
               j = 2
            else
               j = k
            end if
            kap_face = interp_val_to_pt(s% opacity,j,nz,s% dq)
            del_m = 0.5d0*(s% dm(j-1) + s% dm(j))
            del_T4 = s% T(j-1)**4 - s% T(j)**4
            get_L_rad = -s% area(j)**2*crad*clight/(3*kap_face)*(del_T4/del_m)
         end function get_L_rad

         
         real(dp) function get_dlogX_dlogP(j, k)
            integer, intent(in) :: j, k
            integer :: ii, i
            real(dp) :: val, x00, xm1, dlogP, dlogX
            include 'formats.dek'
            get_dlogx_dlogp = 0
            if (k > 1) then
               ii = k
            else
               ii = 2
            end if
            i = s% net_iso(j)
            if (i == 0) return
            x00 = s% xa(i,ii)
            xm1 = s% xa(i,ii-1)
            if (x00 < 1d-20 .or. xm1 < 1d-20) return
            dlogP = (s% lnP(ii) - s% lnP(ii-1))/ln10
            if (dlogP <= 0d0) return
            dlogX = log10(x00/xm1)
            get_dlogX_dlogP = dlogX/dlogP
         end function get_dlogX_dlogP

         
         real(dp) function get_dlog_eps_dlogP(cat, k)
            integer, intent(in) :: cat, k
            integer :: ii
            real(dp) :: val, eps, epsm1, dlogP, dlog_eps
            get_dlog_eps_dlogP = 0
            if (k > 1) then
               ii = k
            else
               ii = 2
            end if
            eps = s% eps_nuc_categories(i_rate,cat,ii)
            epsm1 = s% eps_nuc_categories(i_rate,cat,ii-1)
            if (eps < 1d-3 .or. epsm1 < 1d-3) return
            dlogP = (s% lnP(ii) - s% lnP(ii-1))/ln10
            if (dlogP <= 0d0) return
            dlog_eps = log10(eps/epsm1)
            get_dlog_eps_dlogP = dlog_eps/dlogP
         end function get_dlog_eps_dlogP

         
         real(dp) function pt(v,k)
            integer, intent(in) :: k
            real(dp), pointer :: v(:)
            if (k == 1) then
               pt = v(k)
            else
               pt = (v(k)*s% dq(k-1) + v(k-1)*s% dq(k))/(s% dq(k-1) + s% dq(k))
            endif
         end function pt

         
         real(dp) function if_rot(v,k)
            integer, intent(in) :: k
            real(dp), pointer :: v(:)
            if (s% rotation_flag) then
               if_rot = v(k)
            else
               if_rot = 0
            endif
         end function if_rot
               
         real(dp) function get_ion_info(id,k)
            use ionization_lib, only: eval_ionization
            integer, intent(in) :: id, k
            integer :: ierr
            if (ionization_k /= k) then
               ierr = 0
               call eval_ionization( &
                  1d0 - (s% X(k) + s% Y(k)), s% X(k), s% Rho(k), s% lnd(k)/ln10, &
                  s% T(k), s% lnT(k)/ln10, ionization_res, ierr)
               if (ierr /= 0) ionization_res = 0
               ionization_k = k
            end if
            get_ion_info = ionization_res(id)
         end function get_ion_info
         
      end subroutine getval_for_profile


      end module profile_getval
      
