! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module profile
      
      use star_private_def
      use const_def
      use num_lib, only: safe_log10
      use pulsation_info

      implicit none
      
      ! model log priorities
      integer, parameter :: delta_priority = 1
      integer, parameter :: phase_priority = 2

         
      real(dp), parameter :: del_cntr_rho = 1d0
      real(dp), parameter :: min_cntr_rho = 3d0
      real(dp), parameter :: no_he_ignition_limit = 0.75d0
      real(dp), parameter :: no_cntr_T_drops_limit = 6.5d0
      
      real(dp), parameter :: center_h_gone = 1d-3
      real(dp), parameter :: center_h_going = one_third
      real(dp), parameter :: center_he_going = 5d-2
      
      
      integer, parameter :: category_offset = 10000
      integer, parameter :: abundance_offset = 20000
      integer, parameter :: log_abundance_offset = 30000
      integer, parameter :: xadot_offset = 40000
      integer, parameter :: xaprev_offset = 50000
      integer, parameter :: ionization_offset = 60000
      integer, parameter :: edv_offset = 70000
      integer, parameter :: extra_offset = 80000
      integer, parameter :: err_ratio_offset = 90000
      integer, parameter :: row_log_max_err_ratio_offset = 100000
      

      contains
      
      
      
      recursive subroutine add_profile_columns( &
            s, level, capacity, spec, profile_columns_file, ierr)
         use utils_lib
         use utils_def
         use chem_def
         use chem_lib
         type (star_info), pointer :: s
         integer, intent(in) :: level
         integer, intent(inout) :: capacity
         integer, pointer :: spec(:)
         character (len=*), intent(in) :: profile_columns_file
         integer, intent(out) :: ierr

         integer :: iounit, n, i, t, id, j, k, num
         character (len=256) :: buffer, string, filename
         integer, parameter :: max_level = 20

         logical, parameter :: dbg = .false.

         include 'formats.dek'
         
         if (level > max_level) then
            write(*,*) 'too many levels of nesting for log column files', level
            ierr = -1
            return
         end if

         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         
         ! first try local directory
         filename = profile_columns_file
         if (len_trim(filename) == 0) filename = 'profile_columns.list'
         open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then ! if don't find that file, look in data/star_data
            filename = trim(data_dir_for_mesa) // '/star_data/' // trim(filename)
            ierr = 0
            open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               call free_iounit(iounit)
               write(*,*) 'failed to open ' // trim(filename)
               return
            end if
         end if
         
         if (dbg) then
            write(*,*)
            write(*,*) 'profile_columns_file <' // trim(filename) // '>'
         end if

         call count_specs
         
         n = 0
         i = 0
         
         do
            t = token(iounit, n, i, buffer, string)
            if (dbg) write(*,*) 'token', t
            if (t == eof_token) then
               if (dbg) write(*,*) 'eof token'
               exit
            end if

            select case(t)
               case(name_token)
                  if (dbg) write(*,*) 'name_token ' // trim(string)
                  select case(string)
                  
                     case ('add_row_log_max_error_ratios')
                        do k = 1, s% seulex_kmax-1
                           call insert_spec( &
                              row_log_max_err_ratio_offset + k, 'row_log_max_error_ratios', ierr)
                           if (ierr /= 0) then
                              call error; return
                           end if
                        end do
                  
                     case ('add_log_error_ratios')
                     
                        do k = 1, s% nvar
                           call insert_spec( &
                              err_ratio_offset + k, s% nameofvar(k), ierr)
                           if (ierr /= 0) then
                              call error; return
                           end if
                        end do
                  
                     case ('add_abundances') ! add all of the isos that are in the current net
                     
                        do k = 1, s% species
                           call insert_spec( &
                              abundance_offset + s% chem_id(k), chem_isos% name(s% chem_id(k)), ierr)
                           if (ierr /= 0) then
                              call error; return
                           end if
                        end do
                  
                     case ('add_log_abundances') ! add all of the isos that are in the current net
                     
                        do k = 1, s% species
                           call insert_spec( &
                              log_abundance_offset + s% chem_id(k), chem_isos% name(s% chem_id(k)), ierr)
                           if (ierr /= 0) then
                              call error; return
                           end if
                        end do
                  
                     case ('xadot')
                     
                        t = token(iounit, n, i, buffer, string)
                        if (t /= name_token) then
                           call error; return
                        end if
                        id = get_nuclide_index(string)
                        if (id > 0) then ! store it
                           call insert_spec(xadot_offset + id, string, ierr)
                           if (ierr /= 0) then
                              call error; return
                           end if
                        else
                           write(*,*) 'failed to recognize iso name for xadot ' // trim(string)
                           call error; ierr = -1
                           return
                        end if
                        if (ierr /= 0) then
                           call error; return
                        end if
                  
                     case ('xaprev')
                     
                        t = token(iounit, n, i, buffer, string)
                        if (t /= name_token) then
                           call error; return
                        end if
                        id = get_nuclide_index(string)
                        if (id > 0) then ! store it
                           call insert_spec(xaprev_offset + id, string, ierr)
                           if (ierr /= 0) then
                              call error; return
                           end if
                        else
                           write(*,*) 'failed to recognize iso name for xaprev ' // trim(string)
                           call error; ierr = -1
                           return
                        end if
                        if (ierr /= 0) then
                           call error; return
                        end if
                  
                     case ('ionization')
                     
                        t = token(iounit, n, i, buffer, string)
                        if (t /= name_token) then
                           call error; return
                        end if
                        id = get_nuclide_index(string)
                        if (id > 0) then ! store it
                           call insert_spec(ionization_offset + id, string, ierr)
                           if (ierr /= 0) then
                              call error; return
                           end if
                        else
                           write(*,*) 'failed to recognize iso name for ionization ' // trim(string)
                           call error; ierr = -1
                           return
                        end if
                        if (ierr /= 0) then
                           call error; return
                        end if
                  
                     case ('extra')
                     
                        t = token(iounit, n, i, buffer, string)
                        if (t /= name_token) then
                           call error; return
                        end if                        
                        read(string,fmt=*,iostat=ierr) num
                        if (ierr /= 0 .or. num <= 0 .or. num > max_num_profile_extras) then
                           write(*,*) 'failed to find valid integer for extra ' // trim(string)
                           call error; return
                        end if
                        call insert_spec(extra_offset + num, string, ierr)
                        if (ierr /= 0) then
                           call error; return
                        end if
                  
                     case ('edv')
                     
                        t = token(iounit, n, i, buffer, string)
                        if (t /= name_token) then
                           call error; return
                        end if
                        id = get_nuclide_index(string)
                        if (id > 0) then ! store it
                           call insert_spec(edv_offset + id, string, ierr)
                           if (ierr /= 0) then
                              call error; return
                           end if
                        else
                           write(*,*) 'failed to recognize iso name for edv ' // trim(string)
                           call error; ierr = -1
                           return
                        end if
                        if (ierr /= 0) then
                           call error; return
                        end if
                  
                     case ('add_reaction_categories') ! add all the reaction categories
                        do k = 1, num_categories
                           call insert_spec(category_offset + k, category_name(k), ierr)
                           if (ierr /= 0) return
                        end do
                  
                     case ('include')
                        
                        t = token(iounit, n, i, buffer, string)
                        if (dbg) write(*,*) 'include file token', t
                        if (dbg) write(*,*) 'include file string len', len_trim(string)
                        if (t /= string_token) then
                           call error; return
                        end if
                        if (dbg) write(*,*) 'include file <' // trim(string) // '>'
                        call add_profile_columns(s, level+1, capacity, spec, string, ierr)
                        if (ierr /= 0) then
                           write(*,*) 'failed for included profile columns list ' // trim(string)
                           call error; return
                        end if
                        call count_specs                        
                  
                     case ('log') ! add log of abundance
                        id = get_nuclide_index(string)
                        if (id > 0) then
                           call insert_spec(log_abundance_offset + id, string, ierr)
                           if (ierr /= 0) return
                        else
                           write(*,*) 'failed to recognize species for profile columns ' // trim(string)
                           ierr = -1
                           return
                        end if

                     case default
                        
                        id = do_get_profile_id(string)
                        if (id > 0) then ! store it
                           call insert_spec(id, string, ierr)
                           if (ierr /= 0) return
                        else
                           id = get_nuclide_index(string)
                           if (id > 0) then
                              call insert_spec(abundance_offset + id, string, ierr)
                              if (ierr /= 0) return
                           else
                              id = rates_category_id(string)
                              if (id > 0) then
                                 call insert_spec(category_offset + id, string, ierr)
                                 if (ierr /= 0) return
                              else
                                 write(*,*) 'failed to recognize item for profile columns ' // trim(string)
                                 ierr = -1
                                 return
                              end if
                           end if
                        end if
                        
                  end select
               case default
                  call error; return
            end select
            
         end do
         
         if (dbg) write(*,*) 'finished ' // trim(filename)
         
         close(iounit)
         call free_iounit(iounit)
         
         if (dbg) then
            write(*,*)
            write(*,*) 'done add_profile_columns ' // trim(filename)
            write(*,*)
         end if
         
         
         contains
         
         
         subroutine count_specs
            integer :: i
            j = 1
            do i=1, capacity
               if (spec(i) == 0) then
                  j = i; exit
               end if
            end do
         end subroutine count_specs

         
         subroutine make_room(ierr)
            integer, intent(out) :: ierr
            if (j < capacity) return
            capacity = 50 + (3*capacity)/2
            call realloc_integer(spec,capacity,ierr)
            spec(j+1:capacity) = 0
         end subroutine make_room
  
            
         subroutine insert_spec(c, name, ierr)
            integer, intent(in) :: c
            character (len=*) :: name
            integer, intent(out) :: ierr
            integer :: i
            include 'formats.dek'
            do i=1,j-1
               if (spec(i) == c) return
            end do
            call make_room(ierr)
            if (ierr /= 0) return
            spec(j) = c
            if (dbg) write(*,2) trim(name), spec(j)
            j = j+1
         end subroutine insert_spec
         
         
         subroutine error
            ierr = -1
            call alert(ierr, 'error in profile_columns_file')
            close(iounit)
            call free_iounit(iounit)
         end subroutine error
                  
         
      end subroutine add_profile_columns
      
      
      subroutine set_profile_columns(id, profile_columns_file, ierr)
         use utils_lib, only: realloc_integer
         integer, intent(in) :: id
         character (len=*), intent(in) :: profile_columns_file
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: capacity, cnt, i
         logical, parameter :: dbg = .false.
         if (dbg) write(*,*) 'set_profile_columns'
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         if (associated(s% profile_column_spec)) deallocate(s% profile_column_spec)
         capacity = 100 ! will increase if needed
         allocate(s% profile_column_spec(capacity), stat=ierr)
         if (ierr /= 0) return
         s% profile_column_spec(:) = 0
         call add_profile_columns( &
               s, 1, capacity, s% profile_column_spec, profile_columns_file, ierr)
         if (ierr /= 0) return
         ! delete trailing 0's
         cnt = capacity+1
         do i=1, capacity
            if (s% profile_column_spec(i) == 0) then
               cnt = i; exit
            end if
            if (dbg) write(*,*) 'profile col', i, s% profile_column_spec(i)
         end do
         call realloc_integer(s% profile_column_spec, cnt-1, ierr)
         if (dbg) write(*,*) 'num profile columns', cnt-1
         if (dbg) stop 'debug: set_profile_columns'
      end subroutine set_profile_columns
            
            
      integer function no_extra_profile_columns(s, id, id_extra)
         use star_def, only: star_info
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         no_extra_profile_columns = 0
      end function no_extra_profile_columns
      
      
      subroutine no_data_for_extra_profile_columns(s, id, id_extra, n, nz, names, vals, ierr)
         use star_def, only: maxlen_profile_column_name, star_info
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine no_data_for_extra_profile_columns
      
      
      subroutine do_get_data_for_profile_columns(s, id_extra, numvals, nz, &
            how_many_extra_profile_columns, data_for_extra_profile_columns, &
            names, vals, is_int, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id_extra, numvals, nz
         interface
            include 'extra_profile_cols.dek'
         end interface
         character (len=maxlen_profile_column_name), pointer :: names(:) ! (num_profile_columns)
         real(dp), pointer :: vals(:,:) ! (nz,num_profile_columns)
         logical, pointer :: is_int(:) ! (num_profile_columns) true iff the values in the column are integers
         integer, intent(out) :: ierr
         character (len=0) :: fname
         logical, parameter :: write_flag = .false.
         call do_profile_info(s, fname, &
            id_extra, how_many_extra_profile_columns, data_for_extra_profile_columns, &
            write_flag, names, vals, is_int, ierr)
      end subroutine do_get_data_for_profile_columns
      
      
      subroutine write_profile_info(s, fname, &
            id_extra, how_many_extra_profile_columns, data_for_extra_profile_columns, ierr)
         use chem_def
         use net_def ! categories
         use rates_def, only: i_rate
         type (star_info), pointer :: s
         character (len=*) :: fname
         integer, intent(in) :: id_extra
         interface
            include 'extra_profile_cols.dek'
         end interface
         integer, intent(out) :: ierr
         character (len=maxlen_profile_column_name), pointer :: names(:) ! (num_profile_columns)
         real(dp), pointer :: vals(:,:) ! (nz,num_profile_columns)
         logical, pointer :: is_int(:)
         logical, parameter :: write_flag = .true.
         names => null()
         vals => null()
         is_int => null()
         call do_profile_info(s, fname, &
            id_extra, how_many_extra_profile_columns, data_for_extra_profile_columns, &
            write_flag, names, vals, is_int, ierr)
      end subroutine write_profile_info
      
      
      subroutine do_profile_info(s, fname, &
            id_extra, how_many_extra_profile_columns, data_for_extra_profile_columns, &
            write_flag, names, vals, is_int, ierr)
         use chem_def
         use net_def ! categories
         use rates_def, only: i_rate
         use utils_lib, only:alloc_iounit, free_iounit
         type (star_info), pointer :: s
         character (len=*) :: fname
         integer, intent(in) :: id_extra
         interface
            include 'extra_profile_cols.dek'
         end interface
         logical, intent(in) :: write_flag
         character (len=maxlen_profile_column_name), pointer :: names(:) ! (num_profile_columns)
         real(dp), pointer :: vals(:,:) ! (nz,num_profile_columns)
         logical, pointer :: is_int(:)
         integer, intent(out) :: ierr
         
         real(dp) :: msum, mstar, dt, Lnuc, frac
         integer :: io, i, j, nz, col, k, n, species, h1, he4, numcols, num_extra_cols
         integer, pointer :: chem_id(:)
         logical, parameter :: dbg = .false.
         character (len=256) :: fname1, dbl_fmt, int_fmt, txt_fmt
         character (len=maxlen_profile_column_name), pointer :: extra_col_names(:)
         real(dp), pointer :: extra_col_vals(:,:)

         include "formats.dek"
         
         dbl_fmt = s% profile_dbl_format
         int_fmt = s% profile_int_format
         txt_fmt = s% profile_txt_format
         
         ierr = 0
         nullify(extra_col_names)
         nullify(extra_col_vals)

         nz = s% nz
         species = s% species
         chem_id => s% chem_id
         mstar = s% mstar
         dt = s% dt
         if (.not. associated(s% profile_column_spec)) then
            numcols = 0
         else
            numcols = size(s% profile_column_spec, dim=1)
         end if
         
         if (numcols == 0) then
            write(*,*) 'WARNING: do not have any output specified for profiles.'
            return
         end if
         
         num_extra_cols = how_many_extra_profile_columns(s, s% id, id_extra)
         if (num_extra_cols > 0) then
            allocate(extra_col_names(num_extra_cols), extra_col_vals(nz,num_extra_cols), stat=ierr)
            if (ierr /= 0) return
            call data_for_extra_profile_columns( &
               s, s% id, id_extra, num_extra_cols, nz, extra_col_names, extra_col_vals, ierr)
            if (ierr /= 0) then
               deallocate(extra_col_names, extra_col_vals)
               return
            end if
         end if
         
         if (.not. write_flag) then
         
            if (associated(names)) then
               if (size(names,dim=1) < numcols+num_extra_cols) then
                  write(*,2) 'size(names,dim=1)', size(names,dim=1)
                  write(*,2) 'numcols+num_extra_cols', numcols+num_extra_cols
                  write(*,2) 'numcols', numcols
                  write(*,2) 'num_extra_cols', num_extra_cols
                  write(*,*) 'bad size for names in do_profile_info'
                  ierr = -1
                  return
               end if
            else
               write(*,*) 'failed to provide names array for do_profile_info'
               ierr = -1
               return
            end if
         
            if (associated(vals)) then
               if (size(vals,dim=1) < nz) then
                  write(*,2) 'size(vals,dim=1)', size(vals,dim=1)
                  write(*,2) 'nz', nz
                  write(*,*) 'bad size dim=1 for vals in do_profile_info'
                  ierr = -1
                  return
               end if
               if (size(vals,dim=2) < numcols+num_extra_cols) then
                  write(*,2) 'size(vals,dim=2)', size(vals,dim=2)
                  write(*,2) 'numcols+num_extra_cols', numcols+num_extra_cols
                  write(*,2) 'numcols', numcols
                  write(*,2) 'num_extra_cols', num_extra_cols
                  write(*,*) 'bad size dim=1 for names in do_profile_info'
                  ierr = -1
                  return
               end if
            else
               write(*,*) 'failed to provide vals array for do_profile_info'
               ierr = -1
               return
            end if
         
         end if
                  
         if (write_flag) then
            io = alloc_iounit(ierr)
            if (ierr /= 0) return
            if (len_trim(s% log_data_header_suffix) == 0) then
               fname1 = fname
            else
               fname1 = trim(fname) // s% log_data_header_suffix
            end if
            open(unit=io, file=trim(fname1), action='write', status='replace', iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed to open ' // trim(fname1)
               call free_iounit(io)
               return
            end if
            do i=1, 3
               col = 0
               call do_integer(i, 'model_number', s% model_number)
               call do_integer(i, 'num_zones', s% nz)
               call do_val(i, 'initial_mass', s% initial_mass)
               call do_val(i, 'initial_z', s% initial_z)
               call do_val(i, 'star_age', s% star_age)
               call do_val(i, 'time_step', s% time_step)
            
               call do_val(i, 'Teff', s% Teff)            
               call do_val(i, 'photosphere_L', s% photosphere_L)
               call do_val(i, 'photosphere_r', s% photosphere_r)
            
               call do_val(i, 'center_eta', s% center_degeneracy)
               call do_val(i, 'center_h1', s% center_h1)
               call do_val(i, 'center_he3', s% center_he3)
               call do_val(i, 'center_he4', s% center_he4)
               call do_val(i, 'center_c12', s% center_c12)
               call do_val(i, 'center_n14', s% center_n14)
               call do_val(i, 'center_o16', s% center_o16)
               call do_val(i, 'center_ne20', s% center_ne20)
               call do_val(i, 'star_mass', s% star_mass)
               call do_val(i, 'star_mdot', s% star_mdot)
               call do_val(i, 'star_mass_h1', s% star_mass_h1)
               call do_val(i, 'star_mass_he3', s% star_mass_he3)
               call do_val(i, 'star_mass_he4', s% star_mass_he4)
               call do_val(i, 'star_mass_c12', s% star_mass_c12)
               call do_val(i, 'star_mass_n14', s% star_mass_n14)
               call do_val(i, 'star_mass_o16', s% star_mass_o16)
               call do_val(i, 'star_mass_ne20', s% star_mass_ne20)
               call do_val(i, 'h1_boundary_mass', s% h1_boundary_mass)
               call do_val(i, 'he4_boundary_mass', s% he4_boundary_mass)
               call do_val(i, 'c12_boundary_mass', s% c12_boundary_mass)
               call do_val(i, 'tau10_mass', s% tau10_mass)
               call do_val(i, 'tau10_radius', s% tau10_radius)
               call do_val(i, 'tau100_mass', s% tau100_mass)
               call do_val(i, 'tau100_radius', s% tau100_radius)
               call do_val(i, 'dynamic_time', s% dynamic_timescale)
               call do_val(i, 'kh_timescale', s% kh_timescale)
               call do_val(i, 'nuc_timescale', s% nuc_timescale)
            
               call do_val(i, 'power_nuc_burn', s% power_nuc_burn)
               call do_val(i, 'power_h_burn', s% power_h_burn)
               call do_val(i, 'power_he_burn', s% power_he_burn)
               call do_val(i, 'power_neu', s% power_neutrinos)
            
               call do_val(i, 'h1_boundary_limit', s% h1_boundary_limit)
               call do_val(i, 'he4_boundary_limit', s% he4_boundary_limit)
               call do_val(i, 'c12_boundary_limit', s% c12_boundary_limit)
            
               call do_val(i, 'burn_min1', s% burn_min1)
               call do_val(i, 'burn_min2', s% burn_min2)

               write(io, *)
            end do
            write(io, *)
         end if

         do i = 1, 3
            if (i==3) then
               n = nz
               if (write_flag .and. len_trim(s% log_data_header_suffix) > 0) then
                  close(io)
                  open(unit=io, file=trim(fname), action='write', status='replace', iostat=ierr)
                  if (ierr /= 0) then
                     write(*,*) 'failed to open ' // trim(fname)
                     call free_iounit(io)
                     return
                  end if
               end if
            else
               n = 1
            end if
            do k=1, n
               col = 0
               do j=1,numcols
                  col = col+1
                  call do_col(i, j, k)
               end do
               do j=1,num_extra_cols
                  call do_extra_col(i, j, k)
               end do
               if (write_flag) write(io, *)
            end do
         end do

         if (associated(extra_col_vals)) deallocate(extra_col_vals)
         if (associated(extra_col_names)) deallocate(extra_col_names)
         
         if (write_flag) then
         
            close(io)
            call free_iounit(io)
         
            s% most_recent_profile_filename = trim(fname1)

            write(*, '(/,a, i7,/)') 'save profile ' // trim(fname1), s% model_number
            if (s% extra_terminal_iounit > 0) &
               write(s% extra_terminal_iounit, '(/,a, i7,/)') &
                  'save profile' // trim(fname1), s% model_number
               
            if (s% write_pulse_info_with_profile) then
               fname1 = trim(fname1) // '.' // trim(s% pulse_info_format)
               call save_pulsation_info( &
                  s% id, s% add_atmosphere_to_pulse_info, s% pulse_info_format, fname1, ierr)
            end if
            
         end if
                  
         
         contains
         
                  
         subroutine do_integer(pass, col_name, val)
            integer, intent(in) :: pass
            character (len=*), intent(in) :: col_name
            integer, intent(in) :: val
            col = col+1
            if (pass == 1) then
               write(io, fmt=int_fmt, advance='no') col
            else if (pass == 2) then
               write(io, fmt=txt_fmt, advance='no') trim(col_name)
            else if (pass == 3) then
               write(io, fmt=int_fmt, advance='no') val
            end if
         end subroutine do_integer
         
                  
         subroutine do_val(pass, col_name, val)
            integer, intent(in) :: pass
            character (len=*), intent(in) :: col_name
            real(dp), intent(in) :: val
            col = col+1
            if (pass == 1) then
               write(io, fmt=int_fmt, advance='no') col
            else if (pass == 2) then
               write(io, fmt=txt_fmt, advance='no') trim(col_name)
            else if (pass == 3) then
               write(io, fmt=dbl_fmt, advance='no') val
            end if
         end subroutine do_val
         
         
         subroutine do_extra_col(pass, j, k)
            use rates_def
            use ionization_lib, only: eval_typical_charge
            integer, intent(in) :: pass, j, k       
            if (pass == 1) then
               if (write_flag) write(io, fmt=int_fmt, advance='no') col
            else if (pass == 2) then
               if (write_flag) then
                  write(io, fmt=txt_fmt, advance='no') trim(extra_col_names(j))
               else
                  names(col) = trim(extra_col_names(j))
               end if
            else if (pass == 3) then
               if (write_flag) then
                  write(io, fmt=dbl_fmt, advance='no') extra_col_vals(k,j)
               else
                  vals(k,col) = extra_col_vals(k,j)
                  is_int(col) = .false.
               end if
            end if
         end subroutine do_extra_col
         
         
         subroutine do_col(pass, j, k)
            use rates_def
            use ionization_lib, only: eval_typical_charge
            use profile_getval, only: getval_for_profile
            integer, intent(in) :: pass, j, k
            integer :: i, c, ii, int_val
            real(dp) :: val, cno, z, dr, eps, eps_alt, tau0
            logical :: int_flag
            character (len=128) :: col_name
            logical, parameter :: dbg = .false.
            include 'formats.dek'
            tau0 = 6d13*secyer ! solar diffusion timescale
            c = s% profile_column_spec(j)
            if (pass == 1) then
               if (write_flag) write(io, fmt=int_fmt, advance='no') col
            else if (pass == 2) then
               if (c > row_log_max_err_ratio_offset) then
                  i = c - row_log_max_err_ratio_offset
                  if (i < 9) then
                     write(col_name,'(a,i1,a)') 'row' , i+1, '_log_max_err_ratio'
                  else
                     write(col_name,'(a,i2,a)') 'row' , i+1, '_log_max_err_ratio'
                  end if
               else if (c > err_ratio_offset) then
                  i = c - err_ratio_offset
                  col_name = 'log_err_ratio_' // trim(s% nameofvar(i))
               else if (c > extra_offset) then
                  i = c - extra_offset
                  col_name = trim(s% profile_extra_name(i))
               else if (c > edv_offset) then
                  i = c - edv_offset
                  col_name = 'edv_' // trim(chem_isos% name(i))
               else if (c > ionization_offset) then
                  i = c - ionization_offset
                  col_name = 'ionization_' // trim(chem_isos% name(i))
               else if (c > xaprev_offset) then
                  i = c - xaprev_offset
                  col_name = 'xaprev_' // trim(chem_isos% name(i))
               else if (c > xadot_offset) then
                  i = c - xadot_offset
                  col_name = 'xadot_' // trim(chem_isos% name(i))
               else if (c > log_abundance_offset) then
                  i = c - log_abundance_offset
                  col_name = 'log_' // trim(chem_isos% name(i))
               else if (c > abundance_offset) then
                  i = c - abundance_offset
                  col_name = trim(chem_isos% name(i))
               else if (c > category_offset) then
                  i = c - category_offset
                  col_name = trim(category_name(i))
               else
                  col_name = trim(profile_column_name(c))
               end if
               if (write_flag) then
                  write(io, fmt=txt_fmt, advance='no') trim(col_name)
               else
                  names(col) = trim(col_name)
               end if
            else if (pass == 3) then
               int_flag = .false.
               if (c > row_log_max_err_ratio_offset) then
                  i = c - row_log_max_err_ratio_offset
                  val = 0
                  if (i > 0 .and. associated(s% seulex_error_vectors)) then
                     if (size(s% seulex_error_vectors, dim=1) >= s% nvar .and. &
                         size(s% seulex_error_vectors, dim=2) >= s% nz .and. &
                         size(s% seulex_error_vectors, dim=3) >= i .and. i < s% seulex_rows) then
                        val = safe_log10(maxval(abs(s% seulex_error_vectors(1:s% nvar,k,i))))
                     end if
                  end if
               else if (c > err_ratio_offset) then
                  i = c - err_ratio_offset
                  val = 0
                  if (i > 0 .and. associated(s% seulex_error_vectors)) then
                     if (size(s% seulex_error_vectors, dim=1) >= s% nvar .and. &
                         size(s% seulex_error_vectors, dim=2) >= s% nz .and. &
                         size(s% seulex_error_vectors, dim=3) >= s% seulex_rows) then
                        val = safe_log10(abs(s% seulex_error_vectors(i,k,s% seulex_rows-1)))
                     end if
                  end if
               else if (c > extra_offset) then
                  i = c - extra_offset
                  val = s% profile_extra(k,i)
               else if (c > edv_offset) then
                  i = c - edv_offset
                  ii = s% net_iso(i)
                  if (ii > 0) then
                     val = s% edv(ii,k) !*tau0/Rsun
                  else
                     val = 0
                  end if
               else if (c > ionization_offset) then
                  i = c - ionization_offset
                  ii = s% net_iso(i)
                  val = eval_typical_charge( &
                     i, s% abar(k), exp(s% lnfree_e(k)), &
                     s% T(k), s% lnT(k)/ln10, s% rho(k), s% lnd(k)/ln10)
               else if (c > xaprev_offset) then
                  i = c - xaprev_offset
                  ii = s% net_iso(i)
                  if (ii > 0) then
                     val = s% xa_pre_hydro(ii,k)
                  else
                     val = 0d0
                  end if
               else if (c > xadot_offset) then
                  i = c - xadot_offset
                  ii = s% net_iso(i)
                  if (ii > 0) then
                     val = s% xa(ii,k) - s% xa_pre_hydro(ii,k)
                  else
                     val = 0d0
                  end if
               else if (c > log_abundance_offset) then
                  i = c - log_abundance_offset
                  ii = s% net_iso(i)
                  if (ii > 0) then
                     val = safe_log10(s% xa(ii,k))
                  else
                     val = -99d0
                  end if
               else if (c > abundance_offset) then
                  i = c - abundance_offset
                  ii = s% net_iso(i)
                  if (ii > 0) then
                     val = s% xa(ii,k)
                  else
                     val = 0d0
                  end if
               else if (c > category_offset) then
                  i = c - category_offset
                  val = s% eps_nuc_categories(i_rate,i,k)
               else
                  call getval_for_profile(s, c, k, val, int_flag, int_val)
               end if
               if (write_flag) then
                  if (int_flag) then
                     write(io, fmt=int_fmt, advance='no') int_val
                  else
                     write(io, fmt=dbl_fmt, advance='no') val
                  end if
               else
                  if (int_flag) then
                     vals(k,col) = dble(int_val)
                     is_int(col) = .true.
                  else
                     vals(k,col) = val
                     is_int(col) = .false.
                  end if
               end if
            end if
         end subroutine do_col
         
         
      end subroutine do_profile_info
      


      end module profile
      
