! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_trho_profile

      use star_private_def
      use const_def
      use mod_pgstar_support

      implicit none


      contains
      



      subroutine TRho_Profile_plot(id, device_id, ierr)
         integer, intent(in) :: id, device_id
         integer, intent(out) :: ierr
         real :: xleft, xright, ybot, ytop, txt_scale
         type (star_info), pointer :: s
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         call pgslct(device_id)
         call pgbbuf()
         call pgeras()
         xleft = 0.14
         xright = 0.94
         ybot = 0.13
         ytop = 0.92
         txt_scale = 1
         call draw_TRho_Profile_plot( &
            s, xleft, xright, ybot, ytop, txt_scale, s% show_HR_TRho_with_TRho_Profile, ierr)
         if (ierr /= 0) return
         call pgebuf()
         if (s% show_TRho_Profile_cross_hair) call do_cross_hair
      end subroutine TRho_Profile_plot
      

      subroutine draw_TRho_Profile_plot(s, xleft, xright, ybot, ytop, txt_scale_in, show_HR, ierr)
         use utils_lib
         use mod_pgstar_hr, only: draw_HR_Plot
         use mod_pgstar_trho, only: draw_TRho_Plot
         use mod_pgstar_conv, only: draw_Convection_Plot
         use num_lib, only: safe_log10

         implicit none

         type (star_info), pointer :: s
         real, intent(in) :: xleft, xright, ybot, ytop, txt_scale_in
         logical, intent(in) :: show_HR
         integer, intent(out) :: ierr

         integer :: nz, k
         real :: xmin, xmax, ymin, ymax, xpos, ypos, dx, dy, &
            txt_scale, vpxmin, vpxmax, vpymin, vpymax, vpymargin, vpwinheight, lgT1, lgT2
         real, pointer, dimension(:) :: xvec, yvec
         character (len=128) :: str
         real, parameter :: lgrho1 = -8, lgrho2 = 5
         
         include 'formats.dek'

         ierr = 0
         nz = s% nz        
         allocate (xvec(nz), yvec(nz))
         
         txt_scale = txt_scale_in
         
         if (s% TRho_switch_to_Column_Depth) then
            forall (k=1:s% nz) &
               xvec(k) = safe_log10(s% xmstar*sum(s% dq(1:k-1))/(4*pi*s% r(k)**2))
         else ! log rho
            xvec(1:nz) = s% lnd(1:nz)/ln10
         end if

         xmin = s% TRho_Profile_xmin
         xmax = s% TRho_Profile_xmax
         dx = xmax - xmin

         ! log T
         yvec(1:nz) = s% lnT(1:nz) / ln10
         ymin = s% TRho_Profile_ymin
         ymax = s% TRho_Profile_ymax
         dy = ymax - ymin
                  
         if (show_HR) then
            txt_scale = 0.75
            vpxmin = 0.12
            vpxmax = 0.35
            vpymin = 0.10
            vpymax = 0.90
            vpymargin = 0.12
            vpwinheight = (vpymax - vpymin - vpymargin)/2
            call draw_HR_Plot(s,vpxmin,vpxmax,vpymax-vpwinheight,vpymax,txt_scale,ierr)
            if (ierr /= 0) return
            call draw_TRho_Plot(s,vpxmin,vpxmax,vpymin,vpymin+vpwinheight,txt_scale,ierr)
            if (ierr /= 0) return
            call pgsvp(0.45,0.94,vpymin,vpymax)
         else ! use the values given as args
            call pgsvp(xleft, xright, ybot, ytop)
         end if
         
         call pgswin(xmin, xmax, ymin, ymax)
         call pgscf(1)
         call pgsci(1)
         call pgsch(1.5*txt_scale)
         
         call pgbox( &
            'BCNST1',s% TRho_Pro_xtick,s% TRho_Pro_nxsub,'BCMNSTV1',s% TRho_Pro_ytick,s% TRho_Pro_nysub)
         if (s% TRho_switch_to_Column_Depth) then
            call pgmtxt('B',3.0,0.5,0.5,'log column depth (g cm\u-2\d)')
         else
            call pgmtxt('B',3.0,0.5,0.5,'log Density (g cm\u-3\d)')
         end if
         call pgmtxt('L',3.0,0.5,0.5,'log Temperature (K)')
         call pgsch(1.3*txt_scale)
         
         if (s% TRho_switch_to_Column_Depth) then
            call pgmtxt('T',1.5,0.5,0.5,'Temperature -- Column Depth')
         else
            call pgmtxt('T',1.5,0.5,0.5,'Temperature Density Profile')
         end if
         
         call pgsch(1.2*txt_scale)
         write(str,'(i9)') s% model_number
         call pgmtxt('T',1.8,0.9,0.5,str)
         
         if (.not. s% TRho_switch_to_Column_Depth) then
            if (s% show_TRho_Profile_kap_regions) call do_kap_regions         
            if (s% show_TRho_Profile_eos_regions) call do_eos_regions
            if (s% show_TRho_Profile_degeneracy_line) call do_degeneracy_line      
            if (s% show_TRho_Profile_Pgas_Prad_line) call do_Pgas_Prad_line      
            if (s% show_TRho_Profile_burn_lines) call do_burn_lines
         end if
      
         call show_profile_line(s, xvec, yvec, txt_scale, xmin, xmax, ymin, ymax, &
               s% show_TRho_Profile_legend, s% TRho_Profile_legend_coord, &
               s% TRho_Profile_legend_disp1, s% TRho_Profile_legend_del_disp, s% TRho_Profile_legend_fjust, &
               s% show_TRho_Profile_mass_locs)
      
         if (s% show_TRho_Profile_text_info) &
            call do_show_Profile_text_info(s, txt_scale, &
                     xmin, xmax, ymin, ymax, &
                     s% TRho_Profile_text_info_xfac, s% TRho_Profile_text_info_dxfac, &
                     s% TRho_Profile_text_info_yfac, s% TRho_Profile_text_info_dyfac, &
                     .false., .false.)
      
         if (s% TRho_Profile_show_logQ_limit .and. .not. s% TRho_switch_to_Column_Depth) &
            call do_show_logQ_limit
      
         call show_annotations(s, &
            s% show_TRho_Profile_annotation1, s% show_TRho_Profile_annotation2, s% show_TRho_Profile_annotation3)

         deallocate(xvec, yvec)
                  

         
         contains
         
         
         subroutine do_show_logQ_limit ! logQ = logRho - 2*logT + 12.
            real :: logRho1, logRho2, logT1, logT2, logQ
            logQ = s% logQ_limit
            logRho1 = xmin
            logRho2 = xmax
            logT1 = (logRho1 - logQ + 12)/2
            logT2 = (logRho2 - logQ + 12)/2
            call pgsave
            call pgsci(clr_Gray)
            call pgsls(Line_Type_Dash)
            call pgmove(logRho1, logT1)
            call pgdraw(logRho2, logT2)
            call pgunsa
         end subroutine do_show_logQ_limit
         
         
         subroutine do_degeneracy_line
            call pgsave
            call pgsci(clr_Gray)
            call pgsls(Line_Type_Dash)
            call pgline(size(psi4_logT), psi4_logRho, psi4_logT)
            call pgsls(Line_Type_Solid)
            xpos = psi4_logRho(1)
            ypos = psi4_logT(1)-dy*0.04
            if (inside(xpos, ypos)) call pgptxt(xpos, ypos, 0.0, 0.5, '\ge\dF\u/kT\(0248)4')
            call pgunsa
         end subroutine do_degeneracy_line
         
         
         subroutine add_TR_line(logR1, logT1, logR2, logT2)
            real, intent(in) :: logR1, logT1, logR2, logT2
            real :: logRho1, logRho2
            logRho1 = logR1 + 3 * logT1 - 18
            logRho2 = logR2 + 3 * logT2 - 18
            call pgmove(logRho1, logT1)
            call pgdraw(logRho2, logT2)
         end subroutine add_TR_line
         
         
         subroutine show_label(xpos, ypos, angle, justification, txt)
            real, intent(in) :: xpos, ypos, angle, justification
            character (len=*), intent(in) :: txt
            if (inside(xpos, ypos)) call pgptxt(xpos, ypos, angle, justification, txt)
         end subroutine show_label
         
         
         subroutine do_kap_regions
            real :: logT_lo, logT_hi, logT_max, logR, logRho_lo, logRho_hi, logRho_min
            real, parameter :: min_logR_for_freedman = 1
            real, parameter :: freg_blend_logT2 = 4.10
            real, parameter :: freg_blend_logT1 = 3.93
            
            call pgsave        

            call pgsci(clr_Coral)
            call pgsls(Line_Type_Solid)
            logT_lo = 2.7; logT_hi = 8.7; logT_max = 10.3
            call add_TR_line(-8.0, logT_lo, -8.0, logT_hi)
            call add_TR_line(1.0, logT_lo, 1.0, logT_hi)
            call add_TR_line(1.0, logT_lo, -8.0, logT_lo)
            call add_TR_line(1.0, logT_hi, -8.0, logT_hi)
            call add_TR_line(1.0, 2.7, -8.0, 2.7)
            call add_TR_line(1.0, freg_blend_logT1, -8.0, freg_blend_logT1)
            call add_TR_line(1.0, freg_blend_logT2, -8.0, freg_blend_logT2)
            call add_TR_line(1.0, 8.2, -8.0, 8.2)

            call pgsci(1)
            call add_TR_line(-8.0, logT_hi, -8.0, logT_max)
            call add_TR_line(-8.0, logT_max, 8.0, logT_max)
            !call add_TR_line(8.0, logT_lo, 8.0, logT_hi)
            !call add_TR_line(1.0, logT_lo, 8.0, logT_lo)
            
            ! Freedman
            call pgsci(clr_Tan)
            call pgmove(-8.8,1.88)
            call pgdraw(-3.36,1.88)
            call pgdraw(-1.5,2.5)
            call pgdraw(-2.6,3.6)
            call pgdraw(-11.3,3.6)
            call pgdraw(-9.5,1.88)
            call pgdraw(-8.8,1.88)
            
            
            call pgsci(1)
            call show_label(-4.9, 2.47, 0.0, 0.5, 'FREEDMAN')
            call show_label(-8.5, 3.3, 0.0, 0.5, 'FERGUSON')
            call show_label(-7.5, 5.1, 0.0, 0.5, 'OPAL/OP')
            call show_label(5.5, 9.0, 0.0, 0.5, 'COMPTON')
            call show_label(1.8, 8.35, 0.0, 0.5, 'BLEND')
            call show_label(-8.5, (freg_blend_logT1 + freg_blend_logT2)/2, 0.0, 0.5, 'BLEND')
            call show_label(0.2, 3.9, 0.0, 1.0, '\(0636)\drad\u = \(0636)\dcond\u')
            call pgsci(clr_Crimson)
            call show_label(3.8, 9.4, 0.0, 0.5, 'e\u-\de\u+\d')
            call pgsci(1)
            
            call show_label(-6.8, 6.9, 0.0, 0.5, 'logR = -8')
            call show_label(5.0, 6.9, 0.0, 0.5, 'logR = 1')
            call show_label(2.8, 3.8, 0.0, 0.5, 'logR = 8')
            
            ! show where electron to baryon ratio is twice that expected
            call pgsci(clr_Crimson)
            call pgsls(Line_Type_Dash)
            call pgline(size(elect_data_logT), elect_data_logRho, elect_data_logT)
            ! show where kap_rad == kap_cond
            call pgsci(clr_LightSkyBlue)
            call pgsls(Line_Type_Dot)
            call pgline(size(kap_rad_cond_eq_logT), kap_rad_cond_eq_logRho, kap_rad_cond_eq_logT)
            call pgunsa
         end subroutine do_kap_regions
         
         
         subroutine do_eos_regions
            use eos_lib, only: eos_get_PC_parameters
            real*8 :: mass_fraction_limit_for_PC, &
               logRho1_PC_limit, logRho2_PC_limit, &
               log_Gamma_all_HELM, log_Gamma_all_PC, PC_min_Z, &
               logT_hi, logT_lo, logRho, logT, logG0, abar, zbar
            integer :: ierr
            real :: logRho1, logRho2, logRho3, logRho4, logRho5, logRho6, logRho7
            real :: logT1, logT2, logT3, logT4, logT5, logT6, logT7, logT8
            real :: logRho0
            
            call pgsave
            
            ! blend from table to non-table
            call pgsci(clr_LightSkyGreen)
            call pgsls(Line_Type_Dash)
            
            logRho1 =  2.7
            logRho2 =  2.5
            logRho3 =  -1.71
            logRho4  = -2.21
            logRho5  = -9.0
            logRho6  = -9.99
            logRho7  = -12
            logT1  =  7.7
            logT2 =   7.6
            logT3  =  4.65
            logT4  =  4.75
            logT5  =  3.60
            logT6  =  3.50
            logT7 =   2.3
            logT8  =  2.2

            call stroke_line(logRho2, logT1, logRho7, logT1)        
            call stroke_line(logRho2, logT2, logRho7, logT2)        
            call stroke_line(logRho2, logT1, logRho1, logT2)        
            call stroke_line(logRho1, logT2, logRho1, logT3)        

            call stroke_line(logRho4, logT7, logRho5, logT7)        
            call stroke_line(logRho4, logT8, logRho5, logT8)        
            call stroke_line(logRho4, logT8, logRho3, logT7)        
            call stroke_line(logRho6, logT7, logRho5, logT8)      
            
            call stroke_line(logRho2, logT2, logRho2, logT4)      
            call stroke_line(logRho3, logT7, logRho1, logT3)      
            call stroke_line(logRho4, logT7, logRho2, logT4)      

            call stroke_line(logRho5, logT7, logRho5, logT6)      
            call stroke_line(logRho6, logT7, logRho6, logT6)      
            call stroke_line(logRho5, logT6, logRho6, logT5)    
              
            call stroke_line(logRho6, logT5, logRho7, logT5)      
            call stroke_line(logRho6, logT6, logRho7, logT6)      

            ! blend from OPAL to SCVH
            call pgsci(clr_LightSkyBlue)
            call pgsls(Line_Type_Dot)
            
            logRho0 = logRho1
            logRho1 = 2.2
            logRho2 = 1.2
            logRho3 = -2.0
            logRho4 = -3.8
            logRho5 = -5.8
            logRho6 = -6.8
            logRho7 = -10
            logT1 = 6.6
            logT2 = 6.5
            logT3 = 4.0
            logT4 = 3.4
            logT5 = 3.3

            call stroke_line(logRho0, logT1, logRho2, logT1)        
            call stroke_line(logRho2, logT1, logRho4, logT3)        
            call stroke_line(logRho4, logT3, logRho5, logT4)        
            call stroke_line(logRho5, logT4, logRho7, logT4)        

            call stroke_line(logRho0, logT2, logRho1, logT2)        
            call stroke_line(logRho1, logT2, logRho3, logT3)        
            call stroke_line(logRho3, logT3, logRho5, logT5)        
            call stroke_line(logRho5, logT5, logRho7, logT5)        
            
            call pgsci(1)            
            call show_label(8.6, 8.6, 0.0, 0.5, 'HELM')
            call show_label(-7.2, 5.8, 0.0, 0.5, 'OPAL')
            call show_label(-0.8, 3.7, 0.0, 0.5, 'SCVH')
            call show_label(7.1, 5.1, 0.0, 0.5, 'PC')
            ! show where gamma1 = 4/3
            call pgsci(clr_Crimson)
            call pgsls(Line_Type_Dash)
            call pgline(size(gamma_4_thirds_logT), gamma_4_thirds_logRho, gamma_4_thirds_logT)
            call show_label(3.8, 9.2, 0.0, 0.5, '\(0529)\d1\u < 4/3')
            if (.false.) then ! show where Gamma = 170 for pure oxygen
               logRho1 = 3.8; logT1 = 4.53128 + logRho1/3
               logRho2 = 14; logT2 = 4.53128 + logRho2/3
               call pgsci(1)
               call stroke_line(logRho1, logT1, logRho2, logT2)
               call show_label(6.8, 6.9, 0.0, 1.0, '\u16\dO \(0529)=170')
            end if
            call eos_get_PC_parameters( &
               s% eos_handle, mass_fraction_limit_for_PC,  &
               logRho1_PC_limit, logRho2_PC_limit, &
               log_Gamma_all_HELM, log_Gamma_all_PC, PC_min_Z, ierr)
            if (ierr == 0) then ! show HELM/PC blend
               abar = s% abar(s% nz)
               zbar = s% zbar(s% nz)
               ! gamma = (zbar*qe)**2*((4.0d0/3.0d0)*pi*avo*Rho/abar)**one_third/(kerg*T)
               ! logG = logG0 + logRho/3 - logT
               logG0 = log10((zbar*qe)**2*((4.0d0/3.0d0)*pi*avo/abar)**one_third/kerg)
               logT_lo = logG0 + logRho2_PC_limit/3 - log_Gamma_all_HELM
               logT_hi = logG0 + logRho1_PC_limit/3 - log_Gamma_all_PC
               logRho = 14
               logT = logG0 + logRho/3 - log_Gamma_all_PC
               logRho1 = logRho; logT1 = logT
               logRho2 = logRho1_PC_limit; logT2 = logT_hi
               call pgsls(Line_Type_Dash_Dot)
               call pgsci(clr_Crimson)
               call pgmove(logRho1, logT1)
               call pgdraw(logRho2, logT2)
               call pgdraw(logRho2, 1.0)
               logT = logG0 + logRho/3 - log_Gamma_all_HELM
               logRho1 = logRho; logT1 = logT
               logRho2 = logRho2_PC_limit; logT2 = logT_lo
               call pgmove(logRho1, logT1)
               call pgdraw(logRho2, logT2)
               call pgdraw(logRho2, 1.0)
            end if
            call pgunsa
         end subroutine do_eos_regions
         
         
         subroutine stroke_line(x1, y1, x2, y2)
            real, intent(in) :: x1, y1, x2, y2
            call pgmove(x1, y1)
            call pgdraw(x2, y2)
         end subroutine stroke_line
         
         
         
         subroutine do_Pgas_Prad_line
            lgT1 = log10(3.2d7) + (lgRho1 - log10(0.7d0))/3.0
            lgT2 = log10(3.2d7) + (lgRho2 - log10(0.7d0))/3.0
            call pgsave
            call pgsci(clr_Gray)
            call pgsls(Line_Type_Dash)
            call pgmove(lgRho1, lgT1)
            call pgdraw(lgRho2, lgT2)
            call pgsls(Line_Type_Solid)
            xpos = lgRho1-dx*0.065
            ypos = lgT1-dy*0.025
            if (inside(xpos, ypos)) call pgptxt(xpos, ypos, 0.0, 0.0, 'P\drad\u\(0248)P\dgas\u')
            call pgunsa
         end subroutine do_Pgas_Prad_line
         
         
         subroutine do_burn_lines
            call pgsave
            call pgsci(clr_Gray)
            call pgsls(Line_Type_Dash)
            call write_burn_line(hydrogen_burn_logRho, hydrogen_burn_logT, 'Hydrogen burn')
            call write_burn_line(helium_burn_logRho, helium_burn_logT, 'Helium burn')
            call write_burn_line(carbon_burn_logRho, carbon_burn_logT, 'Carbon burn')
            call write_burn_line(oxygen_burn_logRho, oxygen_burn_logT, 'Oxygen burn')
            call pgsls(Line_Type_Solid)
            call pgunsa
         end subroutine do_burn_lines
         
         
         logical function inside(xpos, ypos)
            real, intent(in) :: xpos, ypos
            inside = .false.
            if (xpos <= s% TRho_Profile_xmin .or. xpos >= s% TRho_Profile_xmax) return
            if (ypos <= s% TRho_Profile_ymin .or. ypos >= s% TRho_Profile_ymax) return
            inside = .true.
         end function inside
         
         
         subroutine write_burn_line(logRho, logT, label)
            real, dimension(:), pointer :: logRho, logT
            character (len=*), intent(in) :: label
            integer :: sz
            real :: xpos, ypos
            character (len=128) :: str
            sz = size(logRho)
            call pgline(sz, logRho, logT)
            if (.not. s% show_TRho_Profile_burn_labels) return
            xpos = logRho(sz)
            ypos = logT(sz)
            if (.not. inside(xpos,ypos)) return
            write(str,'(a)') trim(label)
            call pgptxt(xpos, ypos, 0.0, 1.0, trim(adjustl(str)))
         end subroutine write_burn_line


      end subroutine draw_TRho_Profile_plot

      
      subroutine do_show_Profile_text_info( &
            s, txt_scale, xmin, xmax, ymin, ymax, xfac, dxfac, yfac, dyfac, &
            xaxis_reversed, yaxis_reversed)
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         real, intent(in) :: txt_scale, xmin, xmax, ymin, ymax, xfac, dxfac, yfac, dyfac
         logical, intent(in) :: xaxis_reversed, yaxis_reversed
         
         real :: dxpos, xpos0, dxval, ypos, dypos
         real*8 :: age
         integer :: cnt
         
         include 'formats.dek'
      
         call pgsave
         call pgsch(1.1*txt_scale)
         !call pgsci(clr_Silver)
         call pgsci(1)
         dxpos = 0
         xpos0 = xmin + xfac*(xmax-xmin)
         dxval = dxfac*(xmax-xmin)
         if (xaxis_reversed) dxval = -dxval
         ypos = ymin + yfac*(ymax-ymin)
         dypos = dyfac*(ymax-ymin)
         if (yaxis_reversed) dypos = -dypos
      
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'mass', s% star_mass)
      
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'H rich', s% star_mass - s% h1_boundary_mass)
      
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'H poor', s% h1_boundary_mass)
      
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'He poor', s% he4_boundary_mass)
      
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'lg mdot', safe_log10(abs(s% star_mdot)))
               
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt2(cnt, ypos, xpos0, dxpos, dxval, &
                  'Teff', s% Teff)
               
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'lg L', s% log_surface_luminosity)
               
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'lg LH', safe_log10(s% power_h_burn))
               
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'lg LHe', safe_log10(s% power_he_burn))

         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'lg R', s% log_surface_radius)
               
         cnt = 0; ypos = ypos + dypos
         cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                  'lg dt yr', log10(s% time_step))
               
         cnt = 0; ypos = ypos + dypos
         age = s% star_age + s% profile_age_shift
         !write(*,2) 'age', s% model_number, age, s% star_age, s% profile_age_shift
         cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                  'age yr', age)

            call pgunsa

      end subroutine do_show_Profile_text_info


      end module mod_pgstar_trho_profile

