! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_surf_hist

      use star_private_def
      use const_def
      use mod_pgstar_support

      implicit none


      contains
      

      subroutine surf_hist_plot(id, device_id, ierr)
         use utils_lib
         use chem_def
         use num_lib, only: safe_log10
         use net_def
         use net_lib, only: get_net_reaction_table
         use rates_def, only: rates_reaction_id_max
         use const_def, only: Msun, Rsun
         implicit none

         integer, intent(in) :: id, device_id
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s
         character (len=256) :: str
         real, pointer, dimension(:) :: xvec, yvec
         real :: xmin, xmax, chScale, winxmin, winxmax, winymin, winymax, windy, &
            ymin, ymax, dy, &
            header_frac, footer_frac, plot2_frac, plot3_frac, plot4_frac, plot1_frac
         integer :: grid_min, grid_max, numpts, i, lw, lw_sav, cv_cur
         type (pgstar_data), dimension(:), pointer :: cv
         real, parameter :: margin = 0.08
         
         include 'formats.dek'

         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return

         cv_cur = cv_current(id)
         cv => cv_for_star(id)% cv_data

         if (s% Cntr_Hist_xmin >= cv(cv_cur)% step) s% Cntr_Hist_xmin = -1
         xmin = min(cv(cv_cur)% step, max(cv(1)% step, s% Cntr_Hist_xmin))
         xmax = s% Cntr_Hist_xmax
         if (xmax <= xmin) xmax = cv(cv_cur)% step
         if (s% Cntr_Hist_max_width > 0) xmin = max(xmin, xmax - s% Cntr_Hist_max_width)
         if (xmax <= xmin) xmax = cv(cv_cur)% step
         if (xmax <= xmin) return

         
         call pgslct(device_id)
         call pgbbuf()
         call pgeras()

         allocate (xvec(cv_cur), yvec(cv_cur))

         chScale = 1.0
         winxmin = 0.15
         winxmax = 0.82
         winymin = 0.02
         winymax = 0.93
         windy = winymax - winymin
         
         header_frac = 0.02
         footer_frac = 0.08
         plot2_frac = (1 - (header_frac + footer_frac))/4
         plot3_frac = plot2_frac
         plot4_frac = plot2_frac
         plot1_frac = plot2_frac
         
         call set_grid_min_max(s, xmin, xmax, grid_min, grid_max, numpts, ierr)
         if (ierr /= 0) then
            deallocate(xvec, yvec)
            return
         end if
         
         grid_max = min(grid_max,size(cv,dim=1))
         grid_min = min(grid_min,grid_max)
         numpts = grid_max - grid_min + 1
         if (numpts <= 0) then
            deallocate(xvec, yvec)
            return
         end if

         forall (i=grid_min:grid_max) xvec(i) = cv(i)% step       
                       
         lw = 5
         call pgqlw(lw_sav)
         
         call plot2
         call plot3
         call plot4
         call plot1
         call header
         call footer
         
         call show_annotations(s, &
            s% show_Surf_Hist_annotation1, s% show_Surf_Hist_annotation2, s% show_Surf_Hist_annotation3)

         call pgebuf()

         deallocate(xvec, yvec)


         contains
                  
         
         subroutine plot1
            logical, parameter :: dbg = .false.
         
            include 'formats.dek'
            call pgsave
            
            call set_plot1_vp         
            call pgsch(chScale*1.2)
            call pgmtxt('T',1.5,0.5,0.5,'Surface History')
         
            call pgsch(chScale*0.9)
            write(str,'(i9)') s% model_number
            call pgmtxt('T',1.5*1.2/0.9,0.9,0.5,str)
         
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% log_surface_R
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))
            dy = max(ymax-ymin, 0.08)
            ymax = ymax + margin*dy
            ymin = ymin - margin*dy
         
            call pgswin(xmin, xmax, ymin, ymax)

            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('BCST',0.0,0,'CMSTV',0.0,0)
            call pgsci(clr_Coral)
            call pgmtxt('R',5.0,0.5,0.5,'log R')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% log_surface_Rho
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))        
            ymax = max(ymax, ymin+1e-6)
            dy = max(ymax-ymin, 0.08)
            ymin = ymin-dy*margin
            ymax = ymax + dy*margin

            call pgswin(xmin, xmax, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('',0.0,0,'BNSTV',0.0,0)
            call pgsci(clr_Lilac)
            call pgmtxt('L',4.0,0.5,0.5,'log Rho')
            call pgslw(lw)
            !call pgsls(Line_Type_Dot)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgunsa
         end subroutine plot1


         subroutine plot2
            integer :: k
            logical, parameter :: dbg = .false.
         
            include 'formats.dek'
            call pgsave
                                   
            call set_plot2_vp         
            call pgsch(chScale*1.2)
         
            forall (k=grid_min:grid_max) yvec(k) = &
               safe_log10(cv(k)% cgrav*Msun*cv(k)% star_mass) - 2*(cv(k)% log_surface_R + log10(Rsun))
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))
            dy = max(ymax-ymin, 0.08)
            ymax = ymax + margin*dy
            ymin = ymin - margin*dy
         
            call pgswin(xmin, xmax, ymin, ymax)

            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('BCST',0.0,0,'CMSTV',0.0,0)
            call pgsci(clr_Crimson)
            call pgmtxt('R',5.0,0.5,0.5,'log g')

            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% log_surface_P
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))        
            ymax = max(ymax, ymin+1e-6)
            dy = max(ymax-ymin, 0.08)
            ymin = ymin - dy*margin
            ymax = ymax + dy*margin

            call pgswin(xmin, xmax, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('',0.0,0,'BNSTV',0.0,0)
            call pgsci(clr_LightSkyGreen)
            call pgmtxt('L',4.0,0.5,0.5,'log P')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgunsa
         end subroutine plot2
                  
         
         subroutine plot3
            logical, parameter :: dbg = .false.
         
            include 'formats.dek'
            call pgsave
            
            call set_plot3_vp         
         
            yvec(grid_min:grid_max) = 10.0**(cv(grid_min:grid_max)% log_Teff)
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))
            dy = max(ymax-ymin, 0.08)
            ymax = ymax + margin*dy
            ymin = ymin - margin*dy
         
            call pgswin(xmin, xmax, ymin, ymax)

            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('BCST',0.0,0,'CMSTV',0.0,0)
            call pgsci(clr_Goldenrod)
            call pgmtxt('R',5.0,0.5,0.5,'T\deff')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% log_L
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))        
            ymax = max(ymax, ymin+1e-6)
            dy = max(ymax-ymin, 0.08)
            ymin = ymin-dy*margin
            ymax = ymax + dy*margin

            call pgswin(xmin, xmax, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('',0.0,0,'BNSTV',0.0,0)
            call pgsci(clr_LightSkyBlue)
            call pgmtxt('L',4.0,0.5,0.5,'log L')
            call pgslw(lw)
            !call pgsls(Line_Type_Dot)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgunsa
         end subroutine plot3


         subroutine plot4
            logical, parameter :: dbg = .false.
         
            include 'formats.dek'
            call pgsave
            
            call set_plot4_vp         
         
            yvec(grid_min:grid_max) = log10(max(1e-30,abs(cv(grid_min:grid_max)% star_mdot)))
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))
            dy = max(ymax-ymin, 0.08)
            ymax = ymax + margin*dy
            ymin = ymin - margin*dy
         
            call pgswin(xmin, xmax, ymin, ymax)

            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('BCNST1',0.0,0,'CMSTV',0.0,0)
            call pgsci(clr_MediumSlateBlue)
            call pgmtxt('R',5.0,0.5,0.5,'log Mdot')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% star_mass
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))        

            ymax = max(ymax, ymin+1e-6)
            dy = max(ymax-ymin, 0.08)
            ymin = ymin-dy*margin
            ymax = ymax + dy*margin

            call pgswin(xmin, xmax, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('',0.0,0,'BNSTV',0.0,0)
            call pgsci(clr_Gold)
            call pgmtxt('L',4.0,0.5,0.5,'M')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgunsa
         end subroutine plot4


         subroutine header
         end subroutine header


         subroutine footer
         end subroutine footer


         subroutine set_header_vp
            real :: ymax, ymin
            ymax = winymax
            ymin = ymax - header_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_header_vp


         subroutine set_plot1_vp
            real :: ymax, ymin
            ymax = winymax - header_frac*windy
            ymin = ymax - plot1_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot1_vp


         subroutine set_plot2_vp
            real :: ymax, ymin
            ymax = winymax - (header_frac + plot1_frac)*windy
            ymin = ymax - plot2_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot2_vp


         subroutine set_plot3_vp
            real :: ymax, ymin
            ymax = winymax - (header_frac + plot1_frac + plot2_frac)*windy
            ymin = ymax - plot3_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot3_vp


         subroutine set_plot4_vp
            real :: ymax, ymin
            ymax = winymax - (header_frac + plot1_frac + plot2_frac + plot3_frac)*windy
            ymin = ymax - plot4_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot4_vp


         subroutine set_footer_vp
            real :: ymax, ymin
            ymin = winymin
            ymax = ymin + footer_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_footer_vp
         

      end subroutine surf_hist_plot


      end module mod_pgstar_surf_hist

