! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_support

      use star_private_def
      use const_def
      use rates_def, only: i_rate
      

      implicit none
      
      integer, parameter :: category_offset = 1000
      integer, parameter :: abundance_offset = 2000
      integer, parameter :: extras_offset = 3000

      logical :: have_initialized_pgstar = .false.
      
      real :: sum_dHR_since_last_file_write

      
      ! lines for TRho profile
      real, dimension(:), pointer :: hydrogen_burn_logT, hydrogen_burn_logRho
      real, dimension(:), pointer :: helium_burn_logT, helium_burn_logRho
      real, dimension(:), pointer :: carbon_burn_logT, carbon_burn_logRho
      real, dimension(:), pointer :: oxygen_burn_logT, oxygen_burn_logRho
      real, dimension(:), pointer :: psi4_logT, psi4_logRho
      real, dimension(:), pointer :: elect_data_logT, elect_data_logRho
      real, dimension(:), pointer :: gamma_4_thirds_logT, gamma_4_thirds_logRho
      real, dimension(:), pointer :: kap_rad_cond_eq_logT, kap_rad_cond_eq_logRho
      real, dimension(:), pointer :: opal_clip_logT, opal_clip_logRho
      real, dimension(:), pointer :: scvh_clip_logT, scvh_clip_logRho
      
      integer :: clr_no_mixing, clr_convection, clr_semiconvection, &
         clr_thermo_haline, clr_overshoot
         
      ! Tioga line types
      integer, parameter :: Line_Type_Solid = 1
      integer, parameter :: Line_Type_Dash = 2
      integer, parameter :: Line_Type_Dot_Dash = 3
      integer, parameter :: Line_Type_Dash_Dot = Line_Type_Dot_Dash
      integer, parameter :: Line_Type_Dot = 4

      type pgstar_data     
         integer :: step
         real*8 :: age
         real :: cgrav, star_mass, star_mdot
         real :: &
            conv_mx1_top, conv_mx1_bot, conv_mx2_top, conv_mx2_bot, &
            mx1_top, mx1_bot, mx2_top, mx2_bot
         real :: burn_zone_mass(4,3)
         real :: h1_mass, he4_mass, c12_mass
         real :: log_L, log_LH, log_LHe, log_Lneu
         real :: log_Teff, log_surface_R, log_surface_P, log_surface_Rho
         real :: log_center_T, log_center_Rho, log_center_P
         real :: center_degeneracy, center_gamma, center_ye, center_entropy
         real :: center_h1, center_he4, center_c12, center_n14, center_o16, center_ne20
         real :: surface_h1, surface_he3, surface_he4, surface_c12, surface_n14, surface_o16
      end type pgstar_data
      
      type pgstar_ptr
         type (pgstar_data), dimension(:), pointer :: cv_data
      end type pgstar_ptr
      
      type (pgstar_ptr) :: cv_for_star(max_star_handles)
      integer :: cv_current(max_star_handles)
      integer :: last_data_lim(max_star_handles)
      


      contains
      
      
      subroutine pgstar_clear(s)
         type (star_info), pointer :: s
         integer :: i, id
         type (pgstar_win_file_data), pointer :: p
         id = s% id
         last_data_lim(id) = 0
         if (associated(cv_for_star(id)% cv_data)) then
            deallocate(cv_for_star(id)% cv_data); nullify(cv_for_star(id)% cv_data)
         end if
         cv_current(id) = 0
         if (have_initialized_pgstar) return
         do i = 1, num_pgstar_plots
            p => s% pgstar_win_file_ptr(i)
            p% id_win = 0
         end do
      end subroutine pgstar_clear
      
      
      subroutine init_pgstar(s,n,ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: n
         integer, intent(out) :: ierr
         
         integer :: i
         type (pgstar_win_file_data), pointer :: p
         
         call read_support_info(ierr)
         if (failed('')) return

         ierr = 0
         do i = 1, num_pgstar_plots
            p => s% pgstar_win_file_ptr(i)
            call do_open(p, ierr)
            if (ierr /= 0) return       
         end do

         have_initialized_pgstar = .true.
         sum_dHR_since_last_file_write = 0
         
         contains
         
         
         subroutine do_open(p, ierr)
            type (pgstar_win_file_data), pointer :: p
            integer, intent(out) :: ierr
            character (len=256) :: name
            ierr = 0
            p% do_win = p% win_flag
            if (p% do_win) call open_device(s, p, .false., '/xwin', p% id_win, ierr)
            if (failed(p% name)) return
            p% do_file = p% file_flag
            if (p% do_file) then
               call create_file_name(s, p% file_dir, p% file_prefix, s% model_number, name)
               write(*,*) trim(name)
               name = trim(name) // '/' // trim(s% file_device)
               call open_device(s, p, .true., name, p% id_file, ierr)
               if (failed(p% name)) return
            end if
         end subroutine do_open
         
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
            end if
         end function failed
         
         
      end subroutine init_pgstar
      
      

      subroutine check_windows(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: i
         type (pgstar_win_file_data), pointer :: p

         ierr = 0
         do i = 1, num_pgstar_plots
            p => s% pgstar_win_file_ptr(i)
            call check1win(p, ierr)
            if (ierr /= 0) return
            call check1(p, ierr)
            if (ierr /= 0) return       
         end do

         
         contains
         
         
         subroutine check1win(p, ierr)
            type (pgstar_win_file_data), pointer :: p
            integer, intent(out) :: ierr
            ierr = 0
            if (p% do_win .and. (.not. p% win_flag)) then
               p% do_win = .false.
               if (p% id_win /= 0) then
                  call pgslct(p% id_win)
                  call pgclos
                  p% id_win = 0
               endif
            else if (p% win_flag .and. (.not. p% do_win)) then
               if (p% id_win == 0) &
                  call open_device(s, p, .false., '/xwin', p% id_win, ierr)
               if (ierr == 0) p% do_win = .true.
            end if
         end subroutine check1win
         
         
         subroutine check1(p, ierr)
            type (pgstar_win_file_data), pointer :: p
            integer, intent(out) :: ierr
            include 'formats.dek'
            ierr = 0
            if (p% do_win .and. (p% win_width /= p% prev_win_width .or. &
                                 p% win_aspect_ratio /= p% prev_win_aspect_ratio)) then
               call pgslct(p% id_win)
               call pgpap(p% win_width, p% win_aspect_ratio)
               p% prev_win_width = p% win_width
               p% prev_win_aspect_ratio = p% win_aspect_ratio
            end if
         end subroutine check1
         
         
      end subroutine check_windows
      

      subroutine check_files(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: i
         type (pgstar_win_file_data), pointer :: p

         ierr = 0
         do i = 1, num_pgstar_plots
            p => s% pgstar_win_file_ptr(i)
            call check1(p, ierr)
            if (ierr /= 0) return       
         end do

         
         contains
         
         subroutine check1(p, ierr)
            type (pgstar_win_file_data), pointer :: p
            integer, intent(out) :: ierr
            character (len=256) :: name            
            ierr = 0
            if (p% do_file .and. (.not. p% file_flag)) then
               p% do_file = .false.
            else if (p% file_flag .and. (.not. p% do_file)) then
               if (p% id_file == 0) then
                  call create_file_name(s, p% file_dir, p% file_prefix, s% model_number, name)
                  write(*,*) trim(name)
                  name = trim(name) // '/' // trim(s% file_device)
                  call open_device(s, p, .true., name, p% id_file, ierr)
                  !write(*,*) trim(name) // ' id_file', p% id_file
               end if                  
               p% do_file = .true.
            end if
         end subroutine check1
         
            
      end subroutine check_files      
            
      
      subroutine create_file_name(s, dir, prefix, n, name)
         type (star_info), pointer :: s
         character (len=*), intent(in) :: dir, prefix
         integer, intent(in) :: n
         character (len=*), intent(out) :: name
         integer :: len
         len = len_trim(dir)
         if (len > 0) then
            name = trim(dir) // '/'
         else
            name = ''
         end if
         name = trim(name) // trim(prefix)
         if (n > 9999) then
            write(name,'(a,i5)') trim(name), n
         else if (n > 999) then
            write(name,'(a,i4)') trim(name) // '0', n
         else if (n > 99) then
            write(name,'(a,i3)') trim(name) // '00', n
         else if (n > 9) then
            write(name,'(a,i2)') trim(name) // '000', n
         else
            write(name,'(a,i1)') trim(name) // '0000', n
         end if
         name = trim(name) // '.' // trim(s% file_extension)
      end subroutine create_file_name
      
      
      subroutine open_device(s, p, is_file, dev, id, ierr)
         type (star_info), pointer :: s
         type (pgstar_win_file_data), pointer :: p
         logical, intent(in) :: is_file
         character (len=*), intent(in) :: dev
         integer, intent(out) :: id
         integer, intent(out) :: ierr
         
         integer :: pgopen, system
         character (len=256) :: dir, cmd
         logical :: white_on_black_flag
         real :: width, ratio
         
         if (is_file) then
            dir = p% file_dir
            white_on_black_flag = s% file_white_on_black_flag
         else
            dir = ''
            white_on_black_flag = s% win_white_on_black_flag
         end if
         
         ierr = 0
         id = -1

         retry_loop: do
            id = pgopen(trim(dev))
            if (id > 0) exit
            if (len_trim(dir) > 0) then ! check to see if dir exists
               cmd = 'cd ' // trim(dir)
               ierr = system(trim(cmd))
               if (ierr /= 0) then ! try to create directory
                  cmd = 'mkdir ' // trim(dir)
                  ierr = system(trim(cmd))
                  if (ierr == 0) then
                     write(*,*) 'automatically created ' // trim(dir)
                     cycle retry_loop
                  end if
               end if
            end if
            write(*,*) 
            write(*,*) 
            write(*,*) 'PGSTAR: failed to open ' // trim(dev)
            write(*,*) 
            if (dev == '/xwin' ) then
               write(*,*) 'please check that X Window System is running'
               write(*,*) 'and that you have set the environment variables'
               write(*,*) 'DISPLAY :0'
               write(*,*) 'PGPLOT_DEV /xwin'
               write(*,*) 'and also that you have set PGPLOT_DIR and PGPLOT_FONT'
            else
               write(*,*) 'please check the device specification'
            end if
            write(*,*) 
            write(*,*) 
            ierr = -1
            return
         end do retry_loop
         !write(*,*) 'open device <' // trim(dev) // '> ' // trim(p% name), id
         if (is_file) then
            width = p% file_width; if (width < 0) width = p% win_width
            ratio = p% file_aspect_ratio; if (ratio < 0) ratio = p% win_aspect_ratio
            call pgpap(width, ratio)
         else
            call pgpap(p% win_width, p% win_aspect_ratio)
            p% prev_win_width = p% win_width
            p% prev_win_aspect_ratio = p% win_aspect_ratio
         end if
         call Set_Colours(white_on_black_flag, ierr)      
      end subroutine open_device

      
      subroutine read_support_info(ierr)
         integer, intent(out) :: ierr
         
         ierr = 0
         
         call read_TRho_data( &
            'hydrogen_burn.data', hydrogen_burn_logT, hydrogen_burn_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading hydrogen burn data'
            return
         end if   
               
         call read_TRho_data( &
            'helium_burn.data', helium_burn_logT, helium_burn_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading helium burn data'
            return
         end if   
               
         call read_TRho_data( &
            'carbon_burn.data', carbon_burn_logT, carbon_burn_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading carbon burn data'
            return
         end if  
                
         call read_TRho_data( &
            'oxygen_burn.data', oxygen_burn_logT, oxygen_burn_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading oxygen burn data'
            return
         end if  
                
         call read_TRho_data( &
            'psi4.data', psi4_logT, psi4_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading psi4 data'
            return
         end if         
                
         call read_TRho_data( &
            'elect.data', elect_data_logT, elect_data_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading elect data'
            return
         end if         
                
         call read_TRho_data( &
            'gamma_4_thirds.data', gamma_4_thirds_logT, gamma_4_thirds_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading gamma_4_thirds data'
            return
         end if         
                
         call read_TRho_data( &
            'kap_rad_cond_eq.data', kap_rad_cond_eq_logT, kap_rad_cond_eq_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading kap_rad_cond_eq data'
            return
         end if         
                
         call read_TRho_data( &
            'opal_clip.data', opal_clip_logT, opal_clip_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading opal_clip data'
            return
         end if         
                
         call read_TRho_data( &
            'scvh_clip.data', scvh_clip_logT, scvh_clip_logRho, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading scvh_clip data'
            return
         end if         
      
      end subroutine read_support_info
      
      

      ! remaining routines for setting colours in PGPLOT
      ! by X11 name
      subroutine Set_Colours(white_on_black_flag, ierr)
         implicit none
         logical, intent(in) :: white_on_black_flag
         integer, intent(out) :: ierr
         integer :: index

         index = 0
         ierr = 0
         if (white_on_black_flag) then
            index = setcolour(index, "black", ierr)
            if (ierr /= 0) return
            index = setcolour(index, "white", ierr)
            if (ierr /= 0) return
         else
            index = setcolour(index, "white", ierr)
            if (ierr /= 0) return
            index = setcolour(index, "black", ierr)
            if (ierr /= 0) return
         end if
         index = setcolour(index, "red", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "green", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "blue", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "cyan", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "magenta", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "yellow", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "orange", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "lime green", ierr)
         if (ierr /= 0) return 
         index = setcolour(index, "green yellow", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "dodger blue", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "magenta4", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "plum", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "sandy brown", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "salmon", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "grey59", ierr)
         if (ierr /= 0) return
         index = setcolour(index, "grey30", ierr)
         if (ierr /= 0) return
         
         ! Tioga colors
         index = set_c(index, clr_Black, 0.0, 0.0, 0.0)
         index = set_c(index, clr_Blue, 0.0, 0.0, 1.0)
         index = set_c(index, clr_BrightBlue, 0.0, 0.4, 1.0)
         index = set_c(index, clr_LightSkyBlue, 0.53, 0.808, 0.98)
         index = set_c(index, clr_LightSkyGreen, 0.125, 0.698, 0.668)
         index = set_c(index, clr_MediumSpringGreen, 0.0, 0.98, 0.604)
         index = set_c(index, clr_Goldenrod, 0.855, 0.648, 0.125)
         index = set_c(index, clr_Lilac, 0.8, 0.6, 1.0)
         index = set_c(index, clr_Coral, 1.0, 0.498, 0.312)
         index = set_c(index, clr_FireBrick, 0.698, 0.132, 0.132)
         index = set_c(index, clr_RoyalPurple, 0.4, 0.0, 0.6)
         index = set_c(index, clr_Gold, 1.0, 0.844, 0.0)
         index = set_c(index, clr_Crimson, 0.8, 0.0, 0.2 )
         index = set_c(index, clr_SlateGray, 0.44, 0.5, 0.565)
         index = set_c(index, clr_SeaGreen, 0.18, 0.545, 0.34)
         index = set_c(index, clr_Teal, 0.0, 0.5, 0.5)
         index = set_c(index, clr_LightSteelBlue, 0.69, 0.77, 0.87)
         index = set_c(index, clr_MediumSlateBlue, 0.484, 0.408, 0.932)
         index = set_c(index, clr_MediumBlue, 0.0, 0.0, 0.804)
         index = set_c(index, clr_RoyalBlue, 0.255, 0.41, 0.884)
         index = set_c(index, clr_LightGray, 0.828, 0.828, 0.828)
         index = set_c(index, clr_Silver, 0.752, 0.752, 0.752)
         index = set_c(index, clr_DarkGray, 0.664, 0.664, 0.664)
         index = set_c(index, clr_Gray, 0.5, 0.5, 0.5)
         index = set_c(index, clr_IndianRed, 0.804, 0.36, 0.36)
         index = set_c(index, clr_Tan, 0.824, 0.705, 0.55)

         index = set_c(index, clr_LightOliveGreen, 0.6, 0.8, 0.6)
         index = set_c(index, clr_CadetBlue, 0.372, 0.62, 0.628)
         index = set_c(index, clr_Beige, 0.96, 0.96, 0.864)
      
         clr_no_mixing = clr_SeaGreen
         clr_convection = clr_LightSkyBlue
         clr_semiconvection = clr_SlateGray
         clr_thermo_haline = clr_Lilac
         clr_overshoot = clr_Beige


         contains
         
         integer function set_c(index, clr_i, r, g, b)
            integer :: index, clr_i
            real :: r, g, b
            call pgscr(index, r, g, b)
            clr_i = index
            set_c = index + 1
         end function set_c
         

         integer function setcolour(i, name, ierr)
            implicit none

            integer :: i
            character (len=*) :: name
            integer, intent(out) :: ierr

            call Set_Pgplot_Colour(i, name, ierr)
            setcolour = i + 1
         end function setcolour

      end subroutine Set_Colours


      subroutine Set_Pgplot_Colour(index, name, ierr)
         use utils_lib
         implicit none

         integer :: index
         character (len=*) :: name
         integer, intent(out) :: ierr

         logical, save :: have_colour_list = .false.
         real, allocatable, dimension(:), save :: red, green, blue
         character (len=64), allocatable, dimension(:), save :: colournames
         integer, save :: nrgbcolours, low, hi
         integer :: i

         ierr = 0
         if (.not.have_colour_list) then
            call loadRGBtxt(ierr)
            if (ierr /= 0) return
            have_colour_list = .true.
            call pgqcol(low, hi)
         end if

         if ( .not. (low<=index .and. index<=hi) ) then
            write(*,'(a,i4,a,i3,a,i3)') "Set_Pgplot_Colour: requested index of ", index, &
               " not in ", low, " to ", hi
            return
         endif

         do i=1, nrgbcolours
            if (colournames(i) == name) exit
         end do
         if (i>nrgbcolours) then
            write(*,*) "Set_Pgplot_Colour: colour ", trim(name), " not found"
            ierr = -1
            return
         end if
         call pgscr(index, red(i), green(i), blue(i))


         contains
         

         subroutine loadRGBtxt(ierr)
            integer, intent(out) :: ierr
            
            logical :: fexist
            integer :: iounit, i, j, r, g, b, len
            character (len=1024) :: msg, fname, pgplotdir, line
            ierr = 0
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) then
               write(*,*) "no iounits in loadRGBtxt"
               return
            end if

            call getenv("PGPLOT_DIR", pgplotdir)
            if (len_trim(pgplotdir)==0) then
               write(*,*) "PGPLOT_DIR is not set in your shell"
               ierr = -1
               call free_iounit(iounit)
               return
            end if

            fname = trim(pgplotdir) // "/rgb.txt"
            inquire(file=trim(fname), exist=fexist)
            if (.not.fexist) then
               write(*,*) 'loadRGBtxt: pgplot ', trim(fname), " does not exist"
               write(*,*) 'loadRGBtxt: pgplot rgb.txt file does not exist'
               ierr = -1
               call free_iounit(iounit)
               return
            end if

            open(unit=iounit, file=trim(fname), status="old", iostat=ierr, iomsg=msg)

            if (ierr/=0) then
               write(*,*) trim(msg)
               write(*,*) 'loadRGBtxt: cannot open the pgplot rgb.txt file'
               ierr = -1
               call free_iounit(iounit)
               return
            end if

            ! count colours
            len = 0
            do while(.true.)
               read(iounit, *, iostat=ierr) i
               if (ierr/=0) exit
               len = len + 1
            end do
            nrgbcolours = len
            close(unit=iounit)

            allocate(red(nrgbcolours), green(nrgbcolours), blue(nrgbcolours))
            allocate(colournames(nrgbcolours))
            open(unit=iounit, file=trim(fname), status="old", iostat=ierr, iomsg=msg)
            do i = 1, nrgbcolours
               read(iounit,'(a)') line
               read(line,*) r, g, b
               j = 1
               do while(line(j:j)==char(32) .or. &
                  (line(j:j)>=char(48) .and. line(j:j)<=char(57)))
                  j = j + 1
               end do
               read(line(j:),'(a)') colournames(i)
               red(i) = r/255.0
               green(i) = g/255.0
               blue(i) = b/255.0
            end do
            close(iounit)
            call free_iounit(iounit)
            

         end subroutine loadRGBtxt


      end subroutine Set_Pgplot_Colour

      
      subroutine read_TRho_data(fname, logTs, logRhos, ierr)
         use utils_lib
         character (len=*), intent(in) :: fname
         real, dimension(:), pointer :: logTs, logRhos ! will allocate
         integer, intent(out) :: ierr
         
         character (len=256) :: filename
         real :: logT, logRho
         integer :: iounit, i, sz, cnt
         
         filename = trim(data_dir_for_mesa) // '/star_data/plot_info/' // trim(fname)
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            write(*,*) "no iounits in conv_plot"
            return
         end if

         open(unit=iounit, file=trim(filename), status='old', action='read', iostat=ierr)
         if (ierr/=0) then
            write(*,*) 'failed to open ' // trim(filename)
            call done
            return
         end if
         
         sz = 100
         allocate(logTs(sz), logRhos(sz))
         
         cnt = 0
         do i = 1, 10000
            read(iounit,*,iostat=ierr) logRho, logT
            if (ierr /= 0) then
               ierr = 0; exit
            end if
            if (i > sz) then
               sz = 3*sz/2 + 10
               call realloc(ierr)
               if (ierr /= 0) then
                  call done
                  return
               end if
            end if
            logRhos(i) = logRho
            logTs(i) = logT
            cnt = i
         end do
         
         if (cnt < sz) then
            sz = cnt
            call realloc(ierr)
            if (ierr /= 0) then
               call done
               return
            end if
         end if
         
         call done
         
         
         contains
         
         
         subroutine done
            close(iounit)
            call free_iounit(iounit)
         end subroutine done
         
         
         subroutine realloc(ierr)
            integer, intent(out) :: ierr
            ierr = 0
            call realloc_real(logTs, sz, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in realloc_double'
               return
            end if
            call realloc_real(logRhos, sz, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in realloc_double'
               return
            end if
         end subroutine realloc
         
      
      end subroutine read_TRho_data


      integer function write_info_line_str(cnt, ypos, xpos0, dxpos, str)
         integer, intent(in) :: cnt
         real, intent(in) :: ypos, xpos0, dxpos
         character (len=*), intent(in) :: str
         real :: xpos
         xpos = cnt*dxpos + xpos0
         call pgptxt(xpos, ypos, 0.0, 0.5, trim(adjustl(str)))
         write_info_line_str = cnt+1
      end function write_info_line_str
      
      
      integer function write_info_line_int(cnt, ypos, xpos0, dxpos, dxval, label, val)
         integer, intent(in) :: cnt, val
         real, intent(in) :: ypos, xpos0, dxpos, dxval
         character (len=*), intent(in) :: label

         character (len=128) :: str
         real :: xpos

         write(str,'(a)') trim(label)
         xpos = cnt*dxpos + xpos0
         call pgptxt(xpos, ypos, 0.0, 1.0, trim(adjustl(str)))
         write(str,'(i9)') val
         xpos = xpos +  dxval
         call pgptxt(xpos, ypos, 0.0, 0.0, trim(adjustl(str)))

         write_info_line_int = cnt+1
      end function write_info_line_int
      

      integer function write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, label, val)
         integer, intent(in) :: cnt
         real, intent(in) :: ypos, xpos0, dxpos, dxval
         real*8, intent(in) :: val
         character (len=*), intent(in) :: label

         character (len=128) :: str
         real :: xpos
         integer :: ierr

         write_info_line_flt = cnt+1
         write(str,'(a)')   trim(label)
         xpos = cnt*dxpos + xpos0
         call pgptxt(xpos, ypos, 0.0, 1.0, trim(adjustl(str)))
         ierr = 0
         write(str,'(f12.7)',iostat=ierr) val
         if (ierr /= 0) then
            ierr = 0
            write(str,'(e10.3)',iostat=ierr) val
            if (ierr /= 0) then
               write(*,*) trim(label), val
               write(*,*) 'problem in write_info_line_flt'
               return
            end if
         end if
         xpos = xpos + dxval
         call pgptxt(xpos, ypos, 0.0, 0.0, trim(adjustl(str)))

      end function write_info_line_flt
      

      integer function write_info_line_flt2(cnt, ypos, xpos0, dxpos, dxval, label, val)
         integer, intent(in) :: cnt
         real, intent(in) :: ypos, xpos0, dxpos, dxval
         real*8, intent(in) :: val
         character (len=*), intent(in) :: label

         character (len=128) :: str
         real :: xpos
         integer :: ierr

         write_info_line_flt2 = cnt+1

         write(str,'(a)')   trim(label)
         xpos = cnt*dxpos + xpos0
         call pgptxt(xpos, ypos, 0.0, 1.0, trim(adjustl(str)))
         ierr = 0
         write(str,'(f12.3)',iostat=ierr) val
         if (ierr /= 0) then
            ierr = 0
            write(str,'(e10.3)',iostat=ierr) val
            if (ierr /= 0) then
               write(*,*) trim(label), val
               write(*,*) 'problem in write_info_line_flt2'
               return
            end if
         end if
         xpos = xpos + dxval
         call pgptxt(xpos, ypos, 0.0, 0.0, trim(adjustl(str)))

      end function write_info_line_flt2
      

      integer function write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, label, val)
         integer, intent(in) :: cnt
         real, intent(in) :: ypos, xpos0, dxpos, dxval
         real*8, intent(in) :: val
         character (len=*), intent(in) :: label

         character (len=128) :: str
         real :: xpos

         write(str,'(a)')   trim(label)
         xpos = cnt*dxpos + xpos0
         call pgptxt(xpos, ypos, 0.0, 1.0, trim(adjustl(str)))
         write(str,'(1pe10.3)') val
         xpos = xpos + dxval
         call pgptxt(xpos, ypos, 0.0, 0.0, trim(adjustl(str)))

         write_info_line_exp = cnt+1
      end function write_info_line_exp
         
         
      subroutine set_grid_min_max(s, xmin, xmax, grid_min, grid_max, numpts, ierr)
         type (star_info), pointer :: s
         real, intent(in) :: xmin, xmax
         integer, intent(out) :: grid_min, grid_max, numpts, ierr
         integer :: k, id
         type (pgstar_data), pointer :: cv_data
         ierr = 0
         id = s% id
         grid_min = 1
         do k=2,cv_current(id)
            cv_data => cv_for_star(id)% cv_data(k)
            if (cv_data% step >= xmin) then
               if (cv_data% step > xmin) then
                  grid_min = k-1
               else
                  grid_min = k
               end if
               exit
            end if
         end do
         grid_max = cv_current(id)
         do k=cv_current(id)-1, grid_min, -1
            cv_data => cv_for_star(id)% cv_data(k)
            if (cv_data% step <= xmax) then
               if (cv_data% step < xmax) then
                  grid_max = k+1
               else
                  grid_max = k
               end if
               exit
            end if
         end do
         numpts = grid_max - grid_min + 1
         if (numpts < 0) then
            write(*,*) 'bug in set_grid_min_max'
            ierr = -1
            return
         end if
      end subroutine set_grid_min_max
      
      
      subroutine do_cross_hair
         integer :: cursor_pos, cursor_mode, cursor_posn
         real :: cursor_xref, cursor_yref, cursor_x, cursor_y
         character (len=1) :: cursor_ch         
         integer :: pgband
         do
            cursor_mode = 7
            cursor_posn = 0
            write(*,*) 'position cursor; hit space to resume plotting; hit non-space to repeat.'
            cursor_pos = pgband( &
               cursor_mode, cursor_posn, cursor_xref, cursor_yref, cursor_x, cursor_y, cursor_ch)
            write(*,'(a20,2e16.6)') 'cursor coords', cursor_x, cursor_y
            if (len_trim(cursor_ch) == 0) exit
         end do
      end subroutine do_cross_hair



      subroutine set_grid_minmax( &
            nz, xvec, xmin, xmax, xleft, xright, &
            default_xmin, default_xmax, margin, reversed, grid_min, grid_max, dxmin)
         integer, intent(in) :: nz
         real, intent(in), dimension(:) :: xvec
         real, intent(out) :: xmin, xmax, xleft, xright
         real, intent(in) :: default_xmin, default_xmax, margin
         real, intent(in) :: dxmin
         logical, intent(in) :: reversed
         integer, intent(out) :: grid_min, grid_max
         integer :: k
         real :: dx
         
         if (default_xmin > -100) then
            xmin = default_xmin
         else
            xmin = minval(xvec(1:nz))
         end if
         if (default_xmax > -100) then
            xmax = default_xmax
         else
            xmax = maxval(xvec(1:nz))
         end if
         dx = xmax -xmin
         
         if (default_xmin <= -100) xmin = xmin - margin*dx
         if (default_xmax <= -100) xmax = xmax + margin*dx
         dx = xmax - xmin
         if (dx < dxmin) then
            dx = dxmin
            xmax = (xmax + xmin)/2 + dx/2
            xmin = xmax - dx
         end if
         
         if (xmin == xmax) then
            xmin = xmin - margin/2
            xmax = xmax + margin/2
         end if
      
         if (reversed) then
            xright = xmin; xleft = xmax
         else
            xright = xmax; xleft = xmin
         end if

         if (xvec(1) < xvec(nz)) then ! increasing xs
            grid_max = nz
            do k=nz-1,1,-1 ! in decreasing order
               if (xvec(k) <= xmax) then ! first one <= xmax
                  if (xvec(k) == xmax) then
                     grid_max = k
                  else
                     grid_max = k+1
                  end if
                  exit
               end if
            end do
            grid_min = 1
            do k=nz-1,1,-1
               if (xvec(k) <= xmin) then ! first one <= xmin
                  grid_min = k
                  exit
               end if
            end do
         else ! decreasing
            grid_min = 1
            do k=2,nz ! in decreasing order
               if (xvec(k) <= xmax) then ! first one <= xmax
                  if (xvec(k) == xmax) then
                     grid_min = k
                  else
                     grid_min = k-1
                  end if
                  exit
               end if
            end do
            grid_max = nz
            do k=1,nz
               if (xvec(k) <= xmin) then ! first one <= xmin
                  grid_max = k
                  exit
               end if
            end do
         end if

      end subroutine set_grid_minmax
      
      
      subroutine show_annotations(s, show_annotation1, show_annotation2, show_annotation3)
         type (star_info), pointer :: s
         logical, intent(in) :: show_annotation1, show_annotation2, show_annotation3
         if (show_annotation1 .and. len_trim(s% annotation1_text) > 0) then
            call pgsci(s% annotation1_ci)
            call pgsch(s% annotation1_ch)
            call pgslw(s% annotation1_lw)
            call pgscf(s% annotation1_cf)
            call pgmtxt(s% annotation1_side, s% annotation1_disp, &
                        s% annotation1_coord, s% annotation1_fjust, s% annotation1_text)
         end if
         if (show_annotation2 .and. len_trim(s% annotation2_text) > 0) then
            call pgsci(s% annotation2_ci)
            call pgsch(s% annotation2_ch)
            call pgslw(s% annotation2_lw)
            call pgscf(s% annotation2_cf)
            call pgmtxt(s% annotation2_side, s% annotation2_disp, &
                        s% annotation2_coord, s% annotation2_fjust, s% annotation2_text)
         end if
         if (show_annotation3 .and. len_trim(s% annotation3_text) > 0) then
            call pgsci(s% annotation3_ci)
            call pgsch(s% annotation3_ch)
            call pgslw(s% annotation3_lw)
            call pgscf(s% annotation3_cf)
            call pgmtxt(s% annotation3_side, s% annotation3_disp, &
                        s% annotation3_coord, s% annotation3_fjust, s% annotation3_text)      
         end if
      end subroutine show_annotations


      integer function get_axis_id(name)
         use chem_lib, only: get_nuclide_index, rates_category_id
         character (len=*), intent(in) :: name
         get_axis_id = do_get_profile_id(name)
         if (get_axis_id > 0) return
         get_axis_id = get_nuclide_index(name)
         if (get_axis_id > 0) then
            get_axis_id = get_axis_id + abundance_offset
            return
         end if
         get_axis_id = rates_category_id(name)
         if (get_axis_id > 0) then
            get_axis_id = get_axis_id + category_offset
            return
         end if
         if (name == 'profile_extra1') then
            get_axis_id = 1 + extras_offset
            return
         end if
         if (name == 'profile_extra2') then
            get_axis_id = 2 + extras_offset
            return
         end if
         if (name == 'profile_extra3') then
            get_axis_id = 3 + extras_offset
            return
         end if
         if (name == 'profile_extra4') then
            get_axis_id = 4 + extras_offset
            return
         end if
         if (name == 'profile_extra5') then
            get_axis_id = 5 + extras_offset
            return
         end if
         if (name == 'profile_extra6') then
            get_axis_id = 6 + extras_offset
            return
         end if
         if (name == 'profile_extra7') then
            get_axis_id = 7 + extras_offset
            return
         end if
         if (name == 'profile_extra8') then
            get_axis_id = 8 + extras_offset
            return
         end if
         if (name == 'profile_extra9') then
            get_axis_id = 9 + extras_offset
            return
         end if
      end function get_axis_id
      
      
      real*8 function get_val(s, id, k, in_name, out_name)
         use profile_getval, only: getval_for_profile
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         integer, intent(in) :: id, k
         character (len=*), intent(in) :: in_name
         character (len=*), intent(out) :: out_name
         integer :: ii, int_val
         logical :: int_flag
         if (id > extras_offset) then
            ii = id - extras_offset
            if (ii >= 1 .and. ii <= 9) then
               get_val = s% profile_extra(k,ii)
               out_name = trim(s% profile_extra_name(ii))
            else 
               get_val = 0
               out_name = 'UNKNOWN PROFILE EXTRA'
            end if
         else if (id > abundance_offset) then
            ii = s% net_iso(id - abundance_offset)
            if (ii == 0) then
               get_val = 0d0
            else
               get_val = safe_log10(s% xa(ii,k))
            end if
            out_name = 'log ' // trim(in_name)
         else if (id > category_offset) then
            ii = id - category_offset
            get_val = safe_log10(s% eps_nuc_categories(i_rate,ii,k))
            out_name = 'log ' // trim(in_name)
         else
            call getval_for_profile(s, id, k, get_val, int_flag, int_val)
            if (int_flag) get_val = dble(int_val)
            call remove_underbars(in_name, out_name)
         end if
      end function get_val
            
   
      subroutine remove_underbars(str, name)
         character (len=*), intent(in) :: str
         character (len=*), intent(out) :: name
         integer :: i, len
         len = len_trim(str)
         name = ''
         do i=1,len
            if (str(i:i) == '_') then
               name(i:i) = ' '
            else
               name(i:i) = str(i:i)
            end if
         end do
      end subroutine remove_underbars
      

      subroutine set_xaxis_bounds( &
            s, xaxis_by_in, win_xmin_in, win_xmax_in, xaxis_reversed_in, xmargin, &
            xvec, xmin, xmax, xleft, xright, dx, &
            grid_min, grid_max, npts, ierr)
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         character (len=*), intent(in) :: xaxis_by_in
         real, intent(in) :: win_xmin_in, win_xmax_in, xmargin
         logical, intent(in) :: xaxis_reversed_in
         real, pointer, dimension(:) :: xvec
         real, intent(out) :: xmin, xmax, xleft, xright, dx
         integer, intent(out) :: grid_min, grid_max, npts
         integer, intent(out) :: ierr
         
         integer :: k, nz, xaxis_id
         real :: win_xmin, win_xmax
         real*8 :: dmsum
         logical :: reversed
         character (len=128) :: xaxis_by, xname
      
         include 'formats.dek'
         
         ierr = 0
         win_xmin = win_xmin_in
         win_xmax = win_xmax_in
         nz = s% nz
         xaxis_by = trim(xaxis_by_in)
         
         reversed = .false.
         
         if (xaxis_by == 'by_mass') then
            xvec(1:nz) = s% m(1:nz)/Msun
         else if (xaxis_by == 'by_grid') then
            forall(k=1:nz) xvec(k) = k
            reversed = .true.
         else if (xaxis_by == 'by_radius') then
            xvec(1:nz) = s% r(1:nz)/Rsun
         else if (xaxis_by == 'by_logR') then
            xvec(1:nz) = s% lnR(1:nz)/ln10 - log10(Rsun)
         else if (xaxis_by == 'by_logP') then
            xvec(1:nz) = s% lnP(1:nz)/ln10
            reversed = .true.
         else if (xaxis_by == 'by_logT') then
            xvec(1:nz) = s% lnT(1:nz)/ln10
            reversed = .true.
         else if (xaxis_by == 'by_logxq') then
            xvec(1:nz) = max(s% logxq_cutoff, safe_log10(1d0 - s% q(1:nz)))
            reversed = .true.
         else if (xaxis_by == 'h1_boundary_mass') then
            call set_by_boundary_mass(s% h1_boundary_mass)
         else if (xaxis_by == 'he4_boundary_mass') then
            call set_by_boundary_mass(s% he4_boundary_mass)
         else if (xaxis_by == 'c12_boundary_mass') then
            call set_by_boundary_mass(s% c12_boundary_mass)
         else if (xaxis_by == 'by_r_div_R') then
            xvec(1:nz) = s% r(1:nz)/s% r(1)
         else if (xaxis_by == 'by_log_column_depth') then
            dmsum = 0
            xvec(1) = -99
            do k=2,nz
               dmsum = dmsum + s% dm(k)
               xvec(k) = safe_log10(dmsum/(4*pi*s% r(k)**2))
            end do
            reversed = .true.
         else
            xaxis_id = get_axis_id(xaxis_by)
            if (xaxis_id <= 0) then
               write(*,'(a)') &
                  'pgstar inlist problem: bad value for xaxis_by: <' // trim(xaxis_by) // '>'
               ierr = -1
               return
            end if
            do k=1,nz
               xvec(k) = get_val(s, xaxis_id, k, xaxis_by, xname)
            end do
            reversed = xaxis_reversed_in
         end if
         
         call set_grid_minmax( &
            nz, xvec, xmin, xmax, xleft, xright, &
            win_xmin, win_xmax, xmargin, reversed, grid_min, grid_max, 0.0)
         dx = xmax - xmin         
         npts = grid_max - grid_min + 1
         if (npts <= 0) then
            write(*,*) 'invalid x axis bounds for xaxis_by = ' // trim(xaxis_by)
            write(*,1) 'xmax', xmax
            write(*,1) 'xmin', xmin
            write(*,1) 'dx', dx
            write(*,1) 'xleft', xleft
            write(*,1) 'xright', xright
            write(*,1) 'win_xmin', win_xmin
            write(*,1) 'win_xmax', win_xmax
            write(*,1) 'xmargin', xmargin
            write(*,2) 'grid_min', grid_min
            write(*,2) 'grid_max', grid_max
            write(*,2) 'npts', npts
            write(*,2) 'nz', nz
            write(*,1) 'maxval(xvec(1:nz))', maxval(xvec(1:nz))
            write(*,1) 'minval(xvec(1:nz))', minval(xvec(1:nz))
            ierr = -1
         end if
         
         contains
         
         subroutine set_by_boundary_mass(bdy_mass)
            real*8, intent(in) :: bdy_mass
            xvec = s% m(1:nz)/Msun
            if (win_xmin > -100 .and. win_xmax > -100) then
               win_xmin = max(0.0, bdy_mass - win_xmin)
               win_xmax = bdy_mass + win_xmax
            else
               win_xmin = max(0.0, bdy_mass - 0.001)
               win_xmax = bdy_mass + 0.001
            end if
         end subroutine set_by_boundary_mass
         
      end subroutine set_xaxis_bounds
      
      
      subroutine show_xaxis_by(s,by,ierr)
         type (star_info), pointer :: s
         character (len=*), intent(in) :: by
         integer, intent(out) :: ierr
         ierr = 0
         if (by == 'by_mass') then
            call pgmtxt('B',2.4,0.5,0.5,'m/M\d\(2281)')
         else if (by == 'by_grid') then
            call pgmtxt('B',2.4,0.5,0.5,'grid')
         else if (by == 'by_radius') then
            call pgmtxt('B',2.4,0.5,0.5,'r/R\d\(2281)')
         else if (by == 'by_logR') then
            call pgmtxt('B',2.4,0.5,0.5,'log r/R\d\(2281)')
         else if (by == 'by_logT') then
            call pgmtxt('B',2.4,0.5,0.5,'log T')
         else if (by == 'by_logP') then
            call pgmtxt('B',2.4,0.5,0.5,'log P')
         else if (by == 'by_logxq') then
            if (s% M_center == 0) then
               call pgmtxt('B',2.4,0.5,0.5,'log(1-q) q=fraction of total mass')
            else
               call pgmtxt('B',2.4,0.5,0.5,'log(1-q) q=fraction of envelope mass')
            end if
         else if (by == 'h1_boundary_mass') then
            call pgmtxt('B',2.4,0.5,0.5,'m/M\d\(2281)')
         else if (by == 'he4_boundary_mass') then
            call pgmtxt('B',2.4,0.5,0.5,'m/M\d\(2281)')
         else if (by == 'c12_boundary_mass') then
            call pgmtxt('B',2.4,0.5,0.5,'m/M\d\(2281)')
         else if (by == 'by_r_div_R') then
            call pgmtxt('B',2.4,0.5,0.5,'r/R')
         else if (by == 'by_log_column_depth') then
            call pgmtxt('B',2.4,0.5,0.5,'log column depth (g cm\u-2\d)')
         else
            write(*,*) 'bad value for xaxis_by', by
            ierr = -1
         end if
      end subroutine show_xaxis_by
      
      
      subroutine show_mix_regions_on_xaxis(s,ybot_in,grid_min,grid_max,xvec)
         type (star_info), pointer :: s
         real, intent(in) :: ybot_in
         integer, intent(in) :: grid_min,grid_max
         real, pointer, dimension(:) :: xvec
         real :: eps, ybot
         call pgsci(clr_no_mixing)
         eps = 0.001
         ybot = ybot_in + eps
         call pgmove(xvec(grid_min), ybot)
         call pgdraw(xvec(grid_max), ybot)
         call show_convective_section(s,ybot,xvec)
         call show_semiconvective_section(s,ybot,xvec)
         call show_thermo_haline_section(s,ybot,xvec)
         call show_overshoot_section(s,ybot,xvec)
      end subroutine show_mix_regions_on_xaxis

   
      subroutine show_convective_section(s,ybot,xvec)
         use mlt_def, only: convective_mixing
         type (star_info), pointer :: s
         real, intent(in) :: ybot
         real, pointer, dimension(:) :: xvec
         call show_mixing_section(s,ybot,xvec,convective_mixing,clr_convection)
      end subroutine show_convective_section
   
   
      subroutine show_semiconvective_section(s,ybot,xvec)
         use mlt_def, only: semiconvective_mixing
         type (star_info), pointer :: s
         real, intent(in) :: ybot
         real, pointer, dimension(:) :: xvec
         call show_mixing_section(s,ybot,xvec,semiconvective_mixing,clr_semiconvection)
      end subroutine show_semiconvective_section
   
   
      subroutine show_thermo_haline_section(s,ybot,xvec)
         use mlt_def, only: thermo_haline_mixing
         type (star_info), pointer :: s
         real, intent(in) :: ybot
         real, pointer, dimension(:) :: xvec
         call show_mixing_section(s,ybot,xvec,thermo_haline_mixing,clr_thermo_haline)
      end subroutine show_thermo_haline_section
   
   
      subroutine show_overshoot_section(s,ybot,xvec)
         use mlt_def, only: overshoot_mixing
         type (star_info), pointer :: s
         real, intent(in) :: ybot
         real, pointer, dimension(:) :: xvec
         call show_mixing_section(s,ybot,xvec,overshoot_mixing,clr_overshoot)
      end subroutine show_overshoot_section
   
   
      subroutine show_mixing_section(s,ybot,xvec,mixing_type,clr)
         use mlt_def, only: semiconvective_mixing
         type (star_info), pointer :: s
         real, intent(in) :: ybot
         real, pointer, dimension(:) :: xvec
         integer, intent(in) :: mixing_type,clr
         
         integer :: k, first, last
         logical :: inside
         inside = (s% mixing_type(1) == mixing_type)
         if (inside) first = 1
         call pgsci(clr)
         do k=2,s% nz
            if (.not. inside) then
               if (s% mixing_type(k) == mixing_type) then ! starting
                  inside = .true.
                  first = k
               end if
            else ! inside
               if (s% mixing_type(k) /= mixing_type) then ! ending
                  last = k-1
                  call pgmove(xvec(first), ybot)
                  call pgdraw(xvec(last), ybot)
                  inside = .false.
               end if
            end if
         end do
         if (inside) then
            last = s% nz
            call pgmove(xvec(first), ybot)
            call pgdraw(xvec(last), ybot)
         end if
      end subroutine show_mixing_section

      
      subroutine show_profile_line( &
            s, xvec, yvec, txt_scale, xmin, xmax, ymin, ymax, &
            show_legend, legend_coord, legend_disp1, legend_del_disp, legend_fjust, &
            show_mass_pts)
         type (star_info), pointer :: s
         real, intent(in) :: xvec(:), yvec(:), txt_scale, xmin, xmax, ymin, ymax, &
            legend_coord, legend_disp1, legend_del_disp, legend_fjust
         logical, intent(in) :: show_legend, show_mass_pts
         
         real :: disp
         integer :: nz
         logical :: has_convection, has_overshoot, has_semiconvection, has_thermo_haline
         
         include 'formats.dek'
         
         call pgsave

         nz = s% nz         
         call pgsch(1.2*txt_scale)
            
         call pgsci(clr_Gold)
         call pgslw(14)
         call do_show_eps_nuc_section(1d0)
         call pgslw(1)
         disp = legend_disp1 + 2*legend_del_disp
         if (show_legend) &
            call pgmtxt('T',disp,legend_coord,legend_fjust,'> 1 erg g\u-1\d s\u-1\d')
   
         call pgsci(clr_Coral)
         call pgslw(18)
         call do_show_eps_nuc_section(1d3)
         call pgslw(1)
         disp = legend_disp1 + legend_del_disp
         if (show_legend) &
            call pgmtxt('T',disp,legend_coord,legend_fjust,'> 1000 erg g\u-1\d s\u-1\d')
      
         call pgsci(clr_Crimson)
         call pgslw(20)
         call do_show_eps_nuc_section(1d7)
         call pgslw(1)
         disp = legend_disp1
         if (show_legend) &
            call pgmtxt('T',disp,legend_coord,legend_fjust,'> 10\u7\d erg g\u-1\d s\u-1\d')
         
         disp = legend_disp1 + 2*legend_del_disp
         
         call pgsci(clr_no_mixing)
         call pgslw(10)
         call pgline(nz, xvec, yvec)
         has_convection = do_show_convective_section()
         has_overshoot = do_show_overshoot_section()
         has_semiconvection = do_show_semiconvective_section()
         has_thermo_haline = do_show_thermo_haline_section()
         call pgslw(1)
         if (show_legend) then
            call pgslw(1)
            disp = disp + legend_del_disp
            call show_legend_text(clr_no_mixing, 'no mixing')
            disp = disp + legend_del_disp
            call show_legend_text(clr_convection, 'convection')
            disp = disp + legend_del_disp
            call show_legend_text(clr_overshoot, 'overshoot')
            disp = disp + legend_del_disp
            call show_legend_text(clr_semiconvection, 'semiconvection')
            disp = disp + legend_del_disp
            call show_legend_text(clr_thermo_haline, 'thermo haline')
            call pgslw(10)
         end if
         
         if (show_mass_pts) &
            call show_mass_points(s, xvec, yvec, txt_scale, xmin, xmax, ymin, ymax)    

         call pgunsa
         
         
         contains
         
         subroutine show_legend_text(clr, txt)
            integer, intent(in) :: clr
            character (len=*), intent(in) :: txt
            call pgsci(clr)
            call pgmtxt('T',disp,legend_coord,legend_fjust,txt)
         end subroutine show_legend_text
         
         
         subroutine do_show_eps_nuc_section(eps)
            real*8, intent(in) :: eps
            integer :: k, first, last
            logical :: inside
            inside = (s% eps_nuc(1) > eps)
            if (inside) first = 1
            do k=2,s% nz
               if (.not. inside) then
                  if (s% eps_nuc(k) > eps) then ! starting
                     inside = .true.
                     first = k
                  end if
               else ! inside
                  if (s% eps_nuc(k) <= eps) then ! ending
                     last = k-1
                     call pgline(k-first, xvec(first:last), yvec(first:last))
                     inside = .false.
                  end if
               end if
            end do
            if (inside) then
               last = nz
               call pgline(k-first, xvec(first:last), yvec(first:last))
            end if
         end subroutine do_show_eps_nuc_section
         
         
         logical function do_show_convective_section()
            use mlt_def, only: convective_mixing
            do_show_convective_section = do_show_mixing_section(convective_mixing, clr_convection)
         end function do_show_convective_section

         
         logical function do_show_semiconvective_section()
            use mlt_def, only: semiconvective_mixing
            do_show_semiconvective_section = do_show_mixing_section(semiconvective_mixing, clr_semiconvection)
         end function do_show_semiconvective_section

         
         logical function do_show_thermo_haline_section()
            use mlt_def, only: thermo_haline_mixing
            do_show_thermo_haline_section = do_show_mixing_section(thermo_haline_mixing, clr_thermo_haline)
         end function do_show_thermo_haline_section

         
         logical function do_show_overshoot_section()
            use mlt_def, only: overshoot_mixing
            do_show_overshoot_section = do_show_mixing_section(overshoot_mixing, clr_overshoot)
         end function do_show_overshoot_section
         
         
         logical function do_show_mixing_section(mixing_type, clr)
            integer, intent(in) :: mixing_type, clr
            integer :: k, first, last
            logical :: inside
            include 'formats.dek'
            call pgsave
            call pgsci(clr)
            inside = (s% mixing_type(1) == mixing_type)
            if (inside) first = 1
            do_show_mixing_section = .false.
            do k=2,s% nz
               if (.not. inside) then
                  if (s% mixing_type(k) == mixing_type) then ! starting
                     inside = .true.
                     first = k
                  end if
               else ! inside
                  if (s% mixing_type(k) /= mixing_type) then ! ending
                     last = k-1
                     call pgline(k-first, xvec(first:last), yvec(first:last))
                     do_show_mixing_section = .true.
                     inside = .false.
                  end if
               end if
            end do
            if (inside) then
               last = nz
               call pgline(k-first, xvec(first:last), yvec(first:last))
               do_show_mixing_section = .true.
            end if
            call pgunsa
         end function do_show_mixing_section
         
         
      end subroutine show_profile_line
         
         
      subroutine show_mass_points(s, xvec, yvec, txt_scale, xmin, xmax, ymin, ymax)
         type (star_info), pointer :: s
         real, intent(in) :: xvec(:), yvec(:), txt_scale, xmin, xmax, ymin, ymax
         integer :: i
         do i = 1, s% num_profile_mass_points
            call show_mass_point( &
               s, xvec, yvec, txt_scale, xmin, xmax, ymin, ymax, &
               s% profile_mass_point_q(i), &
               s% profile_mass_point_color_index(i), &
               s% profile_mass_point_symbol(i), &
               s% profile_mass_point_symbol_scale(i), &
               s% profile_mass_point_str(i), &
               s% profile_mass_point_str_clr(i), &
               s% profile_mass_point_str_scale(i))
         end do
      end subroutine show_mass_points
      
      
      subroutine show_mass_point( &
            s, xvec, yvec, txt_scale, xmin, xmax, ymin, ymax, &
            q_in, clr_index, symbol, symbol_scale, str, str_clr, str_scale)
         type (star_info), pointer :: s
         real, intent(in) :: xvec(:), yvec(:), txt_scale, q_in, &
            xmin, xmax, ymin, ymax, symbol_scale, str_scale
         integer, intent(in) :: clr_index, symbol, str_clr
         character (len=*), intent(in) :: str
         real :: q, q0, q1, x, y, dy
         integer :: nz, i, j, k
         include 'formats.dek'
         q = max(0.0,min(1.0,q_in))
         nz = s% nz
         i = nz
         dy = ymax-ymin
         do k=1,s% nz - 1
            if (s% q(k) >= q .and. q > s% q(k+1)) then
               i = k; exit
            end if
         end do
         j = i+1
         if (j >= nz) j = i 
         q0 = s% q(i)
         q1 = s% q(j)
         if ((q0-q)*(q-q1) < 0) then
            j = i-1
            q1 = s% q(j)
         end if
         x = find0(xvec(i), q0-q, xvec(j), q1-q)
         if (x > xmax .or. x < xmin) return
         y = find0(yvec(i), q0-q, yvec(j), q1-q)
         if (y > ymax .or. y < ymin) return
         call pgsave
         call pgsci(clr_index)
         call pgsch(symbol_scale*txt_scale)
         call pgscf(1)
         call pgpt(1, x, y, symbol)
         call pgsch(str_scale*txt_scale)
         call pgsci(str_clr)
         call pgptxt(x, y - 0.015*dy, 0.0, 0.0, trim(str))
         call pgunsa
      end subroutine show_mass_point


      real function find0(xx1,yy1,xx2,yy2)
         real :: xx1,yy1,xx2,yy2
         real :: a, b, xz
         ! returns x where y is 0 on line connecting the points (xx1,yy1) and (xx2,yy2)
         a = (xx1*yy2)-(xx2*yy1)
         b = yy2-yy1
         if ((abs(a) >= abs(b)*1e30) .and. ((yy1 >= 0 .and. yy2 <= 0) &
                  .or. (yy1 <= 0 .and. yy2 > 0))) then
            xz = 0.5*(xx1+xx2)
         else
            xz = a/b
         end if
         find0 = xz
      end function find0
         


      end module mod_pgstar_support

