! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_summary

      use star_private_def
      use const_def
      use mod_pgstar_support

      implicit none


      contains
      
      
      subroutine Summary_Plot(id, device_id, ierr)
         use mod_pgstar_hr, only: draw_HR_Plot
         use mod_pgstar_logg_Teff, only: draw_loggTe_Plot
         use mod_pgstar_trho, only: draw_TRho_Plot
         use mod_pgstar_conv, only: draw_Convection_Plot
         use mod_pgstar_trho_profile, only: draw_TRho_Profile_plot
         use utils_lib
         use chem_def
         use num_lib, only: safe_log10
         use net_def
         use net_lib, only: get_net_reaction_table
         use const_def, only: Msun, Rsun
         implicit none

         integer, intent(in) :: id, device_id
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s
         character (len=256) :: str
         integer :: nz, k, sp, cnt, npts, i, j, grid_min, grid_max, nplots
         logical :: show
         real, pointer, dimension(:) :: xvec, yvec, yvec2, yvec3
         real :: x, xmin, xmax, dx, ymin, ymax, dy, fcol, frow, chScale
         real :: log10Rsun, xleft, xright, xmargin, tmp
         real :: txt_scale, vpxmin, vpxmax, vpymin, vpymax, vpymargin, vpwinheight
         real :: win_gap
         real :: dlnT_dlnP, winxmin, winxmax, winymin, winymax, windy, label_scale
         real :: header_frac, plot1_frac, plot2_frac, plot3_frac, footer_frac, plots_frac
         real*8 :: eps_max, eps, maxv
         real :: xpos_nuc(num_categories)
         
         integer, parameter :: num_colors = 20
         integer :: colors(num_colors)
         integer, parameter :: max_num_labels = 20
         integer :: ii, num_labels, iloc_abundance_label(max_num_labels)
         real :: xloc_abundance_label(max_num_labels)
         integer :: docat(num_categories), eps_k(num_categories), num_cat
         
         include 'formats.dek'
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         colors(:) = (/ &
               clr_MediumSlateBlue, clr_LightSkyBlue, clr_Goldenrod, clr_Lilac, &
               clr_Coral, clr_Crimson, clr_LightSkyGreen, clr_DarkGray, &
               clr_Tan, clr_IndianRed, clr_Gold, &
               clr_Teal, clr_Silver, clr_BrightBlue, clr_FireBrick, &
               clr_RoyalPurple, clr_SlateGray, clr_LightSteelBlue, &
               clr_Gray, clr_RoyalBlue /)

         log10Rsun = log10(Rsun)
         
         call pgslct(device_id)
         call pgbbuf()
         call pgeras()
         
         winymin = 0.04
         winymax = 0.94

         if (s% show_TRho_Profile_with_main) then
            txt_scale = 0.5
            if (s% show_HR_TRho_with_main) then
               vpxmin = 0.1
               vpxmax = 0.88
            else
               vpxmin = 0.2
               vpxmax = 0.85
            end if
            vpymin = 0.62
            vpymax = winymax
            call draw_TRho_Profile_plot(s, vpxmin, vpxmax, vpymin, vpymax, txt_scale, .false., ierr)
            if (ierr /= 0) return
            winymax = 0.54
         end if
         
         if (s% show_HR_TRho_with_main .or. s% show_loggTe_TRho_with_main) then
            nplots = 2
            txt_scale = 0.5
            vpxmin = 0.10
            vpxmax = 0.29
            vpymin = 0.10
            vpymax = winymax - 0.04
            vpymargin = 0.14
            vpwinheight = (vpymax - vpymin - (nplots-1)*vpymargin)/nplots
            if (s% show_HR_TRho_with_main) then
               call draw_HR_Plot(s,vpxmin,vpxmax,vpymax-vpwinheight,vpymax,txt_scale, ierr)
            else if (s% show_loggTe_TRho_with_main) then
               call draw_loggTe_Plot(s,vpxmin,vpxmax,vpymax-vpwinheight,vpymax,txt_scale, ierr)
            else
               ierr = -1
            end if
            if (ierr /= 0) return
            call draw_TRho_Plot(s,vpxmin,vpxmax,vpymin,vpymin+vpwinheight,txt_scale, ierr)
            if (ierr /= 0) return
            winxmin = 0.43
            winxmax = 0.88
            winymax = winymax - 0.02
         else
            winxmin = 0.2
            winxmax = 0.85
         end if

         chScale = 1.0
         windy = winymax - winymin
         
         header_frac = 0.02
         if (s% show_main_win_text_info) then
            footer_frac = 0.38
         else
            footer_frac = 0.08
         end if
         plots_frac = 1 - (header_frac + footer_frac)
         plot1_frac = plots_frac*0.32
         plot3_frac = plot1_frac
         plot2_frac = plots_frac - (plot1_frac + plot3_frac)

         win_gap = 0.03
         xmargin = 0
         
         label_scale = chScale*0.8

         nz = s% nz        
         allocate (xvec(nz), yvec(nz), yvec2(nz), yvec3(nz))
         
         call set_xaxis_bounds( &
            s, s% xaxis_by, s% main_xmin, s% main_xmax, .false., xmargin, &
            xvec, xmin, xmax, xleft, xright, dx, &
            grid_min, grid_max, npts, ierr)
         
         if (ierr /= 0) then
            !stop 'set_xaxis_bounds error in pstar_summary'
            
            
            
            ! skip the plotting
            call set_plot1_vp         
            call pgsch(chScale*1.2)
            call pgmtxt('T',-5.0,0.5,0.5,'summary plot')
            call pgmtxt('T',-7.0,0.5,0.5,'Bad Xmin and/or Xmax specification')
            call pgmtxt('T',-9.0,0.5,0.5,'please check inlist')
         else
            call plot1(.false.)
            call plot2
            call plot3(ierr)
            if (ierr /= 0) return
            call header
            if (s% show_main_win_text_info) call footer(ierr)
            if (ierr /= 0) return
         end if         
         
         call show_annotations(s, &
            s% show_main_annotation1, s% show_main_annotation2, s% show_main_annotation3)

         call pgebuf()

         deallocate(xvec, yvec, yvec2, yvec3)
         

         contains
         
         
         
         
         subroutine plot1(testing)
            logical, intent(in) :: testing
            integer :: lw, lw_sav, ii, jj
            real :: xnuc_cat(num_categories), ylb
            logical, parameter :: dbg = .false.
         
            include 'formats.dek'
            call pgsave
                       
            lw = 6
            call pgqlw(lw_sav)
            
            call set_plot1_vp
         
            !write(str,'(i9)') s% model_number
            !call pgsch(label_scale)
            !if (.not. testing) call pgmtxt('T',5.0*1.2/0.9,0.9,0.5,str)
         
            ! logT
            yvec(1:nz) = s% lnT(1:nz)/ln10
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))
            dy = ymax-ymin
            ymax = ymax + 0.1*dy
            ymin = ymin - 0.1*dy
         
            call pgswin(xleft, xright, ymin, ymax)
            
            if (testing) then
               call pgmove(xmin, ymin)
               call pgdraw(xmax, ymax)
               call pgmove(xmin, ymax)
               call pgdraw(xmax, ymin)
               call pgpt(1, xmin+dx/2, ymin+dy/2, -6)
               call pgunsa
               return
            end if

            call pgscf(1)
            call pgsci(1)
            call pgsch(label_scale)
            call pgbox('BCST',0.0,0,'CMSTV',0.0,0)
            call pgsci(clr_Goldenrod)
            call pgmtxt('R',5.0,0.5,0.5,'log T')
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
         
            yvec(1:nz) = safe_log10(s% non_nuc_neu(1:nz) + s% eps_nuc_neu_total(1:nz))
            ymax = maxval(yvec(grid_min:grid_max))
            yvec2(1:nz) = s% lnd(1:nz)/ln10
            yvec3(1:nz) = safe_log10(s% eps_nuc(1:nz))

            ymax = max(ymax,maxval(yvec2(grid_min:grid_max)))
            ymax = max(ymax,maxval(yvec3(grid_min:grid_max)))
            if (ymax <= 10) then
               ymax = 11.1
            else 
               ymax = ymax*1.1
            end if
            ymin = -6.6         
            ymax = max(ymax, ymin+1)
            dy = ymax-ymin
            ymin = ymin-dy*0.1

            call pgswin(xleft, xright, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(label_scale)
            call pgbox('',0.0,0,'BNSTV',0.0,0)

            call pgsci(clr_Lilac)
            call pgmtxt('L',4.9,0.5,0.5,'log \gr')
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec2(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgsci(clr_LightSkyGreen)
            call pgsch(label_scale)
            call pgmtxt('L',3.7,0.5,0.5,'log \ge\d\gn')
            call pgqlw(lw)
            call pgslw(8)
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgsci(clr_LightSteelBlue)
            call pgsch(label_scale)
            call pgmtxt('L',2.5,0.5,0.5,'log \ge\dnuc\u')
            call pgqlw(lw)
            call pgslw(8)
            call pgline(npts, xvec(grid_min:grid_max), yvec3(grid_min:grid_max))
            call pgslw(lw_sav)

            ! label peaks of eps_nuc
            ! (stack the labels if they are too close in the x-direction)
            num_cat = 0
            do i = 1, num_categories
               if (i == iphoto) cycle
               maxv = safe_log10(maxval(s% eps_nuc_categories(i_rate,i,1:nz)))
               if (maxv > -1) then
                  num_cat = num_cat + 1
                  docat(num_cat) = i
                  xnuc_cat(num_cat) = maxv
               end if
            end do
            call pgsci(1)
            call pgsch(label_scale*0.9)
            do ii = 1, num_cat
               eps_max = -100; i = 0
               do jj = 1, num_cat
                  if (xnuc_cat(jj) < -1) cycle
                  if (xnuc_cat(jj) > eps_max) then
                     eps_max = xnuc_cat(jj)
                     i = jj
                  end if
               end do
               if (i == 0) exit
               if (dbg) write(*,2) 'place ' // category_name(docat(i)), i, eps_max
               xnuc_cat(i) = -1e10 ! mark as done
               eps_max = -100
               eps_k(i) = 0
               do k = 1, nz ! if limit this to grid_min:grid_max, locations jump around too much
                  eps = s% eps_nuc_categories(i_rate,docat(i),k)
                  if(eps > eps_max) then
                     eps_max = eps
                     eps_k(i) = k
                  end if
               end do
               if(eps_k(i) > 0) then
                  k = eps_k(i)
                  xpos_nuc(i) = xvec(k)
                  if (xpos_nuc(i) < xmin .or. xpos_nuc(i) > xmax) cycle
                  cnt = 0
                  do j = 1, num_cat ! compare location to ones already placed
                     if (j == i) cycle
                     if (xnuc_cat(j) > -1) cycle ! haven't done this one yet
                     if (abs(xpos_nuc(i) - xpos_nuc(j)) < 0.1*dx) then
                        cnt = cnt + 1
                        if (dbg) write(*,*) 'conflicts with ' // category_name(docat(j))
                     end if
                  end do
                  if (cnt < 3) then ! only show 3 max
                     str = category_name(docat(i))
                     if (str(1:5) == 'burn_') then
                        str = str(6:len_trim(str))
                     else if (str == 'tri_alfa') then
                        str = '3a'
                     else if (str == 'c12_c12') then
                        str = 'c+c'
                     else if (str == 'c12_o16') then
                        str = 'c+o'
                     else if (str == 'o16_o16') then
                        str = 'o+o'
                     end if

                     if (s% show_main_win_text_info) then
                        ylb = ymax + dy*(0.06 + 0.18*cnt)
                     else
                        ylb = ymax + dy*(0.06 + 0.14*cnt)
                     end if

                     call pgptxt(xpos_nuc(i), ylb, 0.0, 0.5, trim(str))
                  end if
               end if
            end do

            call pgunsa
         end subroutine plot1


         subroutine plot2
            use chem_lib
            integer :: lw, lw_sav, j, num_to_show, cid
            real :: species_ymax, species_ymin, dylbl
            real, parameter :: logL_cutoff = -2, L_cutoff = 1e-2
         
            include 'formats.dek'
            
            
            call pgsave

            lw = 6
            call pgqlw(lw_sav)

            call set_plot2_vp                       
            
            !  abundances
            call pgsci(1)         
            call pgswin(xleft, xright, s% log_mass_frac_ymin, s% log_mass_frac_ymax)
            call pgsch(label_scale)
            call pgbox('BCST',0.0,0,'BCNMSTV1',0.0,0)
            call pgmtxt('L',4.0,0.5,0.5,'log mass frac')
            call pgmtxt('R',5.0,0.5,0.5,'log mass frac')
         
            call pgsch(label_scale)
         
            num_labels = max(0,min(max_num_labels, s% num_abundance_line_labels))
            do i=1,num_labels
               x = xmin + (i-0.5)*dx/num_labels
               do k=2,nz
                  if ((xvec(k-1)-x)*(x-xvec(k)) >= 0) then
                     iloc_abundance_label(i) = k
                     xloc_abundance_label(i) = x
                     exit
                  end if
               end do
            end do
            
            dylbl = (s% log_mass_frac_ymax- s% log_mass_frac_ymin)*0.02
            num_to_show = s% Abundance_num_isos_to_show 
            if (num_to_show < 0) &
               num_to_show = min(s% species, max_Abundance_num_isos_to_show)
            do i = 1, num_to_show
               if (s% Abundance_num_isos_to_show < 0) then
                  cid = s% chem_id(i)
                  str = chem_isos% name(cid)
               else
                  str = s% Adundance_which_isos_to_show(i)
                  cid = chem_get_iso_id(str)
               end if
               if (len_trim(str) == 0 .or. cid <= 0) cycle
               sp = s% net_iso(cid)
               if (sp == 0) cycle
               call pgsci(colors(1 + mod(i-1,num_colors)))
               fcol = mod(i,4)/4.0
               frow = 0.25*(i-mod(i,4))
               str = trim(chem_isos% name(cid))
               select case (i/num_colors)
                  case (0)
                     call pgsls(Line_Type_Solid)
                  case (1)
                     call pgsls(Line_Type_Dot)
                  case (2)
                     call pgsls(Line_Type_Dash)
                  case default
                     call pgsls(Line_Type_Dot_Dash)
               end select
               call pgslw(lw)
               yvec = safe_log10(s% xa(sp,1:nz))
               call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
               call pgslw(lw_sav)
               do j=1,num_labels
                  ii = iloc_abundance_label(j)
                  if (ii > grid_max .or. ii < grid_min) cycle
                  if (xvec(ii) > xmin .and. xvec(ii) < xmax .and. &
                      yvec(ii) > s% log_mass_frac_ymin .and. yvec(ii) < s% log_mass_frac_ymax) &
                     call pgptxt(xloc_abundance_label(j), yvec(ii)+dylbl, 0.0, 0.5, str)
               end do
            end do            

            call pgunsa
         end subroutine plot2


         subroutine plot3(ierr)
            integer, intent(out) :: ierr
            integer :: lw, lw_sav, k
            real :: ybot, eps
         
            include 'formats.dek'
            call pgsave
                       
            lw = 6
            call pgqlw(lw_sav)

            call set_plot3_vp
            
            yvec(1:nz) = exp(s% lnS(1:nz))/(avo*kerg)
            ymin = minval(yvec(grid_min:grid_max))
            ymax = maxval(yvec(grid_min:grid_max))
            dy = ymax-ymin
            ymax = ymax + 0.1*dy
            ymin = ymin - 0.1*dy

            call pgswin(xleft, xright, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(label_scale)
            call pgbox('BCNST',0.0,0,'CMSTV',0.0,0)
            
            call pgsci(1)
            if (s% show_main_win_text_info) then
               call pgmtxt('R',5.0,0.5,0.5,'entropy')
            else
               call pgmtxt('R',5.0,0.5,0.5,'entropy (k\dB\u/m\dp\u)')
            end if
            call pgsci(clr_LightSteelBlue)
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            

            call pgsci(1)
            call pgsch(label_scale)
            call show_xaxis_by(s,s% xaxis_by,ierr)
            if (ierr /= 0) return

            ybot = -0.05
            call pgswin(xleft, xright, ybot, 0.85)
            call pgscf(1)
            call pgsci(1)
            call pgsch(label_scale)
            call pgbox('',s% main_xtick,s% main_nxsub,'BNSTV',0.0,0)
         
            ! actual
            yvec(1:nz) = s% gradT(1:nz)
            call pgsci(clr_FireBrick)
            call pgsch(label_scale)
            call pgmtxt('L',4.9,0.5,0.5,'\(2266)\d\(0856)')
            call pgsch(chScale)
            call pgpt(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max), 850);
            call pgsch(chScale)

            ! adiabatic nabla
            yvec(1:nz) = s% grada(1:nz)
            call pgsci(clr_Tan)
            call pgsch(label_scale)
            call pgmtxt('L',3.3,0.75,0.5,'\(2266)\dad')
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            ! radiative nabla
            yvec(1:nz) = s% gradr(1:nz)
            call pgsci(clr_Coral)
            call pgsch(label_scale)
            call pgmtxt('L',3.3,0.25,0.5,'\(2266)\drad')
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            
            ! show mix regions at bottom of plot
            call pgslw(10)
            call show_mix_regions_on_xaxis(s,ybot,grid_min,grid_max,xvec)
            
            call pgunsa
            
         end subroutine plot3


         subroutine header
            call pgsave
            ! header
            call set_header_vp
            call pgsci(1)
            call pgscf(1)
            call pgswin(0.0,1.0,0.0,1.0)
            write(str,'(i9)') s% model_number
            call pgsch(0.9*label_scale)
            call pgmtxt('T',-0.2,1.16,1.0,str)
            call pgunsa
         end subroutine header


         subroutine footer(ierr)
            use chem_def, only: iphoto
            integer, intent(out) :: ierr
            
            real*8 :: v, vsurf_div_csound, power_photo
            character (len=32) :: why
            real :: xpos0, dxpos, dxval, ypos, dypos
            
            ierr = 0
            call pgsave
            call set_footer_vp
            call pgsch(label_scale*0.7)
            call pgsci(1)
            call pgscf(1)
            call pgswin(0.0,1.0,0.0,1.0)

            dxpos = 0.36
            xpos0 = -0.10
            dxval = 0.014
            ypos = 0.5
            dypos = -0.08
            
            if (s% v_flag) then
               v = s% v(1)
            else
               v = s% r(1)*s% dlnR_dt(1)
            end if
            vsurf_div_csound = v / sqrt(s% gamma1(1)*s% P(1)/s% rho(1))

            power_photo = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
         
            why = dt_why_str(min(numTlim,s% why_Tlim))
            if (s% why_Tlim == Tlim_dX_nuc .and. s% Tlim_dXnuc_species > 0 &
                     .and. s% Tlim_dXnuc_species <= s% species) then
               why = trim(dt_why_str(s% why_Tlim)) // ' ' // &
                  trim(chem_isos% name(s% chem_id(s% Tlim_dXnuc_species)))
            elseif (s% why_Tlim == Tlim_dX .and. s% Tlim_dX_species > 0 &
                     .and. s% Tlim_dX_species <= s% species) then
               why = trim(dt_why_str(s% why_Tlim)) // ' ' // &
                  trim(chem_isos% name(s% chem_id(s% Tlim_dX_species)))
            else if (s% why_Tlim ==  Tlim_dX_div_X .and. s% Tlim_dX_div_X_species > 0 &
                     .and. s% Tlim_dX_div_X_species <= s% species) then
               why = trim(dt_why_str(s% why_Tlim)) // ' ' // &
                  trim(chem_isos% name(s% chem_id(s% Tlim_dX_div_X_species)))
            else if (s% why_Tlim ==  Tlim_dlgL_nuc_cat) then 
               if (s% Tlim_dlgL_nuc_category > 0 &
                     .and. s% Tlim_dlgL_nuc_category <= num_categories ) then
                  why = trim(category_name(s% Tlim_dlgL_nuc_category))
               else
                  write(*,*) 's% Tlim_dlgL_nuc_category', s% Tlim_dlgL_nuc_category
                  write(*,*) 'num_categories', num_categories
                  write(*,*) 
                  write(*,*) 'summary plot'
                  ierr = -1
                  return
               end if
            end if         

            cnt = 0
            ! row 1     lg dt yr lg Lnuc     lg Tcntr    init Z
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg dt yr', safe_log10(s% time_step))
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg Lnuc', safe_log10(s% power_nuc_burn - power_photo))
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'H cntr', s% center_h1)
            if (s% initial_Z > 1d-6) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                        'initial Z', s% initial_Z)
            else
               cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                        'initial Z', s% initial_Z)
            end if
            
            ypos = ypos + dypos
            cnt = 0
            ! row 2     age yr   lg Lneu     lg Dcntr    init Mass
            cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                     'age yr', s% star_age)
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg Lneu', safe_log10(s% power_neutrinos))
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'He cntr', s% center_he3 + s% center_he4)
            if (s% M_center == 0) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                           'init mass', s% initial_mass)
            else
               cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                           'env mass', s% xmstar)
            end if
            
            ypos = ypos + dypos
            cnt = 0
            ! row 3     lg L     lg LH       lg Pcntr    Mass
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg L', s% log_surface_luminosity)
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg LH', safe_log10(s% power_h_burn))
            if (s% M_center == 0 .or. s% net_iso(ih1) == 0) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                        'C cntr', s% center_c12)
            else
               cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                     'H1 (g)', dot_product(s% dm(1:nz),s% xa(s% net_iso(ih1),1:nz)))
            end if
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                  'mass', s% star_mass)
            
            ypos = ypos + dypos
            cnt = 0
            ! row 4     Teff     lg L3a      H cntr      lg Mdot
            cnt = write_info_line_flt2(cnt, ypos, xpos0, dxpos, dxval, &
                     'Teff', s% Teff)
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg L3a', safe_log10(s% power_he_burn))
            if (s% M_center == 0 .or. s% net_iso(ihe4) == 0) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'N cntr', s% center_n14)
            else
               cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                     'He4 (g)', dot_product(s% dm(1:nz),s% xa(s% net_iso(ihe4),1:nz)))
            end if
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg \(0723)Mdot\(0723)', safe_log10(abs(s% star_mdot)))
            
            ypos = ypos + dypos
            cnt = 0
            ! row 5     lg Rsurf lg LZ       He cntr     H rich
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg Rsurf', s% log_surface_radius)
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg LZ', &
                     safe_log10(s% power_nuc_burn - (power_photo + s% power_h_burn + s% power_he_burn)))
            if (s% M_center == 0 .or. s% net_iso(ic12) == 0) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'O cntr', s% center_o16)
            else
               cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                     'C12 (g)', dot_product(s% dm(1:nz),s% xa(s% net_iso(ic12),1:nz)))
            end if
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'H rich', s% star_mass - s% h1_boundary_mass)
            
            ypos = ypos + dypos
            cnt = 0
            ! row 6     lg Psurf  points    Ne cntr      H poor
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg Psurf', s% log_surface_pressure)
            cnt = write_info_line_int(cnt, ypos, xpos0, dxpos, dxval, &
                     'zones', s% nz)
            if (s% M_center == 0 .or. s% net_iso(io16) == 0) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'Ne cntr', s% center_ne20)
            else
               cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                     'O16 (g)', dot_product(s% dm(:),s% xa(s% net_iso(io16),:)))
            end if
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'H poor', s% h1_boundary_mass)
            
            ypos = ypos + dypos
            cnt = 0
            ! row 7     lg grav     iters    Si cntr       He poor
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg grav', safe_log10(s% grav(1)))
            cnt = write_info_line_int(cnt, ypos, xpos0, dxpos, dxval, &
                     'newton', s% num_jacobians)
            if (s% net_iso(isi28) == 0) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                        'lg Pcntr', s% log_center_pressure)
            else if (s% M_center == 0 .or. s% net_iso(ine20) == 0) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                        'Si cntr', s% center_si28)
            else
               cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                     'Ne20 (g)', dot_product(s% dm(:),s% xa(s% net_iso(ine20),:)))
            end if
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'He poor', s% he4_boundary_mass)
            
            ypos = ypos + dypos
            cnt = 0
            ! row 8     v div cs     retry     O cntr    eta cntr
            cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                     'v div cs', vsurf_div_csound)
            cnt = write_info_line_int(cnt, ypos, xpos0, dxpos, dxval, &
                     'retries', s% num_retries)
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg Tcntr', s% log_center_temperature)
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'eta cntr', s% eta(s% nz))
            
            ypos = ypos + dypos
            cnt = 0
            ! row 9     Z surf     why     Ne cntr      gam cntr
            if (s% initial_z >= 1d-5) then
               cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                        'Z surf', 1 - (s% surface_h1 + s% surface_he3 + s% surface_he4))
            else
               cnt = write_info_line_exp(cnt, ypos, xpos0, dxpos, dxval, &
                        'lg Z surf', safe_log10(1 - (s% surface_h1 + s% surface_he3 + s% surface_he4)))
            end if
            cnt = write_info_line_str(cnt, ypos, xpos0, dxpos, why)
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'lg Dcntr', s% log_center_density)
            cnt = write_info_line_flt(cnt, ypos, xpos0, dxpos, dxval, &
                     'gam cntr', s% gam(s% nz))

            call pgunsa
         end subroutine footer


         subroutine set_header_vp
            real :: ymax, ymin
            ymax = winymax
            ymin = ymax - header_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_header_vp


         subroutine set_plot1_vp
            real :: ymax, ymin
            ymax = winymax - header_frac*windy
            ymin = ymax - plot1_frac*windy + win_gap
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot1_vp


         subroutine set_plot2_vp
            real :: ymax, ymin
            ymax = winymax - (header_frac + plot1_frac)*windy
            ymin = ymax - plot2_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot2_vp


         subroutine set_plot3_vp
            real :: ymax, ymin
            ymax = winymax - (header_frac + plot1_frac + plot2_frac)*windy - win_gap
            ymin = ymax - plot3_frac*windy + win_gap
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot3_vp


         subroutine set_footer_vp
            real :: ymax, ymin
            ymin = winymin
            ymax = ymin + footer_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_footer_vp
         

      end subroutine Summary_Plot


            

      end module mod_pgstar_summary

