! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_power

      use star_private_def
      use const_def
      use mod_pgstar_support

      implicit none


      contains
      

      subroutine power_plot(id, device_id, ierr)
         implicit none
         integer, intent(in) :: id, device_id
         integer, intent(out) :: ierr

         real :: winxmin, winxmax, winymin, winymax, label_scale
         type (star_info), pointer :: s

         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call pgslct(device_id)
         call pgbbuf()
         call pgeras()

         label_scale = 1.0         
         winxmin = 0.15
         winxmax = 0.80
         winymin = 0.07
         winymax = 0.93
         
         call do_power_plot(id, device_id, &
            winxmin, winxmax, winymin, winymax, &
            s% Power_xaxis_by, s% Power_xmin, s% Power_xmax, &
            label_scale, ierr)

         call pgebuf()
         
      end subroutine power_plot
      

      subroutine do_power_plot(id, device_id, &
            winxmin, winxmax, winymin, winymax, &
            xaxis_by, xaxis_min, xaxis_max, label_scale, ierr)
         use utils_lib
         use chem_def
         use num_lib, only: safe_log10
         use net_def
         use const_def, only: Msun, Rsun
         implicit none

         integer, intent(in) :: id, device_id
         real, intent(in) :: winxmin, winxmax, winymin, winymax
         character (len=256) :: xaxis_by
         real, intent(in) :: xaxis_min, xaxis_max, label_scale
         integer, intent(out) :: ierr
                  
         character (len=256) :: str
         real, pointer, dimension(:) :: xvec, yvec
         real :: xmin, xmax, xleft, xright, dx, chScale, windy, &
            ymin, ymax, exp10_ymin, legend_xmin, legend_xmax, xmargin
         integer :: lw, lw_sav, grid_min, grid_max, npts, nz
         integer, parameter :: num_colors = 20
         integer :: colors(num_colors)
         type (star_info), pointer :: s
         
         include 'formats.dek'
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         nz = s% nz        
         
         colors(:) = (/ &
               clr_MediumSlateBlue, clr_LightSkyBlue, clr_Goldenrod, clr_Lilac, &
               clr_Coral, clr_Crimson, clr_LightSkyGreen, clr_DarkGray, &
               clr_Tan, clr_IndianRed, clr_Gold, &
               clr_Teal, clr_Silver, clr_BrightBlue, clr_FireBrick, &
               clr_RoyalPurple, clr_SlateGray, clr_LightSteelBlue, &
               clr_Gray, clr_RoyalBlue /)

         chScale = label_scale

         windy = winymax - winymin
         
         legend_xmin = 0.84
         legend_xmax = 0.99

         allocate (xvec(nz), yvec(nz))
         
         xmargin = 0
         call set_xaxis_bounds(s, xaxis_by, xaxis_min, xaxis_max, .false., xmargin, &
            xvec, xmin, xmax, xleft, xright, dx, &
            grid_min, grid_max, npts, ierr)
         
         if (ierr /= 0) then
            ! just skip the plotting
            ierr = 0
            call pgsvp(winxmin, winxmax, winymin, winymax)
            call pgsch(chScale*1.2)
            call pgmtxt('T',-5.0,0.5,0.5,'Power plot')
            call pgmtxt('T',-7.0,0.5,0.5,'Bad Xmin and/or Xmax specification')
            call pgmtxt('T',-9.0,0.5,0.5,'please check inlist')
         else
            call plot(ierr)
            if (ierr /= 0) return
         end if         

         deallocate(xvec, yvec)

         if (s% show_Power_cross_hair) call do_cross_hair

         contains
                  
         
         subroutine plot(ierr)
            use rates_def
            integer, intent(out) :: ierr
            
            integer :: ii, jj, i, cnt
            logical, parameter :: dbg = .false.
            real :: ybot
         
            include 'formats.dek'

            ymax = maxval(s% eps_nuc_categories(i_rate,:,grid_min:grid_max))
            
            if (ymax < 1e-29) ymax = 1e-29
            ymax = log10(ymax)
            if (ymax <= 10) then
               ymax = 10.1
               ymin = -4.1
            else
               ymax = ymax*1.1
               ymin = -0.1
            end if
            
            if (s% Power_ymax > -100) ymax = s% Power_ymax
            if (s% Power_ymin > -100) ymin = s% Power_ymin
            
            exp10_ymin = 10**ymin

            lw = 6
            call pgqlw(lw_sav)

            call pgsave
            call pgsvp(legend_xmin, legend_xmax, winymin, winymax)
            call pgswin(0.0, 1.0, ymin, ymax)
            cnt = 0
            do i=1,num_categories
               cnt = power_line_legend(cnt,i)
            end do
            call pgunsa


            call pgsave
            call pgsvp(winxmin, winxmax, winymin, winymax)
         
            write(str,'(i9)') s% model_number
            call pgsch(label_scale)
            call pgmtxt('T',1.5,0.9,0.5,str)
         
            ybot = -0.05
            call pgswin(xleft, xright, ymin+ybot, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgmtxt('T',1.5,0.5,0.5,'Power')
            call pgsch(label_scale)
            call pgbox('BCNST',0.0,0,'BCNSTV',0.0,0)
            call pgmtxt('L',4.0,0.5,0.5,'log ergs/g/s')
            
            call pgslw(lw)
            cnt = 0
            do i=1,num_categories
               cnt = power_line(cnt,i)
            end do
            call pgslw(lw_sav)
            
            call pgsci(1)
            call pgsch(label_scale)
            call show_xaxis_by(s,xaxis_by,ierr)
            if (ierr /= 0) return
            
            ! show mix regions at bottom of plot
            call pgslw(10)
            call show_mix_regions_on_xaxis(s,ymin+ybot,grid_min,grid_max,xvec)
            
            call pgunsa

         end subroutine plot
         
         
         integer function power_line(cnt, icat)
            integer, intent(in) :: cnt, icat
            real :: ymx, xpos, dx, ypos, xpts(2), ypts(2)
            integer :: iclr
            power_line = cnt
            ymx = maxval(s% eps_nuc_categories(i_rate,icat,grid_min:grid_max))
            if (ymx < exp10_ymin) return
            iclr = cnt - num_colors*(cnt/num_colors) + 1
            power_line = cnt + 1
            call pgsci(colors(iclr))
            yvec = safe_log10(s% eps_nuc_categories(i_rate,icat,1:nz))
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
         end function power_line
         
         
         integer function power_line_legend(cnt, icat)
            integer, intent(in) :: cnt, icat
            real :: ymx, dx, dyline, ypos, xpts(2), ypts(2)
            integer :: iclr
            integer, parameter :: num_max = num_categories
            power_line_legend = cnt
            ymx = maxval(s% eps_nuc_categories(i_rate,icat,grid_min:grid_max))
            if (ymx < exp10_ymin) return
            iclr = cnt - num_colors*(cnt/num_colors) + 1
            power_line_legend = cnt + 1
            call pgsci(colors(iclr))
            dx = 0.1
            dyline = (ymax-ymin)/num_max
            ypos = ymax - (cnt+0.5)*dyline
            xpts(1) = 2*dx
            xpts(2) = xpts(1) + 3*dx
            ypts = ypos + dyline*0.1
            call pgslw(lw)
            call pgline(2, xpts, ypts)
            call pgslw(lw_sav)
            call pgsci(1)
            call pgsch(label_scale*0.75)
            call pgptxt(xpts(2) + dx, ypos, 0.0, 0.0, trim(adjustl(category_name(icat))))
            power_line_legend = cnt + 1
         end function power_line_legend
         

      end subroutine do_power_plot


      end module mod_pgstar_power

