! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_mode_prop

      use star_private_def
      use const_def
      use mod_pgstar_support

      implicit none


      contains
      

      subroutine mode_propagation_plot(id, device_id, ierr)
         implicit none
         integer, intent(in) :: id, device_id
         integer, intent(out) :: ierr

         real :: winxmin, winxmax, winymin, winymax, label_scale
         type (star_info), pointer :: s
         
         include 'formats.dek'

         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call pgslct(device_id)
         call pgbbuf()
         call pgeras()

         label_scale = 1.0         
         winxmin = 0.15
         winxmax = 0.80
         winymin = 0.07
         winymax = 0.93
         
         call do_mode_propagation_plot(id, device_id, &
            winxmin, winxmax, winymin, winymax, &
            s% Mode_Prop_xaxis_by, s% Mode_Prop_xmin, s% Mode_Prop_xmax, &
            label_scale, ierr)

         call pgebuf()
         
      end subroutine mode_propagation_plot
      

      subroutine do_mode_propagation_plot(id, device_id, &
            winxmin, winxmax, winymin, winymax, &
            xaxis_by, xaxis_min, xaxis_max, label_scale, ierr)
         use utils_lib
         use chem_def
         use num_lib, only: safe_log10
         use net_def
         use const_def, only: Msun, Rsun
         implicit none

         integer, intent(in) :: id, device_id
         real, intent(in) :: winxmin, winxmax, winymin, winymax
         character (len=256) :: xaxis_by
         real, intent(in) :: xaxis_min, xaxis_max, label_scale
         integer, intent(out) :: ierr
                  
         character (len=256) :: str
         real, pointer, dimension(:) :: xvec, log_brunt_nu, &
            log_lamb_Sl1, log_lamb_Sl2, log_lamb_Sl3, temp_vec
         real :: xmin, xmax, xleft, xright, dx, chScale, windy, &
            ymin, ymax, exp10_ymin, legend_xmin, legend_xmax, xmargin
         integer :: lw, lw_sav, grid_min, grid_max, npts, nz
         integer, parameter :: num_colors = 20
         integer :: colors(num_colors)
         type (star_info), pointer :: s
         
         include 'formats.dek'
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         nz = s% nz        
         
         colors(:) = (/ &
               clr_MediumSlateBlue, clr_Goldenrod, clr_LightSkyBlue, clr_Lilac, &
               clr_Coral, clr_Crimson, clr_LightSkyGreen, clr_DarkGray, &
               clr_Tan, clr_IndianRed, clr_Gold, &
               clr_Teal, clr_Silver, clr_BrightBlue, clr_FireBrick, &
               clr_RoyalPurple, clr_SlateGray, clr_LightSteelBlue, &
               clr_Gray, clr_RoyalBlue /)

         chScale = label_scale

         windy = winymax - winymin
         
         legend_xmin = 0.84
         legend_xmax = 0.99

         allocate (xvec(nz), log_brunt_nu(nz), &
            log_lamb_Sl1(nz), log_lamb_Sl2(nz), log_lamb_Sl3(nz), temp_vec(nz))
         
         xmargin = 0
         call set_xaxis_bounds(s, xaxis_by, xaxis_min, xaxis_max, .false., xmargin, &
            xvec, xmin, xmax, xleft, xright, dx, &
            grid_min, grid_max, npts, ierr)
         
         if (ierr /= 0) then
            ! just skip the plotting
            ierr = 0
            call pgsvp(winxmin, winxmax, winymin, winymax)
            call pgsch(chScale*1.2)
            call pgmtxt('T',-5.0,0.5,0.5,'Mode Propagation')
            call pgmtxt('T',-7.0,0.5,0.5,'Bad Xmin and/or Xmax specification')
            call pgmtxt('T',-9.0,0.5,0.5,'please check inlist')
         else
            call plot(ierr)
            if (ierr /= 0) return
         end if         

         deallocate(xvec, log_brunt_nu, log_lamb_Sl1, log_lamb_Sl2, log_lamb_Sl3, temp_vec)

         if (s% show_Mode_Prop_cross_hair) call do_cross_hair

         contains
                  
         
         subroutine plot(ierr)
            use rates_def
            integer, intent(out) :: ierr
            
            integer :: ii, jj, i, cnt, k
            logical, parameter :: dbg = .false.
            real :: ybot, nu_max, lg_nu_max, lg_2pt0_nu_max, lg_0pt5_nu_max, lg_nu_max_obs
            real, parameter :: teff_sun = 5777.0, nu_max_sun = 3100.0
        
            include 'formats.dek'

            do k=1,nz
               log_brunt_nu(k) = safe_log10((1d6/(2*pi))*sqrt(max(0d0,s% brunt_N2(k))))
               log_lamb_Sl1(k) = safe_log10((1d6/(2*pi))*sqrt(2d0)*s% csound_at_face(k)/s% r(k))
               log_lamb_Sl2(k) = safe_log10((1d6/(2*pi))*sqrt(6d0)*s% csound_at_face(k)/s% r(k))
               log_lamb_Sl3(k) = safe_log10((1d6/(2*pi))*sqrt(12d0)*s% csound_at_face(k)/s% r(k))
            end do
            
            nu_max = nu_max_sun*s% star_mass/(s% photosphere_r**2*sqrt(s% Teff/teff_sun))
            lg_nu_max = log10(nu_max)
            lg_2pt0_nu_max = log10(2.0*nu_max)
            lg_0pt5_nu_max = log10(0.5*nu_max)
            lg_nu_max_obs = safe_log10(dble(s% Mode_Prop_nu_max_obs))
            

            ymax = 1.33*lg_2pt0_nu_max
            ymin = 0.5*lg_0pt5_nu_max
            
            if (s% Mode_Prop_ymax > -100) ymax = s% Mode_Prop_ymax
            if (s% Mode_Prop_ymin > -100) ymin = s% Mode_Prop_ymin

            lw = 6
            call pgqlw(lw_sav)

            call pgsave
            call pgsvp(legend_xmin, legend_xmax, winymin, winymax)
            call pgswin(0.0, 1.0, ymin, ymax)
            cnt = 0
            cnt = mode_propagation_line_legend(cnt,'log N\dBV\u')
            cnt = mode_propagation_line_legend(cnt,'log S\dl=1\u')
            cnt = mode_propagation_line_legend(cnt,'log S\dl=2\u')
            cnt = mode_propagation_line_legend(cnt,'log S\dl=3\u')
            cnt = mode_propagation_line_legend(cnt,'log 2.0\(2723)\(2139)\dmax\u mod')
            cnt = mode_propagation_line_legend(cnt,'log \(2139)\dmax\u mod')
            call pgsls(4) ! dotted
            cnt = mode_propagation_line_legend(cnt,'log \(2139)\dmax\u obs')
            call pgsls(1) ! solid
            cnt = mode_propagation_line_legend(cnt,'log 0.5\(2723)\(2139)\dmax\u mod')
            call pgunsa


            call pgsave
            call pgsvp(winxmin, winxmax, winymin, winymax)
         
            write(str,'(i9)') s% model_number
            call pgsch(label_scale)
            call pgmtxt('T',1.5,0.9,0.5,str)
         
            ybot = -0.05
            call pgswin(xleft, xright, ymin+ybot, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgmtxt('T',1.5,0.5,0.5,'Mode Propagation')
            call pgsch(label_scale)
            call pgbox('BCNST',0.0,0,'BCNSTV',0.0,0)
            call pgmtxt('L',4.0,0.5,0.5,'log \(2139) (\(2138)Hz)')
            
            call pgslw(lw)
            cnt = 0
            cnt = mode_propagation_line(cnt, log_brunt_nu)
            cnt = mode_propagation_line(cnt, log_lamb_Sl1)
            cnt = mode_propagation_line(cnt, log_lamb_Sl2)
            cnt = mode_propagation_line(cnt, log_lamb_Sl3)
            temp_vec(1:nz) = lg_2pt0_nu_max
            cnt = mode_propagation_line(cnt, temp_vec)
            temp_vec(1:nz) = lg_nu_max
            cnt = mode_propagation_line(cnt, temp_vec)
            call pgsls(4) ! dotted
               temp_vec(1:nz) = lg_nu_max_obs
               cnt = mode_propagation_line(cnt, temp_vec)
            call pgsls(1) ! solid
            temp_vec(1:nz) = lg_0pt5_nu_max
            cnt = mode_propagation_line(cnt, temp_vec)
            call pgslw(lw_sav)
            
            call pgsci(1)
            call pgsch(label_scale)
            call show_xaxis_by(s,xaxis_by,ierr)
            if (ierr /= 0) return
            
            ! show mix regions at bottom of plot
            call pgslw(10)
            call show_mix_regions_on_xaxis(s,ymin+ybot,grid_min,grid_max,xvec)
            
            call pgunsa

         end subroutine plot
         
         
         integer function mode_propagation_line(cnt, yvec)
            integer, intent(in) :: cnt
            real, intent(in) :: yvec(:)
            integer :: iclr
            iclr = cnt - num_colors*(cnt/num_colors) + 1
            mode_propagation_line = cnt + 1
            call pgsci(colors(iclr))
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
         end function mode_propagation_line
         
         
         integer function mode_propagation_line_legend(cnt, name)
            integer, intent(in) :: cnt
            character (len=*), intent(in) :: name
            real :: dx, dyline, ypos, xpts(2), ypts(2)
            integer :: iclr, num_max
            num_max = 20
            mode_propagation_line_legend = cnt
            iclr = cnt - num_colors*(cnt/num_colors) + 1
            call pgsci(colors(iclr))
            dx = 0.1
            dyline = (ymax-ymin)/num_max
            ypos = ymax - (cnt+0.5)*dyline
            xpts(1) = 1*dx
            xpts(2) = xpts(1) + 2*dx
            ypts = ypos + dyline*0.1
            call pgslw(lw)
            call pgline(2, xpts, ypts)
            call pgslw(lw_sav)
            call pgsci(1)
            call pgsch(label_scale*0.75)
            call pgptxt(xpts(2) + dx, ypos, 0.0, 0.0, name)
            mode_propagation_line_legend = cnt + 1
         end function mode_propagation_line_legend
         

      end subroutine do_mode_propagation_plot


      end module mod_pgstar_mode_prop

