! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_dynamo

      use star_private_def
      use const_def
      use mod_pgstar_support

      implicit none


      contains
      

      subroutine Dynamo_plot(id, device_id, ierr)
         implicit none
         integer, intent(in) :: id, device_id
         integer, intent(out) :: ierr
         
         real :: winxmin, winxmax, winymin, winymax, label_scale

         type (star_info), pointer :: s

         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call pgslct(device_id)
         call pgbbuf()
         call pgeras()

         winxmin = 0.14
         winxmax = 0.85
         winymin = 0.13
         winymax = 0.92
         label_scale = 1.2
         
         call do_Dynamo_plot(id, device_id, &
            winxmin, winxmax, winymin, winymax, &
            s% Dynamo_xaxis_by, s% Dynamo_xmin, s% Dynamo_xmax, &
            label_scale, ierr)

         call pgebuf()
      
      end subroutine Dynamo_plot
      

      subroutine do_Dynamo_plot(id, device_id, &
            winxmin, winxmax, winymin, winymax, &
            xaxis_by, xaxis_min, xaxis_max, label_scale, ierr)
         implicit none
         integer, intent(in) :: id, device_id
         real, intent(in) :: winxmin, winxmax, winymin, winymax
         character (len=256) :: xaxis_by
         real, intent(in) :: xaxis_min, xaxis_max, label_scale
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s

         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call Dyn_plot(s, device_id, &
            s% show_Dynamo_cross_hair, &
            s% show_Dynamo_annotation1, s% show_Dynamo_annotation2, s% show_Dynamo_annotation3, &
            xaxis_by, xaxis_min, xaxis_max, &
            s% Dynamo_ymin_left, s% Dynamo_ymax_left, s% Dynamo_dymin_left, &
            s% Dynamo_ymin_right, s% Dynamo_ymax_right, s% Dynamo_dymin_right, &
            winxmin, winxmax, winymin, winymax, label_scale, ierr)
         
      end subroutine do_Dynamo_plot
      


      subroutine Dyn_plot(s, device_id, &
            show_Dyn_cross_hair, &
            show_Dyn_annotation1, show_Dyn_annotation2, show_Dyn_annotation3, &
            Dyn_xaxis_by, Dyn_xmin, Dyn_xmax, &
            Dyn_ymin_left, Dyn_ymax_left, Dyn_dymin_left, &
            Dyn_ymin_right, Dyn_ymax_right, Dyn_dymin_right, &
            winxmin, winxmax, winymin, winymax, label_scale, ierr)
            
         use utils_lib
         use num_lib, only: safe_log10
         implicit none

         type (star_info), pointer :: s
         integer, intent(in) :: device_id
         logical, intent(in) :: show_Dyn_cross_hair
         logical, intent(in) :: show_Dyn_annotation1, show_Dyn_annotation2, show_Dyn_annotation3
         character (len=*), intent(in) :: Dyn_xaxis_by
         real, intent(in) :: &
            Dyn_xmin, Dyn_xmax, &
            Dyn_ymin_left, Dyn_ymax_left, Dyn_dymin_left, &
            Dyn_ymin_right, Dyn_ymax_right, Dyn_dymin_right
         real, intent(in) :: winxmin, winxmax, winymin, winymax, label_scale
         integer, intent(out) :: ierr
         
         real :: windy, xmargin
         real :: xmin, xmax, xleft, xright, dx, tmp, ymin, ymax, ymin2, ymax2, dy
         integer :: grid_min, grid_max, npts, nz
         real, pointer, dimension(:) :: xvec, yvec, yvec2, yvec3
         
         include 'formats.dek'
         ierr = 0
         
         if (.not. s% rotation_flag) return

         
         xmargin = 0

         nz = s% nz
         allocate (xvec(nz), yvec(nz), yvec2(nz), yvec3(nz))
         
         call set_xaxis_bounds( &
            s, Dyn_xaxis_by, Dyn_xmin, Dyn_xmax, .false., xmargin, &
            xvec, xmin, xmax, xleft, xright, dx, &
            grid_min, grid_max, npts, ierr)
         if (ierr /= 0) return

         call plot(ierr)
         if (ierr /= 0) return
         
         call show_annotations(s, &
            show_Dyn_annotation1, show_Dyn_annotation2, show_Dyn_annotation3)

         deallocate(xvec, yvec, yvec2, yvec3)
                  
         if (show_Dyn_cross_hair) call do_cross_hair
         
         
         contains
         
         
         subroutine plot(ierr)
            use mod_pgstar_summary, only: show_convective_section, show_semiconvective_section, &
               show_thermo_haline_section, show_overshoot_section
            integer, intent(out) :: ierr
            
            integer :: lw, lw_sav
            real :: ybot, eps, &
               default_ymax_left, default_ymin_left, &
               default_ymax_right, default_ymin_right
            character (len=128) :: str
         
            include 'formats.dek'
            ierr = 0
            
            call pgsave
                       
            lw = 6
            call pgqlw(lw_sav)
            
            call pgsvp(winxmin, winxmax, winymin, winymax)
            
            ! title
            call pgsch(label_scale*1.2)
            call pgmtxt('T',1.5,0.5,0.5,'Dynamo')
         
            call pgsch(label_scale)
            write(str,'(i9)') s% model_number
            call pgmtxt('T',1.8,0.9,0.5,str)
            
            ! xlabel
            call pgsci(1)
            call pgsch(label_scale)
            call show_xaxis_by(s,Dyn_xaxis_by,ierr)
            if (ierr /= 0) return

            
            ! left axis
            
            default_ymax_left = 10
            default_ymin_left = -2
         
            yvec(1:nz) = safe_log10(s% dynamo_B_phi(1:nz))
            yvec2(1:nz) = safe_log10(s% dynamo_B_r(1:nz))

            if (Dyn_ymax_left > -100) then
               ymax = Dyn_ymax_left
            else
               ymax = max(default_ymax_left,maxval(yvec(grid_min:grid_max)))
               ymax2 = max(default_ymax_left,maxval(yvec2(grid_min:grid_max)))
               if (ymax2 > ymax) ymax = ymax2
            end if
            
            if (Dyn_ymin_left > -100) then
               ymin = Dyn_ymin_left
            else
               ymin = max(default_ymin_left,minval(yvec(grid_min:grid_max)))
               ymin2 = max(default_ymin_left,minval(yvec2(grid_min:grid_max)))
               if (ymin2 < ymin) ymin = ymin2
            end if
            
            dy = ymax-ymin
            if (Dyn_dymin_left > -100) dy = Dyn_dymin_left
            
            ymax = ymax + 0.1*dy
            ymin = ymin - 0.1*dy

            call pgswin(xleft, xright, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(label_scale)
            call pgbox('',0.0,0,'BNSTV',0.0,0)

            call pgsci(clr_Teal)
            call pgsch(label_scale)
            call pgmtxt('L',3.6,0.5,0.5,'log B\dphi\u (Gauss)')
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgsci(clr_Coral)
            call pgmtxt('L',5.3,0.5,0.5,'log B\dr\u (Gauss)')
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec2(grid_min:grid_max))
            call pgslw(lw_sav)
            
            
            ! right axis

            lw = 8
            
            default_ymax_right = 0
            default_ymin_right = -10
         
            ! log omega
            yvec(1:nz) = safe_log10(s% omega(1:nz))
            yvec2(1:nz) = safe_log10(s% j_rot(1:nz)) - 20d0
            
            if (Dyn_ymax_right > -100) then
               ymax = Dyn_ymax_right
            else
               ymax = max(default_ymax_right,maxval(yvec(grid_min:grid_max)))
               ymax2 = max(default_ymax_right,maxval(yvec2(grid_min:grid_max)))
               if (ymax2 > ymax) ymax = ymax2
            end if
            if (Dyn_ymin_right > -100) then
               ymin = Dyn_ymin_right
            else
               ymin = max(default_ymin_right,minval(yvec(grid_min:grid_max)))
               ymin2 = max(default_ymin_right,minval(yvec2(grid_min:grid_max)))
               if (ymin2 < ymin) ymin = ymin2
            end if
            
            dy = ymax-ymin
            if (Dyn_dymin_right > -100) dy = Dyn_dymin_right
            
            ymax = ymax + 0.1*dy
            ymin = ymin - 0.1*dy
         
            call pgswin(xleft, xright, ymin, ymax)

            call pgscf(1)
            call pgsci(1)
            call pgsch(label_scale)
            call pgbox('BCNST',0.0,0,'CMSTV',0.0,0)
            
            call pgsci(clr_FireBrick)
            call pgmtxt('R',5.6,0.5,0.5,'log specific angular momentum (10\u20\d cm\u2\d/s)')
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec2(grid_min:grid_max))
            call pgslw(lw_sav)
            
            call pgsci(clr_RoyalBlue)
            call pgmtxt('R',3.9,0.5,0.5,'log angular velocity (rad/s)')
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            
            ! show mix regions at bottom of plot
            ybot = -0.05
            call pgswin(xleft, xright, ybot, 0.85)
            eps = 0.001
            ybot = ybot + eps
            call pgslw(10)
            call show_mix_regions_on_xaxis(s,ybot,grid_min,grid_max,xvec)


            call pgunsa
            
         end subroutine plot
      
         
      end subroutine Dyn_plot
      

      end module mod_pgstar_dynamo

