! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_cntr_hist

      use star_private_def
      use const_def
      use mod_pgstar_support

      implicit none


      contains
      

      subroutine cntr_hist_plot(id, device_id, ierr)
         use utils_lib
         use chem_def
         use num_lib, only: safe_log10
         use net_def
         use const_def, only: Msun, Rsun
         implicit none

         integer, intent(in) :: id, device_id
         integer, intent(out) :: ierr
         
         real :: xleft, xright, ybot, ytop, txt_scale
         type (star_info), pointer :: s
         
         character (len=256) :: str
         real, pointer, dimension(:) :: xvec, yvec
         real :: xmin, xmax, chScale, winxmin, winxmax, winymin, winymax, windy, &
            ymin, ymax, dy, &
            header_frac, footer_frac, plot2_frac, plot3_frac, plot4_frac, plot1_frac
         integer :: grid_min, grid_max, numpts, i, cv_cur
         type (pgstar_data), dimension(:), pointer :: cv
         
         integer, parameter :: max_num_labels = 20
         integer :: num_labels, iloc_abundance_label(max_num_labels)
         integer, parameter :: num_colors = 8
         integer :: colors(num_colors)
         
         include 'formats.dek'
         
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         cv_cur = cv_current(id)
         cv => cv_for_star(id)% cv_data

         if (s% Cntr_Hist_xmin >= cv(cv_cur)% step) s% Cntr_Hist_xmin = -1
         xmin = min(cv(cv_cur)% step, max(cv(1)% step, s% Cntr_Hist_xmin))
         xmax = s% Cntr_Hist_xmax
         if (xmax <= xmin) xmax = cv(cv_cur)% step
         if (s% Cntr_Hist_max_width > 0) xmin = max(xmin, xmax - s% Cntr_Hist_max_width)
         if (xmax <= xmin) xmax = cv(cv_cur)% step
         if (xmax <= xmin) return
         
         colors(:) = (/ &
               clr_MediumSlateBlue, clr_LightSkyBlue, clr_Goldenrod, clr_Lilac, &
               clr_Coral, clr_Crimson, clr_LightSkyGreen, clr_DarkGray /)

         
         call pgslct(device_id)
         call pgbbuf()
         call pgeras()

         allocate (xvec(cv_cur), yvec(cv_cur))

         chScale = 1.0
         winxmin = 0.15
         winxmax = 0.80
         winymin = 0.02
         winymax = 0.93
         windy = winymax - winymin
         
         header_frac = 0.02
         footer_frac = 0.08
         plot2_frac = (1 - (header_frac + footer_frac))/4
         plot3_frac = plot2_frac
         plot4_frac = plot2_frac
         plot1_frac = plot2_frac
         
         call set_grid_min_max(s, xmin, xmax, grid_min, grid_max, numpts, ierr)
         if (ierr /= 0) return      

         forall (i=grid_min:grid_max) xvec(i) = cv(i)% step       
         
         call plot2
         call plot3
         call plot4
         call plot1
         call header
         call footer
         
         call show_annotations(s, &
            s% show_Cntr_Hist_annotation1, s% show_Cntr_Hist_annotation2, s% show_Cntr_Hist_annotation3)

         call pgebuf()

         deallocate(xvec, yvec)


         contains
                  
         
         subroutine plot1
            integer :: lw, lw_sav
            logical, parameter :: dbg = .false.
         
            include 'formats.dek'
            call pgsave
                       
            lw = 4
            call pgqlw(lw_sav)
            
            call set_plot1_vp         
            call pgsch(chScale*1.2)
            call pgmtxt('T',1.5,0.5,0.5,'Central History')
         
            call pgsch(chScale*0.9)
            write(str,'(i9)') s% model_number
            call pgmtxt('T',1.5*1.2/0.9,0.9,0.5,str)
         
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% log_center_T
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))
            dy = max(ymax-ymin, 0.08)
            ymax = ymax + 0.1*dy
            ymin = ymin - 0.1*dy
            ymax = ymax + dy*0.1
         
            call pgswin(xmin, xmax, ymin, ymax)

            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('BCST',0.0,0,'CMSTV',0.0,0)
            call pgsci(clr_Goldenrod)
            call pgmtxt('R',4.75,0.5,0.5,'log T')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% log_center_Rho
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))        
            ymax = max(ymax, ymin+1e-6)
            dy = max(ymax-ymin, 0.08)
            ymin = ymin-dy*0.1
            ymax = ymax + dy*0.1

            call pgswin(xmin, xmax, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('',0.0,0,'BNSTV',0.0,0)
            call pgsci(clr_Lilac)
            call pgmtxt('L',4.0,0.5,0.5,'log Rho')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgunsa
         end subroutine plot1


         subroutine plot2
            integer :: lw, lw_sav
            logical, parameter :: dbg = .false.
         
            include 'formats.dek'
            call pgsave
                       
            lw = 4
            call pgqlw(lw_sav)
            
            call set_plot2_vp         
            call pgsch(chScale*1.2)
         
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_entropy
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))
            dy = max(ymax-ymin, 0.08)
            ymax = ymax + 0.1*dy
            ymin = ymin - 0.1*dy
            ymax = ymax + dy*0.1
         
            call pgswin(xmin, xmax, ymin, ymax)

            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('BCST',0.0,0,'CMSTV',0.0,0)
            call pgsci(clr_Crimson)
            call pgmtxt('R',4.75,0.5,0.5,'entropy')
            call pgmtxt('R',6.00,0.5,0.5,'(k\dB\u/m\dp\u)')

            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% log_center_P
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))        
            ymax = max(ymax, ymin+1e-6)
            dy = max(ymax-ymin, 0.08)
            ymin = ymin - dy*0.1
            ymax = ymax + dy*0.1

            call pgswin(xmin, xmax, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('',0.0,0,'BNSTV',0.0,0)
            call pgsci(clr_LightSkyGreen)
            call pgmtxt('L',4.0,0.5,0.5,'log P')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgunsa
         end subroutine plot2
                  
         
         subroutine plot3
            integer :: lw, lw_sav
            logical, parameter :: dbg = .false.
         
            include 'formats.dek'
            call pgsave
                       
            lw = 4
            call pgqlw(lw_sav)
            
            call set_plot3_vp         
         
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_gamma
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))
            dy = max(ymax-ymin, 0.08)
            ymax = ymax + 0.1*dy
            ymin = ymin - 0.1*dy
            ymax = ymax + dy*0.1
         
            call pgswin(xmin, xmax, ymin, ymax)

            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('BCST',0.0,0,'CMSTV',0.0,0)
            call pgsci(clr_Coral)
            call pgmtxt('R',4.75,0.5,0.5,'plasma')
            call pgmtxt('R',6.00,0.5,0.5,'gamma')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_degeneracy
            ymax = maxval(yvec(grid_min:grid_max))
            ymin = minval(yvec(grid_min:grid_max))        
            ymax = max(ymax, ymin+1e-6)
            dy = max(ymax-ymin, 0.08)
            ymin = ymin-dy*0.1
            ymax = ymax + dy*0.1

            call pgswin(xmin, xmax, ymin, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgbox('',0.0,0,'BNSTV',0.0,0)
            call pgsci(clr_LightSkyBlue)
            call pgmtxt('L',4.0,0.5,0.5,'eta')
            call pgslw(lw)
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)

            call pgunsa
         end subroutine plot3


         subroutine plot4
            
            integer :: i, j, k, k0, cnt
            real :: x, dx, dy, ymin_limit
            
            include 'formats.dek'
            call pgsave

            call set_plot4_vp
            
            dy = s% Cntr_Hist_mass_frac_max - s% Cntr_Hist_mass_frac_min
            ymin_limit = s% Cntr_Hist_mass_frac_min + 0.05*dy
            
            !  abundances
            call pgsci(1)
            call pgswin(xmin, xmax, s% Cntr_Hist_mass_frac_min, s% Cntr_Hist_mass_frac_max)
            call pgsch(chScale*1.2)
            call pgbox('BCNST1',0.0,0,'BCNMSTV1',0.0,0)
            call pgmtxt('L',4.0,0.5,0.5,'mass frac')
            call pgmtxt('R',4.75,0.5,0.5,'mass frac')
            call pgmtxt('B',3.0,0.5,0.5,'model number')
         
            call pgsch(chScale*0.8)
            j = 0
            
            dx = xmax - xmin
            num_labels = max(0,min(max_num_labels, s% num_abundance_line_labels))
            
            if (.false.) then
               write(*,2) 'num_abundance_line_labels', s% num_abundance_line_labels
               write(*,2) 'max_num_labels', max_num_labels
               write(*,2) 'num_labels', num_labels
               write(*,2) 'grid_min', grid_min
               write(*,2) 'grid_max', grid_max
               write(*,2) 'xmax', int(xmax), xmax
               write(*,2) 'xmin', int(xmin), xmin
               write(*,2) 'dx', int(dx), dx
               write(*,*)
            end if
            
            do i=1,num_labels
               x = xmin + (i-0.5)*dx/num_labels
               if (i == 1) then
                  k0 = grid_min+1
               else
                  k0 = iloc_abundance_label(i-1)+1
               end if
               do k=k0,grid_max
                  if ((xvec(k-1)-x)*(x-xvec(k)) >= 0) then
                     iloc_abundance_label(i) = k
                     !write(*,3) 'label', k, int(x), xvec(k-1), x, xvec(k)
                     exit
                  end if
               end do
            end do

            call pgsch(chScale)
            
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_h1
            call do1_abundance(cnt,'h1',ymin_limit)
 
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_he4
            call do1_abundance(cnt,'he4',ymin_limit)
 
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_c12
            call do1_abundance(cnt,'c12',ymin_limit)
 
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_n14
            call do1_abundance(cnt,'n14',ymin_limit)
 
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_o16
            call do1_abundance(cnt,'o16',ymin_limit)
 
            yvec(grid_min:grid_max) = cv(grid_min:grid_max)% center_ne20
            call do1_abundance(cnt,'ne20',ymin_limit)

            call pgunsa
            
         end subroutine plot4
         
         
         subroutine do1_abundance(j,label,ymin_limit)
            integer, intent(inout) :: j
            character (len=*), intent(in) :: label
            real, intent(in) :: ymin_limit
            integer :: i, ii
            include 'formats.dek'
            j = j+1
            if (maxval(yvec(grid_min:grid_max)) < ymin_limit) return
            call pgsci(colors(1 + mod(j,num_colors)))
            call pgline(numpts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            !write(*,3) 'do1_abundance', j, num_labels
            do i=1,num_labels
               ii = iloc_abundance_label(i)
               if (ii > grid_max .or. ii < grid_min) cycle
               if (xvec(ii) > xmin .and. xvec(ii) < xmax) &
                  call pgptxt(xvec(ii), yvec(ii), 0.0, 0.5, label)
            end do
         end subroutine do1_abundance


         subroutine header
         end subroutine header


         subroutine footer
         end subroutine footer


         subroutine set_header_vp
            real :: ymax, ymin
            ymax = winymax
            ymin = ymax - header_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_header_vp


         subroutine set_plot1_vp
            real :: ymax, ymin
            ymax = winymax - header_frac*windy
            ymin = ymax - plot1_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot1_vp


         subroutine set_plot2_vp
            real :: ymax, ymin
            ymax = winymax - (header_frac + plot1_frac)*windy
            ymin = ymax - plot2_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot2_vp


         subroutine set_plot3_vp
            real :: ymax, ymin
            ymax = winymax - (header_frac + plot1_frac + plot2_frac)*windy
            ymin = ymax - plot3_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot3_vp


         subroutine set_plot4_vp
            real :: ymax, ymin
            ymax = winymax - (header_frac + plot1_frac + plot2_frac + plot3_frac)*windy
            ymin = ymax - plot4_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_plot4_vp


         subroutine set_footer_vp
            real :: ymax, ymin
            ymin = winymin
            ymax = ymin + footer_frac*windy
            call pgsvp(winxmin, winxmax, ymin, ymax)
         end subroutine set_footer_vp
         

      end subroutine cntr_hist_plot


      end module mod_pgstar_cntr_hist

