! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar_abundance

      use star_private_def
      use const_def
      use mod_pgstar_support

      implicit none


      contains
      

      subroutine abundance_plot(id, device_id, ierr)
         implicit none
         integer, intent(in) :: id, device_id
         integer, intent(out) :: ierr

         real :: winxmin, winxmax, winymin, winymax, label_scale
         type (star_info), pointer :: s

         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call pgslct(device_id)
         call pgbbuf()
         call pgeras()

         label_scale = 1.0         
         winxmin = 0.15
         winxmax = 0.80
         winymin = 0.09
         winymax = 0.93
         
         call do_abundance_plot(id, device_id, &
            winxmin, winxmax, winymin, winymax, &
            s% Abundance_xaxis_by, s% Abundance_xmin, s% Abundance_xmax, &
            label_scale, ierr)

         call pgebuf()
         
      end subroutine abundance_plot
      

      subroutine do_abundance_plot(id, device_id, &
            winxmin, winxmax, winymin, winymax, &
            xaxis_by, xaxis_min, xaxis_max, label_scale, ierr)
         use utils_lib
         use chem_def
         use num_lib, only: safe_log10
         use net_def
         use const_def, only: Msun, Rsun
         implicit none

         integer, intent(in) :: id, device_id
         real, intent(in) :: winxmin, winxmax, winymin, winymax
         character (len=256) :: xaxis_by
         real, intent(in) :: xaxis_min, xaxis_max, label_scale
         integer, intent(out) :: ierr
         
         character (len=256) :: str
         real, pointer, dimension(:) :: xvec, yvec
         real :: xmin, xmax, xleft, xright, dx, dylbl, chScale, windy, &
            ymin, ymax, legend_xmin, legend_xmax, xmargin
         integer :: lw, lw_sav, grid_min, grid_max, npts, i, nz
         integer, parameter :: num_colors = 15
         integer :: colors(num_colors)
         integer, parameter :: max_num_labels = 20
         integer :: num_labels, iloc_abundance_label(max_num_labels)
         type (star_info), pointer :: s
         
         include 'formats.dek'
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         nz = s% nz        
         
         colors(:) = (/ &
               clr_MediumSlateBlue, clr_LightSkyBlue, clr_Goldenrod, clr_Lilac, &
               clr_Coral, clr_Crimson, clr_LightSkyGreen, clr_DarkGray, &
               clr_Tan, clr_IndianRed, clr_Gold, &
               clr_Teal, clr_Silver, clr_BrightBlue, &
               clr_RoyalPurple /)

         chScale = label_scale

         windy = winymax - winymin
         
         legend_xmin = 0.84
         legend_xmax = 0.99
         
         allocate (xvec(nz), yvec(nz))
         
         xmargin = 0
         call set_xaxis_bounds(s, xaxis_by, xaxis_min, xaxis_max, .false., &
            xmargin, xvec, xmin, xmax, xleft, xright, dx, &
            grid_min, grid_max, npts, ierr)
         
         if (ierr /= 0) then
            ! skip the plotting
            ierr = 0
            call pgsvp(winxmin, winxmax, winymin, winymax)
            call pgsch(chScale*1.2)
            call pgmtxt('T',-5.0,0.5,0.5,'Abundance plot')
            call pgmtxt('T',-7.0,0.5,0.5,'Bad Xmin and/or Xmax specification')
            call pgmtxt('T',-9.0,0.5,0.5,'please check inlist')
         else
            call plot(ierr)
            if (ierr /= 0) return
         end if         

         deallocate(xvec, yvec)

         if (s% show_Abundance_cross_hair) call do_cross_hair

         contains
                  
         
         subroutine plot(ierr)
            use rates_def
            integer, intent(out) :: ierr
            
            integer :: ii, jj, i, k
            logical, parameter :: dbg = .false.
            real :: lgz, x, ybot
         
            include 'formats.dek'
            ierr = 0
            ymin = s% abundance_log_mass_frac_min
            if (ymin > 0) then
               ymin = -5.1
               lgz = log10(s% initial_z + 1e-9) - 1
               if (lgz-1 < ymin) ymin = lgz
            end if
            ymax = s% abundance_log_mass_frac_max
            if (ymax >= 100) then
               if (ymin < -8) then
                  ymax = 0.5
               else
                  ymax = 0.25
               end if
            end if

            num_labels = max(0,min(max_num_labels, s% num_abundance_line_labels))
            do i=1,num_labels
               x = xmin + (i-0.5)*dx/num_labels
               do k=2,nz
                  if ((xvec(k-1)-x)*(x-xvec(k)) >= 0) then
                     iloc_abundance_label(i) = k
                     exit
                  end if
               end do
            end do
            
            dylbl = (ymax - ymin)*0.015

            lw = 6
            call pgqlw(lw_sav)

            call pgsave
            call pgsvp(legend_xmin, legend_xmax, winymin, winymax)
            call pgswin(0.0, 1.0, ymin, ymax)
            call do_all(.true.)
            call pgunsa

            call pgsave
                       
            call pgsvp(winxmin, winxmax, winymin, winymax)
         
            write(str,'(i9)') s% model_number
            call pgsch(label_scale)
            call pgmtxt('T',1.5,0.9,0.5,str)
         
            ybot = -0.05
            call pgswin(xleft, xright, ymin+ybot, ymax)
            call pgscf(1)
            call pgsci(1)
            call pgsch(chScale*1.2)
            call pgmtxt('T',1.5,0.5,0.5,'Abundance')
            call pgsch(label_scale)
            call pgbox('BCNST',0.0,0,'BCNSTV',0.0,0)
            call pgmtxt('L',4.0,0.5,0.5,'log mass fraction')
            
            call pgsch(label_scale)
            call pgsave
            call do_all(.false.)
            call pgunsa
            
            call pgsci(1)
            call pgsch(label_scale)
            call show_xaxis_by(s,xaxis_by,ierr)
            if (ierr /= 0) return
            
            ! show mix regions at bottom of plot
            call pgslw(10)
            call show_mix_regions_on_xaxis(s,ymin+ybot,grid_min,grid_max,xvec)
            
            call pgunsa

         end subroutine plot

         
         subroutine do_all(legend_flag)
            logical, intent(in) :: legend_flag
            integer :: cnt, num_to_show, i
            include 'formats.dek'
            cnt = 0
            num_to_show = s% Abundance_num_isos_to_show            
            if (num_to_show < 0) then ! show as many as fit
               do i = 1, min(s% species, max_Abundance_num_isos_to_show)
                  cnt = do1(cnt, chem_isos% name(s% chem_id(i)), legend_flag)
               end do
            else
               do i = 1, num_to_show
                  cnt = do1(cnt, s% Adundance_which_isos_to_show(i), legend_flag)
               end do
            end if
         end subroutine do_all
         
         
         integer function do1(cnt, str, legend_flag)
            use chem_lib
            integer, intent(in) :: cnt
            character (len=*), intent(in) :: str
            logical, intent(in) :: legend_flag
            integer :: i, cid
            include 'formats.dek'
            do1 = cnt
            if (len_trim(str) == 0) return
            cid = chem_get_iso_id(str)
            if (cid <= 0) return
            i = s% net_iso(cid)
            if (i == 0) return
            if (legend_flag) then
               do1 = abundance_line_legend(cnt, str)
            else
               do1 = abundance_line(cnt, i, str)
            end if
         end function do1
         
         
         subroutine set_line_style(cnt)
            integer, intent(in) :: cnt
            integer :: iclr, itype
            iclr = cnt - num_colors*(cnt/num_colors) + 1
            call pgsci(colors(iclr))
            if (cnt > num_colors) then
               itype = Line_Type_Dot
            else
               itype = Line_Type_Solid
            end if
            call pgsls(itype)
         end subroutine set_line_style
         
         
         integer function abundance_line(cnt, j, str)
            integer, intent(in) :: cnt, j
            character (len=*), intent(in) :: str
            integer :: i, ii
            call set_line_style(cnt)
            yvec(1:nz) = safe_log10(s% xa(j,1:nz))
            call pgslw(lw)
            call pgline(npts, xvec(grid_min:grid_max), yvec(grid_min:grid_max))
            call pgslw(lw_sav)
            abundance_line = cnt + 1
            
            do i=1,num_labels
               ii = iloc_abundance_label(i)
               if (ii > grid_max .or. ii < grid_min) cycle
               if (xvec(ii) > xmin .and. xvec(ii) < xmax .and. &
                   yvec(ii) > ymin .and. yvec(ii) < ymax) &
                  call pgptxt(xvec(ii), yvec(ii)+dylbl, 0.0, 0.5, str)
            end do

         end function abundance_line
         
         
         integer function abundance_line_legend(cnt, str)
            integer, intent(in) :: cnt
            character (len=*), intent(in) :: str
            real :: ymx, dx, dyline, ypos, xpts(2), ypts(2)
            integer :: iclr
            integer, parameter :: max_cnt = 25
            if (cnt >= max_cnt) return
            call set_line_style(cnt)
            dx = 0.1
            dyline = (ymax-ymin)/max_cnt
            ypos = ymax - (cnt+0.5)*dyline
            xpts(1) = 2*dx
            xpts(2) = xpts(1) + 3*dx
            ypts = ypos + dyline*0.1
            call pgslw(lw)
            call pgline(2, xpts, ypts)
            call pgslw(lw_sav)
            call pgsci(1)
            call pgsch(label_scale*0.75)
            call pgptxt(xpts(2) + dx, ypos, 0.0, 0.0, trim(str))
            abundance_line_legend = cnt + 1
         end function abundance_line_legend
         

      end subroutine do_abundance_plot


      end module mod_pgstar_abundance

