! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_pgstar

      use star_private_def
      use const_def
      use chem_def, only: category_name
      use rates_def, only: i_rate
      use mod_pgstar_support

      implicit none
      

      contains
      
      
      subroutine do_start_new_run_for_pgstar(s, ierr) ! reset logs
         use utils_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         integer :: iounit
         character (len=256) :: fname
         logical :: fexist
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            write(*,*) 'do_start_new_run_for_pgstar: no iounits'
            return
         end if
         fname = trim(s% photo_directory) // '/pgstar.dat'
         inquire(file=trim(fname), exist=fexist)
         if (fexist) then
            open(unit=iounit, file=trim(fname), status='replace', action='write')
            close(iounit)
         end if
         call free_iounit(iounit)
         call pgstar_clear(s)
      end subroutine do_start_new_run_for_pgstar
      
      
      subroutine do_restart_run_for_pgstar(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         ierr = 0
         call pgstar_clear(s)
         cv_current(s% id) = read_pgstar_data(s,ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in read_pgstar_data'
         end if
      end subroutine do_restart_run_for_pgstar
      

      subroutine do_read_pgstar_controls(s, inlist_fname, ierr)
         use pgstar_ctrls_io, only: read_pgstar
         type (star_info), pointer :: s
         character(*), intent(in) :: inlist_fname
         integer, intent(out) :: ierr
         ierr = 0
         !if (have_initialized_pgstar .and. s% pgstar_cnt > 0 .and. &
         !      mod(s% model_number, s% pgstar_cnt) /= 0) return
         call read_pgstar(s, inlist_fname, ierr)
         if (ierr /= 0) then
            write(*,*) 'PGSTAR failed in reading ' // trim(inlist_fname)
            return
         end if
         call set_win_file_data(s, ierr)
      end subroutine do_read_pgstar_controls
      
      
      subroutine set_win_file_data(s, ierr)
         use mod_pgstar_conv, only: Convection_Plot
         use mod_pgstar_summary, only: Summary_Plot
         use mod_pgstar_logg_Teff, only: loggTe_Plot
         use mod_pgstar_hr, only: HR_Plot
         use mod_pgstar_trho, only: TRho_Plot
         use mod_pgstar_hr_trho, only: HR_TRho_Plot
         use mod_pgstar_profile, only: Profile_plot, Profile2_plot, Profile3_plot
         use mod_pgstar_dynamo, only: Dynamo_plot
         use mod_pgstar_mixing_Ds, only: Mixing_Ds_plot
         use mod_pgstar_mixing_dynamo, only: Mixing_Dynamo_plot
         use mod_pgstar_trho_profile, only: TRho_Profile_plot
         use mod_pgstar_cntr_hist, only: cntr_hist_plot
         use mod_pgstar_surf_hist, only: surf_hist_plot
         use mod_pgstar_power, only: power_plot
         use mod_pgstar_mode_prop, only: mode_propagation_plot
         use mod_pgstar_abundance, only: abundance_plot
         use mod_pgstar_abund_power, only: Abundance_Power_plot
         use mod_pgstar_abund_pwr_dmix, only: Abund_Pwr_Dmix_plot
         use mod_pgstar_abund_pwr_dmix_dyn, only: Abund_Pwr_Dmix_Dyn_plot

         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         type (pgstar_win_file_data), pointer :: p
         integer :: i
         
         ! store win and file info in records
         
         p => s% pgstar_win_file_ptr(i_main)
         p% plot => Summary_Plot
         p% id = i_main
         p% name = 'MAIN'
         p% win_flag = s% MAIN_win_flag
         p% win_width = s% MAIN_win_width
         p% win_aspect_ratio = s% MAIN_win_aspect_ratio
         p% file_flag = s% MAIN_file_flag
         p% file_dir = s% MAIN_file_dir
         p% file_prefix = s% MAIN_file_prefix
         p% file_cnt = s% MAIN_file_cnt
         p% file_width = s% MAIN_file_width
         p% file_aspect_ratio = s% MAIN_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_TRho_Profile)
         p% plot => TRho_Profile_plot
         p% id = i_TRho_Profile
         p% name = 'TRho_Profile'
         p% win_flag = s% TRho_Profile_win_flag
         p% win_width = s% TRho_Profile_win_width
         p% win_aspect_ratio = s% TRho_Profile_win_aspect_ratio
         p% file_flag = s% TRho_Profile_file_flag
         p% file_dir = s% TRho_Profile_file_dir
         p% file_prefix = s% TRho_Profile_file_prefix
         p% file_cnt = s% TRho_Profile_file_cnt
         p% file_width = s% TRho_Profile_file_width
         p% file_aspect_ratio = s% TRho_Profile_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Profile)
         p% plot => Profile_plot
         p% id = i_Profile
         p% name = 'Profile'
         p% win_flag = s% Profile_win_flag
         p% win_width = s% Profile_win_width
         p% win_aspect_ratio = s% Profile_win_aspect_ratio
         p% file_flag = s% Profile_file_flag
         p% file_dir = s% Profile_file_dir
         p% file_prefix = s% Profile_file_prefix
         p% file_cnt = s% Profile_file_cnt
         p% file_width = s% Profile_file_width
         p% file_aspect_ratio = s% Profile_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Profile2)
         p% plot => Profile2_plot
         p% id = i_Profile2
         p% name = 'Profile2'
         p% win_flag = s% Profile2_win_flag
         p% win_width = s% Profile2_win_width
         p% win_aspect_ratio = s% Profile2_win_aspect_ratio
         p% file_flag = s% Profile2_file_flag
         p% file_dir = s% Profile2_file_dir
         p% file_prefix = s% Profile2_file_prefix
         p% file_cnt = s% Profile2_file_cnt
         p% file_width = s% Profile2_file_width
         p% file_aspect_ratio = s% Profile2_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Profile3)
         p% plot => Profile3_plot
         p% id = i_Profile3
         p% name = 'Profile3'
         p% win_flag = s% Profile3_win_flag
         p% win_width = s% Profile3_win_width
         p% win_aspect_ratio = s% Profile3_win_aspect_ratio
         p% file_flag = s% Profile3_file_flag
         p% file_dir = s% Profile3_file_dir
         p% file_prefix = s% Profile3_file_prefix
         p% file_cnt = s% Profile3_file_cnt
         p% file_width = s% Profile3_file_width
         p% file_aspect_ratio = s% Profile3_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_loggTe)
         p% plot => loggTe_Plot
         p% id = i_loggTe
         p% name = 'loggTe'
         p% win_flag = s% loggTe_win_flag
         p% win_width = s% loggTe_win_width
         p% win_aspect_ratio = s% loggTe_win_aspect_ratio
         p% file_flag = s% loggTe_file_flag
         p% file_dir = s% loggTe_file_dir
         p% file_prefix = s% loggTe_file_prefix
         p% file_cnt = s% loggTe_file_cnt
         p% file_width = s% loggTe_file_width
         p% file_aspect_ratio = s% loggTe_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_HR)
         p% plot => HR_Plot
         p% id = i_HR
         p% name = 'HR'
         p% win_flag = s% HR_win_flag
         p% win_width = s% HR_win_width
         p% win_aspect_ratio = s% HR_win_aspect_ratio
         p% file_flag = s% HR_file_flag
         p% file_dir = s% HR_file_dir
         p% file_prefix = s% HR_file_prefix
         p% file_cnt = s% HR_file_cnt
         p% file_width = s% HR_file_width
         p% file_aspect_ratio = s% HR_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_TRho)
         p% plot => TRho_Plot
         p% id = i_TRho
         p% name = 'TRho'
         p% win_flag = s% TRho_win_flag
         p% win_width = s% TRho_win_width
         p% win_aspect_ratio = s% TRho_win_aspect_ratio
         p% file_flag = s% TRho_file_flag
         p% file_dir = s% TRho_file_dir
         p% file_prefix = s% TRho_file_prefix
         p% file_cnt = s% TRho_file_cnt
         p% file_width = s% TRho_file_width
         p% file_aspect_ratio = s% TRho_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_HR_TRho)
         p% plot => HR_TRho_Plot
         p% id = i_HR_TRho
         p% name = 'HR_TRho'
         p% win_flag = s% HR_TRho_win_flag
         p% win_width = s% HR_TRho_win_width
         p% win_aspect_ratio = s% HR_TRho_win_aspect_ratio
         p% file_flag = s% HR_TRho_file_flag
         p% file_dir = s% HR_TRho_file_dir
         p% file_prefix = s% HR_TRho_file_prefix
         p% file_cnt = s% HR_TRho_file_cnt
         p% file_width = s% HR_TRho_file_width
         p% file_aspect_ratio = s% HR_TRho_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Dynamo)
         p% plot => Dynamo_plot
         p% id = i_Dynamo
         p% name = 'Dynamo'
         p% win_flag = s% Dynamo_win_flag
         p% win_width = s% Dynamo_win_width
         p% win_aspect_ratio = s% Dynamo_win_aspect_ratio
         p% file_flag = s% Dynamo_file_flag
         p% file_dir = s% Dynamo_file_dir
         p% file_prefix = s% Dynamo_file_prefix
         p% file_cnt = s% Dynamo_file_cnt
         p% file_width = s% Dynamo_file_width
         p% file_aspect_ratio = s% Dynamo_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Mixing_Ds)
         p% plot => Mixing_Ds_plot
         p% id = i_Mixing_Ds
         p% name = 'Mixing_Ds'
         p% win_flag = s% Mixing_Ds_win_flag
         p% win_width = s% Mixing_Ds_win_width
         p% win_aspect_ratio = s% Mixing_Ds_win_aspect_ratio
         p% file_flag = s% Mixing_Ds_file_flag
         p% file_dir = s% Mixing_Ds_file_dir
         p% file_prefix = s% Mixing_Ds_file_prefix
         p% file_cnt = s% Mixing_Ds_file_cnt
         p% file_width = s% Mixing_Ds_file_width
         p% file_aspect_ratio = s% Mixing_Ds_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Mixing_Dynamo)
         p% plot => Mixing_Dynamo_Plot
         p% id = i_Mixing_Dynamo
         p% name = 'Mixing_Dynamo'
         p% win_flag = s% Mixing_Dynamo_win_flag
         p% win_width = s% Mixing_Dynamo_win_width
         p% win_aspect_ratio = s% Mixing_Dynamo_win_aspect_ratio
         p% file_flag = s% Mixing_Dynamo_file_flag
         p% file_dir = s% Mixing_Dynamo_file_dir
         p% file_prefix = s% Mixing_Dynamo_file_prefix
         p% file_cnt = s% Mixing_Dynamo_file_cnt
         p% file_width = s% Mixing_Dynamo_file_width
         p% file_aspect_ratio = s% Mixing_Dynamo_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_CONV)
         p% plot => Convection_Plot
         p% id = i_CONV
         p% name = 'CONV'
         p% win_flag = s% CONV_win_flag
         p% win_width = s% CONV_win_width
         p% win_aspect_ratio = s% CONV_win_aspect_ratio
         p% file_flag = s% CONV_file_flag
         p% file_dir = s% CONV_file_dir
         p% file_prefix = s% CONV_file_prefix
         p% file_cnt = s% CONV_file_cnt
         p% file_width = s% CONV_file_width
         p% file_aspect_ratio = s% CONV_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Cntr_Hist)
         p% plot => cntr_hist_plot
         p% id = i_Cntr_Hist
         p% name = 'Cntr_Hist'
         p% win_flag = s% Cntr_Hist_win_flag
         p% win_width = s% Cntr_Hist_win_width
         p% win_aspect_ratio = s% Cntr_Hist_win_aspect_ratio
         p% file_flag = s% Cntr_Hist_file_flag
         p% file_dir = s% Cntr_Hist_file_dir
         p% file_prefix = s% Cntr_Hist_file_prefix
         p% file_cnt = s% Cntr_Hist_file_cnt
         p% file_width = s% Cntr_Hist_file_width
         p% file_aspect_ratio = s% Cntr_Hist_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Surf_Hist)
         p% plot => surf_hist_plot
         p% id = i_Surf_Hist
         p% name = 'Surf_Hist'
         p% win_flag = s% Surf_Hist_win_flag
         p% win_width = s% Surf_Hist_win_width
         p% win_aspect_ratio = s% Surf_Hist_win_aspect_ratio
         p% file_flag = s% Surf_Hist_file_flag
         p% file_dir = s% Surf_Hist_file_dir
         p% file_prefix = s% Surf_Hist_file_prefix
         p% file_cnt = s% Surf_Hist_file_cnt
         p% file_width = s% Surf_Hist_file_width
         p% file_aspect_ratio = s% Surf_Hist_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Mode_Prop)
         p% plot => mode_propagation_plot
         p% id = i_Mode_Prop
         p% name = 'Mode_Propagation'
         p% win_flag = s% Mode_Prop_win_flag
         p% win_width = s% Mode_Prop_win_width
         p% win_aspect_ratio = s% Mode_Prop_win_aspect_ratio
         p% file_flag = s% Mode_Prop_file_flag
         p% file_dir = s% Mode_Prop_file_dir
         p% file_prefix = s% Mode_Prop_file_prefix
         p% file_cnt = s% Mode_Prop_file_cnt
         p% file_width = s% Mode_Prop_file_width
         p% file_aspect_ratio = s% Mode_Prop_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Power)
         p% plot => power_plot
         p% id = i_Power
         p% name = 'Power'
         p% win_flag = s% Power_win_flag
         p% win_width = s% Power_win_width
         p% win_aspect_ratio = s% Power_win_aspect_ratio
         p% file_flag = s% Power_file_flag
         p% file_dir = s% Power_file_dir
         p% file_prefix = s% Power_file_prefix
         p% file_cnt = s% Power_file_cnt
         p% file_width = s% Power_file_width
         p% file_aspect_ratio = s% Power_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Abundance)
         p% plot => abundance_plot
         p% id = i_Abundance
         p% name = 'Abundance'
         p% win_flag = s% Abundance_win_flag
         p% win_width = s% Abundance_win_width
         p% win_aspect_ratio = s% Abundance_win_aspect_ratio
         p% file_flag = s% Abundance_file_flag
         p% file_dir = s% Abundance_file_dir
         p% file_prefix = s% Abundance_file_prefix
         p% file_cnt = s% Abundance_file_cnt
         p% file_width = s% Abundance_file_width
         p% file_aspect_ratio = s% Abundance_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Abund_Power)
         p% plot => Abundance_Power_plot
         p% id = i_Abund_Power
         p% name = 'Abundance_Power'
         p% win_flag = s% Abund_Power_win_flag
         p% win_width = s% Abund_Power_win_width
         p% win_aspect_ratio = s% Abund_Power_win_aspect_ratio
         p% file_flag = s% Abund_Power_file_flag
         p% file_dir = s% Abund_Power_file_dir
         p% file_prefix = s% Abund_Power_file_prefix
         p% file_cnt = s% Abund_Power_file_cnt
         p% file_width = s% Abund_Power_file_width
         p% file_aspect_ratio = s% Abund_Power_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Abund_Pwr_Dmix)
         p% plot => Abund_Pwr_Dmix_plot
         p% id = i_Abund_Pwr_Dmix
         p% name = 'Abund_Power_Dmix'
         p% win_flag = s% Abund_Pwr_Dmix_win_flag
         p% win_width = s% Abund_Pwr_Dmix_win_width
         p% win_aspect_ratio = s% Abund_Pwr_Dmix_win_aspect_ratio
         p% file_flag = s% Abund_Pwr_Dmix_file_flag
         p% file_dir = s% Abund_Pwr_Dmix_file_dir
         p% file_prefix = s% Abund_Pwr_Dmix_file_prefix
         p% file_cnt = s% Abund_Pwr_Dmix_file_cnt
         p% file_width = s% Abund_Pwr_Dmix_file_width
         p% file_aspect_ratio = s% Abund_Pwr_Dmix_file_aspect_ratio

         p => s% pgstar_win_file_ptr(i_Abund_Pwr_Dmix_Dyn)
         p% plot => Abund_Pwr_Dmix_Dyn_plot
         p% id = i_Abund_Pwr_Dmix_Dyn
         p% name = 'Abund_Power_Dmix_Dyn'
         p% win_flag = s% Abund_Pwr_Dmix_Dyn_win_flag
         p% win_width = s% Abund_Pwr_Dmix_Dyn_win_width
         p% win_aspect_ratio = s% Abund_Pwr_Dmix_Dyn_win_aspect_ratio
         p% file_flag = s% Abund_Pwr_Dmix_Dyn_file_flag
         p% file_dir = s% Abund_Pwr_Dmix_Dyn_file_dir
         p% file_prefix = s% Abund_Pwr_Dmix_Dyn_file_prefix
         p% file_cnt = s% Abund_Pwr_Dmix_Dyn_file_cnt
         p% file_width = s% Abund_Pwr_Dmix_Dyn_file_width
         p% file_aspect_ratio = s% Abund_Pwr_Dmix_Dyn_file_aspect_ratio
         
         do i = 1, max_num_Other_plots
            p => s% pgstar_win_file_ptr(i_Other + i - 1)
            p% win_flag = .false.
            p% file_flag = .false.
         end do

         call s% other_pgstar_plots_info(s% id, ierr)
         
      end subroutine set_win_file_data


      subroutine do_pgstar_plots(s, ierr)
         use star_utils, only: update_time, total_times
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: time0, clock_rate
         real(dp) :: total_all_before

         ierr = 0     

         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         if (s% model_number == s% read_data_at_model_num) call read_text_data(s,ierr)
         if (failed('read_text_data')) return
         call update_pgstar_data(s,ierr)
         if (failed('update_pgstar_data')) return
         call onScreen_Plots(s,ierr)
         if (failed('onScreen_Plots')) return
         if (s% model_number == s% save_data_at_model_num) call save_text_data(s,ierr)
         if (failed('save_text_data')) return
         if (mod(s% model_number, s% photostep) == 0) call write_pgstar_data(s,ierr)
         if (failed('write_pgstar_data')) return
         if (s% doing_timing) call update_time(s, time0, total_all_before,  s% time_pgstar)
         if (s% pause) then
            write(*,*)
            write(*,*) 'model_number', s% model_number
            write(*,*) 'PGSTAR: paused -- hit RETURN to continue'
            read(*,*)
         end if
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
            end if
         end function failed
         
      end subroutine do_pgstar_plots


      ! PGSTAR driver, called after each timestep
      subroutine onScreen_Plots(s, ierr)
         use utils_lib
         use chem_def
         use num_lib, only: safe_log10
         use net_def
         use net_lib, only: get_net_reaction_table
         use rates_def, only: rates_reaction_id_max
         use const_def, only: Msun, Rsun
         
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s
         integer :: i
         type (pgstar_win_file_data), pointer :: p
         logical, parameter :: dbg = .false.
         real(dp) :: dlgL, dlgTeff, dHR
         logical :: must_write_files
         
         include 'formats.dek'
         ierr = 0
         
         if (s% delta_HR_limit_for_file_output <= 0 .or. &
               s% L_phot_old <= 0 .or. s% Teff_old <= 0) then
            must_write_files = .false.
         else
            dlgL = log10(s% L_phot/s% L_phot_old)
            dlgTeff = log10(s% Teff/s% Teff_old)
            dHR = sqrt((s% delta_HR_ds_L*dlgL)**2 + (s% delta_HR_ds_Teff*dlgTeff)**2)
            sum_dHR_since_last_file_write = sum_dHR_since_last_file_write + dHR
            must_write_files = &
               (sum_dHR_since_last_file_write >= s% delta_HR_limit_for_file_output)
            if (must_write_files) sum_dHR_since_last_file_write = 0
         end if

         if (have_initialized_pgstar .and. s% pgstar_cnt > 0 .and. &
               mod(s% model_number, s% pgstar_cnt) /= 0) return

         if ( .not. have_initialized_pgstar ) then
            call init_pgstar(s, s% model_number, ierr)
            if (failed('init_pgstar')) return
         else
            call check_windows(s,ierr)
            if (failed('check_windows')) return
            call check_files(s,ierr)
            if (failed('check_files')) return
         end if
         
         do i = 1, num_pgstar_plots
            p => s% pgstar_win_file_ptr(i)
            if (p% do_win .and. &
                  (mod(s% model_number, s% pgstar_cnt)==0 .or. must_write_files)) then
               call p% plot(s% id, p% id_win, ierr)
               if (failed(p% name)) return
            end if
            if (p% do_file .and. mod(s% model_number, p% file_cnt)==0) then
               call p% plot(s% id, p% id_file, ierr)
               if (failed(p% name)) return
               call pgclos
               p% id_file = 0
               p% do_file = .false.
            end if
         end do
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
            end if
         end function failed

      end subroutine onScreen_Plots

      
      
      subroutine read_text_data(s, ierr)
         use utils_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr

         logical :: fexist
         integer :: iounit, i, id
         character (len=1024) :: fname
         type (pgstar_data), pointer :: cv_data
         
         logical, parameter :: dbg = .true.
         
         include 'formats.dek'
         ierr = 0
         id = s% id
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            write(*,*) 'read_text_data: no iounits'
            return
         end if

         fname = s% read_data_filename
         inquire(file=trim(fname), exist=fexist)
         if (.not.fexist) then
            if (dbg) write(*,*) 'failed to find ' // trim(fname)
            call free_iounit(iounit)
            return
         end if
         
         if (associated(cv_for_star(id)% cv_data)) then
            deallocate(cv_for_star(id)% cv_data); nullify(cv_for_star(id)% cv_data)
         end if
         cv_current(id) = 0
         
         write(*,*) 'PGSTAR read_text_data ' // trim(fname)
         open(iounit, file=trim(fname), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then
            if (dbg) write(*,*) 'failed to open ' // trim(fname)
         else
            read(iounit, *, iostat=ierr) cv_current(id)
            if (ierr /= 0) then
               if (dbg) write(*,*) 'failed read cv_current(id) ' // trim(fname)
            else
               do i=1,3 ! skip
                  read(iounit, *, iostat=ierr) 
                  if (ierr /= 0) then
                     write(*,*) 'failed read cv_current(id) ' // trim(fname)
                  end if
               end do
               if (dbg) write(*,2) 'read cv_current(id) ' // trim(fname), cv_current(id)
               allocate(cv_for_star(id)% cv_data(cv_current(id)+1000), stat=ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in allocate for PGSTAR conv data'
                  call free_iounit(iounit)
                  return
               end if
               do i=1,cv_current(id)
                  cv_data => cv_for_star(id)% cv_data(i)
                  read(iounit, *, iostat=ierr) &
                     cv_data% step, &
                     cv_data% age, &
                     cv_data% cgrav, &
                     cv_data% star_mass, &
                     cv_data% star_mdot, &
                     cv_data% conv_mx1_top, &
                     cv_data% conv_mx1_bot, &
                     cv_data% conv_mx2_top, &
                     cv_data% conv_mx2_bot, &
                     cv_data% mx1_top, &
                     cv_data% mx1_bot, &
                     cv_data% mx2_top, &
                     cv_data% mx2_bot, &
                     cv_data% burn_zone_mass(1,1), &
                     cv_data% burn_zone_mass(2,1), &
                     cv_data% burn_zone_mass(3,1), &
                     cv_data% burn_zone_mass(4,1), &
                     cv_data% burn_zone_mass(1,2), &
                     cv_data% burn_zone_mass(2,2), &
                     cv_data% burn_zone_mass(3,2), &
                     cv_data% burn_zone_mass(4,2), &
                     cv_data% burn_zone_mass(1,3), &
                     cv_data% burn_zone_mass(2,3), &
                     cv_data% burn_zone_mass(3,3), &
                     cv_data% burn_zone_mass(4,3), &
                     cv_data% h1_mass, &
                     cv_data% he4_mass, &
                     cv_data% c12_mass, &
                     cv_data% log_L, &
                     cv_data% log_LH, &
                     cv_data% log_LHe, &
                     cv_data% log_Lneu, &
                     cv_data% log_Teff, &
                     cv_data% log_surface_R, &
                     cv_data% log_surface_P, &
                     cv_data% log_surface_Rho, &
                     cv_data% log_center_T, &
                     cv_data% log_center_Rho, &
                     cv_data% log_center_P, &
                     cv_data% center_degeneracy, &
                     cv_data% center_gamma, &
                     cv_data% center_ye, &
                     cv_data% center_entropy, &
                     cv_data% center_h1, &
                     cv_data% center_he4, &
                     cv_data% center_c12, &
                     cv_data% center_n14, &
                     cv_data% center_o16, &
                     cv_data% center_ne20, &
                     cv_data% surface_h1, &
                     cv_data% surface_he3, &
                     cv_data% surface_he4, &
                     cv_data% surface_c12, &
                     cv_data% surface_n14, &
                     cv_data% surface_o16
                  if (ierr /= 0) then
                     cv_current(id) = i-1;
                     exit
                  end if
               end do
            end if
         end if
         
         write(*,*) 'PGSTAR finished read text data from ' // trim(fname)

         close(unit=iounit)
         call free_iounit(iounit)
         
      end subroutine read_text_data
      
      
      subroutine save_text_data(s, ierr)
         use utils_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr

         integer :: iounit, i, id, ncols
         character (len=1024) :: fname
         type (pgstar_data), pointer :: cv_data
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            write(*,*) 'save_text_data: no iounits'
            return
         end if
         
         id = s% id
         
         fname = s% save_data_filename
         open(iounit, file=trim(fname), action='write', status='replace', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'save_text_data: cannot open new file'
            call free_iounit(iounit)
            return
         end if
         
         write(iounit,'(i6,a)') cv_current(id), ' steps saved in PGSTAR history'
         ncols = 54
         write(iounit,fmt='(i6)',advance='no') 1
         do i=2,ncols
            write(iounit,fmt='(i26,1x)',advance='no') i
         end do
         write(iounit,*)
         write(iounit,fmt='(a6,99(a26,1x))') &
            'step', &
            'age', &
            'cgrav', &
            'star_mass', &
            'star_mdot', &
            'conv_mx1_top', &
            'conv_mx1_bot', &
            'conv_mx2_top', &
            'conv_mx2_bot', &
            'mx1_top', &
            'mx1_bot', &
            'mx2_top', &
            'mx2_bot', &
            'burn_zone_mass_11', &
            'burn_zone_mass_21', &
            'burn_zone_mass_31', &
            'burn_zone_mass_41', &
            'burn_zone_mass_12', &
            'burn_zone_mass_22', &
            'burn_zone_mass_32', &
            'burn_zone_mass_42', &
            'burn_zone_mass_13', &
            'burn_zone_mass_23', &
            'burn_zone_mass_33', &
            'burn_zone_mass_43', &
            'h1_mass', &
            'he4_mass', &
            'c12_mass', &
            'log_L', &
            'log_LH', &
            'log_LHe', &
            'log_Lneu', &
            'log_Teff', &
            'log_surface_R', &
            'log_surface_P', &
            'log_surface_Rho', &
            'log_center_T', &
            'log_center_Rho', &
            'log_center_P', &
            'center_degeneracy', &
            'center_gamma', &
            'center_ye', &
            'center_entropy', &
            'center_h1', &
            'center_he4', &
            'center_c12', &
            'center_n14', &
            'center_o16', &
            'center_ne20', &
            'surface_h1', &
            'surface_he3', &
            'surface_he4', &
            'surface_c12', &
            'surface_n14', &
            'surface_o16'
         write(iounit,*)
         do i = 1, cv_current(id)
            cv_data => cv_for_star(id)% cv_data(i)
            write(iounit,fmt='(i6,99(1pe26.16,1x))') &
               cv_data% step, &
               cv_data% age, &
               cv_data% cgrav, &
               cv_data% star_mass, &
               cv_data% star_mdot, &
               cv_data% conv_mx1_top, &
               cv_data% conv_mx1_bot, &
               cv_data% conv_mx2_top, &
               cv_data% conv_mx2_bot, &
               cv_data% mx1_top, &
               cv_data% mx1_bot, &
               cv_data% mx2_top, &
               cv_data% mx2_bot, &
               cv_data% burn_zone_mass(1,1), &
               cv_data% burn_zone_mass(2,1), &
               cv_data% burn_zone_mass(3,1), &
               cv_data% burn_zone_mass(4,1), &
               cv_data% burn_zone_mass(1,2), &
               cv_data% burn_zone_mass(2,2), &
               cv_data% burn_zone_mass(3,2), &
               cv_data% burn_zone_mass(4,2), &
               cv_data% burn_zone_mass(1,3), &
               cv_data% burn_zone_mass(2,3), &
               cv_data% burn_zone_mass(3,3), &
               cv_data% burn_zone_mass(4,3), &
               cv_data% h1_mass, &
               cv_data% he4_mass, &
               cv_data% c12_mass, &
               cv_data% log_L, &
               cv_data% log_LH, &
               cv_data% log_LHe, &
               cv_data% log_Lneu, &
               cv_data% log_Teff, &
               cv_data% log_surface_R, &
               cv_data% log_surface_P, &
               cv_data% log_surface_Rho, &
               cv_data% log_center_T, &
               cv_data% log_center_Rho, &
               cv_data% log_center_P, &
               cv_data% center_degeneracy, &
               cv_data% center_gamma, &
               cv_data% center_ye, &
               cv_data% center_entropy, &
               cv_data% center_h1, &
               cv_data% center_he4, &
               cv_data% center_c12, &
               cv_data% center_n14, &
               cv_data% center_o16, &
               cv_data% center_ne20, &
               cv_data% surface_h1, &
               cv_data% surface_he3, &
               cv_data% surface_he4, &
               cv_data% surface_c12, &
               cv_data% surface_n14, &
               cv_data% surface_o16
         end do
         close(unit=iounit)
         call free_iounit(iounit)
         
         write(*,*) 'PGSTAR finished write text data to ' // trim(fname)

      end subroutine save_text_data
      

      subroutine write_pgstar_data(s, ierr)
         use utils_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr

         integer :: iounit, i, id
         character (len=1024) :: fname
         type (pgstar_data), pointer :: cv_data
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            write(*,*) 'write_pgstar_data: no iounits'
            return
         end if
         id = s% id
         
         fname = trim(s% log_directory) // '/pgstar.dat'
         open(iounit, file=trim(fname), action='write', &
                  status='replace', iostat=ierr, form='unformatted')
         if (ierr /= 0) then
            write(*,*) 'save_pgstar_data: cannot open new file'
            call free_iounit(iounit)
            return
         end if
         write(iounit) cv_current(id)
         do i = 1, cv_current(id)
            cv_data => cv_for_star(id)% cv_data(i)
            write(iounit) cv_data
         end do
         close(unit=iounit)
         call free_iounit(iounit)
         
         if (dbg) write(*,2) 'write_pgstar_data: ' // trim(fname), cv_current(id)

      end subroutine write_pgstar_data


      integer function read_pgstar_data(s, ierr)
         use utils_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr

         logical :: fexist
         integer :: iounit, i, id
         character (len=1024) :: fname
         type (pgstar_data), pointer :: cv_data
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         ierr = 0
         id = s% id
         cv_current(id) = 0
         read_pgstar_data = 0
         
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) then
            write(*,*) 'read_pgstar_data no iounits'
            return
         end if

         fname = trim(s% log_directory) // '/pgstar.dat'
         inquire(file=trim(fname), exist=fexist)
         if (.not.fexist) then
            if (dbg) write(*,*) 'failed to find ' // trim(fname)
            call free_iounit(iounit)
            return
         end if

         open(iounit, file=trim(fname), action='read', &
                  status='old', iostat=ierr, form='unformatted')
         if (ierr /= 0) then
            if (dbg) write(*,*) 'failed to open ' // trim(fname)
         else
            read(iounit, iostat=ierr) cv_current(id)
            if (ierr /= 0) then
               if (dbg) write(*,*) 'failed read cv_cur ' // trim(fname)
            else
               if (dbg) write(*,2) 'read cv_cur ' // trim(fname), cv_current(id)
               allocate(cv_for_star(id)% cv_data(cv_current(id)+1000), stat=ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in allocate for read_pgstar_data'
                  return
               end if
               do i=1,cv_current(id)
                  cv_data => cv_for_star(id)% cv_data(i)
                  read(iounit, iostat=ierr) cv_data
                  if (ierr/=0 .or. cv_data% age >= s% star_age &
                        .or. cv_data% step >= s% model_number) then
                     cv_current(id) = i-1;
                     exit
                  end if
               end do
            end if
         end if
         
         read_pgstar_data = cv_current(id)
         
         if (dbg) write(*,*) 'read_pgstar_data: cv_current(id)', cv_current(id)

         close(unit=iounit)
         call free_iounit(iounit)

      end function read_pgstar_data
      
      
      subroutine update_pgstar_data(s, ierr)
         use num_lib, only: safe_log10
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: id, sz
      
         type (pgstar_data), pointer :: cv_data
         type (pgstar_data), dimension(:), pointer :: cv2
         
         include 'formats.dek'

         ierr = 0
         id = s% id
         do while (cv_current(id) >= last_data_lim(id))
            last_data_lim(id) = last_data_lim(id) + 100
         end do
         
         if (.not. associated(cv_for_star(id)% cv_data)) &
            allocate(cv_for_star(id)% cv_data(last_data_lim(id)))
         sz = size(cv_for_star(id)% cv_data)
         if (last_data_lim(id) > sz) then
            allocate(cv2(2*last_data_lim(id) + 1000), stat=ierr)
            if (ierr /= 0) then
               write(*,*) 'allocate failed in PGSTAR for update_pgstar_data'
               return
            end if
            cv2(1:sz) = cv_for_star(id)% cv_data(1:sz)
            deallocate(cv_for_star(id)% cv_data)
            cv_for_star(id)% cv_data => cv2
         end if
         
         cv_current(id) = cv_current(id) + 1

         cv_data => cv_for_star(id)% cv_data(cv_current(id))
                  
         cv_data% age = s% star_age
         cv_data% step = s% model_number
         cv_data% cgrav = s% cgrav(1)
         cv_data% star_mass = s% star_mass
         cv_data% star_mdot = s% star_mdot
         
         cv_data% conv_mx1_top = s% conv_mx1_top*s% star_mass
         cv_data% conv_mx1_bot = s% conv_mx1_bot*s% star_mass
         cv_data% conv_mx2_top = s% conv_mx2_top*s% star_mass
         cv_data% conv_mx2_bot = s% conv_mx2_bot*s% star_mass
         
         cv_data% mx1_top = s% mx1_top*s% star_mass
         cv_data% mx1_bot = s% mx1_bot*s% star_mass
         cv_data% mx2_top = s% mx2_top*s% star_mass
         cv_data% mx2_bot = s% mx2_bot*s% star_mass
         
         cv_data% burn_zone_mass = s% burn_zone_mass/Msun

         cv_data% h1_mass = s% h1_boundary_mass
         cv_data% he4_mass = s% he4_boundary_mass
         cv_data% c12_mass = s% c12_boundary_mass
         
         cv_data% log_L = safe_log10(s% L(1)/Lsun)
         cv_data% log_LH = safe_log10(s% power_h_burn)
         cv_data% log_LHe = safe_log10(s% power_he_burn)
         cv_data% log_Lneu = safe_log10(s% power_neutrinos)
         
         cv_data% log_Teff = safe_log10(s% Teff)
         cv_data% log_surface_R = s% log_surface_radius
         cv_data% log_surface_P = s% log_surface_pressure
         cv_data% log_surface_Rho = s% log_surface_density
         
         cv_data% log_center_T = s% log_center_temperature
         cv_data% log_center_Rho = s% log_center_density
         cv_data% log_center_P = s% log_center_pressure

         cv_data% center_degeneracy = s% center_degeneracy
         cv_data% center_gamma = s% center_gamma
         cv_data% center_ye = s% center_ye
         cv_data% center_entropy = s% center_entropy
         
         cv_data% center_h1 = s% center_h1
         cv_data% center_he4 = s% center_he4
         cv_data% center_c12 = s% center_c12
         cv_data% center_n14 = s% center_n14
         cv_data% center_o16 = s% center_o16
         cv_data% center_ne20 = s% center_ne20
         
         cv_data% surface_h1 = s% surface_h1
         cv_data% surface_he3 = s% surface_he3
         cv_data% surface_he4 = s% surface_he4
         cv_data% surface_c12 = s% surface_c12
         cv_data% surface_n14 = s% surface_n14
         cv_data% surface_o16 = s% surface_o16

      end subroutine update_pgstar_data
      

      subroutine do_set_xaxis_bounds( &
            s, xaxis_by, win_xmin_in, win_xmax_in, xmargin, &
            xvec, xmin, xmax, xleft, xright, dx, &
            grid_min, grid_max, npts, ierr)
         type (star_info), pointer :: s
         character (len=*), intent(in) :: xaxis_by
         real, intent(in) :: win_xmin_in, win_xmax_in, xmargin
         real, pointer, dimension(:) :: xvec
         real, intent(out) :: xmin, xmax, xleft, xright, dx
         integer, intent(out) :: grid_min, grid_max, npts
         integer, intent(out) :: ierr
         call set_xaxis_bounds( &
            s, xaxis_by, win_xmin_in, win_xmax_in, .false., xmargin, &
            xvec, xmin, xmax, xleft, xright, dx, &
            grid_min, grid_max, npts, ierr)
      end subroutine do_set_xaxis_bounds
      
      
      subroutine do_show_xaxis_by(s,by,ierr)
         type (star_info), pointer :: s
         character (len=*), intent(in) :: by
         integer, intent(out) :: ierr
         call show_xaxis_by(s,by,ierr)
      end subroutine do_show_xaxis_by


      end module mod_pgstar
