! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module overshoot

      use const_def
      use num_lib
      use utils_lib
      use alert_lib, only: alert, bug_alert
      use star_private_def

      implicit none

      
      contains

      
      ! exponential diffusive overshoot as described in the paper by Falk Herwig, 
      ! "The evolution of AGB stars with convective overshoot", A&A, 360, 952-968 (2000)
      subroutine add_overshooting(s, ierr)
         use mlt_def
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         real(dp) :: D0, vc0, Hp, r0, D_ov, cdc_ov, vc_ov, frac, q_edge, r_edge, &
            f_above, f_below, f0_above, f0_below, f2_above, f2_below, D2_above, D2_below, &
            ov_limit, D_ov_limit, h1_czb_mass, h1_czb_dm, &
            h1_czb_mass_old, top_r, bot_r, zone_dr
         integer :: i, k, loc, k0, n_conv_bdy, nz
         real(dp), pointer, dimension(:) :: grada, gradr

         logical :: dbg, have_h1_czb
         
         include 'formats.dek'
         
         dbg = .false.
         
         ierr = 0         
         nz = s% nz
         grada => s% grada_at_face
         gradr => s% gradr
         n_conv_bdy = s% num_conv_boundaries
         D_ov_limit = s% D_mix_ov_limit
         
         do i=1, n_conv_bdy ! from center to surface

            if (s% conv_bdy_q(i) < s% min_overshoot_q) then
               if (dbg) write(*,*) 'skip since s% conv_bdy_q(i) < min_overshoot_q', i
               cycle
            end if
            
            call set_f(i)
            
            have_h1_czb = .false.
            
            if (f_above == 0 .and. f_below == 0) cycle
            if (s% star_mass <= s% mass_for_overshoot_full_off) cycle
            if (s% star_mass <= s% mass_for_overshoot_full_on) then
               frac = (s% star_mass - s% mass_for_overshoot_full_off) / &
                      (s% mass_for_overshoot_full_on - s% mass_for_overshoot_full_off)
               frac = 0.5d0*(1 - cos(pi*frac))
               if (dbg) write(*,1) 'overshoot frac', frac
               f_above = f_above*frac
               f_below = f_below*frac
            end if
            
            if (f0_above < 0) f0_above = f_above
            if (f0_below < 0) f0_below = f_below
            
            loc = s% conv_bdy_loc(i)
            if (loc < 1 .or. loc > nz) then
               write(*,*) 'bad s% conv_bdy_loc(i)', i, s% conv_bdy_loc(i)
               ierr = -1
               return
            end if
            
            if (s% top_conv_bdy(i)) then ! overshoot toward surface
               if (f_above > 0 .and. s% mixing_type(loc+1) == convective_mixing .and. loc > 1) then
                  if (i <= 1) then
                     bot_r = s% R_center
                  else
                     if (s% top_conv_bdy(i-1)) cycle
                     bot_r = s% r(s% conv_bdy_loc(i-1))
                  end if
                  top_r = s% r(loc)
                  zone_dr = top_r - bot_r
                  if (dbg) write(*,1) 'toward surface zone_dr', zone_dr
                  call overshoot_toward_surface(ierr)
                  if (ierr /= 0) return
               end if
            else ! overshoot toward center
               if (f_below > 0 .and. s% mixing_type(loc-1) == convective_mixing) then
                  if (i >= n_conv_bdy) then
                     top_r = s% r(1)
                  else
                     if (.not. s% top_conv_bdy(i+1)) cycle
                     top_r = s% r(s% conv_bdy_loc(i+1))
                  end if
                  bot_r = s% r(loc)
                  zone_dr = top_r - bot_r
                  if (dbg) write(*,1) 'toward center zone_dr', zone_dr
                  call overshoot_toward_center( &
                     s% burn_h_conv_region(i), s% burn_he_conv_region(i), s% burn_z_conv_region(i), ierr)
                  if (ierr /= 0) return
               end if
            end if
            if (dbg) write(*,*)
         end do
                           
         
         contains
         
         
         subroutine overshoot_toward_surface(ierr)
            integer, intent(out) :: ierr
            include 'formats.dek'
            real(dp) :: h, h2, dr, dr2, dr_limit, r_limit
            ierr = 0
            if (dbg) write(*,1) 'overshoot_toward_surface'
            call get_r0_vc0(loc, .false., f0_above, k0, r0, vc0, D0, Hp, q_edge, r_edge, ierr)
            if (ierr /= 0) then
               write(*,*) 'overshoot toward surface failure'
               return
            end if
            if (k0 <= 0) then
               if (dbg) write(*,2) 'skip overshoot_toward_surface: k0', k0
               return
            end if

            h = f_above*Hp            
            if (D2_above > 0 .and. f2_above > 0 .and. D2_above < D0) then
               dr2 = h*log(D0/D2_above)/2
               h2 = f2_above*Hp
            else
               dr2 = 1d99
            end if

            do k = k0, 1, -1
               if (s% r(k) < r0) cycle
               dr = s% r(k) - r0
               if (k == k0) then
                  ov_limit = 0
               else
                  ov_limit = D_ov_limit
               end if
               if (dr < dr2) then
                  call get_D_ov(k, dr, D0, h, D_ov, ov_limit, cdc_ov, vc_ov)
               else
                  call get_D_ov(k, dr-dr2, D2_above, h2, D_ov, ov_limit, cdc_ov, vc_ov)
               end if
               if (cdc_ov == 0) then
                  if (dbg) write(*,2) 'finished extending upward', k


                  if (s% mixing_type(k+1) == overshoot_mixing) then 
                     ! set mixing_type_change_dq(k)
                     if (s% overshoot_step_fraction > 0) then
                        dr_limit = h
                     else if (dr < dr2) then
                        dr_limit = 0.5d0*h*log(D0/ov_limit)
                     else
                        dr_limit = dr + 0.5d0*h2*log(D2_below/ov_limit)
                     end if
                     r_limit = r0 + dr_limit
                     if (s% r(k) >= r_limit .and. r_limit > s% r(k+1)) then
                        s% mixing_type_change_dq(k) = s% dq(k) * &
                           (s% r(k)**3 - r_limit**3) / (s% r(k)**3 - s% r(k+1)**3)
                     end if
                  end if


                  exit
               end if
               if (dbg) then
                  write(*,2) 'overshoot upward', k, (s% r(k)-r0)/Rsun, &
                     s% r(k)/Rsun, Hp/Rsun, D_ov, s% D_mix(k), s% r(k) - r_edge
               end if
               if (s% r(k) <= r_edge) then ! leave it convective type, but change mixing coeff
                  if (dbg) write(*,*) 'leave it convective type, but change mixing coeff'
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = convective_mixing
               else if (s% mixing_type(k) == overshoot_mixing .or. &
                        s% mixing_type(k) == convective_mixing) then
                  ! have encountered another cz; add effects
                  if (dbg) write(*,*) 'have encountered another cz; add effects'
                  s% cdc(k) = s% cdc(k) + cdc_ov
                  s% D_mix(k) = s% D_mix(k) + D_ov
                  s% conv_vel(k) = s% conv_vel(k) + vc_ov
               else ! change to overshooting
                  if (dbg) write(*,*) 'change to overshooting'
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = overshoot_mixing
               end if
            end do
         end subroutine overshoot_toward_surface
         
         
         subroutine overshoot_toward_center(burn_h, burn_he, burn_z, ierr)
            logical, intent(in) :: burn_h, burn_he, burn_z
            integer, intent(out) :: ierr
            
            real(dp) :: h, h2, dr, dr2, dr_limit, r_limit
            logical :: dbg
            include 'formats.dek'


            dbg = .false. ! r_edge < 0.7*Rsun .and. r_edge > 0.6*Rsun
            !dbg = r_edge < 0.75*Rsun .and. r_edge > 0.65*Rsun

            if (dbg) write(*,*) 'overshoot_toward_center'
            ierr = 0
            call get_r0_vc0(loc-1, .true., f0_below, k0, r0, vc0, D0, Hp, q_edge, r_edge, ierr)
            if (ierr /= 0) return
            if (k0 <= 0) return       
            
            if (dbg) then
               write(*,2) 'loc-1', loc-1
               write(*,2) 'k0', k0
               write(*,1) 'f0_below', f0_below
               write(*,1) 'r0', r0
               write(*,1) 'Hp', Hp
            end if     
            
            if ((.not. (burn_h .or. burn_he .or. burn_z)) .and. (.not. have_h1_czb)) then
               have_h1_czb = .true.
               h1_czb_mass = s% M_center + s% xmstar*q_edge
               s% h1_czb_mass = h1_czb_mass/Msun
               if (.false.) then
                  write(*,2) 's% h1_czb_mass', k0, s% h1_czb_mass
                  write(*,2) 'q_edge', k0, q_edge
                  write(*,2) 'r_edge/Rsun', k0, r_edge/Rsun
                  write(*,2) 'Hp/Rsun', k0, Hp/Rsun
                  write(*,*)
               end if
               call check_TP
               if (dbg) write(*,1) 'f_below', f_below
            end if

            h = f_below*Hp
            if (D2_below > 0 .and. f2_below > 0 .and. D2_below < D0) then
               dr2 = h*log(D0/D2_below)/2
               h2 = f2_below*Hp
               if (dbg) then
                  write(*,2) 'dr2', k0, dr2
                  if (r0 > dr2) write(*,1) 'log10((r0-dr2)/Rsun)', log10((r0-dr2)/Rsun)
                  write(*,1) 'log10(r0/Rsun)', log10(r0/Rsun)
                  write(*,1) 'D2_below', D2_below
                  write(*,1) 'f2_below', f2_below
                  write(*,1) 'f_below', f_below
                  write(*,1) 'Hp', Hp
                  write(*,1) 'h', h
                  write(*,1) 'D0', D0
                  write(*,*)
               end if
            else
               dr2 = 1d99
               h2 = 0
            end if
            
            
            
            do k=k0+1, nz
               if (s% r(k) > r0) cycle
               dr = r0 - s% r(k)
               
               if (dbg) write(*,1) 'dr/(f_below*Hp)', dr/(f_below*Hp)
               
               if (k == k0+1) then
                  ov_limit = 0
               else
                  ov_limit = D_ov_limit
               end if
               if (dr < dr2) then
                  call get_D_ov(k, dr, D0, h, D_ov, ov_limit, cdc_ov, vc_ov)
               else
                  call get_D_ov(k, dr-dr2, D2_below, h2, D_ov, ov_limit, cdc_ov, vc_ov)
               end if
               
               if (dbg) write(*,1) 'D_ov', D_ov
               if (dbg) write(*,1) 'D_ov/D0', D_ov/D0
               
               if (cdc_ov == 0) then
                  if (dbg) write(*,2) 'finished extending downward', k
                  if (s% mixing_type(k-1) == overshoot_mixing) then 
                     ! set mixing_type_change_dq(k-1)
                     if (s% overshoot_step_fraction > 0) then
                        dr_limit = h
                     else if (dr < dr2) then
                        dr_limit = 0.5d0*h*log(D0/ov_limit)
                     else
                        dr_limit = dr + 0.5d0*h2*log(D2_below/ov_limit)
                     end if
                     r_limit = r0 - dr_limit
                     if (s% r(k-1) >= r_limit .and. r_limit > s% r(k)) then
                        s% mixing_type_change_dq(k-1) = s% dq(k-1) * &
                           (s% r(k-1)**3 - r_limit**3) / (s% r(k-1)**3 - s% r(k)**3)
                     end if
                  end if
                  exit
               end if
               if (dbg) then
                  write(*,2) 'overshoot downward', k, s% star_mass*s% q(k), s% r(k)/Rsun
                  write(*,*)
               end if
               if (s% r(k) >= r_edge) then ! leave it convective type, but change mixing coeff
                  if (dbg) write(*,2) 'leave it convective', k
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = convective_mixing
               else if (s% mixing_type(k) == no_mixing) then ! change to overshooting
                  if (dbg) write(*,2) 'was no_mixing', k
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = overshoot_mixing
               else if (s% mixing_type(k) == overshoot_mixing) then
                  ! have encountered overshooting from another cz; add effects
                  if (dbg) write(*,2) 'was overshooting already', k
                  s% cdc(k) = s% cdc(k) + cdc_ov
                  s% D_mix(k) = s% D_mix(k) + D_ov
                  s% conv_vel(k) = s% conv_vel(k) + vc_ov
               else ! have run into a different mixing region
                  exit
               end if
            end do
            
         end subroutine overshoot_toward_center

         
         subroutine set_f(i)
            integer, intent(in) :: i
            if (s% burn_h_conv_region(i)) then
               f_above = s% overshoot_f_above_burn_h
               f_below = s% overshoot_f_below_burn_h
               f0_above = s% overshoot_f0_above_burn_h
               f0_below = s% overshoot_f0_below_burn_h
               D2_above = s% overshoot_D2_above_burn_h
               D2_below = s% overshoot_D2_below_burn_h
               f2_above = s% overshoot_f2_above_burn_h
               f2_below = s% overshoot_f2_below_burn_h
            else if (s% burn_he_conv_region(i)) then            
               f_above = s% overshoot_f_above_burn_he
               f_below = s% overshoot_f_below_burn_he
               f0_above = s% overshoot_f0_above_burn_he
               f0_below = s% overshoot_f0_below_burn_he
               D2_above = s% overshoot_D2_above_burn_he
               D2_below = s% overshoot_D2_below_burn_he
               f2_above = s% overshoot_f2_above_burn_he
               f2_below = s% overshoot_f2_below_burn_he
               if (s% have_done_TP) then
                  f_below = f_below*s% ovr_below_burn_he_factor
                  f0_below = f0_below*s% ovr_below_burn_he_factor
                  f2_below = f2_below*s% ovr_below_burn_he_factor
               end if               
            else if (s% burn_z_conv_region(i)) then            
               f_above = s% overshoot_f_above_burn_z
               f_below = s% overshoot_f_below_burn_z
               f0_above = s% overshoot_f0_above_burn_z
               f0_below = s% overshoot_f0_below_burn_z
               D2_above = s% overshoot_D2_above_burn_z
               D2_below = s% overshoot_D2_below_burn_z
               f2_above = s% overshoot_f2_above_burn_z
               f2_below = s% overshoot_f2_below_burn_z
            else            
               f_above = s% overshoot_f_above_nonburn
               f_below = s% overshoot_f_below_nonburn
               f0_above = s% overshoot_f0_above_nonburn
               f0_below = s% overshoot_f0_below_nonburn               
               D2_above = s% overshoot_D2_above_nonburn
               D2_below = s% overshoot_D2_below_nonburn
               f2_above = s% overshoot_f2_above_nonburn
               f2_below = s% overshoot_f2_below_nonburn
            end if
         end subroutine set_f
         
         
         subroutine check_TP ! check for AGB thermal pulse dredge up
            logical, parameter :: dbg = .false.
            include 'formats.dek'
            if (s% power_h_burn + s% power_he_burn < 0.5d0*s% power_nuc_burn) then
               if (dbg .and. s% TP_state /= 0) then
                  write(*,*) 'advanced burning: change to TP_state = 0'
                  write(*,*)
               end if
               s% TP_state = 0
            else if (s% h1_boundary_mass > s% he4_boundary_mass + 0.1d0) then
               if (dbg .and. s% TP_state /= 0) then
                  write(*,*) 'not yet: change to TP_state = 0'
                  write(*,1) 's% h1_boundary_mass', s% h1_boundary_mass
                  write(*,1) 's% he4_boundary_mass', s% he4_boundary_mass
                  write(*,1) 'h1 - he4_boundary_mass', s% h1_boundary_mass - s% he4_boundary_mass
                  write(*,*)
               end if
               s% TP_state = 0
            else if (s% TP_state == 0) then ! not in TP
               if (s% power_he_burn > s% power_h_burn) then ! starting TP
                  s% TP_state = 1
                  s% TP_M_H_on = h1_czb_mass
                  s% TP_M_H_min = h1_czb_mass
                  if (dbg) then
                     write(*,*) 'change to TP_state = 1'
                     write(*,1) 's% TP_M_H_on', s% TP_M_H_on
                     write(*,*)
                  end if
               end if
            else
               h1_czb_dm = h1_czb_mass*1d-6
               if (s% TP_state == 1) then ! early part of TP
                  if (h1_czb_mass < s% TP_M_H_on - h1_czb_dm) then
                     if (dbg) then
                        write(*,*) 'change to TP_state = 2'
                        write(*,1) 's% TP_M_H_on', s% TP_M_H_on
                        write(*,1) 'h1_czb_dm', h1_czb_dm
                        write(*,1) 's% TP_M_H_on - h1_czb_dm', s% TP_M_H_on - h1_czb_dm
                        write(*,1) 'h1_czb_mass', h1_czb_mass
                        write(*,1) '(s% TP_M_H_on - h1_czb_dm) - h1_czb_mass', &
                           (s% TP_M_H_on - h1_czb_dm) - h1_czb_mass
                        write(*,*)
                     end if
                     s% TP_state = 2
                     s% TP_count = 0
                     s% TP_M_H_min = h1_czb_mass
                     s% have_done_TP = .true.
                  else if (s% power_h_burn > s% power_he_burn) then ! no dredge up
                     if (dbg) write(*,*) 'change to TP_state = 0: no dredge up this time'
                     s% TP_state = 0
                  end if
               else if (s% TP_state == 2) then ! dredge up part of TP
                  if (h1_czb_mass < s% TP_M_H_min) s% TP_M_H_min = h1_czb_mass
                  h1_czb_mass_old = s% h1_czb_mass_old*Msun
                  if (dbg) then
                     write(*,1) '(h1_czb_mass - h1_czb_mass_old)/Msun', (h1_czb_mass - h1_czb_mass_old)/Msun
                     write(*,1) 'h1_czb_dm/Msun', h1_czb_dm/Msun
                     write(*,*)
                  end if
                  if (h1_czb_mass > h1_czb_mass_old + h1_czb_dm .or. &
                        s% power_h_burn > s% power_he_burn) then
                     s% TP_count = s% TP_count + 1
                     if (dbg) write(*,2) '++ TP_count', s% TP_count, &
                           h1_czb_mass - (h1_czb_mass_old + h1_czb_dm), &
                           h1_czb_mass,  h1_czb_mass_old + h1_czb_dm, &
                           s% h1_czb_mass_old, h1_czb_dm
                  else if (h1_czb_mass < h1_czb_mass_old - h1_czb_dm) then
                     s% TP_count = max(0, s% TP_count - 1)
                     if (dbg) write(*,3) '-- TP_count', s% TP_count, s% TP_state
                  end if
                  if (s% TP_count > s% max_DUP_counter) then
                     if (dbg) write(*,*) 'change to TP_state = 0'
                     s% TP_state = 0
                  end if
               end if
               if (s% TP_state == 2) then
                  f_below = f_below*s% overshoot_below_noburn_factor
                  f0_below = f0_below*s% overshoot_below_noburn_factor
               end  if
            end if
         end subroutine check_TP
         
         
         subroutine get_r0_vc0( &
               k00, overshoot_toward_center, f0, k0, r0, vc0, D0, Hp, q_edge, r_edge, ierr)
            integer, intent(in) :: k00 
            ! expect edge of convection somewhere in cell k00
            ! if s% use_Ledoux_criterion, then edge is where gradL == gradr
            ! else, else is where grada_at_face == gradr
            logical, intent(in) :: overshoot_toward_center
            real(dp), intent(in) :: f0
            integer, intent(out) :: k0
            real(dp), intent(out) :: r0, vc0, D0, Hp, q_edge, r_edge
            ! r(k0) >= r0 >= r(k0+1)
            ! r0 is radius where start overshooting
            ! vc0 is convection velocity at r0
            ! Hp is scale height at edge of convection
            integer, intent(out) :: ierr
            
            real(dp) :: dq, ddq, dqfrac, x, x0, x1, x2, a0, a1, a2, &
               lnP_edge, P_edge, lnd_edge, rho_edge, frac, dg00, dgp1, grada_at_face         
            integer :: km1, kp1, kk
            
            logical :: dbg
            
            include 'formats.dek'
            
            dbg = .false.
            k0 = -1
            r0 = 0
            vc0 = 0
            D0 = 0
            Hp = 0
            ierr = 0

            km1 = k00-1
            kp1 = k00+1
            
            if (kp1 > nz) then
               if (.not. overshoot_toward_center) return
               k0 = nz
               r0 = s% r(nz)
               vc0 = s% conv_vel(nz)
               D0 = s% cdc(nz)
               Hp = r0
               return
            end if

            dq = s% dq(k00)
            ddq = s% mixing_type_change_dq(k00)
            dqfrac = ddq/dq ! fractional distance from k00 to kp1 for edge of convection
            q_edge = s% q(k00) - ddq
            r_edge = exp(dqfrac*s% lnR(kp1) + (1-dqfrac)*s% lnR(k00))
            
            x0 = s% dq(km1)/2
            x1 = s% dq(km1) + s% dq(k00)/2
            x2 = s% dq(km1) + s% dq(k00) + s% dq(kp1)/2
            x = s% dq(km1) + ddq
            call two_piece_linear_coeffs(x, x0, x1, x2, a0, a1, a2, ierr)
            if (ierr /= 0) return
            
            lnP_edge = a0*s% lnP(km1) + a1*s% lnP(k00) + a2*s% lnP(kp1)
            P_edge = exp(lnP_edge)
            lnd_edge = a0*s% lnd(km1) + a1*s% lnd(k00) + a2*s% lnd(kp1)
            rho_edge = exp(lnd_edge)
            Hp = P_edge/(rho_edge*s% cgrav(k00)*(s% M_center + s% xmstar*q_edge)/r_edge**2)
            if (s% mixing_length_alpha*Hp > zone_dr) Hp = zone_dr/s% mixing_length_alpha
            if (dbg) then
               write(*,*)
               write(*,*)
               write(*,*)
               write(*,1) 'r_edge+f0*Hp', (r_edge+f0*Hp)/Rsun
               write(*,2) 'r(km1)', k00-1, exp(s% lnR(k00-1))/Rsun
               write(*,2) 'r(k00)', k00, exp(s% lnR(k00))/Rsun
               write(*,1) 'r_edge', r_edge/Rsun
               write(*,2) 'r(kp1)', kp1, exp(s% lnR(kp1))/Rsun
               write(*,2) 'r(kp2)', kp1+1, exp(s% lnR(kp1+1))/Rsun
               write(*,1) 'r_edge-f0*Hp', (r_edge-f0*Hp)/Rsun
               write(*,*)
            end if
            
            if (overshoot_toward_center) then
               r0 = r_edge + f0*Hp
               if (dbg) then
                  write(*,*) 'get_r0_vc0: overshoot toward center'
                  write(*,2) 'r_edge/Rsun', k00, r_edge/Rsun
                  write(*,2) 'Hp/Rsun', k00, Hp/Rsun
                  write(*,2) 'f0/Rsun', k00, f0
                  write(*,2) 'f0*Hp/Rsun', k00, f0*Hp/Rsun
                  write(*,2) '(r_edge + f0*Hp)//Rsun', k00, (r_edge + f0*Hp)/Rsun
                  write(*,2) 'initial k0 for toward center: k00, r0', k00, r0/Rsun
               end if
               k0 = k00
               do kk = k00, 1, -1
                  if (s% r(kk) >= r0) then
                     k0 = kk; exit
                  end if
                  if (s% mixing_type(kk) /= convective_mixing) then
                     if (dbg) write(*,3) 'gap near bottom convective zone', kk, s% model_number
                     if (dbg) write(*,*)
                     k0 = -1; return
                  end if
               end do
               if (dbg) write(*,2) 'final k0, r(k0)/Rsun', k0, s% r(k0)/Rsun, s% r(k0-1)/Rsun
               if (dbg) write(*,*)
            else
               r0 = r_edge - f0*Hp
               if (dbg) then
                  write(*,*) 'get_r0_vc0: overshoot toward surface'
                  write(*,2) 'r_edge', k00, r_edge/Rsun
                  write(*,2) 'Hp', k00, Hp/Rsun
                  write(*,2) 'f0', k00, f0
                  write(*,2) 'f0*Hp', k00, f0*Hp/Rsun
                  write(*,2) 'r_edge - f0*Hp', k00, (r_edge - f0*Hp)/Rsun
                  write(*,2) 'initial k0 for toward surface: k00, r0', k00, r0/Rsun
               end if
               k0 = k00+1
               do kk = k00+1, nz-1
                  if (dbg) write(*,2) 's% r(kk+1)', kk+1, s% r(kk+1)
                  if (s% r(kk+1) <= r0) then
                     k0 = kk; exit
                  end if
                  if (s% mixing_type(kk) /= convective_mixing) then
                     if (dbg) write(*,3) 'gap near top of convective zone', kk, s% model_number
                     if (dbg) write(*,*)
                     k0 = -1
                     return
                  end if
               end do
               if (dbg) write(*,2) 'final k0', k0
               if (dbg) write(*,*)
            end if
            if (k0 < 0) return
            if (s% r(k0) <= s% r(k0+1)) then
               if (dbg) write(*,*) 'get_r0_vc0: bad rs: s% r(k0) <= s% r(k0+1)', k0
               ierr = -1
               return
            end if
            
            frac = (s% r(k0)**3 - r0**3)/(s% r(k0)**3 - s% r(k0+1)**3)
            s% mixing_type_change_dq(k0) = frac*s% dq(k0)
            vc0 = (1-frac)*s% conv_vel(k0) + frac*s% conv_vel(k0+1)
            D0 = vc0*s% mixing_length_alpha*Hp/3 ! diffusion coefficient [cm^2/sec]
                        
         end subroutine get_r0_vc0
         
         
         subroutine get_D_ov(k, z, D0, ov_scale, D_ov, ov_limit, cdc_ov, vc_ov)
            integer, intent(in) :: k
            real(dp), intent(in) :: D0, ov_scale, ov_limit
            real(dp), intent(out) :: D_ov, cdc_ov, vc_ov
            real(dp) :: rho, r, z, fctr
            include 'formats.dek'
            if (s% overshoot_step_fraction <= 0) then
               fctr = exp(-2*z/ov_scale)
            else if (z > ov_scale) then
               fctr = 0d0
            else
               fctr = s% overshoot_step_fraction
            end if
            D_ov = D0*fctr
            if (D_ov < ov_limit) then
               cdc_ov = 0
               vc_ov = 0
               return
            end if
            rho = s% rho(k)
            r = s% r(k)
            cdc_ov = (pi4*r**2*rho)**2*D_ov ! gm^2/sec
            vc_ov = vc0*fctr
         end subroutine get_D_ov

         
      end subroutine add_overshooting

      
      logical function previously_non_convective(s, k)
         type (star_info), pointer :: s
         integer, intent(in) :: k
         real(dp) :: m
         integer :: j
         include 'formats.dek'
         m = s% M_center + s% xmstar*(s% q(k) - s% dq(k)/2)
         previously_non_convective = .true.
         if (s% generations > 2) then
            do j=1, s% n_conv_regions_older
               if (s% cz_bot_mass_older(j) <= m .and. m <= s% cz_top_mass_older(j)) then
                  ! point was in this previously convective region
                  !if (dbg) write(*,3) 'was previously convective', k, j, &
                  !   s% cz_bot_mass_older(j)/Msun, m/Msun, s% cz_top_mass_older(j)/Msun
                  previously_non_convective = .false.
                  return
               end if
            end do
         else
            do j=1, s% n_conv_regions_old
               if (s% cz_bot_mass_old(j) <= m .and. m <= s% cz_top_mass_old(j)) then
                  ! point was in this previously convective region
                  !if (dbg) write(*,3) 'was previously convective', k, j, &
                  !   s% cz_bot_mass_old(j)/Msun, m/Msun, s% cz_top_mass_old(j)/Msun
                  previously_non_convective = .false.
                  return
               end if
            end do
         end if
      end function previously_non_convective


      
      
      end module overshoot
