! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module opacities
      
      use star_private_def
      use alert_lib,only:alert
      use utils_lib,only:is_bad_num
      use const_def
      
      implicit none
      
      logical, parameter :: dbg = .false.
      
      
      logical, parameter :: test_remove_Z = .false.
      

      contains
      
      
      subroutine set_kap_params(s,ierr)
         use kap_lib, only: kap_ptr
         use kap_def, only: Kap_General_Info
         type (star_info), pointer :: s         
         integer, intent(out) :: ierr
         type (Kap_General_Info), pointer :: rq
         call kap_ptr(s% kap_handle, rq, ierr)
         if (ierr /= 0) return
         rq% cubic_interpolation_in_X = s% cubic_interpolation_in_X
         rq% cubic_interpolation_in_Z = s% cubic_interpolation_in_Z
         rq% include_electron_conduction = s% include_electron_conduction
      end subroutine set_kap_params
      
      
      subroutine do_opacities(s,nzlo,nzhi,ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         call set_kap_params(s,ierr)
         if (ierr /= 0) return
         if (dbg) write(*,*) 'call do_kap'
         call do_kap(s,nzlo,nzhi,ierr)
      end subroutine do_opacities
      
      
      subroutine do_kap(s,nzlo,nzhi,ierr)
         use star_utils, only: foreach_cell
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         logical, parameter :: use_omp = .true.
         call foreach_cell(s, nzlo, nzhi, use_omp, do_kap_for_cell,  ierr)
      end subroutine do_kap

      
      subroutine do_kap_test(s,nzlo,nzhi,ierr) ! for finding omp problems
         use chem_def
         use star_utils, only: foreach_cell
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         real(dp), dimension(:), pointer :: kap, dlnd, dlnT
         integer :: nz, k
         integer, pointer :: net_iso(:)
         11 format(a30,99(1pe26.16,3x))
         nz = s% nz
         
         allocate(kap(nz), dlnd(nz), dlnT(nz))
         call foreach_cell(s,nzlo,nzhi,.false.,do_kap_for_cell,ierr)
         kap(1:nz) = s% opacity(1:nz)
         dlnd(1:nz) = s% d_opacity_dlnd(1:nz)
         dlnT(1:nz) = s% d_opacity_dlnT(1:nz)
         call foreach_cell(s,nzlo,nzhi,.true.,do_kap_for_cell,ierr)
         do k=1,nz
            if (kap(k) /= s% opacity(k) .or. &
                dlnd(k) /= s% d_opacity_dlnd(k) .or. &
                dlnT(k) /= s% d_opacity_dlnT(k)) then
                net_iso => s% net_iso
                write(*,*) 'k', k
                write(*,11) 'kap', kap(k), s% opacity(k), kap(k) - s% opacity(k) 
                write(*,11) 'dlnd', dlnd(k), s% d_opacity_dlnd(k), dlnd(k) - s% d_opacity_dlnd(k) 
                write(*,11) 'dlnT', dlnT(k), s% d_opacity_dlnT(k), dlnT(k) - s% d_opacity_dlnT(k)
                write(*,11) 'logT =', s% lnT(k)/ln10
                write(*,11) 'logRho =', s% lnd(k)/ln10
                write(*,11) 'z =', 1 - (s% xa(net_iso(ih1),k) + s% xa(net_iso(ihe3),k) + s% xa(net_iso(ihe4),k))
                write(*,11) 'xh =', s% xa(net_iso(ih1),k)
                write(*,11) 'xc =', s% xa(net_iso(ic12),k)
                write(*,11) 'xn =', s% xa(net_iso(in14),k)
                write(*,11) 'xo =', s% xa(net_iso(io16),k)
                write(*,11) 'xne =', s% xa(net_iso(ine20),k)
                write(*,*)
            end if
         end do

         !call save_kap_test_data(s)
         stop 'debug: do_kap'
      end subroutine do_kap_test
      
      
      subroutine do_kap_for_cell(s,k,ierr)
         use const_def,only:ln10
         use utils_lib,only:is_bad_num
         use net_def,only:net_general_info
         use rates_def, only:i_rate
         use alert_lib
         use chem_def
         use kap_def
         use kap_lib
         use eos_def, only : i_lnfree_e
         use eos_lib, only: Radiation_Pressure
         use star_utils, only: get_XYZ
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         integer, intent(out) :: ierr

         integer, parameter :: nmet = 5
         integer, pointer :: net_iso(:)
         integer :: i, iz, kh
         real(dp) :: &
               z, xh, xhe, xc, xn, xo, xne, base_Z, xheavy, metals(nmet), &
               log10_rho, log10_T, dlnkap_dlnd, dlnkap_dlnT, &
               opacity_max, opacity_max0, opacity_max1, &
               lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
               log_r, log_r_in, log_r_out, log_r_frac, frac, min_cno_for_kap_limit, &
               P, Prad, Pgas, Pgas_div_P, Ledd_factor, Ledd_kap, Ledd_log, &
               a, b, da_dlnd, da_dlnT, db_dlnd, db_dlnT
         character (len=100) :: message
         
         include 'formats.dek'
                  
         ierr = 0
         
         log10_rho = s% lnd(k)/ln10
         log10_T = s% lnT(k)/ln10
         
         lnfree_e = s% lnfree_e(k)
         d_lnfree_e_dlnRho = s% d_eos_dlnd(i_lnfree_e,k)
         d_lnfree_e_dlnT = s% d_eos_dlnT(i_lnfree_e,k)
         
         if (.false.) then ! DEBUG
            call get_XYZ(s, s% xa(:,k), xh, xhe, Z)
            if (is_bad_num(xh) .or. xh < 0 .or. xh > 1) then
               ierr = -1
               !return
               call show_stuff
               stop 'debug: do_kap_for_cell'
            end if
         end if
         
         if (test_remove_Z) then
            P = exp(s% lnP_pre_hydro(k))
            Prad = Radiation_Pressure(exp(s% lnT_pre_hydro(k)))
            Pgas = P - Prad
            Pgas_div_P = max(0d0,Pgas/P)
         else
            Pgas_div_P = 1
         end if
         s% opacity(k) = get1_kap( &
            s, k, s% zbar(k), s% xa(:,k), s% q(k), Pgas_div_P, &
            log10_rho, log10_T, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            dlnkap_dlnd, dlnkap_dlnT, ierr)
         
         if (ierr /= 0 .or. is_bad_num(s% opacity(k))) then
            if (s% report_ierr) then
               return
!$omp critical (star_kap_get)
               write(message,*) 'do_kap_for_cell: kap_get failure for cell ', k
               write(*,*) trim(message)
               call alert(ierr,message)
               call show_stuff
               stop 'debug: do_kap_for_cell'
!$omp end critical (star_kap_get)
            end if
            ierr = -1
            return
         end if
         
         if (s% opacity(k) < 1d-99) then
            s% opacity(k) = 1d-99
            dlnkap_dlnd = 0
            dlnkap_dlnT = 0
         end if

         s% opacity(k) = s% opacity_factor*s% opacity(k)
         s% d_opacity_dlnd(k) = s% opacity(k)*dlnkap_dlnd
         s% d_opacity_dlnT(k) = s% opacity(k)*dlnkap_dlnT
                  
         ! reduce opacity when Pgas << Ptotal
         if (s% Pgas(k) < s% P(k)*s% kap_Pgas_div_P_limit .and. &
               s% q(k) <= s% kap_max_q_for_Pgas_div_P_limit .and. &
               s% q(k) >= s% kap_min_q_for_Pgas_div_P_limit) then
            
            b = s% Pgas(k)/(s% P(k)*s% kap_Pgas_div_P_limit)
            db_dlnd = -b*s% chiRho(k)
            db_dlnT = b*(4 - s% chiT(k))
            a = 1 - b
            da_dlnd = -db_dlnd
            da_dlnT = -db_dlnT
         
            s% opacity(k) = a*s% kap_Pgas_div_P_alt_kap + b*s% opacity(k)
            s% d_opacity_dlnd(k) = b*s% d_opacity_dlnd(k)
            s% d_opacity_dlnT(k) = b*s% d_opacity_dlnT(k)
            
            if (is_bad_num(s% opacity(k))) then
               write(*,2) 's% kap_Pgas_div_P_limit', k, s% kap_Pgas_div_P_limit
               write(*,2) 's% kap_min_q_for_Pgas_div_P_limit', k, s% kap_min_q_for_Pgas_div_P_limit
               write(*,2) 's% kap_max_q_for_Pgas_div_P_limit', k, s% kap_max_q_for_Pgas_div_P_limit
               write(*,2) 's% q(k)', k, s% q(k)
               write(*,2) 's% Pgas(k)', k, s% Pgas(k)
               write(*,2) 's% P(k)', k, s% P(k)
               stop 'do_kap_for_cell'
            end if

         end if

         if (s% opacity(k) > s% opacity_max .and. s% opacity_max > 0) then
            s% opacity(k) = s% opacity_max
            s% d_opacity_dlnd(k) = 0
            s% d_opacity_dlnT(k) = 0
         end if
         
         opacity_max0 = s% opacity_max0
         opacity_max1 = s% opacity_max1
         log_r_frac = s% opacity_max_log_r_frac
         min_cno_for_kap_limit = s% min_cno_for_kap_limit
         if (opacity_max0 < 100) then
            kh = maxloc(s% eps_nuc_categories(i_rate, icno, 1:s% nz), dim=1)
            if (s% eps_nuc_categories(i_rate, icno, kh) > min_cno_for_kap_limit) then
               log_r_in = s% lnr(kh)/ln10
               log_r_out = log_r_in + log_r_frac*(s% lnr(1)/ln10 - log_r_in)
               log_r = s% lnr(k)/ln10
               if (log_r_out > log_r .and. log_r > log_r_in) then
                  frac = (log_r - log_r_in)/(log_r_out - log_r_in)
                  opacity_max = opacity_max0 + (opacity_max1 - opacity_max0)*frac
                  if (s% opacity(k) > opacity_max) then
                     s% opacity(k) = opacity_max
                     s% d_opacity_dlnd(k) = 0
                     s% d_opacity_dlnT(k) = 0
                  end if
               end if
            end if
         end if
         
         
         if (.false. .and. s% model_number == 1 .and. k == 431) then
            call show_stuff
            stop 'debug: do_kap_for_cell'
         end if

         
         contains
         
         subroutine show_stuff
            include 'formats.dek'
            real(dp) :: base_Z
            write(*,2) 'show opacity info'
            write(*,2) 'logT = ', k, log10_T
            write(*,2) 'logRho = ', k, log10_rho
            write(*,2) 'z = ', k, z
            write(*,2) 'xh = ', k, xh
            write(*,2) 'xc = ', k, xc
            write(*,2) 'xn = ', k, xn
            write(*,2) 'xo = ', k, xo
            write(*,2) 'xne = ', k, xne
            write(*,*)
            write(*,2) 'xhe = ', k, xhe
            write(*,2) 'xheavy = ', k, xheavy
            write(*,*)
            write(*,2) 'rho = ', k, s% rho(k)
            write(*,2) 'lnrho = ', k, s% lnd(k)
            write(*,2) 'T = ', k, s% T(k)
            write(*,2) 'lnT = ', k, s% lnT(k)
            write(*,2) 'logKap = ', k, log10(s% opacity(k))
            write(*,2) 'opacity = ', k, s% opacity(k)
            write(*,2) 'dlnkap_dlnd = ', k, dlnkap_dlnd
            write(*,2) 'dlnkap_dlnT = ', k, dlnkap_dlnT
            write(*,2) 'd_opacity_dlnd = ', k, s% d_opacity_dlnd(k)
            write(*,2) 'd_opacity_dlnT = ', k, s% d_opacity_dlnT(k)
            write(*,*)
            base_Z = s% base_Z
            if (base_Z < 0) base_Z = s% initial_z
            write(*,1) 'base_Z', base_Z
            write(*,1) 'dXC', metals(1) - s% base_fC*base_Z
            write(*,1) 'dXO', metals(3) - s% base_fO*base_Z
            write(*,*)
         end subroutine show_stuff
         
      end subroutine do_kap_for_cell
      
      
      subroutine eval_kap_Type2( &
            s, k, zbar, X, Zbase, dXC, dXO, log10_rho, log10_T, xa, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
         use kap_lib
         type (star_info), pointer :: s   
         integer, intent(in) :: k      
         real(dp), intent(in) :: zbar, X, Zbase, dXC, dXO, log10_rho, log10_T, xa(:), &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
         real(dp), intent(out) :: kap, dln_kap_dlnRho, dln_kap_dlnT
         integer, intent(out) :: ierr ! 0 means AOK.
         if (s% use_other_kap) then
            call s% other_kap_get_Type2( &
               s% id, k, s% other_kap_handle, zbar, X, Zbase, dXC, dXO, log10_rho, log10_T, & 
               s% species, s% chem_id, s% net_iso, xa, &
               kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
         else
            call kap_get_Type2( &
               s% kap_handle, zbar, X, Zbase, dXC, dXO, log10_rho, log10_T, & 
               lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
               kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
         end if
      end subroutine eval_kap_Type2
      
      
      subroutine eval_kap_Type1( &
            s, k, zbar, X_in, Z_in, q, Pgas_div_P, log10_rho, log10_T, xa, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
         use kap_lib
         type (star_info), pointer :: s         
         integer, intent(in) :: k      
         real(dp), intent(in) :: zbar, X_in, Z_in, log10_rho, log10_T, xa(:), &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, q, Pgas_div_P
         real(dp), intent(out) :: kap, dln_kap_dlnRho, dln_kap_dlnT
         integer, intent(out) :: ierr ! 0 means AOK.
         real(dp) :: X, Z
         include 'formats.dek'
         if (test_remove_Z .and. Pgas_div_P < 0.5 .and. q > 0.05 .and. q < 0.95) then
            X = X_in + Z_in
            Z = 0
            !write(*,1) 'q', q, Pgas_div_P, Z_in
         else
            X = X_in
            Z = Z_in
         end if
         if (s% use_other_kap) then
            call s% other_kap_get_Type1( &
               s% id, k, s% other_kap_handle, zbar, X, Z, log10_rho, log10_T, & 
               s% species, s% chem_id, s% net_iso, xa, &
               lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
               kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
         else
            call kap_get_Type1( &
               s% kap_handle, zbar, X, Z, log10_rho, log10_T, & 
               lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
               kap, dln_kap_dlnRho, dln_kap_dlnT, ierr)
         end if
      end subroutine eval_kap_Type1

      
      real(dp) function get1_kap( &
            s, k, zbar, xa, q, Pgas_div_P, log10_rho, log10_T, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            dlnkap_dlnd, dlnkap_dlnT, ierr)
         use utils_lib
         use num_lib
         use chem_def, only: ih1, ihe3, ihe4, chem_isos
         use star_utils, only: get_XYZ
         type (star_info), pointer :: s
         integer, intent(in) :: k
         real(dp), intent(in) :: zbar, q, Pgas_div_P, log10_rho, log10_T, xa(:), &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
         real(dp), intent(out) :: dlnkap_dlnd, dlnkap_dlnT
         integer, intent(out) :: ierr

         integer, parameter :: nmet = 5
         integer :: trace_level, i, iz
         real(dp) :: Z, dXC, dXO, opacity, opacity2, dlnkap_dlnd2, dlnkap_dlnT2
         integer, pointer :: net_iso(:)
         logical :: use_CO_enhanced_opacities
         real(dp) :: base_Z, xh, xhe, xc, xn, xo, xne, xheavy, metals(nmet)
         
         include 'formats.dek'
         
         get1_kap = -1d99
         
         net_iso => s% net_iso
         xc = 0; xn = 0; xo = 0; xne =0
         do i=1, s% species
            iz = floor(chem_isos% Z(s% chem_id(i)) + 1d-10)
            select case(iz)
               case (6)
                  xc = xc + xa(i)
               case (7)
                  xn = xn + xa(i)
               case (8)
                  xo = xo + xa(i)
               case (10)
                  xne = xne + xa(i)
            end select
         end do
         
         call get_XYZ(s, xa, xh, xhe, Z)

         use_CO_enhanced_opacities = s% use_CO_enhanced_opacities
         base_Z = s% base_Z
         
         if (q > s% min_q_for_special_kap &
               .or. log10_T < s% max_logT_for_special_kap) then
            use_CO_enhanced_opacities = s% special_kap_use_CO_enhanced
            if (use_CO_enhanced_opacities) then
               base_Z = s% special_kap_Z_base
               !write(*,2) 'special base_Z', k, base_Z
            else
               xh = s% special_kap_xh
               xhe = s% special_kap_xhe
               xc = s% special_kap_xc
               xn = s% special_kap_xn
               xo = s% special_kap_xo
               xne = s% special_kap_xne
            end if
         end if
         
         xheavy = min(1d0, max(0d0, z - (xc + xn + xo + xne)))         
         metals(:) = (/ xc, xn, xo, xne, xheavy /)

         if (s% use_simple_es_for_kap) then
            opacity = 0.2d0*(1 + xh)
            dlnkap_dlnd = 0
            dlnkap_dlnT = 0
         else if (use_CO_enhanced_opacities) then
            if (base_Z < 0) then
               write(*,*) 'you must set base_Z when using CO enhanced opacities'
               ierr = -1
               return
            end if
            dXC = metals(1) - s% base_fC*base_Z
            dXO = metals(3) - s% base_fO*base_Z
            call eval_kap_Type2( &
               s, k, zbar, xh, base_Z, dXC, dXO, log10_rho, log10_T, xa(:), &
               lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
               opacity, dlnkap_dlnd, dlnkap_dlnT, ierr)
         else
            Z = min( max(1 - (xh+xhe), 0d0), 0.1d0 )
            call eval_kap_Type1( &
               s, k, zbar, xh, Z, q, Pgas_div_P, log10_rho, log10_T, xa(:), &
               lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
               opacity, dlnkap_dlnd, dlnkap_dlnT, ierr)
         end if
                  
         get1_kap = opacity
      
      end function get1_kap
      
      

      end module opacities

