! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module mlt_info

      use const_def
      use num_lib
      use utils_lib
      use alert_lib, only: alert, bug_alert
      use star_private_def
      use mlt_def

      implicit none

      logical, parameter :: dbg = .false.
      integer, parameter :: kdbg = -1
      
      contains


      subroutine set_mlt_vars(s, nzlo, nzhi, ierr)
         use star_utils, only:foreach_cell
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         integer :: k, op_err
         include 'formats.dek'
         if (dbg) write(*, *) 'doing set_mlt_vars'
!$OMP PARALLEL DO PRIVATE(k,op_err) SCHEDULE(STATIC,10)
         do k = nzlo, nzhi
            op_err = 0
            call do1_mlt(s,k,s% mixing_length_alpha,op_err)
            if (op_err /= 0) ierr = op_err
         end do
!$OMP END PARALLEL DO
      end subroutine set_mlt_vars


      subroutine do1_mlt_eval(  &
            s, k, cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada,  &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term, &
            alpha_semiconvection, thermo_haline_coeff, &
            mixing_length_alpha, alt_scale_height, thermohaline_option, &
            use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            max_convective_velocity, tau, &
            mixing_type, mlt_basics, mlt_partials, ierr)
         use mlt_lib, only: mlt_eval
         type (star_info), pointer :: s 
         integer, intent(in) :: k        
         real(dp), intent(in) :: cgrav, m, r, T, Rho, L, P
         real(dp), intent(in) :: chiRho, chiT, Cp, Cv, X, opacity, grada, &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, gradr_factor
         logical, intent(in) :: use_Ledoux_criterion
         real(dp), intent(in) :: gradL_composition_term
         real(dp), intent(in) :: alpha_semiconvection, thermo_haline_coeff
         real(dp), intent(in) :: mixing_length_alpha
         logical, intent(in) :: alt_scale_height, use_Henyey_MLT
         character (len=*), intent(in) :: thermohaline_option 
         real(dp), intent(in) :: Henyey_y_param, Henyey_nu_param, max_convective_velocity, tau
         integer, intent(out) :: mixing_type ! none, convective, semiconvective, or salt finger
         real(dp), intent(out) :: mlt_basics(num_mlt_results)
         real(dp), intent(out) :: mlt_partials(num_mlt_partials,num_mlt_results)
            ! e.g., mlt_partials(mlt_dlnT,mlt_gradT) has partial wrt lnT of gradT
         integer, intent(out) :: ierr
         if (s% use_other_mlt) then
            call s% other_mlt(  &
               s% id, k, cgrav, m, r, T, rho, L, P, &
               chiRho, chiT, Cp, Cv, X, opacity, grada,  &
               gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
               gradr_factor, use_Ledoux_criterion, gradL_composition_term, &
               alpha_semiconvection, thermo_haline_coeff, &
               mixing_length_alpha, alt_scale_height, thermohaline_option, &
               use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
               max_convective_velocity, tau, &
               mixing_type, mlt_basics, mlt_partials, ierr)
         else
            call mlt_eval(  &
               cgrav, m, r, T, rho, L, P, &
               chiRho, chiT, Cp, Cv, X, opacity, grada,  &
               gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
               gradr_factor, use_Ledoux_criterion, gradL_composition_term, &
               alpha_semiconvection, thermo_haline_coeff, &
               mixing_length_alpha, alt_scale_height, thermohaline_option, &
               use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
               max_convective_velocity, tau, &
               mixing_type, mlt_basics, mlt_partials, ierr)
         end if
      end subroutine do1_mlt_eval
      

      subroutine do1_mlt(s, k, mixing_length_alpha, ierr) 
         ! get convection info for point k
         use mlt_def
         use mlt_lib
         use eos_def
         use chem_def, only: ih1
         use utils_lib, only: is_bad_num
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         real(dp), intent(in) :: mixing_length_alpha
         integer, intent(out) :: ierr

         real(dp) :: m, L, r, dlnm_dlnq, v0, thc, asc, max_convective_velocity, &
            a, b, Pgas_div_P_limit, da_dlnd, da_dlnT, db_dlnd, db_dlnT, &
            max_q_for_Pgas_div_P_limit, min_q_for_Pgas_div_P_limit, &
            mlt_basics(num_mlt_results), mlt_partials(num_mlt_partials, num_mlt_results), &
            alfa, beta, T_face, rho_face, P_face, Cv_face, gamma1_face, &
            chiRho_face, chiT_face, Cp_face, opacity_face, grada_face, v, &
            dlnmu_dlnrho, dlnmu_dlnT, gradr_factor, gradmu, f, xh_face, tau_face
         integer :: mixing_type, h1
         real(dp), parameter :: conv_vel_mach_limit = 0.9d0
         logical :: Schwarzschild_stable, Ledoux_stable

         include 'formats.dek'
         
         ierr = 0
         
         alfa = 0
                  
         m = s% m(k)
         dlnm_dlnq = 1
         r = s% r(k)
         L = s% L(k)
         
         if (is_bad_num(L)) then
            write(*,2) 'do1_mlt L', k, L
            stop 1
         end if

         ! alfa is the fraction coming from k; (1-alfa) from k-1.
         if (k == 1) then
            alfa = 1
         else
            alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
         end if
         beta = 1 - alfa
         
         h1 = s% net_iso(ih1)
         xh_face = 0
         tau_face = 1 ! this is only used for atmosphere integrations where tau < 2/3
         
         if (alfa == 1) then
         
            T_face = s% T(k)
            rho_face = s% rho(k)
            P_face = s% P(k)
            gamma1_face = s% gamma1(k)
            Cv_face = s% Cv(k)
            chiRho_face = s% chiRho(k)         
            chiT_face = s% chiT(k)         
            Cp_face = s% Cp(k)
            opacity_face = s% opacity(k)         
            grada_face = s% grada(k)
            dlnmu_dlnrho = s% d_eos_dlnd(i_mu,k)/s% mu(k)
            dlnmu_dlnT = s% d_eos_dlnT(i_mu,k)/s% mu(k)
            if (h1 /= 0) xh_face = s% xa(h1, k)
                        
         else
         
            T_face = alfa*s% T(k) + beta*s% T(k-1)
            rho_face = alfa*s% rho(k) + beta*s% rho(k-1)
            P_face = alfa*s% P(k) + beta*s% P(k-1)
            gamma1_face = alfa*s% gamma1(k) + beta*s% gamma1(k-1)
            Cv_face = alfa*s% Cv(k) + beta*s% Cv(k-1)
            chiRho_face = alfa*s% chiRho(k) + beta*s% chiRho(k-1)         
            chiT_face = alfa*s% chiT(k) + beta*s% chiT(k-1)         
            Cp_face = alfa*s% Cp(k) + beta*s% Cp(k-1)         
            opacity_face = alfa*s% opacity(k) + beta*s% opacity(k-1)         
            grada_face = alfa*s% grada(k) + beta*s% grada(k-1)
            dlnmu_dlnrho = &
               alfa*s% d_eos_dlnd(i_mu,k)/s% mu(k) + beta*s% d_eos_dlnd(i_mu,k-1)/s% mu(k-1)
            dlnmu_dlnT = &
               alfa*s% d_eos_dlnT(i_mu,k)/s% mu(k) + beta*s% d_eos_dlnT(i_mu,k-1)/s% mu(k-1)
            if (h1 /= 0) xh_face = alfa*s% xa(h1,k) + beta*s% xa(h1,k-1)
                        
         end if
         
         s% csound_at_face(k) = sqrt(gamma1_face*P_face/rho_face)
         
         thc = s% thermo_haline_coeff
         asc = s% alpha_semiconvection
         if (s% rotation_flag) then
            gradr_factor = s% ft_rot(k)/s% fp_rot(k)
            if (is_bad_num(gradr_factor)) then
               ierr = -1
               if (s% report_ierr) then
                  if (.false.) call show_stuff
                  write(*,2) 'do1_mlt_eval gradr_factor', k, gradr_factor
               end if
               return
            end if
         else
            gradr_factor = 1
         end if
         
         if (s% use_gradmu_alt_for_mlt) then
            gradmu = s% gradmu_alt(k)
         else if (s% use_gradmu_X_for_mlt) then
            gradmu = s% gradmu_X(k)
         else
            gradmu = s% gradmu(k)
         end if
         
         if (s% use_time_dependent_convection) then
            max_convective_velocity = s% upper_limit_for_mlt_vc(k)
         else
            max_convective_velocity = 1d99
         end if
         
         call do1_mlt_eval( &
            s, k, s% cgrav(k), m, r, T_face, rho_face, L, P_face, & 
            chiRho_face, chiT_face, Cp_face, Cv_face, xh_face, &
            opacity_face, grada_face, gradmu, &
            dlnmu_dlnT, dlnmu_dlnrho, s% grad_density(k), s% grad_temperature(k), &
            gradr_factor, s% use_Ledoux_criterion, &
            s% gradL_composition_term(k), &
            asc, thc, mixing_length_alpha, s% alt_scale_height_flag, s% thermohaline_option, & 
            s% use_Henyey_MLT, s% Henyey_MLT_y_param, s% Henyey_MLT_nu_param, &
            max_convective_velocity, tau_face, & 
            mixing_type, mlt_basics, mlt_partials, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) then
               if (.false.) call show_stuff
               write(*,*) 'ierr in do1_mlt_eval for k', k
            end if
            return
         end if
         
         s% mlt_mixing_type(k) = mixing_type
         s% grada_at_face(k) = grada_face
         s% gradr(k) = mlt_basics(mlt_gradr)
         s% mlt_mixing_length(k) = mlt_basics(mlt_Lambda)
         s% mlt_D(k) = mlt_basics(mlt_D)
         s% mlt_vc(k) = mlt_basics(mlt_convection_velocity)
         s% mlt_Gamma(k) = mlt_basics(mlt_Gamma)
         s% scale_height(k) = mlt_basics(mlt_scale_height)
         
         !if (s% scale_height(k) < 0.99d0*P_face/(rho_face*s% cgrav(k)*m/r**2)) then
            !write(*,2) 'reduced scale height', k, s% scale_height(k), &
            !   P_face/(rho_face*s% cgrav(k)*m/r**2)
            !s% scale_height(k) = P_face/(rho_face*s% cgrav(k)*m/r**2)
         !end if

         s% mlt_cdc(k) = mlt_basics(mlt_D)*((pi4*r**2*rho_face)**2)
         if (is_bad_num(s% mlt_cdc(k)) .or. s% mlt_cdc(k) < 0 .or. &
				 is_bad_num(s% mlt_vc(k)) .or. s% mlt_vc(k) < 0) then
            if (s% report_ierr) then
               write(*,2) 's% mlt_cdc(k)', k, s% mlt_cdc(k)
               if (.false.) then
!$omp critical
                  call show_stuff
                  write(*,*)
!$omp end critical
               end if
            end if
            ierr = -1
            return
            stop 'mlt info'
         end if


         s% gradL(k) = mlt_basics(mlt_gradL)

         Schwarzschild_stable = (s% gradr(k) < grada_face)
         Ledoux_stable = (s% gradr(k) < s% gradL(k))
         
         s% d_gradr_dlnR(k) = mlt_partials(mlt_dlnR, mlt_gradr)
         s% d_gradr_dL(k) = mlt_partials(mlt_dL, mlt_gradr)
         
         s% d_gradr_dlnd00(k) = alfa*( &
            mlt_partials(mlt_dlnd, mlt_gradr) + &
            mlt_partials(mlt_dP, mlt_gradr)*s% P(k)*s% chiRho(k) + &
            mlt_partials(mlt_dchiRho, mlt_gradr)*s% d_eos_dlnd(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_gradr)*s% d_eos_dlnd(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_gradr)*s% d_eos_dlnd(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_gradr)*s% d_opacity_dlnd(k) + &
            mlt_partials(mlt_dgrada, mlt_gradr)*s% d_eos_dlnd(i_grad_ad, k))
         
         s% d_gradr_dlnT00(k) = alfa*( &
            mlt_partials(mlt_dlnT, mlt_gradr) + &
            mlt_partials(mlt_dP, mlt_gradr)*s% P(k)*s% chiT(k) + &
            mlt_partials(mlt_dchiRho, mlt_gradr)*s% d_eos_dlnT(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_gradr)*s% d_eos_dlnT(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_gradr)*s% d_eos_dlnT(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_gradr)*s% d_opacity_dlnT(k) + &
            mlt_partials(mlt_dgrada, mlt_gradr)*s% d_eos_dlnT(i_grad_ad, k))

         if (k > 1) then

            s% d_gradr_dlndm1(k) = beta*( &
               mlt_partials(mlt_dlnd, mlt_gradr) + &
               mlt_partials(mlt_dP, mlt_gradr)*s% P(k-1)*s% chiRho(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_gradr)*s% d_eos_dlnd(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_gradr)*s% d_eos_dlnd(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_gradr)*s% d_eos_dlnd(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_gradr)*s% d_opacity_dlnd(k-1) + &
               mlt_partials(mlt_dgrada, mlt_gradr)*s% d_eos_dlnd(i_grad_ad, k-1))
            
            s% d_gradr_dlnTm1(k) = beta*( &
               mlt_partials(mlt_dlnT, mlt_gradr) + &
               mlt_partials(mlt_dP, mlt_gradr)*s% P(k-1)*s% chiT(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_gradr)*s% d_eos_dlnT(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_gradr)*s% d_eos_dlnT(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_gradr)*s% d_eos_dlnT(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_gradr)*s% d_opacity_dlnT(k-1) + &
               mlt_partials(mlt_dgrada, mlt_gradr)*s% d_eos_dlnT(i_grad_ad, k-1))
            
         end if
         
         s% conv_dP_term(k) = mlt_basics(mlt_conv_dP_term)
         s% d_conv_dP_term_dlnR(k) = mlt_partials(mlt_dlnR, mlt_conv_dP_term)
         s% d_conv_dP_term_dL(k) = mlt_partials(mlt_dL, mlt_conv_dP_term)
         
         s% d_conv_dP_term_dlnd00(k) = alfa*( &
            mlt_partials(mlt_dlnd, mlt_conv_dP_term) + &
            mlt_partials(mlt_dP, mlt_conv_dP_term)*s% P(k)*s% chiRho(k) + &
            mlt_partials(mlt_dchiRho, mlt_conv_dP_term)*s% d_eos_dlnd(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_conv_dP_term)*s% d_eos_dlnd(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_conv_dP_term)*s% d_eos_dlnd(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_conv_dP_term)*s% d_opacity_dlnd(k) + &
            mlt_partials(mlt_dgrada, mlt_conv_dP_term)*s% d_eos_dlnd(i_grad_ad, k))
         
         s% d_conv_dP_term_dlnT00(k) = alfa*( &
            mlt_partials(mlt_dlnT, mlt_conv_dP_term) + &
            mlt_partials(mlt_dP, mlt_conv_dP_term)*s% P(k)*s% chiT(k) + &
            mlt_partials(mlt_dchiRho, mlt_conv_dP_term)*s% d_eos_dlnT(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_conv_dP_term)*s% d_eos_dlnT(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_conv_dP_term)*s% d_eos_dlnT(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_conv_dP_term)*s% d_opacity_dlnT(k) + &
            mlt_partials(mlt_dgrada, mlt_conv_dP_term)*s% d_eos_dlnT(i_grad_ad, k))

         if (k > 1) then

            s% d_conv_dP_term_dlndm1(k) = beta*( &
               mlt_partials(mlt_dlnd, mlt_conv_dP_term) + &
               mlt_partials(mlt_dP, mlt_conv_dP_term)*s% P(k-1)*s% chiRho(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_conv_dP_term)*s% d_eos_dlnd(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_conv_dP_term)*s% d_eos_dlnd(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_conv_dP_term)*s% d_eos_dlnd(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_conv_dP_term)*s% d_opacity_dlnd(k-1) + &
               mlt_partials(mlt_dgrada, mlt_conv_dP_term)*s% d_eos_dlnd(i_grad_ad, k-1))
            
            s% d_conv_dP_term_dlnTm1(k) = beta*( &
               mlt_partials(mlt_dlnT, mlt_conv_dP_term) + &
               mlt_partials(mlt_dP, mlt_conv_dP_term)*s% P(k-1)*s% chiT(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_conv_dP_term)*s% d_eos_dlnT(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_conv_dP_term)*s% d_eos_dlnT(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_conv_dP_term)*s% d_eos_dlnT(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_conv_dP_term)*s% d_opacity_dlnT(k-1) + &
               mlt_partials(mlt_dgrada, mlt_conv_dP_term)*s% d_eos_dlnT(i_grad_ad, k-1))
            
         end if
         
         f = s% adjust_mlt_gradT_fraction(k)
         if (f >= 0 .and. f <= 1) then ! replace gradT by combo of grada_at_face and gradr
         
            s% gradT(k) = f*s% grada_at_face(k) + (1-f)*s% gradr(k)
            
            s% d_gradT_dlnR(k) = (1-f)*s% d_gradr_dlnR(k)
            s% d_gradT_dL(k) = (1-f)*s% d_gradr_dL(k)

            s% d_gradT_dlnd00(k) = f*alfa*s% d_eos_dlnd(i_grad_ad, k) + (1-f)*s% d_gradr_dlnd00(k)
            s% d_gradT_dlnT00(k) = f*alfa*s% d_eos_dlnT(i_grad_ad, k) + (1-f)*s% d_gradr_dlnT00(k)

            if (k > 1) then               
               s% d_gradT_dlndm1(k) = f*beta*s% d_eos_dlnd(i_grad_ad, k-1) + (1-f)*s% d_gradr_dlndm1(k)
               s% d_gradT_dlnTm1(k) = f*beta*s% d_eos_dlnT(i_grad_ad, k-1) + (1-f)*s% d_gradr_dlndm1(k)               
            end if         
         
         else
         
            s% gradT(k) = mlt_basics(mlt_gradT)
            s% d_gradT_dlnR(k) = mlt_partials(mlt_dlnR, mlt_gradT)
            s% d_gradT_dL(k) = mlt_partials(mlt_dL, mlt_gradT)
                            
            s% d_gradT_dlnd00(k) = alfa*( &
               mlt_partials(mlt_dlnd, mlt_gradT) + &
               mlt_partials(mlt_dP, mlt_gradT)*s% P(k)*s% chiRho(k) + &
               mlt_partials(mlt_dchiRho, mlt_gradT)*s% d_eos_dlnd(i_chiRho, k) + &
               mlt_partials(mlt_dchiT, mlt_gradT)*s% d_eos_dlnd(i_chiT, k) + &
               mlt_partials(mlt_dCp, mlt_gradT)*s% d_eos_dlnd(i_Cp, k) + &
               mlt_partials(mlt_dopacity, mlt_gradT)*s% d_opacity_dlnd(k) + &
               mlt_partials(mlt_dgrada, mlt_gradT)*s% d_eos_dlnd(i_grad_ad, k))
      
            s% d_gradT_dlnT00(k) = alfa*( &
               mlt_partials(mlt_dlnT, mlt_gradT) + &
               mlt_partials(mlt_dP, mlt_gradT)*s% P(k)*s% chiT(k) + &
               mlt_partials(mlt_dchiRho, mlt_gradT)*s% d_eos_dlnT(i_chiRho, k) + &
               mlt_partials(mlt_dchiT, mlt_gradT)*s% d_eos_dlnT(i_chiT, k) + &
               mlt_partials(mlt_dCp, mlt_gradT)*s% d_eos_dlnT(i_Cp, k) + &
               mlt_partials(mlt_dopacity, mlt_gradT)*s% d_opacity_dlnT(k) + &
               mlt_partials(mlt_dgrada, mlt_gradT)*s% d_eos_dlnT(i_grad_ad, k))
            
            if (k > 1) then

               s% d_gradT_dlndm1(k) = beta*( &
                  mlt_partials(mlt_dlnd, mlt_gradT) + &
                  mlt_partials(mlt_dP, mlt_gradT)*s% P(k-1)*s% chiRho(k-1) + &
                  mlt_partials(mlt_dchiRho, mlt_gradT)*s% d_eos_dlnd(i_chiRho, k-1) + &
                  mlt_partials(mlt_dchiT, mlt_gradT)*s% d_eos_dlnd(i_chiT, k-1) + &
                  mlt_partials(mlt_dCp, mlt_gradT)*s% d_eos_dlnd(i_Cp, k-1) + &
                  mlt_partials(mlt_dopacity, mlt_gradT)*s% d_opacity_dlnd(k-1) + &
                  mlt_partials(mlt_dgrada, mlt_gradT)*s% d_eos_dlnd(i_grad_ad, k-1))
         
               s% d_gradT_dlnTm1(k) = beta*( &
                  mlt_partials(mlt_dlnT, mlt_gradT) + &
                  mlt_partials(mlt_dP, mlt_gradT)*s% P(k-1)*s% chiT(k-1) + &
                  mlt_partials(mlt_dchiRho, mlt_gradT)*s% d_eos_dlnT(i_chiRho, k-1) + &
                  mlt_partials(mlt_dchiT, mlt_gradT)*s% d_eos_dlnT(i_chiT, k-1) + &
                  mlt_partials(mlt_dCp, mlt_gradT)*s% d_eos_dlnT(i_Cp, k-1) + &
                  mlt_partials(mlt_dopacity, mlt_gradT)*s% d_opacity_dlnT(k-1) + &
                  mlt_partials(mlt_dgrada, mlt_gradT)*s% d_eos_dlnT(i_grad_ad, k-1))

            end if
         
         end if

         s% gradT_sub_grada(k) =  s% gradT(k) - grada_face

         !if (.false. .and. abs(T_face-3.515d6) < 1d3  .and. mixing_type == thermo_haline_mixing) then
         if (is_bad_num(s% gradT(k))) then
            if (s% report_ierr) then
               write(*,2) 's% gradT(k)', k, s% gradT(k)
               if (.false.) then
!$omp critical
                  call show_stuff
!$omp end critical
               end if
               return
               stop 'mlt info'
            end if
            ierr = -1
            return
            stop 'mlt info'
         end if
         
         
         if (.false.) then ! switch closer to grada when Pgas << Ptotal
         
            Pgas_div_P_limit = 0.3
            max_q_for_Pgas_div_P_limit = 0.99
            min_q_for_Pgas_div_P_limit = 0.50
            if (s% Pgas(k) < s% P(k)*Pgas_div_P_limit .and. &
                  s% q(k) <= max_q_for_Pgas_div_P_limit .and. &
                  s% q(k) >= min_q_for_Pgas_div_P_limit) then
               
               b = s% Pgas(k)/(s% P(k)*Pgas_div_P_limit)
               db_dlnd = -b*s% chiRho(k)
               db_dlnT = b*(4 - s% chiT(k))
               a = 1 - b
               da_dlnd = -db_dlnd
               da_dlnT = -db_dlnT
            
               s% gradT(k) = a*s% grada(k) + b*s% gradT(k)
               s% d_gradT_dlnd00(k) = &
                  a*s% d_eos_dlnd(i_grad_ad, k) + da_dlnd*s% grada(k) + &
                  b*s% d_gradT_dlnd00(k) + db_dlnd*s% gradT(k)               
               s% d_gradT_dlnT00(k) = &
                  a*s% d_eos_dlnT(i_grad_ad, k) + da_dlnT*s% grada(k) + &
                  b*s% d_gradT_dlnT00(k) + db_dlnT*s% gradT(k)     
               
               s% d_gradT_dlndm1(k) = b*s% d_gradT_dlndm1(k)
               s% d_gradT_dlnTm1(k) = b*s% d_gradT_dlnTm1(k)
               s% d_gradT_dlnR(k) = b*s% d_gradT_dlnR(k)
               s% d_gradT_dL(k) = b*s% d_gradT_dL(k)
            
               !if (k < 600) write(*,2) 'change to grada', k, a
            end if
         
         end if

         
         if (mixing_type == no_mixing) return
         
         if (.not. s% v_flag) return
         
         ! check for high velocities
         return ! DISABLE FOR NOW.
         
         v = s% v(k)
         v0 = s% csound(k)*conv_vel_mach_limit
         if (abs(v) < v0) return
         
         ! switch to radiative gradT for point with supersonic velocity
         call switch_to_radiative(s,k)
         call switch_to_no_mixing(s,k)
         
         
         contains


         subroutine show_stuff
            real(dp) :: vsem, Lambda, D, Dsem, Drad, radiative_conductivity
            include 'formats.dek'
            write(*,*)
            write(*,*) 'do1_mlt info for k, nz', k, s% nz
            write(*,*)
            write(*,1) 'cgrav =', s% cgrav(k)
            write(*,1) 'm =', m
            write(*,1) 'r =', r
            write(*,1) 'T =', T_face
            write(*,1) 'rho =', rho_face
            write(*,1) 'L =', L
            write(*,1) 'P =', P_face
            write(*,1) 'chiRho =', chiRho_face         
            write(*,1) 'chiT =', chiT_face
            write(*,1) 'Cp =', Cp_face
            write(*,1) 'Cv =', Cv_face
            write(*,1) 'opacity =', opacity_face
            write(*,1) 'grada =', grada_face
            write(*,*)
            write(*,1) 'gradmu =', gradmu
            write(*,1) 'dlnmu_dlnT =', dlnmu_dlnT
            write(*,1) 'dlnmu_dlnrho =', dlnmu_dlnrho
            write(*,1) 'grad_rho =', s% grad_density(k)
            write(*,1) 'grad_temp =', s% grad_temperature(k)
            write(*,*)
            write(*,1) 'gradr_factor =', gradr_factor
            if (s% use_Ledoux_criterion) then
               write(*,'(a50)') '         use_Ledoux_criterion = .true.'
            else
               write(*,'(a50)') '         use_Ledoux_criterion = .false.'
            end if
            write(*,1) 'gradL_composition_term =', &
               s% gradL_composition_term_factor*s% gradL_composition_term(k)
            write(*,1) 'alpha_semiconvection =', asc
            write(*,1) 'thermo_haline_coeff =', thc
            write(*,1) 'mixing_length_alpha =', mixing_length_alpha
            if (s% alt_scale_height_flag) then
               write(*,'(a50)') '         alt_scale_height = .true.'
            else
               write(*,'(a50)') '         alt_scale_height = .false.'
            end if
            write(*,*)
            if (s% use_Henyey_MLT) then
               write(*,'(a50)') '         use_Henyey_MLT = .true.'
            else
               write(*,'(a50)') '         use_Henyey_MLT = .false.'
            end if
            write(*,1) 'Henyey_y_param =', s% Henyey_MLT_y_param
            write(*,1) 'Henyey_nu_param =', s% Henyey_MLT_nu_param
            write(*,*)
            write(*,*) '--------------------------------------'
            write(*,*)
            write(*,*)
            write(*,*)

            write(*,1) 'logRho =', s% lnd(k)/ln10
            write(*,1) 'logT =', s% lnT(k)/ln10
            write(*,*)
            write(*,*)
            write(*,3) 'k, nz', k, s% nz
            
            if (ierr /= 0) return
            write(*,1) 's% gradr(k)', s% gradr(k)
            write(*,1) 's% gradT(k)', s% gradT(k)
            write(*,1) 's% gradL(k)', s% gradL(k)
            write(*,1) 's% gradL(k) - grada_at_face', s% gradL(k) - grada_face
            write(*,*)
            write(*,1) 's% mlt_D(k)', s% mlt_D(k)
            write(*,1) 's% mlt_vc(k)', s% mlt_vc(k)
            write(*,2) 's% mlt_mixing_type(k)', s% mlt_mixing_type(k)
            write(*,1) 's% mlt_mixing_length(k)', s% mlt_mixing_length(k)
            write(*,1) 's% d_gradT_dlnd00(k)', s% d_gradT_dlnd00(k)
            write(*,1) 's% d_gradT_dlnT00(k)', s% d_gradT_dlnT00(k)
            write(*,1) 's% d_gradT_dlndm1(k)', s% d_gradT_dlndm1(k)
            write(*,1) 's% d_gradT_dlnTm1(k)', s% d_gradT_dlnTm1(k)
            write(*,1) 's% d_gradT_dlnR(k)', s% d_gradT_dlnR(k)
            write(*,1) 's% d_gradT_dL(k)', s% d_gradT_dL(k)
            write(*,*)
            write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
            write(*,2) 's% eps_nuc(k)', k, s% eps_nuc(k)
            if (k > 1) then
               write(*,2) 's% eps_grav(k-1)', k-1, s% eps_grav(k-1)
               write(*,2) 's% eps_nuc(k-1)', k-1, s% eps_nuc(k-1)
            end if
            write(*,*)
            
            write(*,*) 'Schwarzschild_stable', Schwarzschild_stable
            write(*,*) 'Ledoux_stable', Ledoux_stable
            write(*,*)
            
            !if (s% T(k) < 1d-20) stop 'bogus T: do1_mlt'
            
            !stop 'do1_mlt'
            
         end subroutine show_stuff

      end subroutine do1_mlt


      subroutine set_grads(s, ierr)
         use star_utils, only: smooth
         use chem_def, only: ih1
         type (star_info), pointer :: s         
         integer, intent(out) :: ierr
         integer :: k, i, nz, nweight, h1, k_bot, k_top
         logical :: in_region
         real(dp) :: alfa, beta_factor, chiRho, chiT, lnd_limit, avg
         real(dp), pointer, dimension(:) :: &
            dlnmu, dlnmu_alt, dlnmu_x, dlnP, dlnd, dlnT, beta
            
         include 'formats.dek'
         
         ierr = 0
         nz = s% nz
         h1 = s% net_iso(ih1)
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k = 2, nz
            dlnP(k) = s% lnP(k-1) - s% lnP(k)
            dlnd(k) = s% lnd(k-1) - s% lnd(k)
            dlnT(k) = s% lnT(k-1) - s% lnT(k)
            dlnmu(k) = 2*(s% mu(k-1) - s% mu(k))/(s% mu(k-1) + s% mu(k))
            dlnmu_alt(k) = 2*(s% mu_alt(k-1) - s% mu_alt(k))/(s% mu_alt(k-1) + s% mu_alt(k))
            beta(k) = s% Pgas(k)/s% P(k)
            if (h1 /= 0) then
               dlnmu_X(k) = -(s% xa(h1,k-1) - s% xa(h1,k))/ &
                           (0.5d0*(s% xa(h1,k-1) + s% xa(h1,k)) + 0.6d0)
            else
               dlnmu_X(k) = 0
            end if
         end do
         dlnP(1) = dlnP(2)
         dlnd(1) = dlnd(2)
         dlnT(1) = dlnT(2)
         dlnmu(1) = dlnmu(2)
         dlnmu_alt(1) = dlnmu_alt(2)
         dlnmu_X(1) = dlnmu_X(2)
         beta(1) = beta(2)
         
         call smooth(dlnP,nz)
         call smooth(dlnd,nz)
         call smooth(dlnT,nz)
         call smooth(dlnmu,nz)
         call smooth(dlnmu_alt,nz)
         call smooth(dlnmu_X,nz)
         call smooth(beta,nz)
         
         s% gradmu(1) = 0
         s% gradmu_alt(1) = 0
         s% gradmu_X(1) = 0
         s% grad_density(1) = 0
         s% grad_temperature(1) = 0
         do k = 2, nz
            if (dlnP(k) >= 0) then
               s% gradmu(k) = 0
               s% gradmu_alt(k) = 0
               s% gradmu_X(k) = 0
               s% grad_density(k) = 0
               s% grad_temperature(k) = 0
            else
               s% gradmu(k) = dlnmu(k)/dlnP(k)
               s% gradmu_alt(k) = dlnmu_alt(k)/dlnP(k)
               s% gradmu_X(k) = dlnmu_X(k)/dlnP(k)
               s% grad_density(k) = dlnd(k)/dlnP(k)
               s% grad_temperature(k) = dlnT(k)/dlnP(k)
            end if
         end do
         
         call smooth(s% gradmu,nz)
         call smooth(s% gradmu_alt,nz)
         call smooth(s% gradmu_X,nz)
         call smooth(s% grad_density,nz)
         call smooth(s% grad_temperature,nz)
         
         lnd_limit = s% gradL_comp_term_logRho_switch*ln10
         do k = 2, nz
            if (s% lnd(k) < lnd_limit) then ! ideal gas plus radiation
               beta_factor = beta(k)/(4d0 - 3d0*beta(k))
               if (s% use_gradmu_alt_for_mlt) then
                  s% gradL_composition_term(k) = s% gradmu_alt(k)*beta_factor
               else if (s% use_gradmu_X_for_mlt) then
                  s% gradL_composition_term(k) = s% gradmu_X(k)*beta_factor
               else
                  s% gradL_composition_term(k) = s% gradmu(k)*beta_factor
               end if
            else ! general eos
               alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
               chiRho = alfa*s% chiRho(k) + (1-alfa)*s% chiRho(k-1)         
               chiT = alfa*s% chiT(k) + (1-alfa)*s% chiT(k-1)         
               s% gradL_composition_term(k) = &
                  s% grad_temperature(k) + s% grad_density(k)*chiRho/chiT - 1/chiT
            end if
         end do
         s% gradL_composition_term(1) = s% gradL_composition_term(2)
         
         if (s% gradL_composition_term_factor /= 1) &
            s% gradL_composition_term(1:nz) = &
               s% gradL_composition_term_factor*s% gradL_composition_term(1:nz)

         ! may want to reduce low frequency noise around 0.0 by repeated smoothing
         do i = 1, s% nsmooth_gradL_composition_term
            call smooth(s% gradL_composition_term,nz)
         end do
         
         call dealloc
         
         contains
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            call get_work_array(s, dlnmu, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnmu_alt, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnmu_x, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnP, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnd, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnT, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, beta, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            call return_work_array(s, dlnmu, 'mlt')            
            call return_work_array(s, dlnmu_alt, 'mlt')            
            call return_work_array(s, dlnmu_X, 'mlt')            
            call return_work_array(s, dlnP, 'mlt')            
            call return_work_array(s, dlnd, 'mlt')            
            call return_work_array(s, dlnT, 'mlt')            
            call return_work_array(s, beta, 'mlt')            
         end subroutine dealloc
         
      end subroutine set_grads
      

      subroutine switch_to_no_mixing(s,k)
         type (star_info), pointer :: s
         integer, intent(in) :: k
         !write(*,*) 'switch_to_no_mixing', k
         s% mlt_mixing_type(k) = no_mixing
         s% mlt_mixing_length(k) = 0
         s% mlt_D(k) = 0
         s% mlt_vc(k) = 0
      end subroutine switch_to_no_mixing
      
      
      subroutine switch_to_radiative(s,k)
         type (star_info), pointer :: s
         integer, intent(in) :: k
         !write(*,*) 'switch_to_radiative', k
         s% gradT(k) = s% gradr(k)
         s% d_gradT_dlnd00(k) = s% d_gradr_dlnd00(k)
         s% d_gradT_dlnT00(k) = s% d_gradr_dlnT00(k)
         s% d_gradT_dlndm1(k) = s% d_gradr_dlndm1(k)
         s% d_gradT_dlnTm1(k) = s% d_gradr_dlnTm1(k)
         s% d_gradT_dlnR(k) = s% d_gradr_dlnR(k)
         s% d_gradT_dL(k) = s% d_gradr_dL(k)
      end subroutine switch_to_radiative
      
      
      subroutine switch_to_adiabatic(s,k)
         use eos_def, only: i_grad_ad
         type (star_info), pointer :: s
         integer, intent(in) :: k
         !write(*,*) 'switch_to_adiabatic', k
         s% gradT(k) = s% grada(k)
         s% d_gradT_dlnd00(k) = s% d_eos_dlnd(i_grad_ad, k)
         s% d_gradT_dlnT00(k) = s% d_eos_dlnT(i_grad_ad, k)
         s% d_gradT_dlndm1(k) = 0
         s% d_gradT_dlnTm1(k) = 0
         s% d_gradT_dlnR(k) = 0
         s% d_gradT_dL(k) = 0
      end subroutine switch_to_adiabatic
      
      
      subroutine set_mlt_vc_max(s, ierr)
         use interp_1d_lib
         use interp_1d_def
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer, parameter :: nwork = pm_work_size
         integer :: j, k, nz, prev_nz
         real(dp), pointer :: f(:,:), prev_conv_vel(:), prev_m(:), work(:,:)
         real(dp) :: M_center_older, dt, m, prev_vc, tau
         
         logical, parameter :: dbg = .false.

         include 'formats.dek'
         
         ierr = 0
         if (.not. s% use_time_dependent_convection) return
         dt = s% dt
         nz = s% nz
         s% conv_vel_frac_current(1:nz) = 1
         s% upper_limit_for_mlt_vc(1:nz) = 1d99
         if (s% generations < 3 .or. dt <= secyer) return
         
         ! info for previous model convection velocities
            ! nz_older, conv_vel_older, mixing_type_older
            ! recall m(k) = M_center + q(k)*xmstar; M_center = mstar - xmstar
            ! so to get previous mass coordinates, need q_older, mstar_older, xmstar_older
         
         ! setup to interpolate by mass -- must remove repeated mass coords         
         prev_nz = s% nz_older
         allocate(prev_m(prev_nz), f(4,prev_nz), prev_conv_vel(nz), work(prev_nz,nwork))
         M_center_older = s% mstar_older - s% xmstar_older
         j = 1
         prev_m(j) = s% mstar_older
         do k=2,prev_nz
            m = M_center_older + s% q_older(k)*s% xmstar_older
            if (m >= prev_m(j)) then
               if (dbg) write(*,3) 'm not decreasing inward', j, k, m, prev_m(j)
               cycle ! m must decrease inward
            end if
            j = j+1
            prev_m(j) = m
            f(1,j) = s% conv_vel_older(k)
         end do
         ! j is the number of old points for interpolation by mass
         
         call setup_prev_conv_vel(ierr)
         if (ierr /= 0) return
         
         do k=1,nz
            prev_vc = max(1d3,prev_conv_vel(k))
            tau = s% scale_height(k)/prev_vc
            if (tau > 1d2*dt) then
               s% upper_limit_for_mlt_vc(k) = 1d99
            else
               s% upper_limit_for_mlt_vc(k) = prev_vc*(1d0 + exp(dt/tau))
            end if
         end do

         call dealloc
         
         
         contains
         

         
         subroutine dealloc
            deallocate(prev_m, f, prev_conv_vel, work)
         end subroutine dealloc
         
         
         subroutine setup_prev_conv_vel(ierr)
            integer, intent(out) :: ierr
            integer :: k
            ierr = 0
            call interp_pm(prev_m, j, f, nwork, work, ierr) ! make interpolant
            if (ierr /= 0) then
               write(*,*) 'convection_time_smoothing failed in interp_pm'
               call dealloc
               return
            end if
            ! interpolate prev conv vel to current mass locations
            call interp_values(prev_m, j, f, nz, s% m, prev_conv_vel, ierr)
            if (ierr /= 0) then
               write(*,*) 'convection_time_smoothing failed in interp_values'
               call dealloc
               return
            end if
            do k=1,nz
               if (prev_conv_vel(k) < 1d-2) prev_conv_vel(k) = 0
            end do
         end subroutine setup_prev_conv_vel


      end subroutine set_mlt_vc_max
      
      
      end module mlt_info
