! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module micro
      
      use star_private_def
      use alert_lib,only:alert
      use utils_lib,only:is_bad_num
      use const_def
      use star_utils, only: foreach_cell
      
      implicit none

      logical, parameter :: dbg = .false.
      


      contains
      
      
      subroutine set_micro_vars(s, nzlo, nzhi, skip_net, skip_neu, skip_kap, ierr)
         use const_def,only:ln10,clight
         use opacities, only: do_opacities, set_kap_params, do_kap_for_cell
         use star_utils, only: update_time, total_times
         use net_lib, only: net_work_size
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: skip_net, skip_neu, skip_kap
         integer, intent(out) :: ierr
         
         integer :: j, k, op_err
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         
         include 'formats.dek'

         ierr = 0
         if (dbg) then
            write(*,*) 
            write(*,*) 'set_micro_vars'
            write(*,*) 'nzlo', nzlo
            write(*,*) 'nzhi', nzhi
            write(*,*) 'skip_net', skip_net
            write(*,*) 'skip_kap', skip_kap
         end if

         !call debug('before eos')
         
         if (dbg) write(*,*) 'call do_eos'
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         call do_eos(s,nzlo,nzhi,ierr)
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_eos)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'do_eos returned ierr', ierr
            return
         end if

         !call debug('before neu kap')

         if (.not. (skip_kap .and. skip_neu)) then
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
!$OMP PARALLEL DO PRIVATE(k,op_err) SCHEDULE(STATIC,10)
            do k = nzlo, nzhi
               op_err = 0
               call do1_neu_kap(s,k,op_err)
               if (op_err /= 0) ierr = op_err
            end do
!$OMP END PARALLEL DO
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_kap)
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'do_neu returned ierr', ierr
               return
            end if
         end if
         
         if (ierr /= 0) return
         
         
         !call debug('before net')

         if (.not. skip_net) then
            if (dbg) write(*,*) 'call do_net'
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call do_net(s,nzlo,nzhi,ierr)
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_net)
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'do_net returned ierr', ierr
               return
            end if
         end if
         
         !call debug('after net')
         
         
         contains
         
         
         subroutine debug(str) 
            use chem_def
            character (len=*), intent(in) :: str
            integer :: k, j
            include 'formats.dek'
            k = 1469
            do j=1,1 !s% species
               if (.true. .or. s% xa(j,k) > 1d-9) &
                  write(*,1) trim(str) // ' xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', s% xa(j,k)
            end do
         end subroutine debug
         
         
         subroutine do1_neu_kap(s,k,ierr)
            use neu_def,only:Tmin_neu
            type (star_info), pointer :: s         
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            if (s% T(k) >= Tmin_neu) then
               call do_neu_for_cell(s,k,ierr)
            else
               call do_clear_neu_for_cell(s,k,ierr)
            end if
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'do_neu returned ierr', ierr
               return
            end if      
            if (skip_kap) return
            call do_kap_for_cell(s,k,ierr)
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'do_kap returned ierr', ierr
               return
            end if
         end subroutine do1_neu_kap


      end subroutine set_micro_vars
      
      
      subroutine do_eos(s,nzlo,nzhi,ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         logical, parameter :: use_omp = .true.
         ierr = 0
         if (dbg) write(*,*) 'do_eos call foreach_cell', nzlo, nzhi
         call foreach_cell(s,nzlo,nzhi,use_omp,do_eos_for_cell,ierr)
      end subroutine do_eos
      
      
      subroutine do_eos_for_cell(s,k,ierr)
         use const_def
         use chem_def
         use chem_lib
         use eos_def
         use eos_lib
         use num_lib,only:safe_log
         use net_def,only:net_general_info
         use utils_lib,only:is_bad_num

         type (star_info), pointer :: s         
         integer, intent(in) :: k
         integer, intent(out) :: ierr
         
         real(dp), dimension(num_eos_basic_results) :: res, res_a, res_b, d_dlnd, d_dlnT
         real(dp) :: z, xh, xhe3, xhe4, xhe, abar, zbar, z2bar, ye, sumx, &
            dabar_dx(s% species), dzbar_dx(s% species), dx, dxh_a, dxh_b, &
            Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas
         integer, pointer :: net_iso(:)
         integer :: j, h1, h2, he3, he4, species
         real(dp), parameter :: epsder = 1d-4, Z_limit = 0.5d0
         
         logical, parameter :: testing = .false.
            
         include 'formats.dek'
         
         ierr = 0
         
         net_iso => s% net_iso
         species = s% species
         call composition_info(species, s% chem_id, s% xa(1:species,k), s% X(k), s% Y(k), &
               s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
               s% approx_abar(k), s% approx_zbar(k), sumx, dabar_dx, dzbar_dx)  
         s% mu_alt(k) = s% abar(k)/(1 + s% zbar(k)) ! mu from composition assuming complete ionization
         
         if (.false. .and. k == 1469) then
            j = 2
            write(*,1) 'do_eos_for_cell xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', &
                     s% xa(j,k), s% abar(k)
         end if
         
         h1 = net_iso(ih1)
         h2 = net_iso(ih2)
         he3 = net_iso(ihe3)
         he4 = net_iso(ihe4)
         
         if (h1 /= 0) then
            xh = s% xa(h1,k)
         else
            xh = 0
         end if
         if (h2 /= 0) xh = xh + s% xa(h2,k)
         
         if (he3 /= 0) then
            xhe3 = s% xa(he3,k)
         else
            xhe3 = 0
         end if
         
         if (he4 /= 0) then
            xhe4 = s% xa(he4,k)
         else
            xhe4 = 0
         end if
         
         xhe = xhe3 + xhe4
         z = max(0d0,1d0-(xh+xhe))
         
         if (s% lnPgas_flag) then
            if (s% Pgas(k) == 0) then
               ierr = -1
               return
               
               write(*,2) 's% lnPgas(k)/ln10', k, s% lnPgas(k)/ln10
               stop 'do_eos_for_cell s% Pgas(k) == 0'
            end if
            
            !write(*,2) 'call eval_eosPT', k, s% lnPgas(k)/ln10, s% lnT(k)/ln10
            if (.false. .and. k == 17 .and. &
                  abs(s% lnPgas(k)/ln10 - 2.9752795854874510D+01) < 1d-8 .and. &
                  abs(s% lnT(k)/ln10 - 4.1795250171658598D+00) < 1d-8) then
               call write_info
               stop 'debug: eos'
            end if
            
            call eval_eosPT( &
               s, k, z, xh, s% abar(k), s% zbar(k), s% approx_abar(k), s% approx_zbar(k), s% xa(:,k), &
               s% Pgas(k), s% lnPgas(k)/ln10, s% T(k), s% lnT(k)/ln10, & 
               Rho, log10Rho, s% dlnRho_dlnPgas_const_T(k), s% dlnRho_dlnT_const_Pgas(k), &
               res, s% d_eos_dlnd(:,k), s% d_eos_dlnT(:,k), ierr)
            s% lnd(k) = log10Rho*ln10
            s% rho(k) = Rho
            
            !if (k == 17) &
            !   write(*,2) 'logRho logT logPgas', k, s% lnd(k)/ln10, s% lnT(k)/ln10, s% lnPgas(k)/ln10

         else
            call eos_get( &
               s, k, z, xh, s% abar(k), s% zbar(k), s% approx_abar(k), s% approx_zbar(k), s% xa(:,k), &
               s% rho(k), s% lnd(k)/ln10, s% T(k), s% lnT(k)/ln10, &
               res, s% d_eos_dlnd(:,k), s% d_eos_dlnT(:,k), ierr)
            s% lnPgas(k) = res(i_lnPgas)
            s% Pgas(k) = exp(s% lnPgas(k))
         end if
         
         if (ierr /= 0) return
         
         call store_eos_for_cell(s, k, res, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) then
               call write_info
               stop 'debug: eos'
            end if
            return
         end if
         
         if (.false. .and. k == 1469) then
            call write_info
            write(*,2) 'eos s% grada(k)', k, s% grada(k)
         end if
         
         if (.false. .and. k == 110) then
            call write_info
            stop 'debug: eos'
         end if
         
         if (.false. .and. k == 269 .and. s% lnPgas_flag) then
            write(*,*) 'lnPgas_flag', s% lnPgas_flag
            write(*,2) 'rho', k, s% rho(k)
            write(*,2) 'logRho', k, s% lnd(k)/ln10
            write(*,2) 'Pgas', k, s% Pgas(k)
            write(*,2) 'logPgas', k, s% lnPgas(k)/ln10
            write(*,*)
            call write_info
            stop
         end if
            
         
         contains
         
         subroutine write_info
            1 format(a40,1pd26.16)
            write(*,*) 'do_eos_for_cell k', k
            write(*,*) 's% lnPgas_flag', s% lnPgas_flag
            write(*,1) 'z = ', z
            write(*,1) 'x = ', xh
            write(*,1) 'Xhe3 =', s% xa(he3,k)
            write(*,1) 'Xhe4 =', s% xa(he4,k)
            write(*,1) 'sum(xa) =', sum(s% xa(:,k))
            write(*,1) 'abar = ', s% abar(k)
            write(*,1) 'zbar = ', s% zbar(k)
            write(*,1) 'approx_abar = ', s% approx_abar(k)
            write(*,1) 'approx_zbar = ', s% approx_zbar(k)
            write(*,1) 'Pgas = ', s% Pgas(k)
            write(*,1) 'logPgas = ', s% lnPgas(k)/ln10
            write(*,1) 'rho = ', s% rho(k)
            write(*,1) 'logRho = ', s% lnd(k)/ln10
            write(*,1) 'T = ', s% T(k)
            write(*,1) 'log10_T = ', s% lnT(k)/ln10
            write(*,1) 'logQ = ', s% lnd(k)/ln10 - 2*s% lnT(k)/ln10 + 12
            write(*,1) 'logW = ', s% lnPgas(k)/ln10 - 4*s% lnT(k)/ln10
            write(*,*)
            if (ierr /= 0) return
            write(*,1) 'P = ', s% P(k)
            write(*,1) 'lnP = ', s% lnP(k)
            write(*,1) 'Prad = ', s% Prad(k)
            write(*,1) 'logS = ', s% lnS(k)/ln10
            write(*,1) 'logE = ', s% lnE(k)/ln10
            write(*,1) 'grada = ', s% grada(k)
            write(*,1) 'cv = ', s% cv(k)
            write(*,1) 'cp = ', s% cp(k)
            write(*,1) 'gamma1 = ', s% gamma1(k)
            write(*,1) 'gamma3 = ', s% gamma3(k)
            write(*,1) 'eta = ', s% eta(k)
            write(*,1) 'gam = ', s% gam(k)
            write(*,1) 'mu = ', s% mu(k)
            write(*,1) 'log_free_e = ', s% lnfree_e(k)/ln10
            write(*,1) 'chiRho = ', s% chiRho(k)
            write(*,1) 'chiT = ', s% chiT(k)
         end subroutine write_info
         
      end subroutine do_eos_for_cell
   
      
      subroutine eos_get( &
            s, k, z, xh, abar, zbar, approx_abar, approx_zbar, xa, &
            Rho, log10Rho, T, log10T, & 
            res, d_dlnd, d_dlnT, ierr)
         use eos_lib, only: eosDT_get
         use eos_def, only: num_eos_basic_results, i_Cp, i_Cv
         type (star_info), pointer :: s         
         integer, intent(in) :: k ! 0 means not being called for a particular cell
         real(dp), intent(in) :: &
            z, xh, abar, zbar, approx_abar, approx_zbar, xa(:), Rho, log10Rho, T, log10T
         real(dp), dimension(num_eos_basic_results), intent(out) :: res, d_dlnd, d_dlnT
         integer, intent(out) :: ierr
         
         include 'formats.dek'
         if (s% use_other_eos) then
            call s% other_eosDT_get( &
               s% id, k, s% other_eos_handle, z, xh, approx_abar, approx_zbar, &
               s% species, s% chem_id, s% net_iso, xa, &
               Rho, log10Rho, T, log10T, &
               res, d_dlnd, d_dlnT, ierr)
         else
            call eosDT_get( &
               s% eos_handle, z, xh, approx_abar, approx_zbar, &
               s% species, s% chem_id, s% net_iso, xa, &
               Rho, log10Rho, T, log10T, &
               res, d_dlnd, d_dlnT, ierr)
         end if
         if (ierr /= 0) then
            if (s% report_ierr) then
!$omp critical (star_eos_get)
               write(*,*) 'eos_get ierr', ierr
               write(*,2) 'k', k
               write(*,1) 'z', z
               write(*,1) 'xh', xh
               write(*,1) 'abar', abar
               write(*,1) 'zbar', zbar
               write(*,1) 'approx_abar', approx_abar
               write(*,1) 'approx_zbar', approx_zbar
               write(*,1) 'log10Rho', log10Rho
               write(*,1) 'log10T', log10T
!$omp end critical (star_eos_get)

               !stop 'eos_get'
               
            end if
            return
         end if
         if (.false. .and. (res(i_Cp) < 0 .or. res(i_Cv) < 0)) then
            write(*,*) 'eos_get ierr', ierr
            write(*,2) 'k', k
            write(*,1) 'res(i_Cp)', res(i_Cp)
            write(*,1) 'res(i_Cv)', res(i_Cv)
            write(*,1) 'z', z
            write(*,1) 'xh', xh
            write(*,1) 'abar', abar
            write(*,1) 'zbar', zbar
            write(*,1) 'log10Rho', log10Rho
            write(*,1) 'log10T', log10T
            stop 'debug: eos_get'
         end if
      end subroutine eos_get
      
      
      subroutine check_eos_res(s, k, res, z, xh, ierr)
         use eos_def
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         real(dp) :: res(num_eos_basic_results)
         real(dp) :: z, xh
         integer, intent(out) :: ierr
         1 format(a40,1pe26.16)
         ierr = 0
         if (abs(s% lnPgas(k) - res(i_lnPgas)) > 1d-5*res(i_lnPgas)) then
            write(*,*) 'failure for check_eos_res for lnPgas k', k
            write(*,1) 'res(i_lnPgas)', res(i_lnPgas)
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% lnS(k) - res(i_lnS)) > 1d-5*res(i_lnS)) then
            write(*,*) 'failure for check_eos_res for entropy k', k
            write(*,1) 'res(i_lnS)', res(i_lnS)
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% lnE(k) - res(i_lnE)) > 1d-5*res(i_lnE)) then
            write(*,*) 'failure for check_eos_res for energy k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% grada(k) - res(i_grad_ad)) > 1d-5*res(i_grad_ad)) then
            write(*,*) 'failure for check_eos_res for grada k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% dE_drho(k) - res(i_dE_drho)) > 1d-5*abs(res(i_dE_drho))) then
            write(*,*) 'failure for check_eos_res for dE_dRho k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% cv(k) - res(i_Cv)) > 1d-5*res(i_Cv)) then
            write(*,*) 'failure for check_eos_res for cv k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% cp(k) - res(i_cp)) > 1d-5*res(i_cp)) then
            write(*,*) 'failure for check_eos_res for cp k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% gamma1(k) - res(i_gamma1)) > 1d-5*res(i_gamma1)) then
            write(*,*) 'failure for check_eos_res for gamma1 k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% gamma3(k) - res(i_gamma3)) > 1d-5*res(i_gamma3)) then
            write(*,*) 'failure for check_eos_res for gamma3 k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% chiRho(k) - res(i_chiRho)) > 1d-5*abs(res(i_chiRho))) then
            write(*,*) 'failure for check_eos_res for chiRho k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
         if (abs(s% chiT(k) - res(i_chiT)) > 1d-5*abs(res(i_chiT))) then
            write(*,*) 'failure for check_eos_res for chiT k', k
            call show_res_prev_diffs(s, k, res, z, xh)
            ierr = -1; return
         end if
      end subroutine check_eos_res
   
      
      subroutine eval_eosPT( &
            s, k, z, xh, abar, zbar, approx_abar, approx_zbar, xa, &
            Pgas, log10Pgas, T, log10T, & 
            Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
            res, d_dlnd, d_dlnT, ierr)
         use eos_lib, only: eosPT_get
         use eos_def, only: num_eos_basic_results, i_Cp, i_Cv
         type (star_info), pointer :: s         
         integer, intent(in) :: k ! 0 means not being called for a particular cell
         real(dp), intent(in) :: &
            z, xh, abar, zbar, approx_abar, approx_zbar, xa(:), Pgas, log10Pgas, T, log10T
            
         real(dp), intent(out) :: Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas
         real(dp), dimension(num_eos_basic_results), intent(out) :: res, d_dlnd, d_dlnT
         integer, intent(out) :: ierr
         
         include 'formats.dek'
         ierr = 0
         if (s% use_other_eos) then
            call s% other_eosPT_get( &
               s% id, k, s% eos_handle, z, xh, approx_abar, approx_zbar, &
               s% species, s% chem_id, s% net_iso, xa, &
               Pgas, log10Pgas, T, log10T, &
               Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_dlnd, d_dlnT, ierr)
         else
            call eosPT_get( &
               s% eos_handle, z, xh, approx_abar, approx_zbar, &
               s% species, s% chem_id, s% net_iso, xa, &
               Pgas, log10Pgas, T, log10T, &
               Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_dlnd, d_dlnT, ierr)
         end if
         if (ierr /= 0) then
            if (s% report_ierr) then
!$omp critical (star_eosPT_get)
               write(*,*) 'eval_eosPT ierr', ierr
               write(*,2) 'k', k
               write(*,1) 'z', z
               write(*,1) 'xh', xh
               write(*,1) 'abar', abar
               write(*,1) 'zbar', zbar
               write(*,1) 'approx_abar', approx_abar
               write(*,1) 'approx_zbar', approx_zbar
               write(*,1) 'log10Pgas', log10Pgas
               write(*,1) 'log10T', log10T
!$omp end critical (star_eosPT_get)

               !stop 'eval_eosPT'
               
            end if
            return
         end if
      end subroutine eval_eosPT
      
      
      real(dp) function eval_csound_and_rho(s, k, rho, ierr)
         ! doesn't use any vars other than xh and xa
         use chem_lib
         use chem_def
         use eos_lib
         use eos_def
         
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         real(dp), intent(out) :: rho
         integer, intent(out) :: ierr
         
         integer :: species
         real(dp), dimension(num_eos_basic_results) :: res, d_dlnd, d_dlnT
         real(dp) :: xsum, dabar_dx(s% species), dzbar_dx(s% species), &
            abar, zbar, z2bar, ye, z, y, xh, lnd, T, lnT, &
            gamma1, log10Pgas, Pgas, Prad, P, approx_abar, approx_zbar, &
            log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas

         ierr = 0
         eval_csound_and_rho = -1
         species = s% species
         call composition_info(species, s% chem_id, s% xa(1:species,k), &
               xh, y, abar, zbar, z2bar, ye, approx_abar, approx_zbar, xsum, dabar_dx, dzbar_dx)  
         z = 1 - (xh + y)
         lnT = s% xh(s% i_lnT, k)
         T = exp(lnT)
         
         if (s% lnPgas_flag) then
            log10Pgas = s% xh(s% i_lnPgas,k)/ln10
            Pgas = 10**log10Pgas
            call eval_eosPT( &
               s, k, z, xh, abar, zbar, approx_abar, approx_zbar, s% xa(:,k), &
               Pgas, log10Pgas, T, lnT/ln10, & 
               rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_dlnd, d_dlnT, ierr)
            if (ierr /= 0) then
               if (s% report_ierr) then
                  write(*,*) 'eval_csound: eval_eosPT ierr', ierr
               end if
               return
            end if
         else
            lnd = s% xh(s% i_xlnd, k) - lnd_offset
            rho = exp(lnd)
            call eos_get( &
               s, k, z, xh, abar, zbar, approx_abar, approx_zbar, s% xa(:,k), &
               rho, lnd/ln10, T, lnT/ln10, &
               res, d_dlnd, d_dlnT, ierr)         
            if (ierr /= 0) then
               if (s% report_ierr) then
                  write(*,*) 'eval_csound: eos_get ierr', ierr
               end if
               return
            end if
            Pgas = exp(res(i_lnPgas))
         end if
         
         gamma1 = res(i_gamma1)
         Prad = crad * T**4 / 3
         P = Pgas + Prad
         eval_csound_and_rho = sqrt(gamma1*P/rho)

      end function eval_csound_and_rho
      
      
      subroutine eval_rho( &
            s, z, x, xa, abar, zbar, approx_abar, approx_zbar, T, lnT, Pgas, &
            rho, res, d_eos_dlnd, d_eos_dlnT, ierr)
         use eos_lib, only: eosPT_get, eosDT_get_Rho, Radiation_Pressure
         use eos_def
         
         type (star_info), pointer :: s
         real(dp), intent(in) :: &
            z, x, xa(:), abar, zbar, approx_abar, approx_zbar, T, lnT, Pgas
         real(dp), intent(out) :: rho
         real(dp), intent(out) :: &
            res(num_eos_basic_results), &
            d_eos_dlnd(num_eos_basic_results), &
            d_eos_dlnT(num_eos_basic_results)
         integer, intent(out) :: ierr
         
         real(dp) :: logPgas, logT, &
            logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas
         integer :: which_other, max_iter, eos_calls
         real(dp) :: other, other_tol, logRho_tol, logRho_guess, &
            logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, &
            helm_res(num_helm_results)
            
         include 'formats.dek'
         
         ierr = 0

         logPgas = log10(Pgas)
         logT = lnT/ln10
         
         if (s% use_other_eos) then
            call s% other_eosPT_get( &
               s% id, 0, s% eos_handle, z, x, approx_abar, approx_zbar,  &
               s% species, s% chem_id, s% net_iso, xa, &
               Pgas, logPgas, T, logT, &
               Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_eos_dlnd, d_eos_dlnT, ierr)
         else
            call eosPT_get( &
               s% eos_handle, z, x, approx_abar, approx_zbar,  &
               s% species, s% chem_id, s% net_iso, xa, &
               Pgas, logPgas, T, logT, &
               Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, &
               res, d_eos_dlnd, d_eos_dlnT, ierr)
         end if
         which_other = i_lnPgas
         max_iter = 100
         other = logPgas*ln10
         logRho_tol = 1d-6
         other_tol = 1d-6
         logRho_guess = logRho ! use the eosPT result as guess
         logRho_bnd1 = arg_not_provided
         other_at_bnd1 = arg_not_provided
         logRho_bnd2 = arg_not_provided
         other_at_bnd2 = arg_not_provided
         if (s% use_other_eos) then
            call s% other_eosDT_get_Rho( &
               s% id, 0, s% eos_handle, Z, X, approx_abar, approx_zbar, &
               s% species, s% chem_id, s% net_iso, xa, &
               logT, which_other, other, &
               logRho_tol, other_tol, max_iter, logRho_guess, &
               logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, &
               logRho, res, d_eos_dlnd, d_eos_dlnT, eos_calls, &
               ierr)
         else
            call eosDT_get_Rho( &
               s% eos_handle, Z, X, approx_abar, approx_zbar,  &
               s% species, s% chem_id, s% net_iso, xa, &
               logT, which_other, other, &
               logRho_tol, other_tol, max_iter, logRho_guess, &
               logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, &
               logRho, res, d_eos_dlnd, d_eos_dlnT, helm_res, eos_calls, ierr)
         end if
         Rho = 10**logRho
         
      end subroutine eval_rho
      
      
      subroutine eval_lnPgas( &
            s, z, x, xa, abar, zbar, approx_abar, approx_zbar, T, lnT, Rho, lnd, &
            dbg, lnPgas, ierr)
         use eos_lib, only: eosPT_get_Pgas_for_Rho
         use eos_def
         
         type (star_info), pointer :: s
         real(dp), intent(in) :: &
            z, x, xa(:), abar, zbar, approx_abar, approx_zbar, T, lnT, Rho, lnd
         logical, intent(in) :: dbg
         real(dp), intent(out) :: lnPgas
         integer, intent(out) :: ierr
         
         real(dp) :: logPgas_guess, logPgas_result, logPgas_tol, logRho_tol, &
            logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2, rho_result, logRho_result, &
            dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas
         real(dp), dimension(num_eos_basic_results) :: &
            res, d_eos_dlnd, d_eos_dlnT, d_dlnRho_const_T, d_dlnT_const_Rho
            
         integer :: max_iter, eos_calls
         real(dp) :: helm_res(num_helm_results)
            
         include 'formats.dek'
         
         ierr = 0
         
         if (is_bad_num(z)) then
            ierr = -1
            return
            write(*,1) 'Z', Z
            write(*,1) 'X', X
            write(*,1) 'approx_abar', approx_abar
            write(*,1) 'approx_zbar', approx_zbar
            write(*,1) 'lnT/ln10', lnT/ln10
            write(*,1) 'lnd/ln10', lnd/ln10
            stop 'eval_lnPgas'
         end if

         call eos_get( &
            s, 0, z, x, abar, zbar, approx_abar, approx_zbar, xa, &
            Rho, lnd/ln10, T, lnT/ln10, &
            res, d_eos_dlnd, d_eos_dlnT, ierr)
         if (ierr /= 0) return
         
         logPgas_guess = res(i_lnPgas)/ln10
         max_iter = 100
         logPgas_tol = 1d-6
         logRho_tol = 1d-6
         logPgas_bnd1 = arg_not_provided
         logRho_at_bnd1 = arg_not_provided
         logPgas_bnd2 = arg_not_provided
         logRho_at_bnd2 = arg_not_provided
         
         if (dbg) then
            write(*,*) 'call eosPT_get_Pgas_for_Rho'
            write(*,1) 'lnT/ln10', lnT/ln10
            write(*,1) 'lnd/ln10', lnd/ln10
            write(*,1) 'logPgas_guess', logPgas_guess
            write(*,1) 'Z', Z
            write(*,1) 'X', X
            write(*,1) 'approx_abar', approx_abar
            write(*,1) 'approx_zbar', approx_zbar
            write(*,1) 'guess eta', res(i_eta)
         end if
         
         if (s% use_other_eos) then
            call s% other_eosPT_get_Pgas_for_Rho( &
               s% id, 0, s% eos_handle, Z, X, approx_abar, approx_zbar,  &
               s% species, s% chem_id, s% net_iso, xa, &
               lnT/ln10, lnd/ln10, &
               logPgas_tol, logRho_tol, max_iter, logPgas_guess, &
               logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2, &
               logPgas_result, rho_result, logRho_result, &
               dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, & 
               res, d_dlnRho_const_T, d_dlnT_const_Rho, helm_res, eos_calls, ierr)
         else
            call eosPT_get_Pgas_for_Rho( &
               s% eos_handle, Z, X, approx_abar, approx_zbar,  &
               s% species, s% chem_id, s% net_iso, xa, &
               lnT/ln10, lnd/ln10, &
               logPgas_tol, logRho_tol, max_iter, logPgas_guess, &
               logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2, &
               logPgas_result, rho_result, logRho_result, &
               dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, & 
               res, d_dlnRho_const_T, d_dlnT_const_Rho, helm_res, eos_calls, ierr)
         end if
         lnPgas = logPgas_result*ln10
         
         if (dbg) then
            write(*,1) 'done eosPT_get_Pgas_for_Rho', lnPgas/ln10
            write(*,2) 'ierr', ierr
            stop 'eval_lnPgas'
         end if
         
      end subroutine eval_lnPgas


      subroutine show_res_prev_diffs(s, k, res, z, xh)
         use eos_def
         use chem_def
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         real(dp) :: res(num_eos_basic_results)
         real(dp) :: z, xh
         integer :: he3, he4
         he3 = s% net_iso(ihe3)
         he4 = s% net_iso(ihe4)
         
         1 format(a40,1pe26.16)
         write(*,*)
         write(*,*) 'show_res_prev_diffs: k', k
         write(*,*)
         write(*,1) 'z = ', z
         write(*,1) 'x = ', xh
         write(*,1) 'Xhe3 =', s% xa(he3,k)
         write(*,1) 'Xhe4 =', s% xa(he4,k)
         write(*,1) 'sum(xa) =', sum(s% xa(:,k))
         write(*,1) 'abar = ', s% abar(k)
         write(*,1) 'zbar = ', s% zbar(k)
         write(*,1) 'rho = ', s% rho(k)
         write(*,1) 'log10_rho = ', s% lnd(k)/ln10
         write(*,1) 'T = ', s% T(k)
         write(*,1) 'log10_T = ', s% lnT(k)/ln10
         write(*,*)
         write(*,1) 's% lnPgas(k)', s% lnPgas(k)
         write(*,1) 'res(i_lnPgas)',res(i_lnPgas)
         write(*,1) 's% lnPgas(k)-res(i_lnPgas)',s% lnPgas(k)-res(i_lnPgas)
         write(*,*) 
         write(*,1) 's% lnS(k)', s% lnS(k)
         write(*,1) 'res(i_lnS)', res(i_lnS)
         write(*,1) 's% lnS(k)-res(i_lnS)', s% lnS(k)-res(i_lnS)
         write(*,*)
         write(*,1) 's% lnE(k)', s% lnE(k)
         write(*,1) 'res(i_lnE)', res(i_lnE)
         write(*,1) 's% lnE(k)-res(i_lnE)', s% lnE(k)-res(i_lnE)
         write(*,*)
         write(*,1) 's% grada(k)', s% grada(k)
         write(*,1) 'res(i_grad_ad)' , res(i_grad_ad)
         write(*,1) 's% grada(k)-res(i_grad_ad)' , s% grada(k)-res(i_grad_ad)
         write(*,*)

         write(*,1) 's% dE_drho(k)', s% dE_drho(k)
         write(*,1) 'res(i_dE_drho)' , res(i_dE_drho)
         write(*,1) 's% dE_drho(k)-res(i_dE_drho)' , s% dE_drho(k)-res(i_dE_drho)
         write(*,*)

         write(*,1) 's% cv(k)', s% cv(k)
         write(*,1) 'res(i_Cv)' , res(i_Cv)
         write(*,1) 's% cv(k)-res(i_Cv)' , s% cv(k)-res(i_Cv)
         write(*,*)

         write(*,1) 's% cp(k)', s% cp(k)
         write(*,1) 'res(i_cp)' , res(i_cp)
         write(*,1) 's% cp(k)-res(i_cp)' , s% cp(k)-res(i_cp)
         write(*,*)

         write(*,1) 's% gamma1(k)', s% gamma1(k)
         write(*,1) 'res(i_gamma1)' , res(i_gamma1)
         write(*,1) 's% gamma1(k)-res(i_gamma1)' , s% gamma1(k)-res(i_gamma1)
         write(*,*)

         write(*,1) 's% gamma3(k)', s% gamma3(k)
         write(*,1) 'res(i_gamma3)' , res(i_gamma3)
         write(*,1) 's% gamma3(k)-res(i_gamma3)' , s% gamma3(k)-res(i_gamma3)
         write(*,*)

         write(*,1) 's% eta(k)', s% eta(k)
         write(*,1) 'res(i_eta)' , res(i_eta)
         write(*,1) 's% eta(k)-res(i_eta)' , s% eta(k)-res(i_eta)
         write(*,*)

         write(*,1) 's% mu(k)', s% mu(k)
         write(*,1) 'res(i_mu)' , res(i_mu)
         write(*,1) 's% mu(k)-res(i_mu)' , s% mu(k)-res(i_mu)
         write(*,*)

         write(*,1) 's% lnfree_e(k)', s% lnfree_e(k)
         write(*,1) 'res(i_lnfree_e)' , res(i_lnfree_e)
         write(*,1) 's% lnfree_e(k)-res(i_lnfree_e)' , s% lnfree_e(k)-res(i_lnfree_e)
         write(*,*)

         write(*,1) 's% chiRho(k)', s% chiRho(k)
         write(*,1) 'res(i_chiRho)' , res(i_chiRho)
         write(*,1) 's% chiRho(k)-res(i_chiRho)' , s% chiRho(k)-res(i_chiRho)
         write(*,*)

         write(*,1) 's% chiT(k)', s% chiT(k)
         write(*,1) 'res(i_chiT)' , res(i_chiT)
         write(*,1) 's% chiT(k)-res(i_chiT)' , s% chiT(k)-res(i_chiT)
         write(*,*)

      end subroutine show_res_prev_diffs
      
      
      subroutine store_eos_for_cell(s, k, res, ierr)
         use eos_def
         use eos_lib
         use screen_def, only: classic_screening
         type (star_info), pointer :: s
         integer, intent(in) :: k      
         real(dp), intent(in) ::res(num_eos_basic_results)
         integer, intent(out) :: ierr
         
         real(dp) :: d_theta_e_deta
         
         include 'formats.dek'
         ierr = 0
         s% Prad(k) = crad * s% T(k)**4 / 3
         s% P(k) = s% Prad(k) + s% Pgas(k)
         s% lnP(k) = log(s% P(k))
         s% lnS(k) = res(i_lnS)
         s% lnE(k) = res(i_lnE)
         s% grada(k) = res(i_grad_ad)
         s% dE_drho(k) = res(i_dE_drho)
         s% cv(k) = res(i_Cv)
         s% cp(k) = res(i_cp)
         s% gamma1(k) = res(i_gamma1)
         s% gamma3(k) = res(i_gamma3)
         s% eta(k) = res(i_eta)
         s% gam(k) = (s% zbar(k)*qe)**2 * &
            ((4.0d0/3.0d0)*pi*avo*s% rho(k)/s% abar(k))**one_third / (kerg*s% T(k))
         s% mu(k) = res(i_mu)
         s% lnfree_e(k) = res(i_lnfree_e)
         s% chiRho(k) = res(i_chiRho)
         s% chiT(k) = res(i_chiT)
         if (s% screening_mode == 'classic') &
            s% theta_e(k) = eos_theta_e(res(i_eta), d_theta_e_deta)
         s% csound(k) = sqrt(s% gamma1(k)*s% P(k)/s% rho(k))
         if (is_bad_num(s% csound(k))) then
            if (s% report_ierr) then
               write(*,2) 'csound', k, s% csound(k)
               write(*,2) 's% gamma1(k)', k, s% gamma1(k)
               write(*,2) 's% P(k)', k, s% P(k)
               write(*,2) 's% rho(k)', k, s% rho(k)
            end if
            ierr = -1
         end if
      end subroutine store_eos_for_cell
      
      
      subroutine do_neu(s,nzlo,nzhi,ierr)
         use neu_def,only:Tmin_neu
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         integer :: k,klo
         logical, parameter :: use_omp = .true.
         ierr = 0
         klo = nzhi+1
         do k = nzlo,nzhi ! okay to skip ones where T is less than Tmin_neu
            if (s% T(k) >= Tmin_neu) then
               klo = k
               exit
            end if
         end do
         ! klo is smallest k s.t. T(k) >= Tmin_neu
         if (klo <= nzhi) call foreach_cell(s,klo,nzhi,use_omp,do_neu_for_cell,ierr)
         if (ierr /= 0) return
         if (klo > nzlo) call foreach_cell(s,nzlo,klo-1,use_omp,do_clear_neu_for_cell,ierr)
      end subroutine do_neu
      
      
      subroutine do_clear_neu_for_cell(s,k,ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         integer, intent(out) :: ierr
         ierr = 0
         s% non_nuc_neu(k) = 0
         s% d_nonnucneu_dlnd(k) = 0
         s% d_nonnucneu_dlnT(k) = 0         
         s% nonnucneu_plas(k) = 0
         s% nonnucneu_brem(k) = 0
         s% nonnucneu_phot(k) = 0
         s% nonnucneu_pair(k) = 0
         s% nonnucneu_reco(k) = 0
      end subroutine do_clear_neu_for_cell
      
      
      subroutine do_neu_for_cell(s,k,ierr)
         use neu_def
         use neu_lib
         use chem_def, only: chem_isos
         use const_def,only:ln10
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         integer, intent(out) :: ierr

         real(dp) :: loss(num_neu_rvs) ! total from all sources
         real(dp) :: sources(num_neu_types, num_neu_rvs)
         real(dp) :: log10_rho, log10_T
         real(dp), parameter :: log10_Tlim = 7.5d0
         logical :: flags(num_neu_types) ! true if should include the type of loss
         integer :: j
         
         include 'formats.dek'
         
         flags = .true.
         !flags(reco_neu_type) = .false.
         
         ierr = 0
         
         log10_rho = s% lnd(k)/ln10
         log10_T = s% lnT(k)/ln10

         call neu_get( &
            s% T(k), log10_T, s% rho(k), log10_rho, s% abar(k), s% zbar(k), s% z2bar(k), &
            log10_Tlim, flags, loss, sources, ierr)
         
         if (ierr /= 0) then
            if (s% report_ierr) then
               write(*,3) 'do_neu_for_cell: neu_get ierr', ierr, k
               write(*,1) 'T=', s% T(k)
               write(*,1) 'log10_T=', log10_T
               write(*,1) 'rho=', s% rho(k)
               write(*,1) 'log10_rho=', log10_rho
               write(*,1) 'abar', s% abar(k)
               write(*,1) 'zbar', s% zbar(k)
               write(*,1) 'z2bar', s% z2bar(k)
               write(*,*)
               return
               stop
            end if
            return
         end if
         
         if (s% non_nuc_neu_factor /= 1) loss(:) = loss(:)*s% non_nuc_neu_factor
         s% non_nuc_neu(k) = loss(ineu)
         s% d_nonnucneu_dlnd(k) = loss(idneu_dRho)*s% rho(k)
         s% d_nonnucneu_dlnT(k) = loss(idneu_dT)*s% T(k)
         
         s% nonnucneu_plas(k) = sources(plas_neu_type,ineu)
         s% nonnucneu_brem(k) = sources(brem_neu_type,ineu)
         s% nonnucneu_phot(k) = sources(phot_neu_type,ineu)
         s% nonnucneu_pair(k) = sources(pair_neu_type,ineu)
         s% nonnucneu_reco(k) = sources(reco_neu_type,ineu)
         
         if (.false. .and. (is_bad_num(s% non_nuc_neu(k)) .or. is_bad_num(s% nonnucneu_reco(k)))) then
            write(*,2) 's% non_nuc_neu(k)', k, s% non_nuc_neu(k)
            write(*,2) 's% nonnucneu_plas(k)', k, s% nonnucneu_plas(k)
            write(*,2) 's% nonnucneu_brem(k)', k, s% nonnucneu_brem(k)
            write(*,2) 's% nonnucneu_phot(k)', k, s% nonnucneu_phot(k)
            write(*,2) 's% nonnucneu_pair(k)', k, s% nonnucneu_pair(k)
            write(*,2) 's% nonnucneu_reco(k)', k, s% nonnucneu_reco(k)
            write(*,*)
            write(*,1) 'T =', s% T(k)
            write(*,1) 'log10_T', log10_T
            write(*,1) 'rho =', s% rho(k)
            write(*,1) 'log10_rho =', log10_rho
            write(*,1) 'abar =', s% abar(k)
            write(*,1) 'zbar =', s% zbar(k)
            write(*,*)
            stop 'debug: do_neu_for_cell'
         end if
         
         if (is_bad_num(s% non_nuc_neu(k))) then
            ierr = -1
            if (s% report_ierr) write(*,*) 'do_neu_for_cell ierr for cell', k
            return
            write(*,2) 's% non_nuc_neu(k)', k, s% non_nuc_neu(k)
            write(*,2) 'log10_T', k, log10_T
            write(*,2) 'log10_rho', k, log10_rho
            write(*,2) 's% abar(k)', k, s% abar(k)
            write(*,2) 's% zbar(k)', k, s% zbar(k)
            do j=1,s% species
               write(*,1) trim(chem_isos% name(s% chem_id(j))), s% xa(j,k)
            end do
            stop 'debug: do_neu_for_cell'
         end if

      end subroutine do_neu_for_cell
      
      
      subroutine do1_zero_neu_vars(s,k)
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         s% non_nuc_neu(k) = 0
         s% d_nonnucneu_dlnd(k) = 0
         s% d_nonnucneu_dlnT(k) = 0
         s% nonnucneu_plas(k) = 0
         s% nonnucneu_brem(k) = 0
         s% nonnucneu_phot(k) = 0
         s% nonnucneu_pair(k) = 0
      end subroutine do1_zero_neu_vars
      
      
      subroutine do_net(s,nzlo,nzhi,ierr)
         use net_lib, only: net_work_size
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         logical, parameter :: use_omp = .true.
         ierr = 0
         s% net_lwork = net_work_size(s% net_handle, ierr)
         call foreach_cell(s,nzlo,nzhi,use_omp,do_net_for_cell,ierr)
         !stop 'do_net'
      end subroutine do_net
      
      
      subroutine do_net_for_cell(s,k,ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         integer, intent(out) :: ierr
         logical, parameter :: reuse_given_rates = .false.
         call do1_net( &
            s, k, s% species, s% num_reactions, s% net_lwork, reuse_given_rates, ierr)
      end subroutine do_net_for_cell


      subroutine do1_net(s,k,species,num_reactions,net_lwork,reuse_given_rates,ierr)
         use rates_def
         use net_lib, only: net_get, get_reaction_id_table_ptr, get_net_reaction_table_ptr
         use chem_def
         use const_def,only:ln10
         use utils_lib,only:is_bad_num, return_nan, realloc_double, realloc_double3
         type (star_info), pointer :: s         
         integer, intent(in) :: k, species, num_reactions, net_lwork
         logical, intent(in) :: reuse_given_rates
         integer, intent(out) :: ierr

         integer :: j, screening_mode
         real(dp) :: log10_rho, log10_T, net_work(net_lwork), &
            d_eps_nuc_dRho, d_eps_nuc_dT, &
            category_factors(num_categories), cat_factor
         character (len=100) :: message
         integer, pointer :: net_reaction_ptr(:) 
         real(dp), pointer :: reaction_neuQs(:)
      
         real(dp), parameter :: logT_lim1 = 7.8d0, logT_lim2 = 7.7d0
         real(dp), parameter :: logT_lim3 = 8.8d0, logT_lim4 = 8.7d0

         integer :: ra(9)
         real(dp) :: rfac(9)
         
         logical, parameter :: dbg = .false.

         include 'formats.dek'
         
         ierr = 0
         
         log10_rho = s% lnd(k)/ln10
         log10_T = s% lnT(k)/ln10
         
         category_factors(:) = s% category_factors(:)
         
         if (s% net_rate_factor /= 1) &
            category_factors(:) = category_factors(:)*s% net_rate_factor
         
         ! once H depleted, turn off hydrogen burning by PP where doing advanced burning
         ! else solver seems to hallucinate he3(he3,2p)he4 and other bad things
         if (s% suppress_dubious_PP_burning .and. s% net_iso(ih1) /= 0 .and. s% net_iso(ihe3) /= 0) then
            if (s% xa(s% net_iso(ih1),k) < 1d-15 .and. &
                  s% xa(s% net_iso(ihe3),k) < 1d-15 .and. &
                  category_factors(ipp) == 1) then
               if (log10_T > logT_lim1) then
                  category_factors(ipp) = 0
               else if (log10_T > logT_lim2) then
                  cat_factor = (logT_lim1 - log10_T)/(logT_lim1 - logT_lim2)
                  category_factors(ipp) = cat_factor
               end if
            end if
         end if

         ! once N14 depleted, turn off n14 burning where doing advanced burning
         if (s% suppress_dubious_N_burning .and. s% net_iso(in14) /= 0) then
            if (s% xa(s% net_iso(in14),k) < 1d-15 .and. &
                  category_factors(i_burn_n) == 1) then
               if (log10_T > logT_lim3) then
                  category_factors(i_burn_n) = 0
               else if (log10_T > logT_lim4) then
                  cat_factor = (logT_lim3 - log10_T)/(logT_lim3 - logT_lim4)
                  category_factors(i_burn_n) = cat_factor
               end if
            end if 
         end if
         
         if (s% net_iso(icr56) /= 0) then ! limit rate for fe56 + 2e => cr56
            call get_net_reaction_table_ptr(s% net_handle, net_reaction_ptr, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in do1_net call on get_net_reaction_table_ptr'
               return
            end if
            j = net_reaction_ptr(irfe56ec_fake_to_cr56)
            if (j /= 0) then
               if (s% xa(s% net_iso(icr56),s% nz) >= s% max_center_cr56_for_fe56ec) then
                  s% rate_factors(j) = 0
               else
                  s% rate_factors(j) = s% rate_factor_for_fe56ec_to_cr56
               end if
            end if
         end if
         
         if (s% abar(k) > s% max_abar_for_burning) then
            category_factors = 0
            !write(*,2) 'abar', k, s% abar
            !s% eps_nuc(k) = 0
            !s% d_epsnuc_dlnd(k) = 0
            !s% d_epsnuc_dlnT(k) = 0
            !s% d_epsnuc_dx(:,k) = 0
            !s% reaction_eps_nuc(:,:,k) = 0
            !s% eps_nuc_categories(:,:,k) = 0
            !return
         end if

         screening_mode = get_screening_mode(s,ierr)         
         if (ierr /= 0) then
            write(*,*) 'unknown string for screening_mode: ' // trim(s% screening_mode)
            return
         end if
         
         if (s% reaction_neuQs_factor /= 1d0) then
            allocate(reaction_neuQs(size(std_reaction_neuQs,dim=1)))
            reaction_neuQs(:) = std_reaction_neuQs(:)*s% reaction_neuQs_factor
            !write(*,1) 'reaction_neuQs_factor', s% reaction_neuQs_factor
         else
            reaction_neuQs => std_reaction_neuQs
         end if
         
         call net_get( &
               s% net_handle, species, num_reactions, s% xa(1:species,k), &
               s% T(k), log10_T, s% rho(k), log10_Rho, &
               s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
               s% eta(k), s% rate_factors, category_factors, std_reaction_Qs, reaction_neuQs, &
               s% eps_nuc(k), d_eps_nuc_dRho, d_eps_nuc_dT, s% d_epsnuc_dx(:,k), & 
               s% dxdt_nuc(:,k), s% dxdt_dRho(:,k), s% dxdt_dT(:,k), s% d_dxdt_dx(:,:,k), &
               screening_mode, s% theta_e(k), &
               s% rate_screened(:,:,k), s% rate_raw(:,:,k), reuse_given_rates, &
               s% reaction_eps_nuc(:,:,k), s% eps_nuc_categories(:,:,k),  s% eps_nuc_neu_total(k), &
               net_lwork, net_work, ierr)

         if (s% reaction_neuQs_factor /= 1d0) deallocate(reaction_neuQs)
         
         if (ierr /= 0) then
            write(message,*) 'do1_net: net_get failure for cell ', k
            !call alert(ierr,message)
            if (s% report_ierr) then
               write(*,*) trim(message)
               call show_stuff
            end if
            return
         end if
         
         if (is_bad_num(s% eps_nuc(k))) then
            write(message,*) 'do1_net: net_get for cell ', k
            call show_stuff
            write(*,2) 's% eps_nuc(k)', k, s% eps_nuc(k)
            stop 'net'
            return
         end if
      
         s% d_epsnuc_dlnd(k) = d_eps_nuc_dRho*s% rho(k)
         s% d_epsnuc_dlnT(k) = d_eps_nuc_dT*s% T(k)
         
         if (s% eps_nuc_factor /= 1) then
            s% eps_nuc(k) = s% eps_nuc(k)*s% eps_nuc_factor
            s% d_epsnuc_dlnd(k) = s% d_epsnuc_dlnd(k)*s% eps_nuc_factor
            s% d_epsnuc_dlnT(k) = s% d_epsnuc_dlnT(k)*s% eps_nuc_factor
            s% d_epsnuc_dx(:,k) = s% d_epsnuc_dx(:,k)*s% eps_nuc_factor
            s% reaction_eps_nuc(:,:,k) = s% reaction_eps_nuc(:,:,k)*s% eps_nuc_factor
            s% eps_nuc_categories(:,:,k) = s% eps_nuc_categories(:,:,k)*s% eps_nuc_factor
         end if
         
         if (s% dxdt_nuc_factor /= 1) then
            s% dxdt_nuc(:,k) = s% dxdt_nuc(:,k)*s% dxdt_nuc_factor
            s% dxdt_dRho(:,k) = s% dxdt_dRho(:,k)*s% dxdt_nuc_factor
            s% dxdt_dT(:,k) = s% dxdt_dT(:,k)*s% dxdt_nuc_factor
            s% d_dxdt_dx(:,:,k) = s% d_dxdt_dx(:,:,k)*s% dxdt_nuc_factor
         end if
        
         if (is_bad_num(s% eps_nuc(k))) then
            write(*,*) 'k', k
            write(*,1) 's% eps_nuc(k)', s% eps_nuc(k)
            ierr = -1
            call alert(ierr,'net_get returned bad num for eps_nuc')
            call show_stuff
            write(*,*) '(is_bad_num(s% eps_nuc(k)))'
            write(*,*) 'failed in do1_net'
            return
         end if
         
         if (.false. .and. k == 1440) then
            j = 3 ! he4
            write(*,3) 'net xa', j, k, s% xa(j,k)
            write(*,3) 'net dxdt_nuc', j, k, s% dxdt_nuc(j,k)
            write(*,3) 'net d_dxdt_dx', j, k, s% d_dxdt_dx(j,j,k)
            write(*,*)
         end if

         if (.false. .and. k == 1469) then
            write(*,2) 'eos grada(k)', k, s% grada(k)
            do j=1,s% species
               if (.true. .or. s% xa(j,k) > 1d-9) &
                  write(*,3) 'eos xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', j,k,s% xa(j,k)
            end do
            write(*,*)
         end if
         
         ! DEBUGGING
         if (.false. .and. k == 1469) then
            write(*,*)
            call show_stuff
            write(*,2) 's% eps_nuc(k)', k, s% eps_nuc(k)
            write(*,2) 'sum xa', k, sum(s% xa(:,k))
            if (s% eps_nuc(k) > 1d18) stop 'do1_net'
            return
            
            
            
            write(*,2) 's% d_epsnuc_dlnd(k)', k, s% d_epsnuc_dlnd(k)
            write(*,2) 's% d_epsnuc_dlnT(k)', k, s% d_epsnuc_dlnT(k)
            do j=1,species
               write(*,1) 'd_epsnuc_dx ' // trim(chem_isos% name(s% chem_id(j))), &
                  s% d_epsnuc_dx(j,k)
            end do
            write(*,*)
            do j=1,species
               write(*,1) 'd_dxdt_dx(1,:) ' // trim(chem_isos% name(s% chem_id(j))), &
                  s% d_dxdt_dx(1,j,k)
            end do
            write(*,*)
            write(*,*)
            write(*,*)
            
            stop 'debug: do1_net'
            return

         end if

         if (s% dbg_control > 0 .or. k == -s% nz) then
            write(*,*)
            call show_stuff
            write(*,1) 's% eps_nuc(k)', s% eps_nuc(k)
            write(*,1) 's% d_epsnuc_dlnd(k)', s% d_epsnuc_dlnd(k)
            write(*,1) 's% d_epsnuc_dlnT(k)', s% d_epsnuc_dlnT(k)
            write(*,*)
            write(*,*) 'do1_net'
            ierr = -1
         end if
         
         if (.false.) call show_stuff
         
         
         if (.false. .and. k == s% nz) then
            write(*,2) 's% eps_nuc_factor', k, s% eps_nuc_factor
            write(*,2) 's% reaction_neuQs_factor', k, s% reaction_neuQs_factor
            write(*,2) 's% eps_nuc_neu_total(k)', k, s% eps_nuc_neu_total(k)
            write(*,2) 's% eps_nuc(k)', k, s% eps_nuc(k)
            write(*,*)
            call show_stuff
            stop 'check net'
         end if
         
         
         contains
         
         
         subroutine show_stuff
            use rates_def
            integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id
            include 'formats.dek'
            call get_reaction_id_table_ptr(s% net_handle, reaction_id, ierr) 
            if (ierr /= 0) return
            write(*,2) 'k', k
            write(*,*)
            write(*,*) 'net_name ', trim(s% net_name)
            write(*,*) 'species', species
            if (.false.) then
               write(*, *)
               write(*,*) 'reaction eps_nuc'
               do j=1,s% num_reactions
                  !if (s% reaction_eps_nuc(i_rate, j, k) > 1d18) &
                  write(*,2) trim(reaction_Name(reaction_id(j))), k, s% reaction_eps_nuc(i_rate, j, k)
               end do
            end if
            write(*,*)
            if (.false.) then
               write(*, *)
               write(*,*) 'raw rates'
               do j=1,s% num_reactions
                  !if (s% reaction_eps_nuc(i_rate, j, k) > 1d18) &
                  write(*,2) trim(reaction_Name(reaction_id(j))), k, s% rate_raw(i_rate, j, k)
               end do
            end if
            if (.true.) then
               write(*,*)
               do j=1,species
                  write(*,2) 'dxdt ' // trim(chem_isos% name(s% chem_id(j))), k, s% dxdt_nuc(j, k)
               end do
            end if
            write(*,*)
            do j=1,species
               if (.true. .or. s% xa(j,k) > 1d-9) &
                  write(*,1) 'xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', s% xa(j,k)
            end do
            write(*,*)
            write(*,1) 'T =', s% T(k)
            write(*,1) 'logT =', log10_T
            write(*,1) 'rho =', s% rho(k)
            write(*,1) 'logRho =', log10_Rho
            write(*,1) 'abar =', s% abar(k)
            write(*,1) 'zbar =', s% zbar(k)
            write(*,1) 'z2bar =', s% z2bar(k)
            write(*,1) 'ye =', s% ye(k)
            write(*,1) 'eta =', s% eta(k)
            write(*,*) 'screening_mode = ' // trim(s% screening_mode)
            write(*,1) 'theta_e =', s% theta_e(k)
            
            return
            stop 'do1_net'
            
            if (.false.) then
               write(*,*)
               do j=1,num_categories
                  write(*,2) trim(category_name(j)), k, s% eps_nuc_categories(i_rate, j, k)
               end do
            end if
            if (.true.) then
               write(*, *)
               write(*,*) 'raw rates'
               do j=1,s% num_reactions
                  write(*,2) 'raw rate ' // trim(reaction_Name(reaction_id(j))), &
                     k, s% rate_raw(i_rate, j, k)
               end do
            end if
            if (.false.) then
               write(*, *)
               write(*,*) 'raw rates dlnT'
               do j=1,s% num_reactions
                  write(*,2) 'raw rate dlnT ' // trim(reaction_Name(reaction_id(j))), &
                     k, s% rate_raw(i_rate_dT, j, k)*s% T(k)
               end do
               write(*, *)
            end if
            
            
            
            !return
            
            
            
            if (.true.) then
               write(*,*)
               write(*,*) 'screened rates'
               do j=1,s% num_reactions
                  write(*,2) 'screened rate ' // trim(reaction_Name(reaction_id(j))), &
                     k, s% rate_screened(i_rate, j, k)
               end do
            end if
            if (.true.) then
               write(*, *)
               write(*,*) 'screened rates dlnT'
               do j=1,s% num_reactions
                  write(*,2) 'screened rate dlnT ' // trim(reaction_Name(reaction_id(j))), &
                     k, s% rate_screened(i_rate_dT, j, k)*s% T(k)
               end do
            end if
            if (.true.) then
               write(*, *)
               write(*,*) 'screened rates dlnRho'
               do j=1,s% num_reactions
                  write(*,2) 'screened rate dlnRho ' // trim(reaction_Name(reaction_id(j))), &
                     k, s% rate_screened(i_rate_dRho, j, k)*s% rho(k)
               end do
            end if
            if (.false.) then
               write(*,*)
               do j=1,species
                  write(*,2) 'dxdt_dlnRho ' // trim(chem_isos% name(s% chem_id(j))), k, s% dxdt_dRho(j, k)*s% Rho(k)
               end do
            end if
            if (.false.) then
               write(*,*)
               do j=1,species
                  write(*,2) 'dxdt_dlnT ' // trim(chem_isos% name(s% chem_id(j))), k, s% dxdt_dT(j, k)*s% T(k)
               end do
            end if
            write(*,*) 'X'
            write(*,*)
            write(*,2) 'sum(s% xa(1:species,k))', k, sum(s% xa(1:species,k))
            write(*,2) '1 - sum(s% xa(1:species,k))', k, 1 - sum(s% xa(1:species,k))
            !do j=1,species
            !   write(*,1) trim(chem_isos% name(s% chem_id(j))), s% xa(j,k)
            !end do
            write(*,*)
            write(*,2) 'nnuc = ', species
            write(*,*)
            do j=1,species
               write(*,'(a)') '      j' // trim(chem_isos% name(s% chem_id(j))) // ' = ' // &
                  'get_nuclide_index_in_set("' // trim(chem_isos% name(s% chem_id(j))) // '", set)'
            end do
            write(*,*) 
            do j=1,species
               write(*,'(a)',advance='no') 'j' // trim(chem_isos% name(s% chem_id(j))) // ', '
            end do
            write(*,*)
            write(*,*)
            do j=1,species
               write(*,1) 'x(j' // trim(chem_isos% name(s% chem_id(j))) // ')= ', s% xa(j,k)
            end do
            write(*,*)
            write(*,*)
            do j=1,species
               write(*,1) 'xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', s% xa(j,k)
            end do
            write(*,*)
            write(*,*)
            write(*,1) 'T =', s% T(k)
            write(*,1) 'logT =', log10_T
            write(*,1) 'rho =', s% rho(k)
            write(*,1) 'logRho =', log10_Rho
            write(*,1) 'abar =', s% abar(k)
            write(*,1) 'zbar =', s% zbar(k)
            write(*,1) 'z2bar =', s% z2bar(k)
            write(*,1) 'ye =', s% ye(k)
            write(*,1) 'eta =', s% eta(k)
            write(*,*) 'screening_mode = ' // trim(s% screening_mode)
            write(*,1) 'theta_e =', s% theta_e(k)
            write(*,*) 'reuse_given_rates', reuse_given_rates
            write(*,*)
         end subroutine show_stuff
      
      end subroutine do1_net


      subroutine do1_zero_net_vars(s,k)
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         s% eps_nuc(k) = 0
         s% eps_nuc_categories(:,:,k) = 0
         s% d_epsnuc_dlnd(k) = 0
         s% d_epsnuc_dlnT(k) = 0
         s% d_epsnuc_dx(:,k) = 0
         s% eps_nuc_neu_total(k) = 0
         s% reaction_eps_nuc(:,:,k) = 0
         s% dxdt_nuc(:,k) = 0
         s% dxdt_dRho(:,k) = 0
         s% dxdt_dT(:,k) = 0
         s% d_dxdt_dx(:,:,k) = 0
         s% rate_screened(:,:,k) = 0
         s% rate_raw(:,:,k) = 0
      end subroutine do1_zero_net_vars
      
      
      integer function get_screening_mode(s,ierr)
         use screen_lib, only: screening_option
         type (star_info), pointer :: s 
         integer, intent(out) :: ierr
         get_screening_mode = screening_option(s% screening_mode, ierr)
      end function get_screening_mode

      
      subroutine write_hydro_plot_data(s)
         use utils_lib
         use const_def
         use num_lib, only: safe_log10
         type (star_info), pointer :: s         

         character (len=100) :: filename
         integer :: k, ierr, iounit, nz
         filename = 'plot_data/hydro.data'
         ierr = 0
         nz = s% nz
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         open(iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
         if (ierr == 0) then
            write(*,*) 'write hydro results to ' // trim(filename)
            write(iounit,'(99(a,1x))') 'm', 'lgT', 'lgd', 'lgP', 'lgD', 'lgdm', 'lgR', 'lum', &
               'epsgrav', 'lgddot', 'lgTdot'
            do k=1, nz
               write(iounit,'(99e24.10)') s% m(k)/Msun,&
                     s% lnT(k)/ln10, s% lnd(k)/ln10, s% lnP(k)/ln10, &
                     safe_log10(s% mlt_D(k)), log10(s% dm(k)), &
                     s% lnR(k)/ln10, s% L(k)/Lsun, s% eps_grav(k), &
                     s% dlnd_dt(k)/ln10, s% dlnT_dt(k)/ln10
            end do
            close(iounit)
         else
            write(*,*) 'failed to open file ' // trim(filename)
         end if
         call free_iounit(iounit)      
      end subroutine write_hydro_plot_data

      
      subroutine save_kap_test_data(s)
         use utils_lib
         use const_def
         use chem_def
         type (star_info), pointer :: s         
         character (len=100) :: filename
         integer :: k, ierr, iounit, nz
         integer, pointer :: net_iso(:)
         filename = 'kap_test.data'
         ierr = 0
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         nz = s% nz
         net_iso => s% net_iso
         open(iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
         if (ierr == 0) then
            write(*,*) 'write kap test data to ' // trim(filename)
            write(iounit,*) nz
            write(iounit,'(a8,99(1x,a23))') 'k', 'lgd', 'lgT', &
               'x', 'z', 'c', 'n', 'o', 'ne', &
               'opacity', 'd_opacity_dlnd', 'd_opacity_dlnT'
            do k=1, nz
               write(iounit,'(i8,99e24.10)') k, s% lnd(k)/ln10, s% lnT(k)/ln10, s% xa(net_iso(ih1),k), &
                  1 - (s% xa(net_iso(ih1),k) + s% xa(net_iso(ihe3),k) + s% xa(net_iso(ihe4),k)), &
                  s% xa(net_iso(ic12),k), s% xa(net_iso(in14),k), s% xa(net_iso(io16),k), &
                  s% xa(net_iso(ine20),k), s% opacity(k), s% d_opacity_dlnd(k), s% d_opacity_dlnT(k)
            end do
            close(iounit)
         else
            write(*,*) 'failed to open file ' // trim(filename)
         end if
         call free_iounit(iounit)      
      end subroutine save_kap_test_data


      subroutine init_mesa_micro(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         ierr = 0
         call set_net(s, s% net_name, ierr)
         if (ierr /= 0) return
      end subroutine init_mesa_micro

      
      subroutine do_micro_change_net(s, new_net_name, ierr)
         use net_def
         type (star_info), pointer :: s
         character (len=*), intent(in) :: new_net_name
         integer, intent(out) :: ierr
         ierr = 0
         s% net_name = new_net_name
         call set_net(s, new_net_name, ierr)
      end subroutine do_micro_change_net
      
      
      subroutine set_net(s, new_net_name, ierr)
         use net_lib, only: free_net_handle, alloc_net_handle, get_net_reaction_table_ptr
         use utils_lib, only: realloc_double
         use alloc, only: update_nreactions_allocs
         use rates_def
         type (star_info), pointer :: s
         character (len=*), intent(in) :: new_net_name
         integer, intent(out) :: ierr
         
         integer :: old_num_reactions, i, ir
         integer, parameter :: num_lowT_rates = 10
         integer, pointer :: net_reaction_ptr(:) 
         
         include 'formats.dek'

         old_num_reactions = s% num_reactions

         if (s% net_handle /= 0) call free_net_handle(s% net_handle)
         s% net_handle = alloc_net_handle(ierr)
         if (ierr /= 0) return   
         call setup_new_net_info(s, ierr)
         if (ierr /= 0) return
         
         call update_nreactions_allocs(s, ierr)
         if (ierr /= 0) return

         contains
         
         subroutine show
            use chem_def, only: chem_isos
            integer :: i
            include 'formats.dek'
            return
            
            
            do i=1,s% species
               write(*,2) chem_isos% name(s% chem_id(i)), s% chem_id(i)
            end do
            write(*,*)
         end subroutine show

      end subroutine set_net
      
      
      subroutine setup_new_net_info(s, ierr)
         use net_lib
         use alloc, only: update_nvar_allocs, set_chem_names
         type (star_info), pointer :: s
         integer, intent(out) :: ierr 
         
         integer :: old_nvar_chem
              
         ierr = 0
         call net_tables(s, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_tables'
            return
         end if
         
         s% species = net_num_isos(s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_num_isos'
            return
         end if
         
         s% num_reactions = net_num_reactions(s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_num_reactions'
            return
         end if
         
         old_nvar_chem = s% nvar_chem
         s% nvar_chem = s% species
         call update_nvar_allocs(s, s% nvar_hydro, old_nvar_chem, ierr)
         if (ierr /= 0) return
         
         call get_chem_id_table_ptr(s% net_handle, s% chem_id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in get_chem_id_table_ptr'
            return
         end if
         
         call get_net_iso_table_ptr(s% net_handle, s% net_iso, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in get_net_iso_table_ptr'
            return
         end if

         call set_chem_names(s)
         
         call s% set_rate_factors(s% id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in s% set_rate_factors'
            return
         end if
         
      end subroutine setup_new_net_info
      
      
      subroutine net_tables(s, ierr)
         use net_lib ! setup net
         use rates_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr         
         ierr = 0
         
         call net_start_def(s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_start_def'
            return
         end if
         
         if (len_trim(s% net_name) == 0) then
            write(*,*) 'missing net_name -- please set it and try again'
            ierr = -1
            return
         end if

         call read_net_file(s% net_name, s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in read_net_file ' // trim(s% net_name)
            return
         end if
         
         call net_finish_def(s% net_handle, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_finish_def'
            return
         end if
         
         if (associated(s% rate_factors)) deallocate(s% rate_factors)
         allocate(s% rate_factors(rates_reaction_id_max))
         
         call s% set_rate_factors(s% id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in set_rate_factors'
            return
         end if
         
         if (associated(s% which_rates)) deallocate(s% which_rates)
         allocate(s% which_rates(rates_reaction_id_max))
         
         call s% set_which_rates(s% id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in set_which_rates'
            return
         end if

         call net_set_which_rates(s% net_handle, s% which_rates, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_set_which_rates'
            return
         end if

         call net_set_logTcut(s% net_handle, s% net_logTcut_lo, s% net_logTcut_lim, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_set_logTcut'
            return
         end if
         
         call net_setup_tables( &
            s% net_handle, rate_tables_dir_for_star, rates_cache_suffix_for_star, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'failed in net_setup_tables'
            return
         end if

      end subroutine net_tables
      
      
      subroutine default_set_which_rates(id, ierr)
         use rates_def, only: rates_NACRE_if_available
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         s% which_rates(:) = rates_NACRE_if_available
      end subroutine default_set_which_rates
      
      
      subroutine default_set_rate_factors(id, ierr)
         use rates_def, only: rates_NACRE_if_available
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         s% rate_factors(:) = 1
      end subroutine default_set_rate_factors
      
      
      subroutine shutdown_microphys
         use eos_lib
         use kap_lib
         use net_lib
         call eos_shutdown
         call kap_shutdown
         call net_shutdown
      end subroutine shutdown_microphys


      end module micro

