! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module log
      
      use star_private_def
      use const_def
      use chem_def
      use num_lib, only: linear_interp, find0, safe_log10
      
      implicit none
      
      ! spacing between these must be larger than max number of nuclides
      integer, parameter :: center_xa_offset = 10000
      integer, parameter :: surface_xa_offset = 20000
      integer, parameter :: average_xa_offset = 30000
      integer, parameter :: category_offset = 40000
      integer, parameter :: mixing_offset = 50000
      integer, parameter :: total_mass_offset = 60000
      integer, parameter :: log_total_mass_offset = 70000
      integer, parameter :: log_average_xa_offset = 80000
      integer, parameter :: log_center_xa_offset = 90000
      integer, parameter :: log_surface_xa_offset = 100000
      integer, parameter :: burning_offset = 110000
      integer, parameter :: cz_max_offset = 120000
      integer, parameter :: cz_top_max_offset = 130000
      integer, parameter :: max_eps_nuc_offset = 140000
      integer, parameter :: next_available_offset = 150000



         
      logical, parameter :: open_close_log = .true.



      contains
      
      
      
      recursive subroutine add_log_columns( &
            s, level, capacity, spec, log_columns_file, ierr)
         use utils_lib
         use utils_def
         use chem_def
         use chem_lib
         use rates_lib, only: rates_reaction_id
         type (star_info), pointer :: s
         integer, intent(in) :: level
         integer, intent(inout) :: capacity
         integer, pointer :: spec(:)
         character (len=*), intent(in) :: log_columns_file
         integer, intent(out) :: ierr

         integer :: iounit, n, i, t, id, j, cnt, ii
         character (len=256) :: buffer, string, filename
         integer, parameter :: max_level = 20

         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         if (level > max_level) then
            write(*,*) 'too many levels of nesting for log column files', level
            ierr = -1
            return
         end if

         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return

         ! first try local directory
         filename = log_columns_file
         if (len_trim(filename) == 0) filename = 'log_columns.list'
         open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then ! if don't find that file, look in data/star_data
            filename = trim(data_dir_for_mesa) // '/star_data/' // trim(filename)
            ierr = 0
            open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               call free_iounit(iounit)
               write(*,*) 'failed to open ' // trim(filename)
               return
            end if
         end if
         
         if (dbg) then
            write(*,*)
            write(*,*) 'log_columns_file <' // trim(filename) // '>'
            write(*,*)
         end if

         call count_specs
         
         n = 0
         i = 0
         
         do
            t = token(iounit, n, i, buffer, string)
            if (t == eof_token) exit
            if (t /= name_token) then
               call error; return
            end if
            if (string == 'include') then
               t = token(iounit, n, i, buffer, string)
               if (t /= string_token) then
                  call error; return
               end if
               call add_log_columns(s, level+1, capacity, spec, string, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed for included log columns list ' // trim(string)
                  call error; return
               end if
               call count_specs

            else if (string == 'center' .or. string == 'center') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(center_xa_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns center_xa ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'surface') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(surface_xa_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns surface_xa ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'average') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(average_xa_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns average_xa ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'total_mass') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(total_mass_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns total_mass ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'log_total_mass') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(log_total_mass_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns log_total_mass ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'log_average') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(log_average_xa_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns log_average_xa_offset ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'log_center') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(log_center_xa_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns log_center_xa_offset ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'log_surface') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(log_surface_xa_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns log_surface_xa_offset ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'max_eps_nuc_log_xa') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(max_eps_nuc_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns max_eps_nuc_log_xa ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'cz_top_log_xa') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(cz_top_max_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns cz_top_log_xa ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if
               
            else if (string == 'cz_log_xa') then
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  call error; return
               end if
               id = get_nuclide_index(string)
               if (id > 0) then ! store it
                  call insert_spec(cz_max_offset + id, string, ierr)
                  if (ierr /= 0) then
                     call error; return
                  end if
               else
                  write(*,*) 'failed to recognize iso name for log columns cz_log_xa ' // trim(string)
                  call error; ierr = -1
                  return
               end if
               if (ierr /= 0) then
                  call error; return
               end if

            else
               id = do_get_log_id(string)
               if (id > 0) then ! store it
                  if (id == l_mixing_regions) then
                     t = token(iounit, n, i, buffer, string)
                     if (t /= name_token) then
                        call error; return
                     end if
                     read(string,fmt=*,iostat=ierr) cnt
                     if (ierr /= 0 .or. cnt <= 0 .or. cnt > 1000) then
                        write(*,*) 'failed to find valid integer count following mixing regions ' // trim(string)
                        call error; return
                     end if
                     do ii=1,2*cnt
                        call insert_spec(mixing_offset + ii, string, ierr)
                        if (ierr /= 0) then
                           call error; return
                        end if
                     end do
                  else if (id == l_burning_regions) then
                     t = token(iounit, n, i, buffer, string)
                     if (t /= name_token) then
                        call error; return
                     end if
                     read(string,fmt=*,iostat=ierr) cnt
                     if (ierr /= 0 .or. cnt <= 0 .or. cnt > 1000) then
                        write(*,*) 'failed to find valid integer count following burning regions ' // trim(string)
                        call error; return
                     end if
                     do ii=1,2*cnt
                        call insert_spec(burning_offset + ii, string, ierr)
                        if (ierr /= 0) then
                           call error; return
                        end if
                     end do
                  else
                     call insert_spec(id, string, ierr)
                     if (ierr /= 0) then
                        call error; return
                     end if
                  end if
               else
                  id = rates_category_id(string)
                  if (id > 0) then
                     call insert_spec(category_offset + id, string, ierr)
                     if (ierr /= 0) then
                        call error; return
                     end if
                  else
                     write(*,*) 'failed to recognize item for log columns ' // trim(string)
                     ierr = -1
                     call error; return
                  end if
               end if
            end if            
         end do
         
         if (dbg) write(*,*) 'finished ' // trim(filename)
         
         close(iounit)
         call free_iounit(iounit)
         
         if (dbg) then
            write(*,*)
            write(*,*) 'done add_log_columns ' // trim(filename)
            write(*,*)
         end if
         
         
         contains
         
         
         subroutine count_specs
            integer :: i
            j = 1
            do i=1, capacity
               if (spec(i) == 0) then
                  j = i; exit
               end if
            end do
         end subroutine count_specs
         
         
         subroutine make_room(ierr)
            integer, intent(out) :: ierr
            if (j < capacity) return
            capacity = 50 + (3*capacity)/2
            call realloc_integer(spec,capacity,ierr)
            spec(j+1:capacity) = 0
         end subroutine make_room
  
            
         subroutine insert_spec(c, name, ierr)
            integer, intent(in) :: c
            character (len=*) :: name
            integer, intent(out) :: ierr
            integer :: i
            include 'formats.dek'
            do i=1,j-1
               if (spec(i) == c) return
            end do
            call make_room(ierr)
            if (ierr /= 0) return
            spec(j) = c
            if (dbg) write(*,2) trim(name), spec(j)
            j = j+1
         end subroutine insert_spec
         
         
         subroutine error
            ierr = -1
            call alert(ierr, 'error in log_columns_file')
            close(iounit)
            call free_iounit(iounit)
         end subroutine error
                  
         
      end subroutine add_log_columns
      
      
      subroutine set_log_columns(id, log_columns_file, ierr)
         use utils_lib, only: alloc_iounit, free_iounit, realloc_integer
         integer, intent(in) :: id
         character (len=*), intent(in) :: log_columns_file
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: capacity, cnt, i, io
         logical, parameter :: dbg = .false.
         integer, pointer :: old_log_column_spec(:) => null()
         character (len=strlen) :: fname
         if (dbg) write(*,*) 'set_log_columns'
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         old_log_column_spec => null()
         if (associated(s% log_column_spec)) old_log_column_spec => s% log_column_spec
         nullify(s% log_column_spec)
         capacity = 100 ! will increase if needed
         allocate(s% log_column_spec(capacity), stat=ierr)
         if (ierr /= 0) return
         s% log_column_spec(:) = 0
         call add_log_columns( &
               s, 1, capacity, s% log_column_spec, log_columns_file, ierr)
         if (ierr /= 0) then
            if (associated(old_log_column_spec)) deallocate(old_log_column_spec)
            return
         end if
         ! delete trailing 0's
         cnt = capacity+1
         do i=1, capacity
            if (s% log_column_spec(i) == 0) then
               cnt = i; exit
            end if
         end do
         capacity = cnt-1
         call realloc_integer(s% log_column_spec, capacity, ierr)
         if (ierr /= 0) return
         if (associated(old_log_column_spec)) then
            ! check that haven't changed the cols specs for an existing log file
            io = alloc_iounit(ierr)
            if (ierr /= 0) return
            fname = trim(s% log_directory) // '/' // trim(s% star_log_name)
            if (logfile_exists(fname,io)) then
               if (capacity /= size(old_log_column_spec)) then
                  ierr = -1
                  write(*,*) 'new size of log col specs', capacity
                  write(*,*) 'old size of log col specs', size(old_log_column_spec)
               else
                  do i=1,capacity
                     if (old_log_column_spec(i) /= s% log_column_spec(i)) then
                        write(*,*) 'change in log col spec', i, old_log_column_spec(i), s% log_column_spec(i)
                        ierr = -1
                     end if
                  end do
               end if
               if (ierr /= 0) then
                  write(*,*) 'ERROR: cannot change log columns when have an existing log file'
                  write(*,*) 'please delete the log file or go back to previous log columns list'
               end if
            end if
            call free_iounit(io)
            deallocate(old_log_column_spec)
            if (ierr /= 0) return
         end if
         if (associated(s% log_column_is_integer)) deallocate(s% log_column_is_integer)
         if (associated(s% log_column_values)) deallocate(s% log_column_values)
         allocate(s% log_column_is_integer(capacity), s% log_column_values(capacity), stat=ierr)
         if (ierr /= 0) return
         s% log_column_is_integer(:) = .false.
         s% log_column_values(:) = 0
         s% need_to_write_log_values = .false.
         if (dbg) write(*,*) 'num log columns', capacity
         if (dbg) stop 'debug: set_log_columns'
      end subroutine set_log_columns
      
      
      logical function logfile_exists(fname,io)
         character (len=*) :: fname
         integer, intent(in) :: io
         integer :: ierr
         ierr = 0
         open(unit=io, file=trim(fname), status='old', action='read', iostat=ierr)
         if (ierr == 0) close(io)
         logfile_exists = (ierr == 0)
      end function logfile_exists
      
      
      subroutine do_get_data_for_log_columns(s, id_extra, &
            how_many_extra_log_columns, data_for_extra_log_columns, &
            num_log_columns, nz, names, vals, is_int, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: id_extra, num_log_columns, nz
         interface
            include 'extra_log_cols.dek'
         end interface
         character (len=maxlen_profile_column_name), pointer :: names(:) ! (num_log_columns)
         real(dp), pointer :: vals(:) ! (num_log_columns)
         logical, pointer :: is_int(:)
         integer, intent(out) :: ierr
         logical, parameter :: write_flag = .false.
         call do_log_info ( &
            s, id_extra, how_many_extra_log_columns, data_for_extra_log_columns, &
            write_flag, names, vals, is_int, ierr)
      end subroutine do_get_data_for_log_columns


      subroutine write_log_info ( &
            s, id_extra, how_many_extra_log_columns, data_for_extra_log_columns, ierr)
         use utils_lib, only:alloc_iounit, free_iounit
         use chem_def, only: category_name
         use rates_def, only: rates_reaction_id_max, i_rate
         type (star_info), pointer :: s
         integer, intent(in) :: id_extra
         interface
            include 'extra_log_cols.dek'
         end interface
         integer, intent(out) :: ierr
         character (len=maxlen_profile_column_name), pointer :: names(:) ! (num_log_columns)
         real(dp), pointer :: vals(:) ! (num_log_columns)
         logical, pointer :: is_int(:)
         logical, parameter :: write_flag = .true.
         names => null()
         vals => null()
         is_int => null()
         call do_log_info( &
            s, id_extra, how_many_extra_log_columns, data_for_extra_log_columns, &
            write_flag, names, vals, is_int, ierr)
      end subroutine write_log_info


      subroutine do_log_info ( &
            s, id_extra, how_many_extra_log_columns, data_for_extra_log_columns, &
            write_flag, names, vals, is_int, ierr)
         use utils_lib, only:alloc_iounit, free_iounit
         use chem_def, only: category_name
         use rates_def, only: rates_reaction_id_max, i_rate
         type (star_info), pointer :: s
         integer, intent(in) :: id_extra
         interface
            include 'extra_log_cols.dek'
         end interface
         logical, intent(in) :: write_flag
         character (len=maxlen_profile_column_name), pointer :: names(:) ! (num_log_columns)
         real(dp), pointer :: vals(:) ! (num_log_columns)
         logical, pointer :: is_int(:)
         integer, intent(out) :: ierr
         
         character (len=strlen) :: fname, dbl_fmt, int_fmt, txt_fmt
         integer :: numcols, io, i, nz, col, j, i0, num_extra_cols
         integer, parameter :: num_epsnuc_out = 12
         real(dp) :: &
            epsnuc_out(num_epsnuc_out), csound_surf, v_surf, envelope_fraction_left, csound
         integer :: mixing_regions, burning_regions
         integer, pointer :: mixing_type(:), burning_type(:)
         character (len=maxlen_log_column_name), pointer :: extra_col_names(:)
         real(dp), pointer :: extra_col_vals(:)
         
         include 'formats.dek'
         
         dbl_fmt = s% star_log_dbl_format
         int_fmt = s% star_log_int_format
         txt_fmt = s% star_log_txt_format
         
         ierr = 0
         
         nz = s% nz
         nullify(mixing_type)
         nullify(burning_type)
         nullify(extra_col_names)
         nullify(extra_col_vals)
         
         if (write_flag) then
            io = alloc_iounit(ierr)
            if (ierr /= 0) return
         else
            io = -1 ! set to invalid value to trigger complaint if use it by mistake
         end if
         
         if (.not. associated(s% log_column_spec)) then
            numcols = 0
         else
            numcols = size(s% log_column_spec, dim=1)
         end if
         
         if (numcols == 0) then
            write(*,*) 'WARNING: do not have any output specified for logs.'
            if (io > 0) call free_iounit(io)
            return
         end if
         
         num_extra_cols = how_many_extra_log_columns(s, s% id, id_extra)
         if (num_extra_cols > 0) then
            allocate(extra_col_names(num_extra_cols), extra_col_vals(num_extra_cols), stat=ierr)
            if (ierr /= 0) then
               if (io > 0) call free_iounit(io)
               return
            end if
            call data_for_extra_log_columns( &
               s, s% id, id_extra, num_extra_cols, extra_col_names, extra_col_vals, ierr)
            if (ierr /= 0) then
               deallocate(extra_col_names, extra_col_vals)
               if (io > 0) call free_iounit(io)
               return
            end if
         end if
         
         i0 = 1
         if (write_flag .and. (open_close_log .or. s% model_number == -100)) then
            fname = trim(s% log_directory) // '/' // trim(s% star_log_name)
            if ((.not. logfile_exists(fname,io)) .or. s% doing_first_model_of_run) then
               ierr = 0
               if (len_trim(s% star_log_header_name) > 0) then
                  fname = trim(s% log_directory) // '/' // trim(s% star_log_header_name)
               end if
               open(unit=io, file=trim(fname), action='write', iostat=ierr)
            else
               i0 = 3            
               open(unit=io, file=trim(fname), action='write', position='append', iostat=ierr)
            end if
            if (ierr /= 0) then
               write(*,*) 'failed to open ' // trim(fname)
               if (io > 0) call free_iounit(io)
               return
            end if
         end if
         
         csound_surf = sqrt(s% gamma1(1)*s% P(1)/s% rho(1))
         if (.not. s% v_flag) then
            v_surf =  s% r(1)*s% dlnR_dt(1)
         else
            v_surf = s% v(1)
         end if
         
         if (s% initial_mass > s% h1_boundary_mass) then
            envelope_fraction_left = &
               (s% star_mass - s% h1_boundary_mass)/(s% initial_mass - s% h1_boundary_mass)
         else
            envelope_fraction_left = 1
         end if
         
         if (write_flag .and. i0 == 1) then ! write parameters at start of log
            do i=1,3
               col = 0
               call write_val(io, col, i, 'initial_mass', s% initial_mass)
               call write_val(io, col, i, 'initial_z', s% initial_z)
               call write_val(io, col, i, 'h1_boundary_limit', s% h1_boundary_limit)
               call write_val(io, col, i, 'he4_boundary_limit', s% he4_boundary_limit)
               call write_val(io, col, i, 'c12_boundary_limit', s% c12_boundary_limit)
               call write_val(io, col, i, 'burn_min1', s% burn_min1)
               call write_val(io, col, i, 'burn_min2', s% burn_min2)
               write(io, *)
            end do
            write(io, *)
         end if

         do i=i0,3 ! add a row to the log
            col = 0
            if (i==3) then
               if (write_flag .and. i0 == 1 .and. len_trim(s% star_log_header_name) > 0) then
                  close(io)
                  fname = trim(s% log_directory) // '/' // trim(s% star_log_name)
                  open(unit=io, file=trim(fname), action='write', status='replace', iostat=ierr)
                  if (ierr /= 0) then
                     call free_iounit(io); return
                  end if
               end if
               epsnuc_out(1:4) = s% burn_zone_mass(1:4,1)
               epsnuc_out(5:8) = s% burn_zone_mass(1:4,2)
               epsnuc_out(9:12) = s% burn_zone_mass(1:4,3)
               mixing_regions = count_output_mix_regions()               
               if (mixing_regions > 0) then
                  allocate(mixing_type(nz), stat=ierr)
                  if (ierr /= 0) exit
                  call set_mix_types
                  call prune_weak_mixing_regions
               end if
               burning_regions = count_output_burn_regions()               
               if (burning_regions > 0) then
                  allocate(burning_type(nz), stat=ierr)
                  if (ierr /= 0) exit
                  call set_burn_types
               end if
            end if
            do j=1,numcols
               call do_col(i, j)
            end do
            do j=1,num_extra_cols
               call do_extra_col(i, j)
            end do
            if (write_flag) write(io, *)
         end do
         
         if (open_close_log) close(io)
         
         call free_iounit(io)
         
         if (associated(mixing_type)) deallocate(mixing_type)
         if (associated(burning_type)) deallocate(burning_type)
         if (associated(extra_col_names)) deallocate(extra_col_names)
         if (associated(extra_col_vals)) deallocate(extra_col_vals)
         
         contains


         subroutine do_extra_col(pass, j)
            integer, intent(in) :: pass, j
            if (pass == 1) then
               if (write_flag) write(io, fmt=int_fmt, advance='no') j + numcols
            else if (pass == 2) then
               call do_name(j + numcols, extra_col_names(j))
            else if (pass == 3) then
               call do_val(j + numcols, extra_col_vals(j))
            end if
         end subroutine do_extra_col


         subroutine do_name(j, col_name)
            integer, intent(in) :: j
            character (len=*), intent(in) :: col_name
            if (write_flag) then
               write(io, fmt=txt_fmt, advance='no') trim(col_name)
            else
               names(j) = trim(col_name)
            end if
         end subroutine do_name
         
         
         integer function count_output_burn_regions()
            integer :: j, cnt, c, i, ii
            cnt = 0
            do j=1,numcols
               c = s% log_column_spec(j)
               if (c > burning_offset .and. c <= next_available_offset) then
                  i = c - burning_offset
                  ii = (i+1)/2
                  if (ii > cnt) cnt = ii
               end if
            end do
            count_output_burn_regions = cnt
         end function count_output_burn_regions
         
         
         subroutine set_burn_types
            integer :: cnt, min_ktop, min_kbot, imax, i, prev_cnt, k
            real(dp) :: val
            logical, parameter :: dbg = .false.
            do k=1,nz
               val = s% eps_nuc(k) - s% eps_nuc_neu_total(k) - s% non_nuc_neu(k)
               burning_type(k) = sign(1d0,val)*log10(max(1d0,abs(val)))
            end do
            ! remove smallest regions until <= burning_regions remain
            imax = nz
            do i=1,imax
               call count_burning_regions(cnt, min_ktop, min_kbot)
               if (dbg) write(*,*) 'count_burning_regions', cnt
               if (cnt <= burning_regions) return
               if (i > 1 .and. cnt >= prev_cnt) then
                  write(*,*) 'bug in set_burn_types: cnt, prev_cnt', cnt, prev_cnt
                  if (dbg) stop 'debug: set_burn_types'
                  return
               end if
               prev_cnt = cnt
               if (dbg) write(*,*) 'remove_burning_region', min_ktop, min_kbot, cnt
               call remove_burning_region(min_ktop, min_kbot)
            end do
            if (dbg) stop 'debug: set_burn_types'
         end subroutine set_burn_types

         
         integer function count_output_mix_regions()
            integer :: j, cnt, c, i, ii
            cnt = 0
            do j=1,numcols
               c = s% log_column_spec(j)
               if (c > mixing_offset .and. c <= total_mass_offset) then
                  i = c - mixing_offset
                  ii = (i+1)/2
                  if (ii > cnt) cnt = ii
               end if
            end do
            count_output_mix_regions = cnt
         end function count_output_mix_regions
         
         
         subroutine prune_weak_mixing_regions
            use mlt_def, only: no_mixing
            real(dp) :: D_max_in_region, D_cutoff
            integer :: min_ktop, min_kbot
            integer :: i
            logical, parameter :: dbg = .false.
            include 'formats.dek'
            D_cutoff = s% mixing_D_limit_for_log
            do i = 1, 10000
               call find_weakest_mixing_region(D_max_in_region, min_ktop, min_kbot)
               if (D_max_in_region > D_cutoff) then
                  if (dbg) write(*,3) 'done', min_ktop, min_kbot, D_max_in_region, D_cutoff
                  return
               end if
               if (dbg) write(*,3) 'remove_weak_mixing_region', min_ktop, min_kbot, D_max_in_region
               if (dbg) write(*,3) 'i model mass', i, s% model_number, s% m(min_kbot)/Msun, s% m(min_ktop)/Msun
               if (dbg) write(*,*)
               mixing_type(min_ktop:min_kbot) = no_mixing
            end do
         end subroutine prune_weak_mixing_regions
         
         
         subroutine find_weakest_mixing_region(D_max_in_region, min_ktop, min_kbot)
            use mlt_def
            real(dp), intent(out) :: D_max_in_region
            integer, intent(out) :: min_ktop, min_kbot
            
            integer :: k, kbot, cur_type
            real(dp) :: D_max
            logical, parameter :: dbg = .false.
            include 'formats.dek'
            min_ktop = 1
            min_kbot = nz
            D_max_in_region = 1d99
            kbot = nz
            cur_type = mixing_type(nz)
            do k=nz-1,1,-1
               if (cur_type == mixing_type(k)) cycle
               ! k is bottom of new region; k+1 is top of region cnt
               if (cur_type /= no_mixing .and. cur_type /= semiconvective_mixing) then
                  D_max = maxval(s% D_mix_non_rotation(k+1:kbot))
                  if (D_max < D_max_in_region) then
                     D_max_in_region = D_max; min_ktop = k+1; min_kbot = kbot
                  end if
               end if
               kbot = k
               cur_type = mixing_type(k)
            end do
            if (cur_type /= no_mixing) then
               D_max = maxval(s% D_mix_non_rotation(1:kbot))
               if (D_max < D_max_in_region) then
                  D_max_in_region = D_max; min_ktop = 1; min_kbot = kbot
               end if
            end if

         end subroutine find_weakest_mixing_region
         
         
         subroutine set_mix_types            
            integer :: cnt, min_ktop, min_kbot, imax, i, prev_cnt
            logical, parameter :: dbg = .false.
            mixing_type(1:nz) = s% mixing_type(1:nz)
            ! remove smallest regions until <= mixing_regions remain
            imax = nz
            do i=1,imax
               call count_mixing_regions(cnt, min_ktop, min_kbot)
               if (dbg) write(*,*) 'count_mixing_regions', cnt
               if (cnt <= mixing_regions) exit
               if (i > 1 .and. cnt >= prev_cnt) then
                  write(*,*) 'bug in set_mix_types: cnt, prev_cnt', cnt, prev_cnt
                  if (dbg) stop 'set_mix_types'
                  return
               end if
               prev_cnt = cnt
               if (dbg) write(*,*) 'remove_mixing_region', min_ktop, min_kbot, cnt
               call remove_mixing_region(min_ktop, min_kbot)
            end do        
            if (dbg) call show_regions(mixing_type)
         end subroutine set_mix_types
         
         
         subroutine count_regions(mix_type, cnt, min_ktop, min_kbot)
            integer, intent(in) :: mix_type(:)
            integer, intent(out) :: cnt, min_ktop, min_kbot
            integer :: k, kbot, cur_type
            real(dp) :: prev_qtop, qtop, dq, min_dq
            logical, parameter :: dbg = .false.
            include 'formats.dek'
            cnt = 1
            min_ktop = 1
            min_kbot = nz
            prev_qtop = 0
            min_dq = 1
            kbot = nz
            cur_type = mix_type(nz)
            do k=nz-1,1,-1
               if (cur_type == mix_type(k)) cycle
               ! k is bottom of new region; k+1 is top of region cnt
               qtop = s% q(k+1)
               dq = qtop - prev_qtop
               !if (dbg) write(*,3) 'dq', k+1, cnt, dq, min_dq
               if (dq < min_dq) then
                  min_dq = dq; min_ktop = k+1; min_kbot = kbot
                  if (dbg) write(*,3) 'loop min_ktop min_kbot min_dq', min_ktop, min_kbot, min_dq
               end if
               kbot = k
               cnt = cnt+1
               cur_type = mix_type(k)
               prev_qtop = qtop
            end do       
            qtop = 1
            dq = qtop - prev_qtop
            if (dq < min_dq) then
               min_dq = dq; min_ktop = 1; min_kbot = kbot
               if (dbg) write(*,3) 'final min_ktop min_kbot min_dq', min_ktop, min_kbot, min_dq
            end if
            !if (dbg) write(*,3) 'dq', 1, cnt, dq, min_dq
         end subroutine count_regions
         
         
         subroutine show_regions(mix_type)
            integer, intent(in) :: mix_type(:)
            integer :: k, kbot, cur_type
            include 'formats.dek'
            kbot = nz
            cur_type = mix_type(nz)
            do k=nz-1,1,-1
               if (cur_type == mix_type(k)) cycle
               ! k is bottom of new region; k+1 is top of region cnt
               write(*,4) 'mix region', cur_type, k+1, kbot, s% m(k+1)/Msun, s% m(kbot)/Msun
               kbot = k
               cur_type = mix_type(k)
            end do       
            write(*,4) 'mix region', cur_type, 1, kbot, s% m(1)/Msun, s% m(kbot)/Msun
         end subroutine show_regions
         
         
         subroutine remove_region(mix_type, min_ktop, min_kbot)
            integer, intent(inout) :: mix_type(:)
            integer, intent(in) :: min_ktop, min_kbot
            integer :: new_type
            logical, parameter :: dbg = .false.
            include 'formats.dek'
            if (dbg) then
               write(*,2) 'q top', min_ktop, s% q(min_ktop)
               write(*,2) 'q bot', min_kbot, s% q(min_kbot)
               write(*,2) 'dq', min_kbot, s% q(min_ktop) - s% q(min_kbot)
            end if
            if (min_ktop > 1) then ! merge with above
               new_type = mix_type(min_ktop-1)
            else if (min_kbot < nz) then ! merge with below
               new_type = mix_type(min_kbot+1)
            else
               write(*,*) 'confusion in args for remove_region', min_ktop, min_kbot
               return
            end if
            mix_type(min_ktop:min_kbot) = new_type
            if (dbg) write(*,2) 'new type', min_kbot, new_type
         end subroutine remove_region
         
         
         integer function region_top(mix_type,j)
            integer, intent(in) :: mix_type(:), j
            integer :: k, cnt, cur_type
            cnt = 1
            cur_type = mix_type(nz)
            do k=nz-1,1,-1
               if (cur_type == mix_type(k)) cycle
               ! k is start of new region; k+1 is top of region cnt
               if (cnt == j) then
                  region_top = k+1
                  return
               end if
               cnt = cnt+1
               cur_type = mix_type(k)
            end do
            if (cnt == j) then
               region_top = 1
            else
               region_top = -1
            end if
         end function region_top
         
         
         subroutine count_mixing_regions(cnt, min_ktop, min_kbot)
            integer, intent(out) :: cnt, min_ktop, min_kbot
            call count_regions(mixing_type, cnt, min_ktop, min_kbot)
         end subroutine count_mixing_regions
         
         
         subroutine remove_mixing_region(min_ktop, min_kbot)
            integer, intent(in) :: min_ktop, min_kbot
            call remove_region(mixing_type, min_ktop, min_kbot)
         end subroutine remove_mixing_region
         
         
         integer function mix_region_top(j)
            integer, intent(in) :: j
            mix_region_top = region_top(mixing_type, j)
         end function mix_region_top
         
         
         subroutine count_burning_regions(cnt, min_ktop, min_kbot)
            integer, intent(out) :: cnt, min_ktop, min_kbot
            call count_regions(burning_type, cnt, min_ktop, min_kbot)
         end subroutine count_burning_regions
         
         
         subroutine remove_burning_region(min_ktop, min_kbot)
            integer, intent(in) :: min_ktop, min_kbot
            call remove_region(burning_type, min_ktop, min_kbot)
         end subroutine remove_burning_region
         
         
         integer function burn_region_top(j)
            integer, intent(in) :: j
            burn_region_top = region_top(burning_type, j)
         end function burn_region_top
         
         
         real(dp) function interpolate_burn_bdy(k) result(val)
            use mlt_def
            use num_lib, only: find0
            integer, intent(in) :: k
            integer :: bv, bv0, bv1
            real(dp) :: eps, eps0, eps1, d0, d1, q0, q1
            include 'formats.dek'
            if (k <= 0) then
               val = -1; return
            end if
            val = s% q(k)
            if (k == 1) return
            eps1 = s% eps_nuc(k) - s% eps_nuc_neu_total(k) - s% non_nuc_neu(k)
            bv1 = sign(1d0,eps1)*log10(max(1d0,abs(eps1)))
            eps0 = s% eps_nuc(k-1) - s% eps_nuc_neu_total(k-1) - s% non_nuc_neu(k-1)
            bv0 = sign(1d0,eps0)*log10(max(1d0,abs(eps0)))
            bv = max(bv0,bv1)
            eps = 10**bv
            d0 = eps0 - eps
            d1 = eps1 - eps
            if (d0*d1 >= 0) return
            q1 = s% q(k) - s% dq(k)*0.5d0
            q0 = s% q(k-1) - s% dq(k-1)*0.5d0
            val = find0(q1,d1,q0,d0)
         end function interpolate_burn_bdy
         

         subroutine do_col(pass, j)
            use rates_def
            use star_utils, only: &
               dt_Courant, eval_Ledd, surf_omega_crit, surf_omega, omega_crit
            use mlt_def, only: convective_mixing
            integer, intent(in) :: pass, j
            integer :: i, c, ii, k, int_val
            real(dp) :: val, val1, Ledd, power_photo
            logical :: is_int_val
            character (len=10) :: str
            character (len=100) :: col_name
            logical, parameter :: dbg = .false.
            include 'formats.dek'
            c = s% log_column_spec(j)
            int_val = 0; val = 0; is_int_val = .false.
            if (pass == 1) then
               col = col+1
               if (write_flag) write(io, fmt=int_fmt, advance='no') col
            else if (pass == 2) then
               if (c > max_eps_nuc_offset) then
                  i = c - max_eps_nuc_offset
                  col_name = 'max_eps_nuc_log_' // trim(chem_isos% name(i))
               else if (c > cz_top_max_offset) then
                  i = c - cz_top_max_offset
                  col_name = 'cz_top_log_' // trim(chem_isos% name(i))
               else if (c > cz_max_offset) then
                  i = c - cz_max_offset
                  col_name = 'cz_log_' // trim(chem_isos% name(i))
               else if (c > burning_offset) then
                  i = c - burning_offset
                  ii = (i+1)/2
                  if (ii > burning_regions) burning_regions = ii
                  if (ii < 10) then
                     write(str,'(i1)') ii
                  else if (ii < 100) then
                     write(str,'(i2)') ii
                  else
                     write(str,'(i3)') ii
                  end if
                  if (mod(i,2)==1) then ! burning type
                     col_name = 'burn_type_' // trim(str)
                  else ! location of top
                     col_name = 'burn_qtop_' // trim(str)
                  end if
               else if (c > log_surface_xa_offset) then
                  i = c - log_surface_xa_offset
                  col_name = 'log_surface_' // trim(chem_isos% name(i))
               else if (c > log_center_xa_offset) then
                  i = c - log_center_xa_offset
                  col_name = 'log_center_' // trim(chem_isos% name(i))
               else if (c > log_average_xa_offset) then
                  i = c - log_average_xa_offset
                  col_name = 'log_average_' // trim(chem_isos% name(i))
               else if (c > log_total_mass_offset) then
                  i = c - log_total_mass_offset
                  col_name = 'log_total_mass_' // trim(chem_isos% name(i))
               else if (c > total_mass_offset) then
                  i = c - total_mass_offset
                  col_name = 'total_mass_' // trim(chem_isos% name(i))
               else if (c > mixing_offset) then
                  i = c - mixing_offset
                  ii = (i+1)/2
                  if (ii > mixing_regions) mixing_regions = ii
                  if (ii < 10) then
                     write(str,'(i1)') ii
                  else if (ii < 100) then
                     write(str,'(i2)') ii
                  else
                     write(str,'(i3)') ii
                  end if
                  if (mod(i,2)==1) then ! mixing type
                     col_name = 'mix_type_' // trim(str)
                  else ! location of top
                     col_name = 'mix_qtop_' // trim(str)
                  end if
               else if (c > category_offset) then
                  i = c - category_offset
                  col_name = category_name(i)
               else if (c > average_xa_offset) then
                  i = c - average_xa_offset
                  col_name = 'average_' // trim(chem_isos% name(i))
               else if (c > surface_xa_offset) then
                  i = c - surface_xa_offset
                  col_name = 'surface_' // trim(chem_isos% name(i))
               else if (c > center_xa_offset) then
                  i = c - center_xa_offset
                  col_name = 'center_' // trim(chem_isos% name(i))
               else
                  col_name = trim(log_column_name(c))
               end if
               call do_name(j, col_name)
            else if (pass == 3) then
               if (c > max_eps_nuc_offset) then
                  i = c - max_eps_nuc_offset
                  val = safe_log10(max_eps_nuc_log_x(s% net_iso(i)))
               else if (c > cz_top_max_offset) then
                  i = c - cz_top_max_offset
                  val = safe_log10(cz_top_max_log_x(s% net_iso(i)))
               else if (c > cz_max_offset) then
                  i = c - cz_max_offset
                  val = safe_log10(cz_max_log_x(s% net_iso(i)))
               else if (c > burning_offset) then
                  i = c - burning_offset
                  ii = (i+1)/2
                  k = burn_region_top(ii)
                  if (mod(i,2)==1) then ! burning type
                     is_int_val = .true.
                     if (k > 0) then
                        int_val = burning_type(k)
                     else
                        int_val = -9999
                     end if
                  else ! location of top
                     val = interpolate_burn_bdy(k)
                  end if
               else if (c > log_surface_xa_offset) then
                  i = c - log_surface_xa_offset
                  val = safe_log10(surface_avg_x(s% net_iso(i)))
               else if (c > log_center_xa_offset) then
                  i = c - log_center_xa_offset
                  val = safe_log10(central_avg_x(s% net_iso(i)))
               else if (c > log_average_xa_offset) then
                  i = c - log_average_xa_offset
                  val = safe_log10(star_avg_x(s% net_iso(i)))
               else if (c > log_total_mass_offset) then
                  i = c - log_total_mass_offset
                  val = safe_log10(star_avg_x(s% net_iso(i))*s% xmstar/Msun)
               else if (c > total_mass_offset) then
                  i = c - total_mass_offset
                  val = star_avg_x(s% net_iso(i))*s% xmstar/Msun
               else if (c > mixing_offset) then
                  i = c - mixing_offset
                  ii = (i+1)/2
                  k = mix_region_top(ii)
                  if (mod(i,2)==1) then ! mixing type
                     is_int_val = .true.
                     if (k > 0) then
                        int_val = mixing_type(k)
                     else
                        int_val = -1
                     end if
                  else ! q location of boundary
                     if (k <= 1) then
                        val = 1d0
                     else
                        val = s% q(k-1) - s% mixing_type_change_dq(k-1)
                     end if
                  end if
               else if (c > category_offset) then
                  i = c - category_offset
                  val = category_L(i)
               else if (c > average_xa_offset) then
                  i = c - average_xa_offset
                  val = star_avg_x(s% net_iso(i))
               elseif (c > surface_xa_offset) then
                  i = c - surface_xa_offset
                  val = surface_avg_x(s% net_iso(i))
               else if (c > center_xa_offset) then
                  i = c - center_xa_offset
                  val = central_avg_x(s% net_iso(i))
               else
                  select case(c)
                     case(l_model_number)
                        is_int_val = .true.
                        int_val = s% model_number
                     case(l_star_age)
                        val = s% star_age
                     case(l_star_mass)
                        val = s% star_mass
                     case(l_log_xmstar)
                        val = safe_log10(s% xmstar)
                     case(l_delta_mass)
                        val = s% star_mass - s% initial_mass
                     case(l_star_mdot)
                        val = s% star_mdot
                     case(l_log_abs_mdot)
                        val = safe_log10(abs(s% star_mdot))
                     case(l_time_step)
                        val = s% time_step
                     case(l_e_thermal)
                        val = sum(s% dm(1:s% nz)*s% T(1:s% nz)*s% cp(1:s% nz))
                     case(l_log_dt)
                        val = safe_log10(s% time_step)
                     case(l_log_total_angular_momentum)
                        val = safe_log10(s% total_angular_momentum)
                     case(l_num_zones)
                        int_val = s% nz
                        is_int_val = .true.
                     case(l_num_retries)
                        int_val = s% num_retries
                        is_int_val = .true.
                     case(l_num_backups)
                        int_val = s% num_backups
                        is_int_val = .true.
                     case(l_num_jacobians)
                        int_val = s% num_jacobians
                        is_int_val = .true.
                     case(l_total_num_jacobians)
                        int_val = s% total_num_jacobians
                        is_int_val = .true.
                     case(l_h1_czb_mass)
                        val = s% h1_czb_mass
                     case(l_surf_c12_minus_o16)
                        val = s% surface_c12 - s% surface_o16
                     case(l_surf_num_c12_div_num_o16)
                        val = (16d0/12d0)*s% surface_c12/max(1d-99,s% surface_o16)
                     case(l_conv_mx1_top)
                        val = s% conv_mx1_top
                     case(l_conv_mx1_bot)
                        val = s% conv_mx1_bot
                     case(l_conv_mx2_top)
                        val = s% conv_mx2_top
                     case(l_conv_mx2_bot)
                        val = s% conv_mx2_bot
                     case(l_mx1_top)
                        val = s% mx1_top
                     case(l_mx1_bot)
                        val = s% mx1_bot
                     case(l_mx2_top)
                        val = s% mx2_top
                     case(l_mx2_bot)
                        val = s% mx2_bot
                     case(l_conv_mx1_top_r)
                        val = s% conv_mx1_top_r
                     case(l_conv_mx1_bot_r)
                        val = s% conv_mx1_bot_r
                     case(l_conv_mx2_top_r)
                        val = s% conv_mx2_top_r
                     case(l_conv_mx2_bot_r)
                        val = s% conv_mx2_bot_r
                     case(l_mx1_top_r)
                        val = s% mx1_top_r
                     case(l_mx1_bot_r)
                        val = s% mx1_bot_r
                     case(l_mx2_top_r)
                        val = s% mx2_top_r
                     case(l_mx2_bot_r)
                        val = s% mx2_bot_r
                     case(l_epsnuc_M_1)
                        val = epsnuc_out(1)
                     case(l_epsnuc_M_2)
                        val = epsnuc_out(2)
                     case(l_epsnuc_M_3)
                        val = epsnuc_out(3)
                     case(l_epsnuc_M_4)
                        val = epsnuc_out(4)
                     case(l_epsnuc_M_5)
                        val = epsnuc_out(5)
                     case(l_epsnuc_M_6)
                        val = epsnuc_out(6)
                     case(l_epsnuc_M_7)
                        val = epsnuc_out(7)
                     case(l_epsnuc_M_8)
                        val = epsnuc_out(8)
                     case(l_power_h_burn)
                        val = s% power_h_burn
                     case(l_log_LH)
                        val = safe_log10(s% power_h_burn)
                     case(l_power_he_burn)
                        val = s% power_he_burn
                     case(l_log_LHe)
                        val = safe_log10(s% power_he_burn)
                     case(l_log_LZ)
                        power_photo = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
                        val = safe_log10(s% power_nuc_burn - (power_photo + s% power_h_burn + s% power_he_burn))
                     case(l_luminosity)
                        val = s% L(1)/Lsun
                     case(l_log_L)
                        val = safe_log10(s% L(1)/Lsun)
                     case(l_log_Lneu)
                        val = safe_log10(s% power_neutrinos)
                        
                     case(l_log_Teff)
                        val = safe_log10(s% Teff)
                     case(l_effective_T)
                        val = s% Teff
                     case(l_photosphere_L)
                        val = s% photosphere_L
                     case(l_photosphere_r)
                        val = s% photosphere_r
                        
                     case(l_radius_cm)
                        val = s% R(1)
                     case(l_log_R_cm)
                        val = safe_log10(s% R(1))
                     case(l_radius)
                        val = s% R(1)/Rsun
                     case(l_log_R)
                        val = safe_log10(s% R(1)/Rsun)
                        
                     case(l_gravity)
                        val = s% grav(1)
                     case(l_log_g)
                        val = safe_log10(s% grav(1))
                     case(l_log_center_T)
                        val = s% log_center_temperature
                     case(l_log_center_Rho)
                        val = s% log_center_density
                     case(l_v_div_csound_surf)
                        val = v_surf/csound_surf
                     case(l_surface_accel_div_grav)
                        if (s% dt > 0 .and. s% generations > 1) then
                           val = (v_surf - s% v_surf_old)/(s% dt*s% grav(1))
                        else
                           val = 0
                        end if
                     case(l_log_L_div_Ledd)
                        Ledd = eval_Ledd(s)
                        val = safe_log10(s% L(1)/Ledd)

                     case(l_surf_omega)
                        if (.not. s% rotation_flag) then
                           val = 0
                        else
                           val = s% omega_surf
                        end if
                     case(l_surf_omega_crit)
                        if (.not. s% rotation_flag) then
                           val = 0
                        else
                           val = s% omega_crit_surf
                        end if
                     case(l_surf_omega_div_omega_crit)
                        if (.not. s% rotation_flag) then
                           val = 0
                        else
                           !val = min(1d0,s% omega_surf/s% omega_crit_surf)
                           val = min(2d0,s% omega_surf/s% omega_crit_surf)
                           !val = s% omega_surf/s% omega_crit_surf
                        end if
                     case(l_surf_v_rot)
                        if (.not. s% rotation_flag) then
                           val = 0
                        else
                           val = s% omega_surf*s% photosphere_r*Rsun*1d-5 ! km/sec
                        end if
                     case(l_surf_v_crit)
                        if (.not. s% rotation_flag) then
                           val = 0
                        else
                           val = s% omega_crit_surf*s% photosphere_r*Rsun*1d-5 ! km/sec
                        end if
                     case (l_log_rotational_mdot_boost)
                        if (s% rotation_flag) then
                           val = safe_log10(s% rotational_mdot_boost)
                        else
                           val = 0
                        end if

                     case(l_min_Pgas_div_P)
                        val = minval(s% Pgas(1:nz)/s% P(1:nz))
                     case(l_log_center_P)
                        val = s% log_center_pressure
                     case(l_center_degeneracy)
                        val = s% center_degeneracy
                     case(l_center_eps_grav)
                        val = s% center_eps_grav
                     case(l_center_gamma)
                        val = s% center_gamma
                     case(l_center_mu)
                        val = s% center_mu
                     case(l_center_ye)
                        val = s% center_ye
                     case(l_center_entropy)
                        val = s% center_entropy
                     case(l_iron_core_infall)
                        val = s% iron_core_infall*1d-5 ! convert to km/sec
                     case(l_center_omega)
                        val = s% center_omega
                     case(l_center_omega_div_omega_crit)
                        val = s% center_omega_div_omega_crit

                     case(l_surf_r_equitorial_div_r)
                        val = s% r_equitorial(1)/s% r(1)
                     case(l_surf_r_polar_div_r)
                        val = s% r_polar(1)/s% r(1)
                        
                     case(l_h1_boundary_mass)
                        val = s% h1_boundary_mass
                     case(l_h1_boundary_radius)
                        val = s% h1_boundary_radius
                     case(l_h1_boundary_lgT)
                        val = s% h1_boundary_lgT
                     case(l_h1_boundary_lgRho)
                        val = s% h1_boundary_lgRho
                     case(l_h1_boundary_L)
                        val = s% h1_boundary_L
                     case(l_h1_boundary_v)
                        val = s% h1_boundary_v
                     case(l_h1_bdy_to_phot_sound_time)
                        val = s% h1_bdy_to_phot_sound_time
                     case(l_h1_bdy_omega)
                        val = s% h1_bdy_omega
                     case(l_h1_bdy_omega_div_omega_crit)
                        val = s% h1_bdy_omega_div_omega_crit
                        
                     case(l_he4_boundary_mass)
                        val = s% he4_boundary_mass
                     case(l_he4_boundary_radius)
                        val = s% he4_boundary_radius
                     case(l_he4_boundary_lgT)
                        val = s% he4_boundary_lgT
                     case(l_he4_boundary_lgRho)
                        val = s% he4_boundary_lgRho
                     case(l_he4_boundary_L)
                        val = s% he4_boundary_L
                     case(l_he4_boundary_v)
                        val = s% he4_boundary_v
                     case(l_he4_bdy_to_phot_sound_time)
                        val = s% he4_bdy_to_phot_sound_time
                     case(l_he4_bdy_omega)
                        val = s% he4_bdy_omega
                     case(l_he4_bdy_omega_div_omega_crit)
                        val = s% he4_bdy_omega_div_omega_crit
                        
                     case(l_c12_boundary_mass)
                        val = s% c12_boundary_mass
                     case(l_c12_boundary_radius)
                        val = s% c12_boundary_radius
                     case(l_c12_boundary_lgT)
                        val = s% c12_boundary_lgT
                     case(l_c12_boundary_lgRho)
                        val = s% c12_boundary_lgRho
                     case(l_c12_boundary_L)
                        val = s% c12_boundary_L
                     case(l_c12_boundary_v)
                        val = s% c12_boundary_v
                     case(l_c12_bdy_to_phot_sound_time)
                        val = s% c12_bdy_to_phot_sound_time
                     case(l_c12_bdy_omega)
                        val = s% c12_bdy_omega
                     case(l_c12_bdy_omega_div_omega_crit)
                        val = s% c12_bdy_omega_div_omega_crit
                        
                     case(l_envelope_mass)
                        val = s% star_mass - s% h1_boundary_mass
                     case(l_envelope_fraction_left)
                        val = envelope_fraction_left
                     case(l_tau10_mass)
                        val = s% tau10_mass
                     case(l_tau10_radius)
                        val = s% tau10_radius
                     case(l_tau10_lgP)
                        val = s% tau10_lgP
                     case(l_tau10_T)
                        val = 10**s% tau10_lgT
                     case(l_tau10_lgT)
                        val = s% tau10_lgT
                     case(l_tau10_lgRho)
                        val = s% tau10_lgRho
                     case(l_tau10_L)
                        val = s% tau10_L
                     case(l_tau100_mass)
                        val = s% tau100_mass
                     case(l_tau100_radius)
                        val = s% tau100_radius
                     case(l_tau100_lgP)
                        val = s% tau100_lgP
                     case(l_tau100_T)
                        val = 10**s% tau100_lgT
                     case(l_tau100_lgT)
                        val = s% tau100_lgT
                     case(l_tau100_lgRho)
                        val = s% tau100_lgRho
                     case(l_tau100_L)
                        val = s% tau100_L
                     case(l_dynamic_timescale)
                        val = s% dynamic_timescale
                     case(l_kh_timescale)
                        val = s% kh_timescale
                     case(l_nuc_timescale)
                        val = s% nuc_timescale
                     case(l_eps_grav_integral)
                        val = dot_product(s% dm(1:nz), s% eps_grav(1:nz))/Lsun
                     case(l_extra_L)
                        val = dot_product(s% dm(1:nz), s% extra_heat(1:nz))/Lsun
                     case(l_log_extra_L)
                        val = safe_log10(dot_product(s% dm(1:nz), s% extra_heat(1:nz))/Lsun)
                     case(l_log_abs_Lgrav)
                        val = safe_log10(abs(dot_product(s% dm(1:nz), s% eps_grav(1:nz))/Lsun))
                     case(l_log_Lnuc)
                        power_photo = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
                        val = safe_log10(s% power_nuc_burn - power_photo)
                     case(l_log_Lnuc_sub_log_L)
                        power_photo = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
                        val = safe_log10(s% power_nuc_burn - power_photo)
                        val = val - safe_log10(s% L(1)/Lsun)
                     case(l_mass_loc_of_max_eps_nuc)
                        k = maxloc(s% eps_nuc(1:nz), dim=1)
                        val = (s% m(k) - s% dm(k)/2)/Msun
                     case(l_mass_ext_to_max_eps_nuc)
                        k = maxloc(s% eps_nuc(1:nz), dim=1)
                        val = (1d0 - s% q(k) + 0.5d0*s% dq(k))*s% xmstar/Msun
                        
                     case(l_trace_mass_location)
                        val = s% trace_mass_location
                     case(l_trace_mass_radius)
                        val = s% trace_mass_radius
                     case(l_trace_mass_lgT)
                        val = s% trace_mass_lgT
                     case(l_trace_mass_lgRho)
                        val = s% trace_mass_lgRho
                     case(l_trace_mass_L)
                        val = s% trace_mass_L
                     case(l_trace_mass_v)
                        val = s% trace_mass_v
                     case(l_trace_mass_omega)
                        val = s% trace_mass_omega
                     case(l_trace_mass_omega_div_omega_crit)
                        val = s% trace_mass_omega_div_omega_crit

                     case(l_trace_mass_lgP)
                        val = s% trace_mass_lgP
                     case(l_trace_mass_g)
                        val = s% trace_mass_g
                     case(l_trace_mass_X)
                        val = s% trace_mass_X
                     case(l_trace_mass_Y)
                        val = s% trace_mass_Y
                     case(l_trace_mass_edv_H)
                        val = s% trace_mass_edv_H
                     case(l_trace_mass_edv_He)
                        val = s% trace_mass_edv_He
                     case(l_trace_mass_scale_height)
                        val = s% trace_mass_scale_height
                     case(l_trace_mass_dlnX_dr)
                        val = s% trace_mass_dlnX_dr
                     case(l_trace_mass_dlnY_dr)
                        val = s% trace_mass_dlnY_dr
                     case(l_trace_mass_dlnRho_dr)
                        val = s% trace_mass_dlnRho_dr
                        
                     case(l_max_T_lgT)
                        val = s% max_T_lgT
                     case(l_max_T_mass)
                        val = s% max_T_mass
                     case(l_max_T_radius)
                        val = s% max_T_radius
                     case(l_max_T_lgRho)
                        val = s% max_T_lgRho
                     case(l_max_T_L)
                        val = s% max_T_L
                     case(l_max_T_eps_nuc)
                        val = s% max_T_eps_nuc
                        
                     case(l_max_eps_h)
                        val = s% max_eps_h
                     case(l_max_eps_h_lgT)
                        val = s% max_eps_h_lgT
                     case(l_max_eps_h_lgRho)
                        val = s% max_eps_h_lgRho
                     case(l_max_eps_h_m)
                        val = s% max_eps_h_m/Msun
                     case(l_max_eps_h_xm)
                        val = (s% mstar - s% max_eps_h_m)/Msun
                     case(l_max_eps_h_lgR)
                        val = s% max_eps_h_lgR
                     case(l_max_eps_h_lgP)
                        val = s% max_eps_h_lgP
                     case(l_max_eps_h_opacity)
                        val = s% max_eps_h_opacity
                        
                     case(l_max_eps_he)
                        val = s% max_eps_he
                     case(l_max_eps_he_lgT)
                        val = s% max_eps_he_lgT
                     case(l_max_eps_he_lgRho)
                        val = s% max_eps_he_lgRho
                     case(l_max_eps_he_m)
                        val = s% max_eps_he_m/Msun
                     case(l_max_eps_he_xm)
                        val = (s% mstar - s% max_eps_he_m)/Msun
                     case(l_max_eps_he_lgR)
                        val = s% max_eps_he_lgR
                     case(l_max_eps_he_lgP)
                        val = s% max_eps_he_lgP
                     case(l_max_eps_he_opacity)
                        val = s% max_eps_he_opacity
                        
                     case(l_max_eps_z)
                        val = s% max_eps_z
                     case(l_max_eps_z_lgT)
                        val = s% max_eps_z_lgT
                     case(l_max_eps_z_lgRho)
                        val = s% max_eps_z_lgRho
                     case(l_max_eps_z_m)
                        val = s% max_eps_z_m/Msun
                     case(l_max_eps_z_xm)
                        val = (s% mstar - s% max_eps_z_m)/Msun
                     case(l_max_eps_z_lgR)
                        val = s% max_eps_z_lgR
                     case(l_max_eps_z_lgP)
                        val = s% max_eps_z_lgP
                     case(l_max_eps_z_opacity)
                        val = s% max_eps_z_opacity
                        
                     case(l_max_eps_nuc)
                        val = s% max_eps_nuc
                     case(l_max_eps_nuc_lgT)
                        val = s% max_eps_nuc_lgT
                     case(l_max_eps_nuc_lgRho)
                        val = s% max_eps_nuc_lgRho
                     case(l_max_eps_nuc_m)
                        val = s% max_eps_nuc_m/Msun
                     case(l_max_eps_nuc_xm)
                        val = (s% mstar - s% max_eps_nuc_m)/Msun
                     case(l_max_eps_nuc_lgR)
                        val = s% max_eps_nuc_lgR
                     case(l_max_eps_nuc_lgP)
                        val = s% max_eps_nuc_lgP
                     case(l_max_eps_nuc_opacity)
                        val = s% max_eps_nuc_opacity
                     case(l_max_eps_nuc_cp)
                        val = s% max_eps_nuc_cp
                     case(l_max_eps_nuc_t_heat)
                        if (s% max_eps_nuc >= 0) then
                           val = s% max_eps_nuc_cp*10**s% max_eps_nuc_lgT/s% max_eps_nuc
                        else
                           val = 0
                        end if

                     case(l_max_eps_nuc_csound)
                        val = s% csound(s% max_eps_nuc_k)
                     case(l_max_eps_nuc_pi_r_div_cs)
                        val = pi*s% r(s% max_eps_nuc_k)/s% csound(s% max_eps_nuc_k)
                     case(l_max_eps_nuc_H)
                        val = s% scale_height(s% max_eps_nuc_k)
                     case(l_max_eps_nuc_H_div_cs)
                        val = s% scale_height(s% max_eps_nuc_k)/s% csound(s% max_eps_nuc_k)

                     case(l_log_surf_opacity)
                        val = s% opacity(1)
                     case(l_log_surf_density)
                        val = s% lnd(1)/ln10
                     case(l_surface_temperature)
                        val = s% T_surf
                     case(l_log_surf_temperature)
                        val = safe_log10(s% T_surf)
                     case(l_surface_optical_depth)
                        val = s% tau_base*s% tau_factor
                     case(l_log_surf_optical_depth)
                        val = safe_log10(s% tau_base*s% tau_factor)
                     case(l_log_surf_pressure)
                        val = safe_log10(s% P_surf)
                     case(l_log_surf_z)
                        val = 0
                        if (s% net_iso(ih1) /= 0) val = val + s% xa(s% net_iso(ih1),1)
                        if (s% net_iso(ih2) /= 0) val = val + s% xa(s% net_iso(ih2),1)
                        if (s% net_iso(ihe3) /= 0) val = val + s% xa(s% net_iso(ihe3),1)
                        if (s% net_iso(ihe4) /= 0) val = val + s% xa(s% net_iso(ihe4),1)
                        val = safe_log10(1d0 - val)
                     case(l_log_Ledd)
                        val = safe_log10(eval_Ledd(s)/Lsun)
                        
                     case(l_dt_Courant)
                        val = dt_Courant(s)
                     case(l_log_dt_Courant)
                        val = safe_log10(dt_Courant(s))
                     case(l_dt_Courant_yr)
                        val = dt_Courant(s)/secyer
                     case(l_log_dt_Courant_yr)
                        val = safe_log10(dt_Courant(s)/secyer)
                     case(l_dt_div_dt_Courant)
                        val = s% dt/dt_Courant(s)
                     case(l_log_dt_div_dt_Courant)
                        val = safe_log10(s% dt/dt_Courant(s))

                     case(l_cz_mass)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% m(k)/Msun
                        end if
                     case(l_cz_log_xmass)
                        if (s% largest_conv_mixing_region == 0) then
                           val = -99
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = safe_log10(s% xmstar*sum(s% dq(1:k-1)))
                        end if
                     case(l_cz_log_xmsun)
                        if (s% largest_conv_mixing_region == 0) then
                           val = -99
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/Msun)
                        end if
                     case(l_cz_xm)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% xmstar*sum(s% dq(1:k-1))/Msun
                        end if
                     case(l_cz_logT)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% lnT(k)/ln10
                        end if
                     case(l_cz_logRho)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% lnd(k)/ln10
                        end if
                     case(l_cz_logP)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% lnP(k)/ln10
                        end if
                     case(l_cz_log_column_depth)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/(4*pi*s% r(k)**2))
                        end if
                     case(l_cz_log_radial_depth)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = safe_log10(s% r(1) - s% r(k))
                        end if
                     case(l_cz_luminosity)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% L(k)/Lsun
                        end if
                     case(l_cz_log_tau)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = safe_log10(s% tau(k))
                        end if
                     case(l_cz_opacity)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% opacity(k)
                        end if
                     case(l_cz_log_eps_nuc)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = safe_log10(s% eps_nuc(k))
                        end if
                     case(l_cz_t_heat)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           if (s% eps_nuc(k) <= 0) then
                              val = 1d99
                           else
                              val = s% Cp(k)*s% T(k)/s% eps_nuc(k)
                           end if
                        end if
                     case(l_cz_eta)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% eta(k)
                        end if
                     case(l_cz_csound)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% csound(k)
                        end if
                     case(l_cz_scale_height)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% scale_height(k)
                        end if
                     case(l_cz_grav)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% grav(k)
                        end if
                     case(l_cz_radius)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% R(k)/Rsun
                        end if
                     case(l_cz_zone)
                        if (s% largest_conv_mixing_region == 0) then
                           k = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                        end if
                        int_val = k
                        is_int_val = .true.
                     case(l_cz_omega)
                        if (s% largest_conv_mixing_region == 0 .or. .not. s% rotation_flag) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = s% omega(k)
                        end if
                     case(l_cz_omega_div_omega_crit)
                        if (s% largest_conv_mixing_region == 0 .or. .not. s% rotation_flag) then
                           val = 0
                        else
                           k = s% mixing_region_bottom(s% largest_conv_mixing_region)
                           val = min(1d0,s% omega(k)/omega_crit(s,k))
                        end if

                     case(l_cz_top_mass)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% m(k)/Msun
                        end if
                     case(l_cz_top_log_xmass)
                        if (s% largest_conv_mixing_region == 0) then
                           val = -99
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = safe_log10(s% xmstar*sum(s% dq(1:k-1)))
                        end if
                     case(l_cz_top_log_xmsun)
                        if (s% largest_conv_mixing_region == 0) then
                           val = -99
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/Msun)
                        end if
                     case(l_cz_top_xm)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% xmstar*sum(s% dq(1:k-1))/Msun
                        end if
                     case(l_cz_top_logT)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% lnT(k)/ln10
                        end if
                     case(l_cz_top_logRho)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% lnd(k)/ln10
                        end if
                     case(l_cz_top_logP)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% lnP(k)/ln10
                        end if
                     case(l_cz_top_log_column_depth)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = safe_log10(s% xmstar*sum(s% dq(1:k-1))/(4*pi*s% r(k)**2))
                        end if
                     case(l_cz_top_log_radial_depth)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = safe_log10(s% r(1) - s% r(k))
                        end if
                     case(l_cz_top_luminosity)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% L(k)/Lsun
                        end if
                     case(l_cz_top_log_tau)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = safe_log10(s% tau(k))
                        end if
                     case(l_cz_top_opacity)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% opacity(k)
                        end if
                     case(l_cz_top_log_eps_nuc)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = safe_log10(s% eps_nuc(k))
                        end if
                     case(l_cz_top_t_heat)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           if (s% eps_nuc(k) <= 0) then
                              val = 1d99
                           else
                              val = s% Cp(k)*s% T(k)/s% eps_nuc(k)
                           end if
                        end if
                     case(l_cz_top_eta)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% eta(k)
                        end if
                     case(l_cz_top_csound)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% csound(k)
                        end if
                     case(l_cz_top_scale_height)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% scale_height(k)
                        end if
                     case(l_cz_top_grav)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% grav(k)
                        end if
                     case(l_cz_top_radius)
                        if (s% largest_conv_mixing_region == 0) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% R(k)/Rsun
                        end if
                     case(l_cz_top_zone_logdq)
                        if (s% largest_conv_mixing_region == 0) then
                           k = 0
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = safe_log10(s% dq(k))
                        end if
                     case(l_cz_top_zone)
                        if (s% largest_conv_mixing_region == 0) then
                           k = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                        end if
                        int_val = k
                        is_int_val = .true.
                     case(l_cz_top_omega)
                        if (s% largest_conv_mixing_region == 0 .or. .not. s% rotation_flag) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = s% omega(k)
                        end if
                     case(l_cz_top_omega_div_omega_crit)
                        if (s% largest_conv_mixing_region == 0 .or. .not. s% rotation_flag) then
                           val = 0
                        else
                           k = s% mixing_region_top(s% largest_conv_mixing_region)
                           val = min(1d0,s% omega(k)/omega_crit(s,k))
                        end if
                        
                     case(l_max_conv_vel_div_csound)
                        val = 0
                        do k = 1, nz
                           if (s% q(k) > s% max_conv_vel_div_csound_maxq) cycle
                           csound = sqrt(s% gamma1(k)*s% P(k)/s% rho(k))
                           if (s% conv_vel(k)/csound > val) val = s% conv_vel(k)/csound
                        end do
                        
                     case(l_min_t_eddy)
                        val = 1d99
                        do k = 1, nz
                           if (s% conv_vel(k) <= 0) cycle
                           if (s% scale_height(k)/s% conv_vel(k) < val) &
                              val = s% scale_height(k)/s% conv_vel(k)
                        end do
                        
                     case(l_elapsed_time)
                        val = s% total_elapsed_time
                        
                     case(l_delta_nu)
                        val = 1d6/(2*s% photosphere_acoustic_r)
                     case(l_delta_Pg)
                        val = s% delta_Pg
                     case(l_log_delta_Pg)
                        val = safe_log10(s% delta_Pg)
                     case(l_nu_max)
                        val = s% nu_max
                     case(l_nu_max_3_4th_div_delta_nu)
                        val = s% nu_max**0.75d0/(1d6/(2*s% photosphere_acoustic_r))
                     case(l_acoustic_cutoff)
                        val = s% acoustic_cutoff
                     case(l_gs_per_delta_nu)
                        if (s% nu_max <= 0 .or. s% delta_Pg <= 0) then
                           val = 0
                        else
                           val = 1d6/(2*s% photosphere_acoustic_r) ! delta_nu
                           val = 1d6*val/(s% nu_max**2*s% delta_Pg)
                        end if
                     case(l_ng_for_nu_max)
                        if (s% nu_max <= 0 .or. s% delta_Pg <= 0) then
                           val = 0
                        else
                           val = 1d6/(s% nu_max*s% delta_Pg)
                        end if

                     case(l_int_k_r_dr_nu_max_Sl1)
                        val = get_int_k_r_dr(1,1.0d0)
                     case(l_int_k_r_dr_2pt0_nu_max_Sl1)
                        val = get_int_k_r_dr(1,2d0)
                     case(l_int_k_r_dr_0pt5_nu_max_Sl1)
                        val = get_int_k_r_dr(1,0.5d0)
                     case(l_int_k_r_dr_nu_max_Sl2)
                        val = get_int_k_r_dr(2,1.0d0)
                     case(l_int_k_r_dr_2pt0_nu_max_Sl2)
                        val = get_int_k_r_dr(2,2d0)
                     case(l_int_k_r_dr_0pt5_nu_max_Sl2)
                        val = get_int_k_r_dr(2,0.5d0)
                     case(l_int_k_r_dr_nu_max_Sl3)
                        val = get_int_k_r_dr(3,1.0d0)
                     case(l_int_k_r_dr_2pt0_nu_max_Sl3)
                        val = get_int_k_r_dr(3,2d0)
                     case(l_int_k_r_dr_0pt5_nu_max_Sl3)
                        val = get_int_k_r_dr(3,0.5d0)
                                       
                     case (l_op_split_diff)
                        val = s% op_split_diff
                                       
                     case (l_log_err_ratio_max)
                        val = safe_log10(s% err_ratio_max_hydro)
                     case (l_log_err_ratio_avg)
                        val = safe_log10(s% err_ratio_avg_hydro)
                     case (l_seulex_rows)
                        val = s% seulex_rows

                     case (l_min_L)
                        val = minval(s% L(1:nz))/Lsun
                     case (l_offset_FL)
                        val = s% FL_offset
                        
                     case default
                        write(*,*) 'missing log info for ' // trim(log_column_name(c)), j, k
                        return
                        
                  end select
               end if
               if (is_int_val) then
                  call do_int_val(j,int_val)
               else
                  call do_val(j,val)
               end if
            end if
            
         end subroutine do_col
         
         
         real(dp) function get_int_k_r_dr(el, nu_factor)
            use utils_lib, only: is_bad_num
            integer, intent(in) :: el
            real(dp), intent(in) :: nu_factor
            
            real(dp) :: integral, cs2, r2, n2, sl2, omega2, &
               L2, kr2, dr, r0_outer, r0_inner, sl2_next
            integer :: k, k1, k_inner, k_outer  
            
            logical :: dbg
               
            include 'formats.dek'
            
            dbg = .false. !(el == 1 .and. nu_factor == 1d0)
            
            
            get_int_k_r_dr = 0
            L2 = el*(el+1)
            omega2 = (1d-6*2*pi*s% nu_max*nu_factor)**2
            
            ! k_inner and k_outer are bounds of evanescent region
            
            ! k_outer is outermost k where Sl2 <= omega2 at k-1 and Sl2 > omega2 at k
            ! 1st find outermost where Sl2 <= omega2
            k1 = 0
            do k = 1, s% nz
               r2 = s% r(k)**2
               cs2 = s% csound_at_face(k)**2
               sl2 = L2*cs2/r2
               if (sl2 <= omega2) then
                  k1 = k; exit
               end if
            end do
            if (k1 == 0) return
            ! then find next k where Sl2 >= omega2
            k_outer = 0
            do k = k1+1, s% nz
               r2 = s% r(k)**2
               cs2 = s% csound_at_face(k)**2
               sl2 = L2*cs2/r2
               if (sl2 > omega2) then
                  k_outer = k; exit
               end if
            end do
            if (k_outer == 0) return
            
            ! k_inner is next k where N2 >= omega2 at k+1 and N2 < omega2 at k
            k_inner = 0
            do k = k_outer+1, s% nz
               if (s% brunt_N2(k) >= omega2) then
                  k_inner= k; exit
               end if
            end do
            if (k_inner == 0) return
            
            integral = 0
            do k = k_inner-1, k_outer, -1
               r2 = s% r(k)**2
               cs2 = s% csound_at_face(k)**2
               n2 = s% brunt_N2(k)
               sl2 = L2*cs2/r2
               if (k == k_inner-1) then
                  ! interpolate to estimate where in cell k N2 == omega2
                  ! use that location in place of rmid(k)
                  if (n2 <= omega2 .and. omega2 <= s% brunt_N2(k+1)) then
                     r0_inner = (find0(s% r(k)**3, n2-omega2, s% r(k+1)**3, s% brunt_N2(k+1)-omega2))**(1d0/3d0)
                     if (r0_inner < s% r(k+1) .or. r0_inner > s% r(k)) then
                        if (dbg) write(*,1) 'r0_inner confusion', s% r(k+1), r0_inner, s% r(k)
                     end if
                  else
                     r0_inner = s% r(k+1)
                     if (dbg) write(*,2) 'N2 == omega2 failed for k_inner', k_inner
                  end if
                  dr = s% rmid(k-1) - r0_inner
               else if (k == k_outer) then ! find where in cell k-1 Sl2 == omega2
                  ! use that location in place of rmid(k-1)
                  sl2_next = L2*(s% csound(k-1)/s% r(k-1))**2
                  if (sl2 >= omega2 .and. omega2 >= sl2_next) then
                     r0_outer = (find0(s% r(k)**3, sl2-omega2, s% r(k-1)**3, sl2_next-omega2))**(1d0/3d0)
                     if (r0_outer < s% r(k) .or. r0_outer > s% r(k-1)) then
                        if (dbg) write(*,1) 'r0_outer confusion', s% r(k), r0_outer, s% r(k-1)
                     end if
                  else
                     r0_outer = s% r(k-1)
                     if (dbg) write(*,2) 'Sl2 == omega2 failed for k_outer', k_outer
                  end if
                  dr = r0_outer - s% rmid(k)
               else
                  dr = s% rmid(k-1) - s% rmid(k)
               end if
               kr2 = (1 - n2/omega2)*(1 - Sl2/omega2)*omega2/cs2
               if (kr2 < 0 .and. n2 < omega2 .and. omega2 < Sl2) &
                  integral = integral + sqrt(-kr2)*dr
            end do
            
            
            if (integral == 0) return
            get_int_k_r_dr = 2*integral/ln10
            
            if (dbg) write(*,3) 'r0 inner outer', &
               k_inner, k_outer, r0_inner/Rsun, r0_outer/Rsun, get_int_k_r_dr
               
            if (.not. is_bad_num(get_int_k_r_dr)) return
            
            write(*,2) 'el', el
            write(*,1) 'nu_factor', nu_factor
            write(*,1) 's% nu_max*nu_factor', s% nu_max*nu_factor
            write(*,1) 'log10 nu_max*nu_factor', log10(s% nu_max*nu_factor)
            
            write(*,1) 'get_int_k_r_dr', get_int_k_r_dr
            write(*,1) 'integral', integral
            write(*,2) 'k_inner', k_inner
            write(*,2) 'k_outer', k_outer
            stop 'get_int_k_r_dr'
            
         end function get_int_k_r_dr
         
         
         subroutine do_val(j, val)
            integer, intent(in) :: j
            real(dp), intent(in) :: val
            if (write_flag) then
               write(io, fmt=dbl_fmt, advance='no') val
            else
               vals(j) = val
               is_int(j) = .false.
            end if
         end subroutine do_val
         
         
         subroutine do_int_val(j, val)
            integer, intent(in) :: j
            integer, intent(in) :: val
            if (write_flag) then
               write(io, fmt=int_fmt, advance='no') val
            else
               vals(j) = dble(val)
               is_int(j) = .true.
            end if
         end subroutine do_int_val
         
         
         real(dp) function max_eps_nuc_log_x(j)
            integer, intent(in) :: j
            real(dp) :: sum_x, sum_dq
            integer :: k
            max_eps_nuc_log_x = 0
            if (j == 0) return
            k = maxloc(s% eps_nuc, dim=1)
            if (k < 1 .or. k > s% nz) return
            max_eps_nuc_log_x = s% xa(j,k)
         end function max_eps_nuc_log_x 


         real(dp) function cz_top_max_log_x(j)
            integer, intent(in) :: j
            real(dp) :: sum_x, sum_dq
            integer :: k
            cz_top_max_log_x = 0
            if (s% largest_conv_mixing_region == 0) return
            k = s% mixing_region_top(s% largest_conv_mixing_region)
            if (j == 0 .or. k <= 1) return
            cz_top_max_log_x = s% xa(j,k-1)
         end function cz_top_max_log_x         


         real(dp) function cz_max_log_x(j)
            integer, intent(in) :: j
            real(dp) :: sum_x, sum_dq
            integer :: k
            cz_max_log_x = 0
            if (s% largest_conv_mixing_region == 0) return
            k = s% mixing_region_bottom(s% largest_conv_mixing_region)
            if (j == 0 .or. k <= 1) return
            cz_max_log_x = s% xa(j,k-1)
         end function cz_max_log_x         


         real(dp) function central_avg_x(j)
            integer, intent(in) :: j
            real(dp) :: sum_x, sum_dq
            integer :: k
            if (j == 0) then
               central_avg_x = 0
               return
            end if
            sum_x = 0
            sum_dq = 0
            do k = s% nz, 1, -1
               sum_x = sum_x + s% xa(j,k)*s% dq(k)
               sum_dq = sum_dq + s% dq(k)
               if (sum_dq >= s% center_avg_value_dq) exit
            end do
            central_avg_x = sum_x/sum_dq
         end function central_avg_x         


         real(dp) function surface_avg_x(j)
            integer, intent(in) :: j
            real(dp) :: sum_x, sum_dq
            integer :: k
            if (j == 0) then
               surface_avg_x = 0
               return
            end if
            sum_x = 0
            sum_dq = 0
            do k = 1, s% nz
               sum_x = sum_x + s% xa(j,k)*s% dq(k)
               sum_dq = sum_dq + s% dq(k)
               if (sum_dq >= s% surface_avg_abundance_dq) exit
            end do
            surface_avg_x = sum_x/sum_dq
         end function surface_avg_x         


         real(dp) function star_avg_x(j)
            integer, intent(in) :: j
            if (j == 0) then
               star_avg_x = 0
               return
            end if
            star_avg_x = dot_product(s% xa(j,1:nz),s% dq(1:nz))/sum(s% dq(1:nz))
         end function star_avg_x         


         real(dp) function category_L(i)
            integer, intent(in) :: i
            if (j == 0) then
               category_L = 0
               return
            end if
            category_L = &
               safe_log10(dot_product(s% eps_nuc_categories(i_rate,i,1:nz),s% dm(1:nz))/Lsun)
         end function category_L         


         real(dp) function reaction_eps_nuc(i)
            integer, intent(in) :: i
            if (i == 0) then
               reaction_eps_nuc = 0
               return
            end if
            reaction_eps_nuc = &
               safe_log10(dot_product(s% reaction_eps_nuc(i_rate,i,1:nz),s% dm(1:nz))/Lsun)
         end function reaction_eps_nuc         
                  
         
      end subroutine do_log_info


      subroutine write_integer(io, col, pass, name, val)
         integer, intent(in) :: io, pass
         integer, intent(inout) :: col
         character (len=*), intent(in) :: name
         integer, intent(in) :: val
         if (pass == 1) then
            col = col+1
            write(io, fmt='(i28, 1x)', advance='no') col
         else if (pass == 2) then
            write(io, fmt='(a28, 1x)', advance='no') trim(name)
         else if (pass == 3) then
            write(io, fmt='(i28, 1x)', advance='no') val
         end if
      end subroutine write_integer
      
      
      subroutine write_val(io, col, pass, name, val)
         integer, intent(in) :: io, pass
         integer, intent(inout) :: col
         character (len=*), intent(in) :: name
         real(dp), intent(in) :: val
         if (pass == 1) then
            col = col+1
            write(io, fmt='(i28, 1x)', advance='no') col
         else if (pass == 2) then
            write(io, fmt='(a28, 1x)', advance='no') trim(name)
         else if (pass == 3) then
            write(io, fmt='(1pe28.16e3, 1x)', advance='no') val
         end if
      end subroutine write_val

      

      end module log

