! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module init

      use star_private_def
      use const_def
      use alert_lib
      use read_model

      implicit none
      
      integer, parameter :: do_create_pre_ms_model = 0
      integer, parameter :: do_load_zams_model = 1
      integer, parameter :: do_load_saved_model = 2

      contains
      
      
      subroutine do_star_init( &
            data_dir, chem_isotopes_filename, &
            kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
            kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, other_kappa_file_prefix, &
            eos_file_prefix, other_eos_prefix, eosDT_Z1_suffix, eosPT_Z1_suffix, &
            net_reaction_filename, rate_tables_dir, rates_cache_suffix, &
            ionization_file_prefix, ionization_Z1_suffix, &
            ierr)
         use hydro_rotation, only: init_rotation
         use alloc, only: init_alloc
         character (len=*), intent(in) :: &
            data_dir, chem_isotopes_filename, net_reaction_filename, rate_tables_dir, &
            kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
            other_kappa_file_prefix, &
            eosDT_Z1_suffix, eosPT_Z1_suffix, &
            eos_file_prefix, other_eos_prefix, rates_cache_suffix, &
            ionization_file_prefix, ionization_Z1_suffix
         real(dp), intent(in) :: kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy
         integer, intent(out) :: ierr
         ! ierr will be 0 for a normal return. 
         ! ierr nonzero means something went wrong.
         integer :: iam, nprocs, nprow, npcol
         include 'formats.dek'
         ierr = 0
         call init_alloc
         !call superlu_dist_before(iam, nprocs, nprow, npcol, ierr)
            ! for iam /= 0, this doesn't return until it is time to quit
         if (ierr /= 0) then
            write(*,*) 'superlu_dist_before returned ierr', ierr
            return
         end if
         !if (iam /= 0) then
         !   call superlu_dist_after(ierr)
         !   stop 'do_star_init'
         !end if       
         call stardata_init( &
            data_dir, chem_isotopes_filename, &
            kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
            kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, other_kappa_file_prefix, &
            eos_file_prefix, other_eos_prefix, eosDT_Z1_suffix, eosPT_Z1_suffix, &
            net_reaction_filename, rate_tables_dir, rates_cache_suffix, &
            ionization_file_prefix, ionization_Z1_suffix, &
            ierr)
      	if (ierr /= 0) then
            write(*,*) 'failed in stardata_init'
            return
      	end if
         call init_rotation(ierr)
      	if (ierr /= 0) then
            write(*,*) 'failed in init_rotation'
            return
      	end if
         !if (nprow*npcol > 1) then
         !   write(*,'(a,x,3i3)') 'MPI nprocs nprow npcol', nprocs, nprow, npcol
         !end if
      end subroutine do_star_init


      subroutine do_star_shutdown
         !use mtx_lib, only: superlu_dist_quit_work, superlu_dist_after
         use micro, only: shutdown_microphys
         integer :: ierr
         !call superlu_dist_quit_work(ierr)  
         !call superlu_dist_after(ierr)
         !call shutdown_microphys ! skip this for now
      end subroutine do_star_shutdown
            
      
      integer function alloc_star_data(ierr)
         use kap_lib
         use eos_lib
         use rates_def, only: rates_reaction_id_max, rates_NACRE_if_available
         use chem_def, only: num_categories
         use micro, only: default_set_which_rates, default_set_rate_factors
         use mod_other_wind, only: null_other_wind
         use mod_other_torque, only: default_other_torque
         use mod_other_energy, only: default_other_energy
         use mod_other_energy_implicit, only: default_other_energy_implicit
         use mod_other_mixing, only: null_other_mixing
         use mod_other_atm, only: null_other_atm
         use mod_other_diffusion, only: null_other_diffusion
         use mod_other_mlt, only: null_other_mlt
         use mod_other_cgrav, only: default_other_cgrav
         use mod_other_pgstar_plots, only: null_other_pgstar_plots_info
         use mod_other_mesh_functions
         use mod_other_eos
         use mod_other_kap
         
         
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s
         integer, parameter :: init_alloc_nvar = 20
         character (len=32) :: extra_name
         integer :: i
                  
         ierr = 0
         
         alloc_star_data = alloc_star(ierr)
         if (ierr /= 0) then
            call alert(ierr, 'failed in alloc_star')
            return
         end if
         
         call get_star_ptr(alloc_star_data, s, ierr)
         if (ierr /= 0) return

         s% eos_handle = alloc_eos_handle(ierr)
         if (ierr /= 0) then
            call alert(ierr, 'failed in alloc_eos_handle')
            return
         end if
         
         s% other_eos_handle = -1
         
         s% kap_handle = alloc_kap_handle(ierr)
         if (ierr /= 0) then
            call alert(ierr, 'failed in alloc_kap_handle')
            return
         end if
         
         s% other_kap_handle = -1
         
         s% net_handle = 0
         
         s% generations = 0
         
         s% total_num_jacobians = 0
         
         s% nvar_hydro = 0                 
         s% nvar_chem = 0                 
         s% nvar = 0     
                     
         s% nz = 0     
         s% net_name = ''  
         s% species = 0                  
         s% num_reactions = 0  
               
         s% nz_old = 0         
         s% nz_older = 0         
         
         s% lnPgas_flag = .false.
         s% lnE_flag = .false.
         s% lnTdot_flag = .false.
         s% lnddot_flag = .false.
         s% v_flag = .false.
         s% rotation_flag = .false.
         
         s% just_wrote_terminal_header = .false.
         s% just_did_backup = .false.
         s% prev_create_atm_R0_div_R = -1
         
         s% dt = -1
         s% dt_next = -1
         
         s% FL_offset = s% min_FL_offset
         s% min_L = 0
         s% min_L_old = 0
         s% min_L_older = 0
         
         s% i_xlnd = 0
         s% i_lnT = 0
         s% i_lnR = 0
         s% i_FL = 0
         s% i_lnPgas = 0
         s% i_lnE = 0
         s% i_vel = 0
         s% i_lnTdot = 0
         s% i_lnddot = 0
         s% i_chem1 = 0
         
         s% equP = 0
         s% equT = 0
         s% equR = 0 
         s% equL = 0
         s% equv = 0
         s% equlnE = 0
         s% equlnTdot = 0
         s% equlnddot = 0
         s% equchem1 = 0
         
         nullify(s% dq)
         nullify(s% dq_old)
         nullify(s% dq_older)

         nullify(s% conv_vel)
         nullify(s% conv_vel_old)
         nullify(s% conv_vel_older)

         nullify(s% mixing_type)
         nullify(s% mixing_type_old)
         nullify(s% mixing_type_older)

         nullify(s% q)
         nullify(s% q_old)
         nullify(s% q_older)

         nullify(s% xa)
         nullify(s% xa_old)
         nullify(s% xa_older)

         nullify(s% xh)
         nullify(s% xh_old)
         nullify(s% xh_older)

         nullify(s% atm_structure)
         s% atm_structure_num_pts = 0
         
         nullify(s% chem_id)

         nullify(s% which_rates)
         s% set_which_rates => default_set_which_rates
         
         nullify(s% rate_factors)
         s% set_rate_factors => default_set_rate_factors
         
         allocate(s% nameofvar(init_alloc_nvar),stat=ierr)
         if (ierr /= 0) return
         
         allocate(s% nameofequ(init_alloc_nvar),stat=ierr)
         if (ierr /= 0) return
         
         allocate(s% ode_var(init_alloc_nvar),stat=ierr)
         if (ierr /= 0) return

         allocate(s% category_factors(num_categories))
         
         do i = 1, max_num_profile_extras
            if (i < 10) then
               write(extra_name,'(a,i1)') 'extra_', i
            else
               write(extra_name,'(a,i2)') 'extra_', i
            end if
            s% profile_extra_name(i) = trim(extra_name)
         end do

         nullify(s% other_star_info)

         nullify(s% ipar_decsol)
         nullify(s% hydro_iwork)
         nullify(s% rpar_decsol)
         nullify(s% hydro_work)
         
         nullify(s% seulex_error_vectors)
         s% hydro_seulex_dt_limit = 0
         s% seulex_kopt = 0
         s% seulex_rows = 0
         
         s% other_wind => null_other_wind
         s% other_mixing => null_other_mixing
         s% other_torque => default_other_torque
         s% other_energy => default_other_energy
         s% other_energy_implicit => default_other_energy_implicit
         s% other_cgrav => default_other_cgrav
         s% other_atm => null_other_atm
         s% other_diffusion => null_other_diffusion
         s% other_mlt => null_other_mlt

         s% other_eosDT_get => null_other_eosDT_get
         s% other_eosDT_get_T => null_other_eosDT_get_T
         s% other_eosDT_get_Rho => null_other_eosDT_get_Rho
         
         s% other_eosPT_get => null_other_eosPT_get
         s% other_eosPT_get_T => null_other_eosPT_get_T
         s% other_eosPT_get_Pgas => null_other_eosPT_get_Pgas
         s% other_eosPT_get_Pgas_for_Rho => null_other_eosPT_get_Pgas_for_Rho

         s% other_kap_get_Type1 => null_other_kap_get_Type1
         s% other_kap_get_Type2 => null_other_kap_get_Type2

         s% other_pgstar_plots_info => null_other_pgstar_plots_info
         s% how_many_other_mesh_fcns => null_how_many_other_mesh_fcns
         s% other_mesh_fcn_data => null_other_mesh_fcn_data
         
      end function alloc_star_data    
      

      subroutine create_pre_ms_model(id, ierr)
         use star_utils, only: std_write_internals_to_file
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         character (len=0) :: model_dir
         call model_builder( &
            id, model_dir, do_create_pre_ms_model, &
            .false., 'restart_photo', ierr)
      end subroutine create_pre_ms_model
      

      subroutine load_zams_model(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         call model_builder( &
            id, '', do_load_zams_model, &
            .false., 'restart_photo', ierr)
      end subroutine load_zams_model
      

      subroutine load_saved_model(id, model_fname, ierr)
         integer, intent(in) :: id
         character (len=*), intent(in) :: model_fname
         integer, intent(out) :: ierr
         call model_builder( &
            id, model_fname, do_load_saved_model, &
            .false., 'restart_photo', ierr)
      end subroutine load_saved_model
      

      subroutine load_restart_photo(id, restart_filename, ierr)
         integer, intent(in) :: id
         character (len=*), intent(in) :: restart_filename
         integer, intent(out) :: ierr
         call model_builder( &
            id, '', do_load_zams_model, .true., restart_filename, ierr)
      end subroutine load_restart_photo


      ! for both zams and pre-main-sequence
      subroutine model_builder( &
            id, model_info, do_which, restart, restart_filename, ierr)
         use micro, only: init_mesa_micro
         use alloc, only: set_var_info
         use model_in, only: read_restart
         use init_model, only: get_zams_model
         use hydro_vars, only: set_vars
         use report, only: do_report
         use do_one_utils, only: set_phase_of_evolution
         use evolve_support, only: yrs_for_init_timestep
         use adjust_xyz, only: set_z, set_y
         use pre_ms_model, only: build_pre_ms_model
         use read_model, only: do_read_saved_model
         use relax, only: do_relax_to_limit, do_relax_mass, &
            do_relax_mass_scale, do_relax_num_steps
         integer, intent(in) :: id, do_which
         character (len=*), intent(in) :: model_info, restart_filename
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s
         real(dp) :: initial_mass, initial_z, dlgm_per_step
         
         real(dp), parameter :: lg_max_abs_mdot = -1000
         real(dp), parameter :: change_mass_years_for_dt = 1
         real(dp), parameter :: min_mass_for_create_pre_ms = 0.03d0
         logical :: restore_at_end
         integer :: terminal_cnt, max_model_number, save_D_norm_kappa

         include 'formats.dek'
         
         ierr = 0
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            call alert(ierr, 'bad star id')
            return
         end if

         initial_mass = s% initial_mass
         initial_z = s% initial_z
         
         if (restart) then
            s% doing_first_model_of_run = .false.
            call read_restart(s, restart_filename, ierr)            
            if (ierr /= 0) return         
            s% M_center = s% mstar - s% xmstar
            call check_initials
            call init_mesa_micro(s, ierr)
            if (ierr /= 0) return
            call finish_load_model(s, ierr)
            if (s% max_years_for_timestep > 0) &
               s% dt_next = min(s% dt_next, secyer*s% max_years_for_timestep)
            return
         end if

         s% doing_first_model_of_run = .true.
         call set_zero_age_params(s)
         
         if (do_which == do_load_saved_model) then
            s% dt_next = -1
            call do_read_saved_model(s, model_info, ierr)
            if (ierr /= 0) return
            call check_initials
            if (s% dt_next < 0) s% dt_next = yrs_for_init_timestep(s)*secyer
         else
            s% net_name = 'basic.net'
            s% species = 0
            s% lnPgas_flag = .false.
            s% lnE_flag = .false.
            s% lnTdot_flag = .false.
            s% lnddot_flag = .false.
            s% v_flag = .false.
            s% rotation_flag = .false.
            s% star_mass = s% initial_mass
            s% mstar = s% initial_mass*Msun
            s% M_center = s% mstar - s% xmstar
            call set_var_info(s, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in set_var_info'
               return
            end if
            call init_mesa_micro(s, ierr) ! uses s% net_name
            if (ierr /= 0) then
               write(*,*) 'failed in init_mesa_micro'
               return
            end if
            select case (do_which)
               case (do_create_pre_ms_model)
                  if (.true. .and. s% initial_mass < min_mass_for_create_pre_ms) then
                     write(*,*)
                     write(*,*)
                     write(*,*)
                     write(*,'(a,1x,f5.2)') 'sorry: cannot create pre-ms smaller than', &
                        min_mass_for_create_pre_ms
                     write(*,'(a)') &
                        'please create pre-ms and then relax to lower mass as a separate operation'
                     write(*,*)
                     write(*,'(a)') 'here is an example:'
                     write(*,'(a)') 'in your inlist &controls section, set initial_mass = 0.03'
                     write(*,'(a)') 'in the &star_job section, add something like these lines'
                     write(*,'(a)') '  relax_mass_scale = .true.'
                     write(*,'(a)') '  dlgm_per_step = 1d-3 ! log10(delta M/Msun/step)'
                     write(*,'(a)') '  new_mass = 2.863362d-3 ! 3 Mjupiter in Msun units'
                     write(*,'(a)') '  change_mass_years_for_dt = 1'
                     write(*,*)
                     write(*,*)
                     write(*,*)
                     ierr = -1
                     return
                  end if
                  call build_pre_ms_model(id, s, s% nvar_hydro, s% species, ierr)
                  if (ierr /= 0) then
                     write(*,*) 'failed in build_pre_ms_model'
                     return
                  end if
                  s% generations = 1
                  s% dt_next = 1d-5*secyer
                  !write(*,'(a)') ' done create pre main-sequence model'
                  !write(*,*)
               case (do_load_zams_model)
                  s% generations = 1
                  call get_zams_model(s, s% zams_filename, ierr)
                  if (ierr /= 0) then
                     write(*,*) 'failed in get_zams_model'
                     return
                  end if
                  if (s% dt_next < 0) s% dt_next = yrs_for_init_timestep(s)*secyer
               case default
                  write(*,*) 'bad value for do_which in build_model'
                  ierr = -1
                  return
            end select
         end if

         s% extra_heat(1:s% nz) = 0
         s% rate_factors(1:s% num_reactions) = 1

         call finish_load_model(s, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in finish_load_model'
            return
         end if
         if (s% max_years_for_timestep > 0) &
            s% dt_next = min(s% dt_next, secyer*s% max_years_for_timestep)
         call set_phase_of_evolution(s)
         
         if (do_which == do_create_pre_ms_model) then
            save_D_norm_kappa = s% D_norm_kappa 
            s% D_norm_kappa = 0 ! turn off while do relax
            if (s% mstar > s% initial_mass*Msun) then ! need to reduce mass
               write(*,1) 'reduce mass to', s% initial_mass
               call do_relax_mass(s% id, s% initial_mass, lg_max_abs_mdot, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in do_relax_mass'
                  return
               end if
            else if (s% mstar < s% initial_mass*Msun) then ! need to increase mass
               write(*,1) 'increase mass to', s% initial_mass
               call do_relax_mass(s% id, s% initial_mass, lg_max_abs_mdot, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in do_relax_mass'
                  return
               end if
            end if
            call do_relax_num_steps( &
               s% id, s% pre_ms_relax_num_steps, s% dt_next, ierr) 
            if (ierr /= 0) then
               write(*,*) 'failed in do_relax_num_steps'
               return
            end if
            s% D_norm_kappa = save_D_norm_kappa
         end if

         s% doing_first_model_of_run = .true.
         
         contains
         
         subroutine check_initials
            include 'formats.dek'
            if (abs(initial_mass - s% initial_mass) > 1d-3*initial_mass .and. initial_mass > 0) then
               write(*,1) "WARNING -- inlist initial_mass ignored", initial_mass
               write(*,1) "using saved initial_mass instead", s% initial_mass
               write(*,*)
            end if
            if (abs(initial_z - s% initial_z) > 1d-3*initial_z .and. initial_z > 0) then
               write(*,1) "WARNING -- inlist initial_z ignored", initial_z
               write(*,1) "using saved initial_z instead", s% initial_z
               write(*,*)
            end if
         end subroutine check_initials
         
      end subroutine model_builder
      
      
      logical function doing_restart()
         integer :: ios
         open(unit=99, file='restart_photo', action='read', status='old', iostat=ios)
         if (ios == 0) then
            doing_restart = .true.
            close(99)
         else
            doing_restart = .false.
         end if
      end function doing_restart

         
      end module init
