! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module hydro_vars
      
      use star_private_def
      use const_def
      use chem_def, only: chem_isos
      use utils_lib, only:is_bad_num
      use alert_lib
      
      implicit none

      logical, parameter :: dbg = .false.      

      contains
      
      
      subroutine set_vars(s, nzlo, nzhi, dt, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: nzlo, nzhi
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         logical, parameter :: &
            skip_net = .false., skip_neu = .false., skip_kap = .false., &
            skip_grads = .false., skip_rotation = .false., &
            skip_mixing_info = .false.
         if (dbg) write(*, *) 'set_vars', nzlo, nzhi
         call set_some_vars( &
            s, nzlo, nzhi, skip_net, skip_neu, skip_kap, skip_grads, &
            skip_rotation, skip_mixing_info, dt, ierr)
      end subroutine set_vars
      
      
      subroutine set_all_vars_except_mixing_info(s, nzlo, nzhi, dt, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: nzlo, nzhi
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         logical, parameter :: &
            skip_net = .false., skip_neu = .false., skip_kap = .false., &
            skip_grads = .false., skip_rotation = .false., &
            skip_mixing_info = .true.
         if (dbg) write(*, *) 'set_all_vars_except_mixing_info', nzlo, nzhi
         call set_some_vars( &
            s, nzlo, nzhi, skip_net, skip_neu, skip_kap, skip_grads, &
            skip_rotation, skip_mixing_info, dt, ierr)
      end subroutine set_all_vars_except_mixing_info

      
      subroutine set_some_vars( &
            s, nzlo, nzhi, skip_net, skip_neu, skip_kap, skip_grads, &
            skip_rotation, skip_mixing_info, dt, ierr)
         use utils_lib, only: has_bad_num, is_bad_num
         use mlt_def, only: no_mixing
         use star_utils, only: eval_irradiation_heat, FL_to_L
         type (star_info), pointer :: s
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: &
            skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, skip_mixing_info
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr

         integer :: nz, k, j, i_xlnd, i_lnPgas, i_lnE, i_lnT, i_lnTdot, i_lnddot, &
            i_lnR, i_FL, i_vel
         logical, parameter :: skip_other_cgrav = .false.
         
         include 'formats.dek'
         
         if (dbg) write(*, *) 'set_vars', nzlo, nzhi
         
         nz = s% nz
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         i_lnE = s% i_lnE
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         i_FL = s% i_FL
         i_vel = s% i_vel
         i_lnTdot = s% i_lnTdot
         i_lnddot = s% i_lnddot
         
!$OMP PARALLEL DO PRIVATE(j,k)
         do j=1,s% nvar_hydro
            if (j == i_xlnd) then
               do k=nzlo,nzhi
                  s% lnd(k) = s% xh(i_xlnd,k) - lnd_offset
               end do
            else if (j == i_lnPgas) then
               do k=nzlo,nzhi
                  s% lnPgas(k) = s% xh(i_lnPgas,k)
               end do
            else if (j == i_lnE) then 
               ! NOTE: set lnE_var, not lnE (that's for eos value)
               do k=nzlo,nzhi
                  s% lnE_var(k) = s% xh(i_lnE,k)
               end do
            else if (j == i_lnT) then
               do k=nzlo,nzhi
                  s% lnT(k) = s% xh(i_lnT,k)
               end do
            else if (j == i_lnR) then
               do k=nzlo,nzhi
                  s% lnR(k) = s% xh(i_lnR,k)
               end do
            else if (j == i_FL) then
               do k=nzlo,nzhi
                  s% L(k) = FL_to_L(s, s% xh(i_FL, k))
               end do
            else if (j == i_vel) then
               do k=nzlo,nzhi
                  s% v(k) = s% xh(i_vel,k)
               end do
            else if (j == i_lnTdot) then
               do k=nzlo,nzhi
                  s% lnTdot(k) = s% xh(i_lnTdot,k)
               end do
            else if (j == i_lnddot) then
               do k=nzlo,nzhi
                  s% lnddot(k) = s% xh(i_lnddot,k)
               end do
            end if
         end do
!$OMP END PARALLEL DO
         if (i_vel == 0) then
            do k=nzlo,nzhi
               s% v(k) = 0d0
            end do
         end if
         if (i_lnTdot == 0) then
            do k=nzlo,nzhi
               s% lnTdot(k) = 0d0
            end do
         end if
         if (i_lnddot == 0) then
            do k=nzlo,nzhi
               s% lnddot(k) = 0d0
            end do
         end if
         
         do k=nzlo,nzhi
            s% m(k) = s% M_center + s% q(k)*s% xmstar
            s% dm(k) = s% dq(k)*s% xmstar
         end do

         if (.not. skip_mixing_info) then
            s% mixing_type(nzlo:nzhi) = no_mixing
            s% adjust_mlt_gradT_fraction(nzlo:nzhi) = -1 ! can be set by other_mixing
         end if

         call update_vars( &
            s, nzlo, nzhi, skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, &
            skip_other_cgrav, skip_mixing_info, dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr .or. dbg) write(*, *) 'set_vars: update_hydro_vars returned ierr', ierr
            return
         end if
         
         s% d_extra_heat_dlnd(nzlo:nzhi) = 0
         s% d_extra_heat_dlnT(nzlo:nzhi) = 0
         if (s% inject_uniform_extra_heat /= 0) then 
            do k=1,s% nz 
               s% extra_heat(k) = s% inject_uniform_extra_heat
            end do
         else
            s% extra_heat(nzlo:nzhi) = 0
            call s% other_energy(s% id, ierr)
            if (ierr /= 0) then
               if (s% report_ierr .or. dbg) write(*, *) 'set_vars: other_energy returned ierr', ierr
               return
            end if
         end if
         
         call s% other_torque(s% id, ierr)
         if (ierr /= 0) then
            if (s% report_ierr .or. dbg) write(*, *) 'set_vars: other_torque returned ierr', ierr
            return
         end if
         
         if (s% irradiation_flux /= 0) then
            do k=nzlo,nzhi
               s% irradiation_heat(k) = eval_irradiation_heat(s,k)
            end do
         else
            s% irradiation_heat(nzlo:nzhi) = 0
         end if

      end subroutine set_some_vars
      
      
      subroutine update_vars(s, nzlo, nzhi, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_other_cgrav, skip_mixing_info, dt, ierr)
         use mix_info, only: set_mixing_info
         use star_utils, only: get_tau
         use hydro_gr_factors, only: set_gr_factors
         type (star_info), pointer :: s 
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: skip_net, skip_neu, skip_kap, skip_grads, &
            skip_rotation, skip_other_cgrav, skip_mixing_info
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: i_xlnd, i_lnPgas, i_lnE, i_lnT, i_lnR, i_FL, i_vel, &
            k, species, nvar_chem
         real(dp) :: dt_inv
         logical, parameter :: skip_basic_vars = .false.
         
         include 'formats.dek'
         
         ierr = 0

         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         i_lnE = s% i_lnE
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         i_FL = s% i_FL
         i_vel = s% i_vel
         
         species = s% species
         nvar_chem = s% nvar_chem
         
         ! lagrangian time derivatives
         if (s% generations < 2 .or. dt <= 0 .or. s% nz /= s% nz_old) then
            s% rstar_old = exp(s% lnR(1))
            if (i_xlnd /= 0) s% dlnd_dt(nzlo:nzhi) = 0
            if (i_lnPgas /= 0) s% dlnPgas_dt(nzlo:nzhi) = 0
            if (i_lnE /= 0) s% dlnE_var_dt(nzlo:nzhi) = 0
            s% dlnT_dt(nzlo:nzhi) = 0
            s% dlnR_dt(nzlo:nzhi) = 0
            if (s% v_flag) s% dv_dt(nzlo:nzhi) = 0
            s% dVARdot_dVAR = 0
         else
            s% rstar_old = exp(s% xh_old(i_lnR,1))
            dt_inv = 1/dt
            if (i_xlnd /= 0) then
               do k=nzlo,nzhi 
                  s% dlnd_dt(k) = &
                     ((s% xh(i_xlnd,k) - lnd_offset) - s% lnd_for_d_dt(k))*dt_inv
               end do
            end if
            if (i_lnPgas /= 0) then
               do k=nzlo,nzhi 
                  s% dlnPgas_dt(k) = (s% xh(i_lnPgas,k) - s% lnPgas_for_d_dt(k))*dt_inv
               end do
            end if
            if (i_lnE /= 0) then
               do k=nzlo,nzhi 
                  s% dlnE_var_dt(k) = (s% xh(i_lnE,k) - s% lnE_var_for_d_dt(k))*dt_inv
               end do
            end if
            do k=nzlo,nzhi 
               s% dlnT_dt(k) = (s% xh(i_lnT,k) - s% lnT_for_d_dt(k))*dt_inv
            end do
            do k=nzlo,nzhi 
               s% dlnR_dt(k) = (s% xh(i_lnR,k) - s% lnR_for_d_dt(k))*dt_inv
            end do
            if (s% v_flag) then
               do k=nzlo,nzhi 
                  s% dv_dt(k) = (s% xh(i_vel,k) - s% v_for_d_dt(k))*dt_inv
               end do
            end if
            s% dVARdot_dVAR = dt_inv
            if (nzlo < s% k_below_recently_added) & ! special treatment for newly accreted material
               call do_d_dq_terms_for_d_dt(s, nzlo, s% k_below_recently_added-1)
         end if
                  
         call set_hydro_vars( &
            s, nzlo, nzhi, skip_basic_vars, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_other_cgrav, skip_mixing_info, dt, ierr)
         if (ierr /= 0) return
         
         if (nzlo == 1 .and. nzhi == s% nz .and. .not. skip_mixing_info) then
         
            call set_mixing_info(s, ierr)
            if (ierr /= 0) return
            
            if (s% Teff <= 0) then
               call set_Teff(s, ierr) 
               if (ierr /= 0) return
            end if
            
            call get_tau(s, s% tau)

            if (s% use_gr_factors) then
               call set_gr_factors(s, ierr)
               if (ierr /= 0) return
            end if
            
         end if
         
      end subroutine update_vars
      
      
      subroutine set_Teff(s, ierr)
         use atm_lib, only: atm_option
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         real(dp) :: r_surf, L_surf
         logical, parameter :: skip_partials = .true.
         real(dp) :: Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         ierr = 0         
         call set_Teff_info_for_eqns(s, skip_partials, r_surf, L_surf, Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
      end subroutine set_Teff
      
      
      subroutine set_Teff_info_for_eqns(s, skip_partials, r_surf, L_surf, Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         use atm_lib, only: atm_option
         type (star_info), pointer :: s
         logical, intent(in) :: skip_partials
         real(dp), intent(out) :: r_surf, L_surf, Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         integer, intent(out) :: ierr

         integer :: which_atm_option, off_table_option
         
         r_surf = s% r(1)
         L_surf = s% L(1)
         
         which_atm_option = atm_option(s% which_atm_option, ierr)
         if (ierr /= 0) then
            write(*,*) 'unknown value for which_atm_option ' // trim(s% which_atm_option)
            return
         end if
      
         off_table_option = atm_option(s% which_atm_off_table_option, ierr)  
         if (ierr /= 0) then
            write(*,*) 'unknown value for which_atm_off_table_option ' // trim(s% which_atm_off_table_option)
            return
         end if
         
         call get_surf_PT( &
            s, which_atm_option, off_table_option, r_surf, L_surf, skip_partials, &
            Teff, lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         if (ierr /= 0) then
            return
         end if

         s% Teff = Teff
         s% L_phot = L_surf/Lsun


      end subroutine set_Teff_info_for_eqns
      
      
      subroutine do_d_dq_terms_for_d_dt(s, nzlo, knew) ! only for newly accreted material
         ! assumes have already set the constant q terms
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, knew ! cells from 1 to knew contain new material
         integer :: i_xlnd, i_lnPgas, i_lnE, i_lnT, i_lnR, i_vel, k
         real(dp) :: mdot_div_xmstar, q, dlnR_dq, &
            domega_dq, dvel_dq, dlnd_dq, dlnPgas_dq, dlnT_dq, dlnE_dq, dqsum
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         i_lnE = s% i_lnE
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         i_vel = s% i_vel
         mdot_div_xmstar = s% mstar_dot / s% xmstar
         do k = max(nzlo,2), knew
            ! do values defined at cell boundaries (lnR and vel)
            q = s% q(k)
            dqsum = s% dq(k-1) + s% dq(k)
            dlnR_dq = (s% xh(i_lnR,k-1) - s% xh(i_lnR,k+1))/dqsum
            s% dlnR_dt(k) = s% dlnR_dt(k) - dlnR_dq*q*mdot_div_xmstar
            if (s% v_flag) then
               dvel_dq = (s% xh(i_vel,k-1) - s% xh(i_vel,k+1))/dqsum
               s% dv_dt(k) = s% dv_dt(k) - dvel_dq*q*mdot_div_xmstar
            end if
            ! do values defined at cell centers (lnd and lnT)
            q = q - s% dq(k)
            dqsum = s% dq(k) + 0.5d0*(s% dq(k-1) + s% dq(k+1))
            if (i_xlnd /= 0) then
               dlnd_dq = (s% xh(i_xlnd,k-1) - s% xh(i_xlnd,k+1))/dqsum
               s% dlnd_dt(k) = s% dlnd_dt(k) - dlnd_dq*q*mdot_div_xmstar
            end if
            if (i_lnPgas /= 0) then
               dlnPgas_dq = (s% xh(i_lnPgas,k-1) - s% xh(i_lnPgas,k+1))/dqsum
               s% dlnPgas_dt(k) = s% dlnPgas_dt(k) - dlnPgas_dq*q*mdot_div_xmstar
            end if
            if (i_lnE /= 0) then
               dlnE_dq = (s% xh(i_lnE,k-1) - s% xh(i_lnE,k+1))/dqsum
               s% dlnE_var_dt(k) = s% dlnE_var_dt(k) - dlnE_dq*q*mdot_div_xmstar
            end if
            dlnT_dq = (s% xh(i_lnT,k-1) - s% xh(i_lnT,k+1))/dqsum
            s% dlnT_dt(k) = s% dlnT_dt(k) - dlnT_dq*q*mdot_div_xmstar
         end do
      end subroutine do_d_dq_terms_for_d_dt
      
      
      subroutine set_hydro_vars( &
            s, nzlo, nzhi, skip_basic_vars, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_other_cgrav, skip_mixing_info, dt, ierr)
         use micro, only: set_micro_vars
         use mlt_info, only: set_mlt_vars, set_grads, set_mlt_vc_max
         use star_utils, only: update_time, total_times
         use hydro_rotation, only: set_rotation_info

         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: skip_basic_vars, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_other_cgrav, skip_mixing_info
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: nz
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         
         include 'formats.dek'

         if (dbg) write(*, *) 'set_hydro_vars', nzlo, nzhi

         ierr = 0
         nz = s% nz
         
         if (.not. skip_basic_vars) then
            if (dbg) write(*, *) 'call set_basic_vars'
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call set_basic_vars(s, nzlo, nzhi, skip_other_cgrav, ierr)
            if (s% doing_timing) &
               call update_time(s, time0, total_all_before, s% time_set_basic_vars)
            if (failed('set_basic_vars')) return
         end if
         
         if (dbg) write(*, *) 'call set_micro_vars'
         call set_micro_vars(s, nzlo, nzhi, skip_net, skip_neu, skip_kap, ierr)
         if (failed('set_micro_vars')) return
         
         if (s% rotation_flag .and. .not. skip_rotation) then
            if (dbg) write(*, *) 'call set_rotation_info'
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call set_rotation_info(s, ierr)
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_set_rotation_vars)
            if (failed('set_rotation_info')) return
         end if
         
         if (s% v_flag .and. s% use_artificial_viscosity) then
            if (dbg) write(*, *) 'call set_Pvisc'
            call set_Pvisc(s, nzlo, nzhi, ierr)
            if (failed('set_Pvisc')) return
         end if
         
         if (.not. skip_grads) then
            if (dbg) write(*, *) 'call set_grads'
            call set_grads(s, ierr)
            if (failed('set_grads')) return
         end if

         if (dbg) write(*, *) 'call set_mlt_vars'
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         
         if (s% use_time_dependent_convection .and. .not. skip_mixing_info) then
            if (dbg) write(*,*) 'call set_mlt_vc_max'
            call set_mlt_vc_max(s, ierr)
            if (failed('set_mlt_vc_max')) return
         end if

         call set_mlt_vars(s, nzlo, nzhi, ierr)
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_set_mlt_vars)
         if (failed('set_mlt_vars')) return
         
         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            if (ierr == 0) then
               failed = .false.
               return
            end if
            if (s% report_ierr .or. dbg) write(*,*) 'set_hydro_vars failed in call to ' // trim(str)
            failed = .true.
         end function failed

      end subroutine set_hydro_vars
      
      
      subroutine check_rs(s, ierr)
         type (star_info), pointer :: s         
         integer, intent(out) :: ierr
         integer :: k
         logical :: okay
         include 'formats.dek'
         ierr = 0
         okay = .true.
         do k=2, s% nz
            if (s% r(k) > s% r(k-1)) then
               if (s% report_ierr) then
                  write(*,2) 's% r(k) > s% r(k-1)', k, &
                     s% r(k)/Rsun, s% r(k-1)/Rsun, s% r(k)/Rsun-s% r(k-1)/Rsun
               end if
               okay = .false.
            end if
         end do
         if (okay) return
         ierr = -1
         if (s% report_ierr) write(*,*)
      end subroutine check_rs
      
      
      subroutine set_basic_vars(s, nzlo, nzhi, skip_other_cgrav, ierr)
         use star_utils, only: total_angular_momentum
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         logical, intent(in) :: skip_other_cgrav
         integer, intent(out) :: ierr
         integer :: j, k, species
         real(dp) :: twoGmrc2, r2, alfa, beta, sum_xa
         
         include 'formats.dek'
         
         if (dbg) write(*, *) 'set_basic_vars'
         ierr = 0
         species = s% species
         
!$OMP PARALLEL DO PRIVATE(j,k,twoGmrc2,r2,sum_xa)
         do k=nzlo, nzhi
            if (s% lnPgas_flag) then
               s% Pgas(k) = exp(s% lnPgas(k))
            else
               s% rho(k) = exp(s% lnd(k))
            end if
            s% T(k) = exp(s% lnT(k))
            s% r(k) = exp(s% lnR(k))
            r2 = s% r(k)**2
            s% area(k) = pi4*r2
            if (skip_other_cgrav) then
               s% grav(k) = s% cgrav(k)*s% m(k)/r2
               if (s% use_gr_factors) then ! GR gravity factor = 1/sqrt(1-2Gm/(rc^2))
                  twoGmrc2 = 2*s% cgrav(k)*s% m(k)/(s% r(k)*clight**2)
                  s% grav(k) = s% grav(k)/sqrt(1d0 - twoGmrc2)
               end if
            end if
            if (s% rotation_flag) then ! note: do NOT change j_rot here
               s% i_rot(k) = (2d0/3d0)*r2 ! thin spherical shell
            end if
            do j=1,species 
               s% xa(j,k) = max(0d0, min(1d0, s% xa(j,k)))
            end do
            sum_xa = sum(s% xa(1:species,k))
            if (abs(sum_xa - 1d0) > 1d-12) then
               do j=1,species
                  s% xa(j,k) = s% xa(j,k)/sum_xa
               end do
            end if
         end do    
!$OMP END PARALLEL DO

         s% total_angular_momentum = total_angular_momentum(s)
         !write(*,2) 'set_basic_vars tot J', s% model_number, s% total_angular_momentum

         call set_rmid_and_Amid(s, nzlo, nzhi, ierr)
         
         if (.not. skip_other_cgrav) then
            call s% other_cgrav(s% id, ierr)
            if (ierr /= 0) then
               if (s% report_ierr .or. dbg) write(*, *) 'other_cgrav returned ierr', ierr
               return
            end if
!$OMP PARALLEL DO PRIVATE(k,twoGmrc2,r2,alfa,beta)
            do k=nzlo, nzhi
               if (k == 1) then
                  alfa = 1
                  s% rho_face(k) = s% rho(k)
               else
                  alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
                  beta = 1 - alfa
                  s% rho_face(k) = alfa*s% rho(k) + beta*s% rho(k-1)
               end if
               r2 = s% r(k)**2
               s% grav(k) = s% cgrav(k)*s% m(k)/r2
               if (s% use_gr_factors) then ! GR gravity factor = 1/sqrt(1-2Gm/(rc^2))
                  twoGmrc2 = 2*s% cgrav(k)*s% m(k)/(s% r(k)*clight**2)
                  s% grav(k) = s% grav(k)/sqrt(1d0 - twoGmrc2)
               end if
            end do    
!$OMP END PARALLEL DO
         end if
         
      end subroutine set_basic_vars

      
      subroutine set_rmid_and_Amid(s, nzlo, nzhi, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         
         integer :: k, nz
         
         include 'formats.dek'
         
         ierr = 0
         nz = s% nz
!$OMP PARALLEL DO PRIVATE(k)
         do k=nzlo, nzhi
            call do_Amid(k)
         end do
!$OMP END PARALLEL DO
         
         
         contains
         
         
         subroutine do_Amid(k) ! set Amid, dAmid_dlnR00, and dAmid_dlnRp1
            integer, intent(in) :: k
            real(dp) :: r003, rp13, rmid, rmid2, drmid2_dlnR00, drmid2_dlnRp1
            r003 = s% r(k)**3
            if (k < nz) then
               rp13 = s% r(k+1)**3
            else
               rp13 = s% R_center**3
            end if
            rmid = ((r003 + rp13)/2)**(1d0/3d0)
            s% rmid(k) = rmid
            rmid2 = rmid**2
            drmid2_dlnR00 = r003/rmid
            drmid2_dlnRp1 = rp13/rmid
            s% Amid(k) = 4*pi*rmid2
            s% dAmid_dlnR00(k) = 4*pi*drmid2_dlnR00
            s% dAmid_dlnRp1(k) = 4*pi*drmid2_dlnRp1
         end subroutine do_Amid
         

      end subroutine set_rmid_and_Amid
      
      
      subroutine set_Pvisc(s, nzlo, nzhi, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         
         integer :: k, nz
         real(dp) :: l1, l22
         
         include 'formats.dek'
         
         ierr = 0
         l1 = s% l1_coef
         l22 = s% l2_coef**2
         nz = s% nz
!$OMP PARALLEL DO PRIVATE(k)
         do k=nzlo, nzhi
            call do_Pvisc(k)
         end do
!$OMP END PARALLEL DO
         
         
         contains
         
         
         subroutine do_Pvisc(k) ! set Pvisc and its partials
            use eos_def, only: i_gamma1
            integer, intent(in) :: k
            real(dp) :: dA_dlnR, &
               cs, d_cs_dlnd, d_cs_dlnT, &
               dv, d_dv_dv00, d_dv_dvp1, &
               etav, d_etav_dlnd, d_etav_dlnT, &
               d_etav_dv00, d_etav_dvp1, &
               c00, cp1, Amid2, d_c00_dlnR00, d_c00_dlnRp1, d_cp1_dlnR00, d_cp1_dlnRp1, &
               del, d_del_dlnR00, d_del_dlnRp1, d_del_dv00, d_del_dvp1

            if (l1 == 0) then
               cs = 0; d_cs_dlnd = 0; d_cs_dlnT = 0
            else
               cs = sqrt(s% gamma1(k)*s% P(k)/s% rho(k))
               d_cs_dlnd = (s% d_eos_dlnd(i_gamma1,k)/s% gamma1(k) + s% chiRho(k) - 1)*cs/2
               d_cs_dlnT = (s% d_eos_dlnT(i_gamma1,k)/s% gamma1(k) + s% chiT(k))*cs/2
            end if
            
            if (k == nz) then
               dv = 0; d_dv_dv00 = 0; d_dv_dvp1 = 0
            else if (s% v(k+1) <= s% v(k)) then
               dv = 0; d_dv_dv00 = 0; d_dv_dvp1 = 0
            else
               dv = s% v(k+1) - s% v(k)
               d_dv_dv00 = -1
               d_dv_dvp1 = 1
            end if
            
            etav = 0.75d0*s% rho(k)*(l1*cs + l22*dv)
            if (etav == 0) then
               s% Pvisc(k) = 0
               s% dPvisc_dlnd(k) = 0
               s% dPvisc_dlnT(k) = 0
               s% dPvisc_dlnR00(k) = 0
               s% dPvisc_dvel00(k) = 0
               s% dPvisc_dlnRp1(k) = 0
               s% dPvisc_dvelp1(k) = 0
               return
            end if
            d_etav_dlnd = 0.75d0*s% rho(k)*(l1*(cs + d_cs_dlnd) + l22*dv)
            d_etav_dlnT = 0.75d0*s% rho(k)*l1*d_cs_dlnT
            d_etav_dv00 = 0.75d0*s% rho(k)*l22*d_dv_dv00
            d_etav_dvp1 = 0.75d0*s% rho(k)*l22*d_dv_dvp1
            
            c00 = (1 - s% area(k)/(3*s% Amid(k)))
            Amid2 = s% Amid(k)**2
            dA_dlnR = 2*s% area(k)
            d_c00_dlnR00 = (s% area(k)*s% dAmid_dlnR00(k) - s% Amid(k)*dA_dlnR)/(3*Amid2)
            d_c00_dlnRp1 = s% area(k)*s% dAmid_dlnRp1(k)/(3*Amid2)
            if (k < nz) then
               cp1 = (1 - s% area(k+1)/(3*s% Amid(k)))
               d_cp1_dlnR00 = s% area(k+1)*s% dAmid_dlnR00(k)/(3*Amid2)
               dA_dlnR = 2*s% area(k+1)
               d_cp1_dlnRp1 = (s% area(k+1)*s% dAmid_dlnRp1(k) - s% Amid(k)*dA_dlnR)/(3*Amid2)
               del = s% v(k)*c00 - s% v(k+1)*cp1
               d_del_dlnR00 = s% v(k)*d_c00_dlnR00 - s% v(k+1)*d_cp1_dlnR00
               d_del_dlnRp1 = s% v(k)*d_c00_dlnRp1 - s% v(k+1)*d_cp1_dlnRp1
               d_del_dv00 = c00
               d_del_dvp1 = -cp1
            else
               del = s% v(k)*c00
               d_del_dlnR00 = s% v(k)*d_c00_dlnR00
               d_del_dv00 = c00
               d_del_dvp1 = 0
               d_del_dlnRp1 = 0
            end if
            
            s% Pvisc(k) = -etav*del
            s% dPvisc_dlnd(k) = -d_etav_dlnd*del
            s% dPvisc_dlnT(k) = -d_etav_dlnT*del
            s% dPvisc_dlnR00(k) = -etav*d_del_dlnR00
            s% dPvisc_dvel00(k) = -(d_etav_dv00*del + etav*d_del_dv00)
            if (k < nz) then
               s% dPvisc_dlnRp1(k) = -etav*d_del_dlnRp1
               s% dPvisc_dvelp1(k) = -(d_etav_dvp1*del + etav*d_del_dvp1)
            end if
         
         end subroutine do_Pvisc
         
         
      end subroutine set_Pvisc
      
      
      subroutine eval_eps_grav_and_partials(s, k, dt, ierr)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         real(dp) :: f
         include 'formats.dek'
         ierr = 0
         
         s% eps_grav(k) = 0
         s% d_eps_grav_dlndm1(k) = 0
         s% d_eps_grav_dlnd00(k) = 0
         s% d_eps_grav_dlndp1(k) = 0
         s% d_eps_grav_dlnTm1(k) = 0
         s% d_eps_grav_dlnT00(k) = 0
         s% d_eps_grav_dlnTp1(k) = 0
         s% d_eps_grav_dlnR00(k) = 0
         s% d_eps_grav_dL00(k) = 0
         s% d_eps_grav_dlnPgasm1_const_T(k) = 0
         s% d_eps_grav_dlnPgas00_const_T(k) = 0
         s% d_eps_grav_dlnPgasp1_const_T(k) = 0
         s% d_eps_grav_dlnTm1_const_Pgas(k) = 0
         s% d_eps_grav_dlnT00_const_Pgas(k) = 0
         s% d_eps_grav_dlnTp1_const_Pgas(k) = 0
         s% d_eps_grav_dlnEm1(k) = 0
         s% d_eps_grav_dlnE00(k) = 0
         s% d_eps_grav_dlnEp1(k) = 0
         s% d_eps_grav_dlnRp1(k) = 0
         s% d_eps_grav_dv00(k) = 0
         s% d_eps_grav_dvp1(k) = 0
         s% d_eps_grav_dlnTdot(k) = 0
         s% d_eps_grav_dlnddot(k) = 0

         call eval1_eps_grav_and_partials(s, k, dt, ierr)
         if (ierr /= 0) return
         
         f = s% eps_grav_factor
         if (f == 1) return
         
         s% eps_grav(k) = f*s% eps_grav(k)
         s% d_eps_grav_dlndm1(k) = f*s% d_eps_grav_dlndm1(k)
         s% d_eps_grav_dlnd00(k) = f*s% d_eps_grav_dlnd00(k)
         s% d_eps_grav_dlndp1(k) = f*s% d_eps_grav_dlndp1(k)
         s% d_eps_grav_dlnTm1(k) = f*s% d_eps_grav_dlnTm1(k)
         s% d_eps_grav_dlnT00(k) = f*s% d_eps_grav_dlnT00(k)
         s% d_eps_grav_dlnTp1(k) = f*s% d_eps_grav_dlnTp1(k)
         s% d_eps_grav_dlnR00(k) = f*s% d_eps_grav_dlnR00(k)
         s% d_eps_grav_dL00(k) = f*s% d_eps_grav_dL00(k)
         s% d_eps_grav_dlnPgas00_const_T(k) = f*s% d_eps_grav_dlnPgas00_const_T(k)
         s% d_eps_grav_dlnPgasm1_const_T(k) = f*s% d_eps_grav_dlnPgasm1_const_T(k)
         s% d_eps_grav_dlnPgasp1_const_T(k) = f*s% d_eps_grav_dlnPgasp1_const_T(k)
         s% d_eps_grav_dlnTm1_const_Pgas(k) = f*s% d_eps_grav_dlnTm1_const_Pgas(k)
         s% d_eps_grav_dlnT00_const_Pgas(k) = f*s% d_eps_grav_dlnT00_const_Pgas(k)
         s% d_eps_grav_dlnTp1_const_Pgas(k) = f*s% d_eps_grav_dlnTp1_const_Pgas(k)
         s% d_eps_grav_dlnEm1(k) = f*s% d_eps_grav_dlnEm1(k)
         s% d_eps_grav_dlnE00(k) = f*s% d_eps_grav_dlnE00(k)
         s% d_eps_grav_dlnEp1(k) = f*s% d_eps_grav_dlnEp1(k)
         s% d_eps_grav_dlnRp1(k) = f*s% d_eps_grav_dlnRp1(k)
         s% d_eps_grav_dv00(k) = f*s% d_eps_grav_dv00(k)
         s% d_eps_grav_dvp1(k) = f*s% d_eps_grav_dvp1(k)
         s% d_eps_grav_dlnTdot(k) = f*s% d_eps_grav_dlnTdot(k)
         s% d_eps_grav_dlnddot(k) = f*s% d_eps_grav_dlnddot(k)
         
         
      end subroutine eval_eps_grav_and_partials
      
   
      subroutine eval1_eps_grav_and_partials(s, k, dt, ierr)
         use eos_def
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         real(dp) :: &
            dlnd_dt, dlnPgas_dt, dlnT_dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1, &
            a1, da1_dlnd, da1_dlnT, &
            T1, dT1_dlnd, dT1_dlnTm1, dT1_dlnT00, dT1_dlnTp1, dT1_dlnd_dt, dT1_d_dlnTdt, &
            a2, da2_dlnd, da2_dlnT, &
            T2, dT2_dlnT, dT2_dlndm1, dT2_dlnd00, dT2_dlndp1, &
            T3, dT3_dlnd, dT3_dlnT

         include 'formats.dek'
         
         ierr = 0
         
         s% profile_extra(k,1) = 0
         s% profile_extra(k,2) = 0
         s% profile_extra(k,3) = 0

         if (s% k_below_recently_added > 1) then
            if (k < s% k_below_recently_added .and. k < s% k_below_surface_radiative_zone .and. &
                  s% abar(s% k_below_recently_added-1) < 5d0) then
                  
               if (s% v_flag .and. s% lnE_flag) then
                  write(*,*) 'need to fix do_thin_radiative for v + lnE eps_grav'
                  stop 1
               end if
               
               call do_thin_radiative
               return
            end if
         end if
      
         if (.false. .and. s% v_flag .and. s% lnE_flag) then
            call do_eps_grav_using_E_var_and_v
         else if (s% use_lnE_for_eps_grav) then
            if (s% lnE_flag) then
               call do_eps_grav_using_E_var
            else
               call do_eps_grav_using_E
            end if
         else if (s% lnPgas_flag) then
            call do_eps_grav_with_lnPgas
         else
            call do_eps_grav_with_lnd
         end if
         
         if (is_bad_num(s% eps_grav(k))) then
            ierr = -1
            if (s% report_ierr) write(*,*) 'eval_eps_grav_and_partials -- bad value for eps_grav'
            return
         end if
         
                  
         contains


         subroutine do_eps_grav_using_E
            ! eps_grav = -(dE/dt + P dV/dt), V = 1/rho
            ! eps_grav = -(E*dlnE/dt + P dV/dt)
            ! eps_grav depends on dlnE_dt(k), P(k), rho(k), dlnd_dt(k)
            
            real(dp) :: E, E_prev, dEdt, P, rho, rho_prev, PdVdt, d_dEdt_dlnT, d_dEdt_dlnd, &
               d_PdVdt_dlnT, d_PdVdt_dlnd, conv_fac
            
            include 'formats.dek'
            
            if (dt <= 0) then
               s% eps_grav(k) = 0
               return
            end if
         
            E = exp(s% lnE(k))
            E_prev = exp(s% lnE_pre_hydro(k))
            dEdt = (E - E_prev)/dt

            if (s% conv_dP_term(k) /= 0 .and.  s% conv_dP_term_factor > 0) then 
               ! include effect of convective turbulence on pressure
               conv_fac = s% conv_dP_term_factor*s% conv_dP_term(k)
            else
               conv_fac = 0
            end if
            P = s% P(k)
            rho = s% rho(k)
            rho_prev = exp(s% lnd_pre_hydro(k))           
            PdVdt = P*(1 + conv_fac)*(1d0/rho - 1d0/rho_prev)/dt
            
            s% eps_grav(k) = -(dEdt + PdVdt)

            if (.false. .and. s% model_number == 121 .and. k==s% nz) then
               write(*,*) 'do_eps_grav_using_E'
               write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               write(*,2) 'dEdt', k, dEdt
               write(*,2) 'PdVdt', k, PdVdt
               write(*,2) 'dt', k, dt
               write(*,2) 'rho_prev', k, rho_prev
               write(*,2) 'rho', k, rho
               write(*,2) 'P', k, P
               write(*,2) 'E_prev', k, E_prev
               write(*,2) 'E', k, E
               !stop 'do_eps_grav_using_E'
            end if
                        
            if (.true.) then
               d_dEdt_dlnT = s% T(k)*s% Cv(k)/dt
               d_dEdt_dlnd = rho*s% dE_dRho(k)/dt
            else
               d_dEdt_dlnT = E*get_dlnE_dlnT(s,k)/dt
               d_dEdt_dlnd = E*get_dlnE_dlnd(s,k)/dt
            end if

            d_PdVdt_dlnT = PdVdt*s% chiT(k)
            d_PdVdt_dlnd = PdVdt*s% chiRho(k) - P*(1 + conv_fac)/rho/dt
            
            s% d_eps_grav_dlnT00(k) = -(d_dEdt_dlnT + d_PdVdt_dlnT)
            s% d_eps_grav_dlnd00(k) = -(d_dEdt_dlnd + d_PdVdt_dlnd)
            
            if (s% lnPgas_flag) then
               s% d_eps_grav_dlnPgas00_const_T(k) = &
                   s% d_eps_grav_dlnd00(k)*s% dlnRho_dlnPgas_const_T(k)
               s% d_eps_grav_dlnT00_const_Pgas(k) = &
                  s% d_eps_grav_dlnT00(k) + &
                  s% d_eps_grav_dlnd00(k)*s% dlnRho_dlnT_const_Pgas(k)
            end if
            
            !if (s% hydro_call_number == 305) write(*,4) 'do_eps_grav_using_E', &
            !   k, s% model_number, s% hydro_call_number, &
            !   s% eps_grav(k), s% d_eps_grav_dlnT00(k), s% d_eps_grav_dlnd00(k)
         
         end subroutine do_eps_grav_using_E


         subroutine do_eps_grav_with_lnS
            ! eps_grav = -T*ds/dt
            
            real(dp) :: lnS, entropy, lnS_prev, S_prev, T, dS_dlnT, dS_dlnd           
            
            include 'formats.dek'
            
            if (dt == 0) then
               s% eps_grav(k) = 0
               return
            end if
                        
            lnS = s% lnS(k)
            entropy = exp(lnS)
            lnS_prev = s% lnS_pre_hydro(k)
            S_prev = exp(lnS_prev)
            T = s% T(k)
            
            s% eps_grav(k) = -T*(entropy - S_prev)/dt
            
            dS_dlnT = entropy*s% d_eos_dlnT(i_lnS,k)
            dS_dlnd = entropy*s% d_eos_dlnd(i_lnS,k)
            
            s% d_eps_grav_dlnT00(k) = -T*dS_dlnT/dt + s% eps_grav(k)
            s% d_eps_grav_dlnd00(k) = -T*dS_dlnd/dt
         
         end subroutine do_eps_grav_with_lnS


         subroutine do_eps_grav_with_lnPgas
         
            !s% eps_grav(k) = -s% T(k)*s% cp(k)*(dlnT_dt - s% grada(k)*dlnP_dt)
            ! dlnP_dt = (dPgas_dt + dPrad_dt)/P
            !         = (Pgas*dlnPgas_dt + 4*Prad*dlnT_dt)/P
            !s% eps_grav(k) = -s% T(k)*s% cp(k)* &
            !     ((1 - s% grada(k)*4*Prad/P)*dlnT_dt - s% grada(k)*Pgas/P*dlnPgas_dt)
            
            real(dp) :: dlnd_dlnPgas00, dlnd_dlnPgasm1, dlnd_dlnPgasp1, dgrada_dlnPgas, &
               dT1_dlnPgas, dT2_dlnPgasm1, dT2_dlnPgas00, dT2_dlnPgasp1, dT3_dlnPgas, &
               d_eps_grav_dlnPgasm1, d_eps_grav_dlnPgas00, d_eps_grav_dlnPgasp1, &
               dP_dlnPgas, dPinv_dlnPgas, dPrad_dlnT, &
               da1_dlnPgas, da2_dlnPgas, dP_dlnT_const_Pgas, dPinv_dlnT_const_Pgas, &
               da1_dlnPgas_const_T, da1_dlnT_const_Pgas, da2_dlnPgas_const_T, &
               da2_dlnT_const_Pgas, dcp_dlnPgas_const_T, dcp_dlnT_const_Pgas, &
               dgrada_dlnPgas_const_T, dgrada_dlnT_const_Pgas, dPinv_dlnPgas_const_T, &
               dPgas_dlnPgas_const_T, dPrad_dlnT_const_Pgas, dP_dlnPgas_const_T, &
               dT1_dlnPgas_const_T, dT1_dlnT00_const_Pgas, dT1_dlnTm1_const_Pgas, &
               dT1_dlnTp1_const_Pgas, dT2_dlnPgas00_const_T, dT2_dlnPgasm1_const_T, &
               dT2_dlnPgasp1_const_T, dT2_dlnT_const_Pgas, dT3_dlnPgas_const_T, &
               dT3_dlnT_const_Pgas

            dlnPgas_dt = s% dlnPgas_dt(k)     
            dlnT_dt = s% dlnT_dt(k)         
            call get_dVARDOT_dVAR_cell(s, k, dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1)
            
            dgrada_dlnPgas_const_T = s% d_eos_dlnd(i_grad_ad,k)*s% dlnRho_dlnPgas_const_T(k)
            dgrada_dlnT_const_Pgas = &
               s% d_eos_dlnT(i_grad_ad,k) + s% d_eos_dlnd(i_grad_ad,k)*s% dlnRho_dlnT_const_Pgas(k)
            
            dgrada_dlnPgas_const_T = s% d_eos_dlnd(i_grad_ad,k)*s% dlnRho_dlnPgas_const_T(k)
            dgrada_dlnT_const_Pgas = &
               s% d_eos_dlnT(i_grad_ad,k) + s% d_eos_dlnd(i_grad_ad,k)*s% dlnRho_dlnT_const_Pgas(k)

            dcp_dlnPgas_const_T = s% d_eos_dlnd(i_cp,k)*s% dlnRho_dlnPgas_const_T(k)
            dcp_dlnT_const_Pgas = &
               s% d_eos_dlnT(i_cp,k) + s% d_eos_dlnd(i_cp,k)*s% dlnRho_dlnT_const_Pgas(k)
            
            dPgas_dlnPgas_const_T = s% Pgas(k)
            dP_dlnPgas_const_T = dPgas_dlnPgas_const_T
            dPinv_dlnPgas_const_T = -dP_dlnPgas_const_T/s% P(k)**2
            
            dPrad_dlnT_const_Pgas = 4*s% Prad(k)
            dP_dlnT_const_Pgas = dPrad_dlnT_const_Pgas
            dPinv_dlnT_const_Pgas = -dP_dlnT_const_Pgas/s% P(k)**2
            
            a1 = 1 - s% grada(k)*4*s% Prad(k)/s% P(k)
            da1_dlnPgas_const_T = &
               - dgrada_dlnPgas_const_T*4*s% Prad(k)/s% P(k) &
               - s% grada(k)*4*s% Prad(k)*dPinv_dlnPgas_const_T
            da1_dlnT_const_Pgas = &
               - dgrada_dlnT_const_Pgas*4*s% Prad(k)/s% P(k) &
               - s% grada(k)*4*dPrad_dlnT_const_Pgas/s% P(k) &
               - s% grada(k)*4*s% Prad(k)*dPinv_dlnT_const_Pgas
            
            T1 = dlnT_dt*a1
            dT1_dlnPgas_const_T = dlnT_dt*da1_dlnPgas_const_T
            dT1_dlnT00_const_Pgas = d_dVARdt_dVAR00*a1 + dlnT_dt*da1_dlnT_const_Pgas
            dT1_dlnTm1_const_Pgas = d_dVARdt_dVARm1*a1
            dT1_dlnTp1_const_Pgas = d_dVARdt_dVARp1*a1
            
            a2 = s% grada(k)*s% Pgas(k)/s% P(k)
            da2_dlnPgas_const_T = &
               dgrada_dlnPgas_const_T*s% Pgas(k)/s% P(k) + &
               s% grada(k)*dPgas_dlnPgas_const_T/s% P(k) + &
               s% grada(k)*s% Pgas(k)*dPinv_dlnPgas_const_T
            da2_dlnT_const_Pgas = &
               dgrada_dlnT_const_Pgas*s% Pgas(k)/s% P(k) + &
               s% grada(k)*s% Pgas(k)*dPinv_dlnT_const_Pgas
            
            T2 = dlnPgas_dt*a2
            dT2_dlnT_const_Pgas = dlnPgas_dt*da2_dlnT_const_Pgas
            dT2_dlnPgas00_const_T = d_dVARdt_dVAR00*a2 + dlnPgas_dt*da2_dlnPgas_const_T
            dT2_dlnPgasm1_const_T = d_dVARdt_dVARm1*a2
            dT2_dlnPgasp1_const_T = d_dVARdt_dVARp1*a2
            
            T3 = -s% T(k)*s% cp(k)
            dT3_dlnPgas_const_T = -s% T(k)*dcp_dlnPgas_const_T
            dT3_dlnT_const_Pgas =  -(s% T(k)*s% cp(k) + s% T(k)*dcp_dlnT_const_Pgas)

            s% eps_grav(k) = T3*(T1-T2)
         
            if (is_bad_num(s% eps_grav(k))) then
               ierr = -1
               !call alert(ierr, 'eval_eps_grav_and_partials -- bad value for eps_grav')
               if (s% report_ierr) write(*,*) 'eval_eps_grav_and_partials -- bad value for eps_grav'
               return
            end if

            s% d_eps_grav_dlnPgasm1_const_T(k) = -T3*dT2_dlnPgasm1_const_T
            s% d_eps_grav_dlnPgasp1_const_T(k) = -T3*dT2_dlnPgasp1_const_T
            s% d_eps_grav_dlnPgas00_const_T(k) = &
               (T3*(dT1_dlnPgas_const_T - dT2_dlnPgas00_const_T) + dT3_dlnPgas_const_T*(T1-T2))
            
            s% d_eps_grav_dlnTm1_const_Pgas(k) = T3*dT1_dlnTm1_const_Pgas
            s% d_eps_grav_dlnTp1_const_Pgas(k) = T3*dT1_dlnTp1_const_Pgas
            s% d_eps_grav_dlnT00_const_Pgas(k) = &
               (T3*(dT1_dlnT00_const_Pgas - dT2_dlnT_const_Pgas) + dT3_dlnT_const_Pgas*(T1-T2))

         end subroutine do_eps_grav_with_lnPgas


         subroutine do_eps_grav_with_lnd
            integer :: j
            include 'formats.dek'
         
            !s% eps_grav(k) = -s% T(k)*s% cp(k)* &
            !      ((1-s% grada(k)*s% chiT(k))*dlnT_dt &
            !        - s% grada(k)*s% chiRho(k)*dlnd_dt)
            
            real(dp) :: dT1_dlnTdot, dT2_dlnddot
            
            if (s% lnddot_flag) then
               dlnd_dt = s% lnddot(k)
            else
               dlnd_dt = s% dlnd_dt(k)
            end if   
            
            if (s% lnTdot_flag) then
               dlnT_dt = s% lnTdot(k)
            else
               dlnT_dt = s% dlnT_dt(k)
            end if   
            
            call get_dVARDOT_dVAR_cell(s, k, dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1)

            a1 = 1 - s% grada(k)*s% chiT(k)
            da1_dlnd = -(s% d_eos_dlnd(i_grad_ad,k)*s% chiT(k) + s% grada(k)*s% d_eos_dlnd(i_chiT,k))
            da1_dlnT = -(s% d_eos_dlnT(i_grad_ad,k)*s% chiT(k) + s% grada(k)*s% d_eos_dlnT(i_chiT,k))
   
            T1 = dlnT_dt*a1
            dT1_dlnd = dlnT_dt*da1_dlnd
   
            dT1_d_dlnTdt = a1
            if (s% lnTdot_flag) then
               dT1_dlnT00 = dlnT_dt*da1_dlnT
               dT1_dlnTm1 = 0
               dT1_dlnTp1 = 0
               dT1_dlnTdot = a1
            else
               dT1_dlnT00 = d_dVARdt_dVAR00*a1 + dlnT_dt*da1_dlnT
               dT1_dlnTm1 = d_dVARdt_dVARm1*a1
               dT1_dlnTp1 = d_dVARdt_dVARp1*a1
               dT1_dlnTdot = 0
            end if
            
            a2 = s% grada(k)*s% chiRho(k)
            da2_dlnd = s% d_eos_dlnd(i_grad_ad,k)*s% chiRho(k) + s% grada(k)*s% d_eos_dlnd(i_chiRho,k)
            da2_dlnT = s% d_eos_dlnT(i_grad_ad,k)*s% chiRho(k) + s% grada(k)*s% d_eos_dlnT(i_chiRho,k)
   
            T2 = dlnd_dt*a2
            dT2_dlnT = dlnd_dt*da2_dlnT
            
            if (s% lnddot_flag) then
               dT2_dlnd00 = dlnd_dt*da2_dlnd
               dT2_dlndm1 = 0
               dT2_dlndp1 = 0
               dT2_dlnddot = a2
            else
               dT2_dlnd00 = d_dVARdt_dVAR00*a2 + dlnd_dt*da2_dlnd
               dT2_dlndm1 = d_dVARdt_dVARm1*a2
               dT2_dlndp1 = d_dVARdt_dVARp1*a2
               dT2_dlnddot = 0
            end if

            T3 = -s% T(k)*s% cp(k)
            dT3_dlnd = -s% T(k)*s% d_eos_dlnd(i_Cp,k)
            dT3_dlnT = -s% T(k)*(s% cp(k) + s% d_eos_dlnT(i_Cp,k))
   
            ! eps_grav = T3*(T1-T2)
            s% eps_grav(k) = T3*(T1-T2)
            
            if (.false. .and. k == 1469) then
               write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               write(*,2) 's% grada(k)', k, s% grada(k)
               write(*,2) 's% chiT(k)', k, s% chiT(k)
               write(*,2) 's% chiRho(k)', k, s% chiRho(k)
               write(*,2) 's% T(k)', k, s% T(k)
               write(*,2) 's% rho(k)', k, s% rho(k)
               write(*,2) 's% cp(k)', k, s% cp(k)
               write(*,2) 'T3', k, T3
               write(*,2) 'T1', k, T1
               write(*,2) 'T2', k, T2
               write(*,2) 'dlnd_dt', k, dlnd_dt
               write(*,2) 'dlnT_dt', k, dlnT_dt
               do j=1,s% species
                  if (.true. .or. s% xa(j,k) > 1d-9) &
                     write(*,1) 'xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', s% xa(j,k)
               end do
               write(*,*)
               write(*,1) 'T =', s% T(k)
               write(*,1) 'logT =', s% lnT(k)/ln10
               write(*,1) 'rho =', s% rho(k)
               write(*,1) 'logRho =', s% lnd(k)/ln10
               write(*,1) 'abar =', s% abar(k)
               write(*,1) 'zbar =', s% zbar(k)
               write(*,*)
               write(*,*)
            end if
         
            s% d_eps_grav_dlndm1(k) = -T3*dT2_dlndm1
            s% d_eps_grav_dlndp1(k) = -T3*dT2_dlndp1
            s% d_eps_grav_dlnd00(k) = (T3*(dT1_dlnd - dT2_dlnd00) + dT3_dlnd*(T1-T2))
            s% d_eps_grav_dlnddot(k) = -T3*dT2_dlnddot
            
            s% d_eps_grav_dlnTm1(k) = T3*dT1_dlnTm1
            s% d_eps_grav_dlnTp1(k) = T3*dT1_dlnTp1
            s% d_eps_grav_dlnT00(k) = (T3*(dT1_dlnT00 - dT2_dlnT) + dT3_dlnT*(T1-T2))
            s% d_eps_grav_dlnTdot(k) = T3*dT1_dlnTdot

         
         end subroutine do_eps_grav_with_lnd
         
         
         subroutine do_eps_grav_using_E_var_and_v
            ! eps_grav = -(dlnE_var_dt + P d(1/rho)/dt)
            ! continuity: drho/dt = - rho div(v) 
            ! in 1D spherical, div(v) = 1/r^2 d(r^2 v)/dr = 4 pi rho d(r^2 v)/dm
            ! eps_grav = - dlnE_var_dt - P d(4 pi r^2 v)/dm
            ! eps_grav depends on dlnE_var_dt(k), P(k), r(k), r(k+1), v(k), v(k+1)
            
            real(dp) :: dlnE_var_dt, dm, P, rho, v00, r00, rp1, vp1, PdV, &
               d_PdV_dlnR00, d_PdV_dlnRp1, d_PdV_dv00, d_PdV_dvp1, &
               d_PdV_dlnP, d_PdV_dlnT, d_PdV_dlnd, &
               d_PdV_dlnPgas_const_T, d_PdV_dlnT_const_Pgas
               
            
            dlnE_var_dt = s% dlnE_var_dt(k)
            call get_dVARDOT_dVAR_cell(s, k, dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1)
            
            dm = s% dm(k)
            P = s% P(k)
            rho = s% rho(k)
            v00 = s% v(k)
            r00 = s% r(k)
            
            if (k < s% nz) then
               rp1 = s% R_center
               vp1 = 0 ! s% v_center
            else
               vp1 = s% v(k+1)
               rp1 = s% r(k+1)
            end if
            
            PdV = 4*pi*P*(r00**2*v00 - rp1**2*vp1)/dm
            d_PdV_dlnR00 = 2*4*pi*P*r00**2*v00/dm
            d_PdV_dlnRp1 = -2*4*pi*P*rp1**2*vp1/dm
            d_PdV_dv00 = 4*pi*P*r00**2/dm
            d_PdV_dvp1 = -4*pi*P*rp1**2/dm
            d_PdV_dlnP = PdV
            d_PdV_dlnT = d_PdV_dlnP*s% chiT(k)
            d_PdV_dlnd = d_PdV_dlnP*s% chiRho(k)
            
            s% eps_grav(k) = - dlnE_var_dt - PdV
         
            s% d_eps_grav_dlnEm1(k) = -d_dVARdt_dVARm1
            s% d_eps_grav_dlnE00(k) = -d_dVARdt_dVAR00
            s% d_eps_grav_dlnEp1(k) = -d_dVARdt_dVARp1
            s% d_eps_grav_dlnR00(k) = -d_PdV_dlnR00
            s% d_eps_grav_dlnRp1(k) = -d_PdV_dlnRp1
            s% d_eps_grav_dv00(k) = -d_PdV_dv00
            s% d_eps_grav_dvp1(k) = -d_PdV_dvp1
         
            if (s% lnPgas_flag) then

               d_PdV_dlnPgas_const_T = d_PdV_dlnd*s% dlnRho_dlnPgas_const_T(k)
               d_PdV_dlnT_const_Pgas = d_PdV_dlnT + d_PdV_dlnd*s% dlnRho_dlnT_const_Pgas(k)

               s% d_eps_grav_dlnPgas00_const_T(k) = -d_PdV_dlnPgas_const_T
               s% d_eps_grav_dlnT00_const_Pgas(k) = -d_PdV_dlnT_const_Pgas

            else

               s% d_eps_grav_dlnd00(k) = -d_PdV_dlnd
               s% d_eps_grav_dlnT00(k) = -d_PdV_dlnT

            end if
         
         
         end subroutine do_eps_grav_using_E_var_and_v
         
         
         subroutine do_eps_grav_using_E_var
            ! eps_grav = -(dE/dt + P dV/dt), V = 1/rho
            ! eps_grav = -(E*dlnE/dt + P dV/dt)
            ! eps_grav depends on dlnE_var_dt(k), P(k), rho(k), dlnd_dt(k)
            
            real(dp) :: dlnd_dt, dlnE_var_dt, lnE, E, P, rho, PdVdt, &
               d_PdVdt_dlnP, d_PdVdt_dlnT, d_PdVdt_dlnd00, d_PdVdt_dlndp1, d_PdVdt_dlndm1               
            
            include 'formats.dek'
         
            if (s% lnPgas_flag) then
               write(*,*) 'do_eps_grav_using_E_var not for case of lnPgas'
               stop 1
            end if
            
            dlnd_dt = s% dlnd_dt(k)     
            dlnE_var_dt = s% dlnE_var_dt(k)
            call get_dVARDOT_dVAR_cell(s, k, dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1)
            
            lnE = s% lnE_var(k)
            E = exp(lnE)
            P = s% P(k)
            rho = s% rho(k)
            PdVdt = -P/rho**2*dlnd_dt

            d_PdVdt_dlnP = PdVdt
            d_PdVdt_dlnT = d_PdVdt_dlnP*s% chiT(k)
            d_PdVdt_dlnd00 = -P/rho**2*d_dVARdt_dVAR00 + d_PdVdt_dlnP*s% chiRho(k) - 2*PdVdt
            d_PdVdt_dlndp1 = -P/rho**2*d_dVARdt_dVARp1
            d_PdVdt_dlndm1 = -P/rho**2*d_dVARdt_dVARm1
            
            s% eps_grav(k) = -(E*dlnE_var_dt + PdVdt)
         
            s% d_eps_grav_dlnEm1(k) = -E*d_dVARdt_dVARm1
            s% d_eps_grav_dlnE00(k) = -E*(d_dVARdt_dVAR00 + dlnE_var_dt)
            s% d_eps_grav_dlnEp1(k) = -E*d_dVARdt_dVARp1
            
            ! NOTE: E = exp(lnE_var), so is not dependent on rho or T, only on lnE_var.
            s% d_eps_grav_dlnT00(k) = -d_PdVdt_dlnT
            s% d_eps_grav_dlnd00(k) = -d_PdVdt_dlnd00
            s% d_eps_grav_dlndm1(k) = -d_PdVdt_dlndm1
            s% d_eps_grav_dlndp1(k) = -d_PdVdt_dlndp1
            
            if (k == s% nz) write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
         
         end subroutine do_eps_grav_using_E_var
         
         
         subroutine do_thin_radiative
         
            ! use Lars' formula for eps_grav in thin radiative envelope during accretion
            ! Townsley & Bildsten, The Astrophysical Journal, 600:390–403, 2004 January 1
            
            real(dp) :: inv_4pi_r4, inv_4pi_r4_P, dg, G_m_mdot, CpT, &
               d_inv_4pi_r4_P_dlnR, d_inv_4pi_r4_P_dlnd, d_inv_4pi_r4_P_dlnT, &
               d_dg_dlnd00, d_dg_dlnT00, d_dg_dlndm1, d_dg_dlnTm1, d_dg_dlnR, d_dg_dL, &
               d_CpT_dlnd, d_CpT_dlnT, d_CpT_dlnPgas_const_T, d_CpT_dlnT_const_Pgas, &
               d_dg_dlnPgas00_const_T, d_dg_dlnPgasm1_const_T, d_dg_dlnT00_const_Pgas, &
               d_dg_dlnTm1_const_Pgas, d_inv_4pi_r4_P_dlnPgas_const_T, &
               d_inv_4pi_r4_P_dlnT_const_Pgas, dP_dlnPgas_const_T, dP_dlnT_const_Pgas, &
               dPinv_dlnPgas_const_T, dPinv_dlnT_const_Pgas
               
            ! ideally, this should be revised to use cell centered values for gradT and r.
            
            inv_4pi_r4 = 1/(4*pi*s% r(k)**4)
            inv_4pi_r4_P = inv_4pi_r4/s% P(k)
            dg = s% grada(k) - s% gradT(k)
            G_m_mdot = s% cgrav(k)*s% m(k)*s% mstar_dot
            CpT = s% Cp(k)*s% T(k)
            
            s% eps_grav(k) = dg*G_m_mdot*CpT*inv_4pi_r4_P
            
            d_inv_4pi_r4_P_dlnR = -4*inv_4pi_r4_P
            
            d_dg_dlnR = -s% d_gradT_dlnR(k)
            d_dg_dL = -s% d_gradT_dL(k)

            d_dg_dlnd00 = s% d_eos_dlnd(i_grad_ad,k) - s% d_gradT_dlnd00(k)
            d_dg_dlnT00 = s% d_eos_dlnT(i_grad_ad,k) - s% d_gradT_dlnT00(k)
         
            d_dg_dlndm1 = -s% d_gradT_dlndm1(k)
            d_dg_dlnTm1 = -s% d_gradT_dlnTm1(k)

            d_CpT_dlnd = s% d_eos_dlnd(i_Cp,k)*s% T(k)
            d_CpT_dlnT = s% d_eos_dlnT(i_Cp,k)*s% T(k) + CpT 

            s% d_eps_grav_dlnR00(k) = d_dg_dlnR*G_m_mdot*CpT*inv_4pi_r4_P
            s% d_eps_grav_dL00(k) = d_dg_dL*G_m_mdot*CpT*inv_4pi_r4_P
            
            if (s% lnPgas_flag) then
               
               dP_dlnPgas_const_T = s% Pgas(k)
               dP_dlnT_const_Pgas = 4*s% Prad(k)
               
               dPinv_dlnPgas_const_T = -dP_dlnPgas_const_T/s% P(k)**2
               dPinv_dlnT_const_Pgas = -dP_dlnT_const_Pgas/s% P(k)**2
               
               d_inv_4pi_r4_P_dlnPgas_const_T = inv_4pi_r4*dPinv_dlnPgas_const_T
               d_inv_4pi_r4_P_dlnT_const_Pgas = inv_4pi_r4*dPinv_dlnT_const_Pgas
               
               d_CpT_dlnPgas_const_T = d_CpT_dlnd*s% dlnRho_dlnPgas_const_T(k)
               d_CpT_dlnT_const_Pgas = &
                  d_CpT_dlnT + d_CpT_dlnd*s% dlnRho_dlnT_const_Pgas(k)
               
               d_dg_dlnPgas00_const_T = d_dg_dlnd00*s% dlnRho_dlnPgas_const_T(k)
               d_dg_dlnT00_const_Pgas = &
                  d_dg_dlnT00 + d_dg_dlnd00*s% dlnRho_dlnT_const_Pgas(k)
               
               if (k > 1) then
                  d_dg_dlnPgasm1_const_T = d_dg_dlndm1*s% dlnRho_dlnPgas_const_T(k-1)
                  d_dg_dlnTm1_const_Pgas = &
                     d_dg_dlnTm1 + d_dg_dlndm1*s% dlnRho_dlnT_const_Pgas(k-1)
               else
                  d_dg_dlnPgasm1_const_T = 0
                  d_dg_dlnTm1_const_Pgas = 0
               end if
               
               s% d_eps_grav_dlnPgasm1_const_T(k) = &
                  d_dg_dlnPgasm1_const_T*G_m_mdot*CpT*inv_4pi_r4_P
               s% d_eps_grav_dlnPgas00_const_T(k) = G_m_mdot*( &
                  d_dg_dlnPgas00_const_T*CpT*inv_4pi_r4_P + &
                  dg*d_CpT_dlnPgas_const_T*inv_4pi_r4_P + &
                  dg*CpT*d_inv_4pi_r4_P_dlnPgas_const_T)
               s% d_eps_grav_dlnPgasp1_const_T(k) = 0
               
               s% d_eps_grav_dlnTm1_const_Pgas(k) = &
                  d_dg_dlnTm1_const_Pgas*G_m_mdot*CpT*inv_4pi_r4_P
               s% d_eps_grav_dlnT00_const_Pgas(k) = G_m_mdot*( &
                  d_dg_dlnT00_const_Pgas*CpT*inv_4pi_r4_P + &
                  dg*d_CpT_dlnT_const_Pgas*inv_4pi_r4_P + &
                  dg*CpT*d_inv_4pi_r4_P_dlnT_const_Pgas)
               s% d_eps_grav_dlnTp1_const_Pgas(k) = 0
            
            else

               d_inv_4pi_r4_P_dlnd = -s% chiRho(k)*inv_4pi_r4_P
               d_inv_4pi_r4_P_dlnT = -s% chiT(k)*inv_4pi_r4_P

               s% d_eps_grav_dlndm1(k) = d_dg_dlndm1*G_m_mdot*CpT*inv_4pi_r4_P
               s% d_eps_grav_dlndp1(k) = 0
               s% d_eps_grav_dlnd00(k) = G_m_mdot*( &
                  d_dg_dlnd00*CpT*inv_4pi_r4_P + dg*d_CpT_dlnd*inv_4pi_r4_P + dg*CpT*d_inv_4pi_r4_P_dlnd)

               s% d_eps_grav_dlnTm1(k) = d_dg_dlnTm1*G_m_mdot*CpT*inv_4pi_r4_P
               s% d_eps_grav_dlnTp1(k) = 0
               s% d_eps_grav_dlnT00(k) = G_m_mdot*( &
                  d_dg_dlnT00*CpT*inv_4pi_r4_P + dg*d_CpT_dlnT*inv_4pi_r4_P + dg*CpT*d_inv_4pi_r4_P_dlnT)
            
            end if
            
      
         end subroutine do_thin_radiative
         
         

      end subroutine eval1_eps_grav_and_partials
      
      
      real(dp) function get_Teff(s, ierr) result(Teff)
         use atm_lib, only: atm_option
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: which_atm_option, off_table_option
         real(dp) :: r_surf, L_surf
         logical :: skip_partials
         real(dp) :: &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
            
         ierr = 0
         
         r_surf = s% r(1)
         L_surf = s% L(1)
         skip_partials = .true.
         
         which_atm_option = atm_option(s% which_atm_option, ierr)
         if (ierr /= 0) then
            write(*,*) 'unknown value for which_atm_option ' // trim(s% which_atm_option)
            return
         end if
      
         off_table_option = atm_option(s% which_atm_off_table_option, ierr)  
         if (ierr /= 0) then
            write(*,*) 'unknown value for which_atm_off_table_option ' // trim(s% which_atm_off_table_option)
            return
         end if
         
         call get_surf_PT( &
            s, which_atm_option, off_table_option, r_surf, L_surf, skip_partials, &
            Teff, lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         if (ierr /= 0) then
            return
         end if

      end function get_Teff

      
      subroutine get_surf_PT( &
            s, which_atm_option, off_table_option, r_surf, L_surf, skip_partials, &
            Teff, lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
            ierr)
         use atm_lib
         use atm_def
         use chem_def
         use alert_lib
         use create_atm, only: get_Paczynski_atm_surf_PT

         type (star_info), pointer :: s
         integer, intent(in) :: which_atm_option, off_table_option
         real(dp), intent(in) :: r_surf, L_surf
         logical, intent(in) :: skip_partials
         real(dp), intent(out) :: Teff, &
            lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
            lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap
         integer, intent(out) :: ierr
         
         real(dp) :: opacity, X, Y, Z, tau_surface, tau_center_cell1, &
            kap_surf, err
         integer :: iters, num_atm_structure_points
         real(dp), pointer :: atm_structure_results(:,:)
         logical :: save_atm_structure_info
         
         include 'formats.dek'
      
         opacity = s% opacity(1)
         X = s% X(1); Y = s% Y(1); Z = 1 - (X+Y)
         tau_surface = s% tau_factor*s% tau_base ! tau at outer edge of cell 1
         
         !write(*,1) 'get_surf_PT tau_surface', tau_surface

         !write(*,*) 's% dtau1_pre_hydro > tau_surface', s% dtau1_pre_hydro > tau_surface
         if (s% use_other_atm) then
            call s% other_atm( &
               s% id, s% mstar, r_surf, L_surf, X, Z, opacity, Teff, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               which_atm_option, s% atm_switch_to_grey_as_backup, ierr)
         else if (which_atm_option == atm_grey_and_kap) then
            call atm_get_grey_and_kap( &
               tau_surface, opacity, &
               s% cgrav(1), s% mstar, r_surf, L_surf, X, Z, s% abar(1), s% zbar(1), & 
               s% species, s% chem_id, s% net_iso, s% xa(:,1), &
               s% atm_grey_and_kap_max_tries, s% atm_grey_and_kap_atol, s% atm_grey_and_kap_rtol, &
               s% eos_handle, s% kap_handle, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, & 
               kap_surf, Teff, iters, err, ierr)
            dlnT_dlnkap = 0
            dlnP_dlnkap = 0
            if (s% trace_atm_grey_and_kap .and. .not. skip_partials) then
               write(*,2) 'grey&kap: itrs, err, kap_s, kap1', &
                  iters, err, kap_surf, opacity
            end if    
         else if (which_atm_option == atm_Paczynski_grey) then
            call get_Paczynski_atm_surf_PT( &
               s, r_surf, L_surf, skip_partials, Teff, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               ierr)
            if (s% trace_atm_Paczynski_grey .and. .not. skip_partials) then
               write(*,*) 
               write(*,'(99a20)') &
                  'Paczynski_grey', 'Teff', 'lnT', 'dlnT_dL', 'dlnT_dlnR', 'lnP', 'dlnP_dL', 'dlnP_dlnR'
               write(*,'(i20,99e20.10)') &
                  s% model_number, Teff, lnT_surf, dlnT_dL, dlnT_dlnR, lnP_surf, dlnP_dL, dlnP_dlnR
            end if    
         else if (which_atm_option == atm_grey_irradiated) then
            call atm_grey_irradiated_get( &
               s% atm_grey_irradiated_T_eq, s% atm_grey_irradiated_kap_v, opacity, &
               s% atm_grey_irr_kap_v_div_kap_th, s% atm_grey_irradiated_P_surf, &
               s% cgrav(1), s% mstar, r_surf, L_surf, X, Z, s% abar(1), s% zbar(1), & 
               s% species, s% chem_id, s% net_iso, s% xa(:,1), &
               s% atm_grey_irradiated_max_tries, s% atm_grey_irradiated_atol, &
               s% atm_grey_irradiated_rtol, &
               s% eos_handle, s% kap_handle, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
               kap_surf, tau_surface, Teff, iters, err, ierr)
            s% tau_factor = tau_surface/s% tau_base
            lnP_surf = log(s% atm_grey_irradiated_P_surf)
            dlnP_dL = 0
            dlnP_dlnR = 0
            dlnP_dlnM = 0
            dlnT_dlnkap = 0
            dlnP_dlnkap = 0
            if (s% trace_atm_grey_irradiated .and. .not. skip_partials) then
               write(*,2) 'grey_irradiated: itrs, err, kap_s, kap1, tau_surface, Teff, T', &
                  iters, err, kap_surf, opacity, tau_surface, Teff, exp(lnT_surf)
            end if    
         else if (atm_by_integration(which_atm_option)) then
            save_atm_structure_info = .false.
            nullify(atm_structure_results)
            call atm_get_int_T_tau( &
               s% atm_int_errtol, s% cgrav(1), s% mstar, r_surf, L_surf, X, Z, s% abar(1), s% zbar(1), & 
               s% species, s% chem_id, s% net_iso, s% xa(:,1), &
               which_atm_option, s% eos_handle, s% kap_handle, save_atm_structure_info, &
               tau_surface, skip_partials, Teff, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               num_atm_structure_points, atm_structure_results, ierr)
         else if (atm_by_table(which_atm_option)) then
            if (s% tau_factor /= 1) then
               write(*,*) 'cannot simultaneously have atm by table and tau_factor /= 1'
               ierr = -1
               return
            end if
            s% tau_base = atm_tau_base(which_atm_option, ierr)
            if (ierr == 0) call atm_get_table( &
               which_atm_option, off_table_option, &
               s% cgrav(1), s% mstar, r_surf, L_surf, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, & 
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, & 
               opacity, X, Z, s% abar(1), s% zbar(1), & 
               s% species, s% chem_id, s% net_iso, s% xa(:,1), &
               s% atm_grey_and_kap_max_tries, &
               s% atm_grey_and_kap_atol, s% atm_grey_and_kap_rtol, &
               s% eos_handle, s% kap_handle, Teff, ierr)
            dlnT_dlnkap = 0
            dlnP_dlnkap = 0
         else
            call atm_get_grey( &
               tau_surface, s% cgrav(1), s% mstar, r_surf, L_surf, opacity, Teff, &
               lnT_surf, dlnT_dL, dlnT_dlnR, dlnT_dlnM, dlnT_dlnkap, &
               lnP_surf, dlnP_dL, dlnP_dlnR, dlnP_dlnM, dlnP_dlnkap, &
               ierr)
         end if

         !call show
         !stop 'get PT surf'
         
         if (is_bad_num(lnT_surf) .or. is_bad_num(lnP_surf)) then
            call show
            ierr = -1
            return
            !stop 'get_surf_PT'
         end if

         if (ierr /= 0) then
            if (s% report_ierr) then
               write(*, *) 'atm_get returned ierr', ierr
               write(*, *) trim(alert_message)
               write(*, *)
            end if
            return
         end if
         
         
         contains
         
         subroutine show
            include 'formats.dek'
            write(*,1) 'M =', s% mstar
            write(*,1) 'R =', r_surf
            write(*,1) 'L =', L_surf
            write(*,1) 'X =', X
            write(*,1) 'Z =', Z
            write(*,1) 'opacity =', opacity
            write(*,2) 'which_atm_option =', which_atm_option
            write(*,*) trim(s% which_atm_option)
            write(*,*) 's% atm_switch_to_grey_as_backup', s% atm_switch_to_grey_as_backup
            write(*,*)
            write(*,1) 's% tau_factor', s% tau_factor
            write(*,*)
            write(*,1) 'Teff', Teff
            write(*,1) 'logT_surf', lnT_surf/ln10
            write(*,1) 'logP_surf', lnP_surf/ln10
            write(*,*)
            write(*,1) 'lnT_surf', lnT_surf
            write(*,1) 'lnP_surf', lnP_surf
            write(*,*)
            write(*,1) 'dlnT_dL', dlnT_dL
            write(*,1) 'dlnT_dlnR', dlnT_dlnR
            write(*,1) 'dlnT_dlnM', dlnT_dlnM
            write(*,1) 'dlnT_dlnkap', dlnT_dlnkap
            write(*,*)
            write(*,1) 'dlnP_dL', dlnP_dL
            write(*,1) 'dlnP_dlnR', dlnP_dlnR
            write(*,1) 'dlnP_dlnM', dlnP_dlnM
            write(*,1) 'dlnP_dlnkap', dlnP_dlnkap
            write(*,*)
         end subroutine show
         
      end subroutine get_surf_PT
      
      
      subroutine get_dVARDOT_dVAR_pt(s, k, dt, d_dvarm1, d_dvar00, d_dvarp1)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         real(dp), intent(out) :: d_dvarm1, d_dvar00, d_dvarp1
         real(dp) :: dqsum, q
         if (k >= s% k_below_recently_added .or. dt <= 0) then
            d_dvarm1 = 0
            d_dvar00 = s% dVARdot_dVAR
            d_dvarp1 = 0
         else
            if (k > 1 .and. k < s% nz) then ! see set_d_dt_for_recently_added
               dqsum = s% dq(k-1) + s% dq(k)
               q = s% q(k)
               d_dvarm1 = -q*s% mstar_dot/s% xmstar/dqsum
            else
               d_dvarm1 = 0
            end if
            d_dvar00 = 1/dt
            d_dvarp1 = -d_dvarm1
         end if
      end subroutine get_dVARDOT_dVAR_pt
      
      
      subroutine get_dVARDOT_dVAR_cell(s, k, dt, d_dvarm1, d_dvar00, d_dvarp1)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         real(dp), intent(out) :: d_dvarm1, d_dvar00, d_dvarp1
         real(dp) :: dqsum, q
         if (k >= s% k_below_recently_added .or. dt <= 0) then
            d_dvarm1 = 0
            d_dvar00 = s% dVARdot_dVAR
            d_dvarp1 = 0
         else
            if (k > 1 .and. k < s% nz) then ! see set_d_dt_for_recently_added
               dqsum = s% dq(k) + 0.5d0*(s% dq(k-1) + s% dq(k+1))
               q = s% q(k) - s% dq(k)
               d_dvarm1 = -q*s% mstar_dot/s% xmstar/dqsum
            else
               d_dvarm1 = 0
            end if
            d_dvar00 = 1/dt
            d_dvarp1 = -d_dvarm1
         end if
      end subroutine get_dVARDOT_dVAR_cell

         
      real(dp) function get_dlnE_dlnd(s,k)
         use eos_def, only: i_lnE
         type (star_info), pointer :: s
         integer, intent(in) :: k
         get_dlnE_dlnd = s% d_eos_dlnd(i_lnE,k)
      end function get_dlnE_dlnd

         
      real(dp) function get_dlnE_dlnT(s,k)
         use eos_def, only: i_lnE
         type (star_info), pointer :: s
         integer, intent(in) :: k
         get_dlnE_dlnT = s% d_eos_dlnT(i_lnE,k)
      end function get_dlnE_dlnT

         
      end module hydro_vars

