! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module hydro_rosenbrock
      
      use star_private_def
      use utils_lib, only: alloc_iounit, free_iounit, is_bad_num, has_bad_num
      use const_def
      use num_def
      
      
      implicit none


      contains
      

      integer function do_hydro_rosenbrock( &
            s, dt, report, dumping, numerical_jacobian, &
            decsolblk, lrd, rpar_decsol, lid, ipar_decsol)
         ! return keep_going, retry, backup, or terminate
         
         ! do not require that functions have been evaluated for starting configuration.
         ! when finish, will have functions evaluated for the final set of primary variables.
         
         use hydro_newton, only: check_after_converge, set_xscale_info
         use hydro_mtx, only: set_newton_vars
         use hydro_eqns, only:eval_equ
         use star_utils, only: total_times
         use solve_burn_mix_new
         use utils_lib, only: set_pointer_2

         type (star_info), pointer :: s
         real(dp), intent(in) :: dt 
         logical, intent(in) :: report, dumping, numerical_jacobian
         interface
            include "mtx_decsolblk.dek"
         end interface
         integer, intent(in) :: lrd, lid
         integer, pointer :: ipar_decsol(:) ! (lid)
         real(dp), pointer :: rpar_decsol(:) ! (lrd)
         
         integer :: i, j, k, stage, nz, nvar, nvar_hydro, species, ierr
         logical :: skip_partials, converged, need_jacobian, have_a_factored_mtx
         
         integer :: time0, time1, clock_rate
         real(dp) :: total_all_before, total_all_after
         
         integer, parameter :: lrpar = 0, lipar = 0
         real(dp) :: rpar(lrpar)
         integer :: ipar(lipar)

         real(dp) :: gamma_dt, atol, rtol
         real(dp) :: gamma_c(max_ns,max_ns)
         
         ! copy of pointer; not to be allocated or deallocated
         real(dp), pointer :: U(:,:)
         
         ! temp storage to be allocated and deallocated
         real(dp), pointer, dimension(:,:) :: &
            rhs, error_vector, dx, xscale, wrk_for_refine ! (nvar,nz)
         real(dp), pointer, dimension(:,:,:) :: &
            Un, & ! (nvar,nz,ns)
            lhs_lblk, lhs_dblk, lhs_ublk, & ! (nvar,nvar,nz)
            lhs_lblkF, lhs_dblkF, lhs_ublkF ! (nvar,nvar,nz)
         integer, pointer :: ipiv_blk(:,:) ! (nvar,nz)
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'


         do_hydro_rosenbrock = retry
         converged = .false.
         have_a_factored_mtx = .false.
         
         if (s% ros_solver_name == 'ros2') then
            call ros2_coeffs
         else if (s% ros_solver_name == 'ros3pl') then
            call ros3pl_coeffs
         else if (s% ros_solver_name == 'rodas3') then
            call rodas3_coeffs
         else if (s% ros_solver_name == 'rodasp') then
            call rodasp_coeffs
         else 
            write(*,*) 'unknown ros_solver_name: "' // trim(s% ros_solver_name) // '"'
            stop 1
         end if
         
         if (dbg) then
            write(*,*)
            write(*,2) trim(s% ros_solver_name) // ' model, logdt', &
               s% model_number, log10(dt/secyer)
         end if
         
         if (s% doing_timing) then
            call system_clock(time0,clock_rate)
            total_all_before = total_times(s)
         else
            total_all_before = 0
         end if
         
         ierr = 0         
         nz = s% nz
         nvar = s% nvar
         species = s% species
         nvar_hydro = s% nvar_hydro
         gamma_dt = gamma*dt
         s% num_jacobians = 1
         s% total_num_jacobians = s% total_num_jacobians + s% num_jacobians
         s% num_solves = 0
         s% hydro_matrix_type = block_tridiagonal_matrix_type
            
         call do_alloc(ierr)
         if (ierr /= 0) return

         do i=2,ns
            do j=1,i-1
               gamma_c(i,j) = gamma*c(i,j)
            end do
         end do
         
         if (dbg) write(*,*) 'call set_xscale_info'
         call set_xscale_info(s, nvar, nz, xscale, ierr)
         if (ierr /= 0) then
            !if (s% report_ierr) &
               write(*,2) 'hydro_rosenbrock prepare_jacobian_info: set_xscale_info ierr', ierr
               stop
            return
         end if 

         need_jacobian = .true.
         
         ! compute the stages
         do stage = 1, ns
         
            if (dbg) write(*,3) 'model, stage', s% model_number, stage
            
            if (newf(stage)) then
               
               if (stage > 1) then
!$OMP PARALLEL DO PRIVATE(j,k)
                  do k=1,nz
                     forall (j=1:nvar) &
                        dx(j,k) = sum(a(stage,1:stage-1)*Un(j,k,1:stage-1))
                  end do
!$OMP END PARALLEL DO
                  call set_newton_vars(s, stage, dx, xscale, alpha(stage)*dt, ierr)
                  if (ierr /= 0) then
                     !if (s% report_ierr) then
                        write(*,3) 'ros set_newton_vars err: model, stage', &
                           s% model_number, stage
                        write(*,*)
                        !if (stage == 3) stop
                     !end if
                     call dealloc
                     return
                  end if
               end if

               skip_partials = (.not. need_jacobian)
               call eval_equ(s, nvar, alpha(stage)*dt, skip_partials, xscale, ierr)         
               if (ierr /= 0) then
                  !if (s% report_ierr) then
                     write(*, *) 'hydro_rosenbrock: eval_equ returned ierr', ierr
                  !end if
                  !stop
                  call dealloc
                  return
               end if
         
               if (need_jacobian) then ! calculate jacobian and factor
                  if (dbg) write(*,2) 'call prepare_jacobian_info'
                  call prepare_jacobian_info(ierr)
                  if (ierr /= 0) return
                  !need_jacobian = .false.  ! <<<< fails to converge unless make new J
               end if

            end if
            
!$OMP PARALLEL DO PRIVATE(j,k)
            do j=1,nvar
               forall (k=1:nz) rhs(j,k) = gamma_dt*s% equ(j,k)
               if (stage > 1 .and. s% ode_var(j)) &
                  forall (k=1:nz) rhs(j,k) = rhs(j,k) + &
                     sum(gamma_c(stage,1:stage-1)*Un(j,k,1:stage-1))
                     ! + gamma*gamma_i*dt**2*dequ_dt(j,k) < we assume dequ_dt == 0
            end do
!$OMP END PARALLEL DO
               
            call set_pointer_2(U, Un(:,:,stage), nvar, nz)
            
            if (dbg) write(*,2) 'solve matrix stage', stage
            ! solve (M - gamma_dt*J)*U = rhs

            call solve_mtx(U, ierr)
            if (ierr /= 0) then
               write(*,2) 'rosenbrock: solve_mtx failed', stage
               stop
               call dealloc
               return
            end if
            s% num_solves = s% num_solves + 1
            
         end do
         
         if (dbg) write(*,2) 'calculate final result'
         ! calculate final result and evaluate vars
         forall (k=1:nz,j=1:nvar) dx(j,k) = sum(m(1:ns)*Un(j,k,1:ns))
         call set_newton_vars(s, ns+1, dx, xscale, dt, ierr)
         if (ierr /= 0) then
            write(*,1) 'rosenbrock: final set_newton_vars failed'
            write(*,*)
            !stop
            call dealloc
            return
         end if
         
         ! calculate error estimate
         if (dbg) write(*,2) 'calculate error estimate'         
         if (.true.) then
            forall (k=1:nz,j=1:nvar) &
               error_vector(j,k) = sum(e(1:ns)*Un(j,k,1:ns))/xscale(j,k)
         else ! filter out stiff components of error
            ! solve (M - gamma*dt*J)*error_vector = sum(e(1:ns)*Un(j,k,1:ns))
               ! see L.F. Shampine, L.S. Baca,
               ! "Error estimators for stiff differential equations",
               ! J. Comput. Appl. Math. 11 (1984) 197–207.
            forall (k=1:nz,j=1:species) rhs(j,k) = sum(e(1:ns)*Un(j,k,1:ns))
            call solve_mtx(error_vector, ierr)
            if (ierr /= 0) then
               write(*,1) 'rosenbrock: solve_mtx failed for error vector'
               stop
               call dealloc
               return
            end if
            forall (k=1:nz,j=1:nvar) &
               error_vector(j,k) = error_vector(j,k)/xscale(j,k)
            s% num_solves = s% num_solves + 1
         end if




         
         ! NOTE: the estimate above is for the error of the embedded lower order solution.
         ! to compensate, we adjust tolerances.
         ! see L. Jay, SIAM J. Sci. Comput. 20, 1998. 416–446.    (pp. 438-439)
         ! elo is estimated local order of the main method.
         ! we assume the embedded method is order elo-1.
         rtol = s% hydro_err_ratio_rtol**(dble(elo)/dble(elo+1))
         atol = s% hydro_err_ratio_atol**(dble(elo)/dble(elo+1))
         ! by adjusting the tolerances, we convert the error ratios to estimates
         ! for the local error of the main method rather than the embedded one.
                  
         call get_err_ratio_info( &
            s% err_ratio_max_hydro, s% err_ratio_norm_hydro)
         if (dbg) write(*,1) 's% err_ratio_max_hydro', s% err_ratio_max_hydro
         if (dbg) write(*,1) 's% err_ratio_norm_hydro', s% err_ratio_norm_hydro
         
         converged = (s% err_ratio_norm_hydro <= 1d0)
         
         
         
         call set_dt_next_limit
         
         
         ! REPORT
         !write(*,2) trim(s% ros_solver_name) // ' err max, norm, log_dt, log_dt_limit', &
         !   s% model_number, s% err_ratio_max_hydro, s% err_ratio_norm_hydro, &
         !   log10(dt/secyer), log10(s% hydro_ros_dt_limit/secyer)


                  
         if (converged) then ! set final result and evaluate vars
            ! s% xa has already been updated by final call to set_newton_vars         
            do k=1,nz
               do j=1,nvar_hydro
                  s% xh(j,k) = s% xh_pre_hydro(j,k) + dx(j,k)
               end do
            end do
            ! a few more sanity checks before accept it
            converged = check_after_converge(s, report, ierr)
            if (.not. converged) then
               if (report) write(*,2) 'check_after_converge rejected: model', s% model_number
               !stop 'hydro_rosenbrock'
            end if
         end if

         if (s% doing_timing) then
            call system_clock(time1,clock_rate)
            total_all_after = total_times(s)
            ! see hydro_newton subroutine newt
         end if

         if (dbg) write(*,2) 'done hydro_rosenbrock'
         if (dbg) write(*,2)

         if (dumping) stop 'debug: dumping hydro_rosenbrock' 
         
         
         call dealloc
         
         if (converged) then
            do_hydro_rosenbrock = keep_going

            !write(*, *) 'hydro_rosenbrock converged'
            !write(*,*)

         else
            do_hydro_rosenbrock = retry
            s% result_reason = hydro_failed_to_converge
            if (report) then
               write(*, *) 'hydro_rosenbrock failed to converge'
               write(*,2) 's% model_number', s% model_number
               write(*,2) 's% hydro_call_number', s% hydro_call_number
               write(*,2) 'nz', nz
               write(*,2) 's% num_retries', s% num_retries
               write(*,2) 's% num_backups', s% num_backups
               write(*,2) 's% number_of_backups_in_a_row', s% number_of_backups_in_a_row
               write(*,1) 'log dt/secyer', log10(dt/secyer)
               write(*, *) 
               if (.false.) then
                  do k=1,nz
                     do i=1,nvar
                        if (abs(s% error_vector_hydro_ros(i,k)) > 1) &
                           write(*,2) 'error ratio ' // trim(s% nameofvar(i)), k, &
                              abs(s% error_vector_hydro_ros(i,k))
                     end do
                  end do
               write(*,*)
               end if
               !stop
            else
               write(*, *) 'hydro_rosenbrock failed to converge'
               write(*,*)
            end if
         
         
            !stop
            

            return
         end if
         
         
         contains         



         
         subroutine set_dt_next_limit
            
            real(dp) :: beta1, beta2, alpha2, dt_old, f, f_prev, limtr
            integer :: order
            include 'formats.dek'
            
            order = elo-1 ! order of embedded method
            beta1 = 0.25d0/order
            beta2 = 0.25d0/order
            alpha2 = 0.25d0
            dt_old = s% dt_old

            if (s% err_ratio_max_hydro_old > 0 .and. &
                s% err_ratio_max_hydro > 0 .and. s% err_ratio_max_hydro < 1 .and. dt_old > 0) then 
               ! use 2 values to do "low pass" controller
               ! H211b "low pass" controller.
               ! Soderlind Wang, J of Comp and Applied Math 185 (2006) 225 – 243.
               f = limiter(1/s% err_ratio_max_hydro)
               f_prev = limiter(1/s% err_ratio_max_hydro_old)    
               limtr = limiter(f**beta1 * f_prev**beta2 * (dt/dt_old)**(-alpha2))               
            else ! use the basic 1 step controller
               limtr = limiter(s% err_ratio_max_hydro**(-1d0/elo))
            end if            
            s% hydro_ros_dt_limit = dt * limtr
            
         end subroutine set_dt_next_limit

         
         real(dp) function limiter(x)
            real(dp), intent(in) :: x
            real(dp), parameter :: kappa = 2
            ! for x >= 0 and kappa = 2, limiter value is between 0.07 and 4.14
            ! for x = 1, limiter = 1
            limiter = 1 + kappa*ATAN((x-1)/kappa)
         end function limiter
         
         
         


         
         
         subroutine prepare_jacobian_info(ierr)
            integer, intent(out) :: ierr
            
            integer :: i, j, k

            include 'formats.dek'

            ierr = 0

            !if (s% jacobian_clip_limit > 0) call clip_mtx(s% jacobian_clip_limit)
               
            ! set lhs matrix to (M - gamma_dt*J)
!$OMP PARALLEL DO PRIVATE(i,k,j)
            do k=1,nz
               forall (i=1:nvar,j=1:nvar)
                  lhs_lblk(i,j,k) = -gamma_dt*s% lblk(i,j,k)
                  lhs_dblk(i,j,k) = -gamma_dt*s% dblk(i,j,k)
                  lhs_ublk(i,j,k) = -gamma_dt*s% ublk(i,j,k)
               end forall
               do i=1,nvar
                  if (.not. s% ode_var(i)) cycle
                  lhs_dblk(i,i,k) = lhs_dblk(i,i,k) + 1
               end do
               ! make copy for factoring
               forall (i=1:nvar,j=1:nvar)
                  lhs_lblkF(i,j,k) = lhs_lblk(i,j,k)
                  lhs_dblkF(i,j,k) = lhs_dblk(i,j,k)
                  lhs_ublkF(i,j,k) = lhs_ublk(i,j,k)
               end forall
            end do
!$OMP END PARALLEL DO

            if (have_a_factored_mtx) then
               call dealloc_mtx(ierr)
               if (ierr /= 0) return
            end if
                 
            if (dbg) write(*,*) 'call decsolblk to factor'
            call decsolblk( & ! factor
               0, lhs_lblkF, lhs_dblkF, lhs_ublkF, rhs, ipiv_blk, &
               lrd, rpar_decsol, lid, ipar_decsol, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in decsolblk factor'
               stop 'hydro_rosenbrock prepare_jacobian_info'
            end if
            have_a_factored_mtx = .true.
         
         end subroutine prepare_jacobian_info
         
         
         subroutine get_err_ratio_info(err_ratio_max, err_ratio_norm)
            real(dp), intent(out) :: err_ratio_max, err_ratio_norm         
            real(dp) :: x, err_ratio, sum_err_ratio_squared
            integer :: k_max_err, j_max_err, k, j, stage, n, i_vel, i_FL, i_omega
            include 'formats.dek'
            i_vel = s% i_vel
            i_FL = s% i_FL
            i_omega = s% i_omega
            err_ratio_max = -1
            k_max_err = -1
            j_max_err = -1
            sum_err_ratio_squared = 0
            n = 0
            !write(*,1) 'err ratio rtol, atol', rtol, atol
            do k=1,nz 
               do j=1,nvar
                  
                  
                  
                  !if (j == i_FL .or. j == i_vel .or. j == i_omega) cycle
                  
                  
                  
                  if (j <= s% nvar_hydro) then
                     x = s% xh_pre_hydro(j,k)
                  else
                     x = s% xa_pre_hydro(j-nvar_hydro,k)
                  end if
                  x = x / xscale(j,k)
                  err_ratio = abs(error_vector(j,k))/(atol + rtol*abs(x))
                  error_vector(j,k) = err_ratio
                  if (err_ratio > err_ratio_max) then
                     k_max_err = k
                     j_max_err = j
                     err_ratio_max = err_ratio
                  end if
                  n = n+1
                  sum_err_ratio_squared = &
                     sum_err_ratio_squared + err_ratio**2
               end do
            end do            
            err_ratio_norm = sqrt(sum_err_ratio_squared/n)

            j = j_max_err
            k = k_max_err
            write(*,3) trim(s% ros_solver_name) // ' err max avg lgdt ' // trim(s% nameofvar(j)), &
               s% model_number, k, err_ratio_max, err_ratio_norm, log10(dt/secyer)
            
            return
            
            write(*,1) 's% xh_pre_hydro(j,k)', s% xh_pre_hydro(j,k)
            do stage = 1, ns
               write(*,2) 'Un(j,k,stage)', stage, Un(j,k,stage)
            end do
            
            stop
            
            
         end subroutine get_err_ratio_info
         
         
         subroutine solve_mtx(U, ierr)
            use mtx_lib, only: block_lapack_refine1
            real(dp), pointer :: U(:,:)
            integer, intent(out) :: ierr
            
            integer :: j, k
            
            include 'formats.dek'
            
            ierr = 0
            forall (k=1:nz,j=1:nvar) U(j,k) = rhs(j,k)
            
            call decsolblk( & ! solve.   solution goes in U
               1, lhs_lblkF, lhs_dblkF, lhs_ublkF, U, ipiv_blk, &
               lrd, rpar_decsol, lid, ipar_decsol, ierr)
            if (ierr /= 0) return
            
            if (.false.) & ! this doesn't seem to help
               call block_lapack_refine1( &
                  lhs_lblk, lhs_dblk, lhs_ublk, lhs_lblkF, lhs_dblkF, lhs_ublkF, &
                  ipiv_blk, decsolblk, rhs, U, wrk_for_refine, &
                  lrd, rpar_decsol, lid, ipar_decsol, ierr)
            
            ! J = partials wrt scaled variables
            ! so multiply solution by xscale to get actual (unscaled) result
            forall (k=1:nz,j=1:nvar) U(j,k) = U(j,k)*xscale(j,k)
         
         end subroutine solve_mtx
         
         
         subroutine dealloc_mtx(ierr)
            integer, intent(out) :: ierr            
            ierr = 0
            call decsolblk( & ! deallocate
               2, lhs_lblkF, lhs_dblkF, lhs_ublkF, rhs, ipiv_blk, &
               lrd, rpar_decsol, lid, ipar_decsol, ierr)               
         end subroutine dealloc_mtx
         
         
         subroutine alloc_nvar_nz(p, ierr)
            use alloc, only: get_2d_work_array
            real(dp), pointer :: p(:,:)
            integer, intent(out) :: ierr
            call get_2d_work_array( &
               s, p, nvar, nz, nz_alloc_extra, 'hydro_rosenbrock', ierr)
         end subroutine alloc_nvar_nz
         
         
         subroutine alloc_nvar_nvar_nz(p, ierr)
            use alloc, only: get_3d_work_array
            real(dp), pointer :: p(:,:,:)
            integer, intent(out) :: ierr
            call get_3d_work_array( &
               s, p, nvar, nvar, nz, nz_alloc_extra, 'hydro_rosenbrock', ierr)
         end subroutine alloc_nvar_nvar_nz
      
      
         subroutine do_alloc(ierr)
            use alloc, only: &
               get_2d_work_array, get_3d_work_array, get_integer_2d_work_array
            use mtx_lib, only: lapack_work_sizes
            integer, intent(out) :: ierr
            ierr = 0            
            
            call alloc_nvar_nz(rhs, ierr)
            if (ierr /= 0) return
            
            if (associated(s% error_vector_hydro_ros)) &
               call return_2d(s% error_vector_hydro_ros) ! return previous step info
            call alloc_nvar_nz(s% error_vector_hydro_ros, ierr)
            if (ierr /= 0) return
            error_vector => s% error_vector_hydro_ros
            
            call alloc_nvar_nz(wrk_for_refine, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nz(dx, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nz(xscale, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(s% lblk, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(s% dblk, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(s% ublk, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(lhs_lblk, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(lhs_dblk, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(lhs_ublk, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(lhs_lblkF, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(lhs_dblkF, ierr)
            if (ierr /= 0) return
            
            call alloc_nvar_nvar_nz(lhs_ublkF, ierr)
            if (ierr /= 0) return
            
            if (associated(s% Un_hydro_ros)) &
               call return_3d(s% Un_hydro_ros) ! return previous step info
            call get_3d_work_array( &
               s, s% Un_hydro_ros, nvar, nz, ns, 2, 'hydro_rosenbrock', ierr)
            if (ierr /= 0) return
            Un => s% Un_hydro_ros
            
            call get_integer_2d_work_array( &
               s, ipiv_blk, nvar, nz, nz_alloc_extra, ierr)
            if (ierr /= 0) return         



            return
            
            s% Un_hydro_ros = 0  
            s% error_vector_hydro_ros = 0
            rhs = 0
            dx = 0
            xscale = 0
            s% lblk = 0
            s% dblk = 0
            s% ublk = 0
            lhs_lblk = 0   
            lhs_dblk = 0   
            lhs_ublk = 0   
            lhs_lblkF = 0   
            lhs_dblkF = 0   
            lhs_ublkF = 0   
            ipiv_blk = 0        



         
         end subroutine do_alloc
         
         
         subroutine dealloc
            use alloc
            include 'formats.dek'
            
            if (have_a_factored_mtx) then
               call dealloc_mtx(ierr)
               if (ierr /= 0) return
            end if
            
            call return_2d(wrk_for_refine)
            call return_2d(rhs) 
            call return_2d(dx) 
            call return_2d(xscale) 
            call return_3d(s% lblk)   
            call return_3d(s% dblk)   
            call return_3d(s% ublk)   
            call return_3d(lhs_lblk)   
            call return_3d(lhs_dblk)   
            call return_3d(lhs_ublk)   
            call return_3d(lhs_lblkF)   
            call return_3d(lhs_dblkF)   
            call return_3d(lhs_ublkF)   
            call return_integer_2d(ipiv_blk)         

            ! keep these for plotting
            !call return_2d(s% error_vector_hydro_ros) 
            !call return_3d(s% Un_hydro_ros)   
            
         end subroutine dealloc

         
         subroutine return_1d(p)
            use alloc
            real(dp), pointer :: p(:)
            call return_work_array(s, p, 'hydro_rosenbrock')
         end subroutine return_1d
         
         
         subroutine return_2d(p)
            use alloc
            real(dp), pointer :: p(:,:)
            call return_2d_work_array(s, p, 'hydro_rosenbrock')
         end subroutine return_2d
         
         
         subroutine return_3d(p)
            use alloc
            real(dp), pointer :: p(:,:,:)
            call return_3d_work_array(s, p, 'hydro_rosenbrock')
         end subroutine return_3d
         
         
         subroutine return_integer_2d(p)
            use alloc
            integer, pointer :: p(:,:)
            call return_integer_2d_work_array(s, p)
         end subroutine return_integer_2d

         
         
         
      end function do_hydro_rosenbrock
      

      

      end module hydro_rosenbrock

