! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

#ifdef DBLE
      module hydro_newton_procs_dble
      use hydro_mtx_dble
#else
      module hydro_newton_procs_quad
      use hydro_mtx_quad
#endif
      
      use star_private_def
      use alert_lib, only: alert
      use utils_lib, only: alloc_iounit, free_iounit, &
         is_bad_num, is_bad_quad, has_bad_num
      use const_def
      
      use num_def
      
      implicit none

#ifdef DBLE
#define is_bad is_bad_num
#else
#define is_bad is_bad_quad
#endif


      contains


      subroutine set_xscale( &
            nvar, nz, dx_init, xscale, lrpar, rpar, lipar, ipar, ierr)
         use star_utils, only: update_time, total_times
         integer, intent(in) :: nvar, nz
         real(fltp), pointer :: dx_init(:,:) ! (nvar, nz)
         real(fltp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
         
         type (star_info), pointer :: s   
         integer :: id
         
         include 'formats.dek'

         if (dbg) write(*, *) 'set_xscale'
         
         id = ipar(ipar_id)
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call set_xscale_info(s, nvar, nz, xscale, ierr)
      
      end subroutine set_xscale


      subroutine set_xscale_info(s, nvar, nz, xscale, ierr)
         use star_utils, only: update_time, total_times
         type (star_info), pointer :: s   
         integer, intent(in) :: nvar, nz
         real(fltp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(out) :: ierr

         integer :: i, j, k, nvar_hydro, i_vel
         real(fltp), parameter :: xscale_min = 1
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         real(fltp) :: var_scale, lum_scale, vel_scale, omega_scale
         
         include 'formats.dek'
         
         ierr = 0

         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if         
         
         if (dbg) write(*, *) 'set_xscale'
         nvar_hydro = s% nvar_hydro
         i_vel = s% i_vel
         
         if (.not. s% doing_hydro_newton) then
            do i=1,nvar_hydro
               if (s% ode_var(i)) then
                  if (i == s% i_lnT .and. s% lnTdot_flag) then
                     ! okay
                  else if (i == s% i_xlnd .and. s% lnddot_flag) then
                     ! okay
                  else
                     write(*,2) 'bad value for ode_var ' // trim(s% nameofvar(i)), i
                        write(*,*)
                        do j=1,nvar
                           write(*,*) 's% ode_var(j)', s% ode_var(j)
                        end do
                        write(*,*)
                     stop
                  end if
               end if
            end do
            do i=nvar_hydro+1,nvar
               if (.not. s% ode_var(i)) then
                  write(*,2) 'bad value for ode_var ' // trim(s% nameofvar(i)), i
                        write(*,*)
                        do j=1,nvar
                           write(*,*) 's% ode_var(j)', s% ode_var(j)
                        end do
                        write(*,*)
                  stop
               end if
            end do
         end if
         
         if (i_vel /= 0) then
            vel_scale = 1d1*max(xscale_min, maxval(abs(s% xh_pre_hydro(i_vel,1:nz))))
         else
            vel_scale = 0
         end if
         
         do k=1,nz
            do i=1,nvar
               if ((.not. s% doing_hydro_newton) .and. s% ode_var(i)) then
                  xscale(i,k) = 1
               else if (i <= nvar_hydro) then ! structure variable
                  if (i == i_vel) then
                     xscale(i,k) = vel_scale
                  else
                     xscale(i,k) = max(xscale_min, abs(s% xh_pre_hydro(i,k)))
                  end if
               else ! abundance variable
                  xscale(i,k) = max(s% xa_scale, s% xa_pre_hydro(i-nvar_hydro,k))
               end if
            end do
         end do

         if (s% doing_timing) &
            call update_time(s, time0, total_all_before, s% time_newton_xscale)

         if (.false. .and. s% hydro_call_number == 3) then
            call dump_xscale
         end if
                  
         contains

         subroutine dump_xscale
            integer :: k, j, k0, k1
            include 'formats.dek'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'xscale ' // trim(s% nameofvar(j)), k, xscale(j, k)
               end do
               write(*,*)
            end do
            stop 'set_xscale'      
         end subroutine dump_xscale

      end subroutine set_xscale_info


      subroutine eval_equations(iter, nvar, nz, dx, xscale, equ, lrpar, rpar, lipar, ipar, ierr)
#ifdef DBLE
         use hydro_eqns_dble, only:eval_equ
#else
         use hydro_eqns_quad, only:eval_equ
#endif
         use star_utils, only: update_time, total_times
         use alert_lib, only:bug_alert
         integer, intent(in) :: iter, nvar, nz
         real(fltp), pointer, dimension(:,:) :: dx, xscale, equ ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

			integer :: cnt, i, j
         type (star_info), pointer :: s   
         logical :: skip_partials
         integer :: id
         real(dp) :: dt
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         
         logical, parameter :: check_bad_nums = .false.
         
         include 'formats.dek'
         
         ierr = 0

         id = ipar(ipar_id)
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return




         if (.false. .and. s% hydro_call_number == 3 .and. iter == 1) then
            call dump_eval_equ
         end if




         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if

         if (dbg) write(*, *) 'eval_equations: s% doing_numerical_jacobian', s% doing_numerical_jacobian
         
         dt = rpar(rpar_dt)
         
         if (ipar(ipar_first_call) /= 0) then
            ipar(ipar_first_call) = 0
         else
            if (dbg) write(*, *) 'call set_newton_vars'
            call set_newton_vars(s, iter, dx, xscale, dt, ierr)
            if (ierr /= 0) then
               if (s% report_ierr) write(*, *) 'eval_equations: set_newton_vars returned ierr', ierr
            end if
         end if
         
         if (ierr == 0) then
            equ = 0
            skip_partials = .true.
            if (dbg) write(*, *) 'call eval_equ'
            call eval_equ(s, nvar, dt, skip_partials, xscale, ierr)         
            if (ierr /= 0) then
               if (s% report_ierr) write(*, *) 'eval_equations: eval_equ returned ierr', ierr
            end if
         end if

         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_newton_eval_eqn)

         if (ierr /= 0 .or. .not. dbg .or. .not. check_bad_nums) return
         
         cnt = 0
         do i=1,nz
            do j=1, nvar
               if (is_bad(equ(j, i))) then
                  write(*,3) 'eval_equations: equ has a nan: ', j, i, equ(j, i)
                  cnt = cnt + 1
               end if
               if (abs(equ(j, i)) > 1d100) then
                  write(*,3) 'eval_equations: equ residual too big: ', j, i, equ(j, i)
                  cnt = cnt + 1
               end if
            end do
         end do
         if (cnt > 0) then
            ierr = -1
            call bug_alert(.true., ierr, 'eval_equations')
            return
         end if
         
         write(*,*) 'return from eval_equations'
         
         
         contains


         subroutine dump_eval_equ
            integer :: k, j, k0, k1
            include 'formats.dek'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'dx ' // trim(s% nameofvar(j)), k, dx(j, k)
               end do
               write(*,*)
            end do
            stop 'dump_eval_equ'      
         end subroutine dump_eval_equ

         
      end subroutine eval_equations


      subroutine sizequ( &
            iter, nvar, nz, equ, equ_norm, equ_max, lrpar, rpar, lipar, ipar, ierr)
         use star_utils, only: update_time, total_times
         integer, intent(in) :: iter, nvar, nz
         real(fltp), pointer :: equ(:,:) ! (nvar, nz)
         real(fltp), intent(out) :: equ_norm, equ_max
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         integer :: j, k, num_terms, n, i_chem1, nvar_hydro, nvar_chem, &
            j_max, k_max
         real(fltp) :: sumequ, absq
         type (star_info), pointer :: s 
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         
         include 'formats.dek'  

         if (dbg) write(*, *) 'enter sizequ'
         ierr = 0
         call get_star_ptr(ipar(ipar_id), s, ierr)
         if (ierr /= 0) return

         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         
         nvar_hydro = s% nvar_hydro
         nvar_chem = s% nvar_chem
         n = nz
         num_terms = 0
         sumequ = 0
         equ_max = 0
         j_max = 0
         k_max = 0
         if (s% do_struct_hydro .or. s% do_struct_thermo) then
            if (s% do_burn .or. s% do_mix) then
               num_terms = num_terms + nvar*nz
               do k = 1, nz
                  do j = 1, nvar
                     absq = abs(equ(j,k))
                     sumequ = sumequ + absq
                     if (absq > equ_max) then
                        equ_max = absq
                        j_max = j
                        k_max = k
                     end if
                  end do
               end do
            else
               num_terms = num_terms + n*nvar_hydro
               do k = 1, nz
                  do j = 1, nvar_hydro
                     absq = abs(equ(j,k))
                     !write(*,3) 'equ(j,k)', j, k, equ(j,k)
                     sumequ = sumequ + absq
                     if (absq > equ_max) then
                        equ_max = absq
                        j_max = j
                        k_max = k
                     end if
                  end do
               end do
            end if
         end if
         if (s% do_burn .or. s% do_mix) then
            i_chem1 = s% i_chem1
            num_terms = num_terms + nvar_chem*nz
            do k = 1, nz
               do j = i_chem1, nvar
                  absq = abs(equ(j,k))
                  sumequ = sumequ + absq
                  if (absq > equ_max) then
                     equ_max = absq
                     j_max = j
                     k_max = k
                  end if
               end do
            end do
         end if

         equ_norm = sumequ/num_terms
         
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_newton_size_equ)

         if (dbg) write(*,2) trim(s% nameofequ(j_max)) // ' equ_max', k_max, equ_max, equ_norm

         if (dbg) write(*, *) 'exit sizequ'

         if (.false. .and. equ_max > 1d4) then
            write(*,2) 'equ_norm', num_terms, equ_norm
            write(*,2) trim(s% nameofequ(j_max)) // ' equ_max', k_max, equ_max, equ_norm
            stop 'sizequ'
            call dump_equ
         end if
         
         
         contains


         subroutine dump_equ
            integer :: k, j, k0, k1
            include 'formats.dek'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'equ ' // trim(s% nameofequ(j)), k, equ(j, k)
               end do
               write(*,*)
            end do
            stop 'dump_equ'      
         end subroutine dump_equ


      end subroutine sizequ


      subroutine size_D_norm(s, iter, nvar, nz, B, xscale, D_norm, ierr)
         use star_utils, only: update_time, total_times
         type (star_info), pointer :: s
         integer, intent(in) :: iter, nvar, nz
         real(fltp), pointer, dimension(:,:) :: B, xscale ! (nvar, nz)
         real(fltp), intent(out) :: D_norm
         integer, intent(out) :: ierr
         
         integer :: i, j, k, max_zone, max_var, num_terms, nvar_hydro
         real(fltp) :: x, dx, sc, atol, rtol, alpha, max_sc, sum_sc
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         logical :: skip_var(nvar)
         
         include 'formats.dek'
         
         ierr = 0
         
         nvar_hydro = s% nvar_hydro
         atol = s% D_norm_atol
         rtol = s% D_norm_rtol
         alpha = s% D_norm_alpha
         
         skip_var(1:nvar) = .false.
         do i=1,nvar
            if ((i == s% i_FL .and. s% D_norm_skip_L) .or. &
                (i == s% i_vel .and. s% D_norm_skip_v) .or. &
                (i == s% i_lnddot .and. s% D_norm_skip_lnddot) .or. &
                (i == s% i_lnTdot .and. s% D_norm_skip_lnTdot)) skip_var(i) = .true.
         end do
         
         max_zone = 0
         max_var = 0
         num_terms = 0
         sum_sc = 0
         max_sc = 0
         do k=1,nz
            do i=1,nvar
               if (skip_var(i)) cycle
               if (i <= nvar_hydro) then
                  x = s% xh_pre_hydro(i,k)
               else
                  x = s% xa_pre_hydro(i-nvar_hydro,k)
               end if
               dx = B(i,k)*xscale(i,k)
               sc = abs(dx)/(atol + rtol*abs(x))
               if (sc > max_sc) then
                  max_sc = sc
                  max_zone = k
                  max_var = i
               end if
               if (alpha > 0) sum_sc = sum_sc + sc**alpha
               num_terms = num_terms+1
            end do
         end do
         
         if (alpha > 0) then
            D_norm = (sum_sc/num_terms)**(1d0/alpha)
         else
            D_norm = max_sc
         end if
         
         return
         write(*,2) 's% model_number', s% model_number
         write(*,2) 'iter', iter
         write(*,2) 'num_terms', num_terms
         write(*,2) 'nvar', nvar
         write(*,2) 'nz', nz
         write(*,1) 'D_norm', D_norm
         write(*,3) 'max_sc', max_var, max_zone, max_sc
         write(*,*)
         
      end subroutine size_D_norm


      subroutine sizeB( &
            iter, nvar, nz, x, B, xscale, max_correction, correction_norm, &
            lrpar, rpar, lipar, ipar, ierr)
         use star_utils, only: update_time, total_times
         integer, intent(in) :: iter, nvar, nz
         real(fltp), pointer, dimension(:,:) :: x, B, xscale ! (nvar, nz)
         real(fltp), intent(out) :: correction_norm ! a measure of the average correction
         real(fltp), intent(out) :: max_correction ! magnitude of the max correction
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         integer :: k, i, num_terms, j, max_zone, max_var, n, nvar_hydro, &
            skip1, skip2, skip3, skip4, jmax
         real(fltp) :: abs_corr, sum_corr, x_limit, max_abs_corr_for_k
         type (star_info), pointer :: s
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         
         logical, parameter :: check_for_bad_nums = .false.
         logical, parameter :: save_max_abs_corr_for_k = .true.
         
         include 'formats.dek'

         ierr = 0
         if (dbg) write(*, *) 'sizeB'

         call get_star_ptr(ipar(ipar_id), s, ierr)
         if (ierr /= 0) return
         
         if (s% D_norm_kappa > 0d0) then
            call size_D_norm(s, iter, nvar, nz, B, xscale, correction_norm, ierr)
            max_correction = 0
            return
         end if

         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if

         n = nz
         nvar_hydro = s% nvar_hydro
         
         if (s% include_L_in_error_est) then
            skip1 = 0
         else
            skip1 = s% i_FL
         end if
         
         if (s% include_v_in_error_est) then
            skip2 = 0
         else
            skip2 = s% i_vel
         end if
         
         if (s% include_lnddot_in_error_est) then
            skip3 = 0
         else
            skip3 = s% i_lnddot
         end if
         
         if (s% include_lnTdot_in_error_est) then
            skip4 = 0
         else
            skip4 = s% i_lnTdot
         end if

         max_zone = 0
         max_var = 0
         num_terms = 0
         sum_corr = 0
         max_correction = 0
         x_limit = s% correction_xa_limit
         do k = 1, nz
            max_abs_corr_for_k = 0
            
            if (s% do_struct_hydro .or. s% do_struct_thermo) then
               if (s% do_burn .or. s% do_mix) then
                  jmax = nvar
               else
                  jmax = nvar_hydro
               end if
               do j = 1, jmax
                  if (j == skip1 .or. &
                      j == skip2 .or. &
                      j == skip3 .or. &
                      j == skip4) cycle
                  if (check_for_bad_nums) then
                     if (is_bad(B(j,k))) then
                        ierr = -1
                        cycle
                     end if
                  end if
                  if (j > nvar_hydro) then
                     if (s% xa_pre_hydro(j-nvar_hydro,k) < x_limit) cycle
                  end if
                  num_terms = num_terms + 1
                  abs_corr = abs(B(j,k))
                  sum_corr = sum_corr + abs_corr
                  if (abs_corr > max_abs_corr_for_k) max_abs_corr_for_k = abs_corr
                  if (abs_corr > max_correction) then
                     max_correction = abs_corr
                     max_zone = k
                     max_var = j
                  end if
               end do
            else if (s% do_burn .or. s% do_mix) then
               do j = s% i_chem1, nvar
                  i = j - s% nvar_hydro
                  if (check_for_bad_nums) then
                     if (is_bad(B(j,k))) then
                        ierr = -1
                        cycle
                     end if
                  end if
                  ! recall that correction dx = B*xscale, so B is a relative correction
                  if (s% xa_pre_hydro(i,k) >= x_limit) then
                     abs_corr = abs(B(j,k))
                     if (abs_corr > max_abs_corr_for_k) max_abs_corr_for_k = abs_corr
                     if (abs_corr > max_correction) then
                        max_correction = abs_corr
                        max_zone = k
                        max_var = j
                     end if
                     sum_corr = sum_corr + abs_corr
                     num_terms = num_terms + 1
                  end if
               end do
            end if
            if (save_max_abs_corr_for_k .and. iter <= max_num_profile_extras .and. iter > 0) &
               s% profile_extra(k,iter) = max_abs_corr_for_k
         end do

         correction_norm = sum_corr/num_terms  !sqrt(sum_corr/num_terms)

         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_newton_size_B)

         if (.false. .and. s% hydro_call_number == 2 .and. iter == 1) then
            write(*,2) 'correction_norm', num_terms, correction_norm
            call dump_B
         end if
         
         if (s% hydro_show_correction_info) call show_stuff
         
         abs_corr = abs(max_correction)
         s% abs_max_corr2 = s% abs_max_corr1; s% abs_max_corr1 = abs_corr
         s% max_var2 = s% max_var1; s% max_var1 = max_var
         s% max_zone2 = s% max_zone1; s% max_zone1 = max_zone
         
         if (iter < 3) return
         ! check for flailing
         if ( &
             abs_corr > s% tol_max_correction .and. &
             abs_corr > s% abs_max_corr1 .and. s% abs_max_corr1 > s% abs_max_corr2 .and. &
             max_zone == s% max_zone1 .and. s% max_zone1 == s% max_zone2 .and. &
             max_var == s% max_var1 .and. s% max_var1 == s% max_var2) then
            if (s% hydro_show_correction_info) then
               write(*,*) 'give up because diverging'
            end if
            max_correction = 1d99
         end if
         
         
         return
         
         
         if (s% hydro_call_number < 2) return
         
         call show_stuff
         stop 'sizeB'
         
         contains
         
         
         subroutine show_stuff
            include 'formats.dek'
            if (iter == 1) then
               write(*,*)
               write(*,'(a15,2a7,2a8,99a20)') &
                  'corrections', 'model', 'iter', 'var', 'zone', &
                  'corr norm', 'max corr', 'xscale', 'max corr*xscale', &
                  'mass loc', 'log dt/yr'
            end if
            write(*,'(15x,2i7,a8,i8,4e20.10,99f20.10)') &
               s% model_number, iter, trim(s% nameofvar(max_var)), max_zone, &
               correction_norm, &
               B(max_var,max_zone), &
               xscale(max_var,max_zone), &
               xscale(max_var,max_zone)*B(max_var,max_zone), &
               s% m(max_zone)/Msun, log10(s% dt/secyer)
         end subroutine show_stuff


         subroutine dump_B
            integer :: k, j, k0, k1
            include 'formats.dek'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'B ' // trim(s% nameofequ(j)), k, B(j, k)
               end do
               write(*,*)
            end do
            stop 'dump_equ'      
         end subroutine dump_B

      end subroutine sizeB


      subroutine inspectB(iter, nvar, nz, dx, B, xscale, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: iter, nvar, nz
         real(fltp), pointer, dimension(:,:) :: dx, B, xscale ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         type (star_info), pointer :: s   
         integer :: id
         integer, parameter :: inspectB_iter_stop = -1
         
         id = ipar(ipar_id)
         
         if (dbg) write(*, *) 'inspectB', iter
         ierr = 0

         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         !if (s% hydro_call_number == 3 .and. iter == 1) call dumpB
         
         if (.not. s% hydro_inspectB_flag) return
         
         if (s% hydro_call_number /= s% hydro_dump_call_number) return
         
         call write_solve_logs(s, iter, dx, B, xscale, ierr)

         if (iter == inspectB_iter_stop) then
            stop 'debug: inspectB'
         end if
         
         
         contains


         subroutine dumpB
            integer :: k, j, k0, k1
            include 'formats.dek'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'B ' // trim(s% nameofvar(j)), k, B(j, k)
                  write(*,2) 'xscale ' // trim(s% nameofvar(j)), k, xscale(j, k)
                  write(*,2) 'dx ' // trim(s% nameofvar(j)), k, dx(j, k)
               end do
               write(*,*)
            end do
            stop 'dumpB'      
         end subroutine dumpB
         
      end subroutine inspectB
         
         
      subroutine write_solve_logs(s, iter, dx, B, xscale, ierr)
         use star_utils, only:std_write_internals_to_file
         use utils_lib, only:alloc_iounit, free_iounit, append_data
         type (star_info), pointer :: s   
         integer, intent(in) :: iter
         real(fltp), pointer, dimension(:,:) :: dx, B, xscale ! (nvar, nz)
         integer, intent(out) :: ierr

         integer :: i, j, iounit, i_delta, nvar, nz
         character (len=256) :: logfilename, log_format_str, data_dir, filename
         logical :: initialize_log_file
         real(dp), pointer :: tmp(:), tmp2(:)

         nz = s% nz
         if (s% do_burn .or. s% do_mix) then
            nvar = s% nvar
         else
            nvar = s% nvar_hydro
         end if
         
         !call std_write_internals_to_file(id, iter)

         data_dir = 'plot_data/solve_logs'

         if (iter == 1) then
            iounit = alloc_iounit(ierr); if (ierr /= 0) return
            write(filename, '(2a)') trim(data_dir), '/names.data'
            open(unit = iounit, file = trim(filename), iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'failed to open ', trim(filename)
               ierr = 0
               call free_iounit(iounit)
               return
            end if
            initialize_log_file = .true.
            write(*,*) 'allocate debug_previous_data'
            allocate(debug_previous_data(nz, nvar), stat=ierr)  
               ! we don't deallocate this, but it is only for debugging
               ! so isn't a serious memory leak
            if (ierr /= 0) then
               call free_iounit(iounit)
               return
            end if
         else
            initialize_log_file = .false.
         end if

         log_format_str = '(99999e20.12)'
         
         i_delta = 0
         
         allocate(tmp(nz), tmp2(nz), stat=ierr)
         if (ierr /= 0) then
            call free_iounit(iounit)
            return
         end if
         
         do j = 1, nvar
            tmp(1:nz) = B(j, 1:nz)
            call write_one(nz, tmp, 'corr_' // s% nameofvar(j))
         end do

         do j = 1, nvar
            tmp(1:nz) = B(j, 1:nz)
            call write_delta(nz, tmp, 'corr_' // s% nameofvar(j))
         end do

         do j=1, nvar
            tmp = dx(j,1:nz)
            call write_one(nz, tmp, 'd' // s% nameofvar(j))
         end do

         do j=1, nvar
            i = j - s% nvar_hydro
            if (i > 0) then
               tmp = s% xa_pre_hydro(i,1:nz) + dx(j,1:nz)
            else
               tmp = s% xh_pre_hydro(j,1:nz) + dx(j,1:nz)
            end if
            call write_one(nz, tmp, s% nameofvar(j))
         end do
                  
         call write_one(nz, s% mlt_cdc, 'mlt_cdc')
         call write_one(nz, s% gradT, 'gradT')

         deallocate(tmp, tmp2) ! for write_solve_logs

         if (iter == 1) then
            close(iounit)
            call free_iounit(iounit)
         else
            initialize_log_file = .false.
         end if
         
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         write(filename, '(2a)') trim(data_dir), '/size.data'
         open(unit = iounit, file = trim(filename), iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'failed to open ', trim(filename)
            ierr = 0
            return
         end if
         write(iounit, *) nz, iter
         close(iounit)
         call free_iounit(iounit)

         contains

         subroutine write_delta(nz, vals, name)
            integer, intent(in) :: nz
            real(dp), intent(in) :: vals(nz)
            character (len=*), intent(in) :: name
            real(dp) :: tmp(nz)
            i_delta = i_delta + 1
            if (iter == 1) then
               tmp = 0
            else if (i_delta <= nvar) then
               tmp = vals - debug_previous_data(:,i_delta)
            end if
            call write_one(nz, tmp, 'delta_' // name)          
            debug_previous_data(:,i_delta) = vals
         end subroutine write_delta

         subroutine write_one(nz, vals, varname)
            integer, intent(in) :: nz
            real(dp), intent(in) :: vals(nz)
            character (len=*), intent(in) :: varname
            write(logfilename, '(4a)') trim(data_dir), '/', trim(varname), '.log'
            call append_data(nz, vals(1:nz), logfilename, log_format_str, initialize_log_file, ierr)
            if (ierr /= 0) then
               write(*, *) 'failed in append_data for ', trim(logfilename)
               ierr = 0
               return
            end if
            if (iter == 1) write(iounit, *) trim(varname)
         end subroutine write_one

      end subroutine write_solve_logs
      

#ifdef DBLE
      end module hydro_newton_procs_dble
#else
      end module hydro_newton_procs_quad
#endif

