! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

#ifdef DBLE
      module hydro_mtx_dble
#else
      module hydro_mtx_quad
#endif
      
      use star_private_def
      use alert_lib, only: alert
      use utils_lib, only: is_bad_quad, is_bad_num, has_bad_num
      use num_lib, only: safe_log10
      use const_def
      
      use num_def
      
      implicit none

#ifdef DBLE
      integer, parameter :: fltp = dp
#define is_bad is_bad_num
#else
      integer, parameter :: fltp = qp
#define is_bad is_bad_quad
#endif
      
      logical, parameter :: dbg = .false. 
      
      integer, parameter :: ipar_id = 1
      integer, parameter :: ipar_first_call = 2
      integer, parameter :: hydro_lipar = ipar_first_call
      
      integer, parameter :: rpar_dt = 1
      integer, parameter :: hydro_lrpar = 1

      
      ! for inspectB debugging
      real(dp), pointer :: debug_previous_data(:,:)

      contains
      

      subroutine set_newton_vars(s, iter, dx, xscale, dt, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: iter
         real(fltp), dimension(:,:) :: dx, xscale
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         call set_vars_for_solver(s, 1, s% nz, iter, dx, xscale, dt, ierr)
      end subroutine set_newton_vars


      subroutine set_vars_for_solver(s, nzlo, nzhi, iter, dx, xscale, dt, ierr)
         use utils_lib, only: has_bad_num, is_bad
         use const_def, only: secyer, Msun, Lsun, Rsun
         use hydro_vars, only: set_hydro_vars, set_rmid_and_Amid
         use chem_def
         use star_utils
         type (star_info), pointer :: s
         integer, intent(in) :: nzlo, nzhi, iter
         real(fltp), dimension(:,:) :: dx, xscale
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr

         logical, parameter :: &
            skip_neu = .false., &
            skip_kap = .false., &
            skip_basic_vars = .true., &
            skip_grads = .true., &
            skip_rotation = .true., &
            skip_mixing_info = .true., &
            skip_other_cgrav = .true.
         integer :: i, j, k, kk, klo, khi, &
            i_xlnd, i_lnPgas, i_lnE, i_lnT, i_lnTdot, i_lnddot, i_lnR, i_FL, i_vel, &
            fe56, nvar, nvar_chem, species, i_chem1, nz, nvar_hydro
         real(dp), dimension(:, :), pointer :: xh_old, xa_old
         logical :: do_chem, do_struct, skip_net, try_again
         integer :: time0, clock_rate, op_err, kbad, &
            cnt, max_fixes, loc(2), k_lo, k_hi
         real(dp) :: total_all_before, twoGmrc2, r2, xavg, &
            dq_sum, max_xa_err_ratio, xa_err_norm, dt_inv
         
         include 'formats.dek'

         if (dbg) write(*, *) 'set_vars_for_solver'
         ierr = 0
         
         do_chem = (s% do_burn .or. s% do_mix)
         do_struct = (s% do_struct_hydro .or. s% do_struct_thermo)
         
         xh_old => s% xh_pre_hydro
         xa_old => s% xa_pre_hydro
         
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         i_lnT = s% i_lnT
         i_lnTdot = s% i_lnTdot
         i_lnddot = s% i_lnddot
         i_lnE = s% i_lnE
         i_lnR = s% i_lnR
         i_FL = s% i_FL
         i_vel = s% i_vel
         
         nvar_chem = s% nvar_chem
         species = s% species
         nvar_hydro = s% nvar_hydro
         i_chem1 = s% i_chem1
         
         fe56 = s% net_iso(ife56)
         if (fe56 /= 0) fe56 = i_chem1+fe56-1
         !write(*,2) 'pure_fe56_limit', fe56, s% pure_fe56_limit
         
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = nvar_hydro
         end if

         nz = s% nz
         if (dt == 0) then
            dt_inv = 0
         else 
            dt_inv = 1/dt
         end if
         s% dVARdot_dVAR = dt_inv
      
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
      
         if (do_chem) then
            do k=1,nz
               do j=1,species
                  s% xa(j,k) = xa_old(j,k) + dx(j+nvar_hydro,k)
               end do
            end do
            max_fixes = 0 !5
            do cnt=1,max_fixes
               loc = minloc(s% xa(1:species,1:nz))
               j = loc(1)
               k = loc(2)
               !write(*,4) 'minloc', j, k, cnt, s% xa(j,k)
               if (s% xa(j,k) >= 1d-3*s% min_xa_hard_limit) exit ! too good to fix
               if (s% xa(j,k) < s% min_xa_hard_limit) then
                  if (s% report_ierr) then
                     khi = nz
                     do kk=k+1,nz
                        if (s% xa(j,kk) < s% min_xa_hard_limit) cycle
                        khi = kk-1; exit
                     end do
                     klo = 1
                     do kk=k-1,1,-1
                        if (s% xa(j,kk) < s% min_xa_hard_limit) cycle
                        klo = kk+1; exit
                     end do
                     do k=klo,khi
                        write(*,2) &
                           'negative ' // trim(chem_isos% name(s% chem_id(j))), &
                           k, s% xa(j,k), xa_old(j,k), dx(nvar_hydro+j,k), s% m(k)/Msun
                     end do
                  end if
                  ierr = -1
                  return
                  exit ! too bad to fix
               end if
               if (k == 1) then  
                  k_lo = 1; k_hi = 2
               else if (k == nz) then
                  k_lo = nz-1; k_hi = nz
               else if (s% sig(k) > s% sig(k+1)) then
                  k_lo = k-1; k_hi = k
               else if (s% sig(k+1) > 0) then
                  k_lo = k; k_hi = k+1
               else
                  exit
               end if
               try_again = .true.
               do while (try_again .and. sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi)) < 0)
                  try_again = .false.
                  if (k_lo > 1) then
                     if (s% sig(k_lo) > 0) then
                        k_lo = k_lo-1
                        try_again = .true.
                     end if
                  end if
                  if (k_hi < nz) then
                     if (s% sig(k_hi+1) > 0) then
                        k_hi = k_hi+1
                        try_again = .true.
                     end if
                  end if
               end do
               !write(*,3) 'no extend', k_lo, k_hi
               if (.not. try_again) exit
               dq_sum = sum(s% dq(k_lo:k_hi))
               if (s% report_ierr) then
                  write(*,5) 'fix xa(j,k_lo:k_hi)', j, k_lo, k, k_hi, s% xa(j,k), &
                     sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi))/dq_sum, dq_sum
                  !stop
                end if
               do j=1,species
                  xavg = sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi))/dq_sum
                  do kk=k_lo,k_hi
                     s% xa(j,kk) = xavg
                  end do
               end do
            end do
         end if
         
         kbad = 0
!$OMP PARALLEL DO PRIVATE(k,op_err)
         do k=1,nz
            op_err = 0
            call set1(k,.false.,op_err)
            if (op_err /= 0) then
               kbad = k; ierr = op_err
            end if
         end do
!$OMP END PARALLEL DO
         if (ierr /= 0) then
            if (s% report_ierr) then
               write(*,'(60x,a8,99a20)') 'zone', 'mass loc', 'value', 'prev'
               do k=1,nz ! report the errors sequentially
                  call set1(k,.true.,op_err)
               end do
               write(*,4) 'set_vars_for_solver failed: model, k, nz', s% model_number, kbad, nz
            end if
            
            !stop
            
            
            return
         end if

         do k=1,nz
            if (is_bad_num(s% L(k))) then
               if (s% report_ierr) write(*,2) 'set_vars_for_solver L', k, s% L(k), &
                  xh_old(i_FL,k) + dx(i_FL,k), &
                  xh_old(i_FL,k), dx(i_FL,k)
               ierr = -1
               return               
               stop
            end if
         end do
         !write(*,*) 'all okay at end of set_vars_for_solver'

         if (do_struct .and. s% k_below_recently_added > 1 .and. dt /= 0) then
            do k=1, s% k_below_recently_added-1
               call set_d_dt_for_recently_added(k)
            end do
         end if
         
         if (ierr /= 0) then
            if (s% report_ierr) then
            
               do k=1,nz
                  if (abs(1d0 - sum(s% xa(:,k))) > 1d-3) then
                     write(*,2) 'set_vars_for_solver: bad xa sum', k, &
                        sum(s% xa(:,k)), sum(xa_old(:,k)), sum(dx(i_chem1:nvar,k))
                     write(*,*) '                                                  xa, xa_old+dx, xa_old, dx'
                     do j=1,species
                        write(*,2) trim(chem_isos% name(s% chem_id(j))), k, &
                           s% xa(j,k), xa_old(j,k) + dx(i_chem1-1+j,k), &
                           xa_old(j,k), dx(i_chem1-1+j,k)
                     end do
                     
                     exit
                     
                  end if
               end do
               write(*,*)
               
            end if
            return
         end if
         
         if (do_struct) then
            if ( & !(.not. s% rotation_flag) .and. &
                (.not. s% doing_numerical_jacobian)) then
               if (s% lnPgas_flag) then
                  call edit_lnR_for_lnPgas(s, xh_old, dx, dt)
               else
                  call edit_lnR(s, xh_old, dx, dt)
               end if
            end if
!$OMP PARALLEL DO PRIVATE(k,twoGmrc2,r2)
            do k=1,nz
               s% r(k) = exp(s% lnR(k))
               r2 = s% r(k)**2
               s% area(k) = pi4*r2
               s% grav(k) = s% cgrav(k)*s% m(k)/r2
               if (s% use_gr_factors) then ! GR gravity factor = 1/sqrt(1-2Gm/(rc^2))
                  twoGmrc2 = 2*s% cgrav(k)*s% m(k)/(s% r(k)*clight**2)
                  s% grav(k) = s% grav(k)/sqrt(1d0 - twoGmrc2)
               end if
               if (s% rotation_flag) then ! note: do NOT change j_rot
                  s% i_rot(k) = (2d0/3d0)*r2 ! thin spherical shell
               end if
            end do    
!$OMP END PARALLEL DO
         end if

         s% total_angular_momentum = total_angular_momentum(s)
         !write(*,2) 'set_vars_for_solver tot J', s% model_number, s% total_angular_momentum

         if (dbg) write(*, *) 'call set_hydro_vars'
         skip_net = (s% operator_coupling_choice == -1)
         call set_hydro_vars( &
            s, 1, nz, skip_basic_vars, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_other_cgrav, skip_mixing_info, dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'set_hydro_vars returned ierr', ierr
            return
         end if

         call set_rmid_and_Amid(s, 1, nz, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'set_rmid_and_Amid returned ierr', ierr
            return
         end if
      
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_set_newton_vars)

         
         contains
         
         
         subroutine set1(k,report,ierr)
            use chem_def, only: chem_isos
            use star_utils, only: FL_to_L
            integer, intent(in) :: k
            logical, intent(in) :: report
            integer, intent(out) :: ierr
            real(dp) :: x(nvar)
            ! setting x = xh_old + dx is necessary because of numerical issues.
            ! we want to ensure that we've calculated the variables using exactly the
            ! same values for x as will be returned as the final result.
            real(dp) :: twoGmrc2, r2, sum_xa, xa_clip_limit
            integer :: j, i
            
            include 'formats.dek'
            ierr = 0
            
            xa_clip_limit = s% xa_clip_limit
            
            if (do_struct) then
               x(1:nvar_hydro) = xh_old(1:nvar_hydro,k) + dx(1:nvar_hydro,k)
               if (i_xlnd /= 0) then
                  s% lnd(k) = x(i_xlnd) - lnd_offset
                  s% rho(k) = exp(s% lnd(k))
               end if
               if (i_lnPgas /= 0) then
                  s% lnPgas(k) = x(i_lnPgas)
                  s% Pgas(k) = exp(s% lnPgas(k))
               end if
               if (i_lnE /= 0) s% lnE_var(k) = x(i_lnE)
               s% lnT(k) = x(i_lnT)
               s% T(k) = exp(s% lnT(k))
               s% lnR(k) = x(i_lnR)
               s% L(k) = FL_to_L(s,x(i_FL))
               if (i_vel /= 0) s% v(k) = x(i_vel)
               if (i_lnTdot /= 0) s% lnTdot(k) = x(i_lnTdot)
               if (i_lnddot /= 0) s% lnddot(k) = x(i_lnddot)

               ! set time derivatives
               if (dt == 0) then
                  s% dlnd_dT(k) = 0
                  s% dlnPgas_dT(k) = 0
                  s% dlnT_dt(k) = 0
                  s% dlnR_dt(k) = 0
                  if (i_lnE /= 0) s% dlnE_var_dt(k) = 0
                  if (i_vel /= 0) s% dv_dt(k) = 0
               else if (k >= s% k_below_recently_added) then
                  ! use dx where possible in order to get better accuracy
                  if (i_xlnd /= 0) then
                     if (s% lnd_for_d_dt(k) == xh_old(i_xlnd,k) - lnd_offset) then
                        s% dlnd_dt(k) = dx(i_xlnd,k)*dt_inv
                     else
                        s% dlnd_dt(k) = ((x(i_xlnd) - lnd_offset) - s% lnd_for_d_dt(k))*dt_inv
                     end if
                  end if
                  if (i_lnPgas /= 0) then
                     if (s% lnPgas_for_d_dt(k) == xh_old(i_lnPgas,k)) then
                        s% dlnPgas_dt(k) = dx(i_lnPgas,k)*dt_inv
                     else
                        s% dlnPgas_dt(k) = (x(i_lnPgas) - s% lnPgas_for_d_dt(k))*dt_inv
                     end if
                  end if
                  if (i_lnE /= 0) then
                     if (s% lnE_var_for_d_dt(k) == xh_old(i_lnE,k)) then
                        s% dlnE_var_dt(k) = dx(i_lnE,k)*dt_inv
                     else
                        s% dlnE_var_dt(k) = (x(i_lnE) - s% lnE_var_for_d_dt(k))*dt_inv
                     end if
                  end if
                  if (s% lnT_for_d_dt(k) == xh_old(i_lnT,k)) then
                     s% dlnT_dt(k) = dx(i_lnT,k)*dt_inv
                  else
                     s% dlnT_dt(k) = (x(i_lnT) - s% lnT_for_d_dt(k))*dt_inv
                  end if
                  if (s% lnR_for_d_dt(k) == xh_old(i_lnR,k)) then
                     s% dlnR_dt(k) = dx(i_lnR,k)*dt_inv
                  else
                     s% dlnR_dt(k) = (x(i_lnR) - s% lnR_for_d_dt(k))*dt_inv
                  end if
                  if (i_vel /= 0) then
                     if (s% v_for_d_dt(k) == xh_old(i_vel,k)) then
                        s% dv_dt(k) = dx(i_vel,k)*dt_inv
                     else
                        s% dv_dt(k) = (x(i_vel) - s% v_for_d_dt(k))*dt_inv
                     end if
                  end if
               end if
            end if
            
            if (do_chem) then
               if (s% min_xa_hard_limit > -1d50) then ! .and. .not. s% just_did_backup) then
                  do j=1,species
                     if (s% xa(j,k) < s% min_xa_hard_limit) then
                        if (report) then
                           write(*,'(a60,i8,f20.10,1p,99e20.10)') &
                              'bad negative ' // &
                              trim(chem_isos% name(s% chem_id(j))), &
                              k, s% m(k)/Msun, s% xa(j,k), s% xa_pre_hydro(j,k)
                           return
                        end if
                        ierr = -1
                        s% why_Tlim = Tlim_neg_X
                        if (.not. report) return
                     else if (report .and. s% xa(j,k) < -1d-6) then
                        write(*,'(a60,i8,f20.10,1p,99e20.10)') &
                           'negative ' // &
                           trim(chem_isos% name(s% chem_id(j))), &
                           k, s% m(k)/Msun, s% xa(j,k)
                     end if
                  end do
               else if (s% report_ierr) then
                  do j=1,species
                     if (s% xa(j,k) < -1d-3) then
                        write(*,'(a60,i8,f20.10,1p,99e20.10)') &
                           'negative ' // &
                           trim(chem_isos% name(s% chem_id(j))), &
                           k, s% m(k)/Msun, s% xa(j,k)!, s% D_mix(k), s% D_mix(min(nz,k+1))
                     end if
                  end do
               end if

               do j=1,species
                  s% xa(j,k) = max(0d0, min(1d0, s% xa(j,k)))
               end do

               sum_xa = sum(s% xa(1:nvar_chem,k))
               if (is_bad_num(sum_xa)) then
                  if (report) then
                     write(*,'(a60,i8,99f20.10)') 'bad num sum X', k, s% m(k)/Msun, sum_xa
                  end if
                  ierr = -1
                  s% why_Tlim = Tlim_bad_Xsum
                  if (.not. report) return
               end if
               if (abs(sum_xa - 1d0) > s% sum_xa_tolerance) then
                  if (report) &
                     write(*,'(a60,i8,99f20.10)') &
                        'bad sum X', k, s% m(k)/Msun, &
                        sum_xa, sum(xa_old(1:species,k)), &
                        sum_xa - sum(xa_old(1:species,k))
                  ierr = -1
                  s% why_Tlim = Tlim_bad_Xsum
                  
                  return

                  do j=1,nvar_chem
                     write(*,'(a20,99e20.10)') &
                        trim(chem_isos% name(s% chem_id(j))), &
                        s% xa(j,k) - s% xa_pre_hydro(j,k), &
                        s% xa(j,k), s% xa_pre_hydro(j,k)
                  end do
                  stop 'set1'
               end if

               if (abs(sum_xa - 1d0) > 1d-12) then
                  do j=1,nvar_chem
                     s% xa(j,k) = s% xa(j,k)/sum_xa
                  end do
               end if
               
               if (xa_clip_limit > 0) then
                  do j=1,nvar_chem
                     if (s% xa(j,k) < xa_clip_limit) s% xa(j,k) = 0d0
                  end do
               end if
               
            end if
         
         end subroutine set1
         
         
         subroutine set_d_dt_for_recently_added(k) ! k < k_below_recently_added
            ! uses values done by set1 for k-1 to k+1
            integer, intent(in) :: k
            real(dp) :: mdot_div_xmstar, q, d_dt_const_q, dlnR_dq, dlnE_dq, &
               domega_dq, dvel_dq, dlnd_dq, dlnT_dq, dqsum, dlnPgas_dq
            include 'formats.dek'
            mdot_div_xmstar = s% mstar_dot / s% xmstar
            
            ! do values defined at cell boundaries (lnR and vel)
            q = s% q(k)
            if (k < nz .and. k > 1) then
               dlnR_dq = (s% lnR(k-1) - s% lnR(k+1))/(s% dq(k-1) + s% dq(k))
            else
               dlnR_dq = 0
            end if
            d_dt_const_q = (s% lnR(k) - s% lnR_for_d_dt(k))/dt
            s% dlnR_dt(k) = d_dt_const_q - dlnR_dq*q*mdot_div_xmstar
            if (s% v_flag) then
               if (k < nz .and. k > 1) then
                  dvel_dq = (s% v(k-1) - s% v(k+1))/(s% dq(k-1) + s% dq(k))
               else
                  dvel_dq = 0
               end if
               d_dt_const_q = (s% v(k) - s% v_for_d_dt(k))/dt
               s% dv_dt(k) = d_dt_const_q - dvel_dq*q*mdot_div_xmstar
            end if
            
            ! do values defined at cell centers (lnd, lnPgas, lnE, and lnT)
            q = q - s% dq(k)
            if (k < nz .and. k > 1) then
               dqsum = s% dq(k) + 0.5d0*(s% dq(k-1) + s% dq(k+1))
               dlnd_dq = (s% lnd(k-1) - s% lnd(k+1))/dqsum
               dlnPgas_dq = (s% lnPgas(k-1) - s% lnPgas(k+1))/dqsum
               dlnE_dq = (s% lnE(k-1) - s% lnE(k+1))/dqsum
               dlnT_dq = (s% lnT(k-1) - s% lnT(k+1))/dqsum
            else
               dlnd_dq = 0
               dlnPgas_dq = 0
               dlnE_dq = 0
               dlnT_dq = 0
            end if
            
            if (s% lnPgas_flag) then
               d_dt_const_q = (s% lnPgas(k) - s% lnPgas_for_d_dt(k))/dt
               s% dlnPgas_dt(k) = d_dt_const_q - dlnPgas_dq*q*mdot_div_xmstar
            else
               d_dt_const_q = (s% lnd(k) - s% lnd_for_d_dt(k))/dt
               s% dlnd_dt(k) = d_dt_const_q - dlnd_dq*q*mdot_div_xmstar
            end if
            
            d_dt_const_q = (s% lnT(k) - s% lnT_for_d_dt(k))/dt
            s% dlnT_dt(k) = d_dt_const_q - dlnT_dq*q*mdot_div_xmstar

            if (s% lnE_flag) then
               d_dt_const_q = (s% lnE_var(k) - s% lnE_var_for_d_dt(k))/dt
               s% dlnE_var_dt(k) = d_dt_const_q - dlnE_dq*q*mdot_div_xmstar
            end if

         end subroutine set_d_dt_for_recently_added
         
         
      end subroutine set_vars_for_solver



         
      subroutine edit_lnR(s, xh_old, dx, dt) ! uses mass and density to set radius
         type (star_info), pointer :: s
         real(dp), dimension(:, :) :: xh_old
         real(fltp), dimension(:, :) :: dx
         real(dp), intent(in) :: dt
         real(dp) :: vol00, volp1, cell_vol, dlnR_dq, d_dt_const_q
         integer :: k, nz
         vol00 = (4*pi/3)*s% R_center**3
         nz = s% nz
         do k=nz, 1, -1
            volp1 = vol00
            cell_vol = s% dm(k)/s% rho(k)
            vol00 = volp1 + cell_vol
            s% lnR(k) = log(vol00/(4*pi/3))/3
            dx(s% i_lnR,k) = s% lnR(k) - xh_old(s% i_lnR,k)
            if (dt > 0 .and. k >= s% k_below_recently_added) &
               s% dlnR_dt(k) = (s% lnR(k) - s% lnR_for_d_dt(k))/dt
         end do
         if (dt > 0 .and. s% k_below_recently_added > 2) then
            do k = 2, s% k_below_recently_added - 1
               if (k < nz .and. k > 1) then
                  dlnR_dq = (s% lnR(k-1) - s% lnR(k+1))/(s% dq(k-1) + s% dq(k))
               else
                  dlnR_dq = 0
               end if
               d_dt_const_q = (s% lnR(k) - s% lnR_for_d_dt(k))/dt
               s% dlnR_dt(k) = d_dt_const_q - dlnR_dq*s% q(k)*s% mstar_dot/s% xmstar
            end do
         end if
      end subroutine edit_lnR

      
      subroutine edit_lnR_for_lnPgas(s, xh_old, dx, dt) 
         type (star_info), pointer :: s
         ! cannot use density for this since it hasn't been calculated yet
         real(dp), dimension(:, :) :: xh_old
         real(fltp), dimension(:, :) :: dx
         real(dp), intent(in) :: dt
         integer :: k, kk, k_outer, k_inner, cnt, nz
         real(dp) :: dq_total, dq_sum, r3_outer, r3_inner, r3, alfa, beta
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         cnt = 0
         nz = s% nz
         do k=1, nz-1
            if (s% lnR(k) > s% lnR(k+1)) cycle
            ! find k_outer s.t. lnR(k+1) < lnR(k_outer)
            if (dbg) write(*,*)
            if (dbg) write(*,2) 'bad s% lnR(k)', k, s% lnR(k)
            if (dbg) write(*,2) 'bad s% lnR(k+1)', k+1, s% lnR(k+1)
            cnt = cnt+1
            k_outer = 1
            do kk = k,1,-1
               if (s% lnR(k+1) < s% lnR(kk)) then
                  k_outer = kk; exit
               end if
            end do
            ! find k_inner s.t. lnR(k_inner) < lnR(k)
            k_inner = nz
            do kk = k+1, nz-1
               if (s% lnR(kk) < s% lnR(k)) then
                  k_inner = kk; exit
               end if
            end do
            dq_total = sum(s% dq(k_outer:k_inner-1))
            dq_sum = 0
            r3_outer = exp(3*s% lnR(k_outer))
            r3_inner = exp(3*s% lnR(k_inner))
            if (dbg) write(*,2) 's% lnR(k_outer)', k_outer, s% lnR(k_outer)
            do kk=k_outer+1, k_inner-1
               dq_sum = dq_sum + s% dq(kk-1)
               alfa = dq_sum/dq_total
               beta = 1 - alfa
               r3 = alfa*r3_inner + beta*r3_outer
               s% lnR(kk) = log(r3)/3
               if (dbg) write(*,2) 'new s% lnR(kk)', kk, s% lnR(kk)
               dx(s% i_lnR,kk) = s% lnR(kk) - xh_old(s% i_lnR,kk)
               if (dt > 0 .and. kk >= s% k_below_recently_added) &
                  s% dlnR_dt(kk) = (s% lnR(kk) - s% lnR_for_d_dt(kk))/dt
            end do
            if (dbg) write(*,2) 's% lnR(k_inner)', k_inner, s% lnR(k_inner)
         end do
         if (dbg .and. cnt > 0) stop 'edit_lnR_for_lnPgas'
      end subroutine edit_lnR_for_lnPgas
            

      subroutine enter_setmatrix( &
            iter, nvar_in, nz, neqns, dx, dx_init, xscale, xder, need_solver_to_eval_jacobian, &
            ldA, A, idiag, lrpar, rpar, lipar, ipar, ierr)
         use mtx_lib, only: write_3point_jacobian_info
         use mtx_def, only: lapack
         use utils_lib, only: set_pointer_4, set_quad_pointer_4
         use star_utils, only: update_time, total_times
         integer, intent(in) :: iter, nvar_in, nz, neqns ! (neqns = nvar*nz)
         real(fltp), pointer, dimension(:,:) :: dx, dx_init, xscale, xder ! (nvar, nz)
         logical, intent(out) :: need_solver_to_eval_jacobian
         integer, intent(in) :: ldA ! leading dimension of A
         real(fltp), pointer, dimension(:,:) :: A ! (ldA, neqns)
         integer, intent(inout) :: idiag 
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         type (star_info), pointer :: s                  
         integer :: i, j, k, cnt, nnz, nzlo, nzhi
         logical, parameter :: save_jac_plot_data = .false.  
         real(dp), parameter :: epsder_struct = 1d-7
         real(dp), parameter :: epsder_chem = 1d-7
         real(dp) :: dt
         integer :: id, nvar, nvar_hydro
         logical :: dbg_enter_setmatrix, do_chem
         real(fltp), pointer :: blk3(:, :, :, :)
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         
         include 'formats.dek'
         
         id = ipar(ipar_id)
         dbg_enter_setmatrix = dbg
         
         !write(*,*) 'enter_setmatrix'
         !stop

         ierr = 0   
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            ierr = -1
            return
         end if

         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if

         if (dbg_enter_setmatrix) write(*, '(/,/,/,/,/,/,a)') 'enter_setmatrix'
         
         if (s% model_number == 1) then
            s% num_jacobians = s% num_jacobians + 1
            if (s% num_jacobians > 60 .and. mod(s% num_jacobians,10) == 0) &
               write(*,*) 'first model is slow to converge: num tries', s% num_jacobians
         end if
         
         dt = rpar(rpar_dt)
         
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         
         if (nvar /= nvar_in) then
            write(*,*) 'enter_setmatrix: bad value for nvar_in'
            ierr = -1
            return
         end if

         if (s% numerical_jacobian) then
            write(*,*) 'enter_setmatrix: s% numerical_jacobian', s% numerical_jacobian
            
            if (.false.) then
               write(*,*) 'call eval_equ_for_numerical_jacobian'
               call eval_equ_for_numerical_jacobian(s, dt, xscale, ierr)
               if (ierr /= 0) stop 1
               stop 'after eval_equ_for_numerical_jacobian'
            end if
            
            need_solver_to_eval_jacobian = .true.
            s% doing_numerical_jacobian = .true.
            nvar_hydro = s% nvar_hydro
            do j=1, nvar_hydro
               do k=1,nz
                  xder(j,k) = epsder_struct* &
                     max(1d0, abs(s% xh_pre_hydro(j,k)+dx(j,k)), abs(s% xh_pre_hydro(j,k)+dx_init(j,k)))
               end do
            end do
            if (nvar > nvar_hydro) then
               do j=nvar_hydro+1, nvar
                  do k=1,nz
                     xder(j,k) = epsder_chem* &
                        max(1d0, abs(s% xa_pre_hydro(j-nvar_hydro,k)+dx(j,k)), &
                                          abs(s% xa_pre_hydro(j-nvar_hydro,k)+dx_init(j,k)))
                  end do
               end do
            end if
            call check_xder_lnR
            write(*,*) 'done enter_setmatrix'
            return
         end if

         s% idiag = idiag
#ifdef DBLE
         s% jacobian => A
#endif
                  
         if (s% hydro_matrix_type == block_tridiag_dble_matrix_type .or. &
             s% hydro_matrix_type == block_tridiag_quad_matrix_type) then
!$OMP PARALLEL DO PRIVATE(i,j)
            do i=1,neqns
               do j=1,lda
                  A(j,i) = 0
               end do
            end do
!$OMP END PARALLEL DO
#ifdef DBLE
            call set_pointer_4(blk3, A, nvar, nvar, nz, 3)
            s% ublk_dble => blk3(:,:,:,1)
            s% dblk_dble => blk3(:,:,:,2)
            s% lblk_dble => blk3(:,:,:,3)
#else
            call set_quad_pointer_4(blk3, A, nvar, nvar, nz, 3)
            s% ublk_quad => blk3(:,:,:,1)
            s% dblk_quad => blk3(:,:,:,2)
            s% lblk_quad => blk3(:,:,:,3)
#endif
         end if

         !s% doing_numerical_jacobian = .false.
         
         if (dbg_enter_setmatrix) &
            write(*, *) 'call eval_partials with doing_numerical_jacobian = .false.'
         call eval_partials(s, dt, xscale, ierr)
         if (ierr /= 0) return
         
         if (s% jacobian_clip_limit > 0) call clip_mtx(s% jacobian_clip_limit)

         if (dbg_enter_setmatrix) write(*, *) 'finished enter_setmatrix'
         need_solver_to_eval_jacobian = .false.

         if (s% doing_timing) &
            call update_time(s, time0, total_all_before, s% time_newton_enter_setmatrix)

         if (.not. save_jac_plot_data) return

#ifdef DBLE
         write(*, *) 'save jacobian plot data'
         nzlo = s% hydro_jacobian_nzlo
         nzhi = s% hydro_jacobian_nzhi
         if (nzhi < nzlo) nzhi = nz
         call write_3point_jacobian_info( &
               ldA, nvar*nz, A, idiag, nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
               xscale, s% nameofvar, s% nameofequ, 'plot_data/jacobian_data')
         write(*,*) 'enter_setmatrix: hit CR to continue'
         read(*,*)
#else
         write(*,*) 'enter_setmatrix -- save_jac_plot_data not supported for quad'
         stop 1
#endif
         
         contains
         
         
         subroutine clip_mtx(clip_limit)
            real(dp), intent(in) :: clip_limit
            include 'formats.dek'
            cnt = 0
            nnz = 0
            if (s% hydro_matrix_type == block_tridiag_dble_matrix_type .or. &
                s% hydro_matrix_type == block_tridiag_quad_matrix_type) then
               do k=1,nz
                  do j=1,nvar
                     do i=1,nvar
                        if (k < nz .and. s% ublk_dble(i,j,k) /= 0) then
                           nnz = nnz+1
                           if (abs(s% ublk_dble(i,j,k)) < clip_limit) then
                              cnt = cnt + 1
                              s% ublk_dble(i,j,k) = 0
                           end if
                        end if
                        if (s% dblk_dble(i,j,k) /= 0) then
                           nnz = nnz+1
                           if (abs(s% dblk_dble(i,j,k)) < clip_limit) then
                              cnt = cnt + 1
                              s% dblk_dble(i,j,k) = 0
                           end if
                        end if
                        if (k > 1 .and. s% lblk_dble(i,j,k) /= 0) then
                           nnz = nnz+1
                           if (abs(s% lblk_dble(i,j,k)) < clip_limit) then
                              cnt = cnt + 1
                              s% lblk_dble(i,j,k) = 0
                           end if
                        end if
                     end do
                  end do
               end do
            else
               do i=1,neqns
                  do j=idiag-(2*nvar-1),idiag+(2*nvar-1)
                     if (A(j,i) /= 0) then
                        nnz = nnz+1
                        if (abs(A(j,i)) < clip_limit) then
                           cnt = cnt + 1
                           A(j,i) = 0
                        end if
                     end if
                  end do
               end do
            end if

         end subroutine clip_mtx
         
         
         subroutine write_block_data(ierr)
            use mtx_lib, only: mtx_write_block_tridiagonal
            integer, intent(out) :: ierr
            integer :: iounit, k, i, j, nvar, nz
            character (len=64) :: fname_out
            include 'formats.dek'
            if (s% nvar > 99) then
               write(fname_out,'(a,i3,a)') 'block_tri_', s% nvar, '.data'
            else
               write(fname_out,'(a,i2,a)') 'block_tri_', s% nvar, '.data'
            end if
            nvar = s% nvar
            nz = s% nz
            write(*,'(a,2i6)') 'save test matrix info to ' // trim(fname_out), nvar, nz
            iounit = 33
            ierr = 0
            open(iounit,file=trim(fname_out),iostat=ierr)
            if (ierr /= 0) return
            call mtx_write_block_tridiagonal(iounit,nvar,nz,s% lblk_dble,s% dblk_dble,s% ublk_dble,ierr)
            close(iounit)
            if (ierr /= 0) return
            write(*,'(a,2i6)') 'done saving test matrix info to ' // trim(fname_out), nvar, nz
         end subroutine write_block_data
         
         
         subroutine write_1block_data
            use const_def
            integer :: k, i, j, klo, khi, nvar_lo, nvar_hi
            character (len=64) :: fname_out
            include 'formats.dek'
            klo=1265 ! !s% nz
            khi=1267 ! !s% nz
            nvar_lo = 1
            nvar_hi = s% nvar
            do i = nvar_lo, nvar_hi
               write(*,'(a14)',advance='no') trim(s% nameofvar(i))
               if (i <= s% nvar_hydro) then
                  do k = klo, khi
                     write(*,'(e14.4)',advance='no') s% xh(i,k)
                  end do
               else
                  do k = klo, khi
                     write(*,'(e14.4)',advance='no') s% xa(i-s% nvar_hydro,k)
                  end do
               end if
               write(*,*)
            end do
            do k=klo,khi
               write(*,2) 'k', k, s% sig(k)
               write(*,'(a14)',advance='no') ''
               do i=nvar_lo, nvar_hi
                  write(*,'(a14)',advance='no') trim(s% nameofvar(i))
               end do
               write(*,*)
               do i=nvar_lo, nvar_hi
                  write(*,'(a14)',advance='no') trim(s% nameofequ(i))
                  do j=nvar_lo, nvar_hi
                     write(*,'(e14.4)',advance='no') s% dblk_dble(i,j,k)
                  end do
                  write(*,*)
               end do
            end do
            stop 'write_1block_data'
         end subroutine write_1block_data
         
         
         subroutine check_xder_lnR
            integer :: i_lnR, k
            real(dp) :: lnR00, lnRm1, lnRp1, dlnR_prev, ddx, ddx_limit
            include 'formats.dek'
            i_lnR = s% i_lnR
            do k=1,nz
               ddx = xder(i_lnR,k)
               lnR00 = s% xh_pre_hydro(i_lnR,k) + dx(i_lnR,k)
               if (ddx > 0) then ! compare lnR00 to lnRm1
                  if (k == 1) cycle
                  lnRm1 = s% xh_pre_hydro(i_lnR,k-1) + dx(i_lnR,k-1)
                  ddx_limit = 0.5d0*(lnRm1 - lnR00)
                  if (ddx > ddx_limit) then
                     !if (dbg_enter_setmatrix) write(*,2) 'check_xder_lnR: change ddx for lnR', k, ddx, ddx_limit
                     xder(i_lnR,k) = ddx_limit
                  end if
               else ! compare lnR00 to lnRp1
                  if (k == nz) exit
                  lnRp1 = s% xh_pre_hydro(i_lnR,k+1) + dx(i_lnR,k+1)
                  ddx_limit = 0.5d0*(lnRp1 - lnR00) ! ddx and ddx_limit both < 0
                  if (ddx < ddx_limit) then
                     !if (dbg_enter_setmatrix) write(*,2) 'check_xder_lnR: change ddx for lnR', k, ddx, ddx_limit
                     xder(i_lnR,k) = ddx_limit
                  end if
               end if
            end do            
         end subroutine check_xder_lnR
         
            
      end subroutine enter_setmatrix
      
      
      subroutine eval_equ_for_numerical_jacobian(s, dt, xscale, ierr)
#ifdef DBLE
         use hydro_eqns_dble, only:eval_equ
#else
         use hydro_eqns_quad, only:eval_equ
#endif
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         real(fltp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr
         
         logical :: do_chem
         integer :: nvar
         logical, parameter :: skip_partials = .true.
         ierr = 0
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         call eval_equ(s, nvar, dt, skip_partials, xscale, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'eval_equ_for_numerical_jacobian: eval_equ returned ierr', ierr
            return
         end if
         
      end subroutine eval_equ_for_numerical_jacobian
      
      
      subroutine eval_partials(s, dt, xscale, ierr)
#ifdef DBLE
         use hydro_eqns_dble, only: eval_equ
#else
         use hydro_eqns_quad, only: eval_equ
#endif
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         real(fltp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr
         
         logical :: do_chem
         integer :: nvar
         logical, parameter :: skip_partials = .false.
         ierr = 0
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         call eval_equ(s, nvar, dt, skip_partials, xscale, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'eval_partials: eval_equ returned ierr', ierr
            return
         end if
         
      end subroutine eval_partials
      

      subroutine exit_setmatrix( &
            iter, nvar_in, nz, neqns, dx, ldA, A, idiag, xscale, lrpar, rpar, lipar, ipar, ierr)
         use mtx_lib, only:write_3point_jacobian_info
         use const_def, only: Rsun
         integer, intent(in) :: iter, nvar_in, nz, neqns
         real(fltp), pointer :: dx(:,:) ! (nvar, nz)
         integer, intent(in) :: ldA ! leading dimension of A
         real(fltp), pointer, dimension(:,:) :: A ! (ldA, neqs)
         integer, intent(inout) :: idiag ! row of A with the matrix diagonal entries
         real(fltp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         type (star_info), pointer :: s
         integer :: nvar, id, nvar_hydro, equP, equT, equR, equL, equv, j, i, k, equ_k, var_k
         real(fltp), pointer :: save_A1(:,:), save_A2(:,:)
         logical :: save_numjac_plot_data, test_analytical_jacobian, dbg_exit, do_chem 
         real(dp) :: dt
         
         include 'formats.dek'
         
         dbg_exit = dbg         
         id = ipar(ipar_id)
         
         ierr = 0
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            ierr = -1
            return
         end if
         
         if (dbg_exit) then
            write(*, *) 'exit_setmatrix'
            write(*, 2) 'nvar', nvar
            write(*, 2) 'species', s% species
            write(*, 2) 'nvar_hydro', s% nvar_hydro
            write(*, 2) 'nz', nz
            write(*, 2) 'neqns', neqns
            write(*, 2) 'idiag', idiag
            write(*, 2) 'ldA', ldA
            write(*, *) 
         end if
         
         dt = rpar(rpar_dt)
         
         do_chem = (s% do_burn .or. s% do_mix)
         if (.false. .or. do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         
         if (nvar /= nvar_in .or. nz /= s% nz .or. neqns /= nvar*nz) then
            write(*,*) 'nvar', nvar
            write(*,*) 'nvar_in', nvar_in
            write(*,*) 'nz', nz
            write(*,*) 's% nz', s% nz
            ierr = -1
            return
         end if

         s% doing_numerical_jacobian = .false.

         if (.not. s% numerical_jacobian) return
         
#ifdef DBLE
         ! numerical jacobian is only used for debugging the analytical one
         
         if ((s% hydro_call_number /= s% hydro_dump_call_number) &
            .and. (s% hydro_dump_call_number > 0)) return
         
         save_numjac_plot_data = s% hydro_save_numjac_plot_data        
         
         test_analytical_jacobian = save_numjac_plot_data &
                  .or. (s% hydro_dump_call_number > 0)
         
         nvar_hydro = s% nvar_hydro
         
         equP = s% equP
         equT = s% equT
         equR = s% equR
         equL = s% equL
         equv = s% equv

         if (.not. test_analytical_jacobian) then
            if (save_numjac_plot_data) call write_numeric(nvar)
            return
         end if
         
         s% idiag = idiag
         s% jacobian => A
         
         allocate(save_A1(ldA,neqns),save_A2(ldA,neqns),stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed in exit_setmatrix'
            return
         end if
         
         ! save_A1 gets the numerical jacobian
         do j=1,neqns
            do i=1,ldA
               save_A1(i,j) = A(i,j)
               A(i,j) = 0
            end do
         end do
         
         write(*,*) 'exit_setmatrix: evaluate analytic partials' 
         
         !k = 28
         !write(*,2) 'r(k)/Rsun', k, s% r(k)/Rsun
         !write(*,*)
         
         ! evaluate analytic partials
         !write(*,2) 'lnr(k)', 304, s% lnr(304)
         write(*,*) 'exit_setmatrix: call set_newton_vars'
         call set_newton_vars(s, iter, dx, xscale, dt, ierr)
         !write(*,2) 'lnr(k)', 304, s% lnr(304)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'exit_setmatrix: set_newton_vars returned ierr', ierr
            return
         end if
         
         !write(*,*)
         !write(*,2) 'r(k)/Rsun', k, s% r(k)/Rsun
         !write(*,*)
         !write(*,*)
         !write(*,*)

         write(*,*) 'exit_setmatrix: call eval_partials'
         call eval_partials(s, dt, xscale, ierr)
         if (ierr /= 0) return
         write(*,*) 'exit_setmatrix: done eval_partials'

         if (save_numjac_plot_data) then
            call write_jacobians(s,nvar,save_A1,save_A2)
         end if
         
         ! restore numeric jacobian
         do j=1,neqns
            do i=1,ldA
               A(i,j) = save_A1(i,j)
            end do
         end do

         deallocate(save_A1,save_A2)
#else
         write(*,*) 'exit_setmatrix: no numerical_jacobian for quad'
         stop 1
#endif

         
         contains
         
         
         subroutine get_names(nvar, nmvar, nmequ)
            use chem_def
            integer, intent(in) :: nvar
            character*32, dimension(nvar) :: nmvar, nmequ
            integer :: i
            integer, pointer :: chem_id(:)
            nmvar(1:nvar_hydro) = s% nameofvar(1:nvar_hydro)
            nmequ(1:nvar_hydro) = s% nameofequ(1:nvar_hydro)
            if (nvar == nvar_hydro) return
            chem_id => s% chem_id
            do i=nvar_hydro+1,nvar
               write(nmvar(i),'(a)') 'var_' // trim(chem_isos% name(chem_id(i-nvar_hydro)))
               write(nmequ(i),'(a)') 'equ_' // trim(chem_isos% name(chem_id(i-nvar_hydro)))
            end do
         end subroutine get_names
            
            
#ifdef DBLE         
         subroutine write_numeric(nvar)
            integer, intent(in) :: nvar
            character*32, dimension(nvar) :: nmvar, nmequ
            real(dp) :: rcond
            call get_names(nvar, nmvar, nmequ)
            write(*, *) 'save numerical jacobian plot data'
            call write_3point_jacobian_info( &
                  ldA, neqns, A, idiag, nvar, nz, 1, nz, 1, nvar, 1, nvar, &
                  xscale, nmvar, nmequ, 'plot_data/numerical_jacobian')
            stop 'debug: write_numeric'
         end subroutine write_numeric
#endif
            
            
#ifdef DBLE         
         subroutine write_jacobians(s,nvar,save_A1,save_A2)
            use star_utils, only: std_write_internals_to_file
            type (star_info), pointer :: s
            integer, intent(in) :: nvar
            real(dp), pointer :: save_A1(:,:), save_A2(:,:)
            real(dp) :: rcond
            character*32, dimension(nvar) :: nmvar, nmequ
            
            integer :: i, j, nzlo, nzhi
            
            call get_names(nvar, nmvar, nmequ)

            nzlo = s% hydro_jacobian_nzlo
            nzhi = s% hydro_jacobian_nzhi
            if (nzhi < nzlo) nzhi = nz
            
            write(*, *) 'save analytic jacobian plot data'
            call write_3point_jacobian_info( &
                  ldA, neqns, A, idiag, nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
                  xscale, nmvar, nmequ, 'plot_data/jacobian_data')
            !write(*, *) 'done write_3point_jacobian_info'
               
            ! restore numeric jacobian
            do j=1,neqns
               do i=1,ldA
                  A(i,j) = save_A1(i,j)
               end do
            end do

            write(*, *) 'save numerical jacobian plot data'
            call write_3point_jacobian_info( &
                  ldA, neqns, A, idiag, nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
                  xscale, nmvar, nmequ, 'plot_data/numerical_jacobian')
         
            deallocate(save_A1, save_A2)
            
            call std_write_internals_to_file(s% id, 0)
         
            stop 'done write_jacobians: exit_setmatrix'
            
         end subroutine write_jacobians
#endif
         
         
      end subroutine exit_setmatrix


      subroutine copy_one_to_3point_jacobian( &
            ldA, neqns, nvar, nz, nzlo, nzhi, A, idiag, eqn, em1, e00, ep1, xscale)
         integer, intent(in) :: ldA ! leading dimension of A
         real(dp), intent(out) :: A(ldA, neqns) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         integer, intent(in) :: idiag, neqns, nvar, nz, nzlo, nzhi, eqn
         real(dp), dimension(nvar, nvar, nz), intent(in) :: em1, e00, ep1         
         real(dp), intent(in) :: xscale(nvar, nz)         
         integer :: var, k, dk, ii, jj         
         do dk = -1, 1 ! 3 point stencil
            do var = 1, nvar
               !write(*,*) 'dk, var', dk, var
               ii = eqn - var - nvar*dk + idiag
               jj = var + nvar*(dk-1)
               select case(dk)
                  case(-1) 
                     do k=max(2,nzlo),nzhi
                        A(ii,jj+nvar*k) = em1(eqn, var, k)*xscale(var, k-1)
                     end do
                  case(0) 
                     do k=nzlo,nzhi
                        A(ii,jj+nvar*k) = e00(eqn, var, k)*xscale(var, k)
                     end do
                  case(1) 
                     do k=nzlo,min(nz-1,nzhi)
                        A(ii,jj+nvar*k) = ep1(eqn, var, k)*xscale(var, k+1)
                     end do
               end select
            end do
         end do
      end subroutine copy_one_to_3point_jacobian

      
      subroutine copy_all_from_3point_jacobian( &
               ldA, neqns, nvar, nz, nzlo, nzhi, A, idiag, em1, e00, ep1, xscale)
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz, nzlo, nzhi
         real(dp), intent(in) :: A(ldA, nvar*nz) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         integer, intent(in) :: idiag
         real(dp), dimension(nvar, nvar, nz), intent(out) :: em1, e00, ep1         
         real(dp), intent(in) :: xscale(nvar, nz)
         integer :: eqn
         em1 = 0; e00 = 0; ep1 = 0
         do eqn=1, nvar
            call copy_one_from_3point_jacobian( &
               ldA, neqns, nvar, nz, 1, nz, A, idiag, eqn, em1, e00, ep1, xscale)
         end do
      end subroutine copy_all_from_3point_jacobian

      
      subroutine copy_one_from_3point_jacobian( &
               ldA, neqns, nvar, nz, nzlo, nzhi, A, idiag, eqn, em1, e00, ep1, xscale)
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz, nzlo, nzhi
         integer, intent(in) :: idiag, eqn
         real(dp), intent(in) :: A(ldA, nvar*nz) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         real(dp), dimension(nvar, nvar, nz), intent(out) :: em1, e00, ep1         
         ! em1(i, j, k) is partial of equation i of cell k wrt variable j of cell k-1
         ! e00(i, j, k) is partial of equation i of cell k wrt variable j of cell k
         ! ep1(i, j, k) is partial of equation i of cell k wrt variable j of cell k+1
         real(dp), intent(in) :: xscale(nvar, nz)
         integer :: var, k, dk, ii, jj
         ! move partials from A to em1, e00, and ep1
         do dk = -1, 1 ! 3 point stencil
            do var = 1, nvar
               ii = eqn - var - nvar*dk + idiag
               jj = var + nvar*(dk-1)
               select case(dk)
                  case(-1) 
                     do k=max(2,nzlo),nzhi
                        em1(eqn, var, k) = A(ii, jj+nvar*k)/xscale(var, k-1)
                     end do
                  case(0) 
                     do k=nzlo,nzhi
                        e00(eqn, var, k) = A(ii, jj+nvar*k)/xscale(var, k)
                     end do
                  case(1) 
                     do k=nzlo,min(nz-1,nzhi)
                        ep1(eqn, var, k) = A(ii, jj+nvar*k)/xscale(var, k+1)
                     end do
               end select
            end do
         end do
      end subroutine copy_one_from_3point_jacobian
      
      
      subroutine store1_in_3point_jacobian(ldA, neqns, nvar, nz, A, idiag, eqn, var, k, dk, value, xscale)
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz, idiag, var, eqn, k, dk
         real(dp), intent(out) :: A(ldA, nvar*nz) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         real(dp), intent(in) :: value, xscale(nvar, nz)
         integer :: ii, jj
         ii = eqn - var - nvar*dk + idiag
         jj = var + nvar*(dk-1)
         select case(dk)
            case(-1) 
               A(ii, jj+nvar*k) = value*xscale(var, k-1)
            case(0) 
               A(ii, jj+nvar*k) = value*xscale(var, k)
            case(1) 
               A(ii, jj+nvar*k) = value*xscale(var, k+1)
         end select
      end subroutine store1_in_3point_jacobian
         

      subroutine show_jacobian_block(s, equ_k, var_k, ldA, neqns, nvar, nz, A, idiag, xscale)
         ! show J(i,j) for i in equations for equ_k and j in variables for var_k
         type (star_info), pointer :: s
         integer, intent(in) :: equ_k ! k for equations
         integer, intent(in) :: var_k ! k for vars
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz, idiag
         real(dp), intent(in) :: A(ldA, neqns) ! the jacobian matrix in banded diagonal form
         real(dp), intent(in) :: xscale(nvar, nz)
         
         integer :: i, j, nvar_hydro, equ, var
         real(dp) :: mtx(nvar,nvar)
         logical :: all_zeros, row_all_zeros(nvar), col_all_zeros(nvar)
         
         call get_one_from_3point_jacobian( &
            equ_k, var_k, ldA, neqns, nvar, nz, A, idiag, mtx, xscale)
            
         nvar_hydro = s% nvar_hydro
         
         write(*,*)
         write(*,*)
         write(*,'(12x)', advance='no')
         do var = 1, nvar_hydro
            write(*, '(a24)', advance='no') trim(s% nameofvar(var))
         end do
         write(*,*)
         do equ = 1, nvar_hydro
            write(*, '(a12)', advance='no') trim(s% nameofequ(equ))
            do var = 1, nvar_hydro
               write(*, '(e24.10)', advance='no') mtx(equ,var)
            end do
            write(*,*)
         end do
         write(*,*)
         write(*,*)
         write(*,*) 'equ_k', equ_k
         write(*,*) 'var_k', var_k
         write(*,*)
         write(*,*)
         
         !stop 'debug: show_jacobian_block'
         
         
         
 1       format(i5, 1pe26.16)
         write(*, *)
         do i=1, nvar
            write(*, '(a15, i5)') 'column', i
            all_zeros = .true.
            col_all_zeros(i) = .true.
            do j=1, nvar
               if (mtx(j, i) == 0) then
                  write(*, '(i5, i15)') j, 0
               else
                  write(*, 1) j, mtx(j, i)
                  all_zeros = .false.
                  col_all_zeros(i) = .false.
               end if
            end do
            if (all_zeros) write(*, *) &
                        '****************************** ALL ZEROS ******************************'
            write(*, *)
         end do
         
         write(*, *)
         do i=1, nvar
            write(*, '(a15, i5)') 'row', i
            all_zeros = .true.
            row_all_zeros(i) = .true.
            do j=1, nvar
               if (mtx(i, j) == 0) then
                  write(*, '(i5, i15)') j, 0
               else
                  write(*, 1) j, mtx(i, j)
                  all_zeros = .false.
                  row_all_zeros(i) = .false.
               end if
            end do
            if (all_zeros) write(*, *)  &
                        '****************************** ALL ZEROS ******************************'
            write(*, *)
         end do
         
         write(*, *)
         do i=1, nvar
            if (col_all_zeros(i)) write(*, *) 'all zeros in column', i
            if (row_all_zeros(i)) write(*, *) 'all zeros in row', i
         end do
         write(*, *)


      end subroutine show_jacobian_block

      
      subroutine get_one_from_3point_jacobian( &
               equ_k, var_k, ldA, neqns, nvar, nz, A, idiag, mtx, xscale)
         integer, intent(in) :: equ_k ! k for equations
         integer, intent(in) :: var_k ! k for vars
         integer, intent(in) :: ldA ! leading dimension of A
         integer, intent(in) :: neqns, nvar, nz
         integer, intent(in) :: idiag
         real(dp), intent(in) :: A(ldA, nvar*nz) ! the jacobian matrix
         ! A(idiag+q-v, v) = partial of equation(q) wrt variable(v)
         real(dp), dimension(nvar, nvar), intent(out) :: mtx
         real(dp), intent(in) :: xscale(nvar, nz)         
         integer :: dk, ii, jj, eqn, var       
         ! move partials from A to mtx         
         dk = var_k - equ_k
         if (abs(dk) > 1) then ! BUG
            write(*,*) 'get_one_from_3point_jacobian: bad dk', dk
            mtx = 0
            return
         end if
         do eqn = 1, nvar
            do var = 1, nvar
               ii = eqn - var - nvar*dk + idiag
               jj = var + nvar*(dk-1)
               mtx(eqn, var) = A(ii, jj+nvar*equ_k)!/xscale(var, var_k)
            end do
         end do
      end subroutine get_one_from_3point_jacobian
      

#ifdef DBLE
      end module hydro_mtx_dble
#else
      end module hydro_mtx_quad
#endif

