! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


#ifdef DBLE
      module hydro_eqns_dble
#else
      module hydro_eqns_quad
#endif

      use star_private_def
      use alert_lib
      use const_def
      use utils_lib, only: is_bad_quad, is_bad_num

      implicit none

#ifdef DBLE
      integer, parameter :: fltp = dp
#define is_bad is_bad_num
#else
      integer, parameter :: fltp = qp
#define is_bad is_bad_quad
#endif

      real(fltp), parameter :: one = 1, zero = 0
      
      logical, parameter :: dbg = .false.   
      
      integer, parameter :: dbg_cell = -1
      

      contains


      subroutine eval_equ(s, nvar, dt, skip_partials, xscale, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: nvar
         real(dp), intent(in) :: dt ! time since start of current timestep
         logical, intent(in) :: skip_partials
         real(fltp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr
         call eval_equ_for_solver(s, nvar, 1, s% nz, dt, skip_partials, xscale, ierr)
      end subroutine eval_equ
      

      subroutine eval_equ_for_solver( &
            s, nvar, nzlo, nzhi, dt, skip_partials, xscale, ierr)
         use chem_def
         use mesh_functions
         use star_utils, only: update_time, total_times

         type (star_info), pointer :: s         
         integer, intent(in) :: nvar, nzlo, nzhi
         real(dp), intent(in) :: dt ! time since start of current timestep
         logical, intent(in) :: skip_partials
         real(fltp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr

         integer :: &
            equP, equT, equR, equL, equv, equlnE, equlnTdot, equlnddot, equchem1, &
            k, j, nvar_hydro, nvar_chem, nz, op_err, matrix_type
         integer :: &
            i_xlnd, i_lnPgas, i_lnR, i_lnE, i_lnTdot, i_lnddot, i_lnT, i_FL, i_vel, &
            i_chem1, i_xh1, i_xhe4, kmax_equ(nvar), species
         real(dp) :: max_equ(nvar), total_all_before
         real(dp) :: dVARdot_dVAR, L_phot_old, dot_factor
         real(dp), dimension(:), pointer :: &
            L, lnR, lnP, lnT, dL_dm, dlnP_dm, dlnT_dm, lnE, lnE_var, lnTdot, lnddot
         real(fltp), dimension(:, :), pointer :: equ
         logical :: v_flag, dump_for_debug, do_artificial_viscosity, do_chem, do_mix
         integer :: time0, clock_rate
         
         include 'formats.dek'
         
         ierr = 0
         
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if

         dump_for_debug = .false. !(s% hydro_call_number == 3)
         do_artificial_viscosity = (s% v_flag .and. s% use_artificial_viscosity)
         
         call unpack
         
         dot_factor = max(1d0, dt)
         
         if (dbg) write(*, *) 'eqns', dt
         
         do_mix = s% do_mix
         do_chem = (do_mix .or. s% do_burn)
         
         if (.not. (s% do_struct_hydro .or. s% do_struct_thermo)) then 
         
            if (.not. do_chem) then
               write(*,*) 'bug: all false: do_chem, do_struct_hydro, and do_struct_thermo'
               stop 'eval_equ_for_solver'
            end if
            ! hold structure constant while solve burn and/or mix
            do j=1,nvar_hydro
               call null_eqn(j)
            end do
            if (ierr == 0) &
               call do_chem_eqns(s, xscale, nvar, equchem1, nvar_chem, skip_partials, dt, equ, ierr)
               
         else ! solving structure equations
               
            call s% other_energy_implicit(s% id, ierr)
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'eval_equ_for_solver: ierr from other_energy_implicit'
               return
            end if

!$OMP PARALLEL DO PRIVATE(op_err,k)
            do k = nzlo, nzhi
               if (.not. skip_partials .and. &
                     matrix_type == block_tridiag_dble_matrix_type) then
#ifdef DBLE
                  s% dblk_dble(:,:,k) = 0
                  s% ublk_dble(:,:,k) = 0
                  s% lblk_dble(:,:,k) = 0
#else
                  s% dblk_quad(:,:,k) = 0
                  s% ublk_quad(:,:,k) = 0
                  s% lblk_quad(:,:,k) = 0
#endif
               end if
               op_err = 0
               if (s% do_struct_hydro) then
                  call do1_density_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err            
                  if (k > 1) call do1_pressure_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err 
                  if (equv /= 0) call do1_velocity_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err 
                  if (equlnE /= 0) call do1_lnE_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err 
                  if (equlnTdot /= 0) call do1_lnTdot_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err 
                  if (equlnddot /= 0) call do1_lnddot_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err 
               end if
               if (s% do_struct_thermo) then
                  call do1_luminosity_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err            
                  if (k > 1) call do1_temperature_eqn(k,op_err)
                  if (op_err /= 0) ierr = op_err
               end if                                
               if (do_chem) then
                  call do1_chem_eqns( &
                     s, xscale, k, nvar, equchem1, nvar_chem, skip_partials, dt, equ, op_err)
                  if (op_err /= 0) ierr = op_err
               end if
            end do      
!$OMP END PARALLEL DO

            if (ierr == 0 .and. nzlo == 1 .and. &
                  (s% do_struct_hydro .or. s% do_struct_thermo)) then
               if (dbg) write(*,*) 'call PT_eqns_surf'
               call PT_eqns_surf(s, xscale, equ, skip_partials, nvar, ierr)
               if (dbg) write(*,*) 'done PT_eqns_surf'
            end if

            if (.not. s% do_struct_hydro) then
               call dummy_eqn(equR,i_lnR,nzlo,nzhi) 
               call dummy_eqn(equP,i_xlnd,max(2,nzlo),nzhi)
               if (equv /= 0) call dummy_eqn(equv,i_vel,nzlo,nzhi)
            end if
            
            if (.not. s% do_struct_thermo) then
               call dummy_eqn(equL,i_FL,nzlo,nzhi)
               call dummy_eqn(equT,i_lnT,max(2,nzlo),nzhi)
            end if
            
         end if

         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'eval_equ_for_solver: ierr after eval equations'
            return
         end if

         if (s% hydro_check_everything .and. &
               s% hydro_call_number == s% hydro_dump_call_number) then ! check everything
            call check_everything
         end if
         
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_eval_eqns)
         
         if (dump_for_debug .and. .not. skip_partials) call dump_equ_and_partials
         
         return
         if (s% model_number /= 121) return
         
         !call dump_equL_info ! TESTING
         !call dump_equ ! TESTING
         
         
         contains
         
         
         subroutine unpack
            
            nz = s% nz
            species = s% species
            nvar_hydro = s% nvar_hydro
            matrix_type = s% hydro_matrix_type
            
            lnT => s% lnT
            lnR => s% lnR
            L => s% L
            lnP => s% lnP
            lnTdot => s% lnTdot
            lnddot => s% lnddot
            lnE => s% lnE
            lnE_var => s% lnE_var
            
#ifdef DBLE
            equ => s% equ_dble
#else
            equ => s% equ_quad
#endif
            
            dVARdot_dVAR = s% dVARdot_dVAR
            
            equP = s% equP
            equT = s% equT
            equR = s% equR
            equL = s% equL
            equv = s% equv
            equlnE = s% equlnE
            equlnTdot = s% equlnTdot
            equlnddot = s% equlnddot
            equchem1 = s% equchem1
            
            i_xlnd = s% i_xlnd
            i_lnPgas = s% i_lnPgas
            i_lnT = s% i_lnT
            i_lnTdot = s% i_lnTdot
            i_lnddot = s% i_lnddot
            i_lnE = s% i_lnE
            i_lnR = s% i_lnR
            i_FL = s% i_FL
            i_vel = s% i_vel
            i_chem1 = s% i_chem1
            i_xh1 = i_chem1-1+s% net_iso(ih1)
            i_xhe4 = i_chem1-1+s% net_iso(ihe4)
            
            L_phot_old = s% L_phot_old
            v_flag = s% v_flag
            
            nvar_chem = s% nvar_chem
            
            dL_dm => s% dL_dm
            dlnP_dm => s% dlnP_dm
            dlnT_dm => s% dlnT_dm
            
         end subroutine unpack
         
         
         subroutine null_eqn(j)
            integer, intent(in) :: j
            integer :: k
            do k=nzlo,nzhi
               equ(j,k) = 0 ! s% xs(j,k) - s% xs_pre_pass(j,k)
               if (.not. skip_partials) call e00(s, xscale,j,j,k,nvar,one)
            end do
         end subroutine null_eqn
         
               
         subroutine do1_density_eqn(k,ierr) ! mass-volume-density relation
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            real(fltp) :: rp13, dm, rho, dr3, dequ_ddr3, ddr3_dlnd, &
               ddr3_dlnPgas_const_T, ddr3_dlnT_const_Pgas
            ierr = 0
               
            include 'formats.dek'
      
            dm = s% dm(k)
            rho = s% rho(k)
            ! dm/rho is cell volume
            dr3 = (dm/rho)/(pi4/3)
         
            if (k < nz) then
               rp13 = s% r(k+1)**3
            else
               rp13 = s% R_center**3
            end if
            ! dm = (pi4/3)*(r(k)**3 - rp13)*rho
            ! r(k)**3 = rp13 + (dm/rho)/(pi4/3) = rp13 + dr3            
            equ(equR, k) = lnR(k) - log(rp13 + dr3)/3
         
            if (skip_partials) return

            call e00(s, xscale, equR, i_lnR, k, nvar, one)
            if (k < nz) call ep1(s, xscale, equR, i_lnR, k, nvar, -rp13/(rp13 + dr3))

            dequ_ddr3 = -one_third/(rp13 + dr3)
            ddr3_dlnd = -dr3
            
            if (s% lnPgas_flag) then
               ddr3_dlnPgas_const_T = ddr3_dlnd*s% dlnRho_dlnPgas_const_T(k)
               call e00(s, xscale, equR, i_lnPgas, k, nvar, dequ_ddr3*ddr3_dlnPgas_const_T)
               ddr3_dlnT_const_Pgas = ddr3_dlnd*s% dlnRho_dlnT_const_Pgas(k)
               call e00(s, xscale, equR, i_lnT, k, nvar, dequ_ddr3*ddr3_dlnT_const_Pgas)
            else
               call e00(s, xscale, equR, i_xlnd, k, nvar, dequ_ddr3*ddr3_dlnd)
            end if
            
         end subroutine do1_density_eqn


         subroutine do1_luminosity_eqn(k,ierr) ! luminosity gradient (energy conservation)
            use rates_def
            use net_lib, only: net_get, get_reaction_id_table_ptr, get_net_reaction_table_ptr
            use star_utils, only: FL_to_L, L_to_dL_dFL, L_to_FL, L_to_dFL_dL
            use hydro_vars, only: eval_eps_grav_and_partials

            integer, intent(in) :: k
            integer, intent(out) :: ierr
         
            integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id


            real(dp) :: L00, Lp1, Lx
            real(fltp) :: dA_dlnR, &
               dm_max, dm, L_scale, dLdm, partial, FL00, &
               d_dLdM_dlndm1, d_dLdM_dlnTm1, &
               d_dLdM_dlnd00, d_dLdM_dlnT00, &
               d_dLdM_dlndp1, d_dLdM_dlnTp1, &
               d_dLdM_dlnR00, d_dLdM_dlnRp1, d_dLdM_dL00, &
               diff, diff_old, &
               eps_burn, d_eps_burn_dlnd, d_eps_burn_dlnT, &
               dL_expected, dL_actual, eps_nuc, Flx, dLx, d_dLx_epsnuc, &
               d_del_dlnR00, d_del_dlnRp1, d_del_dvel00, d_del_dvelp1, &
               d_dLAV_dlnd, d_dLAV_dlnT, d_dLAV_dlnR00, d_dLAV_dlnRp1, &
               d_dLAV_dvel00, d_dLAV_dvelp1, dLAV, del, qmid, eps_extra, &
               d_eps_burn_dlnPgas_const_T, d_eps_burn_dlnT_const_Pgas, &
               d_dLdM_dlnPgasm1_const_T, d_dLdM_dlnPgas00_const_T, d_dLdM_dlnPgasp1_const_T, &
               d_dLdM_dlnTm1_const_Pgas, d_dLdM_dlnT00_const_Pgas, d_dLdM_dlnTp1_const_Pgas, &
               d_dLAV_dlnT_const_Pgas, d_dLAV_dlnPgas_const_T, &
               d_dLdM_dlnEm1, d_dLdM_dlnE00, d_dLdM_dlnEp1, d_dLdM_dv00, d_dLdM_dvp1, &
               d_dLdM_dlnTdot, d_dLdM_dlnddot, dL00_dFL00, dLp1_dFLp1, &
               dFLx_dLx, dLx_dLp1, dFLx_dLp1, dFLx_dlnTdot, dFLx_dlnddot, &
               dFLx_dlnT00_const_Pgas, dFLx_dlnTm1_const_Pgas, dFLx_dlnTp1_const_Pgas, &
               dFLx_dlnT00, dFLx_dlnTm1, dFLx_dlnTp1, dFLx_dlnd00, dFLx_dx, &
               dFLx_dlnPgas00_const_T, dFLx_dlnR00, dFLx_dlnE00, dFLx_dv00, dFLx_dL00, &
               dFLx_dlnRp1, dFLx_dlnEp1, dFLx_dvp1, dFLx_dlnPgasp1_const_T, &
               dFLx_dlndp1, dFLx_dlnEm1, dFLx_dlnPgasm1_const_T, dFLx_dlndm1, dFLx_dFLp1, &
               other_equ, other_f



            integer :: j
               
            include 'formats.dek'
            ierr = 0
            
            call eval_eps_grav_and_partials(s, k, dt, ierr) ! get eps_grav info
            if (ierr /= 0) return

            dm = s% dm(k)
            
            eps_nuc = s% eps_nuc(k)
            eps_burn = eps_nuc - s% non_nuc_neu(k) + s% extra_heat(k) + s% irradiation_heat(k)
                     
            dLdm = s% eps_grav(k) + eps_burn ! optional artificial viscosity term treated below
            dL_expected = dm*dLdm
         
            if (do_artificial_viscosity) then
               if (k < nz) then
                  del = s% v(k)*(3*s% Amid(k) - s% area(k)) - s% v(k+1)*(3*s% Amid(k) - s% area(k+1))
               else
                  del = s% v(k)*(3*s% Amid(k) - s% area(k)) ! V_center = 0
               end if
               dLAV = 0.5d0*s% Pvisc(k)*del
               dL_expected = dL_expected + dLAV
               dLdm = dL_expected/dm
            else
               del = 0
            end if
         
            s% dL_dm(k) = dLdm
         
            L00 = L(k)
            if (k < nz) then
               Lp1 = L(k+1)
            else
               Lp1 = s% L_center
            end if
            dL_actual = L00 - Lp1
            
            dLx = dLdm*dm ! dL expected
            Lx = Lp1 + dLx ! Lx = L00 expected = Lp1 + dLdm*dm
            
            FL00 = one*L_to_FL(s,L00) ! transformed L00
            FLx = one*L_to_FL(s,Lx) ! transformed Lx
            
            if (FLx /= Lx) then
               L_scale = 1
            else if (.not. s% doing_first_model_of_run) then
               L_scale = max(1d-3*Lsun, abs(FL_to_L(s,s% xh_pre_hydro(i_FL,k))))
            else
               L_scale = Lsun
            endif
            
            !L_scale = s% energy_pre_hydro(k)*dm/max(1d-9,dt)
            ! this gave convergence problems in test_suite/wd_cool
   
            ! energy conservation (luminosity equation)
            equ(equL, k) = FLx/L_scale - FL00/L_scale
      
            if (is_bad(equ(equL, k))) then
               ierr = -1
               return
            end if

            if (.false. .and. k==s% nz .and. s% hydro_call_number == 126) then
               write(*,2) 'equ(equL,k) lnd lnT egrav eburn', k, equ(equL,k), s% lnd(k), s% lnT(k), &
                  s% eps_grav(k), eps_burn, s% d_eps_grav_dlnd00(k), s% d_eps_grav_dlnT00(k)
            end if
            
            other_f = 1.7656775000060900D+01
            other_equ = 1.5072941697668270D+02
            
            if (.false. .and. k==1059 .and. abs(s% lnT(k) - other_f) > 1d-12) then
            !if (.false. .and. k==1059) then
               
               write(*,*) 'start numeric differences'
               write(*,2) 'equL', equL
               write(*,*)
               write(*,2) 's% lnT(k)', k, s% lnT(k)
               write(*,2) 'other lnT', k, other_f
               write(*,2) 'dlnT', k, s% lnT(k) - other_f
               write(*,*)
               write(*,2) 'equ(equL,k)', k, equ(equL, k)
               write(*,2) 'other equ(equL,k)', k, other_equ
               write(*,2) 'dequ', k, (equ(equL, k) - other_equ)
               write(*,*)
               write(*,2) 'dequ/dlnT', k, (equ(equL,k) - other_equ)/(s% lnT(k) - other_f)
               write(*,*)
               
               write(*,*)
               do j=1,species
                  if (.true. .or. s% xa(j,k) > 1d-9) &
                     write(*,1) 'xin(net_iso(i' // &
                        trim(chem_isos% name(s% chem_id(j))) // '))= ', s% xa(j,k)
               end do
               write(*,*)
               write(*,1) 'T =', s% T(k)
               write(*,1) 'logT =', s% lnT(k)/ln10
               write(*,1) 'rho =', s% rho(k)
               write(*,1) 'logRho =', s% lnd(k)/ln10
               write(*,1) 'abar =', s% abar(k)
               write(*,1) 'zbar =', s% zbar(k)
               write(*,*)
               write(*,*) 'end numeric differences'
               write(*,*)
               !stop 'luminosity_eqn'
               
               
            end if

            if (skip_partials) return
         
            d_dLdM_dlndm1 = 0
            d_dLdM_dlnd00 = 0
            d_dLdM_dlndp1 = 0
            d_dLdM_dlnTm1 = 0
            d_dLdM_dlnT00 = 0
            d_dLdM_dlnTp1 = 0
            d_dLdM_dlnR00 = 0
            d_dLdM_dlnRp1 = 0
            d_dLdM_dL00 = 0

            d_dLdM_dlnEm1 = 0
            d_dLdM_dlnE00 = 0
            d_dLdM_dlnEp1 = 0
            d_dLdM_dv00 = 0
            d_dLdM_dvp1 = 0

            d_eps_burn_dlnd = s% d_epsnuc_dlnd(k) - s% d_nonnucneu_dlnd(k) + s% d_extra_heat_dlnd(k)
            d_eps_burn_dlnT = s% d_epsnuc_dlnT(k) - s% d_nonnucneu_dlnT(k) + s% d_extra_heat_dlnT(k)
      
            d_dLdM_dlnR00 = s% d_eps_grav_dlnR00(k)
            d_dLdM_dlnRp1 = s% d_eps_grav_dlnRp1(k)
            d_dLdM_dL00 = s% d_eps_grav_dL00(k)

            d_dLdM_dv00 = s% d_eps_grav_dv00(k)
            d_dLdM_dvp1 = s% d_eps_grav_dvp1(k)
            
            d_dLdM_dlnEm1 = s% d_eps_grav_dlnEm1(k)
            d_dLdM_dlnE00 = s% d_eps_grav_dlnE00(k)
            d_dLdM_dlnEp1 = s% d_eps_grav_dlnEp1(k)

            if (s% lnPgas_flag) then
               
               d_eps_burn_dlnPgas_const_T = d_eps_burn_dlnd*s% dlnRho_dlnPgas_const_T(k)
               d_eps_burn_dlnT_const_Pgas = &
                  d_eps_burn_dlnT + d_eps_burn_dlnd*s% dlnRho_dlnT_const_Pgas(k)
      
               d_dLdM_dlnPgasm1_const_T = s% d_eps_grav_dlnPgasm1_const_T(k)
               d_dLdM_dlnPgas00_const_T = &
                  s% d_eps_grav_dlnPgas00_const_T(k) + d_eps_burn_dlnPgas_const_T
               d_dLdM_dlnPgasp1_const_T = s% d_eps_grav_dlnPgasp1_const_T(k)
            
               d_dLdM_dlnTm1_const_Pgas = s% d_eps_grav_dlnTm1_const_Pgas(k)
               d_dLdM_dlnT00_const_Pgas = &
                  s% d_eps_grav_dlnT00_const_Pgas(k) + d_eps_burn_dlnT_const_Pgas
               d_dLdM_dlnTp1_const_Pgas = s% d_eps_grav_dlnTp1_const_Pgas(k)

            else
      
               d_dLdM_dlndm1 = s% d_eps_grav_dlndm1(k)
               d_dLdM_dlnd00 = s% d_eps_grav_dlnd00(k) + d_eps_burn_dlnd
               d_dLdM_dlndp1 = s% d_eps_grav_dlndp1(k)
            
               d_dLdM_dlnTm1 = s% d_eps_grav_dlnTm1(k)
               d_dLdM_dlnT00 = s% d_eps_grav_dlnT00(k) + d_eps_burn_dlnT
               d_dLdM_dlnTp1 = s% d_eps_grav_dlnTp1(k)
               
               d_dLdM_dlnPgasm1_const_T = 0
               d_dLdM_dlnPgas00_const_T = 0
               d_dLdM_dlnPgasp1_const_T = 0
            
               d_dLdM_dlnTm1_const_Pgas = 0
               d_dLdM_dlnT00_const_Pgas = 0
               d_dLdM_dlnTp1_const_Pgas = 0

            end if
            
            dLp1_dFLp1 = one*L_to_dL_dFL(s,Lp1)
            dL00_dFL00 = one*L_to_dL_dFL(s,L00)
            dFLx_dLx = one*L_to_dFL_dL(s,Lx)
            ! d_dLx_df = d_dLdm_df*dm
            ! dLx_df = d_dLx_df for f /= Lp1
            ! dFlx_df = dLx_df*dFLx_dLx
            dLx_dLp1 = 1 ! Lx = Lp1 + dLx
            dFLx_dLp1 = dLx_dLp1*dFLx_dLx
            dFLx_dFLp1 = dFLx_dLp1*dLp1_dFLp1
            dFLx_dlnTdot = s% d_eps_grav_dlnTdot(k)*dm*dFLx_dLx
            dFLx_dlnddot = s% d_eps_grav_dlnddot(k)*dm*dFLx_dLx            
            dFLx_dlnT00 = d_dLdm_dlnT00*dm*dFLx_dLx
            dFLx_dlnTm1 = d_dLdM_dlnTm1*dm*dFLx_dLx
            dFLx_dlnTp1 = d_dLdM_dlnTp1*dm*dFLx_dLx
            dFLx_dlnd00 = d_dLdm_dlnd00*dm*dFLx_dLx
            dFLx_dlnR00 = d_dLdM_dlnR00*dm*dFLx_dLx
            dFLx_dlnE00 = d_dLdM_dlnE00*dm*dFLx_dLx
            dFLx_dv00 = d_dLdM_dv00*dm*dFLx_dLx
            dFLx_dL00 = d_dLdM_dL00*dm*dFLx_dLx
            dFLx_dlnRp1 = d_dLdM_dlnRp1*dm*dFLx_dLx
            dFLx_dlnEp1 = d_dLdM_dlnEp1*dm*dFLx_dLx
            dFLx_dvp1 = d_dLdM_dvp1*dm*dFLx_dLx
            dFLx_dlndp1 = d_dLdM_dlndp1*dm*dFLx_dLx
            dFLx_dlnEm1 = d_dLdM_dlnEm1*dm*dFLx_dLx
            dFLx_dlndm1 = d_dLdM_dlndm1*dm*dFLx_dLx
            
            if (s% lnPgas_flag) then
               dFLx_dlnT00_const_Pgas = d_dLdm_dlnT00_const_Pgas*dm*dFLx_dLx
               dFLx_dlnTm1_const_Pgas = d_dLdM_dlnTm1_const_Pgas*dm*dFLx_dLx
               dFLx_dlnTp1_const_Pgas = d_dLdM_dlnTp1_const_Pgas*dm*dFLx_dLx
               dFLx_dlnPgasp1_const_T = d_dLdM_dlnPgasp1_const_T*dm*dFLx_dLx
               dFLx_dlnPgas00_const_T = d_dLdm_dlnPgas00_const_T*dm*dFLx_dLx
               dFLx_dlnPgasm1_const_T = d_dLdM_dlnPgasm1_const_T*dm*dFLx_dLx
            else
               dFLx_dlnT00_const_Pgas = 0
               dFLx_dlnTm1_const_Pgas = 0
               dFLx_dlnTp1_const_Pgas = 0
               dFLx_dlnPgasp1_const_T = 0
               dFLx_dlnPgas00_const_T = 0
               dFLx_dlnPgasm1_const_T = 0
            end if
            
            call e00(s, xscale, equL, i_FL, k, nvar, dFLx_dL00*dL00_dFL00/L_scale - 1/L_scale)
            if (k < nz) call ep1(s, xscale, equL, i_FL, k, nvar, dFLx_dFLp1/L_scale)
               
            if (s% lnTdot_flag) then
               call e00(s, xscale, equL, i_lnTdot, k, nvar, dFLx_dlnTdot/L_scale)
            end if
               
            if (s% lnddot_flag) then
               call e00(s, xscale, equL, i_lnddot, k, nvar, dFLx_dlnddot/L_scale)
            end if
      
            if (s% lnPgas_flag) then
               call e00(s, xscale, equL, i_lnT, k, nvar, dFLx_dlnT00_const_Pgas/L_scale)
               if (k > 1) then
                  call em1(s, xscale, equL, i_lnT, k, nvar, dFLx_dlnTm1_const_Pgas/L_scale)
               end if
               if (k < nz) then
                  call ep1(s, xscale, equL, i_lnT, k, nvar, dFLx_dlnTp1_const_Pgas/L_scale)
               end if
            else
               call e00(s, xscale, equL, i_lnT, k, nvar, dFLx_dlnT00/L_scale)
               if (k > 1) call em1(s, xscale, equL, i_lnT, k, nvar, dFLx_dlnTm1/L_scale)
               if (k < nz) call ep1(s, xscale, equL, i_lnT, k, nvar, dFLx_dlnTp1/L_scale)
            end if

            if (s% do_struct_hydro) then
               if (s% lnPgas_flag) then
                  call e00(s, xscale, equL, i_lnPgas, k, nvar, dFLx_dlnPgas00_const_T/L_scale)
               else
                  call e00(s, xscale, equL, i_xlnd, k, nvar, dFLx_dlnd00/L_scale)
               end if
               
               call e00(s, xscale, equL, i_lnR, k, nvar, dFLx_dlnR00/L_scale)
               if (i_lnE /= 0) call e00(s, xscale, equL, i_lnE, k, nvar, dFLx_dlnE00/L_scale)
               if (i_vel /= 0) call e00(s, xscale, equL, i_vel, k, nvar, dFLx_dv00/L_scale)
               if (k < nz) then
                  call ep1(s, xscale, equL, i_lnR, k, nvar, dFLx_dlnRp1/L_scale)
                  if (i_lnE /= 0) call ep1(s, xscale, equL, i_lnE, k, nvar, dFLx_dlnEp1/L_scale)
                  if (i_vel /= 0) call ep1(s, xscale, equL, i_vel, k, nvar, dFLx_dvp1/L_scale)
                  if (s% lnPgas_flag) then
                     call ep1(s, xscale, equL, i_lnPgas, k, nvar, dFLx_dlnPgasp1_const_T/L_scale)
                  else
                     call ep1(s, xscale, equL, i_xlnd, k, nvar, dFLx_dlndp1/L_scale)
                  end if
               end if
               if (k > 1) then
                  if (i_lnE /= 0) call em1(s, xscale, equL, i_lnE, k, nvar, dFLx_dlnEm1/L_scale)
                  if (s% lnPgas_flag) then
                     call em1(s, xscale, equL, i_lnPgas, k, nvar, dFLx_dlnPgasm1_const_T/L_scale)
                  else
                     call em1(s, xscale, equL, i_xlnd, k, nvar, dFLx_dlndm1/L_scale)
                  end if
               end if
            end if
            
            if (do_chem) then
               do j=1,nvar_chem
                  dFLx_dx = dFLx_dLx*s% d_epsnuc_dx(j,k)*dm
                  call e00(s, xscale, equL, i_chem1+j-1, k, nvar, dFLx_dx/L_scale)
                  
                  
                  if (.false. .and. k == 1358 .and. j == 3) then
                     write(*,3) 's% xa(j,k)', j, k, s% xa(j,k)
                     write(*,2) 'equ(equL,k)', k, equ(equL,k)
                     write(*,2) 'dFLx_dx/L_scale', k, dFLx_dx/L_scale
                     write(*,*)
                     !stop
                  end if


               end do
            end if
            
            
            if (.false. .and. k==1230) then
               !FLp1 = L_to_FL(s,Lp1)
               write(*,*)
               write(*,*)
               write(*,*) 'analytic'
               write(*,2) 'equL', equL
               write(*,*)
               write(*,2) 's% lnT(k)', k, s% lnT(k)
               write(*,2) 'equ(equL,k)', k, equ(equL, k)
               write(*,2) 'dFLx_dlnT00/L_scale', k, dFLx_dlnT00/L_scale

               !stop
               write(*,*)
               do j=1,species
                  if (.true. .or. s% xa(j,k) > 1d-9) &
                     write(*,1) 'xin(net_iso(i' // &
                        trim(chem_isos% name(s% chem_id(j))) // '))= ', s% xa(j,k)
               end do
               write(*,*)
               write(*,1) 'T =', s% T(k)
               write(*,1) 'logT =', s% lnT(k)/ln10
               write(*,1) 'rho =', s% rho(k)
               write(*,1) 'logRho =', s% lnd(k)/ln10
               write(*,1) 'abar =', s% abar(k)
               write(*,1) 'zbar =', s% zbar(k)
               write(*,*)
               write(*,*)
               
            end if
            

            if (.not. do_artificial_viscosity) return
         
            if (k < nz) then
               !del = s% v(k)*(3*s% Amid(k) - s% area(k)) - s% v(k+1)*(3*s% Amid(k) - s% area(k+1))
               dA_dlnR = 2*s% area(k)
               d_del_dlnR00 = &
                  s% v(k)*(3*s% dAmid_dlnR00(k) - dA_dlnR) - s% v(k+1)*3*s% dAmid_dlnR00(k)
               dA_dlnR = 2*s% area(k+1)
               d_del_dlnRp1 = &
                  s% v(k)*3*s% dAmid_dlnRp1(k) - s% v(k+1)*(3*s% dAmid_dlnRp1(k) - dA_dlnR)
               d_del_dvel00 = (3*s% Amid(k) - s% area(k))
               d_del_dvelp1 = -(3*s% Amid(k) - s% area(k+1))
            else
               !del = s% v(k)*(3*s% Amid(k) - s% area(k))
               dA_dlnR = 2*s% area(k)
               d_del_dlnR00 = s% v(k)*(3*s% dAmid_dlnR00(k) - dA_dlnR)
               d_del_dlnRp1 = 0
               d_del_dvel00 = (3*s% Amid(k) - s% area(k))
               d_del_dvelp1 = 0
            end if
         
            !dLAV = 0.5d0*s% Pvisc(k)*del
            d_dLAV_dlnd = 0.5d0*s% dPvisc_dlnd(k)*del
            d_dLAV_dlnT = 0.5d0*s% dPvisc_dlnT(k)*del
            d_dLAV_dlnR00 = 0.5d0*s% Pvisc(k)*d_del_dlnR00
            d_dLAV_dvel00 = 0.5d0*s% Pvisc(k)*d_del_dvel00
            if (k < nz) then
               d_dLAV_dlnRp1 = 0.5d0*s% Pvisc(k)*d_del_dlnRp1
               d_dLAV_dvelp1 = 0.5d0*s% Pvisc(k)*d_del_dvelp1
            else
               d_dLAV_dlnRp1 = 0
               d_dLAV_dvelp1 = 0
            end if
         
            !dL_expected = dL_expected + dLAV
            !equ(equL, k) = (dL_expected - dL_actual)/L_scale
            
            if (s% lnPgas_flag) then
               d_dLAV_dlnT_const_Pgas = &
                  d_dLAV_dlnT + d_dLAV_dlnd*s% dlnRho_dlnT_const_Pgas(k)
               call e00(s, xscale, equL, i_lnT, k, nvar, &
                           d_dLAV_dlnT_const_Pgas*dFLx_dLx/L_scale)
            else
               call e00(s, xscale, equL, i_lnT, k, nvar, d_dLAV_dlnT*dFLx_dLx/L_scale)
            end if

            if (s% do_struct_hydro) then
               if (s% lnPgas_flag) then   
                  d_dLAV_dlnPgas_const_T = d_dLAV_dlnd*s% dlnRho_dlnPgas_const_T(k)
                  call e00(s, xscale, equL, i_lnPgas, k, nvar, &
                           d_dLAV_dlnPgas_const_T*dFLx_dLx/L_scale)
               else
                  call e00(s, xscale, equL, i_xlnd, k, nvar, d_dLAV_dlnd*dFLx_dLx/L_scale)
               end if
               call e00(s, xscale, equL, i_lnR, k, nvar, d_dLAV_dlnR00*dFLx_dLx/L_scale)
               call e00(s, xscale, equL, i_vel, k, nvar, d_dLAV_dvel00*dFLx_dLx/L_scale)
               if (k < nz) then
                  call ep1(s, xscale, equL, i_lnR, k, nvar, d_dLAV_dlnRp1*dFLx_dLx/L_scale)
                  call ep1(s, xscale, equL, i_vel, k, nvar, d_dLAV_dvelp1*dFLx_dLx/L_scale)
               end if
            end if

         end subroutine do1_luminosity_eqn
         
               
         subroutine do1_pressure_eqn(k,ierr)
            ! compare lnP(k-1) to lnP(k)
            use chem_def, only: chem_isos
            use eos_def, only: i_lnPgas
            use star_utils, only: L_to_dL_dFL

            integer, intent(in) :: k
            integer, intent(out) :: ierr
         
            real(fltp) :: &
               lnP_surf, delm, d_delm_dqm1, d_delm_dq00, P00, Pm1, dL00_dFL00, &
               dPr, Ppoint, Ppoint_inv, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               d_Ppoint_inv_dlndm1, d_Ppoint_inv_dlnd00, d_Ppoint_inv_dlnTm1, d_Ppoint_inv_dlnT00, &
               d_Ppoint_inv_dlnPgas00_const_T, d_Ppoint_inv_dlnPgasm1_const_T, &
               d_Ppoint_inv_dlnT00_const_Pgas, d_Ppoint_inv_dlnTm1_const_Pgas, &
               lnPdiff, d_lnPdiff_dlnP00, d_lnPdiff_dlnPm1, var_test, equ_test, &
               d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
               d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
               d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
               d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
               d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
               d_delm_dlndq00, d_delm_dlndqm1, diff, diff_old, &
               dA_dlnR, delQ, dlnPQ, Ainv, &
               d_delQ_dlnd00, d_delQ_dlnT00, d_delQ_dlnR00, d_delQ_dvel00, &
               d_delQ_dlndm1, d_delQ_dlnTm1, d_delQ_dlnRm1, d_delQ_dvelm1, &
               d_delQ_dlnRp1, d_delQ_dvelp1, d_Ainv_dlnR00, &
               d_delQ_dlnPgas00_const_T, d_delQ_dlnPgasm1_const_T, &
               d_delQ_dlnT00_const_Pgas, d_delQ_dlnTm1_const_Pgas, &
               d_dlnPQ_dlnd00, d_dlnPQ_dlnT00, d_dlnPQ_dlnR00, d_dlnPQ_dvel00, &
               d_dlnPQ_dlndm1, d_dlnPQ_dlnTm1, d_dlnPQ_dlnRm1, d_dlnPQ_dvelm1, &
               d_dlnPQ_dlnRp1, d_dlnPQ_dvelp1, &
               r003, rp13, T1, lnT1, lnP1, &
               dlnT1_dL, dlnT1_dlnR, &
               dlnP1_dL, dlnP1_dlnR, &
               dlnP1_dlnm, dlnP1_dlnkap, &
               dlnkap_dlnd, dlnkap_dlnT, &
               dlnP1_dlnd, dlnP1_dlnT, &
               dlnP1_dlnR00, dlnP1_dlnRp1, d_dlnPdm_dL00, &
               dlnP00_dlnPgas_const_T, dlnP00_dlnT_const_Pgas, &
               dlnPm1_dlnPgas_const_T, dlnPm1_dlnT_const_Pgas, &
               d_lnPdiff_dlnTm1_const_Pgas, d_lnPdiff_dlnT00_const_Pgas, &
               d_lnPdiff_dlnPpgas00_const_T, d_lnPdiff_dlnPgasm1_const_T, &
               d_dlnPQ_dlnPgas00_const_T, d_dlnPQ_dlnPgasm1_const_T, &
               d_dlnPQ_dlnT00_const_Pgas, d_dlnPQ_dlnTm1_const_Pgas
            integer :: j   
            logical :: dbg 

            real(dp), parameter :: lnd0 = -1.7048902326588365D+01
            real(dp), parameter :: equ0 = -9.9870027208667683D-05
            real(dp), parameter :: dlnP_dm0 = -2.3395769706524674D-38

            include 'formats.dek'
            ierr = 0

      
            dbg = .false.
         
            ! compare lnP(k-1) to lnP(k)
         
            call do_dlnP_dm(s, k, dt, &
               d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
               d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
               d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
               d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
               d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &         
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
               d_dlnPdm_dL00, &
               Ppoint, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               ierr)
            if (ierr /= 0) return            
         
            if (k == 1) then
               delm = s% dm(k)/2
               Pm1 = 0
            else
               delm = (s% dm(k) + s% dm(k-1))/2
               Pm1 = s% P(k-1)
            end if
         
            P00 = s% P(k)
            dPr = Pm1 - P00
            Ppoint_inv = 1/Ppoint
      
            ! basic eqn is dP = -G m / (4 pi r^4)
            ! divide by <P> to make it unitless
            ! simple average is adequate for <P> since is only for normalizing the equation.
            ! however, be careful to use same <P> for both sides of equation..... 
         
            lnPdiff = dPr/Ppoint ! use this in place of lnP(k-1)-lnP(k)
         
            equ(equP, k) = delm*dlnP_dm(k) - lnPdiff
            
            if (dbg) then
               write(*,2) 'equ(equP, k)', k, equ(equP, k)
               write(*,2) 'delm', k, delm
               write(*,2) 'dlnP_dm(k)', k, dlnP_dm(k)
               write(*,2) 'delm*dlnP_dm(k)', k, delm*dlnP_dm(k)
               write(*,2) 'lnPdiff', k, lnPdiff
               write(*,*)
            end if
         
            if (k == 1 .or. .not. do_artificial_viscosity) then ! to prevent bogus warnings
               Ainv = 0
               delQ = 0
            else
               delQ = s% Pvisc(k-1)*(3*s% Amid(k-1) - s% area(k)) - &
                      s% Pvisc(k)*(3*s% Amid(k) - s% area(k))
               Ainv = 1/s% area(k)
               dlnPQ = -0.5d0*delQ*Ppoint_inv*Ainv
               equ(equP, k) = equ(equP, k) + dlnPQ
            end if
                     
            if (is_bad(equ(equP, k))) then
               ierr = -1
               if (s% report_ierr) write(*,*) 'P_eqn: is_bad(equ(equP, k))'
               return
            end if

            if (.false. .and. s% hydro_call_number == 17417 .and. k == 304 .and. &
                  skip_partials .and. abs(lnd0 - s% lnd(k)) > 1d-12*abs(lnd0)) then
               write(*,*)
               write(*,*) 'numeric'
               write(*,*) 'do_artificial_viscosity', do_artificial_viscosity
               write(*,2) 's% lnd(k)', k, s% lnd(k)
               write(*,2) 'vs', k, lnd0
               write(*,*)
               write(*,2) 'equ(equP, k)', k, equ(equP, k)
               write(*,2) 'vs', k, equ0
               write(*,*)
               write(*,2) 'dequP/dlnd', k, (equ(equP,k) - equ0)/(s% lnd(k) - lnd0)
               write(*,2) 'vs', k, 6.5608434652647929D-02
               write(*,*)
               write(*,2) 'delm*dlnP_dm(k)', k, delm*dlnP_dm(k)
               write(*,2) 'vs', k, delm*(-1.2431625967331853D-31)
               write(*,*)
               write(*,2) 'lnPdiff', k, lnPdiff
               write(*,2) 'vs', k, (-5.1427058225696771D-02)
               write(*,*)
               write(*,2) 'd_dlnPdm_dlnd00', k, (dlnP_dm(k) - (-1.2431625967331853D-31))/(s% lnd(k) - lnd0)
               write(*,2) 'vs', k, 4.0778941386363592D-33
               write(*,*)
               write(*,2) 'd_lnPdiff_dlnP00', k, (lnPdiff - (-5.1427058225696771D-02))/(s% lnd(k) - lnd0)
               write(*,2) 'vs', k, -9.9933881442088235D-01
               write(*,*)
               !stop
            end if
            
            if (.false. .and. s% hydro_call_number == 17417 .and. k == 304) then
               write(*,2) 'dlnP_dm,cgrav,m,r,Ppoint', k, dlnP_dm(k), s% cgrav(k), s% m(k), s% r(k), Ppoint
            end if
            
            if (skip_partials) return

            d_lnPdiff_dlnPm1 = P00*Pm1/Ppoint**2
            d_lnPdiff_dlnP00 = -d_lnPdiff_dlnPm1
            
            dL00_dFL00 = L_to_dL_dFL(s,s% L(k))
         
            call e00(s, xscale, equP, i_lnR, k, nvar, delm*d_dlnPdm_dlnR00)
            call e00(s, xscale, equP, i_FL, k, nvar, delm*d_dlnPdm_dL00*dL00_dFL00)
            
            if (s% lnPgas_flag) then
               dlnP00_dlnPgas_const_T = s% Pgas(k)/s% P(k)
               dlnP00_dlnT_const_Pgas = 4*s% Prad(k)/s% P(k)
               d_lnPdiff_dlnPpgas00_const_T = d_lnPdiff_dlnP00*dlnP00_dlnPgas_const_T
               d_lnPdiff_dlnT00_const_Pgas = d_lnPdiff_dlnP00*dlnP00_dlnT_const_Pgas
               call e00(s, xscale, equP, i_lnPgas, k, nvar, &
                     delm*d_dlnPdm_dlnPgas00_const_T - d_lnPdiff_dlnPpgas00_const_T)
               if (s% do_struct_thermo) &
                  call e00(s, xscale, equP, i_lnT, k, nvar, &
                        delm*d_dlnPdm_dlnT00_const_Pgas - d_lnPdiff_dlnT00_const_Pgas)
            else
               call e00(s, xscale, equP, i_xlnd, k, nvar, &
                     delm*d_dlnPdm_dlnd00 - d_lnPdiff_dlnP00*s% chiRho(k))
               if (s% do_struct_thermo) &
                  call e00(s, xscale, equP, i_lnT, k, nvar, &
                        delm*d_dlnPdm_dlnT00 - d_lnPdiff_dlnP00*s% chiT(k))
            end if
            
            
            if (.false. .and. s% hydro_call_number == 17417 .and. k == 304) then
               write(*,*)
               write(*,*) 'analytic'
               write(*,2) 'equP, lnd', k, equ(equP,k), s% lnd(k)
               write(*,2) 'dequP_dlnd', k, delm*d_dlnPdm_dlnd00 - d_lnPdiff_dlnP00*s% chiRho(k)
               write(*,2) 'd_dlnPdm_dlnd00', k, d_dlnPdm_dlnd00
               write(*,2) 'd_lnPdiff_dlnP00', k, d_lnPdiff_dlnP00
               write(*,2) 's% chiRho(k)', k, s% chiRho(k)
               write(*,2) 'dlnP_dm(k)', k, dlnP_dm(k)
               write(*,2) 'lnPdiff', k, lnPdiff
               write(*,2) 'delm', k, delm
            end if
            
            if (k > 1) then
               call em1(s, xscale, equP, i_lnR, k, nvar, delm*d_dlnPdm_dlnRm1)
               if (s% lnPgas_flag) then
                  dlnPm1_dlnPgas_const_T = s% Pgas(k-1)/s% P(k-1)
                  dlnPm1_dlnT_const_Pgas = 4*s% Prad(k-1)/s% P(k-1)
                  d_lnPdiff_dlnPgasm1_const_T = d_lnPdiff_dlnPm1*dlnPm1_dlnPgas_const_T
                  d_lnPdiff_dlnTm1_const_Pgas = d_lnPdiff_dlnPm1*dlnPm1_dlnT_const_Pgas
                  call em1(s, xscale, equP, i_lnPgas, k, nvar, &
                     delm*d_dlnPdm_dlnPgasm1_const_T - d_lnPdiff_dlnPgasm1_const_T)
                  if (s% do_struct_thermo) &
                     call em1(s, xscale, equP, i_lnT, k, nvar, &
                        delm*d_dlnPdm_dlnTm1_const_Pgas - d_lnPdiff_dlnTm1_const_Pgas)
               else
                  call em1(s, xscale, equP, i_xlnd, k, nvar, &
                     delm*d_dlnPdm_dlndm1 - d_lnPdiff_dlnPm1*s% chiRho(k-1))
                  if (s% do_struct_thermo) &
                     call em1(s, xscale, equP, i_lnT, k, nvar, &
                        delm*d_dlnPdm_dlnTm1 - d_lnPdiff_dlnPm1*s% chiT(k-1))
               end if
            end if

            if (s% v_flag) then
               call e00(s, xscale, equP, i_vel, k, nvar, delm*d_dlnPdm_dvel00)
               if (k > 1) call em1(s, xscale, equP, i_vel, k, nvar, delm*d_dlnPdm_dvelm1)
            end if
         
            if (k == 1 .or. .not. do_artificial_viscosity) return
            
            !delQ = s% Pvisc(k-1)*(3*s% Amid(k-1) - s% area(k))
            !     - s% Pvisc(k)*(3*s% Amid(k) - s% area(k))
         
            d_delQ_dlnd00 = -s% dPvisc_dlnd(k)*(3*s% Amid(k) - s% area(k))
            d_delQ_dlnT00 = -s% dPvisc_dlnT(k)*(3*s% Amid(k) - s% area(k))
            
            dA_dlnR = 2*s% area(k)
            d_delQ_dlnR00 = &
               s% dPvisc_dlnRp1(k-1)*(3*s% Amid(k-1) - s% area(k)) &
               + s% Pvisc(k-1)*(3*s% dAmid_dlnRp1(k-1) - dA_dlnR) &
               - s% dPvisc_dlnR00(k)*(3*s% Amid(k) - s% area(k)) &
               - s% Pvisc(k)*(3*s% dAmid_dlnR00(k) - dA_dlnR)
            d_delQ_dvel00 = &
               s% dPvisc_dvelp1(k-1)*(3*s% Amid(k-1) - s% area(k)) &
               - s% dPvisc_dvel00(k)*(3*s% Amid(k) - s% area(k))
         
            d_delQ_dlndm1 = s% dPvisc_dlnd(k-1)*(3*s% Amid(k-1) - s% area(k))
            d_delQ_dlnTm1 = s% dPvisc_dlnT(k-1)*(3*s% Amid(k-1) - s% area(k))
            d_delQ_dlnRm1 = &
               s% dPvisc_dlnR00(k-1)*(3*s% Amid(k-1) - s% area(k)) &
               + s% Pvisc(k-1)*3*s% dAmid_dlnR00(k-1)
            d_delQ_dvelm1 = s% dPvisc_dvel00(k-1)*(3*s% Amid(k-1) - s% area(k))
         
            if (k < nz) then
               d_delQ_dlnRp1 = -s% dPvisc_dlnRp1(k)*(3*s% Amid(k) - s% area(k))
               d_delQ_dvelp1 = -s% dPvisc_dvelp1(k)*(3*s% Amid(k) - s% area(k))
            else
               d_delq_dlnRp1 = 0
               d_delQ_dvelp1 = 0
            end if               
         
            !Ainv = 1/s% area(k)
            dA_dlnR = 2*s% area(k)
            d_Ainv_dlnR00 = -dA_dlnR*Ainv**2
         
            !dlnPQ = -0.5d0*delQ*Ppoint_inv*Ainv
         
            d_dlnPQ_dlnR00 = &
               -0.5d0*(d_delQ_dlnR00*Ainv + delQ*d_Ainv_dlnR00)*Ppoint_inv
            d_dlnPQ_dvel00 = -0.5d0*d_delQ_dvel00*Ppoint_inv*Ainv
         
            d_dlnPQ_dlnRm1 = -0.5d0*d_delQ_dlnRm1*Ppoint_inv*Ainv
            d_dlnPQ_dvelm1 = -0.5d0*d_delQ_dvelm1*Ppoint_inv*Ainv
         
            !equ(equP, k) = equ(equP, k) + dlnPQ

            call e00(s, xscale, equP, i_lnR, k, nvar, d_dlnPQ_dlnR00)
            call em1(s, xscale, equP, i_lnR, k, nvar, d_dlnPQ_dlnRm1)

            call e00(s, xscale, equP, i_vel, k, nvar, d_dlnPQ_dvel00)         
            call em1(s, xscale, equP, i_vel, k, nvar, d_dlnPQ_dvelm1)
         
            if (s% lnPgas_flag) then               
               d_delQ_dlnPgas00_const_T = d_delQ_dlnd00*s% dlnRho_dlnPgas_const_T(k)
               d_delQ_dlnT00_const_Pgas = &
                  d_delQ_dlnT00 + d_delQ_dlnd00*s% dlnRho_dlnT_const_Pgas(k)
               d_delQ_dlnPgasm1_const_T = d_delQ_dlndm1*s% dlnRho_dlnPgas_const_T(k-1)             
               d_delQ_dlnTm1_const_Pgas = &
                  d_delQ_dlnTm1 + d_delQ_dlndm1*s% dlnRho_dlnT_const_Pgas(k-1)
               d_Ppoint_inv_dlnPgas00_const_T = -dPpoint_dlnPgas00_const_T*Ppoint_inv**2
               d_Ppoint_inv_dlnPgasm1_const_T = -dPpoint_dlnPgasm1_const_T*Ppoint_inv**2
               d_Ppoint_inv_dlnT00_const_Pgas = -dPpoint_dlnT00_const_Pgas*Ppoint_inv**2
               d_Ppoint_inv_dlnTm1_const_Pgas = -dPpoint_dlnTm1_const_Pgas*Ppoint_inv**2
               d_dlnPQ_dlnPgas00_const_T = &
                  -0.5d0*(d_delQ_dlnPgas00_const_T*Ppoint_inv + delQ*d_Ppoint_inv_dlnPgas00_const_T)*Ainv
               call e00(s, xscale, equP, i_lnPgas, k, nvar, d_dlnPQ_dlnPgas00_const_T)
               d_dlnPQ_dlnPgasm1_const_T = &
                  -0.5d0*(d_delQ_dlnPgasm1_const_T*Ppoint_inv + delQ*d_Ppoint_inv_dlnPgasm1_const_T)*Ainv
               call em1(s, xscale, equP, i_lnPgas, k, nvar, d_dlnPQ_dlnPgasm1_const_T)
               if (s% do_struct_thermo) then
                  d_dlnPQ_dlnT00_const_Pgas = &
                     -0.5d0*(d_delQ_dlnT00_const_Pgas*Ppoint_inv + delQ*d_Ppoint_inv_dlnT00_const_Pgas)*Ainv
                  call e00(s, xscale, equP, i_lnT, k, nvar, d_dlnPQ_dlnT00_const_Pgas)
                  d_dlnPQ_dlnTm1_const_Pgas = &
                     -0.5d0*(d_delQ_dlnTm1_const_Pgas*Ppoint_inv + delQ*d_Ppoint_inv_dlnTm1_const_Pgas)*Ainv
                  call em1(s, xscale, equP, i_lnT, k, nvar, d_dlnPQ_dlnTm1_const_Pgas)
               end if
            else
               d_Ppoint_inv_dlnd00 = -dPpoint_dlnd00*Ppoint_inv**2
               d_Ppoint_inv_dlndm1 = -dPpoint_dlndm1*Ppoint_inv**2
               d_Ppoint_inv_dlnT00 = -dPpoint_dlnT00*Ppoint_inv**2
               d_Ppoint_inv_dlnTm1 = -dPpoint_dlnTm1*Ppoint_inv**2
               d_dlnPQ_dlnd00 = &
                  -0.5d0*(d_delQ_dlnd00*Ppoint_inv + delQ*d_Ppoint_inv_dlnd00)*Ainv
               call e00(s, xscale, equP, i_xlnd, k, nvar, d_dlnPQ_dlnd00)
               d_dlnPQ_dlndm1 = &
                  -0.5d0*(d_delQ_dlndm1*Ppoint_inv + delQ*d_Ppoint_inv_dlndm1)*Ainv
               call em1(s, xscale, equP, i_xlnd, k, nvar, d_dlnPQ_dlndm1)
               if (s% do_struct_thermo) then
                  d_dlnPQ_dlnT00 = &
                     -0.5d0*(d_delQ_dlnT00*Ppoint_inv + delQ*d_Ppoint_inv_dlnT00)*Ainv
                  call e00(s, xscale, equP, i_lnT, k, nvar, d_dlnPQ_dlnT00)
                  d_dlnPQ_dlnTm1 = &
                     -0.5d0*(d_delQ_dlnTm1*Ppoint_inv + delQ*d_Ppoint_inv_dlnTm1)*Ainv
                  call em1(s, xscale, equP, i_lnT, k, nvar, d_dlnPQ_dlnTm1)
               end if
            end if
         
            if (k < nz) then
               d_dlnPQ_dlnRp1 = -0.5d0*d_delQ_dlnRp1*Ppoint_inv*Ainv
               d_dlnPQ_dvelp1 = -0.5d0*d_delQ_dvelp1*Ppoint_inv*Ainv
               call ep1(s, xscale, equP, i_lnR, k, nvar, d_dlnPQ_dlnRp1)
               call ep1(s, xscale, equP, i_vel, k, nvar, d_dlnPQ_dvelp1)
            end if

         end subroutine do1_pressure_eqn
         
         
         subroutine do1_lnddot_eqn(k,ierr)
            use hydro_vars, only: get_dVARDOT_dVAR_cell
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            
            real(dp) :: d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1
            logical :: ode_flag
            
            ierr = 0
            ode_flag = (.not. s% doing_hydro_newton) .and. s% ode_var(i_xlnd)
            
            if (ode_flag) then ! value of equ is rate of change for variable lnd
               equ(equlnddot,k) = lnddot(k)
               if (.not. skip_partials) call e00(s, xscale, equlnddot, i_lnddot, k, nvar, one)
               return
            end if

            equ(equlnddot,k) = s% dlnd_dt(k) - lnddot(k)
            if (skip_partials) return
            
            call get_dVARDOT_dVAR_cell(s, k, dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1)

            call e00(s, xscale, equlnddot, i_lnddot, k, nvar, -one)
            call e00(s, xscale, equlnddot, i_xlnd, k, nvar, one*d_dVARdt_dVAR00)
            if (k > 1) call em1(s, xscale, equlnddot, i_xlnd, k, nvar, one*d_dVARdt_dVARm1)
            if (k < nz) call ep1(s, xscale, equlnddot, i_xlnd, k, nvar, one*d_dVARdt_dVARp1)
             
         end subroutine do1_lnddot_eqn
         
         
         subroutine do1_lnTdot_eqn(k,ierr)
            use hydro_vars, only: get_dVARDOT_dVAR_cell
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            
            real(dp) :: d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1
            logical :: ode_flag
            
            ierr = 0
            ode_flag = (.not. s% doing_hydro_newton) .and. s% ode_var(i_lnT)
            
            if (ode_flag) then ! value of equ is rate of change for variable lnT
               equ(equlnTdot,k) = lnTdot(k)
               if (.not. skip_partials) call e00(s, xscale, equlnTdot, i_lnTdot, k, nvar, one)
               return
            end if

            equ(equlnTdot,k) = s% dlnT_dt(k) - lnTdot(k)
            if (skip_partials) return
            
            call get_dVARDOT_dVAR_cell(s, k, dt, d_dVARdt_dVARm1, d_dVARdt_dVAR00, d_dVARdt_dVARp1)

            call e00(s, xscale, equlnTdot, i_lnTdot, k, nvar, -one)
            call e00(s, xscale, equlnTdot, i_lnT, k, nvar, one*d_dVARdt_dVAR00)
            if (k > 1) call em1(s, xscale, equlnTdot, i_lnT, k, nvar, one*d_dVARdt_dVARm1)
            if (k < nz) call ep1(s, xscale, equlnTdot, i_lnT, k, nvar, one*d_dVARdt_dVARp1)
             
         end subroutine do1_lnTdot_eqn
         
               
         subroutine do1_temperature_eqn(k,ierr)
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            
            ! if not in quasi-hydrostatic-equilibrium, this is not correct.
            call do1_quasi_HSE_T_eqn(k,ierr)
         
         end subroutine do1_temperature_eqn
         
               
         subroutine do1_quasi_HSE_T_eqn(k,ierr)
            use star_utils, only: L_to_dL_dFL, L_to_FL

            integer, intent(in) :: k
            integer, intent(out) :: ierr
            
            real(fltp) :: alfa, beta, r, dr_dL00, dr_dlnR00, dr_dlnd00, dr_dlnT00, &
               dr_dlndm1, dr_dlnTm1, dr_lnq00, dr_lndq00, dr_lndqm1, m, dlnPdm, &
               d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
               d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
               dP_dlnPgas00_const_T, dP_dlnPgasm1_const_T, &
               dP_dlnT00_const_Pgas, dP_dlnTm1_const_Pgas, &
               grad_star, d_grad_star_dL, d_grad_star_dlnR, &       
               d_grad_star_dlnd00, d_grad_star_dlndm1, &    
               d_grad_star_dlnT00, d_grad_star_dlnTm1, &
               Ppoint, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               d_dlnTdm_dLum, d_dlnTdm_dlnR, d_dlnTdm_dlnd00, &
               d_dlnTdm_dlnT00, d_dlnTdm_dlndm1, d_dlnTdm_dlnTm1, &
               delm, d_delm_dqm1, d_delm_dq00, T00, Tm1, dT, Tpoint, &
               lnTdiff, d_lnTdiff_dlnT00, d_lnTdiff_dlnTm1, d_dlnTdm_dlnq, &
               d_delm_dlndq00, d_delm_dlndqm1, diff, diff_old, &
               d_grad_star_dlnT00_const_Pgas, d_grad_star_dlnTm1_const_Pgas, &
               d_grad_star_dlnPgas00_const_T, d_grad_star_dlnPgasm1_const_T, &
               d_dlnTdm_dlnPgas00_const_T, d_dlnTdm_dlnPgasm1_const_T, &
               d_dlnTdm_dlnT00_const_Pgas, d_dlnTdm_dlnTm1_const_Pgas, &
               dr_dlnPgas00_const_T, dr_dlnPgasm1_const_T, &
               dr_dlnT00_const_Pgas, dr_dlnTm1_const_Pgas, dL00_dFL00, &
               other, other_equ, FL, other_FL

            
            include 'formats.dek'
            ierr = 0
            ! compare lnT(k-1) to lnT(k)

            call eval_dlnPdm_qhse(s, k, m, &
               dlnPdm, d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, &
               d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
               d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
               Ppoint, &
               dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
               dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
               dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
               ierr)
            if (ierr /= 0) return
         
            call eval_grad_star_info( &
               s, k, grad_star, d_grad_star_dL, d_grad_star_dlnR, &
               d_grad_star_dlnd00, d_grad_star_dlndm1, &
               d_grad_star_dlnT00, d_grad_star_dlnTm1, &
               d_grad_star_dlnT00_const_Pgas, d_grad_star_dlnTm1_const_Pgas, &
               d_grad_star_dlnPgas00_const_T, d_grad_star_dlnPgasm1_const_T, &
               ierr)
                     
            s% dlnT_dm(k) = dlnPdm*grad_star
         
            delm = (s% dm(k) + s% dm(k-1))/2
            Tm1 = s% T(k-1)
            alfa = s% dm(k-1)/(s% dm(k-1) + s% dm(k))
            beta = 1 - alfa

            T00 = s% T(k)
            dT = Tm1 - T00
            Tpoint = alfa*T00 + beta*Tm1
            lnTdiff = dT/Tpoint ! use this in place of lnT(k-1)-lnT(k)
         
            r = delm*dlnT_dm(k) - lnTdiff
            equ(equT, k) = r
            
            if (is_bad(equ(equT, k))) then
               ierr = -1
               return
               write(*,2) 'equ(equT, k)', k, equ(equT, k)
               write(*,2) 'lnTdiff', k, lnTdiff
               write(*,2) 'delm', k, delm
               write(*,2) 'dlnT_dm(k)', k, dlnT_dm(k)
               write(*,2) 'dlnPdm', k, dlnPdm
               write(*,2) 'grad_star', k, grad_star
               stop 'equT'
            endif
            
            other = 8.3276707878306060D+40
            other_equ = 1.4606961473706548D-12
            other_FL = 9.4222987519533120D+01
            if (.false. .and. k==1230 .and. abs(s% L(k) - other) > 1d-12*other) then
               FL = L_to_FL(s,s% L(k))
               write(*,*)
               write(*,*)
               write(*,*) 'start numeric differences'
               write(*,2) 'equT', equT
               write(*,*)
               write(*,2) 'equ(equT,k)', k, equ(equT,k)
               write(*,2) 'L', k, s% L(k)
               write(*,2) 'dr/dL', k, (equ(equT,k) - other_equ) / (s% L(k) - other)
               write(*,*)
               write(*,2) 'gradT', k, s% gradT(k)
               write(*,2) 'other gradT', k, 3.8858264995296010D-01
               write(*,2) 'd_grad_star_dL', k, &
                  (s% gradT(k) - 3.8858264995296010D-01) / (s% L(k) - other)
               write(*,2) 'other d_grad_star_dL', k, 3.6200163240075147D-46
               write(*,*)
               write(*,*)
               write(*,*) 'end numeric differences'
               write(*,*)
               
            end if
         
            if (skip_partials) return
         
            d_lnTdiff_dlnTm1 = T00*Tm1/Tpoint**2
            d_lnTdiff_dlnT00 = -d_lnTdiff_dlnTm1

            d_dlnTdm_dLum = dlnPdm*d_grad_star_dL + d_dlnPdm_dL*grad_star
            dr_dL00 = delm*d_dlnTdm_dLum
            dL00_dFL00 = L_to_dL_dFL(s,s% L(k))

            d_dlnTdm_dlnR = dlnPdm*d_grad_star_dlnR + d_dlnPdm_dlnR*grad_star
            dr_dlnR00 = delm*d_dlnTdm_dlnR
      
            call e00(s, xscale, equT, i_FL, k, nvar, dr_dL00*dL00_dFL00)
            
            if (s% lnPgas_flag) then
               d_dlnTdm_dlnT00_const_Pgas = &
                  dlnPdm*d_grad_star_dlnT00_const_Pgas + d_dlnPdm_dlnT00_const_Pgas*grad_star
               dr_dlnT00_const_Pgas = delm*d_dlnTdm_dlnT00_const_Pgas - d_lnTdiff_dlnT00
               call e00(s, xscale, equT, i_lnT, k, nvar, dr_dlnT00_const_Pgas)
               d_dlnTdm_dlnTm1_const_Pgas =  &
                  dlnPdm*d_grad_star_dlnTm1_const_Pgas + d_dlnPdm_dlnTm1_const_Pgas*grad_star
               dr_dlnTm1_const_Pgas = delm*d_dlnTdm_dlnTm1_const_Pgas - d_lnTdiff_dlnTm1
               call em1(s, xscale, equT, i_lnT, k, nvar, dr_dlnTm1_const_Pgas)
            else
               d_dlnTdm_dlnT00 = dlnPdm*d_grad_star_dlnT00 + d_dlnPdm_dlnT00*grad_star
               dr_dlnT00 = delm*d_dlnTdm_dlnT00 - d_lnTdiff_dlnT00
               call e00(s, xscale, equT, i_lnT, k, nvar, dr_dlnT00)
               d_dlnTdm_dlnTm1 = dlnPdm*d_grad_star_dlnTm1 + d_dlnPdm_dlnTm1*grad_star
               dr_dlnTm1 = delm*d_dlnTdm_dlnTm1 - d_lnTdiff_dlnTm1
               call em1(s, xscale, equT, i_lnT, k, nvar, dr_dlnTm1)
            end if

            if (s% do_struct_hydro) then
               call e00(s, xscale, equT, i_lnR, k, nvar, dr_dlnR00)
               if (s% lnPgas_flag) then
                  d_dlnTdm_dlnPgas00_const_T = &
                     dlnPdm*d_grad_star_dlnPgas00_const_T + d_dlnPdm_dlnPgas00_const_T*grad_star
                  dr_dlnPgas00_const_T = delm*d_dlnTdm_dlnPgas00_const_T
                  call e00(s, xscale, equT, i_lnPgas, k, nvar, dr_dlnPgas00_const_T)
                  d_dlnTdm_dlnPgasm1_const_T = &
                     dlnPdm*d_grad_star_dlnPgasm1_const_T + d_dlnPdm_dlnPgasm1_const_T*grad_star
                  dr_dlnPgasm1_const_T = delm*d_dlnTdm_dlnPgasm1_const_T
                  call em1(s, xscale, equT, i_lnPgas, k, nvar, dr_dlnPgasm1_const_T)
               else
                  d_dlnTdm_dlnd00 = dlnPdm*d_grad_star_dlnd00 + d_dlnPdm_dlnd00*grad_star
                  dr_dlnd00 = delm*d_dlnTdm_dlnd00
                  call e00(s, xscale, equT, i_xlnd, k, nvar, dr_dlnd00)
                  d_dlnTdm_dlndm1 = dlnPdm*d_grad_star_dlndm1 + d_dlnPdm_dlndm1*grad_star
                  dr_dlndm1 = delm*d_dlnTdm_dlndm1
                  call em1(s, xscale, equT, i_xlnd, k, nvar, dr_dlndm1)
               end if
            end if
            
            if (.false. .and. k==1230) then
               write(*,*)
               write(*,*)
               write(*,*) 'start analytic differences'
               write(*,2) 'equT', equT
               write(*,*)
               write(*,2) 'equ(equT,k)', k, equ(equT,k)
               write(*,2) 'L', k, s% L(k)
               write(*,*)
               write(*,2) 'dr_dL00', k, dr_dL00
               write(*,2) 'gradT', k, s% gradT(k)
               write(*,2) 'd_grad_star_dL', k, d_grad_star_dL
               write(*,*)
               write(*,*) 'end analytic differences'
               write(*,*)
               
            end if

         end subroutine do1_quasi_HSE_T_eqn
         
         
         subroutine do1_lnE_eqn(k,ierr)
            use hydro_vars, only: get_dlnE_dlnd, get_dlnE_dlnT
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            
            real(fltp) :: dlnE_dlnd, dlnE_dlnT, &
               dlnE_dlnPgas_const_T, dlnE_dlnT_const_Pgas
            
            ierr = 0

            equ(equlnE, k) = lnE(k) - lnE_var(k)
            if (skip_partials) return
            
            call e00(s, xscale, equlnE, i_lnE, k, nvar, -one)

            dlnE_dlnd = get_dlnE_dlnd(s,k)
            dlnE_dlnT = get_dlnE_dlnT(s,k)
            
            if (s% lnPgas_flag) then
               dlnE_dlnPgas_const_T = dlnE_dlnd*s% dlnRho_dlnPgas_const_T(k)
               dlnE_dlnT_const_Pgas = dlnE_dlnT + dlnE_dlnd*s% dlnRho_dlnT_const_Pgas(k)
               call e00(s, xscale, equlnE, i_lnPgas, k, nvar, dlnE_dlnPgas_const_T)
               call e00(s, xscale, equlnE, i_lnT, k, nvar, dlnE_dlnT_const_Pgas)
            else
               call e00(s, xscale, equlnE, i_xlnd, k, nvar, dlnE_dlnd)
               call e00(s, xscale, equlnE, i_lnT, k, nvar, dlnE_dlnT)
            end if
             
         end subroutine do1_lnE_eqn
         
         
         subroutine do1_velocity_eqn(k,ierr)
            use hydro_vars, only: get_dVARDOT_dVAR_pt
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            real(dp) :: d_dlnRdt_dlnRm1, d_dlnRdt_dlnR00, d_dlnRdt_dlnRp1
            real(fltp) :: r, dlnR_dt, cs
            ierr = 0
            dlnR_dt = s% dlnR_dt(k)
            call get_dVARDOT_dVAR_pt(s, k, dt, d_dlnRdt_dlnRm1, d_dlnRdt_dlnR00, d_dlnRdt_dlnRp1)
            r = s% r(k)
            cs = s% csound_init(k)
            equ(equv, k) = (dlnR_dt*r - s% v(k))/cs
            if (skip_partials) return
            call e00(s, xscale, equv, i_vel, k, nvar, -one/cs)     
            call e00(s, xscale, equv, i_lnR, k, nvar, r*(dlnR_dt + d_dlnRdt_dlnR00)/cs)
            if (k > 1 .and. d_dlnRdt_dlnRm1 /= 0) &
               call em1(s, xscale, equv, i_lnR, k, nvar, r*d_dlnRdt_dlnRm1/cs)
            if (k < s% nz .and. d_dlnRdt_dlnRp1 /= 0) &
               call ep1(s, xscale, equv, i_lnR, k, nvar, r*d_dlnRdt_dlnRp1/cs)
         end subroutine do1_velocity_eqn
         
         
         subroutine dummy_eqn(j,i,nzlo,nzhi)
            integer, intent(in) :: j, i, nzlo, nzhi
            integer :: k
            do k=nzlo,nzhi
               equ(j,k) = 0 !s% xs(i,k) - s% xs_pre_pass(i,k)
               if (.not. skip_partials) call e00(s, xscale,j,i,k,nvar,one)
            end do
         end subroutine dummy_eqn


         subroutine check(j_equ, k)
            integer, intent(in) :: j_equ, k
            integer :: j
            logical, parameter :: dump_it = .false.
            if (abs(equ(j_equ, k)) > max_equ(j_equ)) then
               max_equ(j_equ) = abs(equ(j_equ, k))
               kmax_equ(j_equ) = k
            end if
            
            8 format(a30, 2i6, 1pe26.16, 3x, a)
            9 format(a30, 3i6, 1pe26.16, 3x, a)
                        
            if (dump_it .or. is_bad(equ(j_equ, k))) then
               write(*,8) 'equ(j_equ, k)', j_equ, k, equ(j_equ, k), trim(s% nameofequ(j_equ))
               write(*,*) 'skip_partials', skip_partials
            end if
         end subroutine check
         
         
         subroutine dump_equL_info
            integer :: k, j, k0, k1
            include 'formats.dek'
            write(*,*) 'dump_equL_info'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 's% lnE(k)', k, s% lnE(k)
                  write(*,2) 's% lnE_pre_hydro(k)', k, s% lnE_pre_hydro(k)
                  write(*,2) 's% P(k)', k, s% P(k)
                  write(*,2) 's% rho(k)', k, s% rho(k)
                  write(*,2) 's% lnd_pre_hydro(k)', k, s% lnd_pre_hydro(k)
                  write(*,2) 's% T(k)', k, s% T(k)
                  write(*,2) 's% Cv(k)', k, s% Cv(k)
                  write(*,2) 's% dE_dRho(k)', k, s% dE_dRho(k)
                  write(*,2) 's% chiT(k)', k, s% chiT(k)
                  write(*,2) 's% chiRho(k)', k, s% chiRho(k)
                  write(*,2) 's% d_eps_grav_dlnT00(k)', k, s% d_eps_grav_dlnT00(k)
                  write(*,2) 's% d_eps_grav_dlnd00(k)', k, s% d_eps_grav_dlnd00(k)
                  write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               end do
               write(*,*)
            end do
            stop 'dump_equL_info'      
         end subroutine dump_equL_info


         subroutine dump_equ
            integer :: k, j, k0, k1
            include 'formats.dek'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'equ ' // trim(s% nameofequ(j)), k, equ(j, k)
               end do
               write(*,*)
            end do
            stop 'dump_equ'      
         end subroutine dump_equ


         subroutine dump_equ_and_partials
            integer :: k, j, i
            include 'formats.dek'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'equ ' // trim(s% nameofequ(j)), k, equ(j, k)
                  if (s% use_quad_newton) then
#ifdef DBLE
                  do i=1,nvar
                     write(*,2) 'dblk ' // trim(s% nameofequ(j)) // trim(s% nameofvar(i)), &
                        k, s% dblk_dble(j, i, k)
                     if (k > 1) write(*,2) 'lblk ' // trim(s% nameofequ(j)) // trim(s% nameofvar(i)), &
                           k, s% lblk_dble(j, i, k)
                     if (k < s% nz) write(*,2) 'ublk ' // trim(s% nameofequ(j)) // trim(s% nameofvar(i)), &
                           k, s% ublk_dble(j, i, k)
                  end do
#else
                  do i=1,nvar
                     write(*,2) 'dblk ' // trim(s% nameofequ(j)) // trim(s% nameofvar(i)), &
                        k, s% dblk_quad(j, i, k)
                     if (k > 1) write(*,2) 'lblk ' // trim(s% nameofequ(j)) // trim(s% nameofvar(i)), &
                           k, s% lblk_quad(j, i, k)
                     if (k < s% nz) write(*,2) 'ublk ' // trim(s% nameofequ(j)) // trim(s% nameofvar(i)), &
                           k, s% ublk_quad(j, i, k)
                  end do
#endif
               else
               end if
               end do
               write(*,*)
            end do
            stop 'dump_equ'      
         end subroutine dump_equ_and_partials


         subroutine check_everything
            use star_utils, only: std_write_internals_to_file
            integer :: k, j, k0, k1
            include 'formats.dek'
            write(*,*) 'check everything', nvar, nz
            max_equ(:) = 0; kmax_equ(:) = 0
            do k=nzlo,nzhi
               do j=1, nvar
                  call check(j, k)
               end do
            end do
            write(*,*)
            do j=1,nvar
               write(*,3) trim(s% nameofequ(j)), j, kmax_equ(j), max_equ(j)
            end do
            write(*,*)
            ! set k0 and k1 to locations where want more info
            k0 = 1; k1 = 0
            if (k0 <= k1) then
               do k=k0,k1
                  do j=1,nvar
                     write(*,2) 'equ ' // trim(s% nameofequ(j)), k, equ(j, k)
                  end do
                  write(*,*)
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  write(*,2) '1-q(k)', k, 1-s% q(k)
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  write(*,2) 'lnd(k)/ln10', k, s% lnd(k)/ln10
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  write(*,2) 'lnT(k)/ln10', k, s% lnT(k)/ln10
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  do j=1,nvar_chem
                     write(*,2) 'xa_old(j,k) ' // trim(chem_isos% name(s% chem_id(j))), &
                        j, s% xa_old(j,k)
                  end do
               end do
               write(*,*)
               write(*,*)
               do k=k0,k1
                  do j=1,nvar_chem
                     write(*,2) 'xa(j,k) ' // trim(chem_isos% name(s% chem_id(j))), j, s% xa(j,k)
                  end do
               end do
            end if
            write(*,*) 'nz', nz
            write(*,*)
            write(*,*) 'finished check everything'
            !stop 'check_everything'
            write(*,*)
            write(*,*)
            write(*,*)
            call std_write_internals_to_file(s% id, 0)
            
            if (skip_partials) return
            
         end subroutine check_everything
         

      end subroutine eval_equ_for_solver


      subroutine PT_eqns_surf(s, xscale, equ, skip_partials, nvar, ierr)
         use hydro_vars, only: set_Teff_info_for_eqns
         use chem_def
         use atm_def
         use atm_lib, only: atm_option
         use star_utils, only: L_to_dL_dFL
         
         type (star_info), pointer :: s         
         real(fltp), pointer :: xscale(:,:)
         real(fltp), pointer :: equ(:,:)
         logical, intent(in) :: skip_partials
         integer, intent(in) :: nvar
         integer, intent(out) :: ierr
         
         integer :: i_xlnd, i_lnPgas, i_lnT, i_lnR, i_FL, equP, equT
         real(dp) :: r, L, Teff, &
            lnT_surf, dlnTsurf_dL, dlnTsurf_dlnR, dlnTsurf_dlnM, dlnTsurf_dlnkap, &
            lnP_surf, dlnPsurf_dL, dlnPsurf_dlnR, dlnPsurf_dlnM, dlnPsurf_dlnkap
         real(fltp) :: &
            dlnT_bc_dlnd, dlnT_bc_dlnT, dlnT_bc_dlnR, &
            dlnT_bc_dL, dlnP_bc_dlnd, dlnP_bc_dlnT, dlnP_bc_dlnR, dlnP_bc_dL, &
            dlnkap_dlnd, dlnkap_dlnT, dPinv_dlnd, dPinv_dlnT, dP0, dT0, &
            P_surf, T_surf, dlnP_bc_dlnPsurf, dlnT_bc_dlnTsurf, P_bc, T_bc, lnT_bc, lnP_bc, &
            dP0_dlnR, dT0_dlnR, dT0_dlnT, dT0_dlnd, dT0_dL, dlnP_bc_dP0, dlnT_bc_dT0, &
            dlnP_bc_dlnPgas_const_T, dlnP_bc_dlnT_const_Pgas, dlnP_dlnPgas_const_T, &
            dlnP_dlnT_const_Pgas, dlnT_bc_dlnPgas_const_T, dlnT_bc_dlnT_const_Pgas, dL_dFL
         
         include 'formats.dek'
         
         ierr = 0
         
         if (s% L(1) <= 0) then
            ierr = -1
            if (s% report_ierr) write(*, *) 'P_eqn_surf L <= 0', ierr
            return
         end if
         
         call set_Teff_info_for_eqns(s, skip_partials, r, L, Teff, &
            lnT_surf, dlnTsurf_dL, dlnTsurf_dlnR, dlnTsurf_dlnM, dlnTsurf_dlnkap, &
            lnP_surf, dlnPsurf_dL, dlnPsurf_dlnR, dlnPsurf_dlnM, dlnPsurf_dlnkap, &
            ierr)
         if (ierr /= 0) then
            if (s% report_ierr) then
               write(*,*) 'P_eqn_surf: ierr from set_Teff_info_for_eqns'
            end if
            return
         end if

         equP = s% equP
         equT = s% equT

         i_lnPgas = s% i_lnPgas
         i_xlnd = s% i_xlnd
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         i_FL = s% i_FL
         
         ! P_surf and T_surf are at outer boundary of cell 1
         P_surf = exp(lnP_surf)
         T_surf = exp(lnT_surf)
         s% P_surf = P_surf
         s% T_surf = T_surf

         ! offset P and T from outer edge of cell 1 to center of cell 1
         dP0 = s% surf_bc_offset_factor*s% cgrav(1)*s% mstar*s% dm(1)/(8*pi*r**4)
         dT0 = dP0*s% gradT(1)*s% T(1)/s% P(1)
         
         P_bc = P_surf + dP0
         T_bc = T_surf + dT0

         lnP_bc = log(P_bc)
         lnT_bc = log(T_bc)
         
         if (s% do_struct_hydro) then
            equ(equP, 1) = lnP_bc - s% lnP(1)
            if (is_bad(equ(equP, 1))) then
               write(*,1) 'equ(equP, 1)', equ(equP, 1)
               write(*,1) 's% lnP(1)', s% lnP(1)
               write(*,1) 'lnP_bc', lnP_bc
               write(*,1) 'lnP_surf', lnP_surf
               write(*,1) 'P_surf', P_surf
               write(*,1) 'dP0', dP0
               write(*,1) 'r', r
               write(*,1) 's% dm(1)', s% dm(1)
               write(*,1) 's% mstar', s% mstar
               write(*,1) 's% surf_bc_offset_factor', s% surf_bc_offset_factor
               stop 'PT_eqns_surf'
            end if
         else
            equ(equP,1) = 0
         end if
         
         if (s% do_struct_thermo) then
            equ(equT, 1) = lnT_bc - s% lnT(1)        
         else
            equ(equT,1) = 0
         end if

         if (skip_partials) return

         !dT0 = dP0*s% gradT(1)*s% T(1)/s% P(1)
         
         dP0_dlnR = -4*dP0         
         dT0_dlnR = -4*dT0 + dP0*s% d_gradT_dlnR(1)*s% T(1)/s% P(1)
         
         dPinv_dlnT = -s% chiT(1)/s% P(1)
         dT0_dlnT = &
            dT0 + &
            dP0*s% d_gradT_dlnT00(1)*s% T(1)/s% P(1) + &
            dP0*s% gradT(1)*s% T(1)*dPinv_dlnT
         dPinv_dlnd = -s% chiRho(1)/s% P(1)
         dT0_dlnd = &
            dP0*s% d_gradT_dlnd00(1)*s% T(1)/s% P(1) + &
            dP0*s% gradT(1)*s% T(1)*dPinv_dlnd
            
         dT0_dL = dP0*s% d_gradT_dL(1)*s% T(1)/s% P(1)
         
         dlnP_bc_dP0 = 1/P_bc
         dlnT_bc_dT0 = 1/T_bc
         
         dlnP_bc_dlnPsurf = P_surf/P_bc
         dlnT_bc_dlnTsurf = T_surf/T_bc

         dlnkap_dlnd = s% d_opacity_dlnd(1)/s% opacity(1)
         dlnkap_dlnT = s% d_opacity_dlnT(1)/s% opacity(1)

         dlnP_bc_dlnd = dlnP_bc_dlnPsurf*dlnPsurf_dlnkap*dlnkap_dlnd
         dlnP_bc_dlnT = dlnP_bc_dlnPsurf*dlnPsurf_dlnkap*dlnkap_dlnT

         dlnT_bc_dlnT = dlnT_bc_dlnTsurf*dlnTsurf_dlnkap*dlnkap_dlnT + dlnT_bc_dT0*dT0_dlnT
         dlnT_bc_dlnd = dlnT_bc_dlnTsurf*dlnTsurf_dlnkap*dlnkap_dlnd + dlnT_bc_dT0*dT0_dlnd
         
         dL_dFL = L_to_dL_dFL(s,L)

         if (s% do_struct_thermo) then ! temperature eqn
         
            if (s% lnPgas_flag) then
               dlnT_bc_dlnT_const_Pgas = &
                  dlnT_bc_dlnT + dlnT_bc_dlnd*s% dlnRho_dlnT_const_Pgas(1)
               call e00(s, xscale, equT, i_lnT, 1, nvar, dlnT_bc_dlnT_const_Pgas - 1)
            else
               call e00(s, xscale, equT, i_lnT, 1, nvar, dlnT_bc_dlnT - 1)
            end if
            
            dlnT_bc_dL = dlnT_bc_dlnTsurf*dlnTsurf_dL + dlnT_bc_dT0*dT0_dL
            call e00(s, xscale, equT, i_FL, 1, nvar, dlnT_bc_dL*dL_dFL)
            
            if (s% do_struct_hydro) then ! partial of temperature eqn wrt lnR and (lnd or lnPgas)
            
               dlnT_bc_dlnR = dlnT_bc_dlnTsurf*dlnTsurf_dlnR + dlnT_bc_dT0*dT0_dlnR
               call e00(s, xscale, equT, i_lnR, 1, nvar, dlnT_bc_dlnR)

               if (s% lnPgas_flag) then
                  dlnT_bc_dlnPgas_const_T = dlnT_bc_dlnd*s% dlnRho_dlnPgas_const_T(1)
                  call e00(s, xscale, equT, i_lnPgas, 1, nvar, dlnT_bc_dlnPgas_const_T)
               else
                  call e00(s, xscale, equT, i_xlnd, 1, nvar, dlnT_bc_dlnd)
               end if
               
            end if
            
         else ! dummy eqn
         
            call e00(s,xscale,equT,i_lnT,1,nvar,one)
            
         end if

         if (s% do_struct_hydro) then ! pressure eqn
            
            if (s% lnPgas_flag) then
               dlnP_bc_dlnPgas_const_T = dlnP_bc_dlnd*s% dlnRho_dlnPgas_const_T(1)
               dlnP_dlnPgas_const_T = s% Pgas(1)/s% P(1)
               call e00(s, xscale, equP, i_lnPgas, 1, nvar, &
                  dlnP_bc_dlnPgas_const_T - dlnP_dlnPgas_const_T)
            else
               call e00(s, xscale, equP, i_xlnd, 1, &
                  nvar, dlnP_bc_dlnd - s% chiRho(1))
            end if
            
            dlnP_bc_dlnR = dlnP_bc_dlnPsurf*dlnPsurf_dlnR + dlnP_bc_dP0*dP0_dlnR
            call e00(s, xscale, equP, i_lnR, 1, nvar, dlnP_bc_dlnR)
            
            if (s% do_struct_thermo) then ! partial of pressure eqn wrt lnT
               
               if (s% lnPgas_flag) then
                  dlnP_bc_dlnT_const_Pgas = &
                     dlnP_bc_dlnT + dlnP_bc_dlnd*s% dlnRho_dlnT_const_Pgas(1)
                  dlnP_dlnT_const_Pgas = 4*s% Prad(1)/s% P(1)
                  call e00(s, xscale, equP, i_lnT, 1, &
                  nvar, dlnP_bc_dlnT_const_Pgas - dlnP_dlnT_const_Pgas)
               else
                  call e00(s, xscale, equP, i_lnT, 1, &
                  nvar, dlnP_bc_dlnT - s% chiT(1))
               end if
               
               dlnP_bc_dL = dlnP_bc_dlnPsurf*dlnPsurf_dL
               call e00(s, xscale, equP, i_FL, 1, nvar, dlnP_bc_dL*dL_dFL)
               
            end if
            
         else ! dummy eqn
            
            if (s% lnPgas_flag) then
               call e00(s,xscale,equP,i_lnPgas,1,nvar,one)
            else
               call e00(s,xscale,equP,i_xlnd,1,nvar,one)
            end if
            
         end if

      end subroutine PT_eqns_surf


      subroutine eval_grad_star_info( &
            s, k, grad_star, d_grad_star_dL, d_grad_star_dlnR, &
            d_grad_star_dlnd00, d_grad_star_dlndm1, &
            d_grad_star_dlnT00, d_grad_star_dlnTm1, &
            d_grad_star_dlnT00_const_Pgas, d_grad_star_dlnTm1_const_Pgas, &
            d_grad_star_dlnPgas00_const_T, d_grad_star_dlnPgasm1_const_T, &
            ierr)
         type (star_info), pointer :: s 
         integer, intent(in) :: k        
         real(fltp), intent(out) :: grad_star, d_grad_star_dL, d_grad_star_dlnR, &
            d_grad_star_dlnd00, d_grad_star_dlndm1, d_grad_star_dlnT00, d_grad_star_dlnTm1, &
            d_grad_star_dlnT00_const_Pgas, d_grad_star_dlnTm1_const_Pgas, &
            d_grad_star_dlnPgas00_const_T, d_grad_star_dlnPgasm1_const_T
         integer, intent(out) :: ierr         
         include 'formats.dek'
              
         ierr = 0      
         
         grad_star = s% gradT(k)  
          
         d_grad_star_dL = s% d_gradT_dL(k)   
         d_grad_star_dlnR = s% d_gradT_dlnR(k)

         d_grad_star_dlnd00 = s% d_gradT_dlnd00(k)
         d_grad_star_dlndm1 = s% d_gradT_dlndm1(k)   
         d_grad_star_dlnT00 = s% d_gradT_dlnT00(k)
         d_grad_star_dlnTm1 = s% d_gradT_dlnTm1(k)         
         
         if (s% lnPgas_flag) then
            d_grad_star_dlnPgas00_const_T = &
               d_grad_star_dlnd00*s% dlnRho_dlnPgas_const_T(k)
            d_grad_star_dlnT00_const_Pgas = &
               d_grad_star_dlnT00 + d_grad_star_dlnd00*s% dlnRho_dlnT_const_Pgas(k)
            if (k > 1) then
               d_grad_star_dlnPgasm1_const_T = &
                  d_grad_star_dlndm1*s% dlnRho_dlnPgas_const_T(k-1)
               d_grad_star_dlnTm1_const_Pgas = &
                  d_grad_star_dlnTm1 + d_grad_star_dlndm1*s% dlnRho_dlnT_const_Pgas(k-1)
            else
               d_grad_star_dlnPgasm1_const_T = 0
               d_grad_star_dlnTm1_const_Pgas = 0
            end if
         end if

      end subroutine eval_grad_star_info

            

      subroutine do_chem_eqns( &
            s, xscale, nvar, equchem1, nvar_chem, skip_partials, dt_in, equ, ierr)
         type (star_info), pointer :: s
         real(fltp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: nvar, equchem1, nvar_chem
         logical, intent(in) :: skip_partials
         real(dp), intent(in) :: dt_in
         real(fltp), intent(out) :: equ(:,:)
         integer, intent(out) :: ierr         
         integer :: k, op_err
         include 'formats.dek'         
         ierr = 0
!$OMP PARALLEL DO PRIVATE(k,op_err)
         do k = 1, s% nz
            if (ierr == 0) call do1_chem_eqns( &
               s, xscale, k, nvar, equchem1, nvar_chem, skip_partials, dt_in, equ, op_err)
            if (op_err /= 0) ierr = op_err      
         end do
!$OMP END PARALLEL DO
      end subroutine do_chem_eqns

      subroutine do1_chem_eqns( &
            s, xscale, k, nvar, equchem1, nvar_chem, skip_partials, dt_in, equ, ierr)
         use chem_def
         use num_lib, only: safe_log10
         use net_lib, only: show_net_contents, show_net_reactions, &
            show_net_params, get_reaction_id_table_ptr
         use rates_def, only: reaction_Name, i_rate
         type (star_info), pointer :: s
         real(fltp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: k, nvar, equchem1, nvar_chem
         logical, intent(in) :: skip_partials
         real(dp), intent(in) :: dt_in
         real(fltp), intent(out) :: equ(:,:)
         integer, intent(out) :: ierr

         integer :: nz, i_xlnd, i_lnPgas, i_lnT, i_lnR, op_err
         real(dp), pointer, dimension(:) :: sig
         real(dp) :: dt
         logical :: hit_limit_for_x_expected, ode_flag
         integer :: j, i, jj, ii
         real(fltp) :: dx_expected_dxa, dx_expected, dx_actual, x_actual, x_expected, &
            dq, dm, dx_burning, dx_mixing, dequ, dxdt_nuc, dxdt_mix, &
            xprev, sum_xprev, sum_dxdt_nuc, dx_expected_dlnd, dx_expected_dlnT
         real(fltp) :: d_dxdtmix_dx00, d_dxdtmix_dxm1, d_dxdtmix_dxp1, &
            d_dxmix_dx00, d_dxmix_dxm1, d_dxmix_dxp1
         integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id
         real(fltp) :: &
            x00, xp1, xm1, dx00, dxp1, sigavg, d, w_out00, w_outp1, w_out, w_in, &
            sig00, sigp1, flux00, dflux00_dxm1, dflux00_dx00, &
            fluxp1, dfluxp1_dx00, dfluxp1_dxp1, eqn_scale, d_dxdt_dx, &
            dequ_dlnd, dequ_dlnT, dequ_dlnPgas_const_T, dequ_dlnT_const_Pgas


         include 'formats.dek'
         
         ierr = 0
         
         dt = dt_in
         
         nz = s% nz
         i_xlnd = s% i_xlnd
         i_lnPgas = s% i_lnPgas
         i_lnT = s% i_lnT
         i_lnR = s% i_lnR
         sig => s% sig
         
         dq = s% dq(k)
         dm = s% dm(k)
         w_out00 = 0
         w_outp1 = 1
                  
         sum_xprev = 0
         sum_dxdt_nuc = 0
         d_dxdtmix_dxp1 = 0
         d_dxdtmix_dxm1 = 0
         d_dxdtmix_dx00 = 0
         d_dxmix_dxp1 = 0
         d_dxmix_dxm1 = 0
         d_dxmix_dx00 = 0

         sig00 = sig(k)
   
         if (k < s% nz) then
            sigp1 = sig(k+1)
         else
            sigp1 = 0
         end if
         
         if (.not. s% do_mix) then
         
            dx_mixing = 0

         else
      
            if (k > 1) then
               dflux00_dxm1 = -sig00
               dflux00_dx00 = sig00
            else
               dflux00_dxm1 = 0
               dflux00_dx00 = 0
            end if
      
            if (k < s% nz) then
               dfluxp1_dx00 = -sigp1
               dfluxp1_dxp1 = sigp1
            else
               dfluxp1_dx00 = 0
               dfluxp1_dxp1 = 0
            end if
         
            d_dxdtmix_dx00 = (dfluxp1_dx00 - dflux00_dx00)/dm
            d_dxdtmix_dxm1 = -dflux00_dxm1/dm
            d_dxdtmix_dxp1 = dfluxp1_dxp1/dm

            d_dxmix_dx00 = d_dxdtmix_dx00*dt
            d_dxmix_dxm1 = d_dxdtmix_dxm1*dt
            d_dxmix_dxp1 = d_dxdtmix_dxp1*dt
            
         end if

         do j=1,nvar_chem ! composition equation for species j in cell k
      
            i = equchem1+j-1
            ode_flag = (.not. s% doing_hydro_newton) .and. s% ode_var(i)
         
            if (.not. s% do_burn) then
               dxdt_nuc = 0
            else
               dxdt_nuc = s% dxdt_nuc(j,k)
            end if
            
            x00 = s% xa(j,k)
            if (.not. s% do_mix) then
               dxdt_mix = 0
            else
               if (k > 1) then
                  xm1 = s% xa(j,k-1)
                  dx00 = xm1 - x00
                  flux00 = -sig00*dx00
               else
                  flux00 = 0
               end if
         
               if (k < s% nz) then
                  xp1 = s% xa(j,k+1)
                  dxp1 = x00 - xp1
                  fluxp1 = -sigp1*dxp1
               else
                  fluxp1 = 0
               end if
               dxdt_mix = (fluxp1 - flux00)/dm
            end if
            
            if (ode_flag) then ! equ gets dxdt expected
               
               equ(i,k) = dxdt_nuc + dxdt_mix

               if (is_bad(equ(i,k))) then
                  ierr = -1
                  return

                  write(*,2) 'equ(i,k) ' // trim(chem_isos% name(s% chem_id(j))), k, equ(i,k)
                  write(*,2) 'dxdt_nuc', k, dxdt_nuc
                  write(*,2) 'fluxp1', k, fluxp1
                  write(*,2) 'flux00', k, flux00
                  write(*,2) 'dm', k, dm
                  stop 'chem eqn'
               end if
               
               if (.false. .and. k == 1440 .and. j == 3) then
                  write(*,*)
                  write(*,1) 'chem_eqn'
                  write(*,2) 'equ(i,k) ' // trim(chem_isos% name(s% chem_id(j))), k, equ(i,k)
                  write(*,2) 'x00', k, x00
                  write(*,2) 'dxdt_nuc', k, dxdt_nuc
                  write(*,2) 'dxdt_mix', k, dxdt_mix
                  write(*,2) 'd_dxdt_dx', k, s% d_dxdt_dx(j,j,k)
                  write(*,*)
               end if   
         
               if (skip_partials) cycle
            
               if (s% do_burn) then
            
                  do jj=1,nvar_chem
                     ii = equchem1+jj-1
                     d_dxdt_dx = s% d_dxdt_dx(j,jj,k)
                     call e00(s, xscale, i, ii, k, nvar, d_dxdt_dx)
                  end do

                  dequ_dlnd = s% dxdt_drho(j,k)*s% rho(k)
                  dequ_dlnT = s% dxdt_dT(j,k)*s% T(k)
               
                  if (s% do_struct_hydro) then
                     if (s% lnPgas_flag) then
                        dequ_dlnPgas_const_T = dequ_dlnd*s% dlnRho_dlnPgas_const_T(k)
                        call e00(s, xscale, i, i_lnPgas, k, nvar, dequ_dlnPgas_const_T)
                     else
                        call e00(s, xscale, i, i_xlnd, k, nvar, dequ_dlnd)
                     end if
                  end if
               
                  if (s% do_struct_thermo) then
                     if (s% lnPgas_flag) then
                        dequ_dlnT_const_Pgas = &
                           dequ_dlnT + dequ_dlnd*s% dlnRho_dlnT_const_Pgas(k)
                        call e00(s, xscale, i, i_lnT, k, nvar, dequ_dlnT_const_Pgas)
                     else
                        call e00(s, xscale, i, i_lnT, k, nvar, dequ_dlnT)
                     end if
                  end if

               end if
            
               if (s% do_mix) then
               
                  if (.false. .and. k == 1440 .and. j == 3) then
                     write(*,*)
                     write(*,1) 'chem_eqn partials'
                     write(*,3) 'd_dxdtmix_dx00', j, k, d_dxdtmix_dx00
                     write(*,3) 'dfluxp1_dx00', j, k, dfluxp1_dx00
                     write(*,3) 'dflux00_dx00', j, k, dflux00_dx00
                     write(*,2) 'sig00', k, sig00
                     write(*,2) 'sigp1', k, sigp1
                     write(*,2) 'dm', k, dm
                     write(*,*)
                  end if   


                  call e00(s, xscale, i, i, k, nvar, d_dxdtmix_dx00)
                  if (k > 1) call em1(s, xscale, i, i, k, nvar, d_dxdtmix_dxm1)
                  if (k < nz) call ep1(s, xscale, i, i, k, nvar, d_dxdtmix_dxp1)
               end if



            else ! not ode_flag
                  
               dx_burning = dxdt_nuc*dt
               dx_mixing = dxdt_mix*dt
               eqn_scale = xscale(i,k)
            
               xprev = s% xa_pre_hydro(j,k)
               if (xprev < 0) then
                  write(*,3) 's% xa_pre_hydro(j,k)', j, k, s% xa_pre_hydro(j,k)
                  ierr = -1
                  return
               end if
               x_actual = x00
               dx_actual = x_actual - xprev
            
               dx_expected = dx_mixing + dx_burning
               x_expected = xprev + dx_expected
            
               hit_limit_for_x_expected = .false.
               !hit_limit_for_x_expected = (x_expected < 0)  ! this causes very bad convergence
            
               if (hit_limit_for_x_expected) then
                  !write(*,2) 'bad x_expected ' // trim(chem_isos% name(s% chem_id(j))), k, x_expected
                  x_expected = 0
               end if

               equ(i,k) = (x_expected - x_actual)/eqn_scale            

               if (is_bad(equ(i,k))) then
                  write(*,2) 'equ(i,k) ' // trim(chem_isos% name(s% chem_id(j))), k, equ(i,k)
                  write(*,2) 'dx_expected', k, dx_expected
                  write(*,2) 'dx_actual', k, dx_actual
                  write(*,2) 'eqn_scale', k, eqn_scale
                  stop 'chem eqn'
               end if
               
               !if (k == 321 .and. j == 4 .and. equ(i,k) > 1d4) then
               if (.false. .and. s% hydro_call_number == 12266 .and. j == 4 .and. k==276) then
               !if (.true. .and. equ(i,k) > 1d9 .and. s% hydro_call_number == 12266) then
                  write(*,2) 'equ(i,k) ' // trim(chem_isos% name(s% chem_id(j))), k, equ(i,k)
                  write(*,2) 'dx_expected', k, dx_expected
                  write(*,2) 'dx_actual', k, dx_actual
                  write(*,2) 'dx_burning', k, dx_burning
                  write(*,2) 'dx_mixing', k, dx_mixing
                  write(*,2) 'xprev', k, xprev
                  write(*,2) 'x_actual', k, x_actual
                  write(*,2) 'x_expected', k, x_expected
                  write(*,2) 'eqn_scale', k, eqn_scale
                  write(*,2) 'logT', k, s% lnT(k)/ln10
                  write(*,2) 'logRho', k, s% lnd(k)/ln10
                  write(*,2) 'prev logT', k, s% lnT_pre_hydro(k)/ln10
                  write(*,2) 'prev logRho', k, s% lnd_pre_hydro(k)/ln10
                  write(*,*)
                  if (equ(i,k) > 1d9) stop 'chem eqn'
               end if
                  
            
               if (is_bad(equ(i,k))) then
                  ierr = -1
                  return
               end if
         
               if (skip_partials) cycle
            
               if (.not. ode_flag) then ! jacobian term for x_actual
                  call e00(s, xscale, i, i, k, nvar, -1d0/eqn_scale)
               end if
            
               if (hit_limit_for_x_expected) cycle 
                  ! at limit, x_expected is constant, so partials are 0

               ! jacobian terms for x_expected
            
               if (s% do_burn) then
            
                  do jj=1,nvar_chem
                     ii = equchem1+jj-1
                     dx_expected_dxa = dt*s% d_dxdt_dx(j,jj,k)
                     dequ = dx_expected_dxa/eqn_scale
                     call e00(s, xscale, i, ii, k, nvar, dequ)
                  end do
         
                  dx_expected_dlnd = dt*s% dxdt_drho(j,k)*s% rho(k)
                  dequ_dlnd = dx_expected_dlnd/eqn_scale
               
                  if (s% do_struct_hydro) then
                     if (s% lnPgas_flag) then
                        dequ_dlnPgas_const_T = dequ_dlnd*s% dlnRho_dlnPgas_const_T(k)
                        call e00(s, xscale, i, i_lnPgas, k, nvar, dequ_dlnPgas_const_T)
                     else
                        call e00(s, xscale, i, i_xlnd, k, nvar, dequ_dlnd)
                     end if
                  end if
               
                  if (s% do_struct_thermo) then
                     dx_expected_dlnT = dt*s% dxdt_dT(j,k)*s% T(k)
                     dequ_dlnT = dx_expected_dlnT/eqn_scale
                     if (s% lnPgas_flag) then
                        dequ_dlnT_const_Pgas = &
                           dequ_dlnT + dequ_dlnd*s% dlnRho_dlnT_const_Pgas(k)
                        call e00(s, xscale, i, i_lnT, k, nvar, dequ_dlnT_const_Pgas)
                     else
                        call e00(s, xscale, i, i_lnT, k, nvar, dequ_dlnT)
                     end if
                  end if

               end if
            
               if (s% do_mix) then
            
                  dx_expected_dxa = d_dxmix_dx00
                  dequ = dx_expected_dxa/eqn_scale
                  call e00(s, xscale, i, i, k, nvar, dequ)
               
                  if (k > 1) then
                     dx_expected_dxa = d_dxmix_dxm1
                     dequ = dx_expected_dxa/eqn_scale
                     call em1(s, xscale, i, i, k, nvar, dequ)
                  end if
               
                  if (k < nz) then
                     dx_expected_dxa = d_dxmix_dxp1
                     dequ = dx_expected_dxa/eqn_scale
                     call ep1(s, xscale, i, i, k, nvar, dequ)
                  end if
               
               end if
               
            end if
         
         end do

      end subroutine do1_chem_eqns
      
      
      ! e00(i,j,k) is partial of equ(i,k) wrt var(j,k)
      subroutine e00(s,xscale,i,j,k,nvar,v)
         use utils_lib, only: is_bad
         use num_def, only: &
            block_tridiag_dble_matrix_type, block_tridiag_quad_matrix_type
         type (star_info), pointer :: s
         real(fltp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: i, j, k, nvar
         real(fltp), intent(in) :: v
         integer :: b, q, v00
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         if (dbg) then
            if (is_bad(v)) then
               write(*,4) 'e00(i,j,k)', i, j, k, v
            end if
         end if
         if (j == 0) then
            write(*,*) 'called e00 with j=0 for ' // s% nameofequ(i)
            write(*,*) 's% lnPgas_flag', s% lnPgas_flag
            write(*,*) 's% i_xlnd', s% i_xlnd
            write(*,*) 's% i_lnPgas', s% i_lnPgas
            stop 'e00'
         end if
         if (s% hydro_matrix_type == block_tridiag_dble_matrix_type .or. &
             s% hydro_matrix_type == block_tridiag_quad_matrix_type) then
#ifdef DBLE
            s% dblk_dble(i,j,k) = s% dblk_dble(i,j,k) + v*xscale(j,k)
#else
            s% dblk_quad(i,j,k) = s% dblk_quad(i,j,k) + v*xscale(j,k)
#endif
            return
         end if
         b = nvar*(k-1)
         q = s% idiag + b + i
         v00 = b + j
         s% jacobian(q-v00,v00) = s% jacobian(q-v00,v00) + v*xscale(j,k)
      end subroutine e00

      
      ! em1(i,j,k) is partial of equ(i,k) wrt var(j,k-1)
      subroutine em1(s,xscale,i,j,k,nvar,v)
         use utils_lib, only: is_bad
         use num_def, only: &
            block_tridiag_dble_matrix_type, block_tridiag_quad_matrix_type
         type (star_info), pointer :: s
         real(fltp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: i, j, k, nvar
         real(fltp), intent(in) :: v
         integer :: b, q, vm1
         logical, parameter :: dbg = .false.
         if (k == 1) return
         include 'formats.dek'
         if (dbg) then
            if (is_bad(v)) then
               write(*,4) 'em1(i,j,k)', i, j, k, v
            end if
         end if
         if (s% hydro_matrix_type == block_tridiag_dble_matrix_type .or. &
             s% hydro_matrix_type == block_tridiag_quad_matrix_type) then
#ifdef DBLE
            s% lblk_dble(i,j,k) = s% lblk_dble(i,j,k) + v*xscale(j,k-1)
#else
            s% lblk_quad(i,j,k) = s% lblk_quad(i,j,k) + v*xscale(j,k-1)
#endif
            return
         end if
         b = nvar*(k-1)
         q = s% idiag + b + i
         vm1 = b + j - nvar
         s% jacobian(q-vm1,vm1) = s% jacobian(q-vm1,vm1) + v*xscale(j,k-1)
      end subroutine em1
      
      
      ! ep1(i,j,k) is partial of equ(i,k) wrt var(j,k+1)
      subroutine ep1(s,xscale,i,j,k,nvar,v)
         use utils_lib, only: is_bad
         use num_def, only: &
            block_tridiag_dble_matrix_type, block_tridiag_quad_matrix_type
         type (star_info), pointer :: s
         real(fltp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: i, j, k, nvar
         real(fltp), intent(in) :: v
         integer :: b, q, vp1
         logical, parameter :: dbg = .false.
         if (k == s% nz) return
         include 'formats.dek'
         if (dbg) then
            if (is_bad(v)) then
               write(*,4) 'ep1(i,j,k)', i, j, k, v
               stop 'debug: ep1'
            end if
         end if
         if (s% hydro_matrix_type == block_tridiag_dble_matrix_type .or. &
             s% hydro_matrix_type == block_tridiag_quad_matrix_type) then
#ifdef DBLE
            s% ublk_dble(i,j,k) = s% ublk_dble(i,j,k) + v*xscale(j,k+1)
#else
            s% ublk_quad(i,j,k) = s% ublk_quad(i,j,k) + v*xscale(j,k+1)
#endif
            return
         end if
         b = nvar*(k-1)
         q = s% idiag + b + i
         vp1 = b + j + nvar
         s% jacobian(q-vp1,vp1) = s% jacobian(q-vp1,vp1) + v*xscale(j,k+1)
      end subroutine ep1


      subroutine do_dlnP_dm(s, k, dt, &
            d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
            d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
            d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
            d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
            d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &         
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
            d_dlnPdm_dL00, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
            ierr)
         use hydro_vars, only: get_dVARDOT_dVAR_pt
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(dp), intent(in) :: dt
         real(fltp), intent(out) :: &
            d_dlnPdm_dlnRp1, d_dlnPdm_dlnR00, d_dlnPdm_dlnRm1, &
            d_dlnPdm_dlnq, d_dlnPdm_dlndqm1, d_dlnPdm_dlndq00, &
            d_dlnPdm_dvelp1, d_dlnPdm_dvel00, d_dlnPdm_dvelm1, &
            d_dlnPdm_dlndp1, d_dlnPdm_dlnd00, d_dlnPdm_dlndm1, &
            d_dlnPdm_dlnTp1, d_dlnPdm_dlnT00, d_dlnPdm_dlnTm1, &         
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, & 
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, & 
            d_dlnPdm_dL00, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas
         integer, intent(out) :: ierr
         
         real(fltp) :: &
            m, dv_dt, dlnP_dm, fac, &
            rPterm, accel, d_rPterm_dlnR, dlnP_dm_accel_term, d_accel_dlnR, &
            d_rPterm_dlnd00, d_rPterm_dlnT00, d_rPterm_dlndm1, d_rPterm_dlnTm1, &
            d_rPterm_dlnPgas00_const_T, d_rPterm_dlnT00_const_Pgas, &
            d_rPterm_dlnPgasm1_const_T, d_rPterm_dlnTm1_const_Pgas, &
            twoGmrc2, invfac, gr_factor, dtwoGmrc2_dlnR, d_invfac_dlnR, d_gr_factor_dlnR00, &
            d_conv_dP_term_dlnPgas00_const_T, d_conv_dP_term_dlnPgasm1_const_T, &
            d_conv_dP_term_dlnT00_const_Pgas, d_conv_dP_term_dlnTm1_const_Pgas, &
            d_accel_dlnR00, d_accel_dlnRp1, d_accel_dlnRm1
         real(dp) :: d_dvardt_dvarm1, d_dvardt_dvar00, d_dvardt_dvarp1
            
         logical :: dbg

         include 'formats.dek'
         
         ierr = 0
         
         dbg = .false.

         s% dlnP_dm(k) = 0
         dlnP_dm = 0
         d_dlnPdm_dlnRp1 = 0
         d_dlnPdm_dlnR00 = 0
         d_dlnPdm_dlnRm1 = 0
         d_dlnPdm_dlnq = 0
         d_dlnPdm_dlndqm1 = 0
         d_dlnPdm_dlndq00 = 0
         d_dlnPdm_dvelp1 = 0
         d_dlnPdm_dvel00 = 0
         d_dlnPdm_dvelm1 = 0
         d_dlnPdm_dlndp1 = 0
         d_dlnPdm_dlnd00 = 0
         d_dlnPdm_dlndm1 = 0
         d_dlnPdm_dlnTp1 = 0
         d_dlnPdm_dlnT00 = 0
         d_dlnPdm_dlnTm1 = 0
         d_dlnPdm_dlnPgas00_const_T = 0
         d_dlnPdm_dlnT00_const_Pgas = 0
         d_dlnPdm_dlnPgasm1_const_T = 0
         d_dlnPdm_dlnTm1_const_Pgas = 0
         d_dlnPdm_dL00 = 0

         call eval_dlnPdm_qhse(s, k, m, &
            dlnP_dm, &
            d_dlnPdm_dlnR00, d_dlnPdm_dlnq, d_dlnPdm_dL00, &
            d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
            d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
            ierr)
         if (ierr /= 0) return
         s% dlnP_dm(k) = dlnP_dm

         d_dlnPdm_dlnRm1 = 0

         if (is_bad_num(s% dlnP_dm(k))) then
            write(*,*) 'k', k
            write(*,1) 'dlnP_dm(k)', s% dlnP_dm(k)
            write(*,*)
            ierr = -1
            return
         end if
         
         if (s% conv_dP_term(k) /= 0 .and.  s% conv_dP_term_factor > 0) then 
            ! include effect of convective turbulence on pressure gradient
            
            fac = 1 + s% conv_dP_term_factor*s% conv_dP_term(k)
            s% dlnP_dm(k) = s% dlnP_dm(k)*fac
                                
            if (s% lnPgas_flag) then
               d_conv_dP_term_dlnPgas00_const_T = &
                  s% d_conv_dP_term_dlnd00(k)*s% dlnRho_dlnPgas_const_T(k)
               d_conv_dP_term_dlnPgasm1_const_T = &
                  s% d_conv_dP_term_dlndm1(k)*s% dlnRho_dlnPgas_const_T(k-1)
               d_conv_dP_term_dlnT00_const_Pgas = &
                  s% d_conv_dP_term_dlnT00(k) + &
                  s% d_conv_dP_term_dlnd00(k)*s% dlnRho_dlnT_const_Pgas(k)
               d_conv_dP_term_dlnTm1_const_Pgas = &
                  s% d_conv_dP_term_dlnTm1(k) + &
                  s% d_conv_dP_term_dlndm1(k)*s% dlnRho_dlnT_const_Pgas(k-1)
               d_dlnPdm_dlnPgas00_const_T = d_dlnPdm_dlnPgas00_const_T*fac + &
                  s% dlnP_dm(k)*s% conv_dP_term_factor*d_conv_dP_term_dlnPgas00_const_T
               d_dlnPdm_dlnT00_const_Pgas = d_dlnPdm_dlnT00_const_Pgas*fac + &
                  s% dlnP_dm(k)*s% conv_dP_term_factor*d_conv_dP_term_dlnT00_const_Pgas
               d_dlnPdm_dlnPgasm1_const_T = d_dlnPdm_dlnPgasm1_const_T*fac + &
                  s% dlnP_dm(k)*s% conv_dP_term_factor*d_conv_dP_term_dlnPgasm1_const_T
               d_dlnPdm_dlnTm1_const_Pgas = d_dlnPdm_dlnTm1_const_Pgas*fac + &
                  s% dlnP_dm(k)*s% conv_dP_term_factor*d_conv_dP_term_dlnTm1_const_Pgas
            else
               d_dlnPdm_dlnd00 = d_dlnPdm_dlnd00*fac + &
                  s% dlnP_dm(k)*s% conv_dP_term_factor*s% d_conv_dP_term_dlnd00(k)
               d_dlnPdm_dlnT00 = d_dlnPdm_dlnT00*fac + &
                  s% dlnP_dm(k)*s% conv_dP_term_factor*s% d_conv_dP_term_dlnT00(k)
               d_dlnPdm_dlndm1 = d_dlnPdm_dlndm1*fac + &
                  s% dlnP_dm(k)*s% conv_dP_term_factor*s% d_conv_dP_term_dlndm1(k)
               d_dlnPdm_dlnTm1 = d_dlnPdm_dlnTm1*fac + &
                  s% dlnP_dm(k)*s% conv_dP_term_factor*s% d_conv_dP_term_dlnTm1(k)
            end if
            
            d_dlnPdm_dlnR00 = d_dlnPdm_dlnR00*fac + &
               s% dlnP_dm(k)*s% conv_dP_term_factor*s% d_conv_dP_term_dlnR(k)
            d_dlnPdm_dL00 = d_dlnPdm_dL00*fac + &
               s% dlnP_dm(k)*s% conv_dP_term_factor*s% d_conv_dP_term_dL(k)

         end if
         
         if (s% use_gr_factors) then ! GR gravity factor = 1/sqrt(1-2Gm/(rc^2))
         
            twoGmrc2 = 2*s% cgrav(k)*s% m(k)/(s% r(k)*clight**2)
            invfac = sqrt(1d0 - twoGmrc2)
            gr_factor = 1d0/invfac
            
            dtwoGmrc2_dlnR = -twoGmrc2
            d_invfac_dlnR = -dtwoGmrc2_dlnR/(2*invfac)
            d_gr_factor_dlnR00 = -d_invfac_dlnR/invfac**2
            
            if (dbg) then
               write(*,2) 'std s% dlnP_dm(k)', k, s% dlnP_dm(k)
               write(*,2) 'gr_factor', k, gr_factor
               write(*,2) 'new s% dlnP_dm(k)', k, s% dlnP_dm(k)*gr_factor
               !write(*,2) '', k, 
               !write(*,2) '', k, 
               !write(*,2) '', k, 
            end if

            s% dlnP_dm(k) = s% dlnP_dm(k)*gr_factor                        
            d_dlnPdm_dlnR00 = d_dlnPdm_dlnR00*gr_factor + s% dlnP_dm(k)*d_gr_factor_dlnR00
            
         end if

         if (s% v_flag .or. s% use_artificial_v_damping) then
            
            fac = s% accel_factor
            
            if (s% v_flag) then
               dv_dt = s% dv_dt(k)
            else ! artificial_v_damping following Arnett, ApJSS, 35:145-159, 1977, section IV.
               dv_dt = s% r(k)*s% dlnR_dt(k)/dt
            end if
            accel = dv_dt
            call get_dVARDOT_dVAR_pt(s, k, dt, d_dvardt_dvarm1, d_dvardt_dvar00, d_dvardt_dvarp1)
            
            rPterm = 1/(4 * pi * s% r(k)**2 * Ppoint)            
            dlnP_dm_accel_term = -fac*accel*rPterm
            
            s% dlnP_dm(k) = s% dlnP_dm(k) + dlnP_dm_accel_term
            
            if (s% lnPgas_flag) then
               d_rPterm_dlnPgas00_const_T = -rPterm*dPpoint_dlnPgas00_const_T/Ppoint
               d_rPterm_dlnT00_const_Pgas = -rPterm*dPpoint_dlnT00_const_Pgas/Ppoint
               d_rPterm_dlnPgasm1_const_T = -rPterm*dPpoint_dlnPgasm1_const_T/Ppoint
               d_rPterm_dlnTm1_const_Pgas = -rPterm*dPpoint_dlnTm1_const_Pgas/Ppoint
            else
               d_rPterm_dlnd00 = -rPterm*dPpoint_dlnd00/Ppoint
               d_rPterm_dlnT00 = -rPterm*dPpoint_dlnT00/Ppoint
               d_rPterm_dlndm1 = -rPterm*dPpoint_dlndm1/Ppoint
               d_rPterm_dlnTm1 = -rPterm*dPpoint_dlnTm1/Ppoint
            end if
            d_rPterm_dlnR = -2*rPterm

            if (s% lnPgas_flag) then
               d_dlnPdm_dlnPgas00_const_T = &
                  d_dlnPdm_dlnPgas00_const_T - fac*accel*d_rPterm_dlnPgas00_const_T
               d_dlnPdm_dlnT00_const_Pgas = &
                  d_dlnPdm_dlnT00_const_Pgas - fac*accel*d_rPterm_dlnT00_const_Pgas
               d_dlnPdm_dlnPgasm1_const_T = &
                  d_dlnPdm_dlnPgasm1_const_T - fac*accel*d_rPterm_dlnPgasm1_const_T
               d_dlnPdm_dlnTm1_const_Pgas = &
                  d_dlnPdm_dlnTm1_const_Pgas - fac*accel*d_rPterm_dlnTm1_const_Pgas
            else
               d_dlnPdm_dlnd00 = d_dlnPdm_dlnd00 - fac*accel*d_rPterm_dlnd00
               d_dlnPdm_dlnT00 = d_dlnPdm_dlnT00 - fac*accel*d_rPterm_dlnT00
               d_dlnPdm_dlndm1 = d_dlnPdm_dlndm1 - fac*accel*d_rPterm_dlndm1
               d_dlnPdm_dlnTm1 = d_dlnPdm_dlnTm1 - fac*accel*d_rPterm_dlnTm1
            end if

            d_dlnPdm_dlnR00 = d_dlnPdm_dlnR00 - fac*accel*d_rPterm_dlnR
            
            if (s% v_flag) then
               d_dlnPdm_dvel00 = -fac*d_dvardt_dvar00*rPterm
               d_dlnPdm_dvelp1 = -fac*d_dvardt_dvarp1*rPterm
               d_dlnPdm_dvelm1 = -fac*d_dvardt_dvarm1*rPterm
            else ! artificial_v_damping
               d_accel_dlnR00 = s% r(k)*d_dvardt_dvar00/dt + accel
               d_accel_dlnRp1 = s% r(k)*d_dvardt_dvarp1/dt
               d_accel_dlnRm1 = s% r(k)*d_dvardt_dvarm1/dt
               d_dlnPdm_dlnR00 = d_dlnPdm_dlnR00 - fac*d_accel_dlnR00*rPterm
               d_dlnPdm_dlnRp1 = d_dlnPdm_dlnRp1 - fac*d_accel_dlnRp1*rPterm
               d_dlnPdm_dlnRm1 = d_dlnPdm_dlnRm1 - fac*d_accel_dlnRm1*rPterm
            end if
            
         end if

      end subroutine do_dlnP_dm
      
      
      subroutine eval_dlnPdm_qhse(s, k, &
            m, dlnP_dm_qhse, d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, &
            d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
            d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas, &
            ierr)
         type (star_info), pointer :: s 
         integer, intent(in) :: k
         real(fltp), intent(out) :: &
            m, dlnP_dm_qhse, d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, &
            d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
            d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
            d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
            d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
            Ppoint, &
            dPpoint_dlnd00, dPpoint_dlndm1, dPpoint_dlnT00, dPpoint_dlnTm1, &
            dPpoint_dlnPgas00_const_T, dPpoint_dlnPgasm1_const_T, &
            dPpoint_dlnT00_const_Pgas, dPpoint_dlnTm1_const_Pgas
         integer, intent(out) :: ierr
         
         real(fltp) :: alfa, r, r3, r4, d_r4_dlnR, dlnq_dq, &
            dlnP_dm_qhse0, fs, dfs_dlnT, dfs_dlnR, dfs_dlnd, dfs_dL, &
            dfs_dlnPgas00_const_T, dfs_dlnT00_const_Pgas
         logical :: lnPgas_flag

         include 'formats.dek'
         
         ierr = 0
         lnPgas_flag = s% lnPgas_flag
         
         ! basic eqn is dP = -G m / (4 pi r^4)
         ! divide by <P> to make it unitless
         ! simple average is adequate for <P> since is only for normalizing the equation.
         ! however, be careful to use same <P> for both sides of equation..... 
         
         ! for rotation, multiply by factor fp
         
         ! for tau < 2/3, multiply by Paczynski factor for dilution of radiation
            ! B. Paczynski, 1969, Acta Astr., vol. 19.  eqn 13
                  
         ! dlnP_dm_qhse = -G m / (4 pi r^4 <P>)
         r = s% r(k)
         r3 = r**3
         r4 = r*r3
         d_r4_dlnR = 4*r4
         
         if (k == 1) then
            alfa = 1
            Ppoint = alfa*s% P(k)
            dPpoint_dlndm1 = 0
            dPpoint_dlnTm1 = 0
            dPpoint_dlnPgasm1_const_T = 0
            dPpoint_dlnTm1_const_Pgas = 0
         else
            alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
            Ppoint = alfa*s% P(k) + (1-alfa)*s% P(k-1)
            if (lnPgas_flag) then
               dPpoint_dlnPgasm1_const_T = (1-alfa)*s% Pgas(k-1)
               dPpoint_dlnTm1_const_Pgas = (1-alfa)*4*s% Prad(k-1)
            else
               dPpoint_dlndm1 = (1-alfa)*s% P(k-1)*s% chiRho(k-1)
               dPpoint_dlnTm1 = (1-alfa)*s% P(k-1)*s% chiT(k-1)
            end if
         end if
         
         if (lnPgas_flag) then
            dPpoint_dlnPgas00_const_T = alfa*s% Pgas(k)
            dPpoint_dlnT00_const_Pgas = alfa*4*s% Prad(k)
         else
            dPpoint_dlnd00 = alfa*s% P(k)*s% chiRho(k)
            dPpoint_dlnT00 = alfa*s% P(k)*s% chiT(k)
         end if
         
         m = s% m(k)

         dlnP_dm_qhse = -s% cgrav(k)*m/(pi4*r4*Ppoint)
         if (s% rotation_flag) dlnP_dm_qhse = dlnP_dm_qhse*s% fp_rot(k)
         dlnP_dm_qhse0 = dlnP_dm_qhse
         
         if (s% tau(k) >= 2d0/3d0) then
            fs = 0
            dfs_dlnT = 0
            dfs_dlnR = 0
            dfs_dlnd = 0
            dfs_dL = 0
         else ! (tau < 2d0/3d0) then ! B. Paczynski, 1969, Acta Astr., vol. 19, 1.
            fs = (1d0 - 1.5d0*s% tau(k))*(2*crad*s% T(k)**3d0*sqrt(s% r(k)))/ &
                  (3d0*s% cgrav(k)*m*s% rho(k))*(s% L(k)/(8d0*pi*boltz_sigma))**0.25d0  ! eqn 15
            dfs_dlnT = 3d0*fs
            dfs_dlnR = 0.5d0*fs
            dfs_dlnd = -fs
            dfs_dL = 0.25d0*fs/s% L(k)
            dlnP_dm_qhse = dlnP_dm_qhse0*(1 + fs) ! eqn 13
         end if
         
         !s% profile_extra(k,1) = fs  ! DEBUG
            
         d_dlnPdm_dlnR = -(d_r4_dlnR/r4)*dlnP_dm_qhse + dlnP_dm_qhse0*dfs_dlnR
         dlnq_dq = 1/s% q(k)
         d_dlnPdm_dlnq = dlnP_dm_qhse/s% q(k)/dlnq_dq
         
         if (lnPgas_flag) then      
            dfs_dlnPgas00_const_T = dfs_dlnd*s% dlnRho_dlnPgas_const_T(k)
            dfs_dlnT00_const_Pgas = dfs_dlnT + dfs_dlnd*s% dlnRho_dlnT_const_Pgas(k)
            d_dlnPdm_dlnPgas00_const_T = -(dPpoint_dlnPgas00_const_T/Ppoint)*dlnP_dm_qhse + &
               dlnP_dm_qhse0*dfs_dlnPgas00_const_T
            d_dlnPdm_dlnT00_const_Pgas = -(dPpoint_dlnT00_const_Pgas/Ppoint)*dlnP_dm_qhse + &
               dlnP_dm_qhse0*dfs_dlnT00_const_Pgas
            d_dlnPdm_dlnPgasm1_const_T = -(dPpoint_dlnPgasm1_const_T/Ppoint)*dlnP_dm_qhse
            d_dlnPdm_dlnTm1_const_Pgas = -(dPpoint_dlnTm1_const_Pgas/Ppoint)*dlnP_dm_qhse
         else
            d_dlnPdm_dlnd00 = -(dPpoint_dlnd00/Ppoint)*dlnP_dm_qhse + dlnP_dm_qhse0*dfs_dlnd
            d_dlnPdm_dlnT00 = -(dPpoint_dlnT00/Ppoint)*dlnP_dm_qhse + dlnP_dm_qhse0*dfs_dlnT
            d_dlnPdm_dlndm1 = -(dPpoint_dlndm1/Ppoint)*dlnP_dm_qhse
            d_dlnPdm_dlnTm1 = -(dPpoint_dlnTm1/Ppoint)*dlnP_dm_qhse
         end if
         
         d_dlnPdm_dL = dlnP_dm_qhse0*dfs_dL

         if (is_bad(dlnP_dm_qhse)) then
            ierr = -1
            if (s% report_ierr) write(*,*) 'eval_dlnPdm_qhse: is_bad(dlnP_dm_qhse)'
            return
         end if
         
            if (.false. .and. s% hydro_call_number == 17417 .and. k == 304) then
               write(*,2) 'lnd, dlnP_dm_qhse, d_dlnPdm_dlnd00', k, s% lnd(k), dlnP_dm_qhse, d_dlnPdm_dlnd00
            end if
            if (.false. .and. s% hydro_call_number == 17417 .and. k == 304) then
               write(*,2) 'dlnP_dm_qhse', k, dlnP_dm_qhse
               write(*,2) 's% cgrav(k)', k, s% cgrav(k)
               write(*,2) 's% P(k)', k, s% P(k)
               write(*,2) 's% chiRho(k)', k, s% chiRho(k)
               write(*,2) 'm', k, m
               write(*,2) 'r4', k, r4
               write(*,2) 'Ppoint', k, Ppoint
               write(*,2) 'd_dlnPdm_dlnd00', k, d_dlnPdm_dlnd00
               write(*,*)
            end if

      end subroutine eval_dlnPdm_qhse


#ifdef DBLE
      end module hydro_eqns_dble
#else
      end module hydro_eqns_quad
#endif

