! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module evolve_support

      use star_private_def
      use const_def
      use alert_lib

      implicit none


      contains


      subroutine new_generation(s, ierr)
         use utils_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr

         real*8, pointer :: tmp(:,:), tmp1(:)
         integer, pointer :: itmp1(:)
         integer :: nz
         
         include 'formats.dek'
         !write(*,2) 'new_generation'
         
         ierr = 0
         nz = s% nz
         
         s% nz_older = s% nz_old
         s% nz_old = s% nz

         s% mstar_older = s% mstar_old
         s% mstar_old = s% mstar

         s% xmstar_older = s% xmstar_old
         s% xmstar_old = s% xmstar

         s% total_angular_momentum_older = s% total_angular_momentum_old
         s% total_angular_momentum_old = s% total_angular_momentum
         !write(*,2) 'new_generation J_old', s% model_number, s% total_angular_momentum_old

         tmp1 => s% q_older
         s% q_older => s% q_old
         s% q_old => s% q
         call enlarge_if_needed_1(tmp1,nz,nz_alloc_extra,ierr)
         if (ierr /= 0) return
         s% q => tmp1

         tmp1 => s% dq_older
         s% dq_older => s% dq_old
         s% dq_old => s% dq
         call enlarge_if_needed_1(tmp1,nz,nz_alloc_extra,ierr)
         if (ierr /= 0) return
         s% dq => tmp1

         tmp1 => s% conv_vel_older
         s% conv_vel_older => s% conv_vel_old
         s% conv_vel_old => s% conv_vel
         call enlarge_if_needed_1(tmp1,nz,nz_alloc_extra,ierr)
         if (ierr /= 0) return
         s% conv_vel => tmp1

         tmp1 => s% omega_older
         s% omega_older => s% omega_old
         s% omega_old => s% omega
         call enlarge_if_needed_1(tmp1,nz,nz_alloc_extra,ierr)
         if (ierr /= 0) return
         s% omega => tmp1

         itmp1 => s% mixing_type_older
         s% mixing_type_older => s% mixing_type_old
         s% mixing_type_old => s% mixing_type
         call enlarge_integer_if_needed_1(itmp1,nz,nz_alloc_extra,ierr)
         if (ierr /= 0) return
         s% mixing_type => itmp1

         tmp => s% xh_older
         s% xh_older => s% xh_old
         s% xh_old => s% xh
         call enlarge_if_needed_2(tmp,s% nvar_hydro,nz,nz_alloc_extra,ierr)
         if (ierr /= 0) return
         s% xh => tmp

         tmp => s% xa_older
         s% xa_older => s% xa_old
         s% xa_old => s% xa
         call enlarge_if_needed_2(tmp,s% species,nz,nz_alloc_extra,ierr)
         if (ierr /= 0) return
         s% xa => tmp

         s% time_older = s% time_old
         s% time_old = s% time

         s% L_nuc_burn_total_older = s% L_nuc_burn_total_old
         s% L_nuc_burn_total_old = s% L_nuc_burn_total

         s% L_by_category_older(:) = s% L_by_category_old(:)
         s% L_by_category_old(:) = s% L_by_category(:)

         s% L_phot_older = s% L_phot_old
         s% L_phot_old = s% L_phot

         s% min_L_older = s% min_L_old
         s% min_L_old = s% min_L

         s% FL_offset_older = s% FL_offset_old
         s% FL_offset_old = s% FL_offset

         s% mstar_dot_older = s% mstar_dot_old
         s% mstar_dot_old = s% mstar_dot

         s% v_surf_older = s% v_surf_old
         s% v_surf_old = s% v_surf

         s% omega_surf_older = s% omega_surf_old
         s% omega_surf_old = s% omega_surf

         s% omega_crit_surf_older = s% omega_crit_surf_old
         s% omega_crit_surf_old = s% omega_crit_surf

         s% v_div_v_crit_surf_older = s% v_div_v_crit_surf_old
         s% v_div_v_crit_surf_old = s% v_div_v_crit_surf

         s% h1_czb_mass_older = s% h1_czb_mass_old
         s% h1_czb_mass_old = s% h1_czb_mass_prev

         s% h1_boundary_mass_older = s% h1_boundary_mass_old
         s% h1_boundary_mass_old = s% h1_boundary_mass

         s% he4_boundary_mass_older = s% he4_boundary_mass_old
         s% he4_boundary_mass_old = s% he4_boundary_mass

         s% Teff_older = s% Teff_old
         s% Teff_old = s% Teff

         s% TP_state_older = s% TP_state_old
         s% TP_state_old = s% TP_state

         s% TP_count_older = s% TP_count_old
         s% TP_count_old = s% TP_count

         s% TP_M_H_on_older = s% TP_M_H_on_old
         s% TP_M_H_on_old = s% TP_M_H_on

         s% TP_M_H_min_older = s% TP_M_H_min_old
         s% TP_M_H_min_old = s% TP_M_H_min
         
         s% model_number_older = s% model_number_old
         s% model_number_old = s% model_number

         s% varcontrol_older = s% varcontrol_old
         s% varcontrol_old = s% varcontrol

         s% D_norm_err_est_older = s% D_norm_err_est_old
         s% D_norm_err_est_old = s% D_norm_err_est
         
         s% err_ratio_max_hydro_older = s% err_ratio_max_hydro_old
         s% err_ratio_max_hydro_old = s% err_ratio_max_hydro
         
         s% err_ratio_avg_hydro_older = s% err_ratio_avg_hydro_old
         s% err_ratio_avg_hydro_old = s% err_ratio_avg_hydro
         
         s% dt_old = s% dt

      end subroutine new_generation
            
      
      subroutine save_for_d_dt(s)
         ! these values will be modified as necessary by adjust mass
         type (star_info), pointer :: s
         integer :: k, nz
         nz = s% nz
         do k=1, nz
            s% lnT_for_d_dt(k) = s% xh(s% i_lnT, k)
            s% lnR_for_d_dt(k) = s% xh(s% i_lnR, k)
         end do
         if (s% i_xlnd /= 0) then
            do k=1, nz
               s% lnd_for_d_dt(k) = s% xh(s% i_xlnd, k) - lnd_offset
            end do
         end if
         if (s% i_lnPgas /= 0) then
            do k=1, nz
               s% lnPgas_for_d_dt(k) = s% xh(s% i_lnPgas, k)
            end do
         end if
         if (s% v_flag) then
            do k=1, nz
               s% v_for_d_dt(k) = s% xh(s% i_vel, k)
            end do
         end if
         if (s% lnE_flag) then
            do k=1, nz
               s% lnE_var_for_d_dt(k) = s% xh(s% i_lnE, k)
            end do
         end if
      end subroutine save_for_d_dt
      
      
      subroutine set_L_burn_by_category(s)
         use num_lib, only: safe_log10
         use rates_def, only: i_rate
         type (star_info), pointer :: s
         integer :: k, j
         real*8 :: L_burn_by_category(num_categories)            
         L_burn_by_category(:) = 0         
         do k = s% nz, 1, -1
            do j = 1, num_categories
               L_burn_by_category(j) = &
                  L_burn_by_category(j) + s% dm(k)*s% eps_nuc_categories(i_rate, j, k)
               s% luminosity_by_category(j,k) = L_burn_by_category(j)
            end do
         end do      
      end subroutine set_L_burn_by_category
      
      
      subroutine save_pre_hydro_values(s, ierr)
         ! these are the values before do the hydro step,
         ! but after any element diffusion.
         type (star_info), pointer :: s
         integer, intent(out) :: ierr         
         integer :: k         
         real(dp) :: Ledd
         include 'formats.dek'    
         ierr = 0         
         call set_L_burn_by_category(s) ! save pre_hydro for use in selecting timestep
         do k=1,s% nz
            s% xh_pre_hydro(:,k) = s% xh(:,k)
            s% xa_pre_hydro(:,k) = s% xa(:,k)
            s% lnd_pre_hydro(k) = s% lnd(k)
            s% lnP_pre_hydro(k) = s% lnP(k)
            s% lnT_pre_hydro(k) = s% lnT(k)
            s% lnR_pre_hydro(k) = s% lnR(k)
            s% lnE_pre_hydro(k) = s% lnE(k)
            s% lnS_pre_hydro(k) = s% lnS(k)
            s% L_pre_hydro(k) = s% L(k)
            s% i_rot_pre_hydro(k) = s% i_rot(k)
            Ledd = pi4*clight*s% cgrav(k)*s% m(k)/s% opacity(k)
            s% eps_nuc_pre_hydro(k) = s% eps_nuc(k)
            s% dxdt_nuc_pre_hydro(:,k) = s% dxdt_nuc(:,k)
            s% luminosity_by_category_pre_hydro(:,k) = s% luminosity_by_category(:,k)
            s% gradr_pre_hydro(k) = s% gradr(k)
            s% gradL_pre_hydro(k) = s% gradL(k)
            s% gradT_pre_hydro(k) = s% gradT(k)
            s% grada_at_face_pre_hydro(k) = s% grada_at_face(k)
            s% gradL_comp_pre_hydro(k) = s% gradL_composition_term(k)
            s% gradmu_pre_hydro(k) = s% gradmu(k)
            s% gradmu_alt_pre_hydro(k) = s% gradmu_alt(k)
            s% gradmu_X_pre_hydro(k) = s% gradmu_X(k)
         end do
      end subroutine save_pre_hydro_values


      subroutine restore_older(s)
         type (star_info), pointer :: s
         real*8, pointer :: p1(:), p2(:,:)
         integer, pointer :: ip1(:)
         include 'formats.dek'
         
         if (s% generations < 3) return
         
         s% nz_old = s% nz_older
         s% mstar_old = s% mstar_older
         s% xmstar_old = s% xmstar_older
         s% time_old = s% time_older
         s% total_angular_momentum_old = s% total_angular_momentum_older
         
         s% L_nuc_burn_total_old = s% L_nuc_burn_total_older
         s% L_by_category_old(:) = s% L_by_category_older(:)
         
         s% L_phot_old = s% L_phot_older
         s% min_L_old = s% min_L_older
         s% FL_offset_old = s% FL_offset_older

         s% mstar_dot_old = s% mstar_dot_older
         s% v_surf_old = s% v_surf_older
         s% omega_surf_old = s% omega_surf_older
         s% omega_crit_surf_old = s% omega_crit_surf_older
         s% v_div_v_crit_surf_old = s% v_div_v_crit_surf_older

         s% h1_czb_mass_old = s% h1_czb_mass_older

         s% h1_boundary_mass_old = s% h1_boundary_mass_older
         s% he4_boundary_mass_old = s% he4_boundary_mass_older

         s% Teff_old = s% Teff_older
         
         s% n_conv_regions_old = s% n_conv_regions_older
         s% cz_bot_mass_old(:) = s% cz_bot_mass_older(:)
         s% cz_top_mass_old(:) = s% cz_top_mass_older(:)

         s% TP_state_old = s% TP_state_older
         s% TP_count_old = s% TP_count_older
         s% TP_M_H_on_old = s% TP_M_H_on_older
         s% TP_M_H_min_old = s% TP_M_H_min_older

         s% model_number_old = s% model_number_older
         s% varcontrol_old = s% varcontrol_older
         s% D_norm_err_est_old = s% D_norm_err_est_older
         
         s% err_ratio_max_hydro_old = s% err_ratio_max_hydro_older
         s% err_ratio_avg_hydro_old = s% err_ratio_avg_hydro_older
         
         p1 => s% q_old
         s% q_old => s% q_older
         s% q_older => p1
         
         p1 => s% dq_old
         s% dq_old => s% dq_older
         s% dq_older => p1
         
         p1 => s% conv_vel_old
         s% conv_vel_old => s% conv_vel_older
         s% conv_vel_older => p1
         
         p1 => s% omega_old
         s% omega_old => s% omega_older
         s% omega_older => p1
         
         ip1 => s% mixing_type_old
         s% mixing_type_old => s% mixing_type_older
         s% mixing_type_older => ip1
         
         p2 => s% xh_old 
         s% xh_old => s% xh_older
         s% xh_older => p2
         
         p2 => s% xa_old
         s% xa_old => s% xa_older
         s% xa_older => p2
         
         s% nz_older = 0
         s% mstar_older = 0
         s% xmstar_older = 0
         s% time_older = 0
         s% total_angular_momentum_older = 0

      end subroutine restore_older


      subroutine set_current_to_old(s)
         type (star_info), pointer :: s
         include 'formats.dek'
         s% nz = s% nz_old
         s% mstar = s% mstar_old
         s% xmstar = s% xmstar_old
         s% M_center = s% mstar - s% xmstar
         s% total_angular_momentum = s% total_angular_momentum_old
         ! other current values will be recalculated at start of step         
      end subroutine set_current_to_old


      subroutine output(id, ierr)
         use star_utils, only:get_string_for_model_number
         interface
            subroutine save_restart_info(iounit, id, ierr)
               integer, intent(in) :: iounit
               integer, intent(in) :: id
               integer, intent(out) :: ierr
            end subroutine save_restart_info
         end interface
         integer, intent(in) :: id
         integer, intent(out) :: ierr         
         character (len=256) :: filename, num_str
         type (star_info), pointer :: s
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         call get_string_for_model_number(num_str, s% model_number, s% photo_digits)
         filename = trim(s% photo_directory) // '/' // trim(num_str) 
         call output_to_file(filename, id, ierr) 
         if (ierr /= 0) return        
         write(*, '(2a)') 'save ', trim(filename)
         if (s% extra_terminal_iounit > 0) &
            write(s% extra_terminal_iounit, '(2a)') 'save ', trim(filename)
      end subroutine output
      

      subroutine output_to_file(filename, id, ierr)
         use utils_lib, only:alloc_iounit, free_iounit
         use model_out, only:output_star_model
         character (len=*) :: filename
         integer, intent(in) :: id
         integer, intent(out) :: ierr

         integer :: iounit
         type (star_info), pointer :: s
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return

         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         open(iounit, file=trim(filename), action='write', &
            status='replace', iostat=ierr, form='unformatted')
         if (ierr == 0) then
            s% most_recent_photo_name = trim(filename)
            call output_star_model(s, iounit)
            close(iounit)
         else
            call alert(ierr, "failed to open file for output photo")
         end if
         call free_iounit(iounit)
         
      end subroutine output_to_file
      
      
      real*8 function yrs_for_init_timestep(s)
         type (star_info), pointer :: s
         if (s% initial_mass <= 1) then
            yrs_for_init_timestep = 1d5
         else
            yrs_for_init_timestep = 1d5 / (s% initial_mass**2.5d0)
         end if
      end function yrs_for_init_timestep


      end module evolve_support


