! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module evolve

      use star_private_def
      use alert_lib
      use const_def
      use num_lib, only: safe_log10

      implicit none

      contains
      

      integer function do_evolve_step(id, first_try, just_did_backup)
         use num_def
         use chem_def, only: ih1
         use do_one_utils, only:write_terminal_header
         use report, only:do_report
         use winds, only:do_winds
         use adjust_mass, only: do_adjust_mass         
         use utils_lib, only: is_bad_num, has_bad_num
         use element_diffusion, only: do_element_diffusion
         use alloc, only: check_sizes
         use evolve_support, only: save_pre_hydro_values, save_for_d_dt, set_L_burn_by_category
         use star_utils, only: update_time, total_times, total_angular_momentum, &
            dump_model_for_diff, FL_to_L
         use brunt, only: do_brunt_N2
         use mix_smoothing, only: smooth_newly_non_conv
         use solve_omega_mix, only: do_solve_omega_mix
         use struct_burn_mix, only: do_struct_burn_mix
         use hydro_vars, only: set_vars, set_all_vars_except_mixing_info
         
         use profile
         
         logical, intent(in) :: first_try, just_did_backup
         integer, intent(in) :: id

         type (star_info), pointer :: s            
         integer :: ierr, j_cnt
         integer :: time0, clock_rate
         logical :: trace, skip_global_corr_coeff_limit
         real(dp) :: total_all_before, J_tot1, J_tot2
         
         logical, parameter :: check_for_bad_nums = .true.
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         do_evolve_step = terminate
         
         ierr = 0         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return

         trace = s% trace_evolve
         s% just_did_backup = just_did_backup
         
         if (dbg) then
            call check_sizes(s, ierr)
            if (ierr /= 0) then
               write(*,*) 'do_evolve_step: check_sizes returned ierr', ierr
               return
            end if
         end if

         if (s% doing_first_model_of_run) then
            if (s% do_log_files) then
               if (first_try) then
                  call write_terminal_header(s)
               else
                  write(*,*) 'retry'
               end if
            end if
            call system_clock(time0,clock_rate)
            s% starting_system_clock_time = time0
            s% system_clock_rate = clock_rate
            s% initial_timestep = s% dt_next
            s% last_backup = -s% backup_hold
            s% timestep_hold = -111
            if (first_try) s% model_number_old = s% model_number
         end if

         call debug('before prepare_for_new_step')

         if (first_try) then
            if (trace) write(*,'(/,a,i8)') 'call prepare_for_new_step', s% model_number
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            do_evolve_step = prepare_for_new_step(s)
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_prep_new_step)
            if (do_evolve_step /= keep_going) return
         end if

         call debug('before prepare_for_new_try')

         if (trace) write(*,'(/,a,i8)') 'call prepare_for_new_try', s% model_number
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         do_evolve_step = prepare_for_new_try(s)
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_prep_new_try)
         if (do_evolve_step /= keep_going) return

         call debug('before do_winds')

         if (trace) write(*,'(/,a,i8)') 'call do_winds', s% model_number
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         
         ! set mdot for the step
         call do_winds(s, s% L_phot*Lsun, s% mstar, s% Teff, ierr)
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_do_winds)
         if (ierr /= 0) then
            do_evolve_step = retry
            s% result_reason = nonzero_ierr
            if (s% report_ierr) write(*, *) 'do_evolve_step: do_winds ierr'
            return
         end if
         
         if (s% dt <= 0) then
            do_evolve_step = terminate
            s% result_reason = dt_is_zero
            return
         end if
         
         s% time = s% time_old + s% dt
         
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         call save_for_d_dt(s) ! save information for evaluating lagrangian time derivatives
            ! values may be revised by adjust mass
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_save_for_d_dt)

         call debug('before do_adjust_mass')

         if (trace) write(*,'(/,a)') 'call do_adjust_mass'         
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         call do_adjust_mass(s, s% species, ierr)
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_do_adjust_mass)
         if (ierr /= 0) then
            do_evolve_step = retry
            s% result_reason = adjust_mass_failed
            if (s% report_ierr) write(*, *) 'do_evolve_step: do_adjust_mass ierr'
            return
         end if

         call debug('before do_set_vars')
         
         call do_set_vars(.true.,ierr) ! set mixing info
         if (ierr /= 0) return

         call debug('before do_element_diffusion')
            
         if (s% do_element_diffusion) then
            if (trace) write(*,'(/,a)') 'call do_element_diffusion'
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            if (.not. do_element_diffusion(s, s% dt)) then
               if (s% report_ierr) then
                  write(*, *) 'element diffusion failed: retry', s% model_number
               end if
               do_evolve_step = retry
               s% result_reason = diffusion_failed
               return
            end if
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_diffusion)
            call do_set_vars(.false.,ierr)
            if (ierr /= 0) return
         end if

         call debug('before save_pre_hydro_values')
         
         if (trace) write(*,'(/,a)') 'call save_pre_hydro_values'
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         call save_pre_hydro_values(s, ierr)
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_save_pre_hydro)
         if (ierr /= 0) then
            if (s% report_ierr) then
               write(*, *) 'do_evolve_step: save_pre_hydro_values ierr: retry', s% model_number
            end if
            do_evolve_step = retry
            s% result_reason = nonzero_ierr
            return
         end if

         call debug('before do_struct_burn_mix')

         if (trace) write(*,'(/,a,i8)') 'call do_struct_burn_mix', s% model_number
         skip_global_corr_coeff_limit = &
            (first_try .or. (just_did_backup .and. s% number_of_backups_in_a_row == 1))
         do_evolve_step = do_struct_burn_mix(s, skip_global_corr_coeff_limit)
         if (do_evolve_step /= keep_going) return
        
         if (s% rotation_flag) then
            if (trace) write(*,'(/,a,i8)') 'call do_solve_omega_mix', s% model_number
            do_evolve_step = do_solve_omega_mix(s)
            if (do_evolve_step /= keep_going) return
         end if
                
         if (check_for_bad_nums) then
            if (has_bad_num(s% species*s% nz, s% xa)) then
               write(*, *) 'bad num in xa after calling do_struct_burn_mix: model_number', s% model_number
               do_evolve_step = retry
               s% result_reason = nonzero_ierr
               return
            end if
         end if

         call debug('before smooth_newly_non_conv')
         
         if (s% smooth_convective_bdy) then
            if (trace) write(*,'(/,a)') 'call smooth_newly_non_conv'
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call smooth_newly_non_conv(s, ierr) ! do this after do_struct_burn_mix
            if (s% doing_timing) &
               call update_time(s, time0, total_all_before, s% time_check_newly_non_conv)
            if (ierr /= 0) then
               if (s% report_ierr) then
                  write(*, *) 'do_evolve_step: smooth_newly_non_conv ierr: retry', s% model_number
               end if
               do_evolve_step = retry
               s% result_reason = nonzero_ierr
               return
            end if
         end if
         
         if (dbg) write(*, *) 'call do_brunt_N2'
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         call do_brunt_N2(s, 1, s% nz, ierr)
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_do_brunt)
         if (ierr /= 0) then
            if (s% report_ierr) then
               write(*, *) 'do_evolve_step: do_brunt_N2 ierr: retry', s% model_number
            end if
            do_evolve_step = retry
            s% result_reason = nonzero_ierr
            return
         end if

         if (trace) write(*,'(/,a)') 'call do_report'
         call system_clock(time0,clock_rate)
         s% current_system_clock_time = time0
         s% total_elapsed_time = dble(time0 - s% starting_system_clock_time)/dble(clock_rate)
         if (s% doing_timing) total_all_before = total_times(s)
         call do_report(s, ierr)
         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_do_report)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'do_evolve_step: do_report ierr'
            do_evolve_step = retry
            s% result_reason = nonzero_ierr
            return
         end if
         
         call set_L_burn_by_category(s) ! for use in selecting timestep
         
         s% total_angular_momentum = total_angular_momentum(s)
         
         
         
         contains
         
         
         subroutine debug(str) 
            use chem_def
            character (len=*), intent(in) :: str
            integer :: k, j
            include 'formats.dek'
            return
            k = 1469
            j = 2
            !do j=1,1 !s% species
               if (.true. .or. s% xa(j,k) > 1d-9) &
                  write(*,1) trim(str) // ' xin(net_iso(i' // trim(chem_isos% name(s% chem_id(j))) // '))= ', &
                     s% xa(j,k), s% abar(k)
            !end do
         end subroutine debug
         
         
         subroutine do_set_vars(do_mixing_info,ierr)
            use hydro_vars, only: set_vars, set_all_vars_except_mixing_info
            logical, intent(in) :: do_mixing_info
            integer, intent(out) :: ierr
            integer :: nz
            nz = s% nz
            ierr = 0
            if (trace) write(*,'(/,a)') 'call set_vars'
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            if (do_mixing_info) then
               call set_vars(s, 1, nz, s% dt, ierr)
               ! update "old" mix info to be values from start of this step
               s% conv_vel_old(1:nz) = s% conv_vel(1:nz)
               s% mixing_type_old(1:nz) = s% mixing_type(1:nz)
            else
               call set_all_vars_except_mixing_info(s, 1, nz, s% dt, ierr)
            end if
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_evolve_set_vars)
            if (ierr /= 0) then
               if (s% report_ierr) then
                  write(*, *) 'do_evolve_step: set_vars ierr: retry', s% model_number
               end if
               do_evolve_step = retry
               s% result_reason = nonzero_ierr
               return
            end if
         end subroutine do_set_vars


      end function do_evolve_step

      
      integer function prepare_for_new_step(s)
         use utils_lib, only:has_bad_num, is_bad_num
         use evolve_support, only: new_generation
         use report, only: do_report
         use hydro_vars, only: set_vars
         use star_utils
         use chem_def
         type (star_info), pointer :: s
         
         integer :: ierr, k, nz
         integer :: time0, clock_rate
         logical :: trace
         real(dp) :: total_all_before, safety_factor
         logical, parameter :: check_for_bad_nums = .false.
         
         include 'formats.dek'

         ierr = 0
         trace = s% trace_evolve
         nz = s% nz

         prepare_for_new_step = keep_going

         if (s% dt_next <= 0) then
            prepare_for_new_step = terminate
            if (s% time >= s% max_age*secyer) then
               s% result_reason = result_reason_normal
            else
               s% result_reason = dt_is_zero
            end if
            return
         end if

         if (s% dt_next < s% min_timestep_limit) then
            write(*, *) 's% hydro_dt_next', s% hydro_dt_next
            write(*, *) 's% dt_next', s% dt_next
            write(*, *) 's% min_timestep_limit', s% min_timestep_limit
            prepare_for_new_step = terminate
            s% result_reason = timestep_limits
            return
         end if
         
         s% min_L = FL_to_L(s,minval(s% xh(s% i_FL,1:nz)))
         safety_factor = s% min_FL_offset
         if (-safety_factor*min(s% min_L, s% min_L_old, s% min_L_older) > s% FL_offset) then
            ! need to increase FL_offset to keep all L + FL_offset >> 0
            do k=1,nz
               s% L(k) = FL_to_L(s,s% xh(s% i_FL,k))
            end do
            s% FL_offset = -10*safety_factor*min(s% min_L, s% min_L_old, s% min_L_older)
            do k=1,nz
               s% xh(s% i_FL,k) = L_to_FL(s,s% L(k))
            end do
         end if 
         
         if (s% doing_first_model_of_run) then
            ! this call on set_vars is to prepare for calling do_mesh
            ! if not 1st model of run then have this info leftover from previous step
            if (trace) write(*,*) 'call set_vars'
            call set_vars(s, 1, nz, s% dt, ierr) ! this does set_mixing_info too
            if (ierr /= 0) then
               if (s% report_ierr) then
                  write(*, *) 'prepare_for_new_step: set_vars ierr: retry', s% model_number
               end if
               prepare_for_new_step = retry
               s% result_reason = nonzero_ierr
               return
            end if
            call do_report(s, ierr)
            if (ierr /= 0) then
               if (s% report_ierr) then
                  write(*, *) 'prepare_for_new_step: do_report ierr: retry', s% model_number
               end if
               prepare_for_new_step = retry
               s% result_reason = nonzero_ierr
               return
            end if
         end if
         
         ! save a few things from start of step that will need later
         s% prev_Lmax = maxval(abs(s% L(1:nz)))
         if (s% rotation_flag) then
            s% surf_r_equitorial = s% r_equitorial(1)
         else
            s% surf_r_equitorial = s% r(1)
         end if
         s% surf_opacity = s% opacity(1)
         s% surf_csound = s% csound(1)
         s% surf_rho = s% rho(1)
         s% prev_Ledd = eval_Ledd(s)
         s% hydro_seulex_dt_limit = 0
         
         s% rstar_old = s% r(1)
         if (s% generations == 1) then
            s% surf_accel_grav_ratio = 0
         else
            s% surf_accel_grav_ratio = (s% v_surf - s% v_surf_old)/(s% dt*s% grav(1))
         end if
                  
         if (.not. s% doing_first_model_of_run) then
            if (trace) write(*,*) 'call do_mesh'
            prepare_for_new_step = do_mesh(s)
            if (prepare_for_new_step /= keep_going) then
               write(*,*) 'failed in do_mesh'
               prepare_for_new_step = terminate
               return
            end if
         end if

         if (trace) write(*,*) 'call new_generation'
         call new_generation(s, ierr)         
         if (ierr /= 0) then
            prepare_for_new_step = terminate
            if (s% report_ierr) write(*, *) 'prepare_for_new_step: new_generation ierr'
            s% result_reason = nonzero_ierr
            return
         end if

         s% dt = s% dt_next                  
         s% dt_start = s% dt         
         s% retry_cnt = 0
         s% generations = min(max_generations, s% generations+1)
         
         s% need_to_save_profiles_now = .false.
         s% need_to_update_logfile_now = s% doing_first_model_of_run
         
         if (check_for_bad_nums) then
            if (has_bad_num(s% species*s% nz, s% xa)) then
               write(*, *) 'bad num in xa at end of prepare_for_new_step'
               prepare_for_new_step = terminate
               s% result_reason = nonzero_ierr
               return
            else
               write(*, *) 'no bad num in xa at end of prepare_for_new_step'
            end if
            do k=1, s% nz
               if (is_bad_num(s% dq(k)) .or. s% dq(k) <= 0d0) then
                  write(*, *) 'prepare_for_new_step: s% dq(k)', k, s% dq(k)
                  prepare_for_new_step = terminate
                  s% result_reason = nonzero_ierr
                  return
               end if
            end do
         end if
         
      end function prepare_for_new_step


      integer function do_mesh(s)
         use adjust_mesh, only: remesh
         type (star_info), pointer :: s      
         integer :: ierr
         logical, parameter :: okay_to_merge = .true.
         include 'formats.dek'
         ierr = 0
         do_mesh = remesh(s, okay_to_merge)
         if (do_mesh /= keep_going) then
            s% result_reason = adjust_mesh_failed
            if (s% report_ierr) write(*, *) 'do_mesh: remesh failed'
            return
         end if
      end function do_mesh
      

      integer function prepare_for_new_try(s) 
         ! return keep_going, terminate, retry, or backup
         ! if don't return keep_going, then set result_reason to say why.
         use utils_lib, only: is_bad_num, has_bad_num
         use net_lib, only: clean_up_fractions

         type (star_info), pointer :: s
         
         integer :: ierr, i, k, nz, nvar, nvar_hydro
         real(dp), parameter :: max_sum_abs = 10d0
         real(dp), parameter :: xsum_tol = 1d-2
         logical, parameter :: check_for_bad_nums = .true.

         include 'formats.dek'
         
         nvar = s% nvar
         nvar_hydro = s% nvar_hydro
         nz = s% nz

         s% result_reason = result_reason_normal
         s% model_number = s% model_number_old + 1
         
         ! check dimensions
         if (size(s% xh_old,dim=1) /= nvar_hydro .or. size(s% xh_old,dim=2) < nz) then
            write(*,*) 'bad dimensions for xh_old', size(s% xh_old,dim=1), nvar_hydro, &
               size(s% xh_old,dim=2), nz
            prepare_for_new_try = terminate
            return
         end if
         if (size(s% xa_old,dim=1) /= s% species .or. size(s% xa_old,dim=2) < nz) then
            write(*,*) 'bad dimensions for xa_old', size(s% xa_old,dim=1), s% species, &
               size(s% xa_old,dim=2), nz
            prepare_for_new_try = terminate
            return
         end if
         if (size(s% q_old,dim=1) < nz) then
            write(*,*) 'bad dimensions for q_old', size(s% q_old,dim=1), nz
            prepare_for_new_try = terminate
            return
         end if
         if (size(s% dq_old,dim=1) < nz) then
            write(*,*) 'bad dimensions for dq_old', size(s% dq_old,dim=1), nz
            prepare_for_new_try = terminate
            return
         end if
         
         do k = 1, nz ! write loop to avoid ifort stack size segfaults
            s% xh(:,k) = s% xh_old(:,k) ! start from old structure
            s% xa(:,k) = s% xa_old(:,k) ! start from old composition
            s% q(k) = s% q_old(k) ! start with same q's
            s% dq(k) = s% dq_old(k) ! start with same dq's
            if (s% rotation_flag) then
               s% omega(k) = s% omega_old(k) ! start with same omega's
               s% i_rot(k) = (2d0/3d0)*exp(2d0*s% xh(s% i_lnR,k))
               s% j_rot(k) = s% i_rot(k)*s% omega(k)
            end if
         end do
         
         prepare_for_new_try = keep_going
         
         do k=1, nz
            do i=1, nvar_hydro
               if (is_bad_num(s% xh(i, k))) then
                  write(*, *) 'xh(i, k)', i, k, s% xh(i, k)
                  prepare_for_new_try = terminate
                  return
               end if
            end do
         end do
         
         if (check_for_bad_nums) then
            do k=1, nz
               do i=1, nvar_hydro
                  if (is_bad_num(s% xh(i, k))) then
                     write(*, *) 'xh(i, k)', i, k, s% xh(i, k)
                     prepare_for_new_try = retry
                     return
                  end if
               end do
            end do
            if (has_bad_num(s% species*s% nz, s% xa)) then
               write(*, *) 'bad num in xa at end of prepare_for_new_try'
               prepare_for_new_try = retry
               s% result_reason = nonzero_ierr
               return
            end if
            do k=1, s% nz
               if (is_bad_num(s% dq(k)) .or. s% dq(k) <= 0d0) then
                  write(*, *) 'prepare_for_new_try: s% dq(k)', k, s% dq(k)
                  prepare_for_new_try = retry
                  s% result_reason = nonzero_ierr
                  return
               end if
            end do
         end if
         
         return
         
         if (s% model_number /= 121) return
         write(*,*) 'dump prepare_for_new_try'
         do k=1, s% nz
            do i=1,nvar_hydro
               write(*,3) 'xh', i, k, s% xh(i,k)
            end do
            do i=1,s% species
               write(*,3) 'xa', i, k, s% xa(i,k)
            end do
            write(*,2) 'q', k, s% q(k)
            write(*,2) 'dq', k, s% dq(k)
         end do
         stop 'prepare_for_new_try'
         
      end function prepare_for_new_try
      
      
      integer function pick_next_timestep(id)
         ! determine what we want for the next timestep
         ! if don't return keep_going, then set result_reason to say why.
         use timestep, only: hydro_timestep, timestep_controller
         use star_utils, only: update_time, total_times
         integer, intent(in) :: id
         integer :: ierr
         type (star_info), pointer :: s
         integer :: time0, clock_rate
         real(dp) :: total_all_before
         pick_next_timestep = terminate         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         if (s% max_years_for_timestep > 0) &
            s% max_timestep = secyer*s% max_years_for_timestep
         pick_next_timestep = hydro_timestep(s)
         if (pick_next_timestep /= keep_going) then
            if (s% trace_evolve) write(*,*) 'pick_next_timestep: hydro_timestep /= keep_going'
            call timing
            return
         end if
         pick_next_timestep = timestep_controller(s)
         if (pick_next_timestep /= keep_going) then
            if (s% trace_evolve) write(*,*) 'pick_next_timestep: timestep_controller /= keep_going'
            call timing
            return
         end if
         if ((s% time + s% dt_next) > s% max_age*secyer) &
            s% dt_next = max(0d0, s% max_age*secyer - s% time)
         call timing
         
         contains
         
         subroutine timing
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_next_timestep)
         end subroutine timing
         
      end function pick_next_timestep
      
      
      integer function prepare_to_redo(id)
         use evolve_support, only: set_current_to_old
         integer, intent(in) :: id
         type (star_info), pointer :: s
         integer :: ierr
         include 'formats.dek'        
         ierr = 0 
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            prepare_to_redo = terminate
            return
         end if         
         prepare_to_redo = keep_going         
         if (s% trace_evolve) write(*,'(/,a)') 'prepare_to_redo'        
         call set_current_to_old(s)             
      end function prepare_to_redo
      
      
      integer function prepare_to_retry(id)
         use star_utils, only: update_time, total_times
         use evolve_support, only: set_current_to_old
         integer, intent(in) :: id
         
         real(dp) :: retry_factor, total_all_before
         type (star_info), pointer :: s
         integer :: ierr, time0, clock_rate
         include 'formats.dek'
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            prepare_to_retry = terminate
            return
         end if

         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if
         
         if (s% trace_evolve) write(*,'(/,a)') 'prepare_to_retry'
         
         s% retry_cnt = s% retry_cnt + 1
         if (s% retry_cnt > s% retry_limit) then
            prepare_to_retry = backup
            call timing
            return
         end if

         prepare_to_retry = keep_going
         if (s% retry_limit <= 0) then
            if (s% report_ierr) &
               write(*, *) 'prepare_to_retry: no retries allowed. must make retry_limit > 0'
            prepare_to_retry = terminate
            call timing
            return
         end if
         
         if (s% doing_first_model_of_run) then
            retry_factor = 0.1d0**(1d0/(1+s% retry_limit))
            write(*,1) '1st model retry: new log10 dt/secyer', log10(s% dt*retry_factor/secyer)
         else
            retry_factor = s% timestep_factor_for_backups**(1d0/(1+s% retry_limit))
            !write(*,1) 'retry: new log10 dt/secyer', log10(s% dt*retry_factor/secyer)
         end if
         s% dt = s% dt*retry_factor
         if (s% hydro_seulex_dt_limit > 0) then
            if (s% hydro_seulex_dt_limit < s% dt) s% dt = s% hydro_seulex_dt_limit
            s% hydro_seulex_dt_limit = 0
         end if
         if (s% dt <= 0) then
            if (s% report_ierr) &
               write(*, *) 'error: dt has gone to 0 at time=', s% time
            prepare_to_retry = terminate
            call timing
            return
         end if

         if (s% max_years_for_timestep > 0) &
            s% max_timestep = secyer*s% max_years_for_timestep
         if (s% max_timestep > 0) s% dt = min(s% dt, s% max_timestep)
         
         call set_current_to_old(s)

         s% num_retries = s% num_retries+1
         !write(*,2) 'prepare_to_retry s% num_retries', s% num_retries
         !write(*,2) 's% max_number_retries', s% max_number_retries
         if (s% num_retries > s% max_number_retries .and. s% max_number_retries >= 0) then
            call report_convergence_problems(s, '-- too many retries')
            prepare_to_retry = terminate; return
         end if

         s% model_number_for_last_retry = s% model_number
         
         call timing
         

         contains
         
         subroutine timing
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_prep_for_retry)
         end subroutine timing
            
      end function prepare_to_retry
      
      
      subroutine report_convergence_problems(s,str)
         type (star_info), pointer :: s
         character (len=*), intent(in) :: str
         write(*,*)
         write(*,*) 'stopping because of convergence problems ' // trim(str)
         write(*,*)
      end subroutine report_convergence_problems
      
      
      integer function do1_backup(id)
         ! returns either keep_going or terminate
         ! at end of this routine, must have same vars set as at end of prepare_for_new_step.
         
         use evolve_support, only: restore_older, set_current_to_old
         use alloc, only: free_star_info_arrays, allocate_star_info_arrays, &
            set_var_info, set_chem_names
         integer, intent(in) :: id
         
         integer :: ierr
         type (star_info), pointer :: s
         
         include 'formats.dek'
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            do1_backup = terminate
            return
         end if
         
         if (s% trace_evolve) write(*,'(/,a)') 'do1_backup'
         
         do1_backup = keep_going
         
         s% dt = s% dt_start * &
            (s% timestep_factor_for_backups**(1+2*s% number_of_backups_in_a_row))         
         if (s% doing_first_model_of_run) s% dt = 0.01d0*s% dt

         if (s% hydro_seulex_dt_limit > 0) then
            if (s% hydro_seulex_dt_limit < s% dt) s% dt = s% hydro_seulex_dt_limit
            s% hydro_seulex_dt_limit = 0
         end if

         if (s% dt < s% min_timestep_limit) then
            call report_convergence_problems(s, '-- too many backups')
            do1_backup = terminate
            s% result_reason = timestep_limits
            return
         end if
         
         if (s% generations >= 3) then ! have "older" so can restore it   
            call restore_older(s)
            s% generations = s% generations-1
         end if
         
         call set_current_to_old(s)
         s% varcontrol_old = 0 ! don't use predictive timestep control after backup
         s% D_norm_err_est_old = 0
         
         call set_var_info(s, ierr)
         if (ierr /= 0) then
            do1_backup = terminate
            if (s% report_ierr) write(*, *) 'do1_backup: set_var_info ierr', ierr
            return
         end if
         call set_chem_names(s)
         
         call free_star_info_arrays(s)

         call allocate_star_info_arrays(s, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'do1_backup: allocate_star_info_arrays ierr'
            do1_backup = terminate; return
         end if

         s% dt_start = s% dt                   
         s% last_backup = s% model_number               
         s% num_backups = s% num_backups + 1
         if (s% num_backups > s% max_number_backups .and. s% max_number_backups >= 0) then
            call report_convergence_problems(s, 'too many backups')
            do1_backup = terminate; return
         end if

         s% number_of_backups_in_a_row = s% number_of_backups_in_a_row + 1
         if (s% number_of_backups_in_a_row > s% max_backups_in_a_row &
               .and. s% max_backups_in_a_row > 0) then
            call report_convergence_problems(s, 'too many backups in a row')
            do1_backup = terminate; return
         end if
         
         s% timestep_hold = s% model_number + s% backup_hold
         s% retry_cnt = 0
         s% model_number_for_last_retry = s% model_number
         if (abs(s% backup_drop_varcontrol_target - 1d0) > 1d-6) then
            s% varcontrol_target = s% varcontrol_target*s% backup_drop_varcontrol_target
            write(*,2) 'new varcontrol_target', s% model_number, s% varcontrol_target
         end if
         
         if (s% report_ierr) write(*, *) 'backup model_number', &
            s% model_number, s% num_backups
         
      end function do1_backup


      integer function finish_step( &
            id, id_extra, do_photo, &
            how_many_extra_profile_columns, data_for_extra_profile_columns, &
            how_many_extra_log_columns, data_for_extra_log_columns, ierr)
         ! returns keep_going or terminate
         ! if don't return keep_going, then set result_reason to say why.
         use evolve_support, only: output
         use do_one_utils, only: do_save_profiles
         use log, only: write_log_info
         use utils_lib, only: free_iounit, number_iounits_allocated
         use star_utils, only: update_time, total_times
         use alloc, only: size_work_arrays

         integer, intent(in) :: id, id_extra
         logical, intent(in) :: do_photo ! if true, then save "photo" for restart
         interface
            include 'extra_profile_cols.dek'
            include 'extra_log_cols.dek'
         end interface
         integer, intent(out) :: ierr

         type (star_info), pointer :: s
         integer, parameter :: nvals = 1, n_ivals = 0
         integer :: time0, clock_rate, nz, &
            current_num_iounits_in_use, prev_num_iounits_in_use
         integer :: ivals(n_ivals)
         real(dp) :: vals(nvals)
         real(dp) :: total_all_before
         
         include 'formats.dek'
         
         finish_step = terminate
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         nz = s% nz
         
         ! this is the place to remember values from end of step
         s% h1_czb_mass_prev = s% h1_czb_mass
         
         s% n_conv_regions_older = s% n_conv_regions_old
         s% n_conv_regions_old = s% n_conv_regions
         
         s% cz_bot_mass_older(:) = s% cz_bot_mass_old(:)
         s% cz_bot_mass_old(:) = s% cz_bot_mass(:)
         
         s% cz_top_mass_older(:) = s% cz_top_mass_old(:)
         s% cz_top_mass_old(:) = s% cz_top_mass(:)
         
         prev_num_iounits_in_use = number_iounits_allocated()
         
         finish_step = keep_going
         s% result_reason = result_reason_normal
                  
         if (s% need_to_save_profiles_now .and. s% do_profiles) then
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call do_save_profiles( &
               s, id_extra, how_many_extra_profile_columns, data_for_extra_profile_columns, ierr)
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_write_profile)
            s% need_to_save_profiles_now = .false.
         end if
         
         call check(1)
         
         if (s% need_to_update_logfile_now .and. s% do_log_files) then
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call write_log_info( &
               s, id_extra, how_many_extra_log_columns, data_for_extra_log_columns, ierr)
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_write_log)
            if (ierr /= 0) then
               finish_step = terminate
               if (s% report_ierr) write(*, *) 'finish_step: write_log_info ierr', ierr
               s% result_reason = nonzero_ierr
               return
            end if
            s% need_to_update_logfile_now = .false.
         end if
         
         call check(2)
         
         if (do_photo .or. &
               (s% photostep > 0 .and. mod(s% model_number, s% photostep) == 0)) then
            if (s% doing_timing) then
               total_all_before = total_times(s)
               call system_clock(time0,clock_rate)
            end if
            call output(id, ierr)
            if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_write_photo)
            if (ierr /= 0) then
               finish_step = terminate
               if (s% report_ierr) write(*, *) 'finish_step: output ierr', ierr
               s% result_reason = nonzero_ierr
               return
            end if
         end if
         
         call check(3)

         s% doing_first_model_of_run = .false.
         s% number_of_backups_in_a_row = 0
         if (s% model_number - s% model_number_for_last_retry >= s% backup_drop_vc_target_steps .and. &
             abs(s% backup_drop_varcontrol_target - 1d0) > 1d-6 .and. &
             s% varcontrol_target < s% varcontrol_target_max) then
            s% varcontrol_target = s% varcontrol_target/s% backup_drop_varcontrol_target
            write(*,2) 'new varcontrol_target', s% model_number, s% varcontrol_target
            s% model_number_for_last_retry = s% model_number - 1 ! so delay next change
         end if
                  
         contains
         
         subroutine check(i)
            integer, intent(in) :: i
            include 'formats.dek'
            !return
            
            current_num_iounits_in_use = number_iounits_allocated()
            if (current_num_iounits_in_use > 3 .and. &
                  current_num_iounits_in_use > prev_num_iounits_in_use) then
               write(*,2) 's% model_number', s% model_number
               write(*,2) 'prev_num_iounits_in_use', prev_num_iounits_in_use
               write(*,2) 'current_num_iounits_in_use', current_num_iounits_in_use
               write(*,2) 'i', i
               stop 'finish_step' 
            end if
            prev_num_iounits_in_use = current_num_iounits_in_use
         end subroutine check
         
         
      end function finish_step
      
      
      subroutine set_age(id, age, ierr)
         integer, intent(in) :: id
         real(dp), intent(in) :: age
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         s% time = age*secyer
         s% star_age = age
         s% profile_age = age
         s% post_he_age = age
      end subroutine set_age



      end module evolve


