! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module element_diffusion

      use star_private_def
      use alert_lib
      use const_def
      use utils_lib, only:is_bad_num, has_bad_num

      implicit none
      
      logical, parameter :: dbg = .false.
      
      
      contains
      

      logical function do_element_diffusion(s, dt_in) 
         ! return false if cannot satisfy accuracy requirements
         use diffusion_lib
         use mlt_def, only: convective_mixing, no_mixing
         use chem_def, only: chem_isos, ihe4, ih1
         use chem_lib, only: chem_get_iso_id
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt_in
         
         integer :: ierr, i, j, k, kk, nc, m, nzlo, nzhi, nz, species, iounit, &
            maxsteps_allowed, steps_used, total_num_retries, cid, he4
                     
         logical, parameter :: check_for_bad_nums = .true.
         
         integer, dimension(:), pointer :: &
            class, class_chem_id, mixing_type, mixing_type_arg
         real(dp) :: s1, s2, dqsum, dist, r, Hp, dt, diffusion_AD_factor, &
            diffusion_AD_velocity, diffusion_vgt_max, diffusion_class_factor(20), &
            gradT_mid, gradRho_mid
         real(dp), dimension(:), pointer :: &
            gamma, dlnP_dr_mid, dlnT_dr_mid, dlnRho_dr_mid, free_e, Lmid, &
            dlnP_dm, dlnT_dm, dlnP_dm_mid, dlnT_dm_mid, dlnRho_dm_mid, dm_for_diffusion
         real(dp), dimension(:,:), pointer :: AP, AT, dlnC_dr_mid, &
            X_init, X_final, typical_charge, v, vgt, GT, xa_save
         real(dp), dimension(:,:,:), pointer :: CD, AX
         character (len=8), pointer :: class_name(:)

         logical :: dumping

         include 'formats.dek'
         
         ierr = 0
         dt = dt_in
         nz = s% nz
         
         if ((.not. s% do_element_diffusion) .or. dt < s% diffusion_dt_limit) then
            do_element_diffusion = .true.
            if (s% do_element_diffusion) then
               s% edv(:,1:nz) = 0
            end if
            return
         end if

         if (s% use_other_diffusion) then
            call s% other_diffusion(s% id, dt_in, ierr)
            do_element_diffusion = (ierr == 0)
            return
         end if
         
         s% num_diffusion_solver_steps = 0

         do_element_diffusion = .false.
         nz = s% nz
         nzlo = 1
         nzhi = nz
         species = s% species
         maxsteps_allowed = s% diffusion_maxsteps
         nc = s% diffusion_num_classes
         m = nc+1
         
         call do_alloc(ierr)
         if (ierr /= 0) then
            call alert(ierr, 'allocate failed for do_element_diffusion')
            return
         end if
         
         call set_extras(ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'do_element_diffusion failed in set_extras'
            return
         end if

         !reset nzlo if necessary; nzlo=1 above
         nzlo = 1
         if (s% mixing_type(1) == convective_mixing) then
            do k=1,nz-1
               if(s% mixing_type(k) == convective_mixing .and. &
                     s% mixing_type(k+1) /= convective_mixing) then
                  nzlo=k; exit
               endif
            enddo
         endif
         
         if (s% diffusion_min_dq_at_surface > 0) then
            dqsum = sum(s% dq(1:nzlo))
            k = nzlo
            do while (dqsum < s% diffusion_min_dq_at_surface)
               k = k+1
               if (k > nz) exit
               ! don't go across composition transition
               if (maxloc(s% xa(:,k),dim=1) /= maxloc(s% xa(:,k-1),dim=1)) exit
               dqsum = dqsum + s% dq(k)
               nzlo = k
            end do
            !write(*,2) 'dqsum', nzlo, dqsum, sum(s% dq(1:nzlo)), s% diffusion_min_dq_at_surface
         end if

         !reset nzhi if necessary; nzhi=nz above
         if (s% mixing_type(nzhi) == convective_mixing) then
            do k=nz,2, -1
               if(s% mixing_type(k) == convective_mixing .and. &
                     s% mixing_type(k-1) /= convective_mixing) then
                  nzhi=k; exit
               endif
            enddo
         end if
         
         do j=1,nc
            cid = chem_get_iso_id(s% diffusion_class_representative(j))
            if (cid <= 0) then
               write(*,'(a,3x,i3)') 'bad entry for diffusion_class_representative: ' // &
                  trim(s% diffusion_class_representative(j)), j
               return
            end if
            class_chem_id(j) = cid
         end do

         call set_diffusion_classes( &
            nc, species, s% chem_id, class_chem_id, s% diffusion_class_A_max, &
            class, class_name)
         
         s% diffusion_call_number = s% diffusion_call_number + 1
         dumping = (s% diffusion_call_number == s% diffusion_dump_call_number)
         
         if (.not. s% diffusion_calculates_ionization) then
            do j=1,nc
               typical_charge(j,1:nz) = s% diffusion_class_typical_charge(j)
            end do
         end if
         
         xa_save(:,1:nz) = s% xa(:,1:nz)
         
         mixing_type(1:nz) = no_mixing
         mixing_type_arg => mixing_type

!         call solve_diffusion( &
!            nz, species, nc, m, class, class_chem_id, s% net_iso, &
!            s% abar, s% ye, free_e, s% mstar, s% dm, &
!            s% T, s% lnT, s% rho, s% lnd, s% r, dlnP_dm, dlnT_dm, &
!            mixing_type, dt, maxsteps_allowed, s% diffusion_calculates_ionization, &
!            typical_charge, s% diffusion_min_Y_for_fe_in_he, &
!            s% diffusion_atol, s% diffusion_rtol, &
!            s% diffusion_v_max, s% diffusion_min_rho_face, &
!            gamma, s% diffusion_gamma_full_on, s% diffusion_gamma_full_off, &
!            s% diffusion_T_full_on, s% diffusion_T_full_off, &
!            s% diffusion_X_full_on, s% diffusion_X_full_off, &
!            s% diffusion_Y_full_on, s% diffusion_Y_full_off, &
!            s% xa, steps_used, total_num_retries, nzlo, nzhi, X_init, X_final, &
!            AP, AT, AX, dlnP_dr_mid, dlnT_dr_mid, dlnC_dr_mid, &
!            v, vgt, GT, CD, &
!            max_num_profile_extras, s% profile_extra, s% profile_extra_name, ierr )
         diffusion_AD_factor = 0
         diffusion_AD_velocity = 0
         diffusion_vgt_max = 1d-5
         diffusion_class_factor = 1
         
         do k=1,nz-1
            s1 = dlnP_dm(k)
            s2 = dlnP_dm(k+1)
            if (s1*s2 <= 0) then
               dlnP_dm_mid(k) = 0
            else
               dlnP_dm_mid(k) = 2*s1*s2/(s1+s2)
            end if
            gradT_mid = 0.5d0*(s% gradT(k) + s% gradT(k+1))
            dlnT_dm_mid(k) = gradT_mid*dlnP_dm_mid(k)
            gradRho_mid = (1 - s% chiT(k)*gradT_mid)/s% chiRho(k)
            dlnRho_dm_mid(k) = gradRho_mid*dlnP_dm_mid(k)
            dm_for_diffusion(k) = 0.5d0*(s% dm(k) + s% dm(k+1))
            Lmid(k) = 0.5d0*(s% L(k) + s% L(k+1))
         end do
         dlnP_dm_mid(nz) = dlnP_dm(nz)
         dlnT_dm_mid(nz) = dlnT_dm(nz)
         gradRho_mid = (1 - s% chiT(nz)*s% gradT(nz))/s% chiRho(nz)
         dlnRho_dm_mid(nz) = gradRho_mid*dlnP_dm_mid(nz)
         dm_for_diffusion(1) = 0.5d0*s% dm(2) + s% dm(1)
         Lmid(1) = 0.5d0*s% L(2) + s% L(1)
         dm_for_diffusion(nz) = 0.5d0*s% dm(nz)
         Lmid(nz) = 0.5d0*s% L(nz)
         
         if (dumping) call dump_diffusion_info
         
         ! all args are at cell center points. dm_for_diffusion is mass between cell centers.
         call solve_diffusion( &
            nz, species, nc, m, class, class_chem_id, s% net_iso, &
            s% abar, s% ye, free_e, s% mstar, dm_for_diffusion, s% dm, &
            s% T, s% lnT, s% rho, s% lnd, s% rmid, &
            dlnP_dm_mid, dlnT_dm_mid, dlnRho_dm_mid, Lmid, &
            dt, maxsteps_allowed, s% diffusion_calculates_ionization, typical_charge, &
            s% diffusion_atol, s% diffusion_rtol, &
            diffusion_AD_factor, diffusion_AD_velocity, diffusion_vgt_max, &
            gamma, s% diffusion_gamma_full_on, s% diffusion_gamma_full_off, &
            s% diffusion_T_full_on, s% diffusion_T_full_off, &
            s% diffusion_X_full_on, s% diffusion_X_full_off, &
            s% diffusion_Y_full_on, s% diffusion_Y_full_off, &
            diffusion_class_factor, s% xa, &
            steps_used, total_num_retries, nzlo, nzhi, X_init, X_final, &
            AP, AT, AX, dlnP_dr_mid, dlnT_dr_mid, dlnRho_dr_mid, dlnC_dr_mid, v, vgt, ierr )
         do_element_diffusion = (ierr == 0)
         s% num_diffusion_solver_steps = steps_used
         
         if (dbg .or. s% show_diffusion_info) write(*,'(a,3x,f20.12,8x,99(a,i6,8x))') &
            'log_dt', log10(s% dt/secyer), &
            'diffusion steps_used', steps_used, 'nzlo', nzlo, 'nzhi', nzhi, 'n', nzhi-nzlo+1, &
            'diffusion_call_number', s% diffusion_call_number
         
         if (.not. do_element_diffusion) s% xa(:,1:nz) = xa_save(:,1:nz)

         if (.not. do_element_diffusion .and. s% report_ierr) then
            write(*, *) 
            write(*, *) 'solve_diffusion returned false'
            write(*, *) 's% model_number', s% model_number
            write(*, *) 's% diffusion_call_number', s% diffusion_call_number
            write(*, *) 
         end if
            
         if (dumping) stop 'debug: dump_diffusion_info'
         
         do k=1,nzlo
            do j=1,species
               s% edv(j,k) = 0
            end do
         end do
         do k=nzlo+1,nzhi 
            do j=1,species
               s% edv(j,k) = v(class(j),k)
            end do
         end do
         do k=nzhi+1,nz
            do j=1,species
               s% edv(j,k) = 0
            end do
         end do

         call dealloc
         
         if (.not. do_element_diffusion) then
            return
         end if
         
         if (check_for_bad_nums) then
            if (has_bad_num(species*nz, s% xa)) then
               write(*, *) 'bad num in xa after calling solve_diffusion: model_number', s% model_number
               ierr = -1
               return
            end if
         end if


         contains
         
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            real(dp), pointer :: p(:)
            
            call get_integer_work_array(s, mixing_type, nz, nz_alloc_extra, ierr)
            if (ierr /= 0) return            
            call get_work_array(s, gamma, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnP_dm, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnT_dm, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnP_dm_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnT_dm_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnRho_dm_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnP_dr_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnT_dr_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnRho_dr_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, free_e, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dm_for_diffusion, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, Lmid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            
            call get_2d_work_array(s, AP, m, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, AT, m, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, dlnC_dr_mid, m, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return

            call get_2d_work_array(s, X_init, nc, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, X_final, nc, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, typical_charge, nc, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, v, nc, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, vgt, nc, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, GT, nc, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, xa_save, species, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return

            call get_work_array(s, p, nc*nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call set_pointer_3(CD, p, nc, nc, nz)
            call get_work_array(s, p, m*m*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call set_pointer_3(AX, p, m, m, nz)
            
            allocate(class(species), class_chem_id(nc), class_name(nc), stat=ierr)

         end subroutine do_alloc
            
         subroutine dealloc
            use alloc
            use utils_lib
            real(dp), pointer :: p(:)
            
            call return_integer_work_array(s, mixing_type)
            call return_work_array(s, gamma, 'diffusion')
            call return_work_array(s, dlnP_dm, 'diffusion')
            call return_work_array(s, dlnT_dm, 'diffusion')
            call return_work_array(s, dlnP_dm_mid, 'diffusion')
            call return_work_array(s, dlnT_dm_mid, 'diffusion')
            call return_work_array(s, dlnRho_dm_mid, 'diffusion')
            call return_work_array(s, dlnP_dr_mid, 'diffusion')
            call return_work_array(s, dlnT_dr_mid, 'diffusion')
            call return_work_array(s, dlnRho_dr_mid, 'diffusion')
            call return_work_array(s, free_e, 'diffusion')
            call return_work_array(s, dm_for_diffusion, 'diffusion')
            call return_work_array(s, Lmid, 'diffusion')

            call set_pointer_1(p, X_init, nc*nz)
            call return_work_array(s, p, 'diffusion')
            call set_pointer_1(p, X_final, nc*nz)
            call return_work_array(s, p, 'diffusion')
            call set_pointer_1(p, typical_charge, nc*nz)
            call return_work_array(s, p, 'diffusion')
            call set_pointer_1(p, v, nc*nz)
            call return_work_array(s, p, 'diffusion')
            call set_pointer_1(p, vgt, nc*nz)
            call return_work_array(s, p, 'diffusion')
            call set_pointer_1(p, GT, nc*nz)
            call return_work_array(s, p, 'diffusion')
            call set_pointer_1(p, xa_save, nc*nz)
            call return_work_array(s, p, 'diffusion')

            call set_pointer_1(p, CD, nc*nc*nz)
            call return_work_array(s, p, 'diffusion')
            call set_pointer_1(p, AX, m*m*nz)
            call return_work_array(s, p, 'diffusion')

            deallocate(class, class_chem_id,  class_name)

         end subroutine dealloc
         
         
         subroutine check_xa_sums(ierr)
            integer, intent(out) :: ierr
            integer :: k
            include 'formats.dek'
            do k=1, nz
               if (abs(sum(s% xa(1:species, k)) - 1d0) > 1d-3) then
                  write(*,*) 'k', k
                  write(*,1) 'sum', sum(s% xa(1:species, k))
                  write(*,1) 'sum-1d0', sum(s% xa(1:species, k))-1d0
                  write(*,1) 'abs(sum-1d0)', abs(sum(s% xa(1:species, k))-1d0)
                  do j=1,species
                     write(*,2) 's% xa(j,k)', j, s% xa(j, k)
                  end do
                  ierr = -1
               end if
            end do
         end subroutine check_xa_sums
         
         
         subroutine dump_diffusion_info
            use utils_lib
            use chem_def, only: chem_isos
            integer :: i, k, ierr
            ierr = 0
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) return
            write(*, *)
            write(*, *) 'write diffusion.data'
            write(*, *)
            open(iounit, file='diffusion.data', action='write', status='replace', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'failed to open diffusion dump file'
               call free_iounit(iounit)
               return
            end if
            ! args
            write(iounit, '(99i20)') nz, nzlo, nzhi, species, nc, maxsteps_allowed
            
            do i=1,species
               write(iounit,*) trim(chem_isos% name(s% chem_id(i)))
            end do
            write(iounit, '(99i20)') class(1:species)
            do i=1,nc
               write(iounit, '(a)') trim(chem_isos% name(class_chem_id(i)))
            end do
            do i=1,nc
               write(iounit, '(a)') trim(class_name(i))
            end do
            
            if (s% diffusion_calculates_ionization) then
               write(iounit,*) 1
            else
               write(iounit,*) 0
            end if
            
            write(iounit, '(99(1pe26.16))') &
               s% mstar, s% dt, s% diffusion_atol, s% diffusion_rtol, &
               0d0, 0d0, & ! diffusion_AD_factor, diffusion_AD_velocity
               s% diffusion_v_max, s% diffusion_gamma_full_on, s% diffusion_gamma_full_off, &
               s% diffusion_T_full_on, s% diffusion_T_full_off, &
               s% diffusion_X_full_on, s% diffusion_X_full_off, &
               s% diffusion_Y_full_on, s% diffusion_Y_full_off

            do k=1, nz
               write(iounit, '(99(1pe26.16))') &
                  gamma(k), s% abar(k), s% ye(k), free_e(k), dm_for_diffusion(k), s% dm(k), &
                  s% T(k), s% lnT(k), s% Rho(k), s% lnd(k), s% rmid(k), &
                  dlnP_dm_mid(k), dlnT_dm_mid(k), dlnRho_dm_mid(k), Lmid(k)
               write(iounit, '(99(1pe26.16))') s% xa(1:species, k)
            end do
            
            close(iounit)
            call free_iounit(iounit)
            
         end subroutine dump_diffusion_info
         
         
         subroutine set_extras(ierr)
            integer, intent(out) :: ierr
            integer :: k, op_err
            op_err = 0
            ierr = 0
!$OMP PARALLEL DO PRIVATE(k, op_err)
            do k=1,nz
               call set1_extras(k, op_err)
               if (op_err /= 0) ierr = op_err
            end do
!$OMP END PARALLEL DO
         end subroutine set_extras
         
         
         subroutine set1_extras(k,ierr)
            use eos_lib, only: Plasma_Coupling_Parameter
            use hydro_eqns_dble, only: eval_dlnPdm_qhse
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            real(dp) :: &
               m, d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
               d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, d_dlnPdm_dL, &
               P, dP_dlnd00, dP_dlndm1, dP_dlnT00, dP_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
               dP_dlnPgas00_const_T, dP_dlnPgasm1_const_T, &
               dP_dlnT00_const_Pgas, dP_dlnTm1_const_Pgas
            ierr = 0
            free_e(k) = exp(s% lnfree_e(k))
            gamma(k) = Plasma_Coupling_Parameter(s% T(k), s% rho(k), s% abar(k), s% zbar(k))
            if (k==1) then
               dlnP_dm(k) = 0; dlnT_dm(k) = 0; return
            end if
            call eval_dlnPdm_qhse(s, k, m, &
               dlnP_dm(k), d_dlnPdm_dlnR, d_dlnPdm_dlnq, d_dlnPdm_dL, &
               d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
               d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
               P, &
               dP_dlnd00, dP_dlndm1, dP_dlnT00, dP_dlnTm1, &
               dP_dlnPgas00_const_T, dP_dlnPgasm1_const_T, &
               dP_dlnT00_const_Pgas, dP_dlnTm1_const_Pgas, &
               ierr)
            if (ierr /= 0) return
            dlnT_dm(k) = s% gradT(k)*dlnP_dm(k)
         end subroutine set1_extras


         subroutine merge_cells ! merges cells with mixing_type = convective_mixing
            integer :: new_nz, k, current
            integer, pointer :: new_cell_number(:)
            
            allocate(new_cell_number(nz))
            
            current = 1
            do k = 1, nz
               if( s% mixing_type(k) /= convective_mixing ) then
                  new_cell_number(k)=current
                  current=current+1
               else if (current /=1)then
                  new_cell_number(k)=current
               endif
            enddo
            
            do k=1,nz
               write(*,'(3i6)') k, new_cell_number(k), s% mixing_type(k)
            enddo
            
            new_nz = new_cell_number(nz)
            
            deallocate(new_cell_number)

         end subroutine merge_cells


      end function do_element_diffusion
            

      end module element_diffusion







         
         



