! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module do_one_utils
      
      use star_private_def
      use const_def
      use utils_lib, only:alloc_iounit, free_iounit

      implicit none
      
      ! model log priorities
      integer, parameter :: delta_priority = 1
      integer, parameter :: phase_priority = 2

         
      real(dp), parameter :: del_cntr_rho = 1d0
      real(dp), parameter :: min_cntr_rho = 3d0
      real(dp), parameter :: no_he_ignition_limit = 0.75d0
      real(dp), parameter :: no_cntr_T_drops_limit = 6.5d0
      
      real(dp), parameter :: center_h_gone = 1d-3
      real(dp), parameter :: center_h_going = one_third
      real(dp), parameter :: center_he_going = 5d-2
      
      
      contains
      
      
      logical function model_is_okay(s)
         type (star_info), pointer :: s
         ! for now, just check for valid number in the final dynamic timescale
         model_is_okay = ((s% dynamic_timescale - s% dynamic_timescale) .eq. 0d0) &
                        .and. ((s% dynamic_timescale + 1d0) > 1d0)
      end function model_is_okay

      
      subroutine do_one_finish(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         ierr = id ! for compiler
         ierr = 0 ! nothing to do for now.
      end subroutine do_one_finish
      
      
      subroutine record_last_model(s)
         type (star_info), pointer :: s
         logical :: logged
         logged = log_state ( s, .true. )
         if (s% do_log_files) then
            call set_save_profiles_info(s, phase_priority)
         end if
      end subroutine record_last_model
      
      
      subroutine set_save_profiles_info(s, model_priority)
         type (star_info), pointer :: s
         integer, intent(in) :: model_priority
         s% need_to_save_profiles_now = .true.
         s% save_profiles_model_priority = model_priority
      end subroutine set_save_profiles_info
      
      
      subroutine do_save_profiles( &
            s, id_extra, how_many_extra_profile_columns, data_for_extra_profile_columns, ierr)
         use profile,only:write_profile_info
         use utils_lib, only: number_iounits_allocated
         type (star_info), pointer :: s
         integer, intent(in) :: id_extra
         interface
            include 'extra_profile_cols.dek'
         end interface
         integer, intent(out) :: ierr

         integer, pointer, dimension(:) :: model_numbers, model_priorities, model_logs
         integer :: nz, max_num_mods, num_models, model_log_number
         character (len=256) :: fname
         integer :: model_priority, current_num_iounits_in_use, prev_num_iounits_in_use
         
         include 'formats.dek'
         
         ierr = 0
         nz = s% nz
         prev_num_iounits_in_use = number_iounits_allocated()

         if (.not. s% do_profiles) return
         if (.not. s% v_flag) then
            s% v(1:nz) = 0
            s% dv_dt(1:nz)  = 0
         end if
         if (.not. s% rotation_flag) then
            s% omega(1:nz) = 0
         end if

         max_num_mods = s% max_num_profile_models
         model_priority = s% save_profiles_model_priority

         allocate(model_numbers(max_num_mods), model_priorities(max_num_mods), &
            model_logs(max_num_mods), stat=ierr)
         if (ierr /= 0) return

         write(fname, '(3a)') trim(s% log_directory), '/', trim(s% profiles_index_name)
         
         call read_profiles_info( &
            fname, max_num_mods, num_models, model_numbers, model_priorities, model_logs)
         
         call check(1)

         call make_room_for_profile_info( &
            s% model_number, max_num_mods, num_models, model_numbers, model_priorities, model_logs, ierr)
         if (ierr /= 0) then
            call dealloc; return
         end if
         
         call check(2)
         
         call pick_model_log_number( &
            max_num_mods, num_models, model_logs, model_log_number, ierr)
         if (ierr /= 0) then
            call dealloc; return
         end if
         
         call check(3)
      
         call get_model_profilename(s, model_log_number)
      
         ! add the new model to the list at the end
         num_models = num_models+1
         model_numbers(num_models) = s% model_number
         model_priorities(num_models) = model_priority
         model_logs(num_models) = model_log_number
         
         s% save_profiles_model_priority = delta_priority ! reset it to the default value
						
         ! write the profiles before adding them to the list
         ! so if user interrupts during write, the index is still okay.
         call write_profile_info(s, s% model_profilename, &
            id_extra, how_many_extra_profile_columns, data_for_extra_profile_columns, ierr)
         if (ierr /= 0) then
            call dealloc; return
         end if
         
         call check(4)

         call write_profiles_list( &
            fname, num_models, model_numbers, model_priorities, model_logs, ierr)
         if (ierr /= 0) then
            call dealloc; return
         end if
         
         call check(5)
         
         s% profile_age = s% star_age
         s% prv_log_luminosity = s% log_surface_luminosity
         s% prv_log_surface_temp = s% log_surface_temperature
         s% prv_log_center_temp = s% log_center_temperature
         s% prv_log_center_density = s% log_center_density
         s% next_cntr_rho = s% next_cntr_rho + del_cntr_rho
         
         call dealloc
         
         
         contains
         
         subroutine dealloc
            deallocate(model_numbers, model_priorities, model_logs)
         end subroutine dealloc
         
         subroutine check(i)
            integer, intent(in) :: i
            include 'formats.dek'
            !return
            
            current_num_iounits_in_use = number_iounits_allocated()
            if (current_num_iounits_in_use > 3 .and. &
                  current_num_iounits_in_use > prev_num_iounits_in_use) then
               write(*,2) 's% model_number', s% model_number
               write(*,2) 'prev_num_iounits_in_use', prev_num_iounits_in_use
               write(*,2) 'current_num_iounits_in_use', current_num_iounits_in_use
               write(*,2) 'i', i
               stop 'do_save_profiles' 
            end if
            prev_num_iounits_in_use = current_num_iounits_in_use
         end subroutine check
         
      end subroutine do_save_profiles
      
      
      subroutine write_terminal_header(s)
         type (star_info), pointer :: s
         if (s% model_number <= s% recent_log_header) return
         if (s% just_wrote_terminal_header) return
         s% recent_log_header = s% model_number
         call do_show_terminal_header(s)
         s% just_wrote_terminal_header = .true.
      end subroutine write_terminal_header
      
      
      subroutine do_show_log_description(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         write(*,*)
         write(*,'(a)') " The terminal output contains the following information"
         write(*,*)
         write(*,'(a)') "      'step' is the number of steps since the start of the run,"
         write(*,'(a)') "      'lg_dt_yr' is log10 timestep in years,"
         write(*,'(a)') "      'age_yr' is the simulated years since the start run,"
         write(*,'(a)') "      'lg_Tcntr' is log10 center temperature (K),"
         write(*,'(a)') "      'lg_Dcntr' is log10 center density (g/cm^3),"
         write(*,'(a)') "      'lg_Pcntr' is log10 center pressure (ergs/cm^3),"
         write(*,'(a)') "      'Teff' is the surface temperature (K),"
         write(*,'(a)') "      'lg_R' is log10 surface radius (Rsun),"
         write(*,'(a)') "      'lg_L' is log10 surface luminosity (Lsun),"
         write(*,'(a)') "      'lg_LH' is log10 total PP and CNO hydrogen burning power (Lsun),"
         write(*,'(a)') "      'lg_L3a' is log10 total triple-alpha helium burning power (Lsun),"
         write(*,'(a)') &
         "      'lg_LZ' is log10 total burning power excluding LH and L3a and photodisintegrations (Lsun),"
         write(*,'(a)') "      'lg_LNuc' is log10 nuclear power excluding photodisintegration (Lsun),"
         write(*,'(a)') "      'lg_LNeu' is log10 total neutrino power (Lsun),"
         write(*,'(a)') "      'lg_Psurf' is log10 surface pressure (gas + radiation),"
         write(*,'(a)') "      'Mass' is the total stellar mass (Msun),"
         write(*,'(a)') "      'lg_Mdot' is log10 magnitude of rate of change of mass (Msun/year),"
         write(*,'(a)') "      'lg_Dsurf' is log10 surface density (g/cm^3),"
         write(*,'(a)') "      'H_rich' is the remaining mass outside of the hydrogen poor core,"
         write(*,'(a,e9.2)') "      'H_poor' is the core mass where hydrogen abundance is <=", s% h1_boundary_limit
         write(*,'(a,e9.2)') "      'He_poor' is the core mass where helium abundance is <=", s% he4_boundary_limit
         write(*,'(a)') "      'H_cntr' is the center H1 mass fraction,"
         write(*,'(a)') "      'He_cntr' is the center He4 mass fraction,"
         write(*,'(a)') "      'C_cntr' is the center C12 mass fraction,"
         write(*,'(a)') "      'N_cntr' is the center N14 mass fraction,"
         write(*,'(a)') "      'O_cntr' is the center O16 mass fraction,"
         write(*,'(a)') "      'Ne_cntr' is the center Ne20 mass fraction,"
         write(*,'(a)') "      'X_avg' is the star average hydrogen mass fraction,"
         write(*,'(a)') "      'Y_avg' is the star average helium mass fraction,"
         write(*,'(a)') "      'Z_avg' is the star average metallicity,"
         write(*,'(a)') "      'gam_cntr' is the center plasma interaction parameter,"
         write(*,'(a)') "      'eta_cntr' is the center electron degeneracy parameter,"
         write(*,'(a)') "      'pts' is the number of grid points in the current model,"
         write(*,'(a)') "      'iters' is the number of newton iterations for the current step,"
         write(*,'(a)') "      'retry' is the number of step retries required during the run,"
         write(*,'(a)') "      'bckup' is the number of step backups required during the run,"
         write(*,'(a)') "      'dt_limit' is an indication of what limited the timestep."
         write(*,*)
         write(*,'(a)') " All this and more are saved in 'LOGS/star.log' during the run."
      end subroutine do_show_log_description
      
      
      subroutine do_show_terminal_header(s)
         type (star_info), pointer :: s
         call output_terminal_header(s,terminal_iounit)
         if (s% extra_terminal_iounit > 0) &
            call output_terminal_header(s,s% extra_terminal_iounit)
      end subroutine do_show_terminal_header
      
      
      subroutine output_terminal_header(s,io)
         use chem_def, only: isi28
         type (star_info), pointer :: s
         integer, intent(in) :: io
         character (len=5) :: iters
         if (s% doing_hydro_newton) then
            iters = 'iters'
         else
            iters = 'rows '
         end if
         write(io,'(a)') &
            '_______________________________________________________________________' // &
            '___________________________________________________________________________'
         write(io,*)
         write(io,'(a)') &
            '       step    lg_Tcntr    Teff       lg_LH     lg_Lnuc     Mass       ' // &
            'H_rich     H_cntr     N_cntr     Y_surf     X_avg     eta_cntr   pts  retry'
         if (s% initial_z >= 1d-5) then
            write(io,'(a)') &
               '   lg_dt_yr    lg_Dcntr    lg_R       lg_L3a    lg_Lneu     lg_Mdot    ' // &
               'H_poor     He_cntr    O_cntr     Z_surf     Y_avg     gam_cntr  ' // iters // ' bckup'
         else
            write(io,'(a)') &
               '   lg_dt_yr    lg_Dcntr    lg_R       lg_L3a    lg_Lneu     lg_Mdot    ' // &
               'H_poor     He_cntr    O_cntr     lg_Z_surf  Y_avg     gam_cntr  ' // iters // ' bckup'
         end if
         if (s% net_iso(isi28) == 0) then
            write(io,'(a)') &
               '        age    lg_Pcntr    lg_L       lg_LZ     lg_Psurf    lg_Dsurf   ' // &
               'He_poor    C_cntr     Ne_cntr    Z_cntr     Z_avg     v_div_cs     dt_limit'
         else
            write(io,'(a)') &
               '        age    lg_Pcntr    lg_L       lg_LZ     lg_Psurf    lg_Dsurf   ' // &
               'He_poor    C_cntr     Ne_cntr    Si_cntr    Z_avg     v_div_cs     dt_limit'
         end if
         write(io,'(a)') &
            '_______________________________________________________________________' // &
            '___________________________________________________________________________'
         write(io,*)
      end subroutine output_terminal_header
      
      
      subroutine do_terminal_summary(s)
         type (star_info), pointer :: s
         call output_terminal_summary(s,terminal_iounit)
         if (s% extra_terminal_iounit > 0) then
            call output_terminal_summary(s,s% extra_terminal_iounit)
            flush(s% extra_terminal_iounit)
         end if
      end subroutine do_terminal_summary
      
      
      subroutine output_terminal_summary(s,io)
         use num_def, only:banded_matrix_type
         use num_lib, only:safe_log10
         use const_def, only:secyer
         use chem_def
         use utils_lib, only: is_bad_num
         use rates_def, only: i_rate
         use star_utils, only:eval_current_y, eval_current_z
         type (star_info), pointer :: s
         integer, intent(in) :: io
         
         real(dp) :: age, dt, x_avg, y_avg, z_avg, Xmax, v, vsurf_div_csound, &
            power_nuc_burn, power_h_burn, power_he_burn, power_photo, power_neutrinos, tmp
         integer :: model, ierr, nz, iters
         character (len=3) :: id_str
         character (len=32) :: why
         character (len=90) :: fmt, fmt1, fmt2, fmt3, fmt4, fmt5
         
         include 'formats.dek'
         
         if (s% terminal_show_age_in_years) then
            age = s% star_age
         else
            age = s% time
         end if
         model = s% model_number
         nz = s% nz

         ierr = 0         

         x_avg = s% star_mass_h1/s% star_mass
         y_avg = (s% star_mass_he3 + s% star_mass_he4)/s% star_mass
         z_avg = max(0d0,min(1d0, 1d0 - (x_avg + y_avg)))
         
         Xmax = dot_product(s% dq(1:nz), s% xa(s% species,1:nz))
         
         if (s% v_flag) then
            v = s% v(1)
         else
            v = s% r(1)*s% dlnR_dt(1)
         end if
         vsurf_div_csound = v / sqrt(s% gamma1(1)*s% P(1)/s% rho(1))

         dt = s% time_step*secyer
         
         power_nuc_burn = s% power_nuc_burn
         power_h_burn = s% power_h_burn
         power_he_burn = s% power_he_burn
         power_photo = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,iphoto,1:nz))/Lsun
         power_neutrinos = s% power_neutrinos
         
         if (s% id == 1) then
            id_str = ''
         else
            write(id_str,'(i3)') s% id
         end if
         
         fmt1 = '(a3,i8,f11.6,'
         
         if (s% Teff < 1d4) then
            fmt2 = 'f11.3,'
         else
            fmt2 = '1pe11.3,0p,'
         end if
         
         if (s% star_mass >= 1d2) then
            fmt3 = '2f11.6,2(1pe11.3),0p,'
         else
            fmt3 = '4f11.6,'
         end if
         
         if (s% eta(s% nz) >= 1d3) then
            fmt4 = '4f11.6,e11.3,'
         else
            fmt4 = '4f11.6,f11.6,'
         end if
         fmt5 = '2i7)'
         
         fmt = trim(fmt1) // trim(fmt2) // trim(fmt3) // trim(fmt4) // trim(fmt5)
         write(io,fmt=fmt) &
            id_str, model, &
            s% log_center_temperature, &   ! fmt1
            s% Teff, &   ! fmt2
            safe_log10(power_h_burn), & ! fmt3
            safe_log10(power_nuc_burn - power_photo), &
            s% star_mass, &            
            s% star_mass-s% h1_boundary_mass, &
            s% center_h1, & ! fmt4
            s% center_n14, &
            s% surface_he3 + s% surface_he4, &
            x_avg, &
            s% eta(s% nz), &
            s% nz, & ! fmt5
            s% num_retries
         
         if (s% initial_z >= 1d-5) then
            tmp = 1 - (s% surface_h1 + s% surface_he3 + s% surface_he4)
            fmt1 = '(11f11.6, '
         else
            tmp = safe_log10(1 - (s% surface_h1 + s% surface_he3 + s% surface_he4))
            fmt1 = '(9f11.6, e11.2, f11.6, '
         end if
         if (s% gam(s% nz) >= 1d3) then
            fmt2 = 'e11.3, '
         else
            fmt2 = 'f11.6, '
         end if
         fmt3 = ' 2i7)'
         fmt = trim(fmt1) // trim(fmt2) // trim(fmt3)
         if (s% doing_hydro_newton) then
            iters = s% num_jacobians
         else
            iters = s% seulex_rows
         end if
         write(io,fmt=fmt) &
            log10(s% time_step),  &
            s% log_center_density, &
            s% log_surface_radius, &
            safe_log10(power_he_burn), &
            safe_log10(power_neutrinos), &
            safe_log10(abs(s% star_mdot)), &
            s% h1_boundary_mass, &
            s% center_he3 + s% center_he4, &
            s% center_o16, &
            tmp, &
            y_avg, &
            s% gam(s% nz), &
            iters, &
            s% num_backups 
         
         why = dt_why_str(min(numTlim,s% why_Tlim))
         if (s% why_Tlim == Tlim_dX_nuc .and. s% Tlim_dXnuc_species > 0 &
                  .and. s% Tlim_dXnuc_species <= s% species) then
            why = trim(dt_why_str(s% why_Tlim)) // ' ' // &
               trim(chem_isos% name(s% chem_id(s% Tlim_dXnuc_species)))
         elseif (s% why_Tlim == Tlim_dX .and. s% Tlim_dX_species > 0 &
                  .and. s% Tlim_dX_species <= s% species) then
            why = trim(dt_why_str(s% why_Tlim)) // ' ' // &
               trim(chem_isos% name(s% chem_id(s% Tlim_dX_species)))
         else if (s% why_Tlim ==  Tlim_dX_div_X .and. s% Tlim_dX_div_X_species > 0 &
                  .and. s% Tlim_dX_div_X_species <= s% species) then
            why = trim(dt_why_str(s% why_Tlim)) // ' ' // &
               trim(chem_isos% name(s% chem_id(s% Tlim_dX_div_X_species)))
         else if (s% why_Tlim ==  Tlim_dlgL_nuc_cat) then 
            if (s% Tlim_dlgL_nuc_category > 0 &
                  .and. s% Tlim_dlgL_nuc_category <= num_categories ) then
               why = trim(category_name(s% Tlim_dlgL_nuc_category))
            else
               why = '???'
            end if
         end if         
         
         s% why_Tlim = Tlim_struc ! restore default reason for timestep choice
         
         if (s% net_iso(isi28) == 0) then
            tmp = 1 - (s% center_h1 + s% center_he3 + s% center_he4)
            write(io,'(1pe11.4, 0p, 5f11.6, 3f11.6, 1p2e11.3, 0p, e11.3, a14)') &
               age, &
               s% log_center_pressure, &
               s% log_surface_luminosity, &
               safe_log10(power_nuc_burn - (power_photo + power_h_burn + power_he_burn)), &
               safe_log10(s% P_surf), &
               s% lnd(1)/ln10, &
               s% he4_boundary_mass, &
               s% center_c12, &
               s% center_ne20, &
               tmp, &
               z_avg, &
               vsurf_div_csound, &
               trim(why)
         else
            tmp = s% center_si28
            write(io,'(1pe11.4, 0p, 5f11.6, 0p4f11.6, 1pe11.3, 0p, e11.3, a14)') &
               age, &
               s% log_center_pressure, &
               s% log_surface_luminosity, &
               safe_log10(power_nuc_burn - (power_photo + power_h_burn + power_he_burn)), &
               safe_log10(s% P_surf), &
               s% lnd(1)/ln10, &
               s% he4_boundary_mass, &
               s% center_c12, &
               s% center_ne20, &
               tmp, &
               z_avg, &
               vsurf_div_csound, &
               trim(why)
         end if
                     
         write(io,*)
         
         s% just_wrote_terminal_header = .false.

      end subroutine output_terminal_summary
      
      
      logical function log_state (s, do_write)
         use profile,only:write_profile_info
         type (star_info), pointer :: s
         logical, intent(in) :: do_write                  
         integer :: model
         logical :: write_logfile, write_terminal         
         include 'formats.dek'
         model = s% model_number
         if (s% log_cnt > 0) then
            write_logfile = (mod(model, s% log_cnt) == 0) .or. do_write
         else
            write_logfile = .false.
         end if
         if (s% terminal_cnt > 0) then
            write_terminal = (mod(model, s% terminal_cnt) == 0) .or. do_write
         else
            write_terminal = .false.
         end if
         log_state = write_logfile .or. write_terminal         
         if (.not. log_state) return         
         if ( mod(model, s% write_header_frequency*s% terminal_cnt) .eq. 0 &
               .and. .not. s% doing_first_model_of_run) then
            write(*,*)
            call write_terminal_header(s)
         end if         
         if (write_terminal) call do_terminal_summary(s)  
         if (write_logfile) s% need_to_update_logfile_now = .true.   
               
      end function log_state
      
        
      integer function do_bare_bones_check_model(id)
         integer, intent(in) :: id
         integer :: ierr
         logical :: logged
         type (star_info), pointer :: s
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            do_bare_bones_check_model = terminate
            return
         end if
         logged = log_state( s, .false. )
         do_bare_bones_check_model = keep_going
      end function do_bare_bones_check_model
      
      
      subroutine save_profile(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         call set_save_profiles_info(s, phase_priority)
      end subroutine save_profile

        
      integer function do_check_limits(id)
         use num_lib, only:safe_log10
         use rates_def
         use chem_def
         use chem_lib, only: chem_get_iso_id
         use report, only: center_avg_x, surface_avg_x
         use star_utils, only: dt_Courant, omega_crit
         integer, intent(in) :: id
         type (star_info), pointer :: s
         integer :: ierr, i, j, k, cid, k_burn, k_omega, nz
         real(dp) :: log_surface_gravity, L_H_burn_total, &
            power_nuc_burn, power_h_burn, power_he_burn, power_c_burn, logQ, max_logQ, &
            envelope_fraction_left, avg_x, v_surf, csound_surf, delta_nu, &
            ratio, dt_C, peak_burn_vconv_div_cs, min_pgas_div_p, &
            max_omega_div_omega_crit, omega_div_omega_crit, log_Teff
         
         include 'formats.dek'
         
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            do_check_limits = terminate
            return
         end if

         nz = s% nz
         do_check_limits = keep_going

         csound_surf = sqrt(s% gamma1(1)*s% P(1)/s% rho(1))
         if (.not. s% v_flag) then
            v_surf =  s% r(1) * s% dlnR_dt(1)
         else
            v_surf = s% v(1)
         end if
         
         log_surface_gravity = log10(s% grav(1))
         power_nuc_burn = s% power_nuc_burn
         power_h_burn = s% power_h_burn
         power_he_burn = s% power_he_burn
         power_c_burn = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,i_burn_c,1:nz))/Lsun
         L_H_burn_total = safe_log10(power_h_burn)
         log_Teff = log10(s% Teff)
         
         delta_nu = 1d6/(2*s% photosphere_acoustic_r)
         
         k = maxloc(s% eps_nuc(1:nz), dim=1)
         peak_burn_vconv_div_cs = s% conv_vel(k)/s% csound(k)
         
         if (s% initial_mass > s% h1_boundary_mass) then
            envelope_fraction_left = &
               (s% star_mass - s% h1_boundary_mass)/(s% initial_mass - s% h1_boundary_mass)
         else
            envelope_fraction_left = 1
         end if
         
         max_logQ = -99
         do k = 1, s% nz
            if (s% lnT(k)/ln10 < 5.5d0) then ! only worry about lower T cases
               logQ = s% lnd(k)/ln10 - 2*s% lnT(k)/ln10 + 12
               if (logQ > max_logQ) max_logQ = logQ
            end if
         end do
         
         min_pgas_div_p = 1d99
         do k = s% nz, 1, -1
            if (s% q(k) > s% Pgas_div_P_limit_max_q) exit
            if (s% pgas(k)/s% p(k) < min_pgas_div_p) min_pgas_div_p = s% pgas(k)/s% p(k)
         end do
         
         max_omega_div_omega_crit = 0; k_omega = 0
         if (s% rotation_flag .and. s% omega_div_omega_crit_limit > 0) then
            do k = 1, s% nz
               omega_div_omega_crit = s% omega(k)/omega_crit(s,k) 
               if (omega_div_omega_crit > max_omega_div_omega_crit) then
                  k_omega = k
                  max_omega_div_omega_crit = omega_div_omega_crit
               end if
            end do
         end if
         !write(*,2) 'max_omega_div_omega_crit', k_omega, max_omega_div_omega_crit, &
         !   s% omega(k_omega)/omega_crit(s,k_omega)
         !write(*,2) 'surf omega_div_omega_crit', 1, s% omega(1)/omega_crit(s,1)
               
         !dt_C = dt_Courant(s)
         !if (s% dt < 2*dt_C) write(*,1) 'dt/dt_Courant', s% dt/dt_C
         
         if (peak_burn_vconv_div_cs > 0.75*s% peak_burn_vconv_div_cs_limit) then
            write(*,1) 'peak_burn_vconv_div_cs: ', &
               peak_burn_vconv_div_cs / s% peak_burn_vconv_div_cs_limit, &
               peak_burn_vconv_div_cs, s% peak_burn_vconv_div_cs_limit
            k = maxloc(s% eps_nuc, dim=1)
            write(*,2) 'maxloc eps_nuc', k, s% conv_vel(k), s% csound(k), s% eps_nuc(k)
            stop 'test do_one_utils'
         end if
         
         
         
         
         if (s% iron_core_infall < s% iron_core_infall_limit .and. &
             s% iron_core_infall > 0.1d0*s% iron_core_infall_limit) &
            write(*,1) 'nearing iron_core_infall limit', s% iron_core_infall, s% iron_core_infall_limit
         
         if (max_omega_div_omega_crit > 0.75d0*s% omega_div_omega_crit_limit .and. &
               s% omega_div_omega_crit_limit > 0 .and. k_omega > 0) &
            write(*,2) 'omega_div_omega_crit', k_omega, &
               max_omega_div_omega_crit, s% omega_div_omega_crit_limit, &
               s% m(k_omega)/Msun, s% r_equitorial(k_omega)/Rsun, &
               s% omega(k_omega), sqrt(s% cgrav(k_omega)*s% m(k_omega)/ s% r_equitorial(k_omega)**3)
         
         if (s% star_age >= s% max_age) then 
            call compare_to_target('star_age >= max_age', s% star_age, s% max_age)
            
         else if (max_omega_div_omega_crit >= s% omega_div_omega_crit_limit .and. &
               s% omega_div_omega_crit_limit > 0) then 
            write(*, '(/,a,/, 2e20.10)') &
               'stop max_omega_div_omega_crit >= omega_div_omega_crit_limit', &
               max_omega_div_omega_crit, s% omega_div_omega_crit_limit
            do_check_limits = terminate
            s% result_reason = result_reason_normal
                  
         else if (peak_burn_vconv_div_cs >= s% peak_burn_vconv_div_cs_limit) then 
            write(*, '(/,a,/, 2e20.10)') &
               'stop peak_burn_vconv_div_cs >= peak_burn_vconv_div_cs_limit', &
               peak_burn_vconv_div_cs, s% peak_burn_vconv_div_cs_limit
            do_check_limits = terminate
            s% result_reason = result_reason_normal
            
         else if (s% model_number >= s% max_model_number .and. s% max_model_number >= 0) then 
            write(*, '(/,a,/, 2i9)') 'stop because model_number >= max_model_number', &
               s% model_number, s% max_model_number
            do_check_limits = terminate
            s% result_reason = result_reason_normal
            
         else if (s% center_degeneracy >= s% eta_center_limit) then 
            call compare_to_target('center_degeneracy >= eta_center_limit', &
               s% center_degeneracy, s% eta_center_limit)
            
         else if (s% log_center_temperature >= s% log_center_temp_limit) then 
            call compare_to_target('log_center_temperature >= log_center_temp_limit', &
               s% log_center_temperature, s% log_center_temp_limit)
            
         else if (s% log_center_temperature <= s% log_center_temp_lower_limit) then 
            call compare_to_target('log_center_temperature <= log_center_temp_lower_limit', &
               s% log_center_temperature, s% log_center_temp_lower_limit)
            
         else if (s% log_center_density >= s% log_center_density_limit) then 
            call compare_to_target('log_center_density >= log_center_density_limit', &
               s% log_center_density, s% log_center_density_limit)
            
         else if (s% center_gamma > s% gamma_center_limit) then 
            call compare_to_target('center_gamma > gamma_center_limit', &
               s% center_gamma, s% gamma_center_limit)
            
         else if (s% center_he4 < s% HB_limit .and. s% center_h1 < 1d-4) then 
            call compare_to_target('center he4 < HB_limit', s% center_he4, s% HB_limit)
            
         else if (s% stop_at_TP .and. s% center_he4 < 1d-4 .and. &
                     s% h1_boundary_mass - s% he4_boundary_mass <= s% TP_he_shell_max .and. &
                     any(s% burn_he_conv_region(1:s% num_conv_boundaries))) then
            call compare_to_target('have AGB thermal pulse', &
               s% h1_boundary_mass - s% he4_boundary_mass, s% TP_he_shell_max)
            
         else if (s% star_mass_min_limit > 0 .and. s% star_mass <= s% star_mass_min_limit) then 
            call compare_to_target('star_mass <= star_mass_min_limit', &
               s% star_mass, s% star_mass_min_limit)
            
         else if (s% star_mass_max_limit > 0 .and. s% star_mass >= s% star_mass_max_limit) then 
            call compare_to_target('star_mass >= star_mass_max_limit', &
               s% star_mass, s% star_mass_max_limit)
            
         else if (s% star_H_mass_max_limit > 0 .and. s% star_mass_h1 >= s% star_H_mass_max_limit) then 
            call compare_to_target('star_mass_h1 >= star_H_mass_max_limit', &
               s% star_mass_h1, s% star_H_mass_max_limit)
            
         else if (s% star_H_mass_min_limit > 0 .and. s% star_mass_h1 <= s% star_H_mass_min_limit) then 
            call compare_to_target('star_mass_h1 <= star_H_mass_min_limit', &
               s% star_mass_h1, s% star_H_mass_min_limit)
            
         else if (s% xmstar_min_limit > 0 .and. s% xmstar <= s% xmstar_min_limit) then 
            call compare_to_target('xmstar <= xmstar_min_limit', &
               s% xmstar, s% xmstar_min_limit)
            
         else if (s% xmstar_max_limit > 0 .and. s% xmstar >= s% xmstar_max_limit) then 
            call compare_to_target('xmstar >= xmstar_max_limit', &
               s% xmstar, s% xmstar_max_limit)
            
         else if (s% star_mass - s% h1_boundary_mass < s% envelope_mass_limit) then 
            call compare_to_target('envelope mass < envelope_mass_limit', &
               s% star_mass - s% h1_boundary_mass, s% envelope_mass_limit)
            
         else if (envelope_fraction_left < s% envelope_fraction_left_limit) then 
            call compare_to_target('envelope_fraction_left < limit', &
               envelope_fraction_left, s% envelope_fraction_left_limit)
            
         else if (s% h1_boundary_mass >= s% h1_boundary_mass_limit) then 
            call compare_to_target('h1_boundary_mass >= h1_boundary_mass_limit', &
               s% h1_boundary_mass, s% h1_boundary_mass_limit)
            
         else if (s% he4_boundary_mass >= s% he4_boundary_mass_limit) then 
            call compare_to_target('he4_boundary_mass >= he4_boundary_mass_limit', &
               s% he4_boundary_mass, s% he4_boundary_mass_limit)
            
         else if (s% c12_boundary_mass >= s% c12_boundary_mass_limit) then 
            call compare_to_target('c12_boundary_mass >= c12_boundary_mass_limit', &
               s% c12_boundary_mass, s% c12_boundary_mass_limit)
            
         else if ( &
               s% h1_boundary_mass >= s% he4_boundary_mass .and. s% he4_boundary_mass > 0 .and. &
               s% center_he4 < 1d-4 .and. &
               s% h1_boundary_mass - s% he4_boundary_mass < s% he_layer_mass_lower_limit) then 
            call compare_to_target('he layer mass < he_layer_mass_lower_limit', &
               s% h1_boundary_mass - s% he4_boundary_mass, s% he_layer_mass_lower_limit)
            
         else if (abs(safe_log10(L_H_burn_total/Lsun) - s% log_surface_luminosity) <= &
                  s% abs_diff_lg_LH_lg_Ls_limit &
                  .and. s% abs_diff_lg_LH_lg_Ls_limit > 0) then 
            call compare_to_target('abs(lg_LH - lg_Ls) <= limit', &
               abs(safe_log10(L_H_burn_total/Lsun) - s% log_surface_luminosity), &
                  s% abs_diff_lg_LH_lg_Ls_limit)

         else if (s% Teff <= s% Teff_lower_limit) then 
            call compare_to_target('Teff <= Teff_lower_limit', &
               s% Teff, s% Teff_lower_limit)
         else if (s% Teff >= s% Teff_upper_limit) then 
            call compare_to_target('Teff >= Teff_upper_limit', &
               s% Teff, s% Teff_upper_limit)

         else if (delta_nu <= s% delta_nu_lower_limit .and. s% delta_nu_lower_limit > 0) then 
            call compare_to_target('delta_nu <= delta_nu_lower_limit', &
               delta_nu, s% delta_nu_lower_limit)
         else if (delta_nu >= s% delta_nu_upper_limit .and. s% delta_nu_upper_limit > 0) then 
            call compare_to_target('delta_nu >= delta_nu_upper_limit', &
               delta_nu, s% delta_nu_upper_limit)

         else if (s% delta_Pg <= s% delta_Pg_lower_limit .and. s% delta_Pg_lower_limit > 0) then 
            call compare_to_target('delta_Pg <= delta_Pg_lower_limit', &
               s% delta_Pg, s% delta_Pg_lower_limit)
         else if (s% delta_Pg >= s% delta_Pg_upper_limit .and. s% delta_Pg_upper_limit > 0) then 
            call compare_to_target('delta_Pg >= delta_Pg_upper_limit', &
               s% delta_Pg, s% delta_Pg_upper_limit)

         else if (s% photosphere_r <= s% photosphere_r_lower_limit) then 
            call compare_to_target('photosphere_r <= photosphere_r_lower_limit', &
               s% photosphere_r, s% photosphere_r_lower_limit)
         else if (s% photosphere_r >= s% photosphere_r_upper_limit) then 
            call compare_to_target('photosphere_r >= photosphere_r_upper_limit', &
               s% photosphere_r, s% photosphere_r_upper_limit)

         else if (log_Teff <= s% log_Teff_lower_limit) then 
            call compare_to_target('log_Teff <= log_Teff_lower_limit', &
               log_Teff, s% log_Teff_lower_limit)
         else if (log_Teff >= s% log_Teff_upper_limit) then 
            call compare_to_target('log_Teff >= log_Teff_upper_limit', &
               log_Teff, s% log_Teff_upper_limit)

         else if (s% log_surface_temperature <= s% log_Tsurf_lower_limit) then 
            call compare_to_target('log_surface_temperature <= log_Tsurf_lower_limit', &
               s% log_surface_temperature, s% log_Tsurf_lower_limit)
         else if (s% log_surface_temperature >= s% log_Tsurf_upper_limit) then 
            call compare_to_target('log_surface_temperature >= log_Tsurf_upper_limit', &
               s% log_surface_temperature, s% log_Tsurf_upper_limit)

         else if (s% log_surface_pressure <= s% log_Psurf_lower_limit) then 
            call compare_to_target('log_surface_pressure <= log_Psurf_lower_limit', &
               s% log_surface_pressure, s% log_Psurf_lower_limit)
         else if (s% log_surface_pressure >= s% log_Psurf_upper_limit) then 
            call compare_to_target('log_surface_pressure >= log_Psurf_upper_limit', &
               s% log_surface_pressure, s% log_Psurf_upper_limit)

         else if (s% log_surface_luminosity <= s% log_L_lower_limit) then 
            call compare_to_target('log_surface_luminosity <= log_L_lower_limit', &
               s% log_surface_luminosity, s% log_L_lower_limit)
         else if (s% log_surface_luminosity >= s% log_L_upper_limit) then 
            call compare_to_target('log_surface_luminosity >= log_L_upper_limit', &
               s% log_surface_luminosity, s% log_L_upper_limit)

         else if (log_surface_gravity <= s% log_g_lower_limit) then 
            call compare_to_target('log_surface_gravity <= log_g_lower_limit', &
               log_surface_gravity, s% log_g_lower_limit)
            
         else if (log_surface_gravity >= s% log_g_upper_limit) then 
            call compare_to_target('log_surface_gravity >= log_g_upper_limit', &
               log_surface_gravity, s% log_g_upper_limit)

         else if (power_nuc_burn >= s% power_nuc_burn_upper_limit) then 
            call compare_to_target('power_nuc_burn >= power_nuc_burn_upper_limit', &
               power_nuc_burn, s% power_nuc_burn_upper_limit)

         else if (power_h_burn >= s% power_h_burn_upper_limit) then 
            call compare_to_target('power_h_burn >= power_h_burn_upper_limit', &
               power_h_burn, s% power_h_burn_upper_limit)

         else if (power_he_burn >= s% power_he_burn_upper_limit) then 
            call compare_to_target('power_he_burn >= power_he_burn_upper_limit', &
               power_he_burn, s% power_he_burn_upper_limit)

         else if (power_c_burn >= s% power_c_burn_upper_limit) then 
            call compare_to_target('power_c_burn >= power_c_burn_upper_limit', &
               power_c_burn, s% power_c_burn_upper_limit)

         else if (power_nuc_burn < s% power_nuc_burn_lower_limit) then 
            call compare_to_target('power_nuc_burn < power_nuc_burn_lower_limit', &
               power_nuc_burn, s% power_nuc_burn_lower_limit)

         else if (power_h_burn < s% power_h_burn_lower_limit) then 
            call compare_to_target('power_h_burn < power_h_burn_lower_limit', &
               power_h_burn, s% power_h_burn_lower_limit)

         else if (power_he_burn < s% power_he_burn_lower_limit) then 
            call compare_to_target('power_he_burn < power_he_burn_lower_limit', &
               power_he_burn, s% power_he_burn_lower_limit)

         else if (power_c_burn < s% power_c_burn_lower_limit) then 
            call compare_to_target('power_c_burn < power_c_burn_lower_limit', &
               power_c_burn, s% power_c_burn_lower_limit)

         else if (s% phase_of_evolution == s% phase_of_evolution_stop) then 
            write(*, '(/,a,/, 2i10)') 'stop because phase_of_evolution == phase_of_evolution_stop', &
               s% phase_of_evolution, s% phase_of_evolution_stop
            do_check_limits = terminate
            s% result_reason = result_reason_normal

         else if (s% iron_core_infall > s% iron_core_infall_limit) then 
            write(*, '(/,a,/, 2e20.10)') 'stop because iron_core_infall > iron_core_infall_limit', &
               s% iron_core_infall, s% iron_core_infall_limit
            do_check_limits = terminate
            s% result_reason = result_reason_normal

         else if (abs(v_surf/csound_surf) > s% v_div_csound_surf_limit) then 
            call compare_to_target('v_surf/csound_surf > v_div_csound_surf_limit', &
               abs(v_surf/csound_surf), s% v_div_csound_surf_limit)

         else if (min_pgas_div_p < s% Pgas_div_P_limit) then 
            call compare_to_target('min_pgas_div_p < Pgas_div_P_limit', &
               min_pgas_div_p, s% Pgas_div_P_limit)            

         else if (max_logQ > s% logQ_limit) then 
            call compare_to_target('max_logQ > logQ_limit', max_logQ, s% logQ_limit)

         end if
         
         if (do_check_limits /= keep_going) return
         
         do j=1,num_xa_central_limits
            if (s% xa_central_lower_limit(j) <= 0) cycle
            if (len_trim(s% xa_central_lower_limit_species(j)) == 0) cycle
            cid = chem_get_iso_id(s% xa_central_lower_limit_species(j))
            if (cid == 0) cycle
            i = s% net_iso(cid)
            if (i == 0) cycle
            avg_x = center_avg_x(s,i)
            if (avg_x < s% xa_central_lower_limit(j)) then
               call compare_to_target('have dropped below central lower limit for ' // &
                     trim(s% xa_central_lower_limit_species(j)), &
                     avg_x, s% xa_central_lower_limit(j))
               exit
            end if
         end do
         
         if (do_check_limits /= keep_going) return
         
         do j=1,num_xa_central_limits
            if (s% xa_central_upper_limit(j) <= 0) cycle
            if (s% xa_central_upper_limit(j) >= 1) cycle
            if (len_trim(s% xa_central_upper_limit_species(j)) == 0) cycle
            cid = chem_get_iso_id(s% xa_central_upper_limit_species(j))
            if (cid == 0) cycle
            i = s% net_iso(cid)
            if (i == 0) cycle
            avg_x = center_avg_x(s,i)
            if (avg_x > s% xa_central_upper_limit(j)) then
               call compare_to_target('have risen above central upper limit for ' // &
                     trim(s% xa_central_upper_limit_species(j)), &
                     avg_x, s% xa_central_upper_limit(j))
               exit
            end if
         end do
         
         if (do_check_limits /= keep_going) return
         
         do j=1,num_xa_surface_limits
            if (s% xa_surface_lower_limit(j) <= 0) cycle
            if (len_trim(s% xa_surface_lower_limit_species(j)) == 0) cycle
            cid = chem_get_iso_id(s% xa_surface_lower_limit_species(j))
            if (cid == 0) cycle
            i = s% net_iso(cid)
            if (i == 0) cycle
            avg_x = surface_avg_x(s,i)
            if (avg_x < s% xa_surface_lower_limit(j)) then
               call compare_to_target('have dropped below surface lower limit for ' // &
                     trim(s% xa_surface_lower_limit_species(j)), &
                     avg_x, s% xa_surface_lower_limit(j))
               exit
            end if
         end do
         
         if (do_check_limits /= keep_going) return
         
         do j=1,num_xa_surface_limits
            if (s% xa_surface_upper_limit(j) <= 0) cycle
            if (s% xa_surface_upper_limit(j) >= 1) cycle
            if (len_trim(s% xa_surface_upper_limit_species(j)) == 0) cycle
            cid = chem_get_iso_id(s% xa_surface_upper_limit_species(j))
            if (cid == 0) cycle
            i = s% net_iso(cid)
            if (i == 0) cycle
            avg_x = surface_avg_x(s,i)
            if (avg_x > s% xa_surface_upper_limit(j)) then
               call compare_to_target('have risen above surface upper limit for ' // &
                     trim(s% xa_surface_upper_limit_species(j)), &
                     avg_x, s% xa_surface_upper_limit(j))
               exit
            end if
         end do
         
         if (do_check_limits /= keep_going) return
         
         do j=1,num_xa_average_limits
            if (s% xa_average_lower_limit(j) <= 0) cycle
            if (len_trim(s% xa_average_lower_limit_species(j)) == 0) cycle
            cid = chem_get_iso_id(s% xa_average_lower_limit_species(j))
            if (cid == 0) cycle
            i = s% net_iso(cid)
            if (i == 0) cycle
            avg_x = dot_product(s% dq(1:nz), s% xa(i,1:nz))
            if (avg_x < s% xa_average_lower_limit(j)) then
               call compare_to_target('have dropped below average lower limit for ' // &
                     trim(s% xa_average_lower_limit_species(j)), &
                     avg_x, s% xa_average_lower_limit(j))
               exit
            end if
         end do
         
         if (do_check_limits /= keep_going) return
         
         do j=1,num_xa_average_limits
            if (s% xa_average_upper_limit(j) <= 0) cycle
            if (s% xa_average_upper_limit(j) >= 1) cycle
            if (len_trim(s% xa_average_upper_limit_species(j)) == 0) cycle
            cid = chem_get_iso_id(s% xa_average_upper_limit_species(j))
            if (cid == 0) cycle
            i = s% net_iso(cid)
            if (i == 0) cycle
            avg_x = dot_product(s% dq(1:nz), s% xa(i,1:nz))
            if (avg_x > s% xa_average_upper_limit(j)) then
               call compare_to_target('have risen above average upper limit for ' // &
                     trim(s% xa_average_upper_limit_species(j)), &
                     avg_x, s% xa_average_upper_limit(j))
               exit
            end if
         end do
         
         contains
         
         
         subroutine compare_to_target(str, value, target_value)
            character (len=*), intent(in) :: str
            real(dp), intent(in) :: value, target_value
            real(dp) :: err
            include 'formats.dek'
            err = abs(value - target_value)/ &
               (s% when_to_stop_atol + s% when_to_stop_rtol*max(abs(value),abs(target_value)))
            if (err > 1) then
               do_check_limits = retry
               write(*,'(/,a,5e20.10)') &
                  'retry with smaller timestep to get closer to stopping target', &
                  value, target_value
               if (s% D_norm_kappa /= 0) then
                  write(*,'(a,/,/)') 'have automatically set D_norm_kappa = 0 for this'
                  s% D_norm_kappa = 0 ! turn off error control of timesteps for this
               end if
            else
               do_check_limits = terminate
               s% result_reason = result_reason_normal
               write(*, '(/,a,/, 99e20.10)') 'stop because ' // trim(str), value, target_value, &
                  s% when_to_stop_atol, s% when_to_stop_rtol
            end if
         end subroutine compare_to_target
         
         
      end function do_check_limits

        
      integer function do_one_check_model(id)
         use report, only:std_write_internals_to_file  
         use rates_def, only: i_rate
         use chem_def, only: i_burn_c
         use star_utils, only: update_time, total_times
         integer, intent(in) :: id
         
         logical :: must_do_log, stop_because_he_ignited
         real(dp), parameter :: log_he_temp = 7.8d0
         real(dp), parameter :: d_tau_min = 1d-2, d_tau_max = 1d0
         real(dp), parameter :: little_step_factor = 10d0, little_step_size = 10d0
         real(dp) :: v, surf_dv_dt, surf_grav, power_he_burn, power_c_burn, &
            power_neutrinos, total_all_before
         integer :: model, log_priority, ierr
         integer, parameter :: tau_ramp = 50
         type (star_info), pointer :: s
         logical :: logged
         integer :: time0, clock_rate, nz
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            do_one_check_model = terminate
            return
         end if

         if (s% doing_timing) then
            total_all_before = total_times(s)
            call system_clock(time0,clock_rate)
         end if

         nz = s% nz
         must_do_log = .false.
         log_priority = delta_priority
         stop_because_he_ignited = .false.
         model = s% model_number
         do_one_check_model = keep_going
         
         if ( model <= 1 ) s% next_cntr_rho = max(min_cntr_rho, s% log_center_density + del_cntr_rho)
         
         do_one_check_model = do_check_limits(id)
         if (do_one_check_model /= keep_going) then
            if (dbg) write(*,*) 'do_check_limits /= keep_going'
            must_do_log = .true.
         end if

         if (.not. s% v_flag) then
            v = s% r(1) * s% dlnR_dt(1)
            if (s% dt > 0) then
               surf_dv_dt = (v - s% v_surf_old)/s% dt
               surf_grav = s% grav(1)
               if (abs(surf_dv_dt)/surf_grav > s% surface_accel_div_grav_limit &
                     .and. s% surface_accel_div_grav_limit > 0) then
                  write(*, '(a, e25.15)') 'exceeded surface_accel_div_grav_limit ', &
                     s% surface_accel_div_grav_limit
                  do_one_check_model = terminate
               else if (.false. .and. abs(surf_dv_dt)/surf_grav > 1) then
                  write(*,1) 'abs(surf_dv_dt)/surf_grav', abs(surf_dv_dt)/surf_grav, surf_dv_dt, surf_grav
               end if
            end if
         else
            v = s% v(1)
         end if
         
         power_he_burn = s% power_he_burn
         power_c_burn = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,i_burn_c,1:nz))/Lsun
         power_neutrinos = s% power_neutrinos
         
         if ( s% star_age < s% profile_age ) then
            if (dbg) write(*,*) 'must_do_log for age < profile_age'
            must_do_log = .true. ! in case of backup, do it over
         end if
         
         if ((.not. s% helium_ignition) .and. (s% log_center_temperature > log_he_temp) &
                  .and. (s% phase_of_evolution /= phase_he_igniting)) then
            if ( power_c_burn + power_he_burn > power_neutrinos  .and. (power_neutrinos > 1d0)) then
               must_do_log = .true.
               if (dbg) write(*,*) 'must_do_log for helium_ignition'
               s% helium_ignition = .true.
               s% phase_of_evolution = phase_he_igniting
               s% ignition_center_xhe = s% center_he4
               s% he_luminosity_limit = s% log_surface_luminosity
               s% prev_luminosity = s% log_surface_luminosity
            end if
         end if
         
         if ( (.not. s% carbon_ignition) .and. ( power_c_burn > power_neutrinos ) &
               .and. (power_neutrinos > 1d0)) then
            must_do_log = .true.
            s% carbon_ignition = .true.
            s% phase_of_evolution = phase_carbon_burning
         else if ( (s% phase_of_evolution .eq. phase_he_ignition_over .and. s% prev_age1 .eq. -1d0) &
                  .or. s% star_age <= s% post_he_age ) then
            ! need to check the age since may backup and over the previous saved info
            s% prev_tcntr1 = s% log_center_temperature; s% prev_tcntr2 = s% prev_tcntr1
            s% prev_age1 = s% star_age; s% prev_age2 = s% prev_age1
            must_do_log = .true.
            if (dbg) write(*,*) 'must_do_log for starting phase of steady helium burning'
            s% post_he_age = s% star_age
            if (.not. s% doing_first_model_of_run) &
               write(*, '(/,a, i7,/)') 'starting phase of steady helium burning', &
                  s% model_number
         else if ( time_to_profile(s) ) then
            must_do_log = .true.
            if (dbg) write(*,*) 'must_do_log for time_to_profile'
         end if
         
         if (must_do_log) log_priority = phase_priority
         
         logged = log_state(s, must_do_log)

         if (logged .and. s% do_log_files) then
            if (s% model_number .eq. s% profile_model &
               .or. (s% profile_interval > 0 .and. &
                     (s% doing_first_model_of_run .or. &
                     mod(s% model_number,s% profile_interval) == 0))) then
               if (s% do_log_files) must_do_log = .true.
               if (s% model_number == s% profile_model .or.&
                   s% doing_first_model_of_run .or. &
                   (mod(s% model_number, s% priority_profile_interval) == 0)) then
                  log_priority = phase_priority
               end if
            end if
            if ( must_do_log ) then
               if (dbg) write(*,*) 'do_one_check_model: call set_save_profiles_info'
               call set_save_profiles_info(s, log_priority)
            end if
         end if
         
         if ( stop_because_he_ignited ) then
            write(*, '(a, i7)') 'stop_because_he_ignited at model number', s% model_number
            do_one_check_model = terminate
         end if

         if (s% doing_timing) call update_time(s, time0, total_all_before, s% time_check_model)
         
      end function do_one_check_model
      
      
      logical function time_to_profile(s)
         use chem_def, only: ih1
         type (star_info), pointer :: s
         ! end-of-run and helium break-even are always done. 
         ! this function decides on other models to be profiled.
         real(dp), parameter :: center_he_drop = 1d-2, surface_t_drop = 4d-2
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         time_to_profile = .false.
         
         select case ( s% phase_of_evolution )
         case ( phase_starting )
            if ( arrived_main_seq(s) ) then
               if (dbg) write(*,*) 'arrived_main_seq'
               time_to_profile = .true.
               s% prev_tsurf = s% log_surface_temperature
               if (s% center_h1 > center_h_going) then
                  s% phase_of_evolution = phase_early_main_seq
                  if (abs(s% star_mass - s% initial_mass) < 0.01d0*s% initial_mass &
                        .and. .not. s% doing_first_model_of_run) &
                     write(*, '(/,a, i7,/)') 'starting main sequence', s% model_number
               else if ( s% center_h1 > center_h_gone ) then
                  s% phase_of_evolution = phase_mid_main_seq
               else if ( s% center_he4 > center_he_going ) then
                  s% phase_of_evolution = phase_he_ignition_over
               else
                  s% phase_of_evolution = phase_helium_burning
               end if
            end if
         case ( phase_early_main_seq )
            if ( s% center_h1 < center_h_going ) then
               time_to_profile = .true.
               s% prev_tsurf = s% log_surface_temperature
               s% phase_of_evolution = phase_mid_main_seq
               if (.not. s% doing_first_model_of_run) &
                  write(*, '(/,a, i7,/)') 'center hydrogen more than half gone', &
                     s% model_number
            end if
         case ( phase_mid_main_seq )
            if ( s% center_h1 < center_h_gone &
                  .and. s% log_surface_temperature < s% prev_tsurf-surface_t_drop ) then
               time_to_profile = .true.
               s% phase_of_evolution = phase_wait_for_he
               if (.not. s% doing_first_model_of_run) &
                  write(*, '(/,a, i7,/)') &
                     'center hydrogen less than 0.1% and surface temperature dropping', &
                     s% model_number
            end if
         case ( phase_wait_for_he )
         case ( phase_he_igniting ) ! for non-flash ignition of helium core
            if ( s% center_he4 <= s% ignition_center_xhe-center_he_drop &
                  .and. s% log_surface_luminosity > s% prev_luminosity ) then
               time_to_profile = .true.
               if (.not. s% doing_first_model_of_run) &
                  write(*, '(/,a, i7,/)') 'center helium decreasing and luminosity rising', &
                     s% model_number
               s% phase_of_evolution = phase_he_ignition_over
               s% prev_tcntr2 = s% prev_tcntr1; s% prev_age2 = s% prev_age1
               s% prev_tcntr1 = s% log_center_temperature; s% prev_age1 = s% star_age
               if ( s% log_surface_luminosity > s% he_luminosity_limit ) &
                  s% he_luminosity_limit = s% log_surface_luminosity
            end if
            s% prev_luminosity = s% log_surface_luminosity
         case ( phase_he_ignition_over )
            if ( s% center_he4 < center_he_going ) then
               time_to_profile = .true.
               if (.not. s% doing_first_model_of_run) &
                  write(*, '(/,a, i7,/)') 'center helium low', s% model_number
               s% phase_of_evolution = phase_helium_burning
            end if      
            s% prev_tcntr2 = s% prev_tcntr1; s% prev_age2 = s% prev_age1
            s% prev_tcntr1 = s% log_center_temperature; s% prev_age1 = s% star_age
         case ( phase_carbon_burning )
         case ( phase_helium_burning )
         end select
         
      end function time_to_profile

      
      subroutine pick_model_log_number( &
            max_num_mods, num_models, model_logs, model_log_number, ierr)
         integer, intent(in) :: max_num_mods
         integer, intent(inout) :: num_models
         integer, pointer, dimension(:) :: model_logs
         integer, intent(out) :: model_log_number, ierr
         logical :: in_use(max_num_mods)
         integer :: i
         ! pick log number for the new model
         ierr = 0
         in_use = .false.
         do i=1, num_models
            in_use(model_logs(i)) = .true.
         end do
         model_log_number = 0
         do i=1, max_num_mods
            if (.not. in_use(i)) then
               model_log_number = i; exit
            end if
         end do
         if (model_log_number == 0) then
            write(*, *) 'model_log_number == 0, cannot happen?'
            ierr = -1
            return
         end if
      end subroutine pick_model_log_number
      
      
      subroutine make_room_for_profile_info( &
            model_number, max_num_mods, num_models, model_numbers, model_priorities, model_logs, ierr)
         integer, intent(in) :: model_number, max_num_mods
         integer, intent(inout) :: num_models
         integer, pointer, dimension(:) :: model_numbers, model_priorities, model_logs
         integer, intent(out) :: ierr
         integer :: i, j, nm
         logical, parameter :: dbg = .false.
         include 'formats.dek'
         ierr = 0
         ! delete models with model number greater or equal to current model number
         nm = num_models; j = 0
         do i=1, nm
            if (model_numbers(i) < model_number .and. model_logs(i) <= max_num_mods) then 
               ! keep this one
               j = j+1
               if (j < i) then
                  model_numbers(j) = model_numbers(i)
                  model_priorities(j) = model_priorities(i)
                  model_logs(j) = model_logs(i)
               end if
            end if
         end do
         num_models = j
         if (num_models == max_num_mods) then ! pick one to delete
            j = 1
            do i=2, num_models
               if (dbg) then
                  write(*,3) 'model_priorities(i)', i, model_priorities(i)
                  write(*,3) 'model_priorities(j)', j, model_priorities(j)
                  write(*,3) 'model_numbers(i)', i, model_numbers(i)
                  write(*,3) 'model_numbers(j)', j, model_numbers(j)
                  write(*,*) 'model_priorities(i) < model_priorities(j)', model_priorities(i) < model_priorities(j)
                  write(*,*) 'model_numbers(i) < model_numbers(j)', model_numbers(i) < model_numbers(j)
               end if
               if (model_priorities(i) < model_priorities(j)) then
                  if (dbg) write(*,3) '1 change j'
                  j = i
               else if (model_priorities(i) == model_priorities(j) .and. &
                        model_numbers(i) < model_numbers(j)) then
                  if (dbg) write(*,3) '2 change j'
                  j = i
               end if
               if (dbg) write(*,3) 'new j', j
               if (dbg) write(*,*)
            end do
            ! delete j
            if (dbg) write(*,*) 'delete j', j
            do i=j+1, num_models
               model_numbers(i-1) = model_numbers(i)
               model_priorities(i-1) = model_priorities(i)
               model_logs(i-1) = model_logs(i)
            end do
            num_models = num_models-1
         end if
      end subroutine make_room_for_profile_info
      
      
      subroutine read_profiles_info( &
            fname, max_num_mods, num_models, model_numbers, model_priorities, model_logs)
         character (len=*), intent(in) :: fname
         integer, intent(in) :: max_num_mods
         integer, intent(out) :: num_models
         integer, pointer, dimension(:) :: model_numbers, model_priorities, model_logs
         integer :: iounit, i, ierr
         num_models = 0
         ierr = 0
         iounit = alloc_iounit(ierr); if (ierr /= 0) return         
         open(unit=iounit, file=trim(fname), action='read', status='old', iostat=ierr)
         if (ierr == 0) then ! file exists
            read(iounit, *, iostat=ierr) num_models
            if (ierr == 0) then
               if (num_models > max_num_mods) num_models = max_num_mods
               do i=1, num_models
                  read(iounit, *, iostat=ierr) model_numbers(i), model_priorities(i), model_logs(i)
                  if (ierr /= 0) exit
               end do
            end if
            close(iounit)
            if (ierr /= 0) num_models = 0
         end if
         call free_iounit(iounit)
      end subroutine read_profiles_info         
      
      
      subroutine write_profiles_list( &
            fname, num_models, model_numbers, model_priorities, model_logs, ierr)
         character (len=*), intent(in) :: fname
         integer, intent(in) :: num_models
         integer, pointer, dimension(:) :: model_numbers, model_priorities, model_logs
         integer, intent(out) :: ierr
         integer :: iounit, i
         ierr = 0
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         ! write the new list
         open(unit=iounit, file=trim(fname), action='write', iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'failed to open ' // trim(fname)
         else
            if (num_models == 1) then
               write(iounit, *) num_models, &
                  'model. lines hold model number, priority, and log file number.'
            else
               write(iounit, *) num_models, &
                  'models. lines hold model number, priority, and log file number.'
            end if
            do i=1, num_models
               write(iounit, *) model_numbers(i), model_priorities(i), model_logs(i)
            end do
            close(iounit)
         end if
         call free_iounit(iounit)
      end subroutine write_profiles_list
      
            
      subroutine get_model_profilename(s, model_log_number) ! sets s% model_profilename
         type (star_info), pointer :: s 
         integer, intent(in) :: model_log_number
         if (model_log_number < 10) then
            write(s% model_profilename, '(a, i1, a)') &
               trim(s% log_directory) // '/' // trim(s% log_data_prefix), &
               model_log_number, trim(s% log_data_suffix)
         else if (model_log_number < 100) then
            write(s% model_profilename, '(a, i2, a)') &
               trim(s% log_directory) // '/' // trim(s% log_data_prefix), &
               model_log_number, trim(s% log_data_suffix)
         else if (model_log_number < 1000) then
            write(s% model_profilename, '(a, i3, a)') &
               trim(s% log_directory) // '/' // trim(s% log_data_prefix), &
               model_log_number, trim(s% log_data_suffix)
         else if (model_log_number < 10000) then
            write(s% model_profilename, '(a, i4, a)') &
               trim(s% log_directory) // '/' // trim(s% log_data_prefix), &
               model_log_number, trim(s% log_data_suffix)
         else if (model_log_number < 100000) then
            write(s% model_profilename, '(a, i5, a)') &
               trim(s% log_directory) // '/' // trim(s% log_data_prefix), &
               model_log_number, trim(s% log_data_suffix)
         else
            write(s% model_profilename, '(a, i6, a)') &
               trim(s% log_directory) // '/' // trim(s% log_data_prefix), &
               model_log_number, trim(s% log_data_suffix)
         end if
      end subroutine get_model_profilename
      
      
      subroutine dummy_before_evolve(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         ierr = id ! so that we use that arg
         ierr = 0
      end subroutine dummy_before_evolve
      
      
      subroutine set_phase_of_evolution(s) ! at start of run
         use rates_def, only: i_rate
         use chem_def, only: i_burn_c
         type (star_info), pointer :: s
         real(dp) :: power_he_burn, power_c_burn, power_neutrinos
         integer :: nz
         include 'formats.dek'
         nz = s% nz
         if (.not. arrived_main_seq(s) .or. s% phase_of_evolution == phase_carbon_burning) return
         power_he_burn = s% power_he_burn
         power_c_burn = dot_product(s% dm(1:nz), s% eps_nuc_categories(i_rate,i_burn_c,1:nz))/Lsun
         power_neutrinos = s% power_neutrinos  
         if (s% phase_of_evolution == phase_helium_burning .and. power_c_burn > power_neutrinos) then
            !write(*, *) 'set_phase_of_evolution: phase_carbon_burning'
            s% phase_of_evolution = phase_carbon_burning
         else if (power_c_burn + power_he_burn > power_neutrinos) then
            !write(*, *) 'set_phase_of_evolution: phase_helium_burning'
            s% phase_of_evolution = phase_helium_burning
         else if (s% center_he4 < center_he_going) then
            !write(*, *) 'set_phase_of_evolution: phase_helium_burning'
            s% phase_of_evolution = phase_helium_burning
         else if (s% center_h1 < center_h_gone) then
            !write(*, *) 'set_phase_of_evolution: phase_wait_for_he'
            s% phase_of_evolution = phase_wait_for_he
         else if (s% center_h1 < center_h_going) then
            !write(*, *) 'set_phase_of_evolution: phase_mid_main_seq'
            s% phase_of_evolution = phase_mid_main_seq
         else
            !write(*, *) 'set_phase_of_evolution: phase_early_main_seq'
            s% phase_of_evolution = phase_early_main_seq
         end if
      end subroutine set_phase_of_evolution
      
      
      subroutine show_phase_of_evolution(s)
         type (star_info), pointer :: s
         include 'formats.dek'         
         select case ( s% phase_of_evolution )
         case ( phase_starting )
            write(*, *) 'phase_starting'
         case ( phase_early_main_seq )
            write(*, *) 'phase_early_main_seq'
         case ( phase_mid_main_seq )
            write(*, *) 'phase_mid_main_seq'
         case ( phase_wait_for_he )
            write(*, *) 'phase_wait_for_he'
         case ( phase_he_igniting )
            write(*, *) 'phase_he_igniting'
         case ( phase_he_ignition_over )
            write(*, *) 'phase_he_ignition_over'
         case ( phase_carbon_burning )
            write(*, *) 'phase_carbon_burning'
         case ( phase_helium_burning )
            write(*, *) 'phase_helium_burning'
         end select
      end subroutine show_phase_of_evolution


      logical function arrived_main_seq(s)
         type (star_info), pointer :: s
         include 'formats.dek'   
         arrived_main_seq = &
            (s% L_nuc_burn_total >= s% L_phot) .and. &
            (s% power_h_burn >= s% L_nuc_burn_total/2)
         return
         write(*,1) 's% L_nuc_burn_total', s% L_nuc_burn_total
         write(*,1) 's% L_phot', s% L_phot 
         write(*,1) 's% power_h_burn', s% L_phot 
         write(*,*) 'arrived_main_seq',  arrived_main_seq
         write(*,*)
      end function arrived_main_seq


      
      end module do_one_utils
      
