! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module brunt
      
      use star_private_def
      use alert_lib,only:alert
      use utils_lib,only:is_bad_num
      use const_def
      
      implicit none

      logical, parameter :: dbg = .false.
      


      contains
      
      
      subroutine do_brunt_N2(s,nzlo,nzhi,ierr)
         use interp_1d_def
         use interp_1d_lib
         use star_utils, only: dr_div_R_smooth_nonconv
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         
         integer :: nz, i, k, khi_Brassard, klo_basic
         real(dp) :: Zmin, Zmax, Z
         real(dp), dimension(:), pointer :: brunt_N2_Brassard, cell_dr, work
         
         include 'formats.dek'
         
         ierr = 0
         
         nz = s% nz

         if (.not. s% calculate_Brunt_N2 .or. s% brunt_N2_coefficient == 0) then
            s% brunt_N2(1:nz) = 0
            return
         end if
         
         Zmax = s% brunt_Brassard_full_on_Zmax ! Z <= this means use 100% Brassard option
         Zmin = s% brunt_Brassard_full_off_Zmin ! Z >= this means use 100% basic option
         
         ! if k > khi_Brassard, then Z >= Zmin, so use 100% basic
         ! if k < klo_basic, then Z <= Zmax, so use 100% Brassard
         ! if klo_basic <= k <= khi_Brassard, use blend of both.
         klo_basic = nz + 1
         do k = 1, nz
            Z = 1 - (s% X(k) + s% Y(k))
            if (Z <= Zmax) cycle ! use 100% Brassard
            klo_basic = k ! smallest k with Z > Zmax
            exit
         end do
         khi_Brassard = 0
         do k = nz, 1, -1
            Z = 1 - (s% X(k) + s% Y(k))
            if (Z >= Zmin) cycle ! use 100% basic
            khi_Brassard = k ! largest k with Z < Zmin
            exit
         end do
         
         if (klo_basic > nz) then
            call do_brunt_a_la_Brassard(s,1,nz,ierr)
            if (s% brunt_N2_coefficient /= 1d0) &
               s% brunt_N2(1:nz) = s% brunt_N2_coefficient*s% brunt_N2(1:nz)
            return
         end if
         
         if (khi_Brassard == 0) then
            call do_brunt_basic(s,1,nz,ierr)
            if (s% brunt_N2_coefficient /= 1d0) &
               s% brunt_N2(1:nz) = s% brunt_N2_coefficient*s% brunt_N2(1:nz)
            return
         end if
         
         call do_brunt_a_la_Brassard(s,1,khi_Brassard,ierr)
         if (ierr /= 0) return
         
         call do_alloc(ierr)
         if (ierr /= 0) return

         brunt_N2_Brassard(klo_basic:khi_Brassard) = s% brunt_N2(klo_basic:khi_Brassard)
         
         call do_brunt_basic(s,klo_basic,nz,ierr)         
         if (ierr /= 0) then
            call dealloc
            return
         end if
         
!$OMP PARALLEL DO PRIVATE(k)
         do k = klo_basic, khi_Brassard
            call blend1(k)
         end do
!$OMP END PARALLEL DO         
         
         cell_dr => brunt_N2_Brassard ! reuse it with different name
         do k=2,nz
            cell_dr(k-1) = s% rmid(k-1) - s% rmid(k)
         end do
         cell_dr(nz) = s% rmid(nz) - s% R_center
         do i=1,s% nsmooth_brunt_dlnY_dlnP
            call dr_div_R_smooth_nonconv( &
               s, s% r_div_R_smooth_brunt_dlnY_dlnP, cell_dr, s% brunt_N2, work)
         end do
         
         if (s% brunt_N2_coefficient /= 1d0) &
            s% brunt_N2(1:nz) = s% brunt_N2_coefficient*s% brunt_N2(1:nz)
         
         call dealloc
         

         contains
         

         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            call get_work_array(s, brunt_N2_Brassard, nz, nz_alloc_extra, 'brunt_N2_Brassard', ierr)
            if (ierr /= 0) return          
            call get_work_array(s, work, nz, nz_alloc_extra, 'brunt work', ierr)
            if (ierr /= 0) return          
         end subroutine do_alloc
         
         
         subroutine dealloc
            use alloc
            use utils_lib
            call return_work_array(s, brunt_N2_Brassard, 'brunt_N2_Brassard')            
            call return_work_array(s, work, 'brunt work')            
         end subroutine dealloc
         

         subroutine blend1(k)
            integer, intent(in) :: k
            real(dp) :: Z, alfa, beta, brunt_basic, brunt_Brassard
            brunt_basic = s% brunt_N2(k)
            brunt_Brassard = brunt_N2_Brassard(k)
            Z = 1 - (s% X(k) + s% Y(k))
            if (Z <= Zmax) then
               s% brunt_N2(k) = brunt_Brassard
            else if (Z >= Zmin) then
               s% brunt_N2(k) = brunt_basic
            else
               alfa = (Z - Zmin)/(Zmax - Zmin)
               beta = 1 - alfa
               s% brunt_N2(k) = alfa*brunt_basic + beta*brunt_Brassard
            end if
         end subroutine blend1
         

      end subroutine do_brunt_N2
      
      
      subroutine do_brunt_a_la_Brassard(s,klo_in,khi_in,ierr)
         use star_utils, only: interp_val_to_pt, smooth, dr_div_R_smooth_nonconv
         use interp_1d_def
         use interp_1d_lib
         use chem_def, only: ih1, ih2, ihe3, ihe4
         
         type (star_info), pointer :: s         
         integer, intent(in) :: klo_in, khi_in
         integer, intent(out) :: ierr
         
         real(dp), pointer, dimension(:) :: &
            lnY, dlnY, dlnP, chiY_chiT, rho_P_chiT_chiRho, gradT_sub_grada, work, cell_dr
         real(dp) :: Y, rho_P_chiT_chiRho_at_face, chiY_chiT_at_face, dr_div_R_width
         integer :: h1, h2, he3, he4, nz, nz_x, k, i, op_err, species, klo, khi
         logical, parameter :: dbg = .true.
         
         include 'formats.dek'
         
         ierr = 0

         nz = s% nz
         species = s% species
         klo = max(1, klo_in - 5)
         khi = min(nz, khi_in + 5)

         h1 = s% net_iso(ih1)
         h2 = s% net_iso(ih2)
         he3 = s% net_iso(ihe3)
         he4 = s% net_iso(ihe4)
         if (he4 == 0) then
            write(*,*) 'net must include he3 and he4 for brunt_a_la_Brassard'
            ierr = -1
            return
         end if
         
         call do_alloc(ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed in do_brunt_a_la_Brassard'
            ierr = -1
            return
         end if
         
         do k=klo,khi
            Y = s% Y(k)
            lnY(k) = log(max(1d-99, Y))
            if (k == 1) then
               dlnY(k) = 0
               dlnP(k) = 0
               s% brunt_dlnY_dlnP(k) = 0
               cell_dr(k) = 0
               cycle
            end if
            dlnY(k) = lnY(k-1) - lnY(k)
            dlnP(k) = s% lnP(k-1) - s% lnP(k)
            if (dlnP(k) > -1d-20) then
               s% brunt_dlnY_dlnP(k) = 0
            else
               s% brunt_dlnY_dlnP(k) = dlnY(k)/dlnP(k)
            end if
            cell_dr(k-1) = s% r(k-1) - s% r(k)
            if (.false. .and. k >= 2212 .and. k <= 2218) then
               write(*,2) 'dlnY dlnP dlnY/dlnP', &
                  k, dlnY(k), dlnP(k), s% brunt_dlnY_dlnP(k), cell_dr(k-1)
            end if
         end do
         !write(*,*)
         !stop
         
         cell_dr(nz) = s% r(nz) - s% R_center
         if (klo > 1) then
            do k=1,klo-1
               s% brunt_dlnY_dlnP(k) = 0
               if (k > 1) cell_dr(k-1) = s% r(k-1) - s% r(k)
            end do
         end if
         if (khi < nz) then
            do k=khi+1,nz
               s% brunt_dlnY_dlnP(k) = 0
               if (k > 1) cell_dr(k-1) = s% r(k-1) - s% r(k)
            end do
            cell_dr(nz) = s% r(nz) - s% R_center
         end if
         
         do k=1,nz
            if (is_bad_num(s% brunt_dlnY_dlnP(k))) then
               write(*,2) 's% brunt_dlnY_dlnP(k)', k, s% brunt_dlnY_dlnP(k)
               stop 'debug brunt: before dr_div_R_smooth_nonconv'
            end if
         end do
         
         dr_div_R_width = s% r_div_R_smooth_brunt_dlnY_dlnP
         do i=1,s% nsmooth_brunt_dlnY_dlnP
            call dr_div_R_smooth_nonconv(s, dr_div_R_width, cell_dr, s% brunt_dlnY_dlnP, work)
         end do
         
         do k=1,nz
            if (is_bad_num(s% brunt_dlnY_dlnP(k))) then
               write(*,2) 's% brunt_dlnY_dlnP(k)', k, s% brunt_dlnY_dlnP(k)
               write(*,2) 's% nsmooth_brunt_dlnY_dlnP', s% nsmooth_brunt_dlnY_dlnP
               stop 'debug brunt: after dr_div_R_smooth_nonconv'
            end if
         end do
         
!$OMP PARALLEL DO PRIVATE(k,op_err) SCHEDULE(DYNAMIC,10)
         do k=klo,khi
            if (abs(s% brunt_dlnY_dlnP(k)) <= s% burnt_abs_dlnY_dlnP_cutoff) then
               s% brunt_chiY(k) = 0
            else ! call eos to estimate chiY for cell k
               op_err = 0
               call get_chiY(k, op_err)
               if (op_err /= 0) ierr = op_err
            end if
            chiY_chiT(k) = s% brunt_chiY(k)/s% chiT(k)
            rho_P_chiT_chiRho(k) = (s% rho(k)/s% P(k))*(s% chiT(k)/s% chiRho(k))
            gradT_sub_grada(k) = s% gradT_sub_grada(k)
         end do
!$OMP END PARALLEL DO         
         if (ierr /= 0) then
            call dealloc
            return
         end if
         
         if (klo > 1) s% brunt_chiY(1:klo-1) = 0
         if (khi < nz) s% brunt_chiY(khi+1:nz) = 0
         
!$OMP PARALLEL DO PRIVATE(k, op_err, rho_P_chiT_chiRho_at_face, chiY_chiT_at_face)
         do k=klo,khi
            chiY_chiT_at_face = interp_val_to_pt(chiY_chiT,k,nz,s% dq)
            s% brunt_B(k) = -chiY_chiT_at_face*s% brunt_dlnY_dlnP(k)
            if (s% brunt_B(k) < 0 .and. .not. s% allow_negative_brunt_B) &
               s% brunt_B(k) = 0
            rho_P_chiT_chiRho_at_face = interp_val_to_pt(rho_P_chiT_chiRho,k,nz,s% dq)
            s% brunt_N2(k) = &
               s% grav(k)**2*rho_P_chiT_chiRho_at_face*(s% brunt_B(k) - s% gradT_sub_grada(k))
         end do
!$OMP END PARALLEL DO         
         
         if (klo > 1) s% brunt_B(1:klo-1) = 0
         if (khi < nz) s% brunt_B(khi+1:nz) = 0

         call dealloc
         
         
         contains
         
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            call get_work_array(s, lnY, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnY, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnP, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, chiY_chiT, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, rho_P_chiT_chiRho, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, gradT_sub_grada, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, work, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, cell_dr, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return            
         end subroutine do_alloc

         subroutine dealloc
            use alloc
            use utils_lib
            call return_work_array(s, lnY, 'brunt')            
            call return_work_array(s, dlnY, 'brunt')            
            call return_work_array(s, dlnP, 'brunt')            
            call return_work_array(s, chiY_chiT, 'brunt')            
            call return_work_array(s, rho_P_chiT_chiRho, 'brunt')            
            call return_work_array(s, gradT_sub_grada, 'brunt')            
            call return_work_array(s, work, 'brunt')            
            call return_work_array(s, cell_dr, 'brunt')            
         end subroutine dealloc
         
         
         subroutine get_chiY(k, ierr)
            use eos_def, only: num_eos_basic_results, i_lnPgas
            use chem_lib, only: composition_info
            use micro, only: eos_get
            
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            
            real(dp) :: &
               xa(species), X, Y, Z, X0, Y0, Z0, lnY, delta_lnY, delta_Y, &
               abar, zbar, approx_abar, approx_zbar, xh, xhe, &
               z2bar, ye, xsum, dabar_dx(species), dzbar_dx(species), alt_lnP, Xfac, Zfac
            real(dp), dimension(num_eos_basic_results) :: &
               res, d_eos_dlnd, d_eos_dlnT
            integer :: i
            
            logical, parameter :: dbg = .false.
            
            include 'formats.dek'
            
            ierr = 0
            s% brunt_chiY(k) = 0
            xa(:) = s% xa(:,k)
            Y0 = s% Y(k)
            if (Y0 < 1d-6) then
               if (dbg) write(*,2) 'small Y0', k, Y0
               return
            end if
            X = s% X(k)
            Z = max(0d0,1d0-(X+Y0))
            lnY = log(Y0)
            delta_lnY = 0.0001d0
            do
               Y = exp(lnY + delta_lnY)
               delta_Y = Y-Y0
               if (delta_Y <= X+Z) exit
               if (delta_Y < 1d-6) then
                  return
               end if
               delta_lnY = delta_lnY/2
            end do
            if (he3 /= 0) xa(he3) = xa(he3)*Y/Y0
            xa(he4) = xa(he4)*Y/Y0
            X0 = X
            Z0 = Z
            
            if (.false.) then
               X = max(0d0, X0 - delta_Y)
               Z = max(0d0, min(1d0, 1d0 - (X+Y)))
               if (X0 > 0) then
                  Xfac = X/X0
               else
                  Xfac = 0
               end if
               if (Z0 > 0) then
                  Zfac = Z/Z0
               else
                  Zfac = 0
               end if
            else
               Xfac = (1 - Y)/(X0 + Z0)
               Zfac = Xfac
               X = Xfac*X0
               Z = Zfac*Z0
            end if
            
            do i=1,species
               if (i == h1 .or. i == h2) then
                  xa(i) = xa(i)*Xfac
                  cycle
               end if
               if (i == he3 .or. i == he4) cycle
               xa(i) = xa(i)*Zfac
            end do
         
            call composition_info(species, s% chem_id, xa, xh, xhe, &
               abar, zbar, z2bar, ye, approx_abar, approx_zbar, xsum, dabar_dx, dzbar_dx)  
            
            call eos_get( &
                  s, 0, Z, X, abar, zbar, approx_abar, approx_zbar, xa, &
                  s% rho(k), s% lnd(k)/ln10, s% T(k), s% lnT(k)/ln10, &
                  res, d_eos_dlnd, d_eos_dlnT, ierr)
            if (ierr /= 0) return

            alt_lnP = log(crad*s% T(k)**4/3 + exp(res(i_lnPgas)))
            s% brunt_chiY(k) = (alt_lnP - s% lnP(k))/delta_lnY
            
            if (dbg) then
               write(*,2) 's% brunt_chiY(k)', k, s% brunt_chiY(k)
               write(*,2) 'alt logP', k, alt_lnP/ln10
               write(*,2) 'logP', k, s% lnP(k)/ln10
               write(*,2) 'delta_lnY', k, delta_lnY
               write(*,2) 'delta_Y', k, delta_Y
               write(*,2) 'Z', k, Z
               write(*,2) 'X', k, X
               write(*,*) 'X < Z', k, X < Z
               write(*,2) 'abar', k, abar
               write(*,2) 'zbar', k, zbar
               write(*,2) 'rho', k, s% rho(k)
               write(*,2) 'log rho', k, s% lnd(k)/ln10
               write(*,2) 'T', k, s% T(k)
               write(*,2) 'log T', k, s% lnT(k)/ln10
               write(*,*)
            end if
            
            if (dbg .and. s% brunt_chiY(k) < -50) &
               write(*,2) 's% brunt_chiY(k)', k, s% brunt_chiY(k)

         end subroutine get_chiY


      end subroutine do_brunt_a_la_Brassard
      
      
      subroutine do_brunt_basic(s,klo_in,khi_in,ierr)
         use micro, only: eval_rho
         use chem_def
         use interp_1d_def
         use interp_1d_lib
         type (star_info), pointer :: s         
         integer, intent(in) :: klo_in, khi_in
         integer, intent(out) :: ierr
         
         integer, parameter :: nwork = pm_work_size
         real(dp), pointer, dimension(:,:) :: work, f_lnd, f_lnP
         integer :: nz, k, i, op_err, klo, khi

         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0
         nz = s% nz
         klo = max(1, klo_in - 5)
         khi = min(nz, khi_in + 5)
         
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         f_lnd(1,1:nz) = s% lnd(1:nz)
         call interp_pm(s% rmid, nz, f_lnd, nwork, work, ierr)
         if (ierr /= 0) then
            call dealloc
            write(*,*) 'error in do_brunt_basic interp_pm for lnd'
            return
         end if
         
         f_lnP(1,1:nz) = s% lnP(1:nz)
         call interp_pm(s% rmid, nz, f_lnP, nwork, work, ierr)
         if (ierr /= 0) then
            call dealloc
            write(*,*) 'error in do_brunt_basic interp_pm for lnT'
            return
         end if
         
!$OMP PARALLEL DO PRIVATE(k,op_err)
         do k = klo, khi
            op_err = 0
            call do1(s,k,op_err)
            if (op_err /= 0) ierr = op_err
         end do
!$OMP END PARALLEL DO

         call dealloc

         
         contains
         
         
         subroutine do_alloc(ierr)
            use alloc
            integer, intent(out) :: ierr
            ierr = 0
            call get_2d_work_array(s, f_lnd, 4, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, f_lnP, 4, nz, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return
            call get_2d_work_array(s, work, nz, nwork, nz_alloc_extra, 'brunt', ierr)
            if (ierr /= 0) return
         end subroutine do_alloc
         
         
         subroutine dealloc
            use alloc
            call return_2d_work_array(s, f_lnd, 'brunt')            
            call return_2d_work_array(s, f_lnP, 'brunt')            
            call return_2d_work_array(s, work, 'brunt')            
         end subroutine dealloc
      
      
         subroutine do1(s,k,ierr)
            use utils_lib, only: is_bad_num
            use eos_def
            use eos_lib, only: Radiation_Pressure
            use mlt_def, only: convective_mixing
            type (star_info), pointer :: s         
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            real(dp) :: &
               rmid, grav, scale_height, dr, rmid2, rho2, res(num_eos_basic_results), &
               d_eos_dlnd(num_eos_basic_results), d_eos_dlnT(num_eos_basic_results), &
               lnP_background, P_background, lnT2, T2, x, y, z, lnd2, lnd_background, &
               Pgas2

            include 'formats.dek'
            
            ierr = 0
            
            !if (s% mixing_type(k) == convective_mixing) then
            !   call zero_brunt(s,k)
            !   return
            !end if
            
            rmid = s% rmid(k)
            if (k < nz) then
               grav = 0.5d0*(s% grav(k) + s% grav(k+1))
            else
               grav = s% grav(k)
            end if
            
            scale_height = s% scale_height(k)
            dr = scale_height*s% brunt_H_frac
            rmid2 = rmid + dr
            call interp_value(s% rmid, nz, f_lnP, rmid2, lnP_background, ierr)
            if (ierr /= 0) then
               write(*,2) 'interp_value lnP failed', k, rmid2
               stop 'do_brunt_basic'
            end if
         
            rho2 = 1d0
            lnd_background = 0d0
            lnd2 = 0d0
            
            if (rmid2 < s% rmid(1)) then
            
               lnT2 = s% lnT(k) + s% grada(k)*(lnP_background - s% lnP(k))
               T2 = exp(lnT2)
            
               x = s% X(k)
               y = s% Y(k)
               z = max(0d0, min(1d0, 1d0 - (x+y)))
            
               P_background = exp(lnP_background)
               Pgas2 = P_background - Radiation_Pressure(T2)
            
               if (Pgas2 > 1d6) then
            
                  call eval_rho( &
                     s, z, x, s% xa(:,k), s% abar(k), s% zbar(k), &
                     s% approx_abar(k), s% approx_zbar(k), T2, lnT2, Pgas2, &
                     rho2, res, d_eos_dlnd, d_eos_dlnT, ierr)
                  if (ierr /= 0) then
                     write(*,2) 'eval_rho failed', k, rmid2
                     stop 'do_brunt_basic'
                  end if
            
                  lnd2 = log(rho2)
            
                  call interp_value(s% rmid, nz, f_lnd, rmid2, lnd_background, ierr)
                  if (ierr /= 0) then
                     write(*,2) 'interp_value lnd failed', k, rmid2
                     stop 'do_brunt_basic'
                  end if
            
               end if
            
               s% brunt_N2(k) = grav*(lnd2 - lnd_background)/dr
               
               if (.false.) then
                  s% profile_extra_name(1) = 'log_rho_ad'
                  s% profile_extra_name(2) = 'log_rho_back'
                  s% profile_extra_name(3) = 'log_P_back'
                  s% profile_extra_name(4) = 'log_Ppas2'
                  s% profile_extra_name(5) = 'dlogT'
         
                  s% profile_extra(k,1) = lnd2/ln10
                  s% profile_extra(k,2) = lnd_background/ln10
                  s% profile_extra(k,3) = lnP_background/ln10
                  s% profile_extra(k,4) = log10(Pgas2)
                  s% profile_extra(k,5) = (lnT2-s% lnT(k))/ln10
               end if
               
            else
               
               call zero_brunt(s,k)
            
            end if
            
         end subroutine do1
         
         
         subroutine zero_brunt(s, k)
            type (star_info), pointer :: s         
            integer, intent(in) :: k
            s% brunt_N2(k) = 0
            return
            s% profile_extra_name(1) = 'log_rho_ad'
            s% profile_extra_name(2) = 'log_rho_back'
            s% profile_extra_name(3) = 'log_P_back'
            s% profile_extra_name(4) = 'log_Ppas2'
            s% profile_extra_name(5) = 'dlogT'
            s% profile_extra(k,1) = 0
            s% profile_extra(k,2) = 0
            s% profile_extra(k,3) = 0
            s% profile_extra(k,4) = 0
            s% profile_extra(k,5) = 0
         end subroutine zero_brunt


      end subroutine do_brunt_basic


      end module brunt

