! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************



! -----------------------------------------------------------------------------------
!     the code bimd numerically solves (stiff) differential ode 
!     problems or linearly implicit dae problems of index up to 3 
!     with constant mass matrix
!
!     copyright (c)2005-2007   
!
!     authors: cecilia magherini (cecilia.magherini@ing.unipi.it)
!              luigi   brugnano  (brugnano@math.unifi.it) 
!
!
!     this program is free software; you can redistribute it and/or
!     modify it under the terms of the gnu general public license
!     as published by the free software foundation; either version 2
!     of the license, or (at your option) any later version.
!
!     this program is distributed in the hope that it will be useful,
!     but without any warranty; without even the implied warranty of
!     merchantability or fitness for a particular purpose.  see the
!     gnu general public license for more details.
!
!     licensed under the gnu general public license, version 2 or later.
!       http://www.gnu.org/licenses/info/gplv2orlater.html
!
!     you should have received a copy of the gnu general public license
!     along with this program; if not, write to the free software
!     foundation, inc., 51 franklin street, fifth floor, boston, ma  02110-1301,
!     usa.
! -----------------------------------------------------------------------------------

! L.Brugnano, C.Magherini, F.Mugnai.
! Blended Implicit Methods for the Numerical Solution of DAE Problems,
! Jour. Comput. Appl. Mathematics  189 (2006) 34-50.





      module bimdmtx
      
      
      IMPLICIT real(8)(A-H, O-Z), INTEGER(I-N)

      integer, parameter :: dp = 8

      contains


!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!
!     linear algebra
!
!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      

      subroutine sollu(n,a,lda,b,ml,mu,ipvt,ijob)
      !integer lda,n,ipvt(n),ijob
      !real(dp) a(lda,n),b(n)
      integer lda,n,ipvt(:),ijob
      real(dp) a(:,:),b(:)
      
      if (size(a,dim=1) /= lda .or. size(a,dim=2) /= n .or.
     >      size(b,dim=1) /= n .or. size(ipvt,dim=1) /= n) then
         write(*,*) 'bad sizes for sollu'
         stop 1
      end if

      goto (1,2) ijob

1     call sol(n,n,a,b,ipvt)
      return

2     call solb(n,lda,a,ml,mu,b,ipvt)
      return
      end subroutine sollu


      subroutine declu(n,a,lda,ml,mu,ipvt,ijob,info)
      !integer lda,n,ipvt(n),info,ijob
      !real(dp) a(lda,n)
      integer lda,n,ipvt(:),info,ijob
      real(dp) a(:,:)
      
      if (size(a,dim=1) /= lda .or. size(a,dim=2) /= n .or.
     >      size(ipvt,dim=1) /= n) then
         write(*,*) 'bad sizes for declu'
         stop 1
      end if

      goto(1,2) ijob

   1  call dec(n,n,a,ipvt,info)
      return

   2  call decb(n,lda,a,ml,mu,ipvt,info)
      return

      end subroutine declu


!----------------------- subroutine for the matrix-vector product  ------------------------

      subroutine matvec0(m,m0,ldmas,mlmas,mumas,v,mv,ijob)
!      input
      integer m,ldmas,mlmas,mumas,ijob
      !real(dp) m0(ldmas,m),v(m)
      real(dp) m0(:,:),v(:)
!      output
      !real(dp) mv(m)
      real(dp) mv(:)
!     local variables
      integer i,j
      
      if (size(m0,dim=1) /= ldmas .or. size(m0,dim=2) /= m .or.
     >      size(mv,dim=1) /= m .or. size(v,dim=1) /= m) then
         write(*,*) 'bad sizes for matvec0'
         stop 1
      end if

      goto(10,20) ijob

10    continue
!     full matrix
      do i=1,m
            mv(i)=0d0
            do j=1,m
                  mv(i)=mv(i) + m0(i,j)*v(j)
            end do
      end do

      return

20    continue
!     banded matrix
      do i=1,m
         mv(i)=0d0
         do j=max(1,i-mlmas),min(m,i+mumas)
            mv(i)=mv(i)+m0(i-j+mumas+1,j)*v(j)
         end do
      end do

      return
      end subroutine matvec0






!
!     subroutine dec
!
      subroutine dec (n, ndim, a, ip, ier)
! version real real(dp)
      integer n,ndim,ip,ier,nm1,k,kp1,m,i,j
      real(dp) a,t
      dimension a(ndim,n), ip(n)
!-----------------------------------------------------------------------
!  matrix triangularization by gaussian elimination.
!  input..
!     n = order of matrix.
!     ndim = declared dimension of array  a .
!     a = matrix to be triangularized.
!  output..
!     a(i,j), i.le.j = upper triangular factor, u .
!     a(i,j), i.gt.j = multipliers = lower triangular factor, i - l.
!     ip(k), k.lt.n = index of k-th pivot row.
!     ip(n) = (-1)**(number of interchanges) or o .
!     ier = 0 if matrix a is nonsingular, or k if found to be
!           singular at stage k.
!  use  sol  to obtain solution of linear system.
!  determ(a) = ip(n)*a(1,1)*a(2,2)*...*a(n,n).
!  if ip(n)=o, a is singular, sol will divide by zero.
!
!  reference..
!     c. b. moler, algorithm 423, linear equation solver,
!     c.a.c.m. 15 (1972), p. 274.
!-----------------------------------------------------------------------
      ier = 0
      ip(n) = 1
      if (n .eq. 1) go to 70
      nm1 = n - 1
      do 60 k = 1,nm1
        kp1 = k + 1
        m = k
        do 10 i = kp1,n
          if (dabs(a(i,k)) .gt. dabs(a(m,k))) m = i
 10     continue
        ip(k) = m
        t = a(m,k)
        if (m .eq. k) go to 20
        ip(n) = -ip(n)
        a(m,k) = a(k,k)
        a(k,k) = t
 20     continue
        if (t .eq. 0.d0) go to 80
        t = 1.d0/t
        do 30 i = kp1,n
 30       a(i,k) = -a(i,k)*t
        do 50 j = kp1,n
          t = a(m,j)
          a(m,j) = a(k,j)
          a(k,j) = t
          if (t .eq. 0.d0) go to 45
          do 40 i = kp1,n
 40         a(i,j) = a(i,j) + a(i,k)*t
 45       continue
 50       continue
 60     continue
 70   k = n
      if (a(n,n) .eq. 0.d0) go to 80
      return
 80   ier = k
      ip(n) = 0
      return
!----------------------- end of subroutine dec -------------------------
      end subroutine dec

!
!     subroutine sol
!
      subroutine sol (n, ndim, a, b, ip)
      integer n,ndim,ip(:),nm1,k,kp1,m,i,kb,km1
      !integer n,ndim,ip(n),nm1,k,kp1,m,i,kb,km1
      !real(dp) a(ndim,n),b(n),t
      real(dp) a(:,:),b(:),t
!-----------------------------------------------------------------------
!  solution of linear system, a*x = b .
!  input..
!    n = order of matrix.
!    ndim = declared dimension of array  a .
!    a = triangularized matrix obtained from dec.
!    b = right hand side vector.
!    ip = pivot vector obtained from dec.
!  do not use if dec has set ier .ne. 0.
!  output..
!    b = solution vector, x .
!-----------------------------------------------------------------------
      if (n .eq. 1) go to 50
      nm1 = n - 1
      do 20 k = 1,nm1
        kp1 = k + 1
        m = ip(k)
        t = b(m)
        b(m) = b(k)
        b(k) = t
        do 10 i = kp1,n
 10       b(i) = b(i) + a(i,k)*t
 20     continue
      do 40 kb = 1,nm1
        km1 = n - kb
        k = km1 + 1
        b(k) = b(k)/a(k,k)
        t = -b(k)
        do 30 i = 1,km1
 30       b(i) = b(i) + a(i,k)*t
 40     continue
 50   b(1) = b(1)/a(1,1)
      return
!----------------------- end of subroutine sol -------------------------
      end subroutine sol

!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccca
!
!     subroutine decb
!
      subroutine decb (n, ndim, a, ml, mu, ip, ier)
      real*8 a,t
      dimension a(ndim,n), ip(n)
!-----------------------------------------------------------------------
!  matrix triangularization by gaussian elimination of a banded
!  matrix with lower bandwidth ml and upper bandwidth mu
!  input..
!     n       order of the original matrix a.
!     ndim    declared dimension of array  a.
!     a       contains the matrix in band storage.   the columns
!                of the matrix are stored in the columns of  a  and
!                the diagonals of the matrix are stored in rows
!                ml+1 through 2*ml+mu+1 of  a.
!     ml      lower bandwidth of a (diagonal is not counted).
!     mu      upper bandwidth of a (diagonal is not counted).
!  output..
!     a       an upper triangular matrix in band storage and
!                the multipliers which were used to obtain it.
!     ip      index vector of pivot indices.
!     ip(n)   (-1)**(number of interchanges) or o .
!     ier     = 0 if matrix a is nonsingular, or  = k if found to be
!                singular at stage k.
!  use  solb  to obtain solution of linear system.
!  determ(a) = ip(n)*a(md,1)*a(md,2)*...*a(md,n)  with md=ml+mu+1.
!  if ip(n)=o, a is singular, solb will divide by zero.
!
!  reference..
!     this is a modification of
!     c. b. moler, algorithm 423, linear equation solver,
!     c.a.c.m. 15 (1972), p. 274.
!-----------------------------------------------------------------------
      ier = 0
      ip(n) = 1
      md = ml + mu + 1
      md1 = md + 1
      ju = 0
      if (ml .eq. 0) go to 70
      if (n .eq. 1) go to 70
      if (n .lt. mu+2) go to 7
      do 5 j = mu+2,n
      do 5 i = 1,ml
  5   a(i,j) = 0.d0
  7   nm1 = n - 1
      do 60 k = 1,nm1
        kp1 = k + 1
        m = md
        mdl = min(ml,n-k) + md
        do 10 i = md1,mdl
          if (dabs(a(i,k)) .gt. dabs(a(m,k))) m = i
 10     continue
        ip(k) = m + k - md
        t = a(m,k)
        if (m .eq. md) go to 20
        ip(n) = -ip(n)
        a(m,k) = a(md,k)
        a(md,k) = t
 20     continue
        if (t .eq. 0.d0) go to 80
        t = 1.d0/t
        do 30 i = md1,mdl
 30       a(i,k) = -a(i,k)*t
        ju = min0(max0(ju,mu+ip(k)),n)
        mm = md
        if (ju .lt. kp1) go to 55
        do 50 j = kp1,ju
          m = m - 1
          mm = mm - 1
          t = a(m,j)
          if (m .eq. mm) go to 35
          a(m,j) = a(mm,j)
          a(mm,j) = t
 35       continue
          if (t .eq. 0.d0) go to 45
          jk = j - k
          do 40 i = md1,mdl
            ijk = i - jk
 40         a(ijk,j) = a(ijk,j) + a(i,k)*t
 45       continue
 50       continue
 55     continue
 60     continue
 70   k = n
      if (a(md,n) .eq. 0.d0) go to 80
      return
 80   ier = k
      ip(n) = 0
      return
!----------------------- end of subroutine decb ------------------------
      end subroutine decb

!
!     subroutine solb
!
      subroutine solb (n, ndim, a, ml, mu, b, ip)
      real*8 a,b,t
      dimension a(ndim,n), b(n), ip(n)
!-----------------------------------------------------------------------
!  solution of linear system, a*x = b .
!  input..
!    n      order of matrix a.
!    ndim   declared dimension of array  a .
!    a      triangularized matrix obtained from decb.
!    ml     lower bandwidth of a (diagonal is not counted).
!    mu     upper bandwidth of a (diagonal is not counted).
!    b      right hand side vector.
!    ip     pivot vector obtained from decb.
!  do not use if decb has set ier .ne. 0.
!  output..
!    b      solution vector, x .
!-----------------------------------------------------------------------
      md = ml + mu + 1
      md1 = md + 1
      mdm = md - 1
      nm1 = n - 1
      if (ml .eq. 0) go to 25
      if (n .eq. 1) go to 50
      do 20 k = 1,nm1
        m = ip(k)
        t = b(m)
        b(m) = b(k)
        b(k) = t
        mdl = min(ml,n-k) + md
        do 10 i = md1,mdl
          imd = i + k - md
 10       b(imd) = b(imd) + a(i,k)*t
 20     continue
 25   continue
      do 40 kb = 1,nm1
        k = n + 1 - kb
        b(k) = b(k)/a(md,k)
        t = -b(k)
        kmd = md - k
        lm = max0(1,kmd+1)
        do 30 i = lm,mdm
          imd = i - kmd
 30       b(imd) = b(imd) + a(i,k)*t
 40     continue
 50   b(1) = b(1)/a(md,1)
      return
!----------------------- end of subroutine solb ------------------------
      end subroutine solb


      end module bimdmtx