
      ! L.Brugnano, C.Magherini, F.Mugnai.
      ! Blended Implicit Methods for the Numerical Solution of DAE Problems,
      ! Jour. Comput. Appl. Mathematics  189 (2006) 34-50.





      module bimdax

      use bimdmtx

      implicit none

      integer, parameter :: ns=3


      contains





      !cccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      !                                                         c
      !     in case the isnan function is not supported.        c
      !                                                         c
      !cccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      logical function isnan(a)
         real(dp) a, b
         logical x
         b = a
         x = ( a .eq. b )
         isnan = (.not.x)
      end function isnan


      subroutine blendstep4(m,y0,f0,y,f,h,theta,ipvt,z,gamma, &
         ldlu,mljac,mujac,ijob,ldmas,mlmas,mumas,m0,mz)
         ! blended iteration for the 4th order method
         integer, intent(in) :: m,ipvt(:),ldlu,mljac,mujac,ijob(2),ldmas,mlmas,mumas
         real(dp), intent(in) :: y0(:),f0(:),theta(:,:),gamma,h,m0(:,:)
         real(dp), dimension(:,:), intent(inout) :: y,f,mz  ! (m,ns)
         real(dp), intent(out) :: z(:,:) ! (m,ns)

         integer :: i,j

         real(dp), parameter :: &
            da4_1_1 = -102133d0/405d3, &
            da4_1_2 =   98743d0/18d4, &
            da4_1_3 =   -7387d0/225d2, &
            da4_1_4 =  +51709d0/162d4, &
            da4_2_1 = -950353d0/81d4, &
            da4_2_2 =   +7387d0/9d3, &
            da4_2_3 =   10613d0/18d3, &
            da4_2_4 =  -96031d0/405d3, &
            da4_3_1 =  -22613d0/3d4, &
            da4_3_2 =  -22161d0/2d4, &
            da4_3_3 =  +22161d0/1d4, &
            da4_3_4 =  -21257d0/6d4, &

            a24_1_1 = -302867d0/405d3, &
            a24_1_2 =  81257d0/18d4, &
            a24_1_3 =  7387d0/225d2, &
            a24_1_4 = -51709d0/162d4, &
            a24_2_1 =  140353d0/81d4, &
            a24_2_2 = -7387d0/9d3, &
            a24_2_3 =  7387d0/18d3, &
            a24_2_4 =  96031d0/405d3, &
            a24_3_1 = -7387d0/3d4, &
            a24_3_2 =  22161d0/2d4, &
            a24_3_3 = -22161d0/1d4, &
            a24_3_4 =  81257d0/6d4, &

            db4_1_1 =  919d0/135d2,  &
            db4_1_2 =  4589d0/3d4, &
            db4_1_3 = -37d0/12d1,  &
            db4_1_4 =  3d0/4d1, &
            db4_2_1 =  115387d0/27d4, &
            db4_2_2 =  17d0/15d0, &
            db4_2_3 = -6161d0/3d4,  &
            db4_2_4 = -1d0/15d0, &
            db4_3_1 =  3d0/8d0, &
            db4_3_2 =  9d0/8d0, &
            db4_3_3 =  9d0/8d0, &
            db4_3_4 = -3637d0/1d4, &
            b24_1_1 =  7387d0/27d3, &
            b24_2_1 = -7387d0/27d4,  &
            b24_3_1 =  0d0


         ! z=[y0 y]*(da)'
         do i=1,m
            z(i,1)=y0(i)*da4_1_1+y(i,1)*da4_1_2+y(i,2)*da4_1_3+ y(i,3)*da4_1_4
            z(i,2)=y0(i)*da4_2_1+y(i,1)*da4_2_2+y(i,2)*da4_2_3+ y(i,3)*da4_2_4
            z(i,3)=y0(i)*da4_3_1+y(i,1)*da4_3_2+y(i,2)*da4_3_3+ y(i,3)*da4_3_4
         end do

         ! mz_i=m0*z_i, i=1,2,3
         call matvec0(m,m0,ldmas,mlmas,mumas,z(1:m,1),mz(1:m,1),ijob(2))
         call matvec0(m,m0,ldmas,mlmas,mumas,z(1:m,2),mz(1:m,2),ijob(2))
         call matvec0(m,m0,ldmas,mlmas,mumas,z(1:m,3),mz(1:m,3),ijob(2))

         ! mz=mz-h*[f0 f]*(db)'
         do i=1,m
            mz(i,1) = mz(i,1)- &
               h*(f0(i)* db4_1_1+f(i,1)*db4_1_2+f(i,2)*db4_1_3+f(i,3)*db4_1_4)
            mz(i,2) = mz(i,2)- &
               h*(f0(i)* db4_2_1+f(i,1)*db4_2_2+f(i,2)*db4_2_3+f(i,3)*db4_2_4)
            mz(i,3) = mz(i,3)- &
               h*(f0(i)* db4_3_1+f(i,1)*db4_3_2+f(i,2)*db4_3_3+f(i,3)*db4_3_4)
         end do

         ! solve theta*mz=mz
         call sollu(m,theta,ldlu,mz(1:m,1),mljac,mujac,ipvt,ijob(1))
         call sollu(m,theta,ldlu,mz(1:m,2),mljac,mujac,ipvt,ijob(1))
         call sollu(m,theta,ldlu,mz(1:m,3),mljac,mujac,ipvt,ijob(1))

         ! mz=mz+[y0 y]*(a2)'
         do i=1,m
            mz(i,1)=mz(i,1)+ &
               y0(i)* a24_1_1+y(i,1)*a24_1_2+y(i,2)*a24_1_3+y(i,3)* a24_1_4
            mz(i,2)=mz(i,2)+ &
               y0(i)* a24_2_1+y(i,1)*a24_2_2+y(i,2)*a24_2_3+y(i,3)* a24_2_4
            mz(i,3)=mz(i,3)+ &
               y0(i)* a24_3_1+y(i,1)*a24_3_2+y(i,2)*a24_3_3+y(i,3)* a24_3_4
         end do

         ! z_i=m0*mz_i, i=1,2,3
         call matvec0(m,m0,ldmas,mlmas,mumas,mz(1:m,1),z(1:m,1),ijob(2))
         call matvec0(m,m0,ldmas,mlmas,mumas,mz(1:m,2),z(1:m,2),ijob(2))
         call matvec0(m,m0,ldmas,mlmas,mumas,mz(1:m,3),z(1:m,3),ijob(2))

         ! z=z-h*[f0 f]*(b2)'
         do i=1,m
            z(i,1)=z(i,1)-h*(f0(i)*b24_1_1 + f(i,1)*gamma)
            z(i,2)=z(i,2)-h*(f0(i)*b24_2_1 + f(i,2)*gamma)
            z(i,3)=z(i,3)-h*(f0(i)*b24_3_1 + f(i,3)*gamma)
         end do

         ! theta*z=z
         call sollu(m,theta,ldlu,z(1:m,1),mljac,mujac,ipvt,ijob(1))
         call sollu(m,theta,ldlu,z(1:m,2),mljac,mujac,ipvt,ijob(1))
         call sollu(m,theta,ldlu,z(1:m,3),mljac,mujac,ipvt,ijob(1))

         ! y = y-z
         do j=1,ns
            do i=1,m
               y(i,j) = y(i,j) - z(i,j)
            end do
         end do

      end subroutine blendstep4


      subroutine truncam(m,f0,f,h,z) ! truncation error estimate
         integer, intent(in) :: m
         real(dp), intent(in) :: f0(:), f(:,:), h
         real(dp), intent(out) :: z(:)

         integer :: i

         real(dp), parameter :: psi4_1 = -1d0, psi4_2 =  3d0

         do i=1,m
            z(i)=h*(psi4_1*(f0(i)-f(i,3)) +psi4_2*(f(i,1)-f(i,2)))
         end do

      end subroutine truncam


      subroutine localerr4( &
            m,f0,f,h,z,scal,nerr,nerrup,nlinsys, &
            theta,vmax,ipvt,ldlu,mljac,mujac,ijob, &
            ldmas,mlmas,mumas,m0,index1,index2)
         integer, intent(in) :: m,ipvt(:),ldlu,mljac,mujac,ijob(2),&
            ldmas,mlmas,mumas,index1,index2
         real(dp), intent(in) :: f0(:),f(:,:),theta(:,:),h,scal(:),m0(:,:), vmax(:)
         real(dp), intent(out) :: z(:,:),nerr,nerrup
         integer, intent(inout) :: nlinsys

         integer :: i

         call truncam(m,f0,f,h,z(1:m,1))

         do i=1,m
            z(i,2) = z(i,1)
         end do

         call sollu(m,theta,ldlu,z(1:m,2),mljac,mujac,ipvt,ijob(1))

         call matvec0(m,m0,ldmas,mlmas,mumas,z(1:m,2),z(1:m,3),ijob(2))

         do i=1,m
            z(i,3)=z(i,1)-z(i,3)
         end do

         call sollu(m,theta,ldlu,z(1:m,3),mljac,mujac,ipvt,ijob(1))

         do i=1,index1
            z(i,2)=vmax(1)*z(i,2)
            z(i,3)=vmax(2)*z(i,3)
         end do
         do i =index1+1,index1+index2
            z(i,2) = vmax(2)*z(i,2)
            z(i,3) = vmax(2)*z(i,3)
         end do
         do i = index1+index2+1,m
            z(i,2) = vmax(3)*z(i,2)
            z(i,3) = vmax(3)*z(i,3)/2d0
         end do

         call norm(m,2,scal,z(1:m,2:3),nerr,nerrup)

         nlinsys = nlinsys + 2

      end subroutine localerr4


      subroutine extrapola (m,h0,h,ynew,dd)
         integer, intent(in) :: m
         real(dp), intent(in) :: h0,h,dd(:,:)
         real(dp), intent(out) :: ynew(:,:)

         integer i,j,l
         real(dp) dt,rath

         rath = (h/h0)
         do i=1,m
            ! evaluation of the interpolating polynomial at the points of the new block
            do l=1,ns
               dt = dble(l)*rath
               ynew(i,l)=dd(ns+1,i)
               do j=ns,1,-1
                  dt = dt+1d0
                  ynew(i,l)=ynew(i,l)*dt +dd(j,i)
               end do
            end do
         end do

      end subroutine extrapola


      subroutine diffdiv(m,y0,y,dd)
         ! compute the divided differences of the interpolating polynomial
         integer, intent(in) :: m
         real(dp), intent(in) :: y0(:),y(:,:)
         real(dp), intent(out) :: dd(:,:)

         integer :: i,j,l
         real(dp) :: dt

         do i=1,m
            dd(1,i)=y0(i)
            do j=1,ns
               dd(j+1,i)=y(i,j)
            end do
            do j=2,ns+1
               dt = dble(j-1)
               do l=ns+1,j,-1
                  dd(l,i)=(dd(l,i)-dd(l-1,i))/dt
               end do
            end do
         end do

      end subroutine diffdiv



      real(dp) function contsol(l,t,m,k,t0,tstep,dd)
      !
      !     function to be used if continuous output is desired.
      !     it provides the value, at time t, of the polynomial 
      !     interpolating the l-th component of the numerical solution 
      !     obtained at the last successfully computed step

      implicit none

      integer l,k,m
      real(dp) t,yc,t0,tstep(:),dd(:,:) ! (maxk+1,m)

      integer i
      real(dp) dt      

      if (tstep(1).le.t0) then
      write(6,10) t
      10      format(/,/,'warning: in calling to subroutine contsol',/, &
      'the input parameter t_0 must be strictly lower',/, &
      'than tstep(1). the approximation of the solution',  &
      /, 'at t = ',d18.4, 'is not returned.')
      return
      end if

      dt = (t-tstep(k))/(tstep(1)-t0)
      yc = dd(k+1,l)

      do i=k,1,-1
      dt = dt + 1d0
      yc = yc*dt + dd(i,l)
      end do

      contsol = yc

      end function contsol


      subroutine norm(m,nss,scal,err,nerr,nerrup)
         integer, intent(in) :: m,nss
         real(dp), intent(in) :: scal(:), err(:,:)
         real(dp), intent(out) :: nerr,nerrup

         real(dp) :: nerr0
         integer :: i,j

         nerr = 0d0
         do j=1,nss-1
            nerr0=0d0
            do i=1,m
               nerr0 = nerr0 + (err(i,j)*scal(i))**2
            end do
            nerr = dmax1(nerr,nerr0)
         end do

         nerrup=0d0
         do i=1,m
            nerrup  = nerrup + (err(i,nss)*scal(i))**2
         end do
         nerr = dmax1(nerr,nerrup)
         nerr   = dsqrt(nerr/dble(m))
         nerrup = dsqrt(nerrup/dble(m))

      end subroutine norm


      subroutine set_pointer_2(Aptr, dest, n1, n2)
         real(dp), pointer :: Aptr(:,:)
         real(dp), target :: dest(n1, n2)
         integer, intent(in) :: n1, n2
         Aptr => dest
      end subroutine set_pointer_2


      end module bimdax

