! -----------------------------------------------------------------------------------
!     the code bimd numerically solves (stiff) differential ode 
!     problems or linearly implicit dae problems of index up to 3 
!     with constant mass matrix
!
!     copyright (c)2005-2007   
!
!     authors: cecilia magherini (cecilia.magherini@ing.unipi.it)
!              luigi   brugnano  (brugnano@math.unifi.it) 
!
!
!     this program is free software; you can redistribute it and/or
!     modify it under the terms of the gnu general public license
!     as published by the free software foundation; either version 2
!     of the license, or (at your option) any later version.
!
!     this program is distributed in the hope that it will be useful,
!     but without any warranty; without even the implied warranty of
!     merchantability or fitness for a particular purpose.  see the
!     gnu general public license for more details.
!
!     licensed under the gnu general public license, version 2 or later.
!       http://www.gnu.org/licenses/info/gplv2orlater.html
!
!     you should have received a copy of the gnu general public license
!     along with this program; if not, write to the free software
!     foundation, inc., 51 franklin street, fifth floor, boston, ma  02110-1301,
!     usa.
! -----------------------------------------------------------------------------------

! L.Brugnano, C.Magherini, F.Mugnai.
! Blended Implicit Methods for the Numerical Solution of DAE Problems,
! Jour. Comput. Appl. Mathematics  189 (2006) 34-50.

      subroutine bimd(m,fcn,t0,tend,y0,h, &
                      rtol,atol,itol, &
                      jac,ijac,mljac,mujac, &
                      mas,imas,mlmas,mumas, &
                      solout,iout, &
                      work,lwork,iwork,liwork, &
                      rpar,ipar,idid)
         
         
         use mod_bimd, only: bim0
         
         
! -----------------------------------------------------------------------------------
! -----------------------------------------------------------------------------------
!
!     purpose:    bimd solves a (stiff) differential ode problem,
!     --------
!                               y'    = f(t,y),     t0<=t<=tend,
!                               y(t0) = y0,
!
!                 or a linearly implicit dae problem of index up to 3 with
!                 constant mass matrix, namely problem in the form
!
!                 m y'  = f(t,y),     t0<=t<=tend,
!                 y(t0) = y0,
!
!                 where m is a possibly singular matrix.
!
!                 the code is based on blended implicit methods.
!                 blended implicit methods are a class of block
!                 methods       providing a (relatively) easy definition
!                 of suitable nonlinear splittings for solving the
!                 corresponding discrete problems [1,5-7].
!                 the code bimd implements a variable stepsize-
!                 variable order method. orders: 4-6-8-10-12.
!                 implementation details are in references [1-5].
!
!
!
!     authors:    c.magherini, 
!     --------    dipartimento di matematica applicata "u.dini"
!                 via buonarroti, 1/c 
!                 56127 pisa
!                 italy
!                 e-mail: cecilia.magherini@ing.unipi.it
!
!                 l.brugnano,
!                 dipartimento di matematica "u.dini"
!                 viale morgagni 67/a
!                 50134 firenze
!                 italy
!                 e-mail: brugnano@math.unifi.it
!
!     code home page:   http://www.math.unifi.it/~brugnano/bim/index.html
!     ---------------
!
!     code:       the code is made up of two files:
!     -----        - bimd.f     (i.e. the present file) which contains the main
!                    integration procedure
!                  - bimda.f    containing additional and linear algebra
!                    procedures
!
!     current release:   1.1.1,  september, 2006.
!     ----------------
!
!     release history:   1.0,  october, 2005
!     ---------------    - first version released;
!                        
!                        1.1,  july, 2006
!                        main features (wrt 1.0):
!                        - improved definition of the 
!                          coefficients of the methods
!                        - the results described
!                          in reference [1], have been better
!                          exploited for the definition of 
!                          the stopping criterion for
!                          the splitting newton blended iteration
!                        - improved choice of the initial profile
!                          after a failure due to newton convergence
!                        - possibility of solving the problem
!                          with vector-valued absolute input
!                          tolerances
!                        - new function contsol, to be used when
!                          continuous output is desired
!                        - minor changes concerning the
!                          order variation strategy.
! 
!                        1.1.1, september, 2006
!                        - some minor bugs fixed
!                          
!                             
!
!     references:
!     -----------
!                 [1] l.brugnano, c.magherini, f.mugnai.
!                     blended implicit methods for the numerical solution
!                     of dae problems.
!                     jour. cam 189 (2006) 34-50.
!
!                 [2] l.brugnano, c.magherini
!                     the bim code for the numerical solution of odes
!                     jour. cam 164-165 (2004) 145-158.
!
!                 [3] l.brugnano, c.magherini
!                     some linear algebra issues concerning the implementation
!                     of blended implicit methods
!                     numer. linear alg. appl. 12 (2005) 305-314.
!
!                 [4] l.brugnano, c.magherini
!                     economical error estimates for block implicit methods for
!                     odes via deferred correction.
!                     appl. numer. math. 56 (2006) 608-617.
!
!                 [5] l.brugnano, c.magherini
!                     blended implementation of block implicit methods for odes
!                     appl. numer. math. 42 (2002) 29-45.
!
!                 [6] l.brugnano, d.trigiante
!                     block implicit methods for odes
!                     in "recent trends in numerical analysis", d.trigiante ed.
!                     nova science publ. inc., new york, 2001, pp. 81-105.
!
!                 [7] l.brugnano
!                     blended block bvms (b$_3$vms): a family of economical
!                     implicit methods for odes
!                     jour. cam 116 (2000) 41-62.
!
!
!
!    remark:   the code bimd has been written using a style very similar to the one
!    -------   used in the codes radau and gam. indeed, some subroutines and comments 
!              have been imported from such codes. moreover, the name and the meaning
!              of a number of input parameters and variables have been fully inherited 
!              from them.
!               
! -----------------------------------------------------------------------------------
! -----------------------------------------------------------------------------------
!
!     usage:
!     ------
!
!      call bimd(m,fcn,t0,tend,y0,h,
!     &          rtol,atol,itol,
!     &          jac,ijac,mljac,mujac,
!     &          mas,imas,mlmas,mumas,
!     &          solout,iout,
!     &          work,lwork,iwork,liwork,
!     &          rpar,ipar,idid)
!
!     note:   in order to gain the best performance, the executable has to
!     -----   be created with the option allowing  to continue the execution
!             after a floating-point exception  (e.g., by using the option
!             -fpe; see your fortran compiler reference manual).
!                       the isnan logical function is required, to recognize nans. if
!                       not supported by your compiler, a standard one is provided at
!                       the top of the subbim.f file.
!
!
! -----------------------------------------------------------------------------------
!           input parameters
! -----------------------------------------------------------------------------------
!
! m             size of the problem
!
! fcn           subroutine with the function f(t,y) to be integrated.
!
!
!      subroutine fcn(m,t,y,dy,ierr,rpar,ipar)
!      double precision t,y,dy,rpar(*)
!      integer m,ierr,ipar(*)
!      dimension y(m),dy(m)
!      m      size of the continuous problem
!      t,y    is the point where f is evaluated
!      dy     will contain the value of f(t,y)
!      ierr   is a return code (0 means ok)
!      rpar   possible external real parameters
!      ipar   possible external integer parameters
!      ................
!      return
!      end
!
! t0-tend       integration interval
!
! y0            initial condition
!
! h             initial stepsize
!
! rtol-atol     relative and absolute tolerances. 
!               atol can be either scalar or a 
!               vector of length m.
! 
! itol          switch for atol:
!               
!               itol = 0 --> atol is scalar. 
!                            the code provides a numerical solution 
!                            with the local error of y(i) roughly smaller 
!                            than atol + rtol*abs(y(i))   
!                        
!               itol = 1 --> atol is an array of length m 
!                            the local error of y(i) is kept
!                            below atol(i) + rtol*abs(y(i))   
!
! jac           subroutine evaluating the jacobian of f (dummy, if ijac=0)
!
!
!      subroutine jac(m,t,y,jac,ldjac,ierr,rpar,ipar)
!      double precision t,y,jac,rpar(*)
!      integer m,ldjac,ierr,ipar(*)
!      dimension y(m),jac(ldjac,m)
!      m      size of the continuous problem
!      t,y      is the point where the jacobian is evaluated
!      jac      will contain the value of the jacobian at (t,y)
!      ldjac  leading dimension of the array jac
!      ierr     is a return code (0 means ok)
!      rpar   possible external real parameters
!      ipar     possible external integer parameters
!      ............
!      return
!      end
!
! ijac          flag: 0=numerical jacobian, analytical otherwise
!
! mljac-mujac   lower-upper bandwidth of the jacobian (mljac=m if full jacobian)
!
! mas           subroutine evaluating the mass-matrix (dummy, if imas=0)
!
!
!      subroutine mas(m,mas,ldmas,ierr,rpar,ipar)
!      double precision mas,rpar(*)
!      integer m,ldmas,ierr,ipar(*)
!      dimension mas(ldmas,m)
!      m      size of the continuous problem
!      mas    will contain the evaluated mass-matrix
!      ldmas  leading dimension of the array mas
!      ierr     is a return code (0 means ok)
!      rpar   possible external real parameters
!      ipar     possible external integer parameters
!      ..............
!      return
!      end
!
!
! imas            flag: 0=ode, dae otherwise
!
! mlmas-mumas     lower-upper bandwidth of the mass-matrix (mlmas=m if full mass-matrix)
!                 mlmas is supposed to be .le. mljac
!                 mumas is supposed to be .le. mujac.
!
! lwork   length of work   ( lwork >= 14 +kmax +9*m +5*kmax*m +m*(ldjac+ldlu+ldmas),
!
!         where:
!
!                            ldjac = ldlu = m,            in case of a full jacobian,
!            ldjac = mljac+mujac+1,  ldlu = ldjac+mljac,  in case of a banded jacobian;
!
!                            ldmas = m                    in case of a full mass matrix,
!                            ldmas = mlmas+mumas+1        in case of a banded mass matrix,
!                            ldmas = 1                    in the ode case (i.e. imas = 0)
!
!                             kmax = ordmax-2,            if ordmax>4,
!                                    3,                   if ordmax=4. )
!
! work(1)   uround. machine precision. (default = 1.d-16)
!
! work(2)   hmax. maximum integration step. (default = (tend-t0)/8)
!
! work(3)   factol - safety factor for the stopping criterion of the blendled iteration.
!           method of order 4. (default = 1.d-1)
!
! work(4)   factol - safety factor for the stopping criterion of the blendled iteration.
!           method of order 6. (default = 1.d-1)
!
! work(5)   factol - safety factor for the stopping criterion of the blendled iteration.
!           method of order 8. (default = 1.d-1)
!
! work(6)   factol - safety factor for the stopping criterion of the blendled iteration.
!           method of order 10. (default = 1.d-1)
!
! work(7)   factol - safety factor for the stopping criterion of the blendled iteration.
!           method of order 12. (default = 1.d-1)
!
! work(8)   factol - safety factor for the stopping criterion of the blendled iteration
!           in case of small values of min(abs(y_0)), min(abs(f_0)) and of max(abs(f_0)).
!           (default = 1d-2)
!
! work(9)   factol - safety factor for the stopping criterion of the blendled iteration
!           in case of slowly varying. (default = 5d-2)
!
! work(10)-work(11)  facl-facr. the new stepsize must satisfy facl<=hnew/hold<= facr.
!           (default: work(10)=1.2d-1, work(11)=1d1)
!
! work(12)  sfty - safety factor for predicting the new stepsize for the current order
!           method. (default = 1d0/2d1)
!
! work(13)  sftyup - safety factor for predicting the new stepsize for the higher order
!           method. (default = sfty/2d0)
!
! work(14)  sftydn - safety factor for predicting the new stepsize for the lower order
!           method. (default = sfty)
!
! liwork    length of iwork  (liwork >= m+40)
!
! iwork( 1) max number of integration steps (default = 100000).
!
! iwork( 2) ordmin, 4<=ordmin<=12. (default = 4).
!
! iwork( 3) ordmax, ordmin<=ordmax<=12. (default = 12).
!
! iwork( 4) max number of blended iterations per integration step, method of order 4
!           (default = 10).
!
! iwork( 5) max number of blended iterations per integration step, method of order 6
!           (default = 12).
!
! iwork( 6) max number of blended iterations per integration step, method of order 8
!           (default = 14).
!
! iwork( 7) max number of blended iterations per integration step, method of order 10
!           (default = 16).
!
! iwork( 8) max number of blended iterations per integration step, method of order 12
!           (default = 18).
!
! iwork( 9) dimension of the index 1 variables (default = m).
!           it must be greater than 0.
!
! iwork(10) dimension of the index 2 variables (default = 0).
!
! iwork(11) dimension of the index 3 variables (default = 0).
!
! remark: the variables must be sorted by increasing index.
! -------
!
! rpar,ipar  real and integer parameters (or parameter arrays) which
!            can be used for communication between your calling
!            program and the fcn, jac, mas and solout subroutines.
!
! solout     name (external) of subroutine providing the numerical
!            solution during integration. 
!            if iout = 1, it is called after each successful step.  
!            supply a dummy subroutine if iout = 0.
!            it must have the following form:
!
!      subroutine solout(m,k,ord,t0,t,y,f,dd,rpar,ipar,irtrn)
!      integer m,k,ord,irtrn,ipar(*)
!      double precision t0,t,y,f,dd, rpar(*)
!      dimension t(k),y(m,k),f(m,k),dd(k+1,m)
!
!      m                is the size of the problem
!      k                is the block-size of the method
!      ord              is the order of the method
!      t0               is the starting time point of the step
!      t                contains the (internal) mesh points of 
!                       the step
!      y                is the current numerical solution
!      f                contains the values of fcn(t,y)
!      dd               contains the divided differences of y 
!                       over the internal mesh points of the step
!                       (to be used, for example, if continuous 
!                       output is desired, see below)
!      rpar             possible external real parameters
!      ipar             possible external integer parameters
!      irtrn            is a return code. if set <0, bimd returns
!                       to the calling program.
!
!      ................
!      return
!      end
!
!
!           continuous output:
!           ------------------
!
!           during calls to solout, a continuous solution 
!           for the interval [t0,t(k)] is available through
!           the function
!
!               contsol(i,t,m,k,t0,tstep,dd)
!
!           which provides an approximation to the i-th
!           component of the solution at the time point t.
!
!
! iout      switch for calling the subroutine solout.
!          
!           iout = 0, solout is never called
!           iout = 1, solout is called after each 
!                     successfull step         
!
!
!
! -----------------------------------------------------------------------------------
!           output parameters
! -----------------------------------------------------------------------------------
!
!
! t0        value of t up to where the solution has been computed
!           (if the integration has been succesfull,then t0=tend)
!
! y0        numerical solution at t0
!
! idid      return code:
!              0  succesfull run
!             -1  wrong input parameters
!             -2  larger nmax is needed
!             -3  stepsize too small
!             -4  repeatedly singular matrix
!             -5  too many consecutive newton failures
!             -6  error code returned by the jac subroutine or by the fcn subroutine
!                 at the starting point
!
! iwork(12) number of function evaluations
!
! iwork(13) number of jacobian evaluations
!
! iwork(14) number of lu decomposition
!
! iwork(15) number of linear systems solved
!
! iwork(16)-iwork(20) number of blended iterations per method
!
! iwork(21)-iwork(25) number of step per method
!
! iwork(26)-iwork(30) number of accepted step per method
!
! iwork(31)-iwork(35) number of refused step per method (error test)
!
! iwork(36)-iwork(40) number of refused step per method (newton's convergence)
!
! -----------------------------------------------------------------------------------
! -----------------------------------------------------------------------------------
      implicit none

      external fcn,jac,mas,solout
      integer m,lwork,liwork,iwork(liwork), &
              itol,ijac,mljac,mujac,ldjac, &
              imas,mlmas,mumas,ldmas,iout,idid, &
              ldlu,ijob(2),ipar(*)
      logical jband,mband
      double precision t0,tend,y0(m),h,rtol,atol(*),work(lwork),rpar(*)

      integer nmeth,kmax
      parameter (nmeth=1,kmax=10)

      integer maxstep,ordmin,ordmax,itmax(nmeth),step_ord(nmeth)

      double precision uround, facnewtv(nmeth),facnsmall, &
                       facnrestr,facl,facr, &
                       sfty, sftyup, sftydn, hmax, &
                       rhomuv(nmeth),rhomlv(nmeth),tolestrapr

      integer i,indf0,indt,indipvt,indej0,indy,indf,indtheta,indj0, &
              inderr,indscal,inddelj0,inddelj0old,indtolext, &
              indfj0,indord,indscalext,ind_dd,indm0,indtemp

      logical stopint


      step_ord(1)= 3

      stopint = .false.

!     initial step-size
      if (h.eq.0d0) then
          h=1.d-6
      elseif(h.lt.0d0) then
          write(6,*) 'wrong input h=',h
          stopint=.true.
      end if

!--------------------------------------------------
!     parameters initialization
!--------------------------------------------------
      if (iwork(1).eq.0) then
         maxstep=100000
      else
         maxstep=iwork(1)
         if (maxstep.le.0) then
             write(6,*) 'wrong input iwork(1)=',iwork(1)
             stopint=.true.
         end if
      endif

      if (iwork(2).eq.0) then
         ordmin = 4
      else
         ordmin = iwork(2)
         indord = ordmin/2-1
         if ((indord.le.0).or.(indord.gt.nmeth)) then
             write(6,*) 'wrong input iwork(2)=',iwork(2)
             stopint=.true.
         end if
         ordmin = 2*(indord+1)
      endif

      if (.false. .and. iwork(3).eq.0) then
         ordmax = 12
         indord = nmeth
      else
         iwork(3) = 4
         ordmax = iwork(3)
         indord = ordmax/2 - 1
         if ((indord.le.0).or.(indord.gt.nmeth)) then
             write(6,*) 'wrong input iwork(3)=',iwork(3)
             stopint=.true.
         end if
         ordmax = 2*(indord+1)
      endif

      if (ordmin.gt.ordmax) then
        write(6,1000) iwork(2),iwork(3)
 1000   format(/,/,'invalid values for iwork(2)=',i3,' (ordmin)',/, &
                   '               and iwork(3)=',i3,' (ordmax)',/,/)
        stopint=.true.
      end if

      if (iwork(4) .eq. 0) then
         itmax(1) = 10
      else
         itmax(1) = iwork(4)
         if (itmax(1).le.0) then
             write(6,*) 'wrong input iwork(4)=',iwork(4)
             stopint=.true.
         end if
      end if

        if ((iwork(9)+iwork(10)+iwork(11)).eq.0) then
                iwork( 9) = m
                iwork(10) = 0
                iwork(11) = 0
        else
          if (iwork(9).eq.0) then
             write(6,1010) iwork(9)
 1010        format(/,/,'invalid values for iwork(9)=',i2,/, &
                        'it must be greater than 0',/,/)
             stopint = .true.
          end if
          if ((iwork(9)+iwork(10)+iwork(11)).ne.m) then
            write(6,1020) iwork(9), iwork(10), iwork(11)
 1020       format(/,/,'invalid values for iwork(9)=',i5,/, &
                       '                   iwork(10)=',i5,/, &
                       '               and iwork(11)=',i5,/,/)
              stopint=.true.
          end if
      end if

      if ((imas.ne.0).and.((mlmas.gt.mljac).or.(mumas.gt.mujac)))  &
      then
        write (6,*) &
         'bandwith of "mas" not smaller than bandwith of "jac"'
          stopint=.true.
      end if

      if (work(1) .eq. 0d0) then
         uround = 1.0d-16
      else
         uround = work(1)
         if ((uround.le.0d0).or.(uround.ge.1d0)) then
             write(6,*) 'wrong input work(1)=',work(1)
             stopint=.true.
         end if
      end if

      if (rtol.le.uround) then
          write(6,*) 'rtol is too small'
          stopint = .true.
      end if

      if (itol.eq.0) then
        if (atol(1).le.0d0) then
          write(6,*) 'atol is too small'
          stopint = .true.
        end if
      else
        do i=1,m
          if (atol(i).le.0d0) then
            write(6,1025) i
            stopint=.true.
          end if
        end do
 1025     format(/,/,'atol(',i5,') is too small',/,/)
      end if

      if (work(2) .eq. 0d0) then
         hmax = (tend-t0)/8d0
      else
         hmax = work(2)
         if (hmax.lt.0d0) hmax=(tend-t0)/8d0
         if (hmax.gt.(tend-t0)) hmax = tend-t0
      end if

      if (work(3) .eq. 0d0) then
         facnewtv(1) = 1d-1
      else
         facnewtv(1) = work(3)
         if ((facnewtv(1).le.0d0).or.(facnewtv(1).ge.1d0)) then
             write(6,*) 'wrong input work(3)=',work(3)
             stopint=.true.
         end if
      end if

      if (work(8) .eq. 0d0) then
         facnsmall = 1d-2
      else
         facnsmall = work(8)
         if ((facnsmall.le.0d0).or.(facnsmall.ge.1d0)) then
             write(6,*) 'wrong input work(8)=',work(8)
             stopint=.true.
         end if
      end if

      if (work(9) .eq. 0d0) then
         facnrestr = 5d-2
      else
         facnrestr = work(9)
         if ((facnrestr.le.0d0).or.(facnrestr.ge.1d0)) then
             write(6,*) 'wrong input work(9)=',work(9)
             stopint=.true.
         end if
      end if

      if (work(10) .eq. 0d0) then
         facl = 1.2d-1
      else
         facl = work(10)
         if (facl.lt.0d0) then
             write(6,*) 'wrong input work(10)=',work(10)
             stopint=.true.
         end if
      end if

      if (work(11) .eq. 0d0) then
                facr = 10d0
      else
         facr = work(11)
         if(facr.le.0d0) then
             write(6,*) 'wrong input work(11)=',work(11)
             stopint=.true.
         end if
         if(facl.ge.facr) then
            write(6,1030) work(10),work(11)
 1030       format(/,/,'invalid values for work(10)=',e10.2,' (facl)',/, &
                   '               and work(11)=',e10.2,' (facr)',/,/)
            stopint=.true.
         end if
      end if

      if (work(12) .eq. 0d0) then
         sfty = 1d0/20d0
      else
         sfty = work(12)
         if(sfty.le.0d0) then
             write(6,*) 'wrong input work(13)=',work(12)
             stopint=.true.
         end if
      end if

      if (work(13) .eq. 0d0) then
         sftyup = .5d0*sfty
      else
         sftyup = work(13)
         if(sftyup.le.0d0) then
             write(6,*) 'wrong input work(13)=',work(13)
             stopint=.true.
         end if
      end if

      if (work(14) .eq. 0d0) then
         sftydn = sfty
      else
         sftydn = work(14)
         if(sftydn.le.0d0) then
             write(6,*) 'wrong input work(14)=',work(14)
             stopint=.true.
         end if
      end if

      if (stopint) then
!     invalid input parameters
         idid = -1
         return
      end if

!---------------------------------------------------------
!     fixed parameters
!---------------------------------------------------------

      rhomuv(1) = 1d-2*dabs(dlog10(dmin1(rtol,1d-1)))
      rhomlv(1) = 5d-1

!--------------------------------------------------------
!     banded matrix
!--------------------------------------------------------

        jband = (mljac .lt. m)
        if (jband) then
                ldjac   = mljac+mujac+1
                ldlu    = ldjac+mljac
                ijob(1) = 2
        else
          ldjac = m
                ldlu     = m
                ijob(1)  = 1
        end if

      if (imas.eq.0) then
           iwork(9) = m
           mband     = .false.
           ldmas     = 1
      else
             mband = (mlmas .lt. m)
             if (mband) then
               ldmas   = mlmas+mumas+1
               ijob(2) = 2
           else
               ldmas   = m
               ijob(2) = 1
           end if
      end if


!---------------------------------------------------------
!     compute the vectors entry-point in iwork and work
!---------------------------------------------------------


      indipvt = 41
      if ((indipvt + m-1) .gt. liwork) then
        write(6,*) 'insuff. storage for iwork, min. liwork=',indipvt+m-1
        idid = -1
        return
      end if


      indf0       = 15
      indt        = indf0       + m
      indy        = indt        + step_ord(indord)
      indf        = indy        + m*step_ord(indord)
      indtheta    = indf        + m*step_ord(indord)
      inderr      = indtheta    + m*ldlu
      indtemp     = inderr      + m*step_ord(indord)
      indscal     = indtemp     + m*step_ord(indord)
      indtolext   = indscal     + m
      indscalext  = indtolext   + m
      indj0       = indscalext  + m
      indm0       = indj0       + m*ldjac
      inddelj0    = indm0       + m*ldmas
      inddelj0old = inddelj0    + m
      ind_dd      = inddelj0old + m
      indfj0      = ind_dd      + m*(step_ord(indord)+1)
      indej0      = indfj0      + m

      if ((indej0 + m-1) .gt. lwork) then
        write(6,*) 'insuff. storage for work, min. lwork=',indej0+m-1
        idid = -1
        return
      end if

      tolestrapr   = dmin1(1d-2,1d2*rtol)
      if (itol.eq.0) then
         do i=1,m
            work(indtolext+i-1) = dmin1(1d-2,1d2*atol(1))
         end do
      else
         do i=1,m
            work(indtolext+i-1) = dmin1(1d-2,1d2*atol(i))
         end do
      end if
         
      call  bim0(m,fcn,jac,nmeth,step_ord(indord),y0,work(indf0), &
                 t0,tend,h,rtol,atol,itol, &
                 maxstep,ordmin,ordmax,itmax,uround,hmax,facnewtv, &
                 facnsmall,facnrestr,facl,facr,sfty,sftyup,sftydn, &
                 rhomuv,rhomlv, &
                 iwork(12),iwork(13),iwork(14),iwork(15),iwork(16), &
                 iwork(21),iwork(26),iwork(31),iwork(36), &
                 iwork(indipvt),step_ord, &
                 work(indt),work(indy),work(indf),work(indtheta), &
                 work(inderr),work(indj0),work(inddelj0), &
                 work(inddelj0old),work(indfj0),work(indej0), &
                 work(indscal),work(ind_dd), &
                 tolestrapr,work(indtolext), &
                 work(indscalext), &
                 ijac,mljac,mujac,ldjac,ldlu,jband,ijob, &
                 rpar,ipar,iout,solout,idid, &
                 mas,imas,mlmas,mumas,ldmas,mband, &
                 iwork(9),iwork(10), &
                 work(indm0),work(indtemp))


      return
      end

