! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module screen_lib
      
      use screen_def
      
      implicit none


      contains
      

      subroutine screen_init_AZ_info(
     >         a1, z1, a2, z2,
     >         zg1, zg2, zg3, zg4, zs13,
     >         zhat, zhat2, lzav, aznut, zs13inv,
     >         ierr)
         use screen5, only: screen5_init_AZ_info
         use screen_graboske, only: graboske_init_z_info
         real*8, intent(in) :: a1, z1, a2, z2
         real*8, intent(out) :: zg1, zg2, zg3, zg4
         real*8, intent(out) :: zs13, zhat, zhat2, lzav, aznut, zs13inv
         integer, intent(out) :: ierr
         call graboske_init_z_info(zg1, zg2, zg3, zg4, z1, z2, ierr)
         if (ierr /= 0) return
         call screen5_init_AZ_info(
     >         zs13, zhat, zhat2, lzav, aznut, zs13inv, a1, z1, a2, z2, ierr)
      end subroutine screen_init_AZ_info
      
      
      ! call this once before calling screen_pair for each reaction
      ! sets info that depends only on temp, den, and overall composition         
      subroutine screen_set_context(
     >      sc, temp, den, logT, logRho, zbar, abar, z2bar, 
     >      screening_mode, graboske_cache, 
     >      theta_e, num_isos, y, iso_z)
         use screen, only: do_screen_set_context
         type (Screen_Info), pointer :: sc
         integer, intent(in) :: num_isos
         real*8, intent(in) :: 
     >         temp, den, logT, logRho, zbar, abar, z2bar, y(num_isos), iso_z(num_isos)
            ! y(:) = x(:)/chem_A(chem_id(:))
            ! iso_z(:) = chem_Z(chem_id(:))
         integer, intent(in) :: screening_mode ! if false, use fxt screening scheme.
         real*8, pointer :: graboske_cache(:,:,:)
         real*8 :: theta_e ! for screening_mode == classic_screening
            ! theta_e is used in the Graboske et al screening method.
            ! for non-degenerate electrons, theta_e goes to 1.
            ! for significantly degenerate electrons, it goes to 0.
         call do_screen_set_context(
     >      sc, temp, den, logT, logRho, zbar, abar, z2bar,
     >      screening_mode, graboske_cache, theta_e, num_isos, y, iso_z)
      end subroutine screen_set_context


      ! set jscr = 0 before 1st call.
      ! make calls in exactly the same order as for screen_init_AZ_info
      
      subroutine screen_pair(
     >         sc, a1, z1, a2, z2, screening_mode,
     >         zg1, zg2, zg3, zg4, zs13, zhat, zhat2, lzav, aznut, zs13inv,
     >         theta_e_for_graboske_et_al, graboske_cache, scor, scordt, scordd, ierr)
         use screen_graboske, only: graboske_et_al_screening
         use screen5, only: fxt_screen5
         type (Screen_Info), pointer :: sc ! previously setup 
         real*8, intent(in) :: a1, z1, a2, z2
         integer, intent(in) :: screening_mode ! see screen_def.
         real*8 :: theta_e_for_graboske_et_al ! not used for fxt screening.
         ! cached info
         real*8, intent(in) :: zg1, zg2, zg3, zg4
         real*8, intent(in) :: zs13, zhat, zhat2, lzav, aznut, zs13inv
         real*8, pointer :: graboske_cache(:,:,:) ! (3,max_z_to_cache,max_z_to_cache)
         ! outputs
         real*8, intent(out) :: scor ! screening factor
         real*8, intent(out) :: scordt ! partial wrt temperature
         real*8, intent(out) :: scordd ! partial wrt density
         integer, intent(out) :: ierr
         if (screening_mode == classic_screening) then
            call graboske_et_al_screening(
     >         sc, zg1, zg2, zg3, zg4, 
     >         theta_e_for_graboske_et_al, graboske_cache,
     >         a1, z1, a2, z2, scor, scordt, scordd, ierr)
         else if (screening_mode == extended_screening) then
            call fxt_screen5(
     >         sc, zs13, zhat, zhat2, lzav, aznut, zs13inv, 
     >         a1, z1, a2, z2, scor, scordt, scordd, ierr)
         else if (screening_mode == salpeter_screening) then
            call eval_salpeter_screening(sc, z1, z2, scor, scordt, scordd, ierr)
         else if (screening_mode == no_screening) then
            scor = 1; scordt = 0; scordd = 0
         else
            ierr = -1
            write(*,*) 'screen_pair: unknown value for screening_mode', screening_mode
         end if
      end subroutine screen_pair
      
      
      subroutine eval_salpeter_screening(sc, z1, z2, scor, scordt, scordd, ierr)
         ! weak screening only.  following Salpeter (1954),
         ! with equations (4-215) and (4-221) of Clayton (1968).
         type (Screen_Info), pointer :: sc ! previously setup 
         real*8, intent(in) :: z1, z2
         real*8, intent(out) :: scor ! screening factor
         real*8, intent(out) :: scordt ! partial wrt temperature
         real*8, intent(out) :: scordd ! partial wrt density
         integer, intent(out) :: ierr
         real*8 :: zeta, lnf, rho, T, dlnf_dd, dlnf_dt
         ierr = 0
         rho = sc% den
         T = sc% temp
         zeta = (sc% z2bar + sc% zbar) / sc% abar
         lnf = 1.88d8*z1*z2*sqrt(rho*zeta/T**3)
         dlnf_dd = lnf/(2*rho)
         dlnf_dt = -lnf*3/(2*T)
         scor = exp(lnf)
         scordd = scor*dlnf_dd
         scordt = scor*dlnf_dt
      end subroutine eval_salpeter_screening
      
      
      integer function screening_option(which_screening_option, ierr)
         use screen_def
         use utils_lib, only: StrLowCase
         character (len=*), intent(in) :: which_screening_option
         integer, intent(out) :: ierr
         character (len=64) :: option
         ierr = 0
         option = StrLowCase(which_screening_option)
         
         if (option == 'no_screening' .or. len_trim(option) == 0) then
            screening_option = no_screening
            
         else if (option == 'extended') then
            screening_option = extended_screening
            
         else if (option == 'classic') then
            screening_option = classic_screening
            
         else if (option == 'salpeter') then
            screening_option = salpeter_screening
            
         else
            ierr = -1
            screening_option = -1
         end if 
      end function screening_option
      
      
      subroutine screening_option_str(which_screening_option, screening_option, ierr)
         use screen_def
         integer, intent(in) :: which_screening_option
         character (len=*), intent(out) :: screening_option
         integer, intent(out) :: ierr
         ierr = 0
         
         if (which_screening_option == no_screening) then
            screening_option = 'no_screening'
            
         else if (which_screening_option == classic_screening) then
            screening_option = 'classic'
            
         else if (which_screening_option == extended_screening) then
            screening_option = 'extended'
            
         else if (which_screening_option == salpeter_screening) then
            screening_option = 'salpeter'
            
         else
            ierr = -1
            screening_option = ''
         end if 
      end subroutine screening_option_str
      

      end module screen_lib

