! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, 
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful, 
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
 
      module reaclib_def

      use chem_def, only: iso_name_length, nuclide_data, npart
      use utils_def, only: integer_dict
      
      implicit none




      integer, parameter :: max_nreaclib=80000
      integer, parameter :: max_species_per_reaction=6
      integer, parameter :: ncoefficients=7
      integer, parameter :: nchapters=11
      integer, parameter :: ninverse_coeff = 2
      integer, parameter :: max_terms_per_rate = 20
      integer, parameter :: max_id_length = 36


      ! i/o parameters
      integer, parameter :: internal_format = 0
      integer, parameter :: pretty_print_format = 1
      integer, parameter :: short_format = 2


      ! flags for reaction types -- values are chapter numbers
      integer, parameter :: &
         r_one_one = 1, &
         r_one_two = 2, &
         r_one_three = 3, &
         r_two_one = 4, &
         r_two_two = 5, &
         r_two_three = 6, &
         r_two_four = 7, &
         r_three_one = 8, &
         r_three_two = 9, &
         r_four_two = 10, &
         r_one_four = 11


      integer, dimension(nchapters) :: Nin = (/1, 1, 1, 2, 2, 2, 2, 3, 3, 4, 1/)
      integer, dimension(nchapters) :: Nout = (/1, 2, 3, 1, 2, 3, 4, 1, 2, 2, 4/)


      type reaclib_data
      	integer, dimension(:), pointer :: chapter
      	character(len=iso_name_length), dimension(:, :), pointer :: species
      	character(len=iso_name_length), dimension(:), pointer :: label
      	character, dimension(:), pointer :: reaction_flag
      	character, dimension(:), pointer :: reverse_flag
      	double precision, dimension(:), pointer :: Qvalue
      	double precision, dimension(:, :), pointer :: coefficients
      end type reaclib_data


      type reaction_data
      
      	integer :: nreactions
      	integer :: nchapters_included
      	integer, dimension(nchapters) :: chapters_present
      	type(nuclide_data), pointer :: nuclides
      	
      	integer :: num_from_weaklib
      	integer, dimension(:), pointer :: weaklib_ids ! (num_from_weaklib)
      	
      	integer, dimension(2, nchapters) :: bookmarks
      	character(len=max_id_length), dimension(:), pointer :: reaction_handle, reverse_handle
      	integer, dimension(:), pointer :: category
      	integer, dimension(:), pointer :: chapter
      	integer, dimension(:, :), pointer :: pspecies
      	character, dimension(:), pointer :: reaction_flag
      	double precision, dimension(:), pointer :: weight
      	double precision, dimension(:), pointer :: weight_reverse
      	double precision, dimension(:, :), pointer :: coefficients
      	double precision, dimension(:), pointer :: weak_mask
      	double precision, dimension(:, :), pointer :: inverse_coefficients
      	integer, dimension(:), pointer :: inverse_exp
      	double precision, dimension(:, :), pointer :: inverse_part
         double precision, dimension(:), pointer :: Q
         double precision, dimension(:), pointer :: Qneu
         type (integer_dict), pointer :: reaction_dict, reverse_dict
         
      end type reaction_data
      
      
      character (len=256) :: reaclib_dir
      
   	type(reaclib_data) :: reaclib
   	integer :: nreaclib
      

      contains
      
      
      subroutine reaclib_def_init(data_dir)
         character (*), intent(in) :: data_dir
         reaclib_dir = trim(data_dir) // '/rates_data'
      end subroutine reaclib_def_init
      
      
      subroutine allocate_reaclib_data(r, n, ierr)
      	type(reaclib_data), intent(inout) :: r
      	integer, intent(in) :: n
      	integer, intent(out) :: ierr
      	ierr = 0
      	allocate( &
      	   r% chapter(n), r% species(max_species_per_reaction, n), r% label(n), r% reaction_flag(n), &
      		r% reverse_flag(n), r% Qvalue(n), r% coefficients(ncoefficients, n), stat=ierr)
      end subroutine allocate_reaclib_data
      

      subroutine free_reaclib_data(reaclib, ierr)
      	use alert_lib
      	type(reaclib_data), intent(inout) :: reaclib
      	integer, intent(out) :: ierr
      	ierr = 0
      	if (associated(reaclib% chapter)) & 
      		deallocate( &
      		   reaclib% chapter, reaclib% species, reaclib% label, reaclib% reaction_flag, &
      		   reaclib% reverse_flag, reaclib% Qvalue, reaclib% coefficients, stat=ierr)
      	if (ierr /= 0) call alert(ierr, "free_reaclib: Deallocation request denied")
      end subroutine free_reaclib_data
      

      subroutine allocate_reaction_data(r, n, nweak, ierr)
      	type(reaction_data), intent(out) :: r
      	integer, intent(in) :: n ! number of rates
      	integer, intent(in) :: nweak ! number of weaklib rates
      	integer, intent(out) :: ierr
      	allocate( &
      	   r% reaction_handle(n), r% reverse_handle(n), r% category(n), r% chapter(n), &
      	   r% weaklib_ids(nweak), r% pspecies(max_species_per_reaction, n), r% reaction_flag(n), &
      		r% weight(n), r% weight_reverse(n), r% coefficients(ncoefficients, n), &
      		r% weak_mask(n), r% inverse_coefficients(ninverse_coeff, n), &
      		r% inverse_exp(n), r% inverse_part(npart, n), r% Q(n), r% Qneu(n), stat=ierr)
      	nullify(r% reaction_dict)
      	nullify(r% reverse_dict)
      end subroutine allocate_reaction_data
      

      subroutine free_reaction_data(r)
         use utils_lib, only: integer_dict_free
      	type(reaction_data), intent(inout) :: r
      	if (associated(r% chapter)) then
      		deallocate( &
      		   r% reaction_handle, r% reverse_handle, r% category, r% chapter, r% weaklib_ids, r% pspecies, &
      		   r% reaction_flag, r% weight, r% weight_reverse, r% coefficients, r% weak_mask, &
      		   r% inverse_coefficients, r% inverse_exp, r% inverse_part, r% Q, r% Qneu)
      		if (associated(r% reaction_dict)) call integer_dict_free(r% reaction_dict)
      		if (associated(r% reverse_dict)) call integer_dict_free(r% reverse_dict)
      	end if
      end subroutine free_reaction_data
      

      end module reaclib_def
