! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
 
      module reaclib_eval
      use reaclib_def
      
      implicit none

      double precision, parameter :: lam_max = 1d99, ln1_max = log(lam_max)


      contains
      
      
      subroutine reaction_rates( &
            lo, hi, T9, rates, nuclides, forward_only, &
            lambda, dlambda_dlnT, &
            rlambda, drlambda_dlnT, &
            ierr)
      	integer, intent(in) :: lo, hi
      	double precision, intent(in) :: T9
      	type(reaction_data), intent(in) :: rates
      	type(nuclide_data), intent(in) :: nuclides
      	logical, intent(in) :: forward_only
      	double precision, intent(out) :: lambda, dlambda_dlnT
      	double precision, intent(out) :: rlambda, drlambda_dlnT
      	integer, intent(out) :: ierr
         
         integer :: num_lambdas
      	double precision, dimension(:), pointer :: &
      	   ln_lambdas, lambdas, dlambdas_dlnT, &
      	   ln_rlambdas, rlambdas, drlambdas_dlnT
         
         include 'formats.dek'
         
         ierr = 0

         num_lambdas = size(rates% coefficients,2)
         allocate( &
            ln_lambdas(num_lambdas), lambdas(num_lambdas), dlambdas_dlnT(num_lambdas), &
            ln_rlambdas(num_lambdas), rlambdas(num_lambdas), drlambdas_dlnT(num_lambdas))
         
         call compute_some_lambdas(lo, hi, T9, rates, ln_lambdas, lambdas, dlambdas_dlnT)
         lambda = sum(lambdas(lo:hi))
         dlambda_dlnT = sum(dlambdas_dlnT(lo:hi))
         
         if (forward_only) then
            rlambda = 0
            drlambda_dlnT = 0
         else
            call compute_some_inverse_lambdas( &
               lo, hi, T9, rates, &
               ln_lambdas, lambdas, dlambdas_dlnT, &
               rlambdas, drlambdas_dlnT)
            rlambda = sum(rlambdas(lo:hi))
            drlambda_dlnT = sum(drlambdas_dlnT(lo:hi))
         end if
         
         deallocate(ln_lambdas, lambdas, dlambdas_dlnT)
         deallocate(ln_rlambdas, rlambdas, drlambdas_dlnT)         

      end subroutine reaction_rates


      subroutine compute_lambda(T9,rates,ln_lambda,lambda,dlambda_dlnT)
      	double precision, intent(in) :: T9
      	type(reaction_data), intent(in) :: rates
      	double precision, dimension(size(rates% coefficients,2)), intent(out) :: ln_lambda, lambda, dlambda_dlnT
         call compute_some_lambdas( &
            rates% num_from_weaklib + 1, rates% nreactions, &
            T9, rates, ln_lambda, lambda, dlambda_dlnT)
      end subroutine compute_lambda
      

      subroutine compute_some_lambdas(lo, hi, T9, rates, ln_lambda, lambda, dlambda_dlnT)
         use utils_lib, only: is_bad_num
         integer, intent(in) :: lo, hi ! range of rates to do
      	double precision, intent(in) :: T9
      	type(reaction_data), intent(in) :: rates
      	double precision, dimension(:), intent(out) :: ln_lambda, lambda, dlambda_dlnT
      	
      	double precision, parameter :: onethird = 1d0/3d0, fivethird = 5d0/3d0, &
      	   twothird = 2d0/3d0, fourthird = 4d0/3d0
      	double precision :: T9inv, logT, ln1
      	double precision, dimension(7) :: T9fac, dT9fac_dT9, dT9fac_dlnT
      	integer :: i, j
      	
      	include 'formats.dek'

      	T9inv = 1d0/T9
      	
      	T9fac(1) = 1d0
      	dT9fac_dT9(1) = 0d0
      	
      	T9fac(2) = T9inv
      	dT9fac_dT9(2) = -T9inv*T9inv
      	
      	T9fac(3) = T9inv**onethird
      	dT9fac_dT9(3) = -onethird*T9inv**fourthird
      	
      	T9fac(4) = T9**onethird
      	dT9fac_dT9(4) = onethird*T9inv**twothird
      	
      	T9fac(5) = T9
      	dT9fac_dT9(5) = 1d0
      	
      	T9fac(6) = T9**fivethird
      	dT9fac_dT9(6) = fivethird*T9**twothird
      	
      	T9fac(7) = log(T9)
      	dT9fac_dT9(7) = T9inv
      	
      	dT9fac_dlnT = T9*dT9fac_dT9
      	
!$OMP PARALLEL DO PRIVATE(i, ln1)
      	do i = lo, hi
      	   ln1 = dot_product(T9fac(:), rates% coefficients(:,i))
      	   if (ln1 > ln1_max) then
      	      ln_lambda(i) = ln1_max
      	      lambda(i) = exp(ln1_max)
      	      dlambda_dlnT(i) = 0
      	   else
      	      ln_lambda(i) = ln1
         	   lambda(i) = exp(ln_lambda(i))
         	   dlambda_dlnT(i) = &
         	      dot_product(dT9fac_dlnT(:), rates% coefficients(:,i))*lambda(i)
      	   end if
      	end do
!$OMP END PARALLEL DO
      	
      end subroutine compute_some_lambdas
      

      subroutine compute_inverse_lambda(T9,rates, &
            ln_lambda,lambda,dlambda_dlnT,inv_lambda,dinv_lambda_dlnT)
         use utils_lib, only: is_bad_num
      	use chem_def, only: Tpart, npart
      	use chem_lib, only: get_partition_fcn_indx
      	double precision, intent(in) :: T9
      	type(reaction_data), intent(in) :: rates
      	double precision, dimension(:), intent(in) :: ln_lambda, lambda, dlambda_dlnT
      	double precision, dimension(:), intent(out) :: inv_lambda, dinv_lambda_dlnT
         call compute_some_inverse_lambdas( &
            rates% num_from_weaklib + 1, rates% nreactions, &
            T9, rates, ln_lambda, lambda, dlambda_dlnT, inv_lambda, dinv_lambda_dlnT)
      end subroutine compute_inverse_lambda
      

      subroutine compute_some_inverse_lambdas( &
            lo, hi, T9, rates, &
            ln_lambda, lambda, dlambda_dlnT, &
            inv_lambda, dinv_lambda_dlnT)
         use utils_lib, only: is_bad_num
      	use chem_def, only: Tpart, npart
      	use chem_lib, only: get_partition_fcn_indx
         integer, intent(in) :: lo, hi ! range of rates to do
      	double precision, intent(in) :: T9
      	type(reaction_data), intent(in) :: rates
      	double precision, dimension(:), intent(in) :: ln_lambda, lambda, dlambda_dlnT
      	double precision, dimension(:), intent(out) :: inv_lambda, dinv_lambda_dlnT
      	
      	integer :: indx,indxp
      	integer :: rstart, rend, i, j
      	double precision, dimension(size(lambda)) :: A, Qratio, dQratio_dlnT
      	double precision :: tfac, dtfac_dlnT, lnT9, T9i, dT9i_dlnT, ln1, fac1, dfac1_dlnT, dln1_dlnT
      	
      	include 'formats.dek'
	
      	! find index of partition function and logarithmically interpolate
      	indx = get_partition_fcn_indx(T9)
      	if (indx >= npart) indx = npart-1
      	if (indx < 1) then
      		Qratio(lo:hi) = 1d0
      		dQratio_dlnT(lo:hi) = 0d0
      	else
      		indxp = indx+1
      		tfac = (T9-Tpart(indx))/(Tpart(indxp)-Tpart(indx))
      		dtfac_dlnT = T9/(Tpart(indxp)-Tpart(indx))
!x$OMP PARALLEL DO PRIVATE(i)
      		do i = lo, hi
         		A(i) = (rates% inverse_part(indx,i))* &
         		   (rates% inverse_part(indxp,i)/rates% inverse_part(indx,i))
         		Qratio(i) = A(i)**tfac
         		dQratio_dlnT(i) = Qratio(i)*log(A(i))*dtfac_dlnT
      		end do
!x$OMP END PARALLEL DO
      	end if
		
      	lnT9 = log(T9)
      	T9i = 1d0/T9
      	dT9i_dlnT = -T9i
      	
!x$OMP PARALLEL DO PRIVATE(i, ln1, fac1, dln1_dlnT, dfac1_dlnT)
      	do i = lo, hi
      	
         	ln1 = ln_lambda(i) + &
         	      rates% inverse_coefficients(1,i) + &
         			rates% inverse_coefficients(2,i)*T9i + &
         	      1.5*rates% inverse_exp(i)*lnT9
         	      
         	if (ln1 < ln1_max) then
         		fac1 = exp(ln1)
         		
            	dln1_dlnT = dlambda_dlnT(i)/max(1d-99,lambda(i)) + &
            	      rates% inverse_coefficients(2,i)*dT9i_dlnT + &
            	      1.5*rates% inverse_exp(i)
            	          	
         		dfac1_dlnT = dln1_dlnT*fac1
         		
         	else
         	   ln1 = ln1_max
         	   fac1 = exp(ln1_max)
         	   dln1_dlnT = 0
         	   dfac1_dlnT = 0
         	end if
         	inv_lambda(i) = fac1*Qratio(i)
         	if (lambda(i) < 1d-99) then
         	   dinv_lambda_dlnT(i) = 0
         	   cycle
         	end if
         	dinv_lambda_dlnT(i) = dfac1_dlnT*Qratio(i) + fac1*dQratio_dlnT(i)

      		
      		if (.false. .and. rates% reaction_handle(i) == 'r_na23_pa_ne20') then
      		   write(*,*)
      		   write(*,1) 'T9', T9
      		   write(*,1) 'r_n14_an_f17 lambda(i)', lambda(i)
      		   write(*,1) 'dlambda_dlnT(i)/lambda(i)', dlambda_dlnT(i)/lambda(i)
      		   write(*,1) 'dT9i_dlnT', dT9i_dlnT
      		   write(*,2) 'r..(2,i)', i, rates% inverse_coefficients(2,i)
      		   write(*,1) 'r..(2,i)*dT9i_dlnT', rates% inverse_coefficients(2,i)*dT9i_dlnT
      		   write(*,1) '1.5*rates% inverse_exp(i)', 1.5*rates% inverse_exp(i)
      		   write(*,1) 'ln1', ln1
      		   write(*,1) 'ln1_max', ln1_max
      		   write(*,1) 'r..(1,i)', rates% inverse_coefficients(1,i)
      		   write(*,1) 'r..(2,i)*T9i', rates% inverse_coefficients(2,i)*T9i
      		   write(*,1) '1.5*r..(i)*lnT9', 1.5*rates% inverse_exp(i)*lnT9
      		   write(*,*)
      		   write(*,1) 'fac1', fac1
      		   write(*,1) 'dln1_dlnT', dln1_dlnT
      		   write(*,1) 'dfac1_dlnT', dfac1_dlnT
      		   write(*,*)
      		   write(*,1) 'Qratio(i)', Qratio(i)
      		   write(*,1) 'dQratio_dlnT(i)', dQratio_dlnT(i)
      		   write(*,1) 'dfac1_dlnT*Qratio(i)', dfac1_dlnT*Qratio(i)
      		   write(*,1) 'fac1*dQratio_dlnT(i)', fac1*dQratio_dlnT(i)
      		   write(*,1) 'dinv_lambda_dlnT(i)', dinv_lambda_dlnT(i)
      		   write(*,1) 'inv_lambda', inv_lambda(i)
      		   write(*,*)
      		   stop 'compute_some_inverse_lambdas'
      		end if

      	end do
!x$OMP END PARALLEL DO
      	
      end subroutine compute_some_inverse_lambdas
      


      end module reaclib_eval
