! ***********************************************************************
!
!  Copyright (C) 2010  Bill Paxton
!
!  MESA is free software; you can use it and/or modify
!  it under the combined terms and restrictions of the MESA MANIFESTO
!  and the GNU General Library Public License as published
!  by the Free Software Foundation; either version 2 of the License,
!  or (at your option) any later version.
!
!  You should have received a copy of the MESA MANIFESTO along with
!  this software; if not, it is available at the mesa website:
!  http://mesa.sourceforge.net/
!
!  MESA is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!  See the GNU Library General Public License for more details.
!
!  You should have received a copy of the GNU Library General Public License
!  along with this software; if not, write to the Free Software
!  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module rates_def
      use utils_def, only: integer_dict
      use reaclib_def, only: reaction_data
      
      
      implicit none

      ! reactions
               
         integer, parameter :: ir_h1_h1_wk_h2 = 1 ! p(p, e+nu)h2
         integer, parameter :: irpp_to_he3 = ir_h1_h1_wk_h2+1 ! p(p, e+nu)h2(p, g)he3
         integer, parameter :: ir_h1_h1_ec_h2 = irpp_to_he3+1  ! p(e-p, nu)h2
         integer, parameter :: irpep_to_he3 = ir_h1_h1_ec_h2+1  ! p(e-p, nu)h2(p, g)he3
         integer, parameter :: ir_h2_h2_to_he4 = irpep_to_he3+1 ! h2(h2, g)he4
         integer, parameter :: ir_h1_he3_wk_he4 = ir_h2_h2_to_he4+1 ! he3(p, e+nu)he4
         integer, parameter :: ir_h2_he3_to_h1_he4 = ir_h1_he3_wk_he4+1 ! he3(d, p)he4
         integer, parameter :: ir_he3_he3_to_h1_h1_he4 = ir_h2_he3_to_h1_he4+1 ! he3(he3, 2p)he4
         integer, parameter :: ir_h1_h1_he4_to_he3_he3 = ir_he3_he3_to_h1_h1_he4+1 ! he4(2p, he3)he3
         integer, parameter :: ir_he3_ag_be7 = ir_h1_h1_he4_to_he3_he3+1 ! he3(he4, g)be7
         integer, parameter :: ir34_pp2 = ir_he3_ag_be7+1 ! he4(he3, g)be7(e-, nu)li7(p, a)he4
         integer, parameter :: ir34_pp3 = ir34_pp2+1 ! he4(he3, g)be7(p, g)b8(e+, nu)be8(, a)he4
         integer, parameter :: ir_he4_he4_he4_to_c12 = ir34_pp3+1 ! triple alpha to c12 
         integer, parameter :: ir_c12_to_he4_he4_he4 = ir_he4_he4_he4_to_c12+1 ! c12 to 3 alpha
         integer, parameter :: ir_li7_pa_he4 = ir_c12_to_he4_he4_he4+1 ! li7(p, a)he4
         integer, parameter :: ir_be7_ec_li7 = ir_li7_pa_he4+1 ! be7(e-, nu)li7
         integer, parameter :: irbe7ec_li7_aux = ir_be7_ec_li7+1 ! be7(e-, nu)li7(p, a)he4 
         integer, parameter :: ir_h2_be7_to_h1_he4_he4 = irbe7ec_li7_aux+1 ! be7(d, p)2he4
         integer, parameter :: ir_he3_be7_to_h1_h1_he4_he4 = ir_h2_be7_to_h1_he4_he4+1 ! be7(he3, 2p)2he4
         integer, parameter :: irbe7pg_b8_aux = ir_he3_be7_to_h1_h1_he4_he4+1 ! be7(p, g)b8(e+, nu)be8(, a)he4
         integer, parameter :: ir_b8_wk_he4_he4 = irbe7pg_b8_aux+1 ! b8(p=>n)be8=>2 he4
         integer, parameter :: ir_b8_gp_be7 = ir_b8_wk_he4_he4+1 ! be7(p, g)b8
         integer, parameter :: irc12_to_c13 = ir_b8_gp_be7+1 ! c12(p, g)n13(e+nu)c13
         integer, parameter :: irc12_to_n14 = irc12_to_c13+1 ! c12(p, g)n13(e+nu)c13(p, g)n14
         integer, parameter :: ir_c12_ap_n15 = irc12_to_n14+1 ! c12(a, p)n15
         integer, parameter :: irc12ap_to_o16 = ir_c12_ap_n15+1 ! c12(a, p)n15(p, g)o16
         integer, parameter :: irc12ng_to_o16 = irc12ap_to_o16+1 ! c12(n, g)c13(a, n)o16
         integer, parameter :: ir_c12_ag_o16 = irc12ng_to_o16+1
         integer, parameter :: ir1212a = ir_c12_ag_o16+1 ! c12(c12, a)ne20
         integer, parameter :: ir1212p = ir1212a+1 ! c12(c12, p)na23
         integer, parameter :: ir1212_to_ne20 = ir1212p+1 ! c12 + c12 -> ne20 + he4
         integer, parameter :: ir1216_to_mg24 = ir1212_to_ne20+1 ! ! c12 + o16 -> mg24 + he4
         integer, parameter :: ir1216_to_si28 = ir1216_to_mg24+1 ! ! c12 + o16 -> si28
         integer, parameter :: ir_c13_an_o16 = ir1216_to_si28+1 ! c13(a, n)o16
         integer, parameter :: ir_n13_ap_o16 = ir_c13_an_o16+1 ! n13(a, p)o16
         integer, parameter :: irn13pg_to_n14 = ir_n13_ap_o16+1 ! n13(p, g)o14(e+nu)n14
         integer, parameter :: ir_n13_gp_c12 = irn13pg_to_n14+1 ! n13(g, p)c12
         integer, parameter :: irn14pg_aux = ir_n13_gp_c12+1 ! n14(p, g)o15
         integer, parameter :: irn14_to_n15 = irn14pg_aux+1 ! n14(p, g)o15(e+nu)n15
         integer, parameter :: irn14_to_o16 = irn14_to_n15+1 ! n14(p, g)o15(e+nu)n15(p, g)o16
         integer, parameter :: irn14_to_c12 = irn14_to_o16+1 ! n14(p, g)o15(e+nu)n15(p, a)c12
         integer, parameter :: ir_n14_ap_o17 = irn14_to_c12+1 ! n14(a, p)o17
         integer, parameter :: irn14ecap = ir_n14_ap_o17+1 ! n14(e-, nu)c14
         integer, parameter :: ir_n14_ag_f18 = irn14ecap+1 ! n14(a, g)f18
         integer, parameter :: irn14ag_to_o18 = ir_n14_ag_f18+1 ! n14(a, g)f18(e+nu)o18
         integer, parameter :: irn14ag_to_ne22 = irn14ag_to_o18+1 ! n14(a, g)f18(e+nu)o18(a, g)ne22
         integer, parameter :: irn14ag_lite = irn14ag_to_ne22+1 ! n14 + 1.5 alpha => ne20
         integer, parameter :: ir_n14_gp_c13 = irn14ag_lite+1 ! n14(g, p)c13
         integer, parameter :: irn15pg_aux = ir_n14_gp_c13+1 ! n15(p, g)o16
         integer, parameter :: ir_n15_pa_c12 = irn15pg_aux+1 ! n15(p, a)c12
         integer, parameter :: irn15pa_aux = ir_n15_pa_c12+1 ! n15(p, a)c12
         integer, parameter :: ir_n15_ap_o18 = irn15pa_aux+1 ! n15(a, p)o18
         integer, parameter :: ir_o14_ag_ne18 =ir_n15_ap_o18+1 ! o14(a, g)ne18
         integer, parameter :: ir_o14_ap_f17 = ir_o14_ag_ne18+1 ! o14(a, p)f17
         integer, parameter :: ir_o14_gp_n13 = ir_o14_ap_f17+1 ! o14(g, p)n13
         integer, parameter :: ir_o15_ap_f18 = ir_o14_gp_n13+1 ! o15(a, p)f18
         integer, parameter :: ir_o15_ag_ne19 = ir_o15_ap_f18+1 ! o15(a, g)ne19 
         integer, parameter :: ir_o15_gp_n14 = ir_o15_ag_ne19+1 ! o15(g, p)n14
         integer, parameter :: iro16_to_o17 = ir_o15_gp_n14+1 ! o16(p, g)f17(e+nu)o17
         integer, parameter :: iro16_to_n14 = iro16_to_o17+1 ! o16(p, g)f17(e+nu)o17(p, a)n14
         integer, parameter :: ir_o16_ap_f19 = iro16_to_n14+1 ! o16(a, p)f19
         integer, parameter :: iro16ap_aux = ir_o16_ap_f19+1 ! o16(a, p)f19
         integer, parameter :: iro16ap_to_ne20 = iro16ap_aux+1 ! o16(a, p)f19(p, a)ne20
         integer, parameter :: ir_o16_ag_ne20 = iro16ap_to_ne20+1 ! o16(a, g)ne20
         integer, parameter :: ir_o16_gp_n15 = ir_o16_ag_ne20+1 ! o16(g, p)n15
         integer, parameter :: iro16gp_aux = ir_o16_gp_n15+1 ! o16(g, p)n15 
         integer, parameter :: iro16gp_to_c12 = iro16gp_aux+1 ! o16(g, p)n15(p, a)c12
         integer, parameter :: ir_o16_ga_c12 = iro16gp_to_c12+1 ! o16(g, a)c12
         integer, parameter :: iro16ng_to_mg24 = ir_o16_ga_c12+1 ! o16(n, g)o17(a, g)ne21(a, n)mg24
         integer, parameter :: ir1616 = iro16ng_to_mg24+1 ! o16 + o16 -> si28 + he4
         integer, parameter :: ir1616a = ir1616+1 ! o16(o16, a)si28 
         integer, parameter :: ir1616ppa = ir1616a+1 ! o16(o16, p)p31(p, a)si28 
         integer, parameter :: ir1616ppg = ir1616ppa+1 ! o16(o16, p)p31(p, g)s32 
         integer, parameter :: ir1616p = ir1616ppg+1 ! o16(o16, p)p31 
         integer, parameter :: ir1616p_aux = ir1616p+1 ! o16(o16, p)p31
         integer, parameter :: ir1616g = ir1616p_aux+1 ! o16(o16, g)s32 
         integer, parameter :: ir_o17_pa_n14 = ir1616g+1 ! o17(p, a)n14
         integer, parameter :: ir_o17_ag_ne21 = ir_o17_pa_n14+1 ! o17(a, g)ne21
         integer, parameter :: iro17_to_o18 = ir_o17_ag_ne21+1 ! o17(p, g)f18(e+nu)o18
         integer, parameter :: ir_o18_pa_n15 = iro17_to_o18+1 ! o18(p, a)n15 and n15(a, p)o18
         integer, parameter :: ir_o18_ag_ne22 = ir_o18_pa_n15+1 ! o18(a, g)ne22 
         integer, parameter :: ir_f17_pa_o14 = ir_o18_ag_ne22+1 ! f17(p, a)o14
         integer, parameter :: ir_f17_ap_ne20 = ir_f17_pa_o14+1 ! f17(a, p)ne20
         integer, parameter :: ir_f17_gp_o16 = ir_f17_ap_ne20+1 ! f17(g, p)o16
         integer, parameter :: ir_f18_ap_ne21 = ir_f17_gp_o16+1 ! f18(a, p)ne21
         integer, parameter :: ir_f18_pa_o15 = ir_f18_ap_ne21+1 ! f18(p, a)o15
         integer, parameter :: ir_f18_gp_o17 = ir_f18_pa_o15+1 ! f18(g, p)o17
         integer, parameter :: irf19pg_aux = ir_f18_gp_o17+1 ! f19(p, g)ne20
         integer, parameter :: ir_f19_pa_o16 = irf19pg_aux+1 ! f19(p, a)o16
         integer, parameter :: irf19pa_aux = ir_f19_pa_o16+1 ! f19(p, a)o16
         integer, parameter :: ir_f19_ap_ne22 = irf19pa_aux+1 ! f19(a, p)ne22
         integer, parameter :: ir_f19_gp_o18 = ir_f19_ap_ne22+1 ! f19(g, p)o18
         integer, parameter :: irne18ap_to_mg22 = ir_f19_gp_o18+1 ! ne18(a, p)na21(p, g)mg22
         integer, parameter :: irne18ap_to_mg24 = irne18ap_to_mg22+1 ! ne18(a, p)na21(p, g)mg22 -> mg24
         integer, parameter :: irne18ag_to_mg24 = irne18ap_to_mg24+1 ! ne18(a, g)mg22 -> mg24
         integer, parameter :: irne19pg_to_mg21 = irne18ag_to_mg24+1 ! ne19(p, g)na20(p, g)mg21
         integer, parameter :: irne19pg_to_mg22 = irne19pg_to_mg21+1 ! ne19(p, g)na20(p, g)mg21(e+nu)na21(p, g)mg22 
         integer, parameter :: irne19pg_to_mg24 = irne19pg_to_mg22+1 ! ne19(p, g)na20(p, g)mg21(e+nu)na21(p, g)mg22 -> mg24
         integer, parameter :: ir_ne19_ga_o15 = irne19pg_to_mg24+1 ! ne19(g, a)o15 
         integer, parameter :: irne20pg_to_ne21 = ir_ne19_ga_o15+1 ! ne20(p, g)na21(e+nu)ne21
         integer, parameter :: irne20pg_to_mg22 = irne20pg_to_ne21+1 ! ne20(p, g)na21(p, g)mg22 
         integer, parameter :: irne20pg_to_mg24 = irne20pg_to_mg22+1 ! ne20(p, g)na21(p, g)mg22 -> mg24
         integer, parameter :: ir_ne20_ag_mg24 = irne20pg_to_mg24+1 ! ne20(a, g)mg24
         integer, parameter :: ir_ne20_ga_o16 = ir_ne20_ag_mg24+1 ! ne20(g, a)o16
         integer, parameter :: ir_ne20_gp_f19 = ir_ne20_ga_o16+1 ! ne20(g, p)f19
         integer, parameter :: irne20gp_aux = ir_ne20_gp_f19+1 ! ne20(g, p)f19
         integer, parameter :: irne20gp_to_o16 = irne20gp_aux+1 ! ne20(g, p)f19(p, a)o16
         integer, parameter :: ir_ne20_ap_na23 = irne20gp_to_o16+1 ! ne20(a, p)na23
         integer, parameter :: irne20ap_aux = ir_ne20_ap_na23+1 ! ne20(a, p)na23
         integer, parameter :: irne20ap_to_mg24 = irne20ap_aux+1 ! ne20(a, p)na23(p, g)mg24
         integer, parameter :: irne20ng_to_mg24 = irne20ap_to_mg24+1 ! ne20(n, g)ne21(a, n)mg24
         integer, parameter :: ir_ne22_ag_mg26 = irne20ng_to_mg24+1 ! ne22(a, g)mg26
         integer, parameter :: ir_ne22_ga_o18 = ir_ne22_ag_mg26+1 ! ne22(g, a)o18
         integer, parameter :: irne22ag_fake = ir_ne22_ga_o18+1 ! ne22(0.5a, g)mg24
         integer, parameter :: irne22ga_fake = irne22ag_fake+1 ! ne22(g, 0.5a)ne20
         integer, parameter :: irna22pg_to_na23 = irne22ga_fake+1 ! na22(p, g)mg23(e+nu)na23
         integer, parameter :: ir_na23_gp_ne22 = irna22pg_to_na23+1 ! na23(g, p)ne22
         integer, parameter :: ir_na23_ap_mg26 = ir_na23_gp_ne22+1 ! na23(a, p)mg26 
         integer, parameter :: irna23ap_aux = ir_na23_ap_mg26+1 ! na23(a, p)mg26 
         integer, parameter :: ir_na23_pa_ne20 = irna23ap_aux+1 ! na23(p, a)ne20 
         integer, parameter :: irna23pa_aux = ir_na23_pa_ne20+1 ! na23(p, a)ne20 
         integer, parameter :: irna23pg_aux = irna23pa_aux+1 ! na23(p, g)mg24 
         integer, parameter :: ir_mg24_ap_al27 = irna23pg_aux+1 ! mg24(a, p)al27 
         integer, parameter :: irmg24ap_aux = ir_mg24_ap_al27+1 ! mg24(a, p)al27 
         integer, parameter :: irmg24ap_to_si28 = irmg24ap_aux+1 
         integer, parameter :: irmg24pg_to_mg25 = irmg24ap_to_si28+1 ! mg24(p, g)al25(e+nu)mg25 
         integer, parameter :: ir_mg24_ag_si28 = irmg24pg_to_mg25+1 ! mg24(a, g)si28 
         integer, parameter :: ir_mg24_ga_ne20 = ir_mg24_ag_si28+1 ! mg24(g, a)ne20
         integer, parameter :: ir_mg24_gp_na23 = ir_mg24_ga_ne20+1 ! mg24(g, p)na23 
         integer, parameter :: irmg24gp_aux = ir_mg24_gp_na23+1 ! mg24(g, p)na23 
         integer, parameter :: irmg24gp_to_ne20 = irmg24gp_aux+1 ! mg24(g, p)na23(p, a)ne20 
         integer, parameter :: irmg25pg_to_mg26 = irmg24gp_to_ne20+1 ! mg25(p, g)al26(beta)mg26
         integer, parameter :: ir_mg26_ag_si30 = irmg25pg_to_mg26+1 ! mg26(a, g)si30
         integer, parameter :: iral27pg_aux = ir_mg26_ag_si30+1 ! al27(p, g)si28
         integer, parameter :: ir_al27_gp_mg26 = iral27pg_aux+1 ! al27(g, p)mg26
         integer, parameter :: ir_al27_pa_mg24 = ir_al27_gp_mg26+1 ! al27(p, a)mg24 
         integer, parameter :: iral27pa_aux = ir_al27_pa_mg24+1 ! al27(p, a)mg24 
         integer, parameter :: ir_al27_ap_si30 = iral27pa_aux+1 ! al27(a, p)si30
         integer, parameter :: irsi28pg_to_si29 = ir_al27_ap_si30+1 ! si28(p, g)p29(e+nu)si29
         integer, parameter :: ir_si28_ap_p31 = irsi28pg_to_si29+1 ! si28(a, p)p31 
         integer, parameter :: irsi28ap_aux = ir_si28_ap_p31+1 ! si28(a, p)p31 
         integer, parameter :: irsi28ap_to_s32 = irsi28ap_aux+1 
         integer, parameter :: ir_si28_ag_s32 = irsi28ap_to_s32+1 ! si28(a, g)s32 
         integer, parameter :: ir_si28_gp_al27 = ir_si28_ag_s32+1 ! si28(g, p)al27
         integer, parameter :: irsi28gp_aux = ir_si28_gp_al27+1 ! si28(g, p)al27
         integer, parameter :: ir_si28_ga_mg24 = irsi28gp_aux+1 ! si28(g, a)mg24
         integer, parameter :: irsi28gp_to_mg24 = ir_si28_ga_mg24+1 
         integer, parameter :: irsi29pg_to_si30 = irsi28gp_to_mg24+1 ! si29(p, g)p30(e+nu)si30
         integer, parameter :: ir_si30_ga_mg26 = irsi29pg_to_si30+1 ! si30(g, a)mg26
         integer, parameter :: ir_p31_pg_s32 = ir_si30_ga_mg26+1 ! p31(g, p)si30 
         integer, parameter :: irp31pg_aux = ir_p31_pg_s32+1 ! p31(p, g)s32 
         integer, parameter :: ir_p31_pa_si28 = irp31pg_aux+1 ! p31(p, a)si28 
         integer, parameter :: irp31pa_aux = ir_p31_pa_si28+1 ! p31(p, a)si28 
         integer, parameter :: ir_s32_ap_cl35 = irp31pa_aux+1 ! s32(a, p)cl35 
         integer, parameter :: irs32ap_aux = ir_s32_ap_cl35+1 ! s32(a, p)cl35 
         integer, parameter :: irs32ap_to_ar36 = irs32ap_aux+1 ! 
         integer, parameter :: ir_s32_ag_ar36 = irs32ap_to_ar36+1 ! s32(a, g)ar36 
         integer, parameter :: ir_s32_gp_p31 = ir_s32_ag_ar36+1 ! s32(g, p)p31 
         integer, parameter :: irs32gp_aux = ir_s32_gp_p31+1 ! s32(g, p)p31 
         integer, parameter :: ir_s32_ga_si28 = irs32gp_aux+1 ! s32(g, a)si28 
         integer, parameter :: irs32gp_to_si28 = ir_s32_ga_si28+1 !    
         integer, parameter :: ircl35pg_aux = irs32gp_to_si28+1 ! cl35(p, g)ar36
         integer, parameter :: ir_cl35_pa_s32 = ircl35pg_aux+1 ! cl35(p, a)s32 
         integer, parameter :: ircl35pa_aux = ir_cl35_pa_s32+1 ! cl35(p, a)s32 
         integer, parameter :: ir_ar36_ap_k39 = ircl35pa_aux+1 ! ar36(a, p)k39
         integer, parameter :: irar36ap_aux = ir_ar36_ap_k39+1 ! ar36(a, p)k39
         integer, parameter :: irar36ap_to_ca40 = irar36ap_aux+1 !  
         integer, parameter :: ir_ar36_ag_ca40 = irar36ap_to_ca40+1 ! ar36(a, g)ca40 
         integer, parameter :: ir_ar36_gp_cl35 = ir_ar36_ag_ca40+1 ! ar36(g, p)cl35
         integer, parameter :: irar36gp_aux = ir_ar36_gp_cl35+1 ! ar36(g, p)cl35
         integer, parameter :: ir_ar36_ga_s32 = irar36gp_aux+1 ! ar36(g, a)s32 
         integer, parameter :: irar36gp_to_s32 = ir_ar36_ga_s32+1 ! 
         integer, parameter :: irk39pg_aux = irar36gp_to_s32+1 ! k39(p, g)ca40 
         integer, parameter :: ir_k39_pa_ar36 = irk39pg_aux+1 ! k39(p, a)ar36
         integer, parameter :: irk39pa_aux = ir_k39_pa_ar36+1 ! k39(p, a)ar36
         integer, parameter :: ir_ca40_ap_sc43 = irk39pa_aux+1 ! ca40(a, p)sc43 
         integer, parameter :: irca40ap_aux = ir_ca40_ap_sc43+1 ! ca40(a, p)sc43 
         integer, parameter :: irca40ap_to_ti44 = irca40ap_aux+1 !
         integer, parameter :: ir_ca40_ag_ti44 = irca40ap_to_ti44+1 ! ca40(a, g)ti44 
         integer, parameter :: ir_ca40_gp_k39 = ir_ca40_ag_ti44+1 ! ca40(g, p)k39 
         integer, parameter :: irca40gp_aux = ir_ca40_gp_k39+1 ! ca40(g, p)k39 
         integer, parameter :: ir_ca40_ga_ar36 = irca40gp_aux+1 ! ca40(g, a)ar36 
         integer, parameter :: irca40gp_to_ar36 = ir_ca40_ga_ar36+1 !  
         integer, parameter :: irsc43pg_aux = irca40gp_to_ar36+1 ! sc43(p, g)ti44 
         integer, parameter :: ir_sc43_pa_ca40 = irsc43pg_aux+1 ! sc43(p, a)ca40 
         integer, parameter :: irsc43pa_aux = ir_sc43_pa_ca40+1 ! sc43(p, a)ca40 
         integer, parameter :: ir_ti44_ap_v47 = irsc43pa_aux+1 ! ti44(a, p)v47 
         integer, parameter :: irti44ap_aux = ir_ti44_ap_v47+1 ! ti44(a, p)v47 
         integer, parameter :: irti44ap_to_cr48 = irti44ap_aux+1 ! 
         integer, parameter :: ir_ti44_ag_cr48 = irti44ap_to_cr48+1 ! ti44(a, g)cr48
         integer, parameter :: ir_ti44_gp_sc43 = ir_ti44_ag_cr48+1 ! ti44(g, p)sc43 
         integer, parameter :: irti44gp_aux = ir_ti44_gp_sc43+1 ! ti44(g, p)sc43 
         integer, parameter :: ir_ti44_ga_ca40 = irti44gp_aux+1 ! ti44(g, a)ca40 
         integer, parameter :: irti44gp_to_ca40 = ir_ti44_ga_ca40+1
         integer, parameter :: irv47pg_aux = irti44gp_to_ca40+1 ! v47(p, g)cr48 
         integer, parameter :: ir_v47_pa_ti44 = irv47pg_aux+1 ! v47(p, a)ti44 
         integer, parameter :: irv47pa_aux = ir_v47_pa_ti44+1 ! v47(p, a)ti44 
         integer, parameter :: ir_cr48_ap_mn51 = irv47pa_aux+1 ! cr48(a, p)mn51 
         integer, parameter :: ircr48ap_aux = ir_cr48_ap_mn51+1 ! cr48(a, p)mn51 
         integer, parameter :: ircr48ap_to_fe52 = ircr48ap_aux+1
         integer, parameter :: ir_cr48_ag_fe52 = ircr48ap_to_fe52+1 ! cr48(a, g)fe52
         integer, parameter :: ir_cr48_gp_v47 = ir_cr48_ag_fe52+1 ! cr48(g, p)v47 
         integer, parameter :: ircr48gp_aux = ir_cr48_gp_v47+1 ! cr48(g, p)v47 
         integer, parameter :: ir_cr48_ga_ti44 = ircr48gp_aux+1 ! cr48(g, a)ti44
         integer, parameter :: ircr48gp_to_ti44 = ir_cr48_ga_ti44+1
         integer, parameter :: irmn51pg_aux = ircr48gp_to_ti44+1 ! mn51(p, g)fe52 
         integer, parameter :: ir_mn51_pa_cr48 = irmn51pg_aux+1 ! mn51(p, a)cr48 
         integer, parameter :: irmn51pa_aux = ir_mn51_pa_cr48+1 ! mn51(p, a)cr48 
         integer, parameter :: irfe52aprot = irmn51pa_aux+1 ! fe52(a, prot)co55 
         integer, parameter :: irfe52aprot_aux = irfe52aprot+1 ! fe52(a, prot)co55 
         integer, parameter :: irfe52ap_aux = irfe52aprot_aux+1 ! fe52(a, p)co55 
         integer, parameter :: irfe52aprot_to_fe54 = irfe52ap_aux+1 ! fe52(a, prot)co55(g, prot)fe54
         integer, parameter :: irfe52aprot_to_ni56 = irfe52aprot_to_fe54+1 ! fe52(a, prot)co55(prot, g)ni56
         integer, parameter :: irfe52ap_to_ni56 = irfe52aprot_to_ni56+1 ! fe52(a, p)co55(p, g)ni56
         integer, parameter :: ir_fe52_gp_mn51 = irfe52ap_to_ni56+1 ! fe52(g, p)mn51 
         integer, parameter :: irfe52gp_aux = ir_fe52_gp_mn51+1 ! fe52(g, p)mn51 
         integer, parameter :: ir_fe52_ag_ni56 = irfe52gp_aux+1 ! fe52(a, g)ni56
         integer, parameter :: ir_fe52_ga_cr48 = ir_fe52_ag_ni56+1 ! fe52(g, a)cr48
         integer, parameter :: irfe52gp_to_cr48 = ir_fe52_ga_cr48+1
         integer, parameter :: irfe52ng_to_fe54 = irfe52gp_to_cr48+1 ! fe52(n, g)fe53(neut, g)fe54
         integer, parameter :: irfe52ng_aux = irfe52ng_to_fe54+1 ! fe52(n, g)fe53
         integer, parameter :: irfe52neut_to_fe54 = irfe52ng_aux+1 ! fe52(neut, g)fe53(neut, g)fe54
         integer, parameter :: irfe53ng_aux = irfe52neut_to_fe54+1 ! fe53(n, g)fe54
         integer, parameter :: irfe53gn_aux = irfe53ng_aux+1 ! fe53(g, n)fe52
         integer, parameter :: irfe54gn_to_fe52 = irfe53gn_aux+1 ! fe54(g, n)fe53
         integer, parameter :: irfe54gn_aux = irfe54gn_to_fe52+1 ! fe54(g, n)fe53
         integer, parameter :: irfe54g_to_fe52 = irfe54gn_aux+1 ! fe54(g, neut)fe53(g, neut)fe52
         integer, parameter :: irfe54protg = irfe54g_to_fe52+1 ! fe54(prot, g)co55
         integer, parameter :: irfe54protg_aux = irfe54protg+1 ! fe54(prot, g)co55
         integer, parameter :: irfe54prot_to_fe52 = irfe54protg_aux+1 ! fe54(prot, g)co55(prot, a)fe52
         integer, parameter :: irfe54prot_to_ni56 = irfe54prot_to_fe52+1 ! fe54(prot, g)co55(prot, g)ni56
         integer, parameter :: irfe54a_to_ni56 = irfe54prot_to_ni56+1 ! fe54 + alpha -> ni56 + 2 neut
         integer, parameter :: irfe54ng_aux = irfe54a_to_ni56+1 ! fe54(neut, g)fe55                        
         integer, parameter :: irfe55ng_aux = irfe54ng_aux+1 ! fe55(neut, g)fe56                            
         integer, parameter :: irfe55gn_aux = irfe55ng_aux+1 ! fe55(g, neut)fe54                            
         integer, parameter :: irfe56gn_aux = irfe55gn_aux+1 ! fe56(g, neut)fe55                            
         integer, parameter :: irfe54ng_to_fe56 = irfe56gn_aux+1 ! fe54(neut, g)fe55(neut, g)fe56
         integer, parameter :: irfe56gn_to_fe54 = irfe54ng_to_fe56+1 ! fe56(g, neut)fe55(g, neut)fe54
         integer, parameter :: irfe56ec_fake_to_cr56 = irfe56gn_to_fe54+1 ! fe56 + 2 e- => cr56
         integer, parameter :: irco55g_prot = irfe56ec_fake_to_cr56+1 ! co55(g, prot)fe54 
         integer, parameter :: irco55gprot_aux = irco55g_prot+1 ! co55(g, prot)fe54 
         integer, parameter :: irco55prot_a = irco55gprot_aux+1 ! co55(p, a)fe52 
         integer, parameter :: irco55prota_aux = irco55prot_a+1 ! co55(prot, a)fe52 
         integer, parameter :: irco55pa_aux = irco55prota_aux+1 ! co55(p, a)fe52 
         integer, parameter :: irco55protg = irco55pa_aux+1 ! co55(prot, g)ni56 
         integer, parameter :: irco55protg_aux = irco55protg+1 ! co55(prot, g)ni56 
         integer, parameter :: irco55pg_aux = irco55protg_aux+1 ! co55(p, g)ni56 
         integer, parameter :: irni56g_prot = irco55pg_aux+1 ! ni56(g, p)co55 
         integer, parameter :: irni56gprot_aux = irni56g_prot+1 ! ni56(g, prot)co55 
         integer, parameter :: irni56gp_aux = irni56gprot_aux+1 ! ni56(g, p)co55 
         integer, parameter :: irni56gprot_to_fe52 = irni56gp_aux+1 ! ni56(g, prot)co55(prot, a)fe52 
         integer, parameter :: irni56gp_to_fe52 = irni56gprot_to_fe52+1 ! ni56(g, p)co55(p, a)fe52 
         integer, parameter :: irni56gprot_to_fe54 = irni56gp_to_fe52+1 ! ni56(g, prot)co55(g, prot)fe54 
         integer, parameter :: ir_ni56_ga_fe52 = irni56gprot_to_fe54+1 ! ni56(g, a)fe52
         integer, parameter :: irni56ec_to_fe54 = ir_ni56_ga_fe52+1 ! ni56 + 2 e- => 56/54*fe54
         integer, parameter :: irni56ec_to_fe56 = irni56ec_to_fe54+1 ! ni56 + 2 e- => fe56
         integer, parameter :: irprot_to_neut = irni56ec_to_fe56+1 ! prot(e-nu)neut
         integer, parameter :: irneut_to_prot = irprot_to_neut+1 ! neut(e+nu)prot
         integer, parameter :: irhe4gn_to_pieces = irneut_to_prot+1 ! he4(g, neut)he3(g, prot)h2(g, neut)prot
         integer, parameter :: irhe4g_neut_aux = irhe4gn_to_pieces+1 ! he4(g, neut)he3
         integer, parameter :: irhe3gprot_aux = irhe4g_neut_aux+1 ! he3(g, prot)h2
         integer, parameter :: irh2g_neut_aux = irhe3gprot_aux+1 ! h2(g, neut)prot
         integer, parameter :: irhe4_breakup = irh2g_neut_aux+1 ! he4(g, neut)he3(g, prot)h2(g, neut)prot
         integer, parameter :: irprot_neutg_aux = irhe4_breakup+1 ! prot(neut, g)h2
         integer, parameter :: irh2_protg_aux = irprot_neutg_aux+1 ! h2(prot, g)he3
         integer, parameter :: irhe3_neutg_aux = irh2_protg_aux+1 ! he3(neut, g)he4
         integer, parameter :: irhe4_rebuild = irhe3_neutg_aux+1 ! prot(neut, g)h2(prot, g)he3(neut, g)he4
         integer, parameter :: irn14gc12 = irhe4_rebuild+1 ! n14 => c12 + neut + prot
         integer, parameter :: ir_h2_pg_he3 = irn14gc12+1
         integer, parameter :: ir_be7_pg_b8 = ir_h2_pg_he3+1
         integer, parameter :: ir_c12_pg_n13 = ir_be7_pg_b8+1
         integer, parameter :: ir_c13_pg_n14 = ir_c12_pg_n13+1
         integer, parameter :: ir_n13_pg_o14 = ir_c13_pg_n14+1
         integer, parameter :: ir_n14_pg_o15 = ir_n13_pg_o14+1
         integer, parameter :: ir_n15_pg_o16 = ir_n14_pg_o15+1 !
         integer, parameter :: ir_o16_pg_f17 = ir_n15_pg_o16+1 !
         integer, parameter :: ir_o17_pg_f18 = ir_o16_pg_f17+1 !
         integer, parameter :: ir_o18_pg_f19 = ir_o17_pg_f18+1 !
         integer, parameter :: ir_f19_pg_ne20 = ir_o18_pg_f19+1 !
         integer, parameter :: ir_ne22_pg_na23 = ir_f19_pg_ne20+1 !
         
         integer, parameter :: num_predefined_reactions = ir_ne22_pg_na23
         integer :: rates_reaction_id_max

         
         
      
      ! reactions information
         
         integer, parameter :: maxlen_reaction_Name = 32
         character (len=maxlen_reaction_Name), pointer :: reaction_Name(:) ! (rates_reaction_id_max)
   
         integer, parameter :: maxlen_reaction_Info = 72
         character (len=maxlen_reaction_Info), pointer :: reaction_Info(:) ! (rates_reaction_id_max)
         
         double precision, pointer :: std_reaction_Qs(:) ! (rates_reaction_id_max) 
            ! set at initialization; read-only afterwards.
            ! avg energy including neutrinos
         
         double precision, pointer :: std_reaction_neuQs(:) ! (rates_reaction_id_max) 
            ! set at initialization; read-only afterwards.
            ! avg neutrino loss
            
         integer, pointer :: reaction_screening_info(:,:) !(3,rates_reaction_id_max)
            ! reaction_screen_info(1:2,i) = [chem_id1, chem_id2] for screening.  0's if no screening.
            
         integer, pointer :: weak_reaction_info(:,:) ! (2,rates_reaction_id_max)
            ! weak_reaction_info(1:2,i) = [chem_id_in, chem_id_out].  0's if not a weak reaction.

         integer, pointer :: reaction_ye_rho_exponents(:,:) ! (2,rates_reaction_id_max)
            ! multiply T dependent rate by Ye^a(i) * Rho^b(i)
            ! reaction_ye_rho_coeffs(1,i) is a(i)
            ! reaction_ye_rho_coeffs(2,i) is b(i)
            ! (0,0) for photodisintegrations and decays
            ! (0,1) for standard 2 body reactions
            ! (0,2) for 3 body reactions such as triple alpha
            ! (1,1) for 2 body electron captures
            ! (1,2) for 3 body electron captures (e.g., pep)      
      
         integer, parameter :: max_num_reaction_inputs = 3
         integer, pointer :: reaction_inputs(:,:) ! (2*max_num_reaction_inputs,rates_reaction_id_max)
            ! up to max_num_reaction_inputs pairs of coefficients and chem id's, terminated by 0's.
            ! e.g.,  o16(p,g)f17 would be (/ 1, io16, 1, ih1, 0 /)
            ! triple alpha would be (/ 3, ihe4, 0 /)
            ! he3(he4, g)be7(e-,nu)li7(p,a)he4 would be (/ 1, ihe3, 1, ihe4, i, ih1, 0 /)

         integer, parameter :: max_num_reaction_outputs = 4
         integer, pointer :: reaction_outputs(:,:) ! (2*max_num_reaction_outputs,rates_reaction_id_max)
            ! up to max_num_reaction_outputs pairs of coefficients and chem id's, terminated by 0's.
            ! e.g.,  o16(p,g)f17 would be (/ 1, if17, 0 /)
            ! c12(a, p)n15 would be (/ 1, in15, 1, ih1, 0 /)
         



      ! reaction categories

         
         integer, pointer :: reaction_categories(:) ! (rates_reaction_id_max) set by net using reactions.list info




      ! choices for various rates
         ! NOTE: if change these, must edit raw_rates to match.
         
         ! NACRE = Angulo et al. 1999 Nucl. Phys. A, 656, 3
         ! JR = jina reaclib -- file mesa/data/jina_data/results05301331 -- (Sakharuk et al. 2006)
         ! CF88 = Frank Timmes' version of 
            ! Caughlin, G. R. & Fowler, W. A. 1988, Atom. Data and Nuc. Data Tables, 40, 283
         
         
         ! when possible, NACRE is 1, jina reaclib is 2
         integer, parameter :: rates_NACRE_if_available = 1
         integer, parameter :: rates_JR_if_available = 2
         
         ! the jina reaclib rates are not valid below T = 10^7.
         ! if we have the option, we automatically blend over to nacre for low temperatures.
         ! the following values determine the blend region.
         double precision :: JR_T_full_off = 1.0d7 ! don't use JR below this
         double precision :: JR_T_full_on = 1.1d7 ! don't need to blend above this
         
         
      
         ! p(p,e+nu)h2
         integer, parameter :: use_rate_pp_NACRE = 1
         integer, parameter :: use_rate_pp_JR = 2 ! cf88
         integer, parameter :: use_rate_pp_CF88 = 3
         
         ! h2(p,g)he3 
         integer, parameter :: use_rate_dpg_NACRE = 1
         integer, parameter :: use_rate_dpg_JR = 2 ! Descouvemont et al (2004)
         integer, parameter :: use_rate_dpg_CF88 = 3
         
         ! he3(he3, 2p)he4
         integer, parameter :: use_rate_he3he3_NACRE = 1
         integer, parameter :: use_rate_he3he3_JR = 2 ! cf88
         integer, parameter :: use_rate_he3he3_CF88 = 3
         
         ! he4(he3,g)be7
         integer, parameter :: use_rate_he3he4_NACRE = 1
         integer, parameter :: use_rate_he3he4_JR = 2 ! Descouvemont et al (2004)
         integer, parameter :: use_rate_he3he4_CF88 = 3
         
         ! triple alpha
         integer, parameter :: use_rate_3a_NACRE = 1
         integer, parameter :: use_rate_3a_JR = 2 ! Fynbo et al. 2005 Nature 433, 136-139
         integer, parameter :: use_rate_3a_CF88 = 3
         integer, parameter :: use_rate_3a_FL87 = 4 ! Fushiki and Lamb, Apj, 317, 368-388, 1987
            ! note: use_rate_3a_FL87 is a special case. see eval_FL_epsnuc_3alf in rate_lib
      
         ! be7(p,g)b8
         integer, parameter :: use_rate_be7pg_NACRE = 1
         integer, parameter :: use_rate_be7pg_JR = 2 ! cf88
         integer, parameter :: use_rate_be7pg_CF88 = 3
         
         ! li7(p,a)he4
         integer, parameter :: use_rate_li7pa_NACRE = 1
         integer, parameter :: use_rate_li7pa_JR = 2 ! Descouvemont et al (2004)
         integer, parameter :: use_rate_li7pa_CF88 = 3
         
         ! c12(p,g)n13
         integer, parameter :: use_rate_c12pg_NACRE = 1
         integer, parameter :: use_rate_c12pg_JR = 2 ! nacre
         integer, parameter :: use_rate_c12pg_CF88 = 3
         
         ! c12(a,g)o16
         integer, parameter :: use_rate_c12ag_NACRE = 1
         integer, parameter :: use_rate_c12ag_JR = 2 ! Buchmann (1996)
         integer, parameter :: use_rate_c12ag_Kunz = 3 ! Kunz et al. (2002)
         
         ! c13(p,g)n14
         integer, parameter :: use_rate_c13pg_NACRE = 1
         integer, parameter :: use_rate_c13pg_JR = 2 ! nacre
         
         ! c12(c12,a)ne20
         integer, parameter :: use_rate_1212_CF88 = 1
         integer, parameter :: use_rate_1212_JR = 2
         integer, parameter :: use_rate_1212_G05 = 3 ! Gasques, et al. Phys Review C, 72, 025806 (2005)
            ! NOTE: Gasques option for c12+c12 is implemented in net, not in rates.
         
         ! n13(p,g)o14
         integer, parameter :: use_rate_n13pg_NACRE = 1
         integer, parameter :: use_rate_n13pg_JR = 2 ! Li et al. (2006)
      
         ! n14(p,g)o15
         integer, parameter :: use_rate_n14pg_NACRE = 1
         integer, parameter :: use_rate_n14pg_JR = 2 ! Imbriani et al. (2004)
         integer, parameter :: use_rate_n14pg_CF88 = 3
         
         ! n14(a,g)f18
         integer, parameter :: use_rate_n14ag_NACRE = 1
         integer, parameter :: use_rate_n14ag_JR = 2 ! Gorres, Arlandini, Giesen (2000)
         integer, parameter :: use_rate_n14ag_CF88 = 3
         
         ! n15(p,g)o16
         integer, parameter :: use_rate_n15pg_NACRE = 1
         integer, parameter :: use_rate_n15pg_JR = 2 ! nacre
         
         ! n15(p,a)c12
         integer, parameter :: use_rate_n15pa_NACRE = 1
         integer, parameter :: use_rate_n15pa_JR = 2 ! nacre
         integer, parameter :: use_rate_n15pa_CF88 = 3
         
         ! o14(a,p)f17
         integer, parameter :: use_rate_o14ap_CF88 = 1
         integer, parameter :: use_rate_o14ap_JR = 2 ! Hahn 1996    PhRvC 54, 4, p1999-2013
         
         ! o15(a,g)ne19
         integer, parameter :: use_rate_o15ag_CF88 = 1
         integer, parameter :: use_rate_o15ag_JR = 2 ! Hahn 1996    PhRvC 54, 4, p1999-2013
         
         ! o16(p,g)f17
         integer, parameter :: use_rate_o16pg_NACRE = 1
         integer, parameter :: use_rate_o16pg_JR = 2 ! nacre
         integer, parameter :: use_rate_o16pg_CF88 = 3
         
         ! o16(a,g)ne20
         integer, parameter :: use_rate_o16ag_NACRE = 1
         integer, parameter :: use_rate_o16ag_JR = 2 ! nacre
         integer, parameter :: use_rate_o16ag_CF88 = 3
         
         ! o17(p,a)n14
         integer, parameter :: use_rate_o17pa_NACRE = 1
         integer, parameter :: use_rate_o17pa_JR = 2 ! Chafa et al. (2007)
      
         ! o17(p,g)f18
         integer, parameter :: use_rate_o17pg_NACRE = 1
         integer, parameter :: use_rate_o17pg_JR = 2 ! Chafa et al. (2007)
         
         ! o18(p,a)n15
         integer, parameter :: use_rate_o18pa_NACRE = 1
         integer, parameter :: use_rate_o18pa_JR = 2 ! nacre
         
         ! o18(p,g)f19
         integer, parameter :: use_rate_o18pg_NACRE = 1
         integer, parameter :: use_rate_o18pg_JR = 2 ! nacre
         
         ! o18(a,g)ne22
         integer, parameter :: use_rate_o18ag_NACRE = 1
         integer, parameter :: use_rate_o18ag_JR = 2  ! Dababneh (2003)
               
         ! f17(p,g)ne18
         integer, parameter :: use_rate_f17pg_wk82 = 1 ! wiescher and kettner, ap. j., 263, 891 (1982)
         integer, parameter :: use_rate_f17pg_JR = 2 ! Bardayan et al. (2000)
      
         ! f18(p,g)ne19
         integer, parameter :: use_rate_f18pg_wk82 = 1 ! wiescher and kettner, ap. j., 263, 891 (1982)
         integer, parameter :: use_rate_f18pg_JR = 2 ! Shu (2003)
      
         ! f18(p,a)o15
         integer, parameter :: use_rate_f18pa_wk82 = 1 ! wiescher and kettner, ap. j., 263, 891 (1982)
         integer, parameter :: use_rate_f18pa_JR = 2 ! Shu (2003)
         
         ! f19(p,g)ne20
         integer, parameter :: use_rate_f19pg_NACRE = 1
         integer, parameter :: use_rate_f19pg_JR = 2 ! cf88
         
         ! f19(p,a)o16
         integer, parameter :: use_rate_f19pa_NACRE = 1
         integer, parameter :: use_rate_f19pa_JR = 2 ! cf88
         
         ! ne20(a,g)mg24
         integer, parameter :: use_rate_ne20ag_NACRE = 1
         integer, parameter :: use_rate_ne20ag_JR = 2
         integer, parameter :: use_rate_ne20ag_CF88 = 3
         
         ! o16 + o16
         integer, parameter :: use_rate_1616_CF88 = 1
         integer, parameter :: use_rate_1616_reaclib = 2

      
      ! for mazurek's ni56 electron capture rate interpolation
         double precision :: tv(7),rv(6),rfdm(4),rfd0(4),rfd1(4),rfd2(4),tfdm(5),tfd0(5),tfd1(5),tfd2(5)


      type T_Factors
         double precision :: lnT9
         double precision :: T9
         double precision :: T92
         double precision :: T93
         double precision :: T94
         double precision :: T95
         double precision :: T96
         double precision :: T912
         double precision :: T932
         double precision :: T952
         double precision :: T972
         double precision :: T913
         double precision :: T923
         double precision :: T943
         double precision :: T953
         double precision :: T973
         double precision :: T9113
         double precision :: T914
         double precision :: T934
         double precision :: T954
         double precision :: T974
         double precision :: T915
         double precision :: T935
         double precision :: T945
         double precision :: T965
         double precision :: T917
         double precision :: T927
         double precision :: T947
         double precision :: T918
         double precision :: T938
         double precision :: T958
         double precision :: T9i
         double precision :: T9i2
         double precision :: T9i3
         double precision :: T9i12
         double precision :: T9i32
         double precision :: T9i52
         double precision :: T9i72
         double precision :: T9i13
         double precision :: T9i23
         double precision :: T9i43
         double precision :: T9i53
         double precision :: T9i14
         double precision :: T9i34
         double precision :: T9i54
         double precision :: T9i15
         double precision :: T9i35
         double precision :: T9i45
         double precision :: T9i65
         double precision :: T9i17
         double precision :: T9i27
         double precision :: T9i47
         double precision :: T9i18
         double precision :: T9i38
         double precision :: T9i58
         double precision :: T916
         double precision :: T976
         double precision :: T9i76
      end type T_Factors
      

      ! info for rates being evaluated using tables (rate_list.txt)
      type rate_table_info
         logical :: use_rate_table
         logical :: need_to_read
         character (len=132) :: rate_fname
         integer :: nT8s
         double precision, pointer :: T8s(:) ! (nT8s)
         double precision, pointer :: f(:,:) ! (4,nT8s)
      end type rate_table_info
      
      type (rate_table_info), pointer :: raw_rates_records(:)
      character (len=164) :: rates_table_dir

      type (integer_dict), pointer :: reaction_names_dict
      type (integer_dict), pointer :: skip_warnings_dict
   	
      type (reaction_data), target :: reaclib_rates


      ! rate results components
      
      integer, parameter :: i_rate = 1        
      integer, parameter :: i_rate_dT = 2  
      integer, parameter :: i_rate_dRho = 3 
      integer, parameter :: num_rvs = 3
      
      
      ! for tabular evaluation of the raw reaction rates
      double precision :: rattab_thi != 10.301029995664d0 ! log10(highest temp = 2e10)
      double precision :: rattab_tlo != 5.30102999566398d0 ! log10(lowest temp = 2e5)
      
      integer :: rattab_points_per_decade = 2000
      integer :: nrattab ! number of reaction rate table temperatures
         ! nrattab = <points per decade>*(rattab_thi - rattab_tlo) + 1
               
      double precision :: rattab_tstp != (rattab_thi-rattab_tlo)/(nrattab-1)! step size
         
      character (len=256) :: rates_dir


      
      contains


      subroutine start_rates_def_init(data_dir)
         use utils_lib, only: integer_dict_define
         character (*), intent(in) :: data_dir
         integer :: i, ierr         
         rates_dir = trim(data_dir) // '/rates_data'
         ierr = 0
         nullify(reaction_names_dict)
         do i=1,rates_reaction_id_max
            call integer_dict_define(reaction_names_dict, reaction_Name(i), i, ierr)
            if (ierr /= 0) then
               write(*,*) 'FATAL ERROR: rates_def_init failed in integer_dict_define'
               return
            end if
         end do
         call create_skip_warnings_dict(ierr)  
         call set_rattab_range(5.30102999566398d0, 10.301029995664d0)
      end subroutine start_rates_def_init
      
      
      subroutine set_rattab_range(tlo, thi)
         double precision, intent(in) :: tlo, thi
         if (abs(thi - tlo) < 1d-6) then
            rattab_tlo = tlo
            rattab_thi = rattab_tlo
            nrattab = 1
            rattab_tstp = 0
            return
         end if
         rattab_thi = thi
         rattab_tlo = tlo
         nrattab = rattab_points_per_decade*(rattab_thi - rattab_tlo) + 1
         if (nrattab <= 1) then
            rattab_thi = rattab_tlo
            nrattab = 1
            rattab_tstp = 0
         else
            rattab_tstp = (rattab_thi-rattab_tlo)/(nrattab-1)
         end if
      end subroutine set_rattab_range
      
      
      subroutine create_skip_warnings_dict(ierr)
         use utils_lib
         use utils_def
         integer, intent(out) :: ierr
         
         integer :: iounit, n, i, t, id, read_int
         character (len=256) :: buffer, string, filename, list_filename
         
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         
         nullify(skip_warnings_dict)

         list_filename = 'skip_warnings.list'
         ! first try the local directory
         filename = trim(list_filename)
         open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then ! if don't find that file, look in rates_data
            filename = trim(rates_dir) // '/' // trim(list_filename)
            ierr = 0
            open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               call free_iounit(iounit)
               write(string,*) 'failed to open file ' // trim(list_filename)
               call alert(ierr, string)
               write(*,*) trim(string)
               return
            end if
         end if
         
         n = 0
         i = 0
         
      reaction_loop: do
            t = token(iounit, n, i, buffer, string)
            if (t == eof_token) exit
            if (t /= name_token) then
               call error; return
            end if
            call integer_dict_define(skip_warnings_dict, string, 1, ierr)
            if (ierr /= 0) then
               write(*,*) 'FATAL ERROR: create_skip_warnings_dict failed in integer_dict_define'
               return
            end if
         end do reaction_loop
         
         close(iounit)
         call free_iounit(iounit)

         call integer_dict_create_hash(skip_warnings_dict, ierr)
         if (ierr /= 0) then
            write(*,*) 'FATAL ERROR: create_skip_warnings_dict failed'
            return
         end if

         contains
         
         subroutine error
            ierr = -1
            call alert(ierr, 'error in ' // trim(list_filename))
            close(iounit)
            call free_iounit(iounit)
         end subroutine error

      end subroutine create_skip_warnings_dict
		
		
		integer function get_num_reaction_inputs(ir)
		   integer, intent(in) :: ir
		   integer :: j
		   include 'formats.dek'
         get_num_reaction_inputs = max_num_reaction_inputs
         do j = 1, 2*max_num_reaction_inputs-1, 2
            if (reaction_inputs(j,ir) == 0) then
               get_num_reaction_inputs = (j-1)/2
               exit
            end if
         end do
		end function get_num_reaction_inputs
		
		
		integer function get_num_reaction_outputs(ir)
		   integer, intent(in) :: ir
		   integer :: j
         get_num_reaction_outputs = max_num_reaction_outputs
         do j = 1, 2*max_num_reaction_outputs-1, 2
            if (reaction_outputs(j,ir) == 0) then
               get_num_reaction_outputs = (j-1)/2
               exit
            end if
         end do
		end function get_num_reaction_outputs
      
      
      integer function reaclib_index(handle) result(indx)
      	use utils_lib, only: integer_dict_lookup
      	character(len=*), intent(in) :: handle ! as in rates% reaction_handle
         integer :: ierr
         ierr = 0
      	call integer_dict_lookup(reaclib_rates% reaction_dict, handle, indx, ierr)
      	if (ierr /= 0) indx = 0
      end function reaclib_index
      
      
      integer function reaclib_reverse(handle) result(indx)
      	use utils_lib, only: integer_dict_lookup
      	character(len=*), intent(in) :: handle ! as in rates% reaction_handle
         integer :: ierr
         ierr = 0
      	call integer_dict_lookup(reaclib_rates% reverse_dict, handle, indx, ierr)
      	if (ierr /= 0) indx = 0
      end function reaclib_reverse


      end module rates_def

