! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************





      module pycno
      use rates_def
      use utils_lib
      
      implicit none
      
      
      contains
      

      subroutine G05_epsnuc_CC(T, Rho, X12, eps, deps_dT, deps_dRho)
         
         ! from Gasques, et al, Nuclear fusion in dense matter.  astro-ph/0506386.
         ! Phys Review C, 72, 025806 (2005)
         
         use const_def
         
         double precision, intent(in) :: T
         double precision, intent(in) :: Rho
         double precision, intent(in) :: X12 ! mass fraction of c12
         double precision, intent(out) :: eps ! rate in ergs/g/sec
         double precision, intent(out) :: deps_dT ! partial wrt temperature
         double precision, intent(out) :: deps_dRho ! partial wrt density
         
         
         double precision, parameter :: exp_cutoff = 200d0
         double precision, parameter :: exp_max_result = 1d200
         double precision, parameter :: exp_min_result = 1d-200

         double precision, parameter :: sqrt3 = 1.73205080756888d0 ! sqrt[3]
         
         double precision, parameter :: Z = 6 ! charge of C12
         double precision, parameter :: A = 12 ! atomic number for C12

         double precision, parameter :: b_ne20 = 160.64788d0
         double precision, parameter :: b_c12  =  92.1624d0
         double precision, parameter :: b_he4  =  28.2928d0
         
         ! coefficients from "optimal" model 1
         double precision, parameter :: Cexp = 2.638
         double precision, parameter :: Cpyc = 3.90
         double precision, parameter :: Cpl = 1.25
         double precision, parameter :: CT = 0.724
         double precision, parameter :: Csc = 1.0754
         double precision, parameter :: Lambda = 0.5  
         
         double precision :: m, Z2e2, Q1212, MeV_to_erg, barn_to_cm2, MeV_barn_to_erg_cm2, ergs_c12c12
         
         double precision :: Tk, n12, a_ion, Gamma, omega_p, Tp
         double precision :: Ea, tau, Epk, SEpk, lam, Rpyc, Ppyc, Fpyc
         double precision :: Ttilda, Gammatilda, tautilda, phi, gam1, gam2, gam
         double precision :: P, lnF, F, deltaR, R

         double precision :: Epk1, Epk2x, SEpkx1, SEpkx2x, SEpkx2d, SEpkx2n, SEpkx2, phin, phid
         double precision :: Epk2c, dEpk1_dT, dn12_dRho, dTk_dT, da_ion_dRho, dGamma_dT, 
     >         dGamma_dRho, domega_p_dRho, dTp_dRho
         double precision :: dtau_dT, dEpk1_dRho, dEpk2x_dT, dEpk2x_dRho, dEpk2c_dT, dEpk2c_dRho, dEpk_dT, dEpk_dRho
         double precision :: dSEpkx1_dT, dSEpkx1_dRho, dSEpkx2x_dT, dSEpkx2x_dRho, dSEpkx2d_dT, dSEpkx2d_dRho
         double precision :: dSEpkx2n_dT, dSEpkx2n_dRho, dSEpkx2_dT, dSEpkx2_dRho, dSEpk_dT, dSEpk_dRho
         double precision :: dlam_dRho, dPpyc_dRho, dFpyc_dRho, dRpyc_dT, dRpyc_dRho, gam_n, gam_d, dgam_dT, dgam_dRho
         double precision :: dphin_dGamma, dphin_dT, dphin_dRho, dphid_dGamma, 
     >         dphid_dT, dphid_dRho, dphidT, dphidRho
         double precision :: dTtilda_dT, dTtilda_dRho, dGammatilda_dT, dGammatilda_dRho, 
     >         dtautilda_dTtilda, dtautilda_dT, dtautilda_dRho
         double precision :: dgam_n_dT, dgam_n_dRho, dgam_d_dT, dgam_d_dRho, dP_dgam, dP_dTtilda, dP_dT, dP_dRho
         double precision :: lnF1, lnF2x, lnF2, lnF3, dlnF1_dT, dlnF1_dRho, dlnF2x_dT, dlnF2x_dRho, dlnF2_dT, dlnF2_dRho
         double precision :: dlnF3_dT, dlnF3_dRho, dlnF_dT, dlnF_dRho, dF_dT, dF_dRho, n12term, dn12term_dRho
         double precision :: d_deltaR_dT, d_deltaR_dRho, dRdT, dRdRho, exponent
         
         include 'formats.dek'
   
         m = A * amu  ! ion mass of C12 = 12 amu by definition
         Z2e2 = (Z*qe)**2
         Q1212 = b_ne20 + b_he4 - 2 * b_c12 ! MeV
         MeV_to_erg = 1d6 * ev2erg
         barn_to_cm2 = 1d-24
         MeV_barn_to_erg_cm2 = MeV_to_erg * barn_to_cm2
         ergs_c12c12 = MeV_to_erg * Q1212

         Tk = T * kerg ! temperature in ergs
         dTk_dT = kerg
         
         n12 = X12 * Rho * avo / A ! number density of C12, cm^-3
         dn12_dRho = X12 * avo / A 
         
         ! section III.A.
         a_ion = (3 / (4 * Pi * n12))**one_third ! ion-sphere radius, cm
         da_ion_dRho = -one_third * a_ion * dn12_dRho / n12
         
         Gamma = Z2e2 / (a_ion * Tk) ! Coulomb coupling parameter
         dGamma_dT = - Gamma * dTk_dT / Tk
         dGamma_dRho =  - Gamma * da_ion_dRho / a_ion
         
         omega_p = DSQRT(4 * Pi * Z2e2 * n12 / m) ! ion plasma frequency
         domega_p_dRho = 0.5d0 * omega_p * dn12_dRho / n12
         
         Tp = hbar * omega_p ! ion plasma temperature
         dTp_dRho = hbar * domega_p_dRho
         
         ! section III.B.
         Ea = m * (Z2e2 / hbar)**2 ! (eqn 16)
         
         tau = 3 * (Pi/2)**two_thirds * (Ea / Tk)**one_third ! (eqn 16)
         dtau_dT = -one_third * tau * dTk_dT / Tk
         
         Epk1 = hbar * omega_p
         dEpk1_dRho = hbar * domega_p_dRho
         dEpk1_dT = 0
         
         exponent = -Lambda*Tp/Tk
         if (exponent < -exp_cutoff) then
            Epk2x = exp_min_result
            dEpk2x_dT = 0
            dEpk2x_dRho = 0
         else if (exponent > exp_cutoff) then
            Epk2x = exp_max_result
            dEpk2x_dT = 0
            dEpk2x_dRho = 0
         else
            Epk2x = exp(exponent)
            dEpk2x_dT = Epk2x * Lambda * Tp * dTk_dT / Tk**2
            dEpk2x_dRho = -Epk2x * Lambda * Tp * dTp_dRho / Tk
         end if
         
         Epk2c = Z2e2/a_ion + Tk*tau/3
         dEpk2c_dT = (dTk_dT*tau + Tk*dtau_dT)/3
         dEpk2c_dRho = -Z2e2*da_ion_dRho/a_ion**2
         
         Epk = (Epk1 + Epk2c * Epk2x) / MeV_to_erg ! (eqn 32)
         dEpk_dT = (dEpk1_dT + dEpk2c_dT * Epk2x + Epk2c * dEpk2x_dT) / MeV_to_erg
         dEpk_dRho = (dEpk1_dRho + dEpk2c_dRho * Epk2x + Epk2c * dEpk2x_dRho) / MeV_to_erg

         SEpkx1 = -0.428*Epk
         dSEpkx1_dT = -0.428*dEpk_dT
         dSEpkx1_dRho = -0.428*dEpk_dRho
         
         SEpkx2x = 0.613*(8-Epk)
         dSEpkx2x_dT = -0.613*dEpk_dT
         dSEpkx2x_dRho = -0.613*dEpk_dRho
         
         if (SEpkx2x < -exp_cutoff) then
            SEpkx2d = 1 + exp_min_result
            dSEpkx2d_dT = 0
            dSEpkx2d_dRho = 0
         else if (SEpkx2x > exp_cutoff) then
            SEpkx2d = 1 + exp_max_result
            dSEpkx2d_dT = 0
            dSEpkx2d_dRho = 0
         else
            SEpkx2d = 1 + exp(SEpkx2x)
            dSEpkx2d_dT = (SEpkx2d-1)*dSEpkx2x_dT
            dSEpkx2d_dRho = (SEpkx2d-1)*dSEpkx2x_dRho
         end if
         
         SEpkx2n = 3 * Epk**0.308
         dSEpkx2n_dT = 3 * 0.308 * dEpk_dT / Epk
         dSEpkx2n_dRho = 3 * 0.308 * dEpk_dRho / Epk
         
         SEpkx2 = SEpkx2n / SEpkx2d
         dSEpkx2_dT = (dSEpkx2n_dT - SEpkx2n * dSEpkx2d_dT / SEpkx2d) / SEpkx2d
         dSEpkx2_dRho = (dSEpkx2n_dRho - SEpkx2n * dSEpkx2d_dRho / SEpkx2d) / SEpkx2d
         
         exponent = SEpkx1+SEpkx2
         if (exponent < -exp_cutoff) then
            SEpk = 5.15d16 * exp_min_result
            dSEpk_dT = 0
            dSEpk_dRho = 0
         else if (exponent > exp_cutoff) then
            SEpk = 5.15d16 * exp_max_result
            dSEpk_dT = 0
            dSEpk_dRho = 0
         else
            SEpk = 5.15d16 * exp(exponent) ! (eqn 12)
            dSEpk_dT = SEpk * (dSEpkx1_dT + dSEpkx2_dT)
            dSEpk_dRho = SEpk * (dSEpkx1_dRho + dSEpkx2_dRho)
         end if

         ! section III.D.
         lam = ((Rho * X12 / (1.3574d11 * A))**one_third) / (A * Z**2) ! (eqn 24)
         dlam_dRho = one_third * lam / Rho
         
         Ppyc = 8*Cpyc*11.515/(lam**Cpl) ! (eqn 23)
         dPpyc_dRho = - Ppyc * dlam_dRho / lam
         
         Fpyc = DEXP(-Cexp/DSQRT(lam)) ! (eqn 23)
         dFpyc_dRho = Fpyc * Cexp * dlam_dRho / (2 * lam**1.5)
         
         Rpyc = Rho * X12 * A * Z**4 * (1 / (8 * 11.515)) * 1d46 * lam**3 * SEpk * Fpyc * Ppyc ! (eqn 25)
         if (Rpyc < exp_min_result) then
            Rpyc = 0
            dRpyc_dT = 0
            dRpyc_dRho = 0
         else
            dRpyc_dT = Rpyc * (dSEpk_dT / SEpk)
            dRpyc_dRho = Rpyc * (3 * dlam_dRho / lam + dSEpk_dRho / SEpk + dFpyc_dRho / Fpyc + dPpyc_dRho / Ppyc)
         end if

         ! section III.G.
         phin = DSQRT(Gamma)
         dphin_dGamma = 0.5 * phin / Gamma
         dphin_dT = dGamma_dT * dphin_dGamma
         dphin_dRho = dGamma_dRho * dphin_dGamma
         
         phid = (Csc**4 / 9 + Gamma**2) ** 0.25
         dphid_dGamma = 0.5d0 * Gamma / phid**3
         dphid_dT = dGamma_dT * dphid_dGamma
         dphid_dRho = dGamma_dRho *dphid_dGamma
         
         phi = phin / phid
         dphidT = (dphin_dT - dphid_dT * phin / phid) / phid
         dphidRho = (dphin_dRho - dphid_dRho * phin / phid) / phid
         
         Ttilda = DSQRT(Tk**2 + (CT * Tp)**2) ! (eqn 29)
         dTtilda_dT = Tk * dTk_dT / Ttilda
         dTtilda_dRho = CT**2 * Tp * dTp_dRho / Ttilda
         
         Gammatilda = Z2e2 / (a_ion * Ttilda) ! (eqn 29)
         dGammatilda_dT = -Gammatilda * dTtilda_dT / Ttilda
         dGammatilda_dRho = -Gammatilda * (dTtilda_dRho / Ttilda + da_ion_dRho / a_ion)
         
         tautilda = 3 * (Pi/2)**two_thirds * (Ea / Ttilda)**one_third ! (eqn 29)
         dtautilda_dTtilda = - tautilda / (3 * Ttilda)
         dtautilda_dT = dtautilda_dTtilda * dTtilda_dT
         dtautilda_dRho = dtautilda_dTtilda * dTtilda_dRho
         
         gam1 = two_thirds
         gam2 = two_thirds * (Cpl + 0.5)
         gam_n = Tk**2 * gam1 + Tp**2 * gam2
         dgam_n_dT = 2 * Tk * dTk_dT * gam1
         dgam_n_dRho = 2 * Tp * dTp_dRho * gam2
         
         gam_d = Tk**2 + Tp**2
         dgam_d_dT = 2 * Tk * dTk_dT
         dgam_d_dRho = 2 * Tp * dTp_dRho

         gam = gam_n / gam_d ! (eqn 31)
         dgam_dT = (dgam_n_dT - gam_n * dgam_d_dT / gam_d)/gam_d
         dgam_dRho = (dgam_n_dRho - gam_n * dgam_d_dRho / gam_d)/gam_d
         
         P = (8 * (Pi / 2) ** one_third / sqrt3) * (Ea / Ttilda) ** gam ! (eqn 29)
         if (P < exp_min_result) then
            P = 0
            dP_dgam = 0
            dP_dTtilda = 0
            dP_dT = 0
            dP_dRho = 0
         else
            dP_dgam = P * DLOG(Ea / Ttilda)
            dP_dTtilda = -P * gam / Ttilda
            dP_dT = dP_dgam * dgam_dT + dP_dTtilda * dTtilda_dT
            dP_dRho = dP_dgam * dgam_dRho + dP_dTtilda * dTtilda_dRho
         end if
         
         lnF1 = -tautilda
         exponent = -Lambda * Tp / Tk
         if (exponent < -exp_cutoff) then
            lnF2x = exp_min_result
            dlnF2x_dT = 0
            dlnF2x_dRho = 0
         else if (exponent > exp_cutoff) then
            lnF2x = exp_max_result
            dlnF2x_dT = 0
            dlnF2x_dRho = 0
         else
            lnF2x = exp(exponent)
            dlnF2x_dT = lnF2x * Lambda * Tp * dTk_dT / Tk**2
            dlnF2x_dRho = -lnF2x * Lambda * dTp_dRho / Tk
         end if
         
         lnF2 = Csc * Gammatilda * phi * lnF2x
         lnF3 = - Lambda * Tp / Tk
         lnF = lnF1 + lnF2 + lnF3 ! (eqn 29)
         if (lnF < -exp_cutoff) then
            F = exp_min_result
            dF_dT = 0
            dF_dRho = 0
         else if (lnF > exp_cutoff) then
            F = exp_max_result
            dF_dT = 0
            dF_dRho = 0
         else
            F = exp(lnF)
            dlnF1_dT = -dtautilda_dT
            dlnF1_dRho = -dtautilda_dRho
            dlnF2_dT = lnF2 * (dGammatilda_dT / Gammatilda + dphidT / phi + dlnF2x_dT / lnF2x)
            dlnF2_dRho = lnF2 * (dGammatilda_dRho / Gammatilda + dphidRho / phi + dlnF2x_dRho / lnF2x)
            dlnF3_dT = -lnF3 * dTk_dT / Tk
            dlnF3_dRho = lnF3 * dTp_dRho / Tp
            dlnF_dT = dlnF1_dT + dlnF2_dT + dlnF3_dT
            dlnF_dRho = dlnF1_dRho + dlnF2_dRho + dlnF3_dRho
            dF_dT = dlnF_dT * F
            dF_dRho = dlnF_dRho * F
         end if

         ! NOTE: when calculating deltaR we need to convert SEpk to erg cm^2 from MeV barn.
         ! That conversion is included in the numerical coefficient of the eqn for Rpyc.
         
         n12term = n12**2 * (hbar / (2 * m * Z2e2))
         dn12term_dRho = 2 * n12term * dn12_dRho / n12
         
         deltaR = n12term * SEpk * MeV_barn_to_erg_cm2 * P * F ! (eqn 29)
         if (deltaR < 1d-100) then
            deltaR = 0
            d_deltaR_dT = 0
            d_deltaR_dRho = 0
         else
            d_deltaR_dT = deltaR * (dSEpk_dT / SEpk + dP_dT / P + dF_dT / F)
            d_deltaR_dRho = deltaR * (dn12term_dRho / n12term + dSEpk_dRho / SEpk + dP_dRho / P + dF_dRho / F)
         end if
         
         R = Rpyc + deltaR ! eqn 28, s^-1 cm^-3
         dRdT = dRpyc_dT + d_deltaR_dT
         dRdRho = dRpyc_dRho + d_deltaR_dRho

         eps = R * ergs_c12c12 / Rho ! ergs/g/sec
         deps_dT = dRdT * ergs_c12c12 / Rho
         deps_dRho = (dRdRho * ergs_c12c12 - eps) / Rho
      
         if (.true. .or. Rho < 1.7d10) return
         write(*,1) 'T', T
         write(*,1) 'Rho', Rho
         write(*,1) 'X12', X12
         write(*,1) 'eps', eps
         write(*,1) 'deps_dT', deps_dT
         write(*,1) 'deps_dRho', deps_dRho
         write(*,*) 
   
      end subroutine G05_epsnuc_CC
      

      subroutine FL_epsnuc_3alf(T, Rho, Y, UE, r, drdT, drdRho)       
         double precision, intent(in) :: T ! temperature
         double precision, intent(in) :: Rho ! density
         double precision, intent(in) :: Y ! helium mass fraction
         double precision, intent(in) :: UE ! electron molecular weight
         double precision, intent(out) :: r ! rate in ergs/g/sec
         double precision, intent(out) :: drdT ! partial wrt temperature
         double precision, intent(out) :: drdRho ! partial wrt density


         double precision :: T6, R6, R6T, R6T13, R6T16, T62, T612, T623, T653, T632, T613, U, AF
         double precision :: G1, dG1dRho, dG1dT, G2, dG2dRho, dG2dT
         double precision :: B1, dB1dRho, dB1dT, B2, dB2dRho, dB2dT
         double precision :: dUdT, dUdRho, U32, U52, dAFdT, dAFdRho
         double precision :: E1, dE1dT, dE1dRho, E2, dE2dT, dE2dRho
         double precision :: F1, dF1dT, dF1dRho, F2, dF2dT, dF2dRho
         double precision :: dR6dRho, dR6TdRho, dR6T13dRho, dR6T16dRho
         double precision :: dT6dT, dT612dT, dT62dT, dT613dT, dT623dT, dT632dT, dT653dT
         
         double precision, parameter :: CLN = 2.30258509299405d0 ! ln(10)
         
         ! DEBUG
         double precision, parameter :: AF_0  =  1.9005324047511074D+00
         double precision, parameter :: B1_denom_0  =  2.9602238143383192D-01
         double precision, parameter :: B1_0  =  1.2227955158250397D-08
         double precision, parameter :: B2_denom_0  =  1.7563773044362474D+00
         double precision, parameter :: B2_0  =  1.0173166567483392D-14
         double precision, parameter :: E1_0  =  -2.2308014220480969D+00
         double precision, parameter :: F1_0  =  1.5176626709750911D-04
         double precision, parameter :: E2_0  =  -2.2350904778008243D+01
         double precision, parameter :: F2_0  =  2.7741209323605414D-13
         double precision, parameter :: T_0  =  7.9432823472428218D+07
         double precision, parameter :: RHO_0  =  3.1622776917911558D+09
         double precision, parameter :: r_0  =  2.2348420508311778D+20
         double precision, parameter :: G1_0  =  1.5177849505266735D-04
         double precision, parameter :: G2_0  =  2.8758525980353755D-13
         double precision, parameter :: U_0  =  1.0723431204522564D+00

         double precision :: 
     >         B1_denom, dB1_denom_dRho, dB1_denom_dT,
     >         B2_denom, dB2_denom_dRho, dB2_denom_dT
         double precision :: A1, dA1dT, B1_numerator, dB1_numerator_dT
         double precision :: A2, dA2dT, B2_numerator, dB2_numerator_dT

         include 'formats.dek'
         
         R6=RHO*1d-6
         dR6dRho = 1d-6
         R6T=2.0*R6/UE
         dR6TdRho = 2.0*dR6dRho/UE

         R6T16=R6T**(1d0/6d0)
         dR6T16dRho = (1d0/6d0)*dR6TdRho*R6T16/R6T
         R6T13=R6T16**2d0        
         dR6T13dRho = 2*R6T16*dR6T16dRho
         T6=T*1d-6
         dT6dT=1d-6
         dT62dT=2*T6*dT6dT
         T613=T6**(1d0/3d0)
         dT613dT=(1d0/3d0)*dT6dT*T613/T6
         T623=T613**2.0
         dT623dT=2*T613*dT613dT
         T653=T623*T6
         dT653dT = dT623dT*T6 + T623*dT6dT
         
         T62=T6**2d0
         T612=T6**0.5d0
         dT612dT=0.5*dT6dT/T612
         T632=T6*T612
         dT632dT=1.5*T612*dT6dT        
         
         U=1.35D0*R6T13/T623
         dUdT = -U * dT623dT / T623
         dUdRho = U * dR6T13dRho / R6T13
         
         U32 = U**1.5
         U52 = U*U32

         if (U < 1) then ! strong screening regime, eqn 4.8a in F&L
         
            A1 = (1-4.222D-2*T623)**2 + 2.643D-5*T653
            dA1dT = -2*4.222d-2*dT623dT*(1-4.222D-2*T623) + 2.643D-5*dT653dT
            
            B1_denom=A1*T623
            dB1_denom_dT = dA1dT*T623 + A1*dT623dT
            
            B1_numerator = 16.16D0*DEXP(-134.92/T613)
            dB1_numerator_dT = B1_numerator*134.92*dT613dT/T613**2
            
            B1=B1_numerator/B1_denom
            dB1dT = dB1_numerator_dT/B1_denom - B1*dB1_denom_dT/B1_denom
            
            A2 = (1-2.807D-2*T623)**2 + 2.704D-6*T653
            dA2dT = -2*2.807D-2*dT623dT*(1-2.807D-2*T623) + 2.704D-6*dT653dT
            
            B2_denom=A2*T623
            dB2_denom_dT = dA2dT*T623 + A2*dT623dT
            
            B2_numerator = 244.6D0*(1.0+3.528D-3*T623)**5 * DEXP(-235.72D0/T613)
            dB2_numerator_dT = B2_numerator*
     >            (5*3.528D-3*dT623dT/(1.0+3.528D-3*T623) + 235.72D0*dT613dT/T623)

            B2=B2_numerator/B2_denom
            dB2dT = dB2_numerator_dT/B2_denom - B2*dB2_denom_dT/B2_denom
            
            if (5.458D3 > R6T) then
            
               E1 = -1065.1D0/T6
               dE1dT = -E1 * dT6dT / T6
               
               F1 = DEXP(E1)/T632
               dF1dT = F1 * (dE1dT - dT632dT/T632)
               
               B1=B1+F1
               dB1dT = dB1dT + dF1dT
               
            endif
            
            if (1.836D4 > R6T) then
            
               E2 = -3336.4D0/T6
               dE2dT = -E2 * dT6dT / T6
               
               F2 = DEXP(E2)/T632
               dF2dT = F2 * (dE2dT - dT632dT/T632)
            
               B2=B2+F2
               dB2dT = dB2dT + dF2dT
               
            endif
            
            G1=B1*DEXP(60.492D0*R6T13/T6)
            dG1dT = G1*(dB1dT/B1 - 60.492D0*R6T13*dT6dT/T6**2)
            dG1dRho=0
            
            G2=B2*DEXP(106.35D0*R6T13/T6)
            dG2dT = G2*(dB2dT/B2 - 106.35D0*R6T13*dT6dT/T6**2)
            dG2dRho=0            

         else ! pycnonuclear regime, eqn 4.8b in F&L
         
            AF=1.0/U32 + 1.0
            dAFdT = -1.5 * dUdT/U52
            dAFdRho = -1.5 * dUdRho/U52
         
            B1_denom=T612*((1.0-5.680D-2*R6T13)**2+8.815D-7*T62)
            dB1_denom_dT = B1_denom*dT612dT/T612 + T612*8.815D-7*dT62dT
            dB1_denom_dRho = -2*5.680D-2*(1.0-5.680D-2*R6T13)*T612*dR6T13dRho
            
            B1=1.178D0*AF*DEXP(-77.554/R6T16)/B1_denom
            dB1dT = B1 * (dAFdT/AF - dB1_denom_dT/B1_denom)
            dB1dRho = B1 * (dAFdRho/AF + 77.554*dR6T16dRho/R6T16**2 - dB1_denom_dRho/B1_denom)
         
            B2_denom=T612*((1.0-3.791D-2*R6T13)**2+5.162D-8*T62)
            dB2_denom_dT = B2_denom*dT612dT/T612 + T612*5.162D-8*dT62dT
            
            dB2_denom_dRho = T612*(-0.000252733 + 9.58112d-8*(Rho/UE)**(1d0/3d0))*(Rho/UE)**(1d0/3d0)/Rho
            !dB2_denom_dRho = T612*(-0.000252733 + 9.58112d-8*(Rho/UE)**(1d0/3d0))*(Rho/UE)**(1d0/3d0)/Rho
            
            B2=13.48D0*AF*(1.0+5.070D-3*R6T13)**5*DEXP(-135.08D0/R6T16)/B2_denom
            dB2dT = B2 * (dAFdT/AF - dB2_denom_dT/B2_denom)
            dB2dRho = B2 * (dAFdRho/AF + 135.08D0*dR6T16dRho/R6T16**2 - dB2_denom_dRho/B2_denom)            
            
            !write(*,1) 'AF', AF
            !write(*,1) 'B1_denom', B1_denom
            !write(*,1) 'B1', B1
            !write(*,1) 'B2_denom', B2_denom
            !write(*,1) 'B2', B2
            
            if (5.458D3 > R6T) then
            
               E1 = (60.492*R6T13-1065.1D0)/T6
               dE1dT = -E1 * dT6dT / T6
               dE1dRho = 60.492*dR6T13dRho/T6
               
               F1 = DEXP(E1)/T632
               dF1dT = F1 * (dE1dT - dT632dT/T632)
               dF1dRho = F1 * dE1dRho
               
               !write(*,1) 'E1', E1
               !write(*,1) 'F1', F1

               G1=B1+F1
               dG1dT = dB1dT + dF1dT
               dG1dRho = dB1dRho + dF1dRho
               
            else
            
               G1=B1; dG1dRho = dB1dRho; dG1dT = dB1dT
               
            endif
            
            if (1.836D4 > R6T) then
            
               E2 = (106.35D0*R6T13-3336.4D0)/T6
               dE2dT = -E2 * dT6dT / T6
               dE2dRho = 106.35D0*dR6T13dRho/T6
               
               F2 = DEXP(E2)/T632
               dF2dT = F2 * (dE2dT - dT632dT/T632)
               dF2dRho = F2 * dE2dRho
               
               !write(*,1) 'E2', E2
               !write(*,1) 'F2', F2
            
               G2=B2+F2
               dG2dT = dB2dT + dF2dT
               dG2dRho = dB2dRho + dF2dRho
               
            else
            
               G2=B2; dG2dRho = dB2dRho; dG2dT = dB2dT
               
            endif

         endif
      
         r=5.120D29*G1*G2*Y**3*R6**2 ! ergs/g/sec, eqn 4.7 in F&L

         if (r < 1d-99 .or. G1 < 1d-99 .or. G2 < 1d-99) then
            drdT = 0
            drdRho = 0
         else
            drdT = r * (dG1dT/G1 + dG2dT/G2)
            drdRho = r * (dG1dRho/G1 + dG2dRho/G2 + 2*dR6dRho/R6)

            return
         
            write(*,1) 'T', T
            write(*,1) 'RHO', RHO
            write(*,1) 'r', r
            write(*,1) 'G1', G1
            write(*,1) 'G2', G2
            write(*,1) 'U', U
            write(*,*)
            
            write(*,1) 'abs(Rho_0 - Rho)', abs(Rho_0 - Rho)
            
            if (.true. .and. abs(Rho_0 - Rho) > 1d-2) then
               write(*,*)
               write(*,1) 'analytic drdRho', drdRho
               write(*,1) 'numeric drdRho', (r_0 - r) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic dG1dRho', dG1dRho
               write(*,1) 'numeric dG1dRho', (G1_0 - G1) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic dG2dRho', dG2dRho
               write(*,1) 'numeric dG2dRho', (G2_0 - G2) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic dUdRho', dUdRho
               write(*,1) 'numeric dUdRho', (U_0 - U) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic AF', dAFdRho 
               write(*,1) 'numeric AF', (AF_0 - AF) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic B1_denom', dB1_denom_dRho
               write(*,1) 'numeric B1_denom', (B1_denom_0 - B1_denom) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic B1', dB1dRho
               write(*,1) 'numeric B1', (B1_0 - B1) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic B2_denom', dB2_denom_dRho 
               write(*,1) 'numeric B2_denom', (B2_denom_0 - B2_denom) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic B2', dB2dRho 
               write(*,1) 'numeric B2', (B2_0 - B2) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic E1', dE1dRho 
               write(*,1) 'numeric E1', (E1_0 - E1) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic F1', dF1dRho 
               write(*,1) 'numeric F1', (F1_0 - F1) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic E2', dE2dRho 
               write(*,1) 'numeric E2', (E2_0 - E2) / (Rho_0 - Rho)
               write(*,*)
               write(*,1) 'analytic F2', dF2dRho 
               write(*,1) 'numeric F2', (F2_0 - F2) / (Rho_0 - Rho)
               write(*,*)
               stop 'FL_epsnuc_3alf' 
            end if
            
            
            
         end if
   
         
         if (.true.) then
         end if


      end subroutine FL_epsnuc_3alf


      end module pycno
      
      


