      module test_support
      use num_def
      use num_lib
      
      implicit none

      contains

      
      subroutine show_results(nv,y,expect,show_all)
         integer, intent(in) :: nv
         double precision, dimension(nv), intent(in) :: y, expect
         logical, intent(in) :: show_all
         integer :: i
         if (show_all) then
            write(*,'(/,a5,99a20)') 'i', 'calculated    ', 'reference    ', 'lg(abs rel diff)'
            do i=1,nv
               write(*,'(i5,2e20.10,f20.10)') i, y(i), expect(i), log10(abs(y(i)-expect(i))/max(1d-299,abs(expect(i))))
            end do
         else
            write(*,'(/,a5,99a20)') 'i', 'calculated    ', 'reference    '
            do i=1,nv
               write(*,'(i5,2e20.10,f20.10)') i, y(i), expect(i)
            end do
         end if
         write(*,*)
      end subroutine show_results
      
      
      subroutine show_statistics(nfcn,njac,nstep,show_all)
         integer, intent(in) :: nfcn,njac,nstep
         logical, intent(in) :: show_all
         if (.not. show_all) return
			write(*,*) 'number of steps         ', nstep
			write(*,*) 'number of function evals', nfcn
			write(*,*) 'number of jacobians     ', njac
			write(*,*) 'functions + jacobians   ', nfcn+njac
			write(*,*)
      end subroutine show_statistics

      
      subroutine check_results(nv,y,expect,tol,ierr)
         integer, intent(in) :: nv
         double precision, dimension(nv), intent(in) :: y, expect
         double precision, intent(in) :: tol
         integer, intent(out) :: ierr
         integer :: i
         logical :: okay
         include 'formats.dek'
         okay = .true.
         ierr = 0
         do i=1,nv
            if (abs(expect(i)) < tol) cycle
            if (abs(y(i)-expect(i)) > tol) then
               write(*,*) 'check results result#', i
               write(*,*) 'log10 abs diff', log10(abs(y(i)-expect(i)))
               write(*,*) 'y(i)', y(i)
               write(*,*) 'expect(i)', expect(i)
               write(*,*) 'log10 tol', log10(tol)
               write(*,*)
               ierr = -1
               okay = .false.
            end if
         end do
         if (okay) return
         write(*,*)
         do i=1,nv
            write(*,2) 'y expected', i, y(i), expect(i)
         end do
         write(*,*)
         stop 1
         return
      end subroutine check_results

      double precision function f(x,dfdx,lrpar,rpar,lipar,ipar,ierr)
         integer, intent( in ) :: lrpar, lipar
         double precision, intent( in ) :: x
         double precision, intent( out ) :: dfdx
         double precision, intent( inout ), target :: rpar(lrpar)
         integer, intent( inout ), target :: ipar(lipar)
         integer, intent( out ) :: ierr
         ierr = 0
         f = x-3*sin(1-x)
         dfdx = 1+3*cos(1-x)
      end function f
      
      subroutine test_root
         integer, parameter :: lrpar=0, lipar=0
         double precision, target :: rpar(lrpar)
         integer, target :: ipar(lipar)
         double precision :: x, dfdx, y
         double precision :: x1, x3 ! bounds for x
            ! values of f at x1 and x3 must have opposite sign
            ! return value for safe_root will be bracketed by x1 and x3
         double precision :: y1, y3 ! f(x1) and f(x3)
         integer :: imax ! max number of iterations for search
         double precision :: epsx, epsy 
         ! stop seaching when x is determined to within epsx
         ! or when abs(f(x)) is less than epsy
         integer :: i, ierr
         double precision :: expected_root = 0.74800611d0
         ierr = 0
         imax = 100
         x1 = 0
         x3 = 10
         y1 = f(x1,dfdx,lrpar,rpar,lipar,ipar,ierr)
         y3 = f(x3,dfdx,lrpar,rpar,lipar,ipar,ierr)
         epsx = 1d-6
         epsy = 1d-6
         x = safe_root(f,x1,x3,y1,y3,imax,epsx,epsy,lrpar,rpar,lipar,ipar,ierr)
         if (abs(x-expected_root) > 1d-6) stop 1
         write(*,*) 'root', x
      end subroutine test_root
      

      double precision function test_f(x,dfdx,lrpar,rpar,lipar,ipar,ierr)
         ! returns with ierr = 0 if was able to evaluate f and df/dx at x
         double precision, intent(in) :: x
         double precision, intent(out) :: dfdx
         integer, intent(in) :: lipar, lrpar
         integer, intent(inout), target :: ipar(lipar)
         double precision, intent(inout), target :: rpar(lrpar)
         integer, intent(out) :: ierr
         test_f = tanh(x) - 0.4621171572600098d0
         dfdx = 1/cosh(x)**2
         ierr = 0       
      end function test_f
      
      
      subroutine test_root2
         double precision :: x ! provide starting guess on input
         double precision :: x1,x3 ! bounds for x
         double precision :: y1,y3 ! f(x1) and f(x3)
         integer, parameter :: imax = 50, lipar = 0, lrpar = 0
         double precision, target :: rpar(lipar)
         integer, target :: ipar(lrpar)
         double precision :: dx
         double precision, parameter :: epsx = 1d-10, epsy = 1d-10
         integer :: ierr      
         x = -1d0
         dx = 0.1d0
         ierr = 0      
         write(*,*) 'test_root2'         
         call look_for_brackets(x,dx,x1,x3,test_f,y1,y3,imax,lrpar,rpar,lipar,ipar,ierr)
         if (ierr /= 0) stop 1
         write(*,*) 'x1', x1
         write(*,*) 'x3', x3         
         write(*,*) 'y1', y1
         write(*,*) 'y3', y3
         x = safe_root(test_f,x1,x3,y1,y3,imax,epsx,epsy,lrpar,rpar,lipar,ipar,ierr)   
         if (ierr /= 0) stop 2
         write(*,*) 'safe_root', x
         write(*,*)         
      end subroutine test_root2
      
      
      subroutine test_root3
         double precision :: x ! provide starting guess on input
         double precision :: x1,x3 ! bounds for x
         double precision :: y1,y3 ! f(x1) and f(x3)
         integer, parameter :: newt_imax = 10, imax = 50, lipar = 0, lrpar = 0
         double precision, target :: rpar(lipar)
         integer, target :: ipar(lrpar)
         double precision :: dx
         double precision, parameter :: epsx = 1d-10, epsy = 1d-10
         integer :: ierr      
         dx = 0.1d0
         ierr = 0      
         write(*,*) 'test_root3'         
         x = 0.1d0 ! not too bad initial guess.  newton should find it okay.
         x = safe_root_without_brackets( &
                  test_f,x,dx,newt_imax,imax,epsx,epsy,lrpar,rpar,lipar,ipar,ierr)   
         if (ierr /= 0) stop 3
         write(*,*) 'first safe_root_without_brackets', x
         x = -1d0 ! really bad guess will make it give up on newton
         x = safe_root_without_brackets( &
                  test_f,x,dx,newt_imax,imax,epsx,epsy,lrpar,rpar,lipar,ipar,ierr)   
         if (ierr /= 0) stop 3
         write(*,*) 'second safe_root_without_brackets', x
         write(*,*)         
      end subroutine test_root3


      subroutine van_der_Pol_derivs(n,x,y,f,lrpar,rpar,lipar,ipar,ierr)
         integer, intent(in) :: n, lrpar, lipar
         double precision, intent(in) :: x
         double precision, intent(inout) :: y(n)
         double precision, intent(out) :: f(n)
         double precision, intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         integer, intent(out) :: ierr ! nonzero means retry with smaller timestep.
         ierr = 0
         f(1) = y(2)
         f(2) = ((1 - y(1)**2) * y(2) - y(1))/rpar(1)
         ! the derivatives do not depend on x         
      end subroutine van_der_Pol_derivs


      subroutine solout(nr,xold,x,n,y,rwork,iwork,interp_y,lrpar,rpar,lipar,ipar,irtrn)
         ! nr is the step number.
         ! x is the current x value; xold is the previous x value.
         ! y is the current y value.
         ! irtrn negative means terminate integration.
         ! rwork and iwork hold info for 
         integer, intent(in) :: nr, n, lrpar, lipar
         double precision, intent(in) :: xold, x
         double precision, intent(inout) :: y(n)
         double precision, intent(inout), target :: rpar(lrpar), rwork(*)
         integer, intent(inout), target :: ipar(lipar), iwork(*)
         interface
            ! this subroutine can be called from your solout routine.
            ! it computes interpolated values for y components during the just completed step.
            double precision function interp_y(i,s,rwork,iwork,ierr)
               integer, intent(in) :: i ! result is interpolated approximation of y(i) at x=s.
               double precision, intent(in) :: s ! interpolation x value (between xold and x).
               double precision, intent(inout), target :: rwork(*)
               integer, intent(inout), target :: iwork(*)
               integer, intent(out) :: ierr
            end function interp_y
         end interface
         integer, intent(out) :: irtrn
         ! --- prints solution at equidistant output-points
         ! --- by using "contd8", the continuous collocation solution
         double precision :: xout, y1, y2
         integer, parameter :: iprint = 6
         integer :: ierr
         xout = rpar(2)
         irtrn = 1
         if (ipar(1) /= 1) return ! no output
        
         if (nr.eq.1) then
            write (6,99) x,y(1),y(2),nr-1
            xout=x+0.2d0
         else
            do 
               if (x >= xout-1d-10) then
                  ierr = 0
                  y1 = interp_y(1,xout,rwork,iwork,ierr)
                  if (ierr /= 0) stop 10
                  y2 = interp_y(2,xout,rwork,iwork,ierr)
                  if (ierr /= 0) stop 11
                  write (6,99) xout,y1,y2,nr-1
                  xout=xout+0.2d0
                  cycle
               end if
               exit
            end do
        end if
 99     format(1x,'x =',f5.2,'    y =',2e18.10,'    nstep =',i6)
        rpar(2) = xout
        end subroutine solout


      subroutine test_dopri(do_853,show_all)
         logical, intent(in) :: do_853,show_all
         integer, parameter :: nv = 2  ! the number of variables in the van der Pol system of ODEs
         double precision, parameter :: eps = 1d-3 ! stiffness parameter for van der Pol
         double precision :: rtol(1) ! relative error tolerance(s)
         double precision :: atol(1) ! absolute error tolerance(s)
         double precision :: x ! starting value for the interval of integration
         double precision :: xend ! ending value for the interval of integration
         double precision :: y(nv), expect(nv), yprime(nv)
         character (len=64) :: str
         character (len=256) :: dir, fname
         integer, parameter :: lrpar = 2, lipar = 1, nrdens = nv
         integer, parameter :: liwork = nrdens+100, lwork = 11*nv+8*nrdens+100
         double precision :: rpar(lrpar), max_abs_yp2, work(lwork), h, max_step_size
         integer :: io_unit, i, ipar(lipar), iwork(liwork), lout, iout, idid, itol, j
         integer :: check_liwork, check_lwork, max_steps, ierr

         write(*,*)        
         write(*,*) 'vdpol'
         if (do_853) then
            write(*,*) 'dop853'
         else
            write(*,*) 'dopri5'
         end if

         x = 0
         xend = 2.0
         y(1) = 2
         y(2) = 0
         
         lout = 0
         max_steps = 0
         max_step_size = 9

         itol = 0 ! scalar tolerances
         iout = 2 ! want dense output
         
         rtol(1) = 1d-4
         atol(1) = 1d-4
         h = 1d-6
         
         rpar(1) = eps
         rpar(2) = 0
         if (show_all) then
            ipar(1) = 1
         else
            ipar(1) = 0
         end if
         
         iwork = 0
         work = 0

         iwork(5)=nrdens ! want dense output for all components
         iwork(4)=1 ! test for stiffness at each step
         
         if (do_853) then
            call dopri5_work_sizes(nv,nrdens,check_liwork,check_lwork)
         else
            call dop853_work_sizes(nv,nrdens,check_liwork,check_lwork)
         end if
         
         if (check_liwork > liwork .or. check_lwork > lwork) then
            write(*,*) 'need to enlarge work arrays for dopri5'
            stop 1
         end if
         
         ierr = 0
         if (do_853) then
            call dop853( &
                  nv,van_der_Pol_derivs,x,y,xend, &
                  h,max_step_size,max_steps, &
                  rtol,atol,itol, &
                  solout,iout,work,lwork,iwork,liwork, &
                  lrpar,rpar,lipar,ipar,lout,idid)
         else
            call dopri5( &
                  nv,van_der_Pol_derivs,x,y,xend, &
                  h,max_step_size,max_steps, &
                  rtol,atol,itol, &
                  solout,iout,work,lwork,iwork,liwork, &
                  lrpar,rpar,lipar,ipar,lout,idid)
         end if

         if (idid /= 1) then ! trouble
            write(*,*) 'idid', idid
            stop 1
         end if
         
         expect(1:2) = (/ 1.7632345401889102d+00, -8.3568868191466206d-01 /)
         
         call show_results(nv,y,expect,show_all)
         if (.not. show_all) return
         
         ! typical: fcn=   21530     step=  1468     accpt=  1345     rejct=  122
         write (6,91) (iwork(j),j=17,20)
 91      format(' fcn=',i8,'     step=',i6,'     accpt=',i6,'     rejct=',i5)
 
         write(*,*)
      
      end subroutine test_dopri



      subroutine test_cash_karp(show_all)
         logical, intent(in) :: show_all
         
         integer, parameter :: nv = 2  ! the number of variables in the van der Pol system of ODEs
         double precision, parameter :: eps = 1d-3 ! stiffness parameter for van der Pol
         double precision :: rtol(1) ! relative error tolerance(s)
         double precision :: atol(1) ! absolute error tolerance(s)
         double precision :: x ! starting value for the interval of integration
         double precision :: xend ! ending value for the interval of integration
         double precision :: y(nv), expect(nv), yprime(nv)
         character (len=64) :: str
         character (len=256) :: dir, fname
         integer, parameter :: lrpar = 2, lipar = 1
         double precision :: rpar(lrpar), max_abs_yp2, h, max_step_size
         integer :: io_unit, i, ipar(lipar), lout, iout, idid, itol, j
         integer :: liwork, lwork, max_steps, ierr
         double precision, pointer :: work(:)
         integer, pointer :: iwork(:)

         write(*,*)        
         write(*,*) 'vdpol'
         write(*,*) 'cash_karp'
         

         x = 0
         xend = 2.0
         y(1) = 2
         y(2) = 0
         
         lout = 6
         max_steps = 10000
         max_step_size = 9

         itol = 0 ! scalar tolerances
         iout = 0 ! no intermediate output
         
         rtol(1) = 1d-4
         atol(1) = 1d-4
         h = 1d-6
         
         rpar(1) = eps
         rpar(2) = 0
         ipar(1) = 0

         call cash_karp_work_sizes(nv,liwork,lwork)
         allocate(work(lwork), iwork(liwork))
         
         iwork = 0
         work = 0
         
         ierr = 0
         call cash_karp( &
               nv,van_der_Pol_derivs,x,y,xend, &
               h,max_step_size,max_steps, &
               rtol,atol,itol, &
               solout,iout,work,lwork,iwork,liwork, &
               lrpar,rpar,lipar,ipar,lout,idid)

         if (idid /= 1) then ! trouble
            write(*,*) 'idid', idid
            stop 1
         end if

         expect(1:2) = (/ 1.7632345401889102d+00, -8.3568868191466206d-01 /)
         
         call show_results(nv,y,expect,show_all)
         
         if (.not. show_all) return
         
         write (6,91) (iwork(j),j=1,4)
 91      format(' fcn=',i8,'     step=',i6,'     accpt=',i6,'     rejct=',i5)
 
         write(*,*)
         
         deallocate(work, iwork)
      
      end subroutine test_cash_karp

      
      subroutine test_binary_search
         integer, parameter :: n = 100
         integer :: k, result
         
         double precision :: vec(n), val
         
         1 format(a40,f16.6)
         
         do k=1,n
            vec(k) = dble(k)**2
         end do
         
         val = dble(n/3)**2 + 2
         result = binary_search(n, 0, vec, val)
         
         write(*,*) 
         write(*,*) 'binary_search, increasing values, result =', result
         if (vec(result) <= val .and. val < vec(result+1)) then
            write(*,1) 'vec(result)', vec(result)
            write(*,1) 'val', val
            write(*,1) 'vec(result+1)', vec(result+1)
            write(*,*) 'okay'
         else
            stop 1
         end if
         
         vec(:) = -vec(:) ! test decreasing values
         val = -val
         result = binary_search(n, 0, vec, val)
         
         write(*,*) 
         write(*,*) 'binary_search, increasing values, result =', result
         if (vec(result) >= val .and. val > vec(result+1)) then
            write(*,1) 'vec(result)', vec(result)
            write(*,1) 'val', val
            write(*,1) 'vec(result+1)', vec(result+1)
            write(*,*) 'okay'
         else
            stop 1
         end if         
         
         write(*,*)
      
      end subroutine test_binary_search
      
      
      subroutine test_qsort
         use const_def
         integer, parameter :: n = 100
         integer :: ord(n), i
         real*8 :: a(n)
         write(*,*)
         write(*,*) 'qsort into increasing order'
         do i=1,n
            a(i) = sin(2.1*pi*dble(i)/dble(n))
         end do
         call qsort(ord, n, a)
         do i=1,n
            write(*,'(2i5,f12.5)') i, ord(i), a(ord(i))
         end do
         write(*,*)
      end subroutine test_qsort
      
      
      real*8 function g(x) result(y)
         real*8, intent(in) :: x
         y = (x-3)*(x-8)
      end function g
      
      
      subroutine test_find0_quadratic
         real*8 :: xx1, yy1, xx2, yy2, xx3, yy3, x, y
         integer :: ierr
         include 'formats.dek'
         write(*,*) 'test_find0_quadratic'
         xx1 = 1
         yy1 = g(xx1)
         xx2 = 2
         yy2 = g(xx2)
         xx3 = 4
         yy3 = g(xx3)
         x = find0_quadratic(xx1, yy1, xx2, yy2, xx3, yy3, ierr)
         if (ierr /= 0) then
            stop 1
         end if
         y = g(x)
         write(*,1) 'x', x
         write(*,1) 'y', y

         xx1 = 6
         yy1 = g(xx1)
         xx2 = 7
         yy2 = g(xx2)
         xx3 = 8
         yy3 = g(xx3)
         x = find0_quadratic(xx1, yy1, xx2, yy2, xx3, yy3, ierr)
         if (ierr /= 0) then
            stop 1
         end if
         y = g(x)
         write(*,1) 'x', x
         write(*,1) 'y', y
         write(*,*)         
      end subroutine test_find0_quadratic

      
            
      end module test_support
