! ***********************************************************************
!
!   Copyright (C) 2008  Bill Paxton
!
!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful, 
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module vdpol
      
      ! for information about this problem,
      ! see http://pitagora.dm.uniba.it/~testset/problems/vdpol.php

      use num_def
      use num_lib
      use mtx_lib
      use alert_lib
      
      implicit none
      
      
      ! args for isolve -- see num_isolve.dek in num/public
      
      integer, parameter :: which_solver = ros3p_solver ! as defined in num_def.f
      
      integer, parameter :: n = 2 ! the number of variables in the "vdpol" system of ODEs

      double precision :: x 
         ! input: initial x value
         ! output: x value for which the solution has been computed.
      double precision :: y(n) 
         ! input: initial values for y
         ! output: values of y for final value of x.
      double precision :: xend ! desired final x value (positive or negative)
      double precision :: h 
         ! input: initial step size guess
         ! output: predicted next step size from the last accepted step
      double precision :: max_step_size
      integer :: max_steps
      
      ! absolute and relative error tolerances
      double precision :: rtol(1), atol(1) 
      integer :: itol

      ! information about the jacobian matrix
      integer :: ijac, nzmax, isparse, mljac, mujac

      ! information about the "mass" matrix
      integer :: imas, mlmas, mumas
      
      ! switch for calling the subroutine solout or nor
      integer :: iout
      
      integer :: lrd, lid
      double precision, pointer :: rpar_decsol(:) ! (lrd)
      integer, pointer :: ipar_decsol(:) ! (lid)
      
      ! work arrays.
      integer :: lwork, liwork
      double precision, pointer :: work(:) ! (lwork)
      integer, pointer :: iwork(:) ! (liwork)
         
      ! parameter arrays.
      integer, parameter :: lrpar = 1, lipar = 3
      double precision, target :: rpar(lrpar)
      integer, target :: ipar(lipar)
               
      ! io unit for warnings and errors
      integer :: lout
      
      ! result code
      integer :: idid
      
      ! stiffness parameter
      double precision, parameter :: mu = 1d-3
         
      
      contains
      
      
      subroutine solve_vdpol
      
         integer :: ierr, i
         double precision :: yexact(n)
         
         x = 0
         
         y(1) = 2d0
         y(2) = 0d0

         xend = 2d0

         h = 1d-10

         max_step_size = 0 
         max_steps = 500000

         rtol(1) = 1d-8
         atol(1) = 1d-8
         itol = 0
         
         ijac = 1
         nzmax = 0
         isparse = 0
         mljac = n ! square matrix
         mujac = n

         imas = 0
         mlmas = 0
         mumas = 0        
         
         iout = 1
         
         lid = 0
         lrd = 0

         ipar = 0
         rpar = 0         

         lout = 6

         call lapack_work_sizes(n, lrd, lid)

         call isolve_work_sizes(n, nzmax, imas, mljac, mujac, mlmas, mumas, liwork, lwork)
         
         allocate(iwork(liwork), work(lwork), ipar_decsol(lid), rpar_decsol(lrd), stat=ierr)
         if (ierr /= 0) then
            write(*, *) 'allocate ierr', ierr
            stop 1
         end if
      
         iwork = 0
         work = 0
         
         write(*,*)
         write(*,*) 'vdpol'
         write(*,*)
         
         call isolve( &
            which_solver, n, vdpol_derivs, x, y, xend, & 
            h, max_step_size, max_steps, & 
            rtol, atol, itol, & 
            vdpol_jacob, ijac, null_sjac, nzmax, isparse, mljac, mujac, & 
            null_mas, imas, mlmas, mumas, & 
            vdpol_solout, iout, & 
            lapack_decsol, null_decsols, lrd, rpar_decsol, lid, ipar_decsol, &  
            work, lwork, iwork, liwork, & 
            lrpar, rpar, lipar, ipar, & 
            lout, idid)
            
         if (idid /= 1) ierr = -1
         if (ierr /= 0) then
            write(*, *) 'alert message: ' // trim(alert_message)
            stop 1
         end if
         
         write(*,*)
         write(*,*) 'nsteps', iwork(16)
         
         deallocate(iwork, work, ipar_decsol, rpar_decsol)
         
         ! expected solution for stiffness param = 1d-3
         yexact(1) =  1.7632345401889102d+00           
         yexact(2) = -8.3568868191466206d-01
         
         write(*,'(/,a5,99a20)') 'i', 'calculated    ', 'reference    ', 'lg(abs rel diff)'
         do i=1, n
            write(*,'(i5,2e20.10,f20.10)') i, y(i), yexact(i), &
                  log10(abs(y(i)-yexact(i))/max(1d-299, abs(yexact(i))))
         end do
         write(*,*)
         write(*,*)

      end subroutine solve_vdpol


      subroutine vdpol_derivs(n, x, vars, dvars_dx, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: n, lrpar, lipar
         double precision, intent(in) :: x
         double precision, intent(inout) :: vars(n)
         double precision, intent(out) :: dvars_dx(n)
         double precision, intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         integer, intent(out) :: ierr
         ierr = 0
         dvars_dx(1) = vars(2)
         dvars_dx(2) = ((1-vars(1)**2)*vars(2)-vars(1))/mu
      end subroutine vdpol_derivs


      subroutine vdpol_jacob(n, x, y, f, dfdy, ld_dfdy, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: n, ld_dfdy, lrpar, lipar
         double precision, intent(in) :: x
         double precision, intent(inout) :: y(n)
         double precision, intent(out) :: f(n), dfdy(ld_dfdy, n)
         double precision, intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         integer, intent(out) :: ierr
         integer :: nz, i, j
         ierr = 0
         dfdy(1, 1) = 0d0
         dfdy(1, 2) = 1d0
         dfdy(2, 1) = (-2.0d0*y(1)*y(2)-1d0)/mu
         dfdy(2, 2) = (1d0-y(1)**2)/mu
         call vdpol_derivs(n, x, y, f, lrpar, rpar, lipar, ipar, ierr)
      end subroutine vdpol_jacob


      subroutine vdpol_solout(nr, xold, x, n, y, rwork, iwork, interp_y, lrpar, rpar, lipar, ipar, irtrn)
         integer, intent(in) :: nr, n, lrpar, lipar
         double precision, intent(in) :: xold, x
         double precision, intent(inout) :: y(n)
         double precision, intent(inout), target :: rpar(lrpar), rwork(*)
         integer, intent(inout), target :: ipar(lipar), iwork(*)
         interface
            double precision function interp_y(i, s, rwork, iwork, ierr)
               integer, intent(in) :: i
               double precision, intent(in) :: s
               double precision, intent(inout), target :: rwork(*)
               integer, intent(inout), target :: iwork(*)
               integer, intent(out) :: ierr
            end function interp_y
         end interface
         integer, intent(out) :: irtrn
         
         double precision :: xout, y1, y2
         integer :: ierr
         
         ierr = 0
         irtrn = 0
         xout = rpar(1)
         if (nr.eq.1) then
            write (6, 99) x, y(1), y(2), nr-1
            xout=0.2d0
         else
            do
               if (x >= xout) then
                  y1 = interp_y(1, xout, rwork, iwork, ierr)
                  if (ierr /= 0) exit
                  y2 = interp_y(2, xout, rwork, iwork, ierr)
                  write (6, 99) xout, y1, y2, nr-1
                  if (ierr /= 0) exit
                  xout=xout+0.2d0
                  cycle
               end if
               exit
            end do
         end if
         if (ierr /= 0) then
            write(*, *) 'problem with interp_y in vdpol_solout'
            irtrn = -1
         end if
         rpar(1) = xout
  99     format(1x, 'x =', f5.2, '    y =', 2e18.10, '    nstep =', i8)
  
      end subroutine vdpol_solout


      end module vdpol
      

      program sample_ode_solver
      use vdpol
      implicit none
      
      call solve_vdpol
      
      end program sample_ode_solver
