      module test_rosenbrock
      
      
      integer :: nvar
      integer nfun,njac,nstp,nacc,nrej,ndec,nsol,nsng
      
      logical, parameter :: dbg = .false.
      
      
      contains

  
 
      subroutine func(t,y,f)
      integer neqn
      double precision t,y(nvar),f(nvar)

      f(1) = y(2)
      f(2) = ((1-y(1)**2)*y(2)-y(1))/1.0d-3
      
      if (dbg) then
         write(*,*) 'func y(1)', y(1)
         write(*,*) 'func y(2)', y(2)
         write(*,*) 'func f(1)', f(1)
         write(*,*) 'func f(2)', f(2)
         write(*,*)
      end if

      return
      end subroutine func

      subroutine jac_fcn(t,y,f,dfdy)
      integer ldim,neqn
      double precision t,y(nvar),dfdy(nvar,nvar),f(nvar)

      integer i,j

      dfdy(1,1) = 0d0
      dfdy(1,2) = 1d0
      dfdy(2,1) = (-2.0d0*y(1)*y(2)-1d0)/1.0d-3
      dfdy(2,2) = (1d0-y(1)**2)/1.0d-3
      
      if (dbg) then
         write(*,*) 'jac_fcn y(1)', y(1)
         write(*,*) 'jac_fcn y(2)', y(2)
         write(*,*) 'jac_fcn dfdy(1,1)', dfdy(1,1)
         write(*,*) 'jac_fcn dfdy(1,2)', dfdy(1,2)
         write(*,*) 'jac_fcn dfdy(2,1)', dfdy(2,1)
         write(*,*) 'jac_fcn dfdy(2,2)', dfdy(2,2)
         write(*,*)
      
      end if
      
      call func(t,y,f)
      
      return
      end subroutine jac_fcn
   
      
      
   
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      subroutine do_rosenbrock(y,tstart,tend,abstol,reltol,rpar,ipar,ierr)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!      
!    solves the system y'=f(t,y) using a rosenbrock method defined by:
!
!     g = 1/(h*gamma(1)) - jac_fcn(t0,y0)
!     t_i = t0 + alpha(i)*h
!     y_i = y0 + \sum_{j=1}^{i-1} a(i,j)*k_j
!     g * k_i = func( t_i, y_i ) + \sum_{j=1}^s c(i,j)/h * k_j +
!                  gamma(i)*df/dt(t0, y0)
!     y1 = y0 + \sum_{j=1}^s m(j)*k_j 
!
!    for details on rosenbrock methods and their implementation consult:
!         e. hairer and g. wanner
!         "solving odes ii. stiff and differential-algebraic problems".
!         springer series in computational mathematics, springer-verlag, 1996.  
!    the codes contained in the book inspired this implementation.             
!
!    (c)  adrian sandu, august 2004
!    virginia polytechnic institute and state university    
!    contact: sandu@cs.vt.edu
!    this implementation is part of kpp - the kinetic preprocessor
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!       
!~~~>      input arguments: 
!       
!-     y(nvar)       = vector of initial conditions (at t=tstart)
!-    [tstart,tend]  = time range of integration
!        (if tstart>tend the integration is performed backwards in time)  
!-    reltol, abstol = user precribed accuracy
!-    subroutine func( t, y, ydot ) = ode function, 
!                                         returns ydot = y' = f(t,y) 
!-    subroutine func( t, y, ydot ) = jacobian of the ode function,
!                                         returns jcb = df/dy 
!-    ipar(1:10)    = integer inputs parameters
!-    rpar(1:10)    = real inputs parameters
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!
!~~~>     output arguments:  
!        
!-    y(nvar)       -> vector of final states (at t->tend)
!-    ipar(11:20)   -> integer output parameters
!-    rpar(11:20)   -> real output parameters
!-    ierr          -> job status upon return
!          - succes (positive value) or failure (negative value) -
!                    =  1 : success
!                    = -1 : improper value for maximal no of steps
!                    = -2 : selected rosenbrock method not implemented
!                    = -3 : hmin/hmax/hstart must be positive
!                    = -4 : facmin/facmax/facrej must be positive
!                    = -5 : improper tolerance values
!                    = -6 : no of steps exceeds maximum bound
!                    = -7 : step size too small
!                    = -8 : matrix is repeatedly singular
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! 
!~~~>     input parameters:
!
!    note: for input parameters equal to zero the default values of the
!          corresponding variables are used.
!
!    ipar(1)   = 1: f = f(y)   independent of t (autonomous)
!              = 0: f = f(t,y) depends on t (non-autonomous)
!    ipar(2)   = 0: abstol, reltol are nvar-dimensional vectors
!              = 1:  abstol, reltol are scalars
!    ipar(3)  -> maximum number of integration steps
!              for ipar(3)=0) the default value of 100000 is used
!
!    ipar(4)  -> selection of a particular rosenbrock method
!              = 0 :  default method is rodas3
!              = 1 :  method is  ros2
!              = 2 :  method is  ros3 
!              = 3 :  method is  ros4 
!              = 4 :  method is  rodas3
!              = 5:   method is  rodas4
!
!    rpar(1)  -> hmin, lower bound for the integration step size
!                it is strongly recommended to keep hmin = zero 
!    rpar(2)  -> hmax, upper bound for the integration step size
!    rpar(3)  -> hstart, starting value for the integration step size
!                
!    rpar(4)  -> facmin, lower bound on step decrease factor (default=0.2)
!    rpar(5)  -> facmin,upper bound on step increase factor (default=6)
!    rpar(6)  -> facrej, step decrease factor after multiple rejections
!                        (default=0.1)
!    rpar(7)  -> facsafe, by which the new step is slightly smaller 
!                  than the predicted value  (default=0.9)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! 
!~~~>     output parameters:
!
!    note: each call to rosenbrock adds the corrent no. of fcn calls
!         to previous value of ipar(11), and similar for the other params.
!         set ipar(11:20) = 0 before call to avoid this accumulation.
!
!    ipar(11) = no. of function calls
!    ipar(12) = no. of jacobian calls
!    ipar(13) = no. of steps
!    ipar(14) = no. of accepted steps
!    ipar(15) = no. of rejected steps (except at the beginning)
!    ipar(16) = no. of lu decompositions
!    ipar(17) = no. of forward/backward substitutions
!    ipar(18) = no. of singular matrix decompositions
!
!    rpar(11)  -> texit, the time corresponding to the 
!                        computed y upon return
!    rpar(12)  -> hexit, last accepted step before exit
!    for multiple restarts, use hexit as hstart in the following run 
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      implicit none
      
      double precision tstart,tend
      double precision y(nvar),abstol(nvar),reltol(nvar)
      integer ipar(20)
      double precision rpar(20)
      integer ierr
!~~~>  the method parameters      
      integer smax
      parameter (smax = 6)
      integer  method, ros_s
      double precision ros_m(smax), ros_e(smax)
      double precision ros_a(smax*(smax-1)/2), ros_c(smax*(smax-1)/2)
      double precision ros_alpha(smax), ros_gamma(smax), ros_elo
      logical  ros_newf(smax)
      character*12 ros_name
!~~~>  local variables     
      double precision roundoff,facmin,facmax,facrej,facsafe
      double precision hmin, hmax, hstart, hexit
      double precision texit
      integer i, uplimtol, max_no_steps
      logical autonomous, vectortol
!~~~>   statistics on the work performed
!~~~>   parameters
      double precision zero, one, deltamin 
      parameter (zero = 0.0d0)
      parameter (one  = 1.0d0)
      parameter (deltamin = 1.0d-5)
      
      double precision :: dlamch

!~~~>  initialize statistics
      nfun = ipar(11)
      njac = ipar(12)
      nstp = ipar(13)
      nacc = ipar(14)
      nrej = ipar(15)
      ndec = ipar(16)
      nsol = ipar(17)
      nsng = ipar(18)
      
!~~~>  autonomous or time dependent ode. default is time dependent.
      autonomous = .not.(ipar(1).eq.0)

!~~~>  for scalar tolerances (ipar(2).ne.0)  the code uses abstol(1) and reltol(1)
!      for vector tolerances (ipar(2).eq.0) the code uses abstol(1:nvar) and reltol(1:nvar)
      if (ipar(2).eq.0) then
         vectortol = .true.
	      uplimtol  = nvar
      else 
         vectortol = .false.
	      uplimtol  = 1
      end if
      
!~~~>   the maximum number of steps admitted
      if (ipar(3).eq.0) then
         max_no_steps = 100000
      elseif (max_no_steps.gt.0) then
         max_no_steps=ipar(3)
      else 
         write(6,*)'user-selected max no. of steps: ipar(3)=',ipar(3)
         call ros_errormsg(-1,tstart,zero,ierr)
	      return         
      end if

!~~~>  the particular rosenbrock method chosen
      if (ipar(4).eq.0) then
         method = 3
      elseif ( (ipar(4).ge.1).and.(ipar(4).le.5) ) then
         method = ipar(4)
      else  
         write (6,*) 'user-selected rosenbrock method: ipar(4)=', method
         call ros_errormsg(-2,tstart,zero,ierr)
	 return         
      end if
      
!~~~>  unit roundoff (1+roundoff>1)  
      roundoff = dlamch('e')

!~~~>  lower bound on the step size: (positive value)
      if (rpar(1).eq.zero) then
         hmin = zero
      elseif (rpar(1).gt.zero) then 
         hmin = rpar(1)
      else	 
         write (6,*) 'user-selected hmin: rpar(1)=', rpar(1)
         call ros_errormsg(-3,tstart,zero,ierr)
	      return         
      end if
!~~~>  upper bound on the step size: (positive value)
      if (rpar(2).eq.zero) then
         hmax = abs(tend-tstart)
      elseif (rpar(2).gt.zero) then
         hmax = min(abs(rpar(2)),abs(tend-tstart))
      else	 
         write (6,*) 'user-selected hmax: rpar(2)=', rpar(2)
         call ros_errormsg(-3,tstart,zero,ierr)
	      return         
      end if
!~~~>  starting step size: (positive value)
      if (rpar(3).eq.zero) then
         hstart = max(hmin,deltamin)
      elseif (rpar(3).gt.zero) then
         hstart = min(abs(rpar(3)),abs(tend-tstart))
      else	 
         write (6,*) 'user-selected hstart: rpar(3)=', rpar(3)
         call ros_errormsg(-3,tstart,zero,ierr)
	      return         
      end if
!~~~>  step size can be changed s.t.  facmin < hnew/hexit < facmax 
      if (rpar(4).eq.zero) then
         facmin = 0.2d0
      elseif (rpar(4).gt.zero) then
         facmin = rpar(4)
      else	 
         write (6,*) 'user-selected facmin: rpar(4)=', rpar(4)
         call ros_errormsg(-4,tstart,zero,ierr)
	      return         
      end if
      if (rpar(5).eq.zero) then
         facmax = 6.0d0
      elseif (rpar(5).gt.zero) then
         facmax = rpar(5)
      else	 
         write (6,*) 'user-selected facmax: rpar(5)=', rpar(5)
         call ros_errormsg(-4,tstart,zero,ierr)
	      return         
      end if
!~~~>   facrej: factor to decrease step after 2 succesive rejections
      if (rpar(6).eq.zero) then
         facrej = 0.1d0
      elseif (rpar(6).gt.zero) then
         facrej = rpar(6)
      else	 
         write (6,*) 'user-selected facrej: rpar(6)=', rpar(6)
         call ros_errormsg(-4,tstart,zero,ierr)
	      return         
      end if
!~~~>   facsafe: safety factor in the computation of new step size
      if (rpar(7).eq.zero) then
         facsafe = 0.9d0
      elseif (rpar(7).gt.zero) then
         facsafe = rpar(7)
      else	 
         write (6,*) 'user-selected facsafe: rpar(7)=', rpar(7)
         call ros_errormsg(-4,tstart,zero,ierr)
	      return         
      end if
!~~~>  check if tolerances are reasonable
       do i=1,uplimtol
         if ( (abstol(i).le.zero) .or. (reltol(i).le.10.d0*roundoff)
     &          .or. (reltol(i).ge.1.0d0) ) then
            write (6,*) ' abstol(',i,') = ',abstol(i)
            write (6,*) ' reltol(',i,') = ',reltol(i)
            call ros_errormsg(-5,tstart,zero,ierr)
	         return
         end if
       end do
     
 
!~~~>   initialize the particular rosenbrock method

      if (method .eq. 1) then
         call ros2(ros_s, ros_a, ros_c, ros_m, ros_e, 
     &             ros_alpha, ros_gamma, ros_newf, ros_elo, ros_name)
      elseif (method .eq. 2) then
         call ros3(ros_s, ros_a, ros_c, ros_m, ros_e, 
     &             ros_alpha, ros_gamma, ros_newf, ros_elo, ros_name)
      elseif (method .eq. 3) then
         call ros4(ros_s, ros_a, ros_c, ros_m, ros_e, 
     &             ros_alpha, ros_gamma, ros_newf, ros_elo, ros_name)
      elseif (method .eq. 4) then
         call rodas3(ros_s, ros_a, ros_c, ros_m, ros_e, 
     &             ros_alpha, ros_gamma, ros_newf, ros_elo, ros_name)
      elseif (method .eq. 5) then
         call rodas4(ros_s, ros_a, ros_c, ros_m, ros_e, 
     &             ros_alpha, ros_gamma, ros_newf, ros_elo, ros_name)
      else
         write (6,*) 'unknown rosenbrock method: ipar(4)=', method
         call ros_errormsg(-2,tstart,zero,ierr) 
	 return        
      end if
      
      write(*,*) trim(ros_name)
      write(*,*)

!~~~>  call rosenbrock method   
      call rosenbrockintegrator(y,tstart,tend,texit,
     &      abstol,reltol,
!  rosenbrock method coefficients     
     &      ros_s, ros_m, ros_e, ros_a, ros_c, 
     &      ros_alpha, ros_gamma, ros_elo, ros_newf,
!  integration parameters
     &      autonomous, vectortol, max_no_steps,
     &      roundoff, hmin, hmax, hstart, hexit,
     &      facmin, facmax, facrej, facsafe,  
!  error indicator
     &      ierr)


!~~~>  collect run statistics
      ipar(11) = nfun
      ipar(12) = njac
      ipar(13) = nstp
      ipar(14) = nacc
      ipar(15) = nrej
      ipar(16) = ndec
      ipar(17) = nsol
      ipar(18) = nsng
!~~~> last t and h
      rpar(11) = texit
      rpar(12) = hexit       
      
      return
      end subroutine do_rosenbrock

      
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      subroutine rosenbrockintegrator(y,tstart,tend,t,
     &      abstol,reltol,
!~~~> rosenbrock method coefficients     
     &      ros_s, ros_m, ros_e, ros_a, ros_c, 
     &      ros_alpha, ros_gamma, ros_elo, ros_newf,
!~~~> integration parameters
     &      autonomous, vectortol, max_no_steps,
     &      roundoff, hmin, hmax, hstart, hexit,
     &      facmin, facmax, facrej, facsafe,  
!~~~> error indicator
     &      ierr)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!   template for the implementation of a generic rosenbrock method 
!            defined by ros_s (no of stages)  
!            and its coefficients ros_{a,c,m,e,alpha,gamma}
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      implicit none
      
!~~~> input: the initial condition at tstart; output: the solution at t      
      double precision y(nvar)
!~~~> input: integration interval   
      double precision tstart,tend         
!~~~> output: time at which the solution is returned (t=tend if success)   
      double precision t         
!~~~> input: tolerances            
      double precision  abstol(nvar), reltol(nvar)
!~~~> input: the rosenbrock method parameters      
      integer  ros_s
      double precision ros_m(ros_s), ros_e(ros_s) 
      double precision ros_a(ros_s*(ros_s-1)/2), ros_c(ros_s*(ros_s-1)/2)
      double precision ros_alpha(ros_s), ros_gamma(ros_s), ros_elo
      logical ros_newf(ros_s)
!~~~> input: integration parameters      
      logical autonomous, vectortol
      double precision hstart, hmin, hmax
      integer max_no_steps
      double precision roundoff, facmin, facmax, facrej, facsafe 
!~~~> output: last accepted step      
      double precision hexit 
!~~~> output: error indicator
      integer ierr
! ~~~~ local variables           
      double precision ynew(nvar), fcn0(nvar), fcn(nvar),
     &       k(nvar*ros_s), dfdt(nvar), jac(nvar,nvar), mtx(nvar,nvar)
      double precision h, hnew, hc, hg, fac, tau, ytol
      double precision err, yerr(nvar)
      integer pivot(nvar), direction, ioffset, j, istage, i
      logical rejectlasth, rejectmoreh, singular
!~~~>  local parameters
      double precision zero, one, deltamin 
      parameter (zero = 0.0d0)
      parameter (one  = 1.0d0)
      parameter (deltamin = 1.0d-5)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      
!~~~>  initial preparations
      t = tstart
      hexit = 0.0d0
      h = min(hstart,hmax) 
      if (abs(h).le.10.d0*roundoff) then
           h = deltamin
      end if	   
      
      if (tend .ge. tstart) then
        direction = +1
      else
        direction = -1
      end if		

      rejectlasth=.false.
      rejectmoreh=.false.
   
!~~~> time loop begins below 

      do while ( (direction.gt.0).and.(tend-t.ge.roundoff)
     &     .or. (direction.lt.0).and.((tend-t)+roundoff.le.zero) ) 
      if ( nstp.gt.max_no_steps ) then  ! too many steps
         call ros_errormsg(-6,t,h,ierr)
	      return
      end if
      if ( ((t+0.1d0*h).eq.t).or.(h.le.roundoff) ) then  ! step size too small
	      call ros_errormsg(-7,t,h,ierr)
	      return
      end if
      
!~~~>  limit h if necessary to avoid going beyond tend   
      hexit = h
      h = min(h,abs(tend-t))
      
      if (dbg) then
         write(*,*)
         write(*,*) 'direction', direction
         write(*,*) 'roundoff', roundoff
         write(*,*) 't', t
         write(*,*) 'tend', tend
         write(*,*) '(tend-t)', (tend-t)
         write(*,*) '(tend-t) .ge. roundoff', (tend-t) .ge. roundoff
         write(*,*) 'nstep', nstp
         write(*,*) 'h', h
         write(*,*) 'hexit', hexit
      end if   

!~~~>   compute the function at current time
      !call func(t,y,fcn0)
      nfun=nfun+1
  
!~~~>   compute the jacobian at current time
      call jac_fcn(t,y,fcn0,jac)
      njac=njac+1

!~~~>  compute the function derivative with respect to t
      if (.not.autonomous) then
         call ros_funtimederivative ( t, roundoff, y, fcn0, func, dfdt )
      end if
 
!~~~>  repeat step calculation until current step accepted
      do while (.true.) ! while step not accepted
      
      mtx = -jac
      mtx(1,1) = mtx(1,1) + 1/(h*ros_Gamma(1))
      mtx(2,2) = mtx(2,2) + 1/(h*ros_Gamma(1))
      call dgetrf(nvar, nvar, mtx, nvar, pivot, ierr)
      if (ierr /= 0) return

!~~~>   compute the stages
      do istage = 1, ros_s
      
         if (dbg) write(*,*) 'stage', istage

         ! current istage offset. current istage vector is k(ioffset+1:ioffset+nvar)
         ioffset = nvar*(istage-1)

         ! for the 1st istage the function has been computed previously
         if ( istage.eq.1 ) then
            fcn = fcn0
         ! istage>1 and a new function evaluation is needed at the current istage
         else if ( ros_newf(istage) ) then
            ynew = y
            do j = 1, istage-1
               call daxpy(nvar,ros_a((istage-1)*(istage-2)/2+j),k(nvar*(j-1)+1),1,ynew,1) 
            end do
            tau = t + ros_alpha(istage)*direction*h
            call func(tau,ynew,fcn)
            nfun=nfun+1
         end if ! if istage.eq.1 elseif ros_newf(istage)
         
         call dcopy(nvar,fcn,1,k(ioffset+1),1)
         
         do j = 1, istage-1
            hc = ros_c((istage-1)*(istage-2)/2+j)/(direction*h)
            call daxpy(nvar,hc,k(nvar*(j-1)+1),1,k(ioffset+1),1)
         end do
         
         if ((.not. autonomous).and.(ros_gamma(istage).ne.zero)) then
            hg = direction*h*ros_gamma(istage)
            call daxpy(nvar,hg,dfdt,1,k(ioffset+1),1)
         end if
         
         if (dbg) write(*,*) 'rhs(1)', k(ioffset+1)
         if (dbg) write(*,*) 'rhs(2)', k(ioffset+2)
         call dgetrs('N', nvar, 1, mtx, nvar, pivot, k(ioffset+1), nvar, ierr)
         if (ierr /= 0) return
         if (dbg) write(*,*) 'sol(1)', k(ioffset+1)
         if (dbg) write(*,*) 'sol(2)', k(ioffset+2)
         if (dbg) write(*,*)

      end do  ! istage	    
	    

!~~~>  compute the new solution 
      ynew = y
      do j=1,ros_s
	      call daxpy(nvar,ros_m(j),k(nvar*(j-1)+1),1,ynew,1)
      end do

!~~~>  compute the error estimation 
      call dscal(nvar,zero,yerr,1)
      do j=1,ros_s        
	      call daxpy(nvar,ros_e(j),k(nvar*(j-1)+1),1,yerr,1)
      end do 
      
        ERR=0.d0
        do i=1,NVAR
           ytol = AbsTol(i) + RelTol(i)*max(abs(y(i)),abs(ynew(i)))
           ERR = ERR + ( yerr(i)/ytol )**2
        end do      
        ERR = DSQRT( ERR/NVAR ) 
        
        if (dbg) then
            write(*,*) 'nstep', nstp+1
            write(*,*) 'x', t
            write(*,*) 'h', h
            write(*,*) 'err', err
            write(*,*)
        end if

!~~~> new step size is bounded by facmin <= hnew/h <= facmax
      fac  = min(facmax,max(facmin,facsafe/err**(one/ros_elo)))
      hnew = h*fac  

!~~~>  check the error magnitude and adjust step size
      nstp = nstp+1
      if ( (err.le.one).or.(h.le.hmin) ) then  !~~~> accept step
         nacc = nacc+1
	      y = ynew
         t = t + direction*h
	      hnew = max(hmin,min(hnew,hmax))
         if (rejectlasth) then  ! no step size increase after a rejected step
	         hnew = min(hnew,h) 
	      end if   
         rejectlasth = .false.  
         rejectmoreh = .false.
         h = hnew
	      goto 101  ! exit the loop: while step not accepted
      else                 !~~~> reject step
         if (rejectmoreh) then
	         hnew=h*facrej
	      end if   
         rejectmoreh = rejectlasth
         rejectlasth = .true.
         h = hnew
         if (nacc.ge.1) then
	         nrej = nrej+1
	      end if    
      end if ! err <= 1

      end do ! loop: while step not accepted

101   continue

      end do ! time loop    
      
!~~~> succesful exit
      ierr = 1  !~~~> the integration was successful

      return
      end subroutine rosenbrockintegrator
      subroutine ros_funtimederivative ( t, roundoff, y, 
     &                       fcn0, func, dfdt )
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
!~~~> the time partial derivative of the function by finite differences
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      implicit none	 

!~~~> input arguments   
      double precision t, roundoff, y(nvar), fcn0(nvar) 
      external func  
!~~~> output arguments      
      double precision dfdt(nvar)   
!~~~> global variables     
!~~~> local variables     
      double precision delta, deltamin, one     
      parameter ( deltamin = 1.0d-6 )   
      parameter ( one = 1.0d0 )
      
      delta = sqrt(roundoff)*max(deltamin,abs(t))
      call func(t+delta,y,dfdt)
      call daxpy(nvar,(-one),fcn0,1,dfdt,1)
      call dscal(nvar,(one/delta),dfdt,1)

      return
      end subroutine ros_funtimederivative


!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
      subroutine ros_errormsg(code,t,h,ierr)
      double precision t, h
      integer ierr, code
      
      ierr = code
      write(6,*) 
     &   'forced exit from rosenbrock due to the following error:' 
     
      if    (code .eq. -1) then   
         write(6,*) '--> improper value for maximal no of steps'
      elseif (code .eq. -2) then   
         write(6,*) '--> selected rosenbrock method not implemented'
      elseif (code .eq. -3) then   
         write(6,*) '--> hmin/hmax/hstart must be positive'
      elseif (code .eq. -4) then   
         write(6,*) '--> facmin/facmax/facrej must be positive'
      elseif (code .eq. -5) then
         write(6,*) '--> improper tolerance values'
      elseif (code .eq. -6) then
         write(6,*) '--> no of steps exceeds maximum bound'
      elseif (code .eq. -7) then
         write(6,*) '--> step size too small: t + 10*h = t',
     &                ' or h < roundoff'
      elseif (code .eq. -8) then   
         write(6,*) '--> matrix is repeatedly singular'
      else
         write(6,102) 'unknown error code: ',code
      end if
      
  102 format('       ',a,i4)    
      write(6,103) t, h
      
  103 format('        t=',e15.7,' and h=',e15.7)    
     
      return
      end subroutine ros_errormsg
      
      

!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
      subroutine ros2 (ros_s,ros_a,ros_c,ros_m,ros_e,ros_alpha,
     &                ros_gamma,ros_newf,ros_elo,ros_name)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
! --- an l-stable method, 2 stages, order 2
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
      implicit none
      integer s
      parameter (s=2)
      integer  ros_s
      double precision ros_m(s), ros_e(s), ros_a(s*(s-1)/2), ros_c(s*(s-1)/2)
      double precision ros_alpha(s), ros_gamma(s), ros_elo
      logical  ros_newf(s)
      character*12 ros_name
      double precision g
      
       g = 1.0d0 + 1.0d0/sqrt(2.0d0)
      
!~~~> name of the method
       ros_name = 'ros-2'      
!~~~> number of stages
       ros_s = 2
      
!~~~> the coefficient matrices a and c are strictly lower triangular.
!   the lower triangular (subdiagonal) elements are stored in row-wise order:
!   a(2,1) = ros_a(1), a(3,1)=ros_a(2), a(3,2)=ros_a(3), etc.
!   the general mapping formula is:
!             a(i,j) = ros_a( (i-1)*(i-2)/2 + j )       
!             c(i,j) = ros_c( (i-1)*(i-2)/2 + j )  
      
       ros_a(1) = (1.d0)/g
       ros_c(1) = (-2.d0)/g
!~~~> does the stage i require a new function evaluation (ros_newf(i)=true)
!   or does it re-use the function evaluation from stage i-1 (ros_newf(i)=false)
       ros_newf(1) = .true.
       ros_newf(2) = .true.
!~~~> m_i = coefficients for new step solution
       ros_m(1)= (3.d0)/(2.d0*g)
       ros_m(2)= (1.d0)/(2.d0*g)
! e_i = coefficients for error estimator       
       ros_e(1) = 1.d0/(2.d0*g)
       ros_e(2) = 1.d0/(2.d0*g)
!~~~> ros_elo = estimator of local order - the minimum between the
!       main and the embedded scheme orders plus one
       ros_elo = 2.0d0       
!~~~> y_stage_i ~ y( t + h*alpha_i )
       ros_alpha(1) = 0.0d0
       ros_alpha(2) = 1.0d0 
!~~~> gamma_i = \sum_j  gamma_{i,j}       
       ros_gamma(1) = g
       ros_gamma(2) =-g
      
      return
      end subroutine ros2


!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
      subroutine ros3 (ros_s,ros_a,ros_c,ros_m,ros_e,ros_alpha,
     &               ros_gamma,ros_newf,ros_elo,ros_name)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
! --- an l-stable method, 3 stages, order 3, 2 function evaluations
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
      implicit none
      integer s
      parameter (s=3)
      integer  ros_s
      double precision ros_m(s), ros_e(s), ros_a(s*(s-1)/2), ros_c(s*(s-1)/2)
      double precision ros_alpha(s), ros_gamma(s), ros_elo
      logical  ros_newf(s)
      character*12 ros_name
      
!~~~> name of the method
      ros_name = 'ros-3'      
!~~~> number of stages
      ros_s = 3
      
!~~~> the coefficient matrices a and c are strictly lower triangular.
!   the lower triangular (subdiagonal) elements are stored in row-wise order:
!   a(2,1) = ros_a(1), a(3,1)=ros_a(2), a(3,2)=ros_a(3), etc.
!   the general mapping formula is:
!             a(i,j) = ros_a( (i-1)*(i-2)/2 + j )       
!             c(i,j) = ros_c( (i-1)*(i-2)/2 + j )  
     
      ros_a(1)= 1.d0
      ros_a(2)= 1.d0
      ros_a(3)= 0.d0

      ros_c(1) = -0.10156171083877702091975600115545d+01
      ros_c(2) =  0.40759956452537699824805835358067d+01
      ros_c(3) =  0.92076794298330791242156818474003d+01
!~~~> does the stage i require a new function evaluation (ros_newf(i)=true)
!   or does it re-use the function evaluation from stage i-1 (ros_newf(i)=false)
      ros_newf(1) = .true.
      ros_newf(2) = .true.
      ros_newf(3) = .false.
!~~~> m_i = coefficients for new step solution
      ros_m(1) =  0.1d+01
      ros_m(2) =  0.61697947043828245592553615689730d+01
      ros_m(3) = -0.42772256543218573326238373806514d+00
! e_i = coefficients for error estimator       
      ros_e(1) =  0.5d+00
      ros_e(2) = -0.29079558716805469821718236208017d+01
      ros_e(3) =  0.22354069897811569627360909276199d+00
!~~~> ros_elo = estimator of local order - the minimum between the
!       main and the embedded scheme orders plus 1
      ros_elo = 3.0d0       
!~~~> y_stage_i ~ y( t + h*alpha_i )
      ros_alpha(1)= 0.0d+00
      ros_alpha(2)= 0.43586652150845899941601945119356d+00
      ros_alpha(3)= 0.43586652150845899941601945119356d+00
!~~~> gamma_i = \sum_j  gamma_{i,j}       
      ros_gamma(1)= 0.43586652150845899941601945119356d+00
      ros_gamma(2)= 0.24291996454816804366592249683314d+00
      ros_gamma(3)= 0.21851380027664058511513169485832d+01
      return
      end subroutine ros3

!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      


!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   
      subroutine ros4 (ros_s,ros_a,ros_c,ros_m,ros_e,ros_alpha,
     &               ros_gamma,ros_newf,ros_elo,ros_name)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   
!     l-stable rosenbrock method of order 4, with 4 stages
!     l-stable embedded rosenbrock method of order 3 
!
!         e. hairer and g. wanner, solving ordinary differential
!         equations ii. stiff and differential-algebraic problems.
!         springer series in computational mathematics,
!         springer-verlag (1990)               
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   

      implicit none
      integer s
      parameter (s=4)
      integer  ros_s
      double precision ros_m(s), ros_e(s), ros_a(s*(s-1)/2), ros_c(s*(s-1)/2)
      double precision ros_alpha(s), ros_gamma(s), ros_elo
      logical  ros_newf(s)
      character*12 ros_name
      
!~~~> name of the method
      ros_name = 'ros-4'      
!~~~> number of stages
      ros_s = 4
      
!~~~> the coefficient matrices a and c are strictly lower triangular.
!   the lower triangular (subdiagonal) elements are stored in row-wise order:
!   a(2,1) = ros_a(1), a(3,1)=ros_a(2), a(3,2)=ros_a(3), etc.
!   the general mapping formula is:
!             a(i,j) = ros_a( (i-1)*(i-2)/2 + j )       
!             c(i,j) = ros_c( (i-1)*(i-2)/2 + j )  
     
      ros_a(1) = 0.2000000000000000d+01
      ros_a(2) = 0.1867943637803922d+01
      ros_a(3) = 0.2344449711399156d+00
      ros_a(4) = ros_a(2)
      ros_a(5) = ros_a(3)
      ros_a(6) = 0.0d0

      ros_c(1) =-0.7137615036412310d+01
      ros_c(2) = 0.2580708087951457d+01
      ros_c(3) = 0.6515950076447975d+00
      ros_c(4) =-0.2137148994382534d+01
      ros_c(5) =-0.3214669691237626d+00
      ros_c(6) =-0.6949742501781779d+00
!~~~> does the stage i require a new function evaluation (ros_newf(i)=true)
!   or does it re-use the function evaluation from stage i-1 (ros_newf(i)=false)
      ros_newf(1)  = .true.
      ros_newf(2)  = .true.
      ros_newf(3)  = .true.
      ros_newf(4)  = .false.
!~~~> m_i = coefficients for new step solution
      ros_m(1) = 0.2255570073418735d+01
      ros_m(2) = 0.2870493262186792d+00
      ros_m(3) = 0.4353179431840180d+00
      ros_m(4) = 0.1093502252409163d+01
!~~~> e_i  = coefficients for error estimator       
      ros_e(1) =-0.2815431932141155d+00
      ros_e(2) =-0.7276199124938920d-01
      ros_e(3) =-0.1082196201495311d+00
      ros_e(4) =-0.1093502252409163d+01
!~~~> ros_elo  = estimator of local order - the minimum between the
!       main and the embedded scheme orders plus 1
      ros_elo  = 4.0d0       
!~~~> y_stage_i ~ y( t + h*alpha_i )
      ros_alpha(1) = 0.d0
      ros_alpha(2) = 0.1145640000000000d+01
      ros_alpha(3) = 0.6552168638155900d+00
      ros_alpha(4) = ros_alpha(3)
!~~~> gamma_i = \sum_j  gamma_{i,j}       
      ros_gamma(1) = 0.5728200000000000d+00
      ros_gamma(2) =-0.1769193891319233d+01
      ros_gamma(3) = 0.7592633437920482d+00
      ros_gamma(4) =-0.1049021087100450d+00
      return
      end subroutine ros4
      
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
      subroutine rodas3 (ros_s,ros_a,ros_c,ros_m,ros_e,ros_alpha,
     &                ros_gamma,ros_newf,ros_elo,ros_name)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
! --- a stiffly-stable method, 4 stages, order 3
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
      implicit none
      integer s
      parameter (s=4)
      integer  ros_s
      double precision ros_m(s), ros_e(s), ros_a(s*(s-1)/2), ros_c(s*(s-1)/2)
      double precision ros_alpha(s), ros_gamma(s), ros_elo
      logical  ros_newf(s)
      character*12 ros_name
      
!~~~> name of the method
      ros_name = 'rodas-3'      
!~~~> number of stages
      ros_s = 4
      
!~~~> the coefficient matrices a and c are strictly lower triangular.
!   the lower triangular (subdiagonal) elements are stored in row-wise order:
!   a(2,1) = ros_a(1), a(3,1)=ros_a(2), a(3,2)=ros_a(3), etc.
!   the general mapping formula is:
!             a(i,j) = ros_a( (i-1)*(i-2)/2 + j )       
!             c(i,j) = ros_c( (i-1)*(i-2)/2 + j )  
 
      ros_a(1) = 0.0d+00
      ros_a(2) = 2.0d+00
      ros_a(3) = 0.0d+00
      ros_a(4) = 2.0d+00
      ros_a(5) = 0.0d+00
      ros_a(6) = 1.0d+00

      ros_c(1) = 4.0d+00
      ros_c(2) = 1.0d+00
      ros_c(3) =-1.0d+00
      ros_c(4) = 1.0d+00
      ros_c(5) =-1.0d+00 
      ros_c(6) =-(8.0d+00/3.0d+00) 
               
!~~~> does the stage i require a new function evaluation (ros_newf(i)=true)
!   or does it re-use the function evaluation from stage i-1 (ros_newf(i)=false)
      ros_newf(1)  = .true.
      ros_newf(2)  = .false.
      ros_newf(3)  = .true.
      ros_newf(4)  = .true.
!~~~> m_i = coefficients for new step solution
      ros_m(1) = 2.0d+00
      ros_m(2) = 0.0d+00
      ros_m(3) = 1.0d+00
      ros_m(4) = 1.0d+00
!~~~> e_i  = coefficients for error estimator       
      ros_e(1) = 0.0d+00
      ros_e(2) = 0.0d+00
      ros_e(3) = 0.0d+00
      ros_e(4) = 1.0d+00
!~~~> ros_elo  = estimator of local order - the minimum between the
!       main and the embedded scheme orders plus 1
      ros_elo  = 3.0d+00       
!~~~> y_stage_i ~ y( t + h*alpha_i )
      ros_alpha(1) = 0.0d+00
      ros_alpha(2) = 0.0d+00
      ros_alpha(3) = 1.0d+00
      ros_alpha(4) = 1.0d+00
!~~~> gamma_i = \sum_j  gamma_{i,j}       
      ros_gamma(1) = 0.5d+00
      ros_gamma(2) = 1.5d+00
      ros_gamma(3) = 0.0d+00
      ros_gamma(4) = 0.0d+00
      return
      end subroutine rodas3
    
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~      
      subroutine rodas4 (ros_s,ros_a,ros_c,ros_m,ros_e,ros_alpha,
     &                 ros_gamma,ros_newf,ros_elo,ros_name)
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   
!     stiffly-stable rosenbrock method of order 4, with 6 stages
!
!         e. hairer and g. wanner, solving ordinary differential
!         equations ii. stiff and differential-algebraic problems.
!         springer series in computational mathematics,
!         springer-verlag (1996)               
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   

      implicit none
      integer s
      parameter (s=6)
      integer  ros_s
      double precision ros_m(s), ros_e(s), ros_a(s*(s-1)/2), ros_c(s*(s-1)/2)
      double precision ros_alpha(s), ros_gamma(s), ros_elo
      logical  ros_newf(s)
      character*12 ros_name

!~~~> name of the method
       ros_name = 'rodas-4'      
!~~~> number of stages
       ros_s = 6

!~~~> y_stage_i ~ y( t + h*alpha_i )
       ros_alpha(1) = 0.000d0
       ros_alpha(2) = 0.386d0
       ros_alpha(3) = 0.210d0 
       ros_alpha(4) = 0.630d0
       ros_alpha(5) = 1.000d0
       ros_alpha(6) = 1.000d0
	
!~~~> gamma_i = \sum_j  gamma_{i,j}       
       ros_gamma(1) = 0.2500000000000000d+00
       ros_gamma(2) =-0.1043000000000000d+00
       ros_gamma(3) = 0.1035000000000000d+00
       ros_gamma(4) =-0.3620000000000023d-01
       ros_gamma(5) = 0.0d0
       ros_gamma(6) = 0.0d0

!~~~> the coefficient matrices a and c are strictly lower triangular.
!   the lower triangular (subdiagonal) elements are stored in row-wise order:
!   a(2,1) = ros_a(1), a(3,1)=ros_a(2), a(3,2)=ros_a(3), etc.
!   the general mapping formula is:  a(i,j) = ros_a( (i-1)*(i-2)/2 + j )       
!                                    c(i,j) = ros_c( (i-1)*(i-2)/2 + j )  
     
       ros_a(1) = 0.1544000000000000d+01
       ros_a(2) = 0.9466785280815826d+00
       ros_a(3) = 0.2557011698983284d+00
       ros_a(4) = 0.3314825187068521d+01
       ros_a(5) = 0.2896124015972201d+01
       ros_a(6) = 0.9986419139977817d+00
       ros_a(7) = 0.1221224509226641d+01
       ros_a(8) = 0.6019134481288629d+01
       ros_a(9) = 0.1253708332932087d+02
       ros_a(10) =-0.6878860361058950d+00
       ros_a(11) = ros_a(7)
       ros_a(12) = ros_a(8)
       ros_a(13) = ros_a(9)
       ros_a(14) = ros_a(10)
       ros_a(15) = 1.0d+00

       ros_c(1) =-0.5668800000000000d+01
       ros_c(2) =-0.2430093356833875d+01
       ros_c(3) =-0.2063599157091915d+00
       ros_c(4) =-0.1073529058151375d+00
       ros_c(5) =-0.9594562251023355d+01
       ros_c(6) =-0.2047028614809616d+02
       ros_c(7) = 0.7496443313967647d+01
       ros_c(8) =-0.1024680431464352d+02
       ros_c(9) =-0.3399990352819905d+02
       ros_c(10) = 0.1170890893206160d+02
       ros_c(11) = 0.8083246795921522d+01
       ros_c(12) =-0.7981132988064893d+01
       ros_c(13) =-0.3152159432874371d+02
       ros_c(14) = 0.1631930543123136d+02
       ros_c(15) =-0.6058818238834054d+01

!~~~> m_i = coefficients for new step solution
       ros_m(1) = ros_a(7)
       ros_m(2) = ros_a(8)
       ros_m(3) = ros_a(9)
       ros_m(4) = ros_a(10)
       ros_m(5) = 1.0d+00
       ros_m(6) = 1.0d+00

!~~~> e_i  = coefficients for error estimator       
       ros_e(1) = 0.0d+00
       ros_e(2) = 0.0d+00
       ros_e(3) = 0.0d+00
       ros_e(4) = 0.0d+00
       ros_e(5) = 0.0d+00
       ros_e(6) = 1.0d+00

!~~~> does the stage i require a new function evaluation (ros_newf(i)=true)
!   or does it re-use the function evaluation from stage i-1 (ros_newf(i)=false)
       ros_newf(1) = .true.
       ros_newf(2) = .true.
       ros_newf(3) = .true.
       ros_newf(4) = .true.
       ros_newf(5) = .true.
       ros_newf(6) = .true.
     
!~~~> ros_elo  = estimator of local order - the minimum between the
!       main and the embedded scheme orders plus 1
       ros_elo = 4.0d0
     
      return
      end subroutine rodas4


      subroutine do_test_rosenbrock
         use test_support,only:show_results,show_statistics
         integer, parameter :: n = 2
         integer :: nfcn,njac,nstep, ipar(20)
         double precision ::   t, tend, Hmin, Hmax, Hstart, AbsTol(n), RelTol(n)
         double precision ::   y(n), yexact(n), rpar(20)
         logical :: show_all
         show_all = .true.
         
         nvar = n
         
         y(1) = 2d0
         y(2) = 0d0
         
         t = 0
         tend = 2d0
         
         RelTol = 1d-4
         AbsTol = 1d-4
         Hstart = 1d-4 ! initial step size
         Hmax = 1d0
         Hmin = 0d0
         
         
!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
! 
!~~~>     input parameters:
!
!    note: for input parameters equal to zero the default values of the
!          corresponding variables are used.
!
!    ipar(1)   = 1: f = f(y)   independent of t (autonomous)
!              = 0: f = f(t,y) depends on t (non-autonomous)
!    ipar(2)   = 0: abstol, reltol are nvar-dimensional vectors
!              = 1:  abstol, reltol are scalars
!    ipar(3)  -> maximum number of integration steps
!              for ipar(3)=0) the default value of 100000 is used
!
!    ipar(4)  -> selection of a particular rosenbrock method
!              = 0 :  default method is rodas3
!              = 1 :  method is  ros2
!              = 2 :  method is  ros3 
!              = 3 :  method is  ros4 
!              = 4 :  method is  rodas3
!              = 5:   method is  rodas4
!
!    rpar(1)  -> hmin, lower bound for the integration step size
!                it is strongly recommended to keep hmin = zero 
!    rpar(2)  -> hmax, upper bound for the integration step size
!    rpar(3)  -> hstart, starting value for the integration step size
!                
!    rpar(4)  -> facmin, lower bound on step decrease factor (default=0.2)
!    rpar(5)  -> facmin,upper bound on step increase factor (default=6)
!    rpar(6)  -> facrej, step decrease factor after multiple rejections
!                        (default=0.1)
!    rpar(7)  -> facsafe, by which the new step is slightly smaller 
!                  than the predicted value  (default=0.9)
         
         ipar = 0
         rpar = 0
         
         ipar(1) = 1 ! autonomous
         ipar(2) = 1 ! scalar tolerances
         ipar(3) = 0 ! default max steps
         ipar(4) = 4 ! method
         rpar(1) = Hmin
         rpar(2) = Hmax
         rpar(3) = Hstart
         
         call do_rosenbrock(y,t,tend,abstol,reltol,rpar,ipar,ierr)         
!    ipar(11) = no. of function calls
!    ipar(12) = no. of jacobian calls
!    ipar(13) = no. of steps
         nfcn = ipar(11)
         njac = ipar(12)
         nstep = ipar(13)
         
         yexact(1) =  1.7632345401889102d+00           
         yexact(2) = -8.3568868191466206d-01
         call show_results(n,y,yexact,show_all)
         call show_statistics(nfcn,njac,nstep,show_all)
         
      
      end subroutine do_test_rosenbrock
      
      
      end module test_rosenbrock
