! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module num_lib
      ! various numerical routines
      
      use const_def, only: dp
      use num_def
      
      ! NOTE: because of copyright restrictions, 
      !       mesa doesn't use any routines from Numerical Recipes.
      ! all the routines in mesa are open-source.  NR is not.  ;(
         
      ! this does NOT mean we've had to settle for 2nd best algorithms.
      ! the best ones are often too complex to fit in NR anyway.  ;)

      
      implicit none

      contains ! the procedure interface for the library
      ! client programs should only call these routines.
      
      

      
      ! QuickSort. ACM Algorithm 402, van Emden, 1970
      include "num_qsort.dek"

      
      ! safe root finding
      ! uses alternating bisection and inverse parabolic interpolation
      ! also have option to use derivative as accelerator (newton method)
      include "num_safe_root.dek"

      
      ! solvers for ODEs and DAEs.
      
      
      ! sometimes you just want a simple runge-kutta
         include "num_rk4.dek"
         
      
      ! but there are lots of fancier options too.
      

      ! selections from the Hairer family of ODE/DAE integrators.
      ! from Ernst Hairer's website: http://www.unige.ch/~hairer/


      ! explicit ODE solver based on method of Cash and Karp (for non-stiff problems)

         ! explicit Runge-Kutta ODE integrator with variable step size
         include "num_cash_karp.dek"


      ! explicit ODE solvers based on methods of Dormand and Prince (for non-stiff problems)

         ! explicit Runge-Kutta ODE integrator of order 5
         ! with dense output of order 4
         include "num_dopri5.dek" !  "DOramand PRInce order 5"

         ! explicit Runge-Kutta ODE integrator of order 8
         ! with dense output of order 7
         include "num_dop853.dek" !  "DOramand Prince order 8(5, 3)"
         
         ! both integrators have automatic step size control and monitoring for stiffness.
      
         ! For a description see:
         ! Hairer, Norsett and Wanner (1993): 
         ! Solving Ordinary Differential Equations. Nonstiff Problems. 2nd edition. 
         ! Springer Series in Comput. Math., vol. 8.
         ! http://www.unige.ch/~hairer/books.html
      
      
      ! implicit solvers (for stiff problems)
      
         ! there are a bunch of implicit solvers to pick from (listed below), 
         ! but they all have pretty much the same arguments, 
         ! so I've provided a general routine, called "isolve", that let's you
         ! pass an extra argument to specify which one of the particular solvers
         ! that you want to use.
         include "num_isolve.dek"
         
         ! if possible, you should write your code to call isolve
         ! rather than calling one of the particular solvers.
         ! only call a specific solver if you need a feature it provides
         ! that isn't supported by isolve.
         
         ! you can find an example program using isolve in num/test/src/sample_ode_solver.f
         
         
      ! the implicit solver routines
      
         ! for detailed descriptions of these routines see: 
         ! Hairer and Wanner (1996): 
         ! Solving Ordinary Differential Equations. 
         ! Stiff and Differential-Algebraic Problems. 2nd edition. 
         ! Springer Series in Comput. Math., vol. 14.
         ! http://www.unige.ch/~hairer/books.html
      
         ! linearly implicit Runge-Kutta method (Rosenbrock)
         include "num_ros2.dek"    ! L-stable; 2 stages; order 2, 2 function evaluations.
            ! ros2 is suitable for use with approximate jacobians such as from numerical differences.
            ! ros2 is designed for use with Strang splitting and is reported to be able to cope
            ! with large time steps and artificial transients introduced at the beginning of split intervals.
            ! see Verwer et al, "A Second-Order Rosenbrock Method Applied to Photochemical Dispersion Problems", 
            ! SIAM J. Sci. Comput. (20), 1999, 1456-1480.
            
         include "num_rose2.dek"    ! L-stable; 3 stages; order 2, 3 function evaluations.   
            ! rose2 is unique among the implicit solvers in that the final function evaluation
            ! uses the solution vector for the step.

         include "num_rodas3.dek"  ! L-stable; 4 stages; order 3, 3 function evaluations.
         include "num_rodas4.dek"  ! L-stable; 6 stages; order 4, 6 function evaluations.     

         ! 3rd order; for parabolic equations.
         include "num_ros3p.dek"   ! A-stable; 3 stages; order 3, 2 function evaluations.
         include "num_ros3pl.dek"  ! L-stable; 4 stages; order 3, 3 function evaluations.
         ! 4th order; for parabolic equations.
         include "num_rodasp.dek"  ! L-stable; 6 stages; order 4, 6 function evaluations. 
      
         ! implicit extrapolation integrator of variable order.
         ! seulex is based on the linearly implicit Euler method.
         include "num_seulex.dek"   ! "Stiff EULer EXtrapolation"
      
         ! implicit extrapolation integrator of variable order.
         ! based on linearly implicit mid-point rule of bader-deuflhard.
         include "num_sodex.dek"   
         
         ! I've included radau5 in case you'd like to try it.
         ! However, it isn't part of the "standard" set supported by isolve
         ! because, unlike the other solvers, it requires complex linear algebra routines.
         include "num_radau5.dek"


         include "num_solvers_options.dek"


      ! which implicit solver should you use?
         
         ! somewhat surprisingly, in some cases the solvers
         ! that work well at high tolerances will fail with low
         ! tolerances and vice-versa.  so you need to match
         ! the solver to the problem.
      
         ! your best bet is to try them all on some typical cases.
         ! happily this isn't too hard to do since they all
         ! use the same function arguments and have (almost)
         ! identical calling sequences and options.
      
      
      ! flexible choice of linear algebra routines
      
         ! the solvers need to solve linear systems.
         ! this is typically done by first factoring the matrix A
         ! and then repeatedly using the factored form to solve
         ! A*x=b for various vectors b.
         
         ! rather than build-in a particular matrix solver, 
         ! the mesa versions of the solvers take as arguments
         ! routines to perform these tasks.  the mesa/mtx package
         ! includes several choices for implementations of the
         ! required routines.
         
      
      ! dense, banded, or sparse matrix
      
         ! All the packages allow the matrix to be in dense or banded form.
         ! the choice of sparse matrix package is not fixed by the solvers.
         ! the only constraint is the the sparse format must be either
         ! compressed row sparse or compressed column sparse.
         ! the mesa/mtx package comes with one option for a sparse
         ! package (based on SPARSKIT), and also has hooks for another (Super_LU).
         ! Since the sparse routines are passed as arguments to the solvers, 
         ! it is possible to experiment with different linear algebra
         ! packages without a great deal of effort.
         
         
      ! analytical or numerical jacobian
      
         ! to solve M*y' = f(y), the solvers need to have the jacobian matrix, df/dy.
         ! the jacobian can either be calculated analytically by a user supplied routine, 
         ! or the solver can form a numerical difference estimate by repeatedly
         ! evaluating f(y) with slightly different y's.  Such numerical jacobians
         ! are supported by all the solvers for both dense and banded matrix forms.
         ! For the sparse matrix case, only analytical jacobians are allowed.
         
         ! NOTE: for most implicit solvers, the accuracy of the jacobian influences
         ! the rate of convergence, but doesn't impact the accuracy of the solution.
         ! however, the rodas solvers are an exception to this rule.
         ! they are based on the rosenbrock method which replaces the newton iteration
         ! by formulas that directly use the jacobian in the formula for
         ! the result.  as a result, the rodas solvers depend on having
         ! accurate jacobians in order to produce accurate results.
      
      
      ! explicit or implicit ODE systems
      
         ! systems of the form y' = f(y) are called "explicit ODE systems".
         
         ! systems of the form M*y' = f(y), with M not equal to the identity matrix, 
         ! are called "implicit ODE systems".
         
         ! in addition to the usual explicit systems, 
         ! the solvers can also handle implicit ODE systems
         ! in which M is an arbitrary constant matrix, 
         ! even including the case of M singular.
         
         ! for M non-constant, see the discussion of "problems with special structure"
      
      
      ! problems with special structure
      
         ! 3 special cases can be handled easily
         
            ! case 1, second derivatives: y'' = f(t, y, y')
            ! case 2, nonconstant matrix: C(x, y)*y' = f(t, y)
            ! case 3, both of the above: C(x, y)*y'' = f(t, y, y')
            
         ! these all work by adding auxiliary variables to the problem and
         ! converting back to the standard form with a constant matrix M.
         
            ! case 1: y'' = f(t, y, y')
               ! after add auxiliary variables z, this becomes
               ! y' = z
               ! z' = f(t, y, z)
               
            ! case 2: C(x, y)*y' = f(t, y)
               ! after add auxiliary variables z, this becomes
               ! y' = z
               ! 0 = C(x, y)*z - f(t, y)
               
            ! case 3: C(x, y)*y'' = f(t, y, y')
               ! after add auxiliary variables z and u, this becomes
               ! y' = z
               ! z' = u
               ! 0 = C(x, y)*u - f(t, y, z)
         
         ! The last two cases take advantage of the ability to have M singular.
         
         ! If the matrix for df/dy is dense in these special cases, all the solvers
         ! (with the exception of the older code sodex) can reduce the cost of the
         ! linear algebra operations by special treatment of the auxiliary variables.
         
      
      ! "projection" of solution to valid range of values.
      
         ! it is often the case that the n-dimensional solution
         ! is actually constrained to a subspace of full n dimensional
         ! space of numbers.  The proposed solutions at each step
         ! need to be projected back to the allowed subspace in order
         ! to maintain valid results.  The routines all provide for this
         ! option by calling a "solout" routine, supplied by the user, 
         ! after every accepted step.  The user's solout routine can modify
         ! the solution y before returning to continue the integration.
      
         
      ! "dense output"
      
         ! the routines provide estimates of the solution over entire step.
         ! useful for tabulating the solution at prescribed points
         ! or for smooth graphical presentation of the solution.
         ! also very useful for "event location" -- e.g., at what
         ! value of x do we get a solution y(x) s.t. some relation
         ! g(x, y(x))=0.  The dense output option is very helpful here.
         ! All of the solvers support dense output.
         ! BTW: there is typically a certain overhead associated with
         ! providing the option for dense output, so don't request it
         ! unless you'll really be using it.

         ! here is a special version of safe_root for use with dense output "solout" routines
         include "num_solout_root.dek"



      ! "null" implementations of routines used by the solvers
      ! are for use in cases in which you aren't actually using the routine, 
      ! but something must be provided for the required argument.
      

      subroutine null_jac(n, x, y, f, dfy, ldfy, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: n, ldfy, lrpar, lipar
         real(dp), intent(in) :: x
         real(dp), intent(inout) :: y(n)
         real(dp), intent(out) :: f(n) ! dy/dx
         real(dp), intent(out) :: dfy(ldfy, n)
         real(dp), intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         integer, intent(out) :: ierr ! nonzero means terminate integration
         f=0; dfy=0; ierr=0
      end subroutine null_jac


      subroutine null_sjac(n, x, y, f, nzmax, ia, ja, values, lrpar, rpar, lipar, ipar, ierr)  
         ! sparse jacobian. format either compressed row or compressed column.
         integer, intent(in) :: n, nzmax, lrpar, lipar
         real(dp), intent(in) :: x
         real(dp), intent(inout) :: y(n)
         real(dp), intent(out) :: f(n) ! dy/dx
         integer, intent(out) :: ia(n+1), ja(nzmax)
         real(dp), intent(out) :: values(nzmax)
         real(dp), intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         integer, intent(out) :: ierr ! nonzero means terminate integration
         f=0; values = 0; ia = 0; ja = 0; ierr = 0
      end subroutine null_sjac


      subroutine null_mas(n, am, lmas, lrpar, rpar, lipar, ipar)
         integer, intent(in) :: n, lmas, lrpar, lipar
         real(dp), intent(out) :: am(lmas, n)
         real(dp), intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         am = 0
      end subroutine null_mas


      subroutine null_solout(nr, xold, x, n, y, rwork, iwork, interp_y, lrpar, rpar, lipar, ipar, irtrn)
         integer, intent(in) :: nr, n, lrpar, lipar
         real(dp), intent(in) :: xold, x
         real(dp), intent(inout) :: y(n)
         real(dp), intent(inout), target :: rpar(lrpar), rwork(*)
         integer, intent(inout), target :: ipar(lipar), iwork(*)
         interface
            real(dp) function interp_y(i, s, rwork, iwork, ierr)
               use const_def, only: dp
               integer, intent(in) :: i ! result is interpolated approximation of y(i) at x=s.
               real(dp), intent(in) :: s ! interpolation x value (between xold and x).
               real(dp), intent(inout), target :: rwork(*)
               integer, intent(inout), target :: iwork(*)
               integer, intent(out) :: ierr
            end function interp_y
         end interface
         integer, intent(out) :: irtrn
         irtrn = 0
      end subroutine null_solout
      
      
      subroutine null_dfx(n, x, y, fx, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: n, lrpar, lipar
         real(dp), intent(in) :: x, y(n)
         real(dp), intent(out) :: fx(n)
         real(dp), intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         integer, intent(out) :: ierr
         ierr = 0
         fx = 0
      end subroutine null_dfx
      
      
      subroutine null_radau5_debug_routine( 
     >     message, info, n, x0, x, y, xend, hmax, h,  
     >     rtol, atol, itol, okay_to_reuse_jac,  
     >     ijac, nzmax, isparse, mljac, mujac, mlmas, mumas,  
     >     ia, ja, sparse_jac, sa, sar, sai,  
     >     nmax, uround, safe, thet, fnewt, quot1, quot2, nit, ijob, startn,  
     >     nind1, nind2, nind3, pred, facl, facr, m1, m2, nm1, nerror,  
     >     implct, banded, ldjac, lde1, ldmas, z1, z2, z3,  
     >     dydx, scal, f1, f2, f3, fjac, e1, e2r, e2i, fmas, ip1, ip2,  
     >     rwork, nfcn, njac, nstep, naccpt, nrejct, ndec, nsol, newt, ierr_cnt, 
     >     caljac, first, last, err, theta, faccon, dyno, 
     >     lrpar, rpar, lipar, ipar, lout, ierr)
      character (len=*), intent(in) :: message
      integer, intent(in) ::  
     >     info, n, itol, ijac, nzmax,  
     >     isparse, mljac, mujac, mlmas, mumas, nit, ijob, 
     >     nmax, nind1, nind2, nind3, m1, m2, nm1, nerror, ldjac, lde1, ldmas, 
     >     nfcn, njac, nstep, naccpt, nrejct, ndec, nsol, newt, ierr_cnt, lout
      integer, intent(in) ::  
     >     ia(n+1), ja(nzmax), ip1(nm1), ip2(nm1)
      logical, intent(in) ::  
     >     startn, pred, caljac, first, last, okay_to_reuse_jac, implct, banded
      real(dp), intent(in) :: 
     >     x0, ! initial 
     >     x, ! current 
     >     xend, ! end 
     >     hmax, ! max stepsize 
     >     h, ! step size 
     >     uround, ! smallest number s.t. not dropped by roundoff when add to 1.0 
     >     safe, ! safety factor in step size prediction  --  work 2 
     >     thet, ! decides whether the jacobian should be recomputed  --  work 3 
     >     fnewt, ! stopping criterion for newton's method  --  work 4 
     >     sparse_jac(nzmax),  
     >     sa(nzmax),  
     >     sar(nzmax),  
     >     sai(nzmax), 
     >     fjac(ldjac,n), ! jacobian matrix (dense or banded) 
     >     fmas(ldmas,nm1), ! mass matrix (dense or banded) 
     >     quot1, ! if quot1 < hnew/hold < quot2, step size = const  --  work 5 
     >     quot2,  ! if quot1 < hnew/hold < quot2, step size = const  --  work 6 
     >     facl, ! parameter for step size selection  --  work 8 
     >     facr, ! parameter for step size selection  --  work 9 
     >     y(n),  
     >     z1(n),  
     >     z2(2*n),  
     >     z3(n),  
     >     dydx(n),  
     >     scal(n),  
     >     f1(n),  
     >     f2(n),  
     >     f3(n), 
     >     e1(lde1,nm1), ! fac1*M - fjac, then factored 
     >     e2r(lde1,nm1), ! alphn*M - fjac, then factored 
     >     e2i(lde1,nm1), ! betan*M, then factored 
     >     atol(*), rtol(*), err, theta, faccon, dyno
      integer, intent(in) :: lrpar, lipar
      real(dp), intent(inout), target :: rpar(lrpar)
      integer, intent(inout), target :: ipar(lipar)
      real(dp), target :: rwork(4 + 4*n)
      integer, intent(out) :: ierr
      ierr = 0
      end subroutine null_radau5_debug_routine
      
      
      
      
      ! Newton-Raphson iterative solver for nonlinear systems
      ! square, banded, or sparse matrix      
      ! analytic or numerical difference jacobian      
      ! where possible, reuses jacobian to improve efficiency      
      ! uses line search method to improve "global" convergence
      include "num_newton_dble.dek"            
      include "num_newton_quad.dek"            
      
      
      
      !  MEBDFI_95 is a routine for solving stiff initial value problems
      !  with fully implicit systems of differential algebraic equations:
      !  
      !      g(t,y,y')=0, with vector y=(y(1),y(2),y(3),.....,y(n)).
      !  
      !  The code is based on MEBDFI, backward differentiation formulas (BDF)
      !  as modified (M) and extended (E) by Jeff Cash, and then modified some 
      !  more by him to handle fully implicit equations (I).  So the name 
      !  can be parsed as M-E-BDF-I, a modified extended backward difference 
      !  formula scheme for stiff fully implicit initial value problems.
      !  
      !      References
      !  
      !  J. R. Cash, Efficient numerical methods for the solution of stiff 
      !         initial-value problems and differential algebraic equations", 
      !         Proc. Roy. Soc. London, Ser. A, vol 459, (2003) pp 797-815.
      !  
      !  E. Hairer and G. Wanner, Solving Ordinary Differential
      !         Equations II, Stiff and Differential-Algebraic Problems,
      !         Springer 1996, Page 267.
      !
      include "mebdfi_solver.dek"     
      
      
      
      
      ! BOBYQA -- minimize scalar function of many variables.
      !     This subroutine seeks the least value of a function of many variables,
      !     by applying a trust region method that forms quadratic models by
      !     interpolation. There is usually some freedom in the interpolation
      !     conditions, which is taken up by minimizing the Frobenius norm of
      !     the change to the second derivative of the model, beginning with the
      !     zero matrix. The values of the variables are constrained by upper and
      !     lower bounds.
      ! by M.J.D. Powell (mjdp@cam.ac.uk)      
      include "num_bobyqa.dek"
      
      
      ! HOOKE-JEEVES -- minimize scalar function of many variables.
      !    This routine finds a point X where the nonlinear objective function
      !    F(X) has a local minimum.  X is an N-vector and F(X) is a scalar.
      !    The objective function F(X) is not required to be differentiable
      !    or even continuous.  The program does not use or require derivatives
      !    of the objective function.
      include "num_hooke.dek"
      
      
      ! global or local minimum of scalar function of 1 variable
      include "num_brent.dek"


      
      ! a few miscellaneous math routines
      ! reminder: elemental functions can be called with vectors
      ! as well as with scalar args.
      
      elemental real(dp) function safe_log10(x)
         real(dp), intent(in) :: x
         safe_log10 = log10(max(1d-99, x))
      end function safe_log10
      
      elemental real(dp) function safe_log(x)
         real(dp), intent(in) :: x
         safe_log = log(max(1d-99, x))
      end function safe_log      
            
            
      elemental real(dp) function arcsinh(x)
         real(dp), intent(in) :: x
         arcsinh = sign(1d0, x)*log(sqrt(x**2+1)+abs(x))
      end function arcsinh
            
      elemental real(dp) function d_arcsinh_dx(x)
         real(dp), intent(in) :: x
         d_arcsinh_dx = 1/sqrt(x**2+1)
      end function d_arcsinh_dx      
      
      
      elemental real(dp) function signed_log(x)
         real(dp), intent(in) :: x
         signed_log = sign(1d0,x)*log(1d0 + abs(x))
      end function signed_log
      
      elemental real(dp) function d_signed_log_dx(x)
         real(dp), intent(in) :: x
         d_signed_log_dx = 1d0 / (1d0 + abs(x))
      end function d_signed_log_dx
      
      elemental real(dp) function inverse_signed_log(sl)
         real(dp), intent(in) :: sl
         inverse_signed_log = sign(1d0,sl)*(exp(abs(sl)) - 1d0)
      end function inverse_signed_log
      
      
      ! binary search
      include "num_binary_search.dek"

      
      real(dp) function linear_interp(x1, y1, x2, y2, x)
         real(dp), intent(in) :: x1, y1, x2, y2, x
         if (x2 == x1) then
            linear_interp = (y1+y2)/2
         else
            linear_interp = y1 + (y2-y1)*(x-x1)/(x2-x1)
         end if
      end function linear_interp


      real(dp) function find0(xx1, yy1, xx2, yy2) result(x)
         ! find x between xx1 and xx2 s.t. linear_interp(xx1, yy1, xx2, yy2, x) == 0
         real(dp), intent(in) :: xx1, yy1, xx2, yy2
         real(dp) :: a, b
         a = (xx1*yy2)-(xx2*yy1)
         b = yy2-yy1
         if ((abs(a) .ge. abs(b)*1d99) .or. 
     >        ((abs(a) .ge. abs(b)*1d30) .and. 
     >           ((yy1 .ge. 0d0 .and. yy2 .le. 0d0) .or. (yy1 .le. 0d0 .and. yy2 .ge. 0d0)))) then
            x = 0.5d0*(xx1+xx2)
         else
            x = a/b
         end if
         if (yy1*yy2 <= 0) then ! sanity check
            if (x > max(xx1,xx2)) x = max(xx1,xx2)
            if (x < min(xx1,xx2)) x = min(xx1,xx2)
         end if
      end function find0


      real(dp) function find0_quadratic(xx1, yy1, xx2, yy2, xx3, yy3, ierr) result(x)
         ! find x between xx1 and xx3 s.t. quad_interp(xx1, yy1, xx2, yy2, xx3, yy3, x) == 0
         ! xx2 between xx1 and xx3; yy1 and yy3 different sign; yy2 between yy1 and yy3.
         real(dp), intent(in) :: xx1, yy1, xx2, yy2, xx3, yy3
         integer, intent(out) :: ierr
         real(dp) :: a, b, s2, denom
         ierr = 0
         s2 = (xx3**2*(-yy1 + yy2) + xx2**2*(yy1 - yy3) + xx1**2*(-yy2 + yy3))**2 -  
     >        4*(xx3*(-yy1 + yy2) + xx2*(yy1 - yy3) + xx1*(-yy2 + yy3)) 
     >           *(xx1*xx3*(-xx1 + xx3)*yy2 +  
     >              xx2**2*(xx3*yy1 - xx1*yy3) + xx2*(-xx3**2*yy1 + xx1**2*yy3))
         if (s2 < 0) then
            ierr = -1
            return
         end if
         b = sqrt(s2)
         a = xx3**2*(yy1 - yy2) + xx1**2*(yy2 - yy3) + xx2**2*(-yy1 + yy3)
         denom = 2*(xx3*(yy1 - yy2) + xx1*(yy2 - yy3) + xx2*(-yy1 + yy3))
         x = (a + b)/denom
         if (x > max(xx1,xx2,xx3)) x = (a - b)/denom
         if (x < min(xx1,xx2,xx3) .or. x > max(xx1,xx2,xx3)) ierr = -1
      end function find0_quadratic
      
      
      subroutine two_piece_linear_coeffs(x, x0, x1, x2, a0, a1, a2, ierr)
         ! interpolation value at x is a0*f(x0) + a1*f(x1) + a2*f(x2)
         real(dp), intent(in) :: x, x0, x1, x2
         real(dp), intent(out) :: a0, a1, a2
         integer, intent(out) :: ierr
         ierr = 0
         if (x0 < x1 .and. x1 < x2) then
            if (x <= x0) then
               a0 = 1; a1 = 0; a2 = 0
            else if (x >= x2) then
               a0 = 0; a1 = 0; a2 = 1
            else if (x <= x1) then
               a1 = min(1d0, max(0d0, (x - x0)/(x1 - x0)))
               a0 = 1 - a1; a2 = 0
            else if (x < x2) then
               a2 = min(1d0, max(0d0, (x - x1)/(x2 - x1))) ! a2 => 1 as x => x2
               a1 = 1 - a2; a0 = 0
            end if
         else if (x0 > x1 .and. x1 > x2) then
            if (x >= x0) then
               a0 = 1; a1 = 0; a2 = 0
            else if (x <= x2) then
               a0 = 0; a1 = 0; a2 = 1
            else if (x >= x1) then
               a1 = min(1d0, max(0d0, (x - x0)/(x1 - x0)))
               a0 = 1 - a1; a2 = 0
            else if (x > x2) then
               a2 = min(1d0, max(0d0, (x - x1)/(x2 - x1))) ! a2 => 1 as x => x2
               a1 = 1 - a2; a0 = 0
            end if
         else
            ierr = -1
         end if
      end subroutine two_piece_linear_coeffs


      end module num_lib

